
      module lbc
!
! This software is part of the NCAR TIE-GCM.  Use is governed by the 
! Open Source Academic Research License Agreement contained in the file 
! tiegcmlicense.txt.
!
! Calculate lower boundary conditions for T,U,V,Z
!
      use params_module,only: nlonp4,nlat,nlev,dz
      use cons_module,only: pi,atm_amu,gask,grav,freq_semidi,
     |  dt,re,dlamda,tgrad,cs,cor,tn
      use cons_module,only: tbound,zbound
      use addfld_module,only: addfld
      implicit none

      private calc_tzm		! calc_tzm from dt.F in TIMEGCM

!
! Total lower boundary conditions returned by this module.
! (dimensioned at full global grid, but defined at subdomains only)
!
      real,dimension(nlonp4,nlat) :: t_lbc, u_lbc, v_lbc, z_lbc
!
! Diurnal and semi-diurnal tidal perturbations using Hough functions: 
!
      complex,dimension(nlat) :: 
     |  t_di , u_di , v_di , z_di, t_sdi, u_sdi, v_sdi, z_sdi
      complex,parameter :: ci=(0.,1.), expta=1.
      complex :: bnd_sdi(nlonp4), bnd_di(nlonp4)
!
! For bndcmp:
      real :: b(nlonp4,2,2),fb(nlonp4,2)
!
! This t0 is different than the t0 in cons.
      real :: t0(nlev+1) 
!
      contains
!-----------------------------------------------------------------------
      subroutine tuvz_lbc(istep,itp)

!  HME/CTMT: 2 extra arguements istep,itp in tuzvz_lbc called from advance.F
!
! Update lower boundary subdomains of T,U,V,Z
! This is called every timestep from advance, after getgswm and 
!   before addiag.
!
      use mpi_module,only: lon0,lon1,lat0,lat1
      use init_module,only: iter
      use input_module,only: step,
     |  gswm_mi_di_ncfile,  ! gswm migrating diurnal data file
     |  gswm_mi_sdi_ncfile, ! gswm migrating semi-diurnal data file
     |  gswm_nm_di_ncfile,  ! gswm non-migrating diurnal data file
     |  gswm_nm_sdi_ncfile  ! gswm non-migrating semi-diurnal data file
      use gswm_module,only: ! (nlonp4,nlat)
     |  gswm_mi_di_z, gswm_mi_sdi_z, gswm_nm_di_z, gswm_nm_sdi_z,
     |  gswm_mi_di_t, gswm_mi_sdi_t, gswm_nm_di_t, gswm_nm_sdi_t,
     |  gswm_mi_di_u, gswm_mi_sdi_u, gswm_nm_di_u, gswm_nm_sdi_u,
     |  gswm_mi_di_v, gswm_mi_sdi_v, gswm_nm_di_v, gswm_nm_sdi_v
      use input_module,only: saber_ncfile, tidi_ncfile
      use saber_tidi,only: get_saber_tidi,saber_t,saber_z,
     |  tidi_u,tidi_v

      use input_module,only: ctmt_di_ncfile,ctmt_nlev
      use ctmt_module,only: getctmt,nalt5,ctmt_u,ctmt_v,ctmt_t,ctmt_d,
     |  ctmt_nudge,alt_ctmt,zm_t,zm_d,zm_u,zm_v
     |  ,t_zmctmt,u_zmctmt,v_zmctmt
!     |  ,tzm_tie,uzm_tie,vzm_tie,zzm_tie
! interface p levels for Z,DEN and midpoint p levels for TN,UN,VN
      use fields_module,only: z,	! z(nlev,nlon,nlat,2) usually allocated
     |  tn,un,vn

      use hist_module,only: modeltime
#ifdef MPI
      use mpi_module,only: mp_periodic_f2d
#endif
      implicit none
! Args:
      integer,intent(in) :: istep,itp
! Local:
      integer :: i,j,j2,k,ki,nalt,nalts,ifold
      real :: rstep,xkm,fac1,xkmn
      complex :: t_expt_sdi, t_expt_di, uvz_expt_sdi, uvz_expt_di
!  NOTE:  May not have right dimensions for altitude/p level w calc_tzm subroutine!
      real,dimension(1:nalt5+1,lat0:lat1) :: tzm_tie,uzm_tie,vzm_tie
      real,dimension(lon0:lon1,lat0:lat1) :: t_pert,z_pert,d_pert,
     |  u_pert,v_pert
      real :: fctmt(lon0:lon1,lat0:lat1,3) ! for mp calls
!
! Calculate exponentials
      rstep = float(step)
      t_expt_sdi = cexp(ci*freq_semidi*rstep*iter)
      t_expt_di  = cexp(ci*.5*freq_semidi*rstep*iter)
      uvz_expt_sdi = cexp(ci*freq_semidi*dt*iter)
      uvz_expt_di  = cexp(ci*.5*freq_semidi*dt*iter)
!
! Set background constants (see cons module): (zbound=96.3723km in cm, T=181.0K)
      t_lbc(lon0:lon1,lat0:lat1) = tbound
      u_lbc(lon0:lon1,lat0:lat1) = 0.
      v_lbc(lon0:lon1,lat0:lat1) = 0.
      z_lbc(lon0:lon1,lat0:lat1) = zbound

      if (len_trim(saber_ncfile) > 0 .or.
     |    len_trim(tidi_ncfile) > 0) goto 100

      if (len_trim(ctmt_di_ncfile) > 0) go to 150
!
! Add gswm or Hough mode perturbations:
! GSWM 12076: 141-212K,-96 to +79 m/s Un, -115 to +119 m/s Vn, z 94.10-97.53km 
!
! GSWM migrating diurnal:
      if (len_trim(gswm_mi_di_ncfile) > 0) then
        do j=lat0,lat1
          do i=lon0,lon1
            t_lbc(i,j) = t_lbc(i,j)+gswm_mi_di_t(i,j)
            u_lbc(i,j) = u_lbc(i,j)+gswm_mi_di_u(i,j)
            v_lbc(i,j) = v_lbc(i,j)+gswm_mi_di_v(i,j)
            z_lbc(i,j) = z_lbc(i,j)+gswm_mi_di_z(i,j)
          enddo
        enddo
!
! Hough mode diurnal:
      else ! use Hough functions for diurnal tide
        do j=lat0,lat1
          do i=lon0,lon1
            t_lbc(i,j) = t_lbc(i,j)+real(t_di(j)*bnd_di(i)*t_expt_di)
            u_lbc(i,j) = u_lbc(i,j)+real(u_di(j)*bnd_di(i)*uvz_expt_di)
            v_lbc(i,j) = v_lbc(i,j)+real(v_di(j)*bnd_di(i)*uvz_expt_di)
            z_lbc(i,j) = z_lbc(i,j)+real(z_di(j)*bnd_di(i)*uvz_expt_di)
          enddo
        enddo
      endif
!
! GSWM migrating semi-diurnal:
      if (len_trim(gswm_mi_sdi_ncfile) > 0) then
        do j=lat0,lat1
          do i=lon0,lon1
            t_lbc(i,j) = t_lbc(i,j)+gswm_mi_sdi_t(i,j)
            u_lbc(i,j) = u_lbc(i,j)+gswm_mi_sdi_u(i,j)
            v_lbc(i,j) = v_lbc(i,j)+gswm_mi_sdi_v(i,j)
            z_lbc(i,j) = z_lbc(i,j)+gswm_mi_sdi_z(i,j)
          enddo
        enddo
!
! Hough mode semi-diurnal:
      else ! use Hough functions for semi-diurnal tide
        do j=lat0,lat1
          do i=lon0,lon1
            t_lbc(i,j)=t_lbc(i,j)+real(t_sdi(j)*bnd_sdi(i)*t_expt_sdi)
            u_lbc(i,j)=u_lbc(i,j)+real(u_sdi(j)*bnd_sdi(i)*uvz_expt_sdi)
            v_lbc(i,j)=v_lbc(i,j)+real(v_sdi(j)*bnd_sdi(i)*uvz_expt_sdi)
            z_lbc(i,j)=z_lbc(i,j)+real(z_sdi(j)*bnd_sdi(i)*uvz_expt_sdi)
          enddo
        enddo
      endif
!
! GSWM non-migrating diurnal:
      if (len_trim(gswm_nm_di_ncfile) > 0) then
        do j=lat0,lat1
          do i=lon0,lon1
            t_lbc(i,j) = t_lbc(i,j)+gswm_nm_di_t(i,j)
            u_lbc(i,j) = u_lbc(i,j)+gswm_nm_di_u(i,j)
            v_lbc(i,j) = v_lbc(i,j)+gswm_nm_di_v(i,j)
            z_lbc(i,j) = z_lbc(i,j)+gswm_nm_di_z(i,j)
          enddo
        enddo
      endif
!
! GSWM non-migrating semi-diurnal:
      if (len_trim(gswm_nm_sdi_ncfile) > 0) then
        do j=lat0,lat1
          do i=lon0,lon1
            t_lbc(i,j) = t_lbc(i,j)+gswm_nm_sdi_t(i,j)
            u_lbc(i,j) = u_lbc(i,j)+gswm_nm_sdi_u(i,j)
            v_lbc(i,j) = v_lbc(i,j)+gswm_nm_sdi_v(i,j)
            z_lbc(i,j) = z_lbc(i,j)+gswm_nm_sdi_z(i,j)
          enddo
        enddo
      endif
!
! Add SABER and/or TIDI perturbations:
 100  continue
      if (len_trim(saber_ncfile) > 0 .or.
     |    len_trim(tidi_ncfile) > 0)
     |  call get_saber_tidi(modeltime)

      if (len_trim(saber_ncfile) > 0) then
        do j=lat0,lat1
          do i=lon0,lon1
            t_lbc(i,j) = t_lbc(i,j)+saber_t(i,j)
            z_lbc(i,j) = z_lbc(i,j)+saber_z(i,j)
          enddo
        enddo

!       write(6,"('tuvz_lbc: added saber_t,z to t_lbc,z_lbc')")
!       write(6,"('tuvz_lbc: saber_z min,max=',2e12.4)") 
!    |    minval(saber_z),maxval(saber_z)
!       write(6,"('tuvz_lbc: z_lbc min,max=',2e12.4)") 
!    |    minval(z_lbc),maxval(z_lbc)

      endif
      if (len_trim(tidi_ncfile) > 0) then
        do j=lat0,lat1
          do i=lon0,lon1
            u_lbc(i,j) = u_lbc(i,j)+tidi_u(i,j)
            v_lbc(i,j) = v_lbc(i,j)+tidi_v(i,j)
          enddo
        enddo
!       write(6,"('tuvz_lbc: added tidi_u,v to u_lbc,v_lbc')")
      endif
!
! Add CTMT:
 150  continue
      if (len_trim(ctmt_di_ncfile) > 0) then
       ifold = 0
       xkmn = 0.
       if (ifold==1 .and. ctmt_nlev>1) then
        write (6,"(1x,'STOP since ifold=1 and ctmt_nlev>1,=',i2)")
     |    ctmt_nlev
        stop
       endif
!  Must read files first for istep=1 in getctmt
        call getctmt (itp)
! TEMP:  The zonal means are not used
!  Calculate TIEGCM zonal means (zm) for TN, UN and VN       
! calc_tzm is not working!  istep==1, minvals ALL zero, but tn(1,1,17,itp) OK!!?
       call calc_tzm(tn(:,lon0:lon1,lat0:lat1,itp),tzm_tie,
     |    1,ctmt_nlev,lon0,lon1,lat0,lat1)
        call calc_tzm(un(:,lon0:lon1,lat0:lat1,itp),uzm_tie,
     |    1,ctmt_nlev,lon0,lon1,lat0,lat1)
        call calc_tzm(vn(:,lon0:lon1,lat0:lat1,itp),vzm_tie,
     |    1,ctmt_nlev,lon0,lon1,lat0,lat1)
      if (istep<3) then
       do i=1,nalt5+1
        write (6,"('zm at nalt min t,u,v =',i2,8e12.5)") i,
     |   minval(tzm_tie(i,:)),minval(uzm_tie(i,:)),
     |   minval(vzm_tie(i,:))
!     |   ,tn(1,1,17,itp),tn(1,nlonp4,17,itp)
       enddo
      endif

! Want to add z_pert to zbound=136.291*1.e5/sqrt(2.)~96.37km
! CTMT 12076: 155-205K,-35 to +43 m/s Un, -64 to +47 m/s Vn, z 95.24-97.94km, rho +/-15%
! CTMT 12080 w t_zm: 151-200K or -25to+24K, -34to+43m/s Un, -63to+45m/s Vn, 
!   z 95.3-97.9km or -1.065to+1.545km, -14.98to+15.68% Nden
       if (istep<2) write(6,"('tuvz_lbc 97.5km: ctmt_rho,un,vn,tn ',
     |    'pert =',4e12.4,4f8.0,2f7.1)") 
     |    minval(ctmt_d(:,:,2)),maxval(ctmt_d(:,:,2)),
     |    minval(ctmt_u(:,:,2)),maxval(ctmt_u(:,:,2)),
     |    minval(ctmt_v(:,:,2)),maxval(ctmt_v(:,:,2)),
     |    minval(ctmt_t(:,:,2)),maxval(ctmt_t(:,:,2))
! For nudging, try to put zm in u,v,den as well for lb and calc for alts>lbc
!   Only nudge z at lbc - so only need to use den and cal_ctmt_z for ki=1
!  Need to interpolate first to z_pert+zbound to interpolate between 95-97.5km 
!   Could find z_pert again 
! Nudge and revise for actual z of ctmt_nlev p levels, going from top to bottom (ki)
        xkm = 0.
        do k=1,ctmt_nlev
         ki = ctmt_nlev + 1 - k
! Use .._lbc for higher altitudes and then revise to .._lbc at lbc (bottom)
         do j=lat0,lat1
	  j2 = 2*j
          do i=lon0,lon1
! Interpolate alt_ctmt to z
           nalts = 0
           do nalt=2,nalt5
             if (z(ki,i,j,itp)*1.e-5>=alt_ctmt(nalt-1) .and.
     |           z(ki,i,j,itp)*1.e-5<=alt_ctmt(nalt)) nalts=nalt
           enddo
! Extrapolate if z<alt_ctmt or z>alt_ctmt 
           if (nalts==0) then
             if (alt_ctmt(1)>z(ki,i,j,itp)*1.e-5) nalts = 2
             if (alt_ctmt(nalt5)<z(ki,i,j,itp)*1.e-5) then
              nalts = nalt5
!  Check to see how much extrap is needed above 105 (or 120) km
              xkm = max(xkm,z(ki,i,j,itp)*1.e-5 - alt_ctmt(nalt5))
             endif
! Check to see if still nalts=0
             if (nalts==0) then
	      write (6,"(1x,'ki,j,i,itp,nalts,5,alt_ctmt,z=',6i3,5f7.1,
     |         e12.5)") ki,j,i,itp,nalts,nalt5,alt_ctmt,z(ki,i,j,itp)
	       stop
             endif
           endif	! if (nalts==0) then
           fac1 = (alt_ctmt(nalts)-z(ki,i,j,itp)*1.e-5)/2.5
!  Set t,u,v_lbc to t,u,v-zonal_mean + zonal mean of t,u,v_zm
!   Correct for fact that t,u,v on half-levels 
!  z_lbc was set in beginning to zbound, so does not need setting again since 
!         is ONLY used at LBC (ki=1)
           if (ki==1) then
	    if (ifold==1) then
             t_lbc(i,j) = tbound
             u_lbc(i,j) = 0.
             v_lbc(i,j) = 0.
            else
           if (nlat==72) then
            t_lbc(i,j) = fac1*zm_t(j,nalts-1) + (1.-fac1)*zm_t(j,nalts)
            u_lbc(i,j) = fac1*zm_u(j,nalts-1) + (1.-fac1)*zm_u(j,nalts)
            v_lbc(i,j) = fac1*zm_v(j,nalts-1) + (1.-fac1)*zm_v(j,nalts)
           else
            t_lbc(i,j) = fac1*0.5*(zm_t(j2,nalts-1)+zm_t(j2-1,nalts-1)) 
     |        + (1.-fac1)*0.5*(zm_t(j2,nalts)+zm_t(j2-1,nalts)) 
            u_lbc(i,j) = fac1*0.5*(zm_u(j2,nalts-1)+zm_u(j2-1,nalts-1)) 
     |       + (1.-fac1)*0.5*(zm_u(j2,nalts)+zm_u(j2-1,nalts)) 
            v_lbc(i,j) = fac1*0.5*(zm_v(j2,nalts-1)+zm_v(j2-1,nalts-1)) 
     |       + (1.-fac1)*0.5*(zm_v(j2,nalts)+zm_v(j2-1,nalts)) 
	   endif
            endif	! if (ifold==1) then
           else		! for ki>1 below
           if (nlat==72) then
            t_lbc(i,j) = 
!     |       0.5*(tn(ki,i,j,itp)-tzm_tie(ki,j)
!     |       + tn(ki+1,i,j,itp)-tzm_tie(ki+1,j)) + 
     |        fac1*zm_t(j,nalts-1) + (1.-fac1)*zm_t(j,nalts)
            u_lbc(i,j) = 
!     |       0.5*(un(ki,i,j,itp)-uzm_tie(ki,j)
!     |       + un(ki+1,i,j,itp)-uzm_tie(ki+1,j)) +   
     |        fac1*zm_u(j,nalts-1) + (1.-fac1)*zm_u(j,nalts)
            v_lbc(i,j) = 
!     |       0.5*(vn(ki,i,j,itp)-vzm_tie(ki,j) 
!     |       + vn(ki+1,i,j,itp)-vzm_tie(ki+1,j)) + 
     |        fac1*zm_v(j,nalts-1) + (1.-fac1)*zm_v(j,nalts)
           else
            t_lbc(i,j) = 
!     |       0.5*(tn(ki,i,j,itp)-tzm_tie(ki,j)
!     |       + tn(ki+1,i,j,itp)-tzm_tie(ki+1,j)) + 
     |       fac1*0.5*(zm_t(j2,nalts-1)+zm_t(j2-1,nalts-1)) +
     |       (1.-fac1)*0.5*(zm_t(j2,nalts)+zm_t(j2-1,nalts)) 
            u_lbc(i,j) = 
!     |       0.5*(un(ki,i,j,itp)-uzm_tie(ki,j)
!     |       + un(ki+1,i,j,itp)-uzm_tie(ki+1,j)) + 
     |       fac1*0.5*(zm_u(j2,nalts-1)+zm_u(j2-1,nalts-1)) +
     |       (1.-fac1)*0.5*(zm_u(j2,nalts)+zm_u(j2-1,nalts)) 
            v_lbc(i,j) = 
!     |       0.5*(vn(ki,i,j,itp)-vzm_tie(ki,j) 
!     |       + vn(ki+1,i,j,itp)-vzm_tie(ki+1,j)) + 
     |       fac1*0.5*(zm_v(j2,nalts-1)+zm_v(j2-1,nalts-1)) +
     |       (1.-fac1)*0.5*(zm_v(j2,nalts)+zm_v(j2-1,nalts)) 
	   endif
           endif	! if (ki==1) then,else
! Set t,d,u,v_pert(i,j) to perturbation ctmt_t,d,u,v (NOTE:  THESE ARE WRONG!?)
           t_pert(i,j) = fac1*ctmt_t(i,j,nalts-1) + 
     |      (1.-fac1)*ctmt_t(i,j,nalts)
           u_pert(i,j) = fac1*ctmt_u(i,j,nalts-1) + 
     |      (1.-fac1)*ctmt_u(i,j,nalts)
           v_pert(i,j) = fac1*ctmt_v(i,j,nalts-1) + 
     |      (1.-fac1)*ctmt_v(i,j,nalts)
! Find z from t_lbc and t,d_pert for ki=1 (lbc)
        if (ki==1) then
           d_pert(i,j) = fac1*ctmt_d(i,j,nalts-1) + 
     |      (1.-fac1)*ctmt_d(i,j,nalts)
! Save 2d boundaries to secondary histories:
      call addfld('D_LBC','D_LBC',' ',d_pert(lon0:lon1,lat0:lat1),
     |  'lon',lon0,lon1,'lat',lat0,lat1,0)
!  Calculate z_pert
        call cal_ctmt_z(istep,itp,t_lbc,z_pert,t_pert,d_pert)
! Set t_lbc for various heights - only saved for k=1,ctmt-nlev for k=1 (or lb)
            z_lbc(i,j) = z_lbc(i,j)+z_pert(i,j)
        endif	! if (ki==1) then
            t_lbc(i,j) = t_lbc(i,j)+t_pert(i,j)
            u_lbc(i,j) = u_lbc(i,j)+u_pert(i,j)
            v_lbc(i,j) = v_lbc(i,j)+v_pert(i,j)
          enddo	! do i=lon0,lon1
         enddo	! do j=lat0,lat1
! Set nalt=ki
       nalt = ki
! Print out min/max values of lbc
       if(istep<3) then
        write(6,"('nalt,s,fac1,tuv_lbc(zm),_pert,ctmt min,x=',
     |    2i2,f7.2,6e12.4/1x,6e12.4/1x,6e12.4)") nalt,nalts,fac1,
     |    minval(t_lbc),maxval(t_lbc),
     |    minval(u_lbc),maxval(u_lbc),
     |    minval(v_lbc),maxval(v_lbc),
     |    minval(t_pert),maxval(t_pert),
     |    minval(u_pert),maxval(u_pert),
     |    minval(v_pert),maxval(v_pert),
     |    minval(ctmt_t(:,:,nalts)),maxval(ctmt_t(:,:,nalts)),
     |    minval(ctmt_u(:,:,nalts)),maxval(ctmt_u(:,:,nalts)),
     |    minval(ctmt_v(:,:,nalts)),maxval(ctmt_v(:,:,nalts))
        if (ki==1) write (6,"('dz_lbc(zm),_pert,ctmt min,x=',8e12.4)")
     |    minval(z_lbc),maxval(z_lbc),
     |    minval(z_pert),maxval(z_pert),
     |    minval(d_pert),maxval(d_pert),
     |    minval(ctmt_d(:,:,nalts)),maxval(ctmt_d(:,:,nalts))
       endif

! Transfer to 3D module data 
        t_zmctmt(lon0:lon1,lat0:lat1,nalt) = t_lbc(lon0:lon1,lat0:lat1)
        u_zmctmt(lon0:lon1,lat0:lat1,nalt) = u_lbc(lon0:lon1,lat0:lat1)
        v_zmctmt(lon0:lon1,lat0:lat1,nalt) = v_lbc(lon0:lon1,lat0:lat1)
!
! Do mpi periodic points exchange for gswm with f2d(:)
! lons 1,2 <- nlonp4-3,nlonp4-2 and nlonp4-1,nlonp4 <- 3,4
!
#ifdef MPI
           fctmt(:,:,1) = t_zmctmt(lon0:lon1,lat0:lat1,nalt)
           fctmt(:,:,2) = u_zmctmt(lon0:lon1,lat0:lat1,nalt)
           fctmt(:,:,3) = v_zmctmt(lon0:lon1,lat0:lat1,nalt)
           call mp_periodic_f2d(fctmt,lon0,lon1,lat0,lat1,3)
        t_zmctmt(lon0:lon1,lat0:lat1,nalt) = fctmt(:,:,1)
        u_zmctmt(lon0:lon1,lat0:lat1,nalt) = fctmt(:,:,2)
        v_zmctmt(lon0:lon1,lat0:lat1,nalt) = fctmt(:,:,3)
#else
! lons 1,2 <- nlonp4-3,nlonp4-2 and nlonp4-1,nlonp4 <- 3,4
      t_zmctmt(1:2,:,nalt) = t_zmctmt(nlonp4-3:nlonp4-2,:,nalt)
      u_zmctmt(1:2,:,nalt) = u_zmctmt(nlonp4-3:nlonp4-2,:,nalt)
      v_zmctmt(1:2,:,nalt) = v_zmctmt(nlonp4-3:nlonp4-2,:,nalt)
      t_zmctmt(nlonp4-1:nlonp4,:,nalt) = t_zmctmt(3:4,:,nalt)
      u_zmctmt(nlonp4-1:nlonp4,:,nalt) = u_zmctmt(3:4,:,nalt)
      v_zmctmt(nlonp4-1:nlonp4,:,nalt) = v_zmctmt(3:4,:,nalt)
#endif
        enddo	! do k=1,ctmt_nlev
       if (xkm>xkmn) write (6,"('WARNING!!! Extrapolate beyond 105km ',
     |  'by xkm=',f5.1,' for ctmt_nlev=',i2)") xkm,ctmt_nlev
      endif		! for CTMT

! Print out min/max values of lbc
       if (istep<3) write(6,"('tuvz_lbc: t_lbc min,max=',2e12.4,
     |    ' u_lbc=',2e12.4,' v_lbc=',2e12.4,' z_lbc=',2e12.4)")
     |    minval(t_lbc),maxval(t_lbc),
     |    minval(u_lbc),maxval(u_lbc),
     |    minval(v_lbc),maxval(v_lbc),
     |    minval(z_lbc),maxval(z_lbc)

!
! Save 2d boundaries to secondary histories:
      call addfld('T_LBC','T_LBC',' ',t_lbc(lon0:lon1,lat0:lat1),
     |  'lon',lon0,lon1,'lat',lat0,lat1,0)
      call addfld('U_LBC','U_LBC',' ',u_lbc(lon0:lon1,lat0:lat1),
     |  'lon',lon0,lon1,'lat',lat0,lat1,0)
      call addfld('V_LBC','V_LBC',' ',v_lbc(lon0:lon1,lat0:lat1),
     |  'lon',lon0,lon1,'lat',lat0,lat1,0)
      call addfld('Z_LBC','Z_LBC',' ',z_lbc(lon0:lon1,lat0:lat1),
     |  'lon',lon0,lon1,'lat',lat0,lat1,0)
      
      end subroutine tuvz_lbc
!-----------------------------------------------------------------------
      subroutine bndry_semidiurnal
!
! Lower boundary conditions for semi-diurnal tide, using Hough functions.
! This is called once per run from init, and returns t_sdi, u_sdi, v_sdi, 
!   z_sdi at nlat latitudes. 
!
      use input_module,only: tide
      implicit none
!
! Local:
      integer,parameter :: nalf=19, malf=2
      real :: p(nlat,nalf,malf),hough(nlat,5,malf),cp(nalf/2+1)
      complex :: dzb(nlat)
      real :: b(5,19),rl(5),bhour(5),rlamda,xdot(19),ydot(19),
     |  ptscal,theta,ptjm(2*nlat+1)
      integer :: n,jm,l,lm1,m,mm1,j,ld,i,nm1
!
      complex zee(5),cl(5),expdlm
      data b/
     | 0.969152, 0.0     , 0.216046, 0.0     , 0.093838,
     | 0.0     , 0.909763, 0.0     , 0.342113, 0.0     ,
     |-0.245226, 0.0     , 0.798445, 0.0     , 0.421218,
     | 0.0     ,-0.408934, 0.0     , 0.645517, 0.0     ,
     | 0.024633, 0.0     ,-0.543993, 0.0     , 0.464159,
     | 0.0     , 0.071127, 0.0     ,-0.643189, 0.0     ,
     |-0.001292, 0.0     , 0.139613, 0.0     ,-0.699495,
     | 0.0     ,-0.006673, 0.0     , 0.225090, 0.0     ,
     | 0.000042, 0.0     ,-0.019654, 0.0     , 0.320141,
     | 0.0     , 0.000394, 0.0     ,-0.043345, 0.0     ,
     |-0.000001, 0.0     , 0.001772, 0.0     ,-0.079831,
     | 0.0     ,-0.000016, 0.0     , 0.005401, 0.0     ,
     | 0.0     , 0.0     ,-0.000112, 0.0     , 0.012932,
     | 0.0     , 0.0     , 0.0     ,-0.000476, 0.0     ,
     | 0.0     , 0.0     , 0.000005, 0.0     ,-0.001490,
     | 0.0     , 0.0     , 0.0     , 0.000031, 0.0     ,
     | 0.0     , 0.0     , 0.0     , 0.0     , 0.000129,
     | 0.0     , 0.0     , 0.0     ,-0.000002, 0.0     ,
     | 0.0     , 0.0     , 0.0     , 0.0     ,-0.000009/
      data rl/7.8519E5, 3.6665E5, 2.1098E5, 1.3671E5, 0.9565E5/
      real,external :: sddot ! util.F
!
! t0 is local to this module (different than the t0 in cons)
      t0(:) = 0.
      t0(1) = tbound
      t0(2) = tbound+dz*tgrad
!
! Longitudinal structure:
      rlamda = -2.*dlamda
      bnd_sdi(1)=cexp(ci*2.*rlamda)
      expdlm=cexp(ci*2.*dlamda)
      do i=2,nlonp4
        bnd_sdi(i)=bnd_sdi(i-1)*expdlm
      enddo
!
! Zero out if user did not provide amp/phase:
      if (all(tide==0.)) then
        t_sdi = 0.
        u_sdi = 0.
        v_sdi = 0.
        z_sdi = 0.
        return
      endif
!
      bhour = tide(6:10)
      do n=1,5
        zee(n)=tide(n)*cexp(ci*pi*bhour(n)/6.)
        cl(n)=csqrt(cmplx(gask/(atm_amu*grav*rl(n))*
     |    (t0(1)*2./7.+(t0(2)-t0(1))/dz)-.25))-.5*ci
      enddo
      jm=2*nlat+1
!
! Set up hough functions (see sphpac.F)
      do n=2,nalf+1
        nm1 = n-1
        do m=2,malf+1
          mm1=m-1
          call alfk(n,m,cp)
          do j=1,jm
            theta = float(j-1)*pi/float(jm-1)
            call lfpt(n,m,theta,cp,ptscal)
            ptjm(j) = ptscal
          enddo
          do j=1,nlat
            p(j,nm1,mm1) = ptjm(2*(nlat+1-j))
          enddo
        enddo
        do j=1,nlat
          p(j,nm1,2)=sqrt(float(n*(n+1)-6))*p(j,nm1,2)-2.*tn(j)*
     |               p(j,nm1,1)
        enddo
      enddo
!
! util.F: real function sddot(n,x,y)
      do l=1,5
        do ld=1,2
          do j=1,nlat
            xdot(:) = p(j,:,ld)
            ydot(:) = b(l,:)
            hough(j,l,ld)=sddot(19,xdot,ydot)
          enddo
        enddo
      enddo
!
! Define module data:
      do j=1,nlat
        t_sdi(j)=0.
        z_sdi(j)=0.
        dzb(j)=0.
      enddo
      do l=1,5
        do j=1,nlat
          z_sdi(j)=z_sdi(j)+zee(l)*hough(j,l,1)
          dzb(j)=dzb(j)+zee(l)*hough(j,l,2)
          t_sdi(j)=t_sdi(j)+ci*atm_amu*grav/gask*zee(l)*cl(l)*
     |      hough(j,l,1)
        enddo
      enddo
      do j=1,nlat
        u_sdi(j)=freq_semidi*re*(1.-(cor(j)/freq_semidi)**2)
        v_sdi(j)=ci*grav*(dzb(j)-2.*cor(j)/(freq_semidi*cs(j))*
     |    z_sdi(j))/u_sdi(j)
        u_sdi(j)=grav*(cor(j)/freq_semidi*dzb(j)-2./cs(j)*
     |    z_sdi(j))/u_sdi(j)
      enddo
!     write(6,"('bndry_semidiurnal: t_sdi min,max=',2e12.4)")
!    |  minval(real(t_sdi)),maxval(real(t_sdi))
!     write(6,"('bndry_semidiurnal: u_sdi min,max=',2e12.4)")
!    |  minval(real(u_sdi)),maxval(real(u_sdi))
!     write(6,"('bndry_semidiurnal: v_sdi min,max=',2e12.4)")
!    |  minval(real(v_sdi)),maxval(real(v_sdi))
!     write(6,"('bndry_semidiurnal: z_sdi min,max=',2e12.4)")
!    |  minval(real(z_sdi)),maxval(real(z_sdi))
      end subroutine bndry_semidiurnal
!-----------------------------------------------------------------------
      subroutine bndry_diurnal
!
! Lower boundary conditions for diurnal tide, using Hough functions.
! This is called once per run from init, and returns t_di, u_di, v_di, 
!   z_di at nlat latitudes. 
!
      use input_module,only: tide2
!
! Local:
      integer,parameter :: nalf=19, malf=2
      real :: p(nlat,nalf,malf),hough(nlat,5,malf),cp(nalf/2+1)
      complex :: dzb(nlat)
      real :: b(1,19),rl (1),bhour(1),rlamda,xdot(19),ydot(19),
     |  ptscal,theta,ptjm(2*nlat+1),pik
      integer :: l,m,j,n,jm,ld,i
      complex zee(1),cl(1),expdlm
!
      data b/
     | 0.282710,
     | 0.0     ,
     |-0.638229,
     | 0.0     ,
     | 0.620521,
     | 0.0     ,
     |-0.336408,
     | 0.0     ,
     | 0.117021,
     | 0.0     ,
     |-0.028332,
     | 0.0     ,
     | 0.005042,
     | 0.0     ,
     |-0.000686,
     | 0.0     ,
     | 0.000074,
     | 0.0     ,
     |-0.000006/
      data rl/0.6909E5/
      real,external :: sddot ! in util.F
!
! t0 is local to this module (different than the t0 in cons)
      t0(:) = 0.
      t0(1) = tbound
      t0(2) = tbound+dz*tgrad
!
! Calculate longitudinal structure
      rlamda = -2.*dlamda
      bnd_di(1)=cexp(ci*rlamda)
      expdlm=cexp(ci*dlamda)
      do i=2,nlonp4
        bnd_di(i)=bnd_di(i-1)*expdlm
      enddo
!
! Zero out if user did not provide amp/phase:
      if (all(tide2==0.)) then
        t_di = 0.
        u_di = 0.
        v_di = 0.
        z_di = 0.
        return
      endif
      bhour(1) = tide2(2)
      pik = 3.14159265358979312 
      do n=1,1
        zee(n)=tide2(n)*cexp(ci*pik*bhour(n)/12.)
        cl(n)=csqrt(cmplx(gask/(atm_amu*grav*rl(n))*
     |  (t0(1)*2./7.+(t0(2)-t0(1))/dz)-.25))-.5*ci
      enddo
      jm=2*nlat+1
!
! Set up hough functions:
!
      do n=1,19
        do m=1,2
          call alfk(n,m,cp)
          do j=1,jm
            theta = float(j-1)*pi/float(jm-1)
            call lfpt(n,m,theta,cp,ptscal)
            ptjm(j)=ptscal
          enddo
          do j=1,nlat
            p(j,n,m) = ptjm(2*(nlat+1-j))
          enddo
        enddo
        do j=1,nlat
          p(j,n,2)=sqrt(float(n*(n+1)-2))*p(j,n,2)-tn(j)*p(j,n,1)
        enddo
      enddo
!
! util.F: real function sddot(n,x,y)
      do l=1,1
        do ld=1,2
          do j=1,nlat
            xdot(:) = p(j,:,ld)
            ydot(:) = b(l,:)
            hough(j,l,ld)=sddot(19,xdot,ydot)
          enddo
        enddo
      enddo
!
! Generate t_di, u_di, v_di, z_di:
      do j=1,nlat
        t_di(j)=0.
        z_di(j)=0.
        dzb(j)=0.
      enddo
      do l=1,1
        do j=1,nlat
          z_di(j)=z_di(j)+zee(l)*hough(j,l,1)
          dzb(j)=dzb(j)+zee(l)*hough(j,l,2)
          t_di(j)=t_di(j)+ci*atm_amu*grav/gask*zee(l)*cl(l)*hough(j,l,1)
        enddo
      enddo
      do j=1,nlat
        u_di(j)=.5*freq_semidi*re*(1.-(cor(j)/(.5*freq_semidi))**2)
        v_di(j)=ci*grav*(dzb(j)-cor(j)/(.5*freq_semidi*cs(j))*z_di(j))/
     |    u_di(j)
        u_di(j)=grav*(cor(j)/(.5*freq_semidi)*dzb(j)-1./cs(j)*z_di(j))/
     |    u_di(j)
      enddo
!     write(6,"('bndry_diurnal: t_di min,max=',2e12.4)")
!    |  minval(real(t_di)),maxval(real(t_di))
!     write(6,"('bndry_diurnal: u_di min,max=',2e12.4)")
!    |  minval(real(u_di)),maxval(real(u_di))
!     write(6,"('bndry_diurnal: v_di min,max=',2e12.4)")
!    |  minval(real(v_di)),maxval(real(v_di))
!     write(6,"('bndry_diurnal: z_di min,max=',2e12.4)")
!    |  minval(real(z_di)),maxval(real(z_di))
      end subroutine bndry_diurnal
!-----------------------------------------------------------------------
      subroutine bndcmp
C     ****
C     ****     CALCULATE MATRICES B(nlonp4,2,2) AND VECTORS FB(nlonp4,2)
C     ****       REPRESENTING THE LOWER BOUBNDARY CONDITION IN COMP,
C     ****       WHERE PSI1 AND PSI2 ARE CALCULATED:
C     ****
C     ****         PSI(K=-1/2) = B * PSI(K=1/2) + FB
C     ****
C     ****     BNDCMP CALLS THE SUBROUTINE BNDEF TO DEFINE THE 2 X 2
C     ****       MATRICES E, F AND THE 2 VECTOR G IN THE GENERAL
C     ****       LOWER BOUNDARY CONDITION:
C     ****
C     ****         E * D(PSI)/DS + F * PSI + G = 0.
C     ****
C     ****         WHERE:
C     ****           PSI = |PSI1| AND THE BOUNDARY CONDITION IS APPLIED
C     ****                 |    |
C     ****                 |PSI2|
C     ****
C     ****           AT LEVEL ZERO
C     ****
C     ****     THIS SUBROUTINE THEN EVALUATES B AND FB FROM:
C     ****
C     ****       B = (E/DS - F/2.)**(-1) * (E/DS + F/2.)
C     ****
C     ****       FB = (E/DS - F/2.)**(-1) * G
C     ****
!
! Local:
      real :: ee(nlonp4,2,2),ff(nlonp4,2,2),gg(nlonp4,2),
     |  wm1(nlonp4,2,2),wm2(nlonp4,2,2),wm3(nlonp4,2,2),
     |  ws1(nlonp4)
      integer :: l,m,i
C     ****
C     ****     CALL BNDEF TO DEFINE E, F AND G IN S1, S2 AND S3
C     ****
      call bndef(ee,ff,gg)
C     ****
C     ****     WM1 = (E/DS - F/2.)
C     ****
C     ****     WM1 = (E/DS + F/2.)
C     ****
      do l = 1,2
        do m = 1,2
          do i = 1,nlonp4
            wm1(i,l,m) = ee(i,l,m)/dz-ff(i,l,m)/2.
            wm2(i,l,m) = ee(i,l,m)/dz+ff(i,l,m)/2.
          enddo
        enddo
      enddo
C     ****
C     ****     WM3 = WM1**(-1)
C     ****
C     ****       WS1 = DET(WM1)
C     ****
      do i = 1,nlonp4
        ws1(i) = wm1(i,1,1)*wm1(i,2,2)-wm1(i,1,2)*wm1(i,2,1)
      enddo
C     ****
C     ****     NOW INVERSE OF WM1 IN WM3
C     ****
      do i = 1,nlonp4
        wm3(i,1,1) =  wm1(i,2,2)/ws1(i)
        wm3(i,1,2) = -wm1(i,1,2)/ws1(i)
        wm3(i,2,1) = -wm1(i,2,1)/ws1(i)
        wm3(i,2,2) =  wm1(i,1,1)/ws1(i)
      enddo
C     ****
C     ****     B = WM3 * WM2
C     ****
! b and fb are bndry_module module data.
      do l = 1,2
        do m = 1,2
          do i = 1,nlonp4
            b(i,l,m) = wm3(i,l,1)*wm2(i,1,m)+wm3(i,l,2)*wm2(i,2,m)
          enddo
        enddo
      enddo
C     ****
C     ****     FB = WM3 * G
C     ****
      do l = 1,2
        do i = 1,nlonp4
          fb(i,l) = wm3(i,l,1)*gg(i,1)+wm3(i,l,2)*gg(i,2)
        enddo
      enddo
      end subroutine bndcmp
!-----------------------------------------------------------------------
      subroutine bndef(ee,ff,gg)
C     ****
C     ****     BNDEF DEFINES THE LOWER BOUNDARY CONDITION FOR THIS
C     ****       VERSION OF THE MODEL
C     ****
C     ****     THE LOWER BOUNDARY CONDITION FOR COMP IS:
C     ****
C     ****       E * D(PSI)/DS + F * PSI +G = 0.
C     ****
C     ****         WHERE:
C     ****           PSI = VECTOR(PSI1,PSI2)
C     ****           E AND F ARE 2 X 2 MATRICES
C     ****           G = VECTOR(G1,G2)
C     ****
C     ****           E, F AND G MAY BE FUNCTIONS OF LATITUDE & LONGITUDE
C     ****
C     ****       THIS SUBROUTINE DEFINES E, F AND G for BNDCMP
C     ****
!
! Args:
      real,intent(out) :: ee(nlonp4,2,2),ff(nlonp4,2,2),gg(nlonp4,2)
!
! Local:
      real :: alfa
      integer :: i
C     ****
C     ****     IN TIGCM AND TIEGCM:
C     ****
C     ****       E = |0.  0.|
C     ****           |      |
C     ****           |0.  1.|
C     ****
C     ****       F = |1.  1.|
C     ****           |      |
C     ****           |0. -1.|
C     ****
C     ****       G = |-ALFA|
C     ****           |     |
C     ****           |   0.|
C     ****
C     ****       WHERE:
C     ****         ALFA = 0.22 + 0.014 = 0.234
C     ****
      data alfa/0.234/
      do i = 1,nlonp4
        ee(i,1,1) = 0.
        ee(i,1,2) = 0.
        ee(i,2,1) = 0.
        ee(i,2,2) = 1.
        ff(i,1,1) = 1.
        ff(i,1,2) = 1.
        ff(i,2,1) = 0.
        ff(i,2,2) = -1.
        gg(i,1) = -alfa
        gg(i,2) = 0.
      enddo
      end subroutine bndef
!-----------------------------------------------------------------------
! 03/12 bae:  calc_tzm taken from dt.F TIMEGCM to calc zonal means (zm) for 
!  TN,UN,VN,Z for levels 1,ctmt_nlev in order to nudge with MSIS00 and TIDI zm
!  Had to add 'name' ('lbc') to calling sequence for call mpi_scatterlons_f3d
      subroutine calc_tzm(tn,tzm,lev0,lev1,lon0,lon1,lat0,lat1)
      use params_module,only: nlonp4
#ifdef MPI
      use mpi_module,only: mp_gatherlons_f3d,mp_scatterlons_f3d,mytidi
      implicit none
#else
      implicit none
      integer :: mytidi=0
#endif
!
! Args:
      integer,intent(in) :: lev0,lev1,lon0,lon1,lat0,lat1
      real,intent(in) :: tn(lev0:lev1,lon0:lon1,lat0:lat1)
      real,intent(out) :: tzm(lev0:lev1,lat0:lat1)
!
! Local:
      integer :: k,i,j
      real :: f_kij(lev0:lev1,nlonp4,lat0:lat1),rlon

      f_kij = 0.
      do j=lat0,lat1
        do i=lon0,lon1
          f_kij(:,i,j) = tn(:,i,j)
        enddo
      enddo

#ifdef MPI
!
! Gather global longitudes to western-most task in lat row:
      call mp_gatherlons_f3d(f_kij,lev0,lev1,lon0,lon1,lat0,lat1,1)
#endif
      tzm = 0.
      rlon = 1./real(nlonp4-4)
!
! Calculate zonal mean:
      if (mytidi==0) then
        do j=lat0,lat1
          do i=3,nlonp4-2
            tzm(:,j) = tzm(:,j)+f_kij(:,i,j)
          enddo
          tzm(:,j) = tzm(:,j)*rlon 
        enddo
!
! Copy tzm redundantly to i-index of f_kij:
        do j=lat0,lat1
          do i=1,nlonp4
            f_kij(:,i,j) = tzm(:,j)
          enddo
        enddo
      endif ! mytidi==0
#ifdef MPI
!
! Scatter 3d w/ zonal means to other tasks in my lat row:
!      call mp_scatterlons_f3d(f_kij,lev0,lev1,lon0,lon1,lat0,lat1,1)
      call mp_scatterlons_f3d(f_kij,lev0,lev1,lon0,lon1,lat0,lat1,1,
     |  'lbc')
#endif
!
! Finally copy zm to my subdomain tzm:
      do j=lat0,lat1
        tzm(:,j) = f_kij(:,lon0,j) ! all i's in f_kij are zm, so just use lon0
      enddo
      end subroutine calc_tzm
!-----------------------------------------------------------------------
      end module lbc
