module runtime_opts

!----------------------------------------------------------------------- 
! 
! Purpose: This module is responsible for reading CAM namelist cam_inparm 
!          and broadcasting namelist values if needed.  
! 
! Author:
!   Original routines:  CMS
!   Module:             T. Henderson, September 2003
!
!-----------------------------------------------------------------------

use shr_kind_mod,    only: r8=>shr_kind_r8
use spmd_utils,      only: masterproc
use namelist_utils,  only: find_group_name
use pmgrid,          only: plon
use cam_instance,    only: inst_suffix
use cam_control_mod, only: print_step_cost, indirect, pertlim, nlvdry
use cam_logfile,     only: iulog
use pspect
use units
use constituents,    only: readtrace
use time_manager,    only: timemgr_set_step_size
use filenames,       only: absems_data
use cam_abortutils,  only: endrun
use rayleigh_friction, only: rayk0, raykrange, raytau0

implicit none
private
save


!-----------------------------------------------------------------------
! Public interfaces ----------------------------------------------------
!-----------------------------------------------------------------------
public read_namelist        ! Set and/or get all runtime options


!-----------------------------------------------------------------------
!
! SOMEWHAT ALPHABETICAL listing of variables in the cam_inparm namelist:
!
! variable                description
! --------             -----------------
!
! absems_data          Dataset with absorption and emissivity factors.
!
! dtime = nnnn,        Model time step in seconds.
integer :: dtime

! 
! 
! nlvdry = nn,         Number of layers over which to do dry
!                      adjustment. Defaults to 3.
! 
! cam_branch_file      Filepath of restart file to branch from.
!                      Full pathname required.
character(len=256) :: cam_branch_file = ' '
!

!------------------------------------------------------------------
! The following 3 are specific to Rayleigh friction
! integer rayk0         vertical level at which rayleigh friction term is centered
! real(r8) raykrange    range of rayleigh friction profile; if 0, range is set automatically
! real(r8) raytau0      approximate value of decay time at model top (days);
!                       if 0., no rayleigh friction is applied
!------------------------------------------------------------------
!
!
! pertlim = n.n        Max size of perturbation to apply to initial
!                      temperature field.
!
! phys_alltoall        Dynamics/physics transpose option. See phys_grid module.
!
integer :: phys_alltoall
! 
! phys_loadbalance     Load balance option for performance tuning of 
!                      physics chunks.  See phys_grid module.  
integer :: phys_loadbalance
! 
! phys_twin_algorithm  Load balance option for performance tuning of 
!                      physics chunks.  See phys_grid module.  
integer :: phys_twin_algorithm
! 
! phys_chnk_per_thd    Performance tuning option for physics chunks.  See 
!                      phys_grid module.  
integer :: phys_chnk_per_thd
! 
! readtrace = .T.      If true, tracer initial conditions obtained from 
!                      initial file. 
!
! print_step_cost      true => print per timestep cost info
!
!
!   logical indirect     
!                    ! true => include indirect radiative effects of
!                    ! sulfate aerosols.  Default is false.
!
! met_data_file        name of file that contains the offline meteorology data
! met_data_path        name of directory that contains the offline meteorology data
!
! met_filenames_list   name of file that contains names of the offline 
!                      meteorology data files
!
! met_remove_file      true => the offline meteorology file will be removed
!
! met_cell_wall_winds  true => the offline meteorology winds are defined on the model
!                      grid cell walls
! Physics buffer
logical :: pbuf_global_allocate       ! allocate all buffers as global (default: .true.)


! Conservation checks

logical            :: print_energy_errors ! switch for diagnostic output from check_energy module

! Radiative heating rate calculation options

integer :: iradsw        ! freq. of shortwave radiation calc in time steps (positive)
                         ! or hours (negative).  Default: -1
integer :: iradlw        ! frequency of longwave rad. calc. in time steps (positive)
                         ! or hours (negative).  Default: -1
integer :: iradae        ! frequency of absorp/emis calc in time steps (positive)
                         ! or hours (negative).  Default: -12
integer :: irad_always   ! Specifies length of time in timesteps (positive)
                         ! or hours (negative) SW/LW radiation will be run continuously
                         ! from the start of an initial run.  Default: 0
logical :: spectralflux  ! calculate fluxes (up and down) per band. Default: FALSE
logical :: use_rad_dt_cosz  ! if true, uses the radiation dt for all cosz calculations

! SCM Options
integer, parameter :: max_chars = 128
character(len=max_chars) iopfile
character(len=200) :: scm_clubb_iop_name
logical  :: scm_iop_srf_prop
logical  :: scm_relaxation
logical  :: scm_diurnal_avg
logical  :: scm_crm_mode

contains

!=======================================================================

  subroutine read_namelist(nlfilename, single_column, scmlat, scmlon)

   !----------------------------------------------------------------------- 
   ! 
   ! Purpose: 
   ! Read data from namelist cam_inparm to define the run. Process some of the
   ! namelist variables to determine history and restart/branch file path 
   ! names.  Check input namelist variables for validity and print them
   ! to standard output. 
   ! 
   ! Method: 
   ! Important Note for running on SUN systems: "implicit automatic (a-z)"
   ! will not work because namelist data must be static.
   !
   ! Author: 
   ! Original version:  CCM1
   ! Standardized:      L. Bath, June 1992
   !                    T. Acker, March 1996
   !     
   !-----------------------------------------------------------------------

   ! Note that the following interfaces are prototypes proposed by Henderson 
   ! and Eaton.  They minimize coupling with other modules.  Design of these 
   ! interfaces should be refined via review by other CAM developers.  
   ! Interface *_defaultopts() gets default values from the responsible 
   ! module (Expert) prior to namelist read.  
   ! Interface *_setopts() sends values to the responsible module (Expert) 
   ! after namelist read.  Erroneous values are handled by Experts.  
   ! TBH  9/8/03 
   !
   use cam_initfiles,    only: cam_initfiles_readnl

   use phys_grid,        only: phys_grid_defaultopts, phys_grid_setopts

   use chem_surfvals,    only: chem_surfvals_readnl
   use check_energy,     only: check_energy_defaultopts, check_energy_setopts
   use radiation,        only: radiation_defaultopts, radiation_setopts, radiation_printopts
   use cam_restart,      only: restart_defaultopts, restart_setopts, restart_printopts
   use carma_flags_mod,  only: carma_readnl
   use co2_cycle,        only: co2_cycle_readnl
   use shr_string_mod,   only: shr_string_toUpper
   use scamMod,          only: scam_setopts,scam_default_opts

   ! Some modules read their own namelist input.
   use spmd_utils,          only: spmd_utils_readnl
   use cam_history,         only: history_readnl
   use physconst,           only: physconst_readnl
   use phys_control,        only: phys_ctl_readnl
   use wv_saturation,       only: wv_sat_readnl
   use ref_pres,            only: ref_pres_readnl
   use cam3_aero_data,      only: cam3_aero_data_readnl
   use cam3_ozone_data,     only: cam3_ozone_data_readnl
   use macrop_driver,       only: macrop_driver_readnl
   use microp_driver,       only: microp_driver_readnl
   use microp_aero,         only: microp_aero_readnl
   use subcol,              only: subcol_readnl
   use cloud_fraction,      only: cldfrc_readnl
   use cldfrc2m,            only: cldfrc2m_readnl
   use rk_stratiform,       only: rk_stratiform_readnl
   use unicon_cam,          only: unicon_cam_readnl
   use zm_conv_intr,        only: zm_conv_readnl
   use hk_conv,             only: hkconv_readnl
   use uwshcu,              only: uwshcu_readnl
   use pkg_cld_sediment,    only: cld_sediment_readnl
   use gw_drag,             only: gw_drag_readnl
   use qbo,                 only: qbo_readnl
   use iondrag,             only: iondrag_readnl
   use phys_debug_util,     only: phys_debug_readnl
   use conv_water,          only: conv_water_readnl
   use rad_constituents,    only: rad_cnst_readnl
   use radiation_data,      only: rad_data_readnl
   use modal_aer_opt,       only: modal_aer_opt_readnl
   use clubb_intr,          only: clubb_readnl
   use chemistry,           only: chem_readnl
   use prescribed_volcaero, only: prescribed_volcaero_readnl
   use prescribed_strataero,only: prescribed_strataero_readnl
   use aerodep_flx,         only: aerodep_flx_readnl
   use solar_data,          only: solar_data_readnl
   use solar_euv_data,      only: solar_euv_data_readnl
   use tropopause,          only: tropopause_readnl
   use aoa_tracers,         only: aoa_tracers_readnl
   use prescribed_ozone,    only: prescribed_ozone_readnl
   use prescribed_aero,     only: prescribed_aero_readnl
   use prescribed_ghg,      only: prescribed_ghg_readnl
   use aircraft_emit,       only: aircraft_emit_readnl
   use cospsimulator_intr,  only: cospsimulator_intr_readnl
   use sat_hist,            only: sat_hist_readnl
   ! Needed by sat_hist_readnl
   use cam_history,         only: hfilename_spec, mfilt, fincl, nhtfrq, avgflag_pertape
   use vertical_diffusion,  only: vd_readnl
   use cam_history_support, only: fieldname_len, fieldname_lenp2
   use cam_diagnostics,     only: diag_readnl
   use radheat,             only: radheat_readnl
#if ( defined OFFLINE_DYN )
   use metdata,             only: metdata_readnl
#endif
   use offline_driver,      only: offline_driver_readnl
   use rate_diags,          only: rate_diags_readnl
   use tracers,             only: tracers_readnl

!---------------------------Arguments-----------------------------------

   character(len=*), intent(in) :: nlfilename
   logical,          intent(in) :: single_column
   real(r8),         intent(in) :: scmlat
   real(r8),         intent(in) :: scmlon
   !-----------------------------------------------------------------------

   include 'netcdf.inc'

   !---------------------------Local variables-----------------------------
   character(len=*), parameter ::  subname = "read_namelist"

   integer ntspdy         ! number of timesteps per day
   integer t              ! history tape index
   integer ierr           ! error code
   integer unitn          ! namelist unit number

   integer f, i
   integer, parameter :: max_chars = 128


! Define the cam_inparm namelist
! ***NOTE*** If a namelist option is not described in the CAM Users Guide,
!            it is not supported.  

  ! Input data namelist items
  namelist /cam_inparm/    &
                    cam_branch_file, &
                    absems_data, & ! Why is this here?
                    dtime, &
                    nlvdry,  &
                    pertlim ,&
                    readtrace, rayk0, raykrange, raytau0, &
                    indirect, &
                    print_step_cost, &
                    phys_alltoall, phys_loadbalance, phys_twin_algorithm, &
                    phys_chnk_per_thd

  ! physics buffer
  namelist /cam_inparm/ pbuf_global_allocate

  ! conservation checks
  namelist /cam_inparm/ print_energy_errors

  ! radiative heating calculation options
  namelist /cam_inparm/ iradsw, iradlw, iradae, irad_always, spectralflux, use_rad_dt_cosz

  ! scam
  namelist /cam_inparm/ iopfile,scm_iop_srf_prop,scm_relaxation, &
                        scm_diurnal_avg,scm_crm_mode, scm_clubb_iop_name

! 
!-----------------------------------------------------------------------
!
! Determine preset values (this is currently being phased out)
!
   call preset ()
!
! Preset sulfate aerosol related variables

   indirect  = .false.

   ! restart write interval
   call restart_defaultopts( &
      cam_branch_file_out          =cam_branch_file            )

   ! Get default values of runtime options for physics chunking.
   call phys_grid_defaultopts(                      &
      phys_loadbalance_out    =phys_loadbalance,    &
      phys_twin_algorithm_out =phys_twin_algorithm, &
      phys_alltoall_out       =phys_alltoall,       &
      phys_chnk_per_thd_out   =phys_chnk_per_thd    )

   ! conservation
   call check_energy_defaultopts( &
      print_energy_errors_out = print_energy_errors )

   ! radiative heating calcs
   call radiation_defaultopts( &
      iradsw_out      = iradsw,     &
      iradlw_out      = iradlw,     &
      iradae_out      = iradae,     &
      irad_always_out = irad_always, &
      spectralflux_out = spectralflux, &
      use_rad_dt_cosz_out = use_rad_dt_cosz )

   call scam_default_opts( &
      iopfile_out=iopfile, &
      scm_iop_srf_prop_out=scm_iop_srf_prop,&
      scm_relaxation_out=scm_relaxation, &
      scm_diurnal_avg_out=scm_diurnal_avg, &
      scm_crm_mode_out=scm_crm_mode, &
      scm_clubb_iop_name_out=scm_clubb_iop_name)

   ! Read in the cam_inparm namelist from input filename

   if (masterproc) then
      write(iulog,*) 'Read in cam_inparm namelist from: ', trim(nlfilename)
      unitn = getunit()
      open( unitn, file=trim(nlfilename), status='old' )

      ! Look for cam_inparm group name in the input file.  If found, leave the
      ! file positioned at that namelist group.
      call find_group_name(unitn, 'cam_inparm', status=ierr)
      if (ierr == 0) then  ! found cam_inparm
         read(unitn, cam_inparm, iostat=ierr)  ! read the cam_inparm namelist group
         if (ierr /= 0) then
            call endrun( subname//':: namelist read returns an'// &
                         ' error condition for cam_inparm' )
         end if
      else
         call endrun(subname // ':: can''t find cam_inparm in file ' // trim(nlfilename))
      end if
      close( unitn )
      call freeunit( unitn )
   end if
!
! Scatter namelist data to all processes
#if ( defined SPMD )
   call distnl ( )
#endif

   call timemgr_set_step_size(dtime)

   ! restart write interval
   call restart_setopts( &
      cam_branch_file_in = cam_branch_file)


   ! Set runtime options for physics chunking.
   call phys_grid_setopts(                          &
       phys_loadbalance_in    =phys_loadbalance,    &
       phys_twin_algorithm_in =phys_twin_algorithm, &
       phys_alltoall_in       =phys_alltoall,       &
       phys_chnk_per_thd_in   =phys_chnk_per_thd    )

   ! conservation
   call check_energy_setopts( &
      print_energy_errors_in = print_energy_errors )

   call radiation_setopts( dtime, nhtfrq(1), &
      iradsw_in      = iradsw,     &
      iradlw_in      = iradlw,     &
      iradae_in      = iradae,     &
      irad_always_in = irad_always, &
      spectralflux_in = spectralflux, &
      use_rad_dt_cosz_in = use_rad_dt_cosz )

   ! Set runtime options for single column mode
   call scam_setopts(scmlat_in=scmlat, scmlon_in=scmlon, &
      iopfile_in=iopfile, single_column_in=single_column,&
      scm_iop_srf_prop_in=scm_iop_srf_prop,&
      scm_relaxation_in=scm_relaxation, &
      scm_diurnal_avg_in=scm_diurnal_avg, &
      scm_crm_mode_in=scm_crm_mode, &
      scm_clubb_iop_name_in=scm_clubb_iop_name)

   ! Call subroutines for modules to read their own namelist.
   ! In some cases namelist default values may depend on settings from
   ! other modules, so there may be an order dependence in the following
   ! calls.
   ! ***N.B.*** In particular, physconst_readnl should be called before
   !            the other readnl methods in case that method is used to set
   !            physical constants, some of which are set at runtime
   !            by the physconst_readnl method.
   ! Modules that read their own namelist are responsible for making sure
   ! all processes receive the values.

   call cam_initfiles_readnl(nlfilename)
   call spmd_utils_readnl(nlfilename)
   call history_readnl(nlfilename, dtime)
   call physconst_readnl(nlfilename)
   call chem_surfvals_readnl(nlfilename)
   call phys_ctl_readnl(nlfilename)
   call wv_sat_readnl(nlfilename)
   call ref_pres_readnl(nlfilename)
   call cam3_aero_data_readnl(nlfilename)
   call cam3_ozone_data_readnl(nlfilename)
   call macrop_driver_readnl(nlfilename)
   call microp_driver_readnl(nlfilename)
   call microp_aero_readnl(nlfilename)
   call clubb_readnl(nlfilename)
   call subcol_readnl(nlfilename)
   call cldfrc_readnl(nlfilename)
   call cldfrc2m_readnl(nlfilename)
   call unicon_cam_readnl(nlfilename)
   call zm_conv_readnl(nlfilename)
   call rk_stratiform_readnl(nlfilename)
   call hkconv_readnl(nlfilename)
   call uwshcu_readnl(nlfilename)
   call cld_sediment_readnl(nlfilename)
   call gw_drag_readnl(nlfilename)
   call qbo_readnl(nlfilename)
   call iondrag_readnl(nlfilename)
   call phys_debug_readnl(nlfilename)
   call conv_water_readnl(nlfilename)
   call rad_cnst_readnl(nlfilename)
   call rad_data_readnl(nlfilename)
   call modal_aer_opt_readnl(nlfilename)
   call chem_readnl(nlfilename)
   call prescribed_volcaero_readnl(nlfilename)
   call prescribed_strataero_readnl(nlfilename)
   call solar_data_readnl(nlfilename)
   call solar_euv_data_readnl(nlfilename)
   call carma_readnl(nlfilename)
   call tropopause_readnl(nlfilename)
   call aoa_tracers_readnl(nlfilename)
   call tracers_readnl(nlfilename)
   call aerodep_flx_readnl(nlfilename)
   call prescribed_ozone_readnl(nlfilename)
   call prescribed_aero_readnl(nlfilename)
   call prescribed_ghg_readnl(nlfilename)
   call co2_cycle_readnl(nlfilename)
   call aircraft_emit_readnl(nlfilename)
   call cospsimulator_intr_readnl(nlfilename)
   call sat_hist_readnl(nlfilename, hfilename_spec, mfilt, fincl, nhtfrq, avgflag_pertape)
   call diag_readnl(nlfilename)
   call radheat_readnl(nlfilename)
   call vd_readnl(nlfilename)
#if ( defined OFFLINE_DYN )
   call metdata_readnl(nlfilename)
#endif
   call offline_driver_readnl(nlfilename)
   call rate_diags_readnl(nlfilename)


! 
! Print cam_inparm input variables to standard output
! 
   if (masterproc) then
      write(iulog,*)'Time-invariant (absorption/emissivity) factor dataset is: ', trim(absems_data)

      call restart_printopts()

!
! Write physics variables from namelist cam_inparm to std. output
!
      write(iulog,9108) nlvdry
9108 format('Lowest level for dry adiabatic adjust (NLVDRY)',i10)


      call radiation_printopts()

    end if

#ifdef PERGRO
    if (masterproc) then
       write(iulog,*)'pergro for cloud water is true'
    end if
#endif

    ntspdy = nint(86400._r8/dtime) ! no. timesteps per day


end subroutine read_namelist


!=======================================================================

#ifdef SPMD
subroutine distnl
!-----------------------------------------------------------------------
!     
! Purpose:     
! Distribute namelist data to all processors.
!
! The cpp SPMD definition provides for the funnelling of all program i/o
! through the master processor. Processor 0 either reads restart/history
! data from the disk and distributes it to all processors, or collects
! data from all processors and writes it to disk.
!     
!---------------------------Code history-------------------------------
!
! Original version:  CCM2
! Standardized:      J. Rosinski, Oct 1995
!                    J. Truesdale, Feb. 1996
!
!-----------------------------------------------------------------------
   use spmd_utils,     only: masterproc, masterprocid, mpicom
   use spmd_utils,     only: mpi_integer, mpi_logical, mpi_character, mpi_real8
!-----------------------------------------------------------------------
   integer :: ierr
!
!-----------------------------------------------------------------------
! 
   call mpi_bcast(dtime,       1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(nlvdry, 1, mpi_integer, masterprocid, mpicom, ierr)

   call mpi_bcast(rayk0, 1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(raykrange,1, mpi_real8, masterprocid, mpicom, ierr)
   call mpi_bcast(raytau0, 1, mpi_real8, masterprocid, mpicom, ierr)

   call mpi_bcast(readtrace, 1, mpi_logical, masterprocid, mpicom, ierr)

   call mpi_bcast(print_step_cost,1, mpi_logical, masterprocid, mpicom, ierr)
   call mpi_bcast(pertlim, 1, mpi_real8,  masterprocid, mpicom, ierr )

   call mpi_bcast(absems_data,len(absems_data), mpi_character, masterprocid, mpicom, ierr)
   call mpi_bcast(cam_branch_file, len(cam_branch_file), mpi_character, masterprocid, mpicom, ierr)

   call mpi_bcast(indirect, 1, mpi_logical, masterprocid, mpicom, ierr)
   call mpi_bcast (absems_data,len(absems_data),mpi_character,masterprocid,mpicom, ierr)
   call mpi_bcast (cam_branch_file  ,len(cam_branch_file) ,mpi_character,masterprocid,mpicom, ierr)

   ! Physics chunk tuning
   call mpi_bcast(phys_loadbalance, 1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(phys_twin_algorithm,1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(phys_alltoall, 1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(phys_chnk_per_thd, 1, mpi_integer, masterprocid, mpicom, ierr)

   ! Physics buffer
   call mpi_bcast(pbuf_global_allocate, 1, mpi_logical, masterprocid, mpicom, ierr)

   ! Conservation
   call mpi_bcast(print_energy_errors, 1, mpi_logical, masterprocid, mpicom, ierr)

   ! Radiative heating calculation
   call mpi_bcast(iradsw,     1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(iradlw,     1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(iradae,     1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(irad_always,1, mpi_integer, masterprocid, mpicom, ierr)
   call mpi_bcast(spectralflux,1, mpi_logical, masterprocid, mpicom, ierr)
   call mpi_bcast(use_rad_dt_cosz,1, mpi_logical, masterprocid, mpicom, ierr)

end subroutine distnl
#endif



subroutine preset
!----------------------------------------------------------------------- 
! 
! Purpose: Preset namelist CAM_INPARM input variables and initialize some other variables
! 
! Method: Hardwire the values
! 
! Author: CCM Core Group
! 
!-----------------------------------------------------------------------
   use rgrid
!-----------------------------------------------------------------------
   include 'netcdf.inc'
!-----------------------------------------------------------------------
!
!
! Flags
!
   print_step_cost = .false.   ! print per timestep cost info
!
! rgrid: set default to full grid
!
   nlon(:) = plon

end subroutine preset

end module runtime_opts
