module inidat
  !----------------------------------------------------------------------- 
  ! 
  ! Purpose: Read initial dataset and spectrally truncate as appropriate.
  !
  ! Method: Initialize one or a few fields at a time, to minimize the 
  !         memory  requirements
  ! 
  ! Author: 
  ! Modified: P. Worley, to implement initialization of subsets
  !           of fields. (8/03)
  !
  !           A. Gettelman and C. Craig (Nov 2010) - put micro/macro physics 
  !           into separate routines
  ! 
  !-----------------------------------------------------------------------
  use cam_logfile, only : iulog
  use element_mod, only : element_t
  use shr_kind_mod, only: r8 => shr_kind_r8
  use spmd_utils,   only: iam, masterproc
  use cam_control_mod, only: ideal_phys, aqua_planet, pertlim
  use cam_initfiles,   only: initial_file_get_id, topo_file_get_id

  implicit none
  private

  public read_inidat

contains



  subroutine read_inidat(dyn_in)
    use dyn_comp,            only: dyn_import_t
    use parallel_mod,        only: par
    use bndry_mod,           only: bndry_exchangev
    use constituents,        only: cnst_name, cnst_read_iv, qmin
    use dimensions_mod,      only: nelemd, nlev, np, npsq
    use dof_mod,             only: putUniquePoints
    use edge_mod,            only: edgevpack, edgevunpack, InitEdgeBuffer, FreeEdgeBuffer
    use edge_mod,            only: EdgeBuffer_t
    use ncdio_atm,           only: infld
    use shr_vmath_mod,       only: shr_vmath_log
    use hycoef,              only: ps0
    use cam_abortutils,      only: endrun
    use pio,                 only: file_desc_t, io_desc_t, pio_double, pio_get_local_array_size, pio_freedecomp
    use dyn_grid,            only: get_horiz_grid_dim_d, dyn_decomp
    use chemistry   ,        only: chem_implements_cnst, chem_init_cnst
    use carma_intr,          only: carma_implements_cnst, carma_init_cnst
    use tracers     ,        only: tracers_implements_cnst, tracers_init_cnst
    use aoa_tracers ,        only: aoa_tracers_implements_cnst, aoa_tracers_init_cnst
    use clubb_intr,          only: clubb_implements_cnst, clubb_init_cnst
    use rk_stratiform,       only: rk_stratiform_implements_cnst, rk_stratiform_init_cnst
    use microp_driver,       only: microp_driver_implements_cnst, microp_driver_init_cnst
    use phys_control,        only: phys_getopts
    use co2_cycle   ,        only: co2_implements_cnst, co2_init_cnst
    use unicon_cam,          only: unicon_implements_cnst, unicon_init_cnst
    use nctopo_util_mod,     only: nctopo_util_inidat
    use cam_history_support, only: max_fieldname_len
    use cam_grid_support,    only: cam_grid_get_local_size, cam_grid_get_gcid
    use cam_map_utils,       only: iMap

    type (dyn_import_t), target, intent(inout) :: dyn_in   ! dynamics import

    type(file_desc_t), pointer :: fh_ini, fh_topo

    type(element_t), pointer :: elem(:)
    real(r8), allocatable :: tmp(:,:,:)    ! (npsp,nlev,nelemd)
    real(r8), allocatable :: qtmp(:,:)     ! (npsp*nelemd,nlev)
    logical,  allocatable :: tmpmask(:,:)  ! (npsp,nlev,nelemd) unique grid vals
    integer :: ie, k, t
    character(len=max_fieldname_len) :: fieldname
    logical :: found
    integer :: kptr, m_cnst
    type(EdgeBuffer_t) :: edge
    integer :: lsize

    integer,parameter :: pcnst = PCNST
    integer(iMap), pointer :: ldof(:) => NULL() ! Basic (2D) grid dof
    integer,       pointer :: gcid(:) => NULL() ! ID based on ldof with no holes

    integer :: rndm_seed_sz
    integer, allocatable :: rndm_seed(:)
    real(r8) :: pertval
    integer :: i, j, indx
    real(r8), parameter :: D0_0 = 0.0_r8
    real(r8), parameter :: D0_5 = 0.5_r8
    real(r8), parameter :: D1_0 = 1.0_r8
    real(r8), parameter :: D2_0 = 2.0_r8
    character(len=*), parameter :: subname='READ_INIDAT'

    fh_ini  => initial_file_get_id()
    fh_topo => topo_file_get_id()

    if(iam < par%nprocs) then
       elem=> dyn_in%elem
    else
       nullify(elem)
    end if

    lsize = cam_grid_get_local_size(dyn_decomp)	

    if (lsize /= (np*np*nelemd)) then
      call endrun(trim(subname)//': mismatch in local input array size')
    end if
    allocate(tmp(npsq,nlev,nelemd))
    tmp = 0.0_r8
    allocate(qtmp(npsq*nelemd,nlev))

    if (iam < par%nprocs) then
      if(elem(1)%idxP%NumUniquePts <=0 .or. elem(1)%idxP%NumUniquePts > np*np) then
         write(iulog,*)  elem(1)%idxP%NumUniquePts
         call endrun(trim(subname)//': invalid idxP%NumUniquePts')
      end if
    end if

    fieldname = 'U'
    tmp = 0.0_r8
    call infld(fieldname, fh_ini, 'ncol', 'lev', 1, npsq,          &
         1, nlev, 1, nelemd, tmp, found, gridname='GLL')
    if(.not. found) then
       call endrun('Could not find U field on input datafile')
    end if
    
    do ie=1,nelemd
       elem(ie)%state%v=0.0_r8
       indx = 1
       do j = 1, np
          do i = 1, np
             elem(ie)%state%v(i,j,1,:,1) = tmp(indx,:,ie)
             indx = indx + 1
          end do
       end do
    end do

    fieldname = 'V'
    tmp = 0.0_r8
    call infld(fieldname, fh_ini, 'ncol', 'lev', 1, npsq,          &
         1, nlev, 1, nelemd, tmp, found, gridname='GLL')
    if(.not. found) then
       call endrun('Could not find V field on input datafile')
    end if

    do ie=1,nelemd
       indx = 1
       do j = 1, np
          do i = 1, np
             elem(ie)%state%v(i,j,2,:,1) = tmp(indx,:,ie)
             indx = indx + 1
          end do
       end do
    end do

    fieldname = 'T'
    tmp = 0.0_r8
    call infld(fieldname, fh_ini, 'ncol', 'lev', 1, npsq,          &
         1, nlev, 1, nelemd, tmp, found, gridname='GLL')
    if(.not. found) then
       call endrun('Could not find T field on input datafile')
    end if

    do ie=1,nelemd
       elem(ie)%state%T=0.0_r8
       indx = 1
       do j = 1, np
          do i = 1, np
             elem(ie)%state%T(i,j,:,1) = tmp(indx,:,ie)
             indx = indx + 1
          end do
       end do
    end do

    if (pertlim .ne. D0_0) then
      if(masterproc) then
        write(iulog,*) trim(subname), ': Adding random perturbation bounded', &
                       'by +/- ', pertlim, ' to initial temperature field'
      end if

      call random_seed(size=rndm_seed_sz)
      allocate(rndm_seed(rndm_seed_sz))

      do ie=1,nelemd
        ! seed random number generator based on element ID
        ! (possibly include a flag to allow clock-based random seeding)
        rndm_seed = elem(ie)%GlobalId
        call random_seed(put=rndm_seed)
        do i=1,np
          do j=1,np
            do k=1,nlev
              call random_number(pertval)
              pertval = D2_0*pertlim*(D0_5 - pertval)
              elem(ie)%state%T(i,j,k,1) = elem(ie)%state%T(i,j,k,1)*(D1_0 + pertval)
            end do
          end do
        end do
      end do

      deallocate(rndm_seed)
    end if

    if (associated(ldof)) then
       call endrun(trim(subname)//': ldof should not be associated')
    end if
    call cam_grid_get_gcid(dyn_decomp, ldof)
    if (associated(gcid)) then
       call endrun(trim(subname)//': gcid should not be associated')
    end if
    allocate(gcid(size(ldof)))
    !!XXgoldyXX: Hack to please the <param>_init_cnst routines which
    !!XXgoldyXX: expect valid gcids (no holes in input array)
    !!XXgoldyXX: Best fix is to pass in lats, lons, and a mask
    where (ldof == 0)
       gcid = 1
    elsewhere
       gcid = ldof
    end where

    ! qmin = 1e-12,0,0

    do m_cnst = 1, pcnst

       found = .false.

       if(cnst_read_iv(m_cnst)) then
          tmp = 0.0_r8
          call infld(cnst_name(m_cnst), fh_ini, 'ncol', 'lev',      &
               1, npsq, 1, nlev, 1, nelemd, tmp, found, gridname='GLL')
       end if

       if(.not. found) then

          if(par%masterproc  ) write(iulog,*) 'Field ',cnst_name(m_cnst),' not found on initial dataset'

          if (microp_driver_implements_cnst(cnst_name(m_cnst))) then
             call microp_driver_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), ' initialized by "microp_driver_init_cnst"'
          else if (clubb_implements_cnst(cnst_name(m_cnst))) then
             call clubb_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "clubb_init_cnst"'
          else if (rk_stratiform_implements_cnst(cnst_name(m_cnst))) then
             call rk_stratiform_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "rk_stratiform_init_cnst"'
          else if (chem_implements_cnst(cnst_name(m_cnst))) then
             call chem_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "chem_init_cnst"'
          else if (tracers_implements_cnst(cnst_name(m_cnst))) then
             call tracers_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "tracers_init_cnst"'
          else if (aoa_tracers_implements_cnst(cnst_name(m_cnst))) then
             call aoa_tracers_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "aoa_tracers_init_cnst"'
          else if (carma_implements_cnst(cnst_name(m_cnst))) then
             call carma_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "carma_init_cnst"'
          else if (co2_implements_cnst(cnst_name(m_cnst))) then
             call co2_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "co2_init_cnst"'
          else if (unicon_implements_cnst(cnst_name(m_cnst))) then
             call unicon_init_cnst(cnst_name(m_cnst), qtmp, gcid)
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), &
                   ' initialized by "unicon_init_cnst"'
          else
              if(par%masterproc) write(iulog,*) '          ', cnst_name(m_cnst), ' set to 0.'
              qtmp = 0.0_r8
            end if
            ! Since the rest of processing uses tmp, copy qtmp into tmp
            do ie = 1, nelemd
               do k=1,nlev
                  do i = 1, npsq
                     ! Implicit reshape (qtmp is (np*np*nelemd, nlev)
                     tmp(i,k,ie) = qtmp(i+((ie-1)*npsq),k)
                  end do
               end do
            end do
       end if
       indx = 0
       do ie = 1, nelemd
          do k=1,nlev
             do i = 1, npsq
                ! Zero out the tmp values which might have been set
                ! erroneously by <param>_init_const
                if (ldof(indx + i) /= 0) then
                   ! Implicit reshape (qtmp is (np*np*nelemd, nlev)
                   tmp(i,k,ie)=max(qmin(m_cnst),tmp(i,k,ie))
                else
                   tmp(i,k,ie) = 0._r8
                end if
             end do
          end do
          indx = indx + npsq
       end do
       
       do ie=1,nelemd
          elem(ie)%state%Q(:,:,:,m_cnst)=0.0_r8
          indx = 1
          do j = 1, np
             do i = 1, np
                elem(ie)%state%Q(i,j,:,m_cnst) = tmp(indx,:,ie)
                indx = indx + 1
             end do
          end do
       end do
    end do
    ! Cleanup
    if (associated(gcid)) then
      deallocate(gcid)
      nullify(gcid)
    end if

    fieldname = 'PS'
    tmp(:,1,:) = 0.0_r8
    call infld(fieldname, fh_ini, 'ncol',      &
         1, npsq, 1, nelemd, tmp(:,1,:), found, gridname='GLL')
    if(.not. found) then
       call endrun('Could not find PS field on input datafile')
    end if

    ! Check read-in data to make sure it is in the appropriate units
    allocate(tmpmask(npsq,nelemd))
    tmpmask = (reshape(ldof, (/npsq,nelemd/)) /= 0)

    if(minval(tmp(:,1,:), mask=tmpmask) < 10000._r8) then
       call endrun('Problem reading ps field')
    end if
    deallocate(tmpmask)

    do ie=1,nelemd
       elem(ie)%state%ps_v=0.0_r8
          indx = 1
          do j = 1, np
             do i = 1, np
                elem(ie)%state%ps_v(i,j,1) = tmp(indx,1,ie)
                indx = indx + 1
             end do
          end do
    end do

    if (ideal_phys .or. aqua_planet .or. .not. associated(fh_topo)) then
       tmp(:,:,:) = 0._r8
    else    
       fieldname = 'PHIS'
       tmp(:,1,:) = 0.0_r8
       call infld(fieldname, fh_topo, 'ncol',      &
            1, npsq, 1, nelemd, tmp(:,1,:), found, gridname='GLL')
       if(.not. found) then
          call endrun('Could not find PHIS field on input datafile')
       end if
    end if

    do ie=1,nelemd
       elem(ie)%state%phis=0.0_r8
       indx = 1
       do j = 1, np
          do i = 1, np
             elem(ie)%state%phis(i,j) = tmp(indx,1,ie)
             indx = indx + 1
          end do
       end do
    end do
    
    ! once we've read all the fields we do a boundary exchange to 
    ! update the redundent columns in the dynamics
    if(iam < par%nprocs) then
       call initEdgeBuffer(par, edge, (3+pcnst)*nlev+2)
    end if
    do ie=1,nelemd
       kptr=0
       call edgeVpack(edge, elem(ie)%state%ps_v(:,:,1),1,kptr,elem(ie)%desc)
       kptr=kptr+1
       call edgeVpack(edge, elem(ie)%state%phis,1,kptr,elem(ie)%desc)
       kptr=kptr+1
       call edgeVpack(edge, elem(ie)%state%v(:,:,:,:,1),2*nlev,kptr,elem(ie)%desc)
       kptr=kptr+2*nlev
       call edgeVpack(edge, elem(ie)%state%T(:,:,:,1),nlev,kptr,elem(ie)%desc)
       kptr=kptr+nlev
       call edgeVpack(edge, elem(ie)%state%Q(:,:,:,:),nlev*pcnst,kptr,elem(ie)%desc)
    end do
    if(iam < par%nprocs) then
       call bndry_exchangeV(par,edge)
    end if
    do ie=1,nelemd
       kptr=0
       call edgeVunpack(edge, elem(ie)%state%ps_v(:,:,1),1,kptr,elem(ie)%desc)
       kptr=kptr+1
       call edgeVunpack(edge, elem(ie)%state%phis,1,kptr,elem(ie)%desc)
       kptr=kptr+1
       call edgeVunpack(edge, elem(ie)%state%v(:,:,:,:,1),2*nlev,kptr,elem(ie)%desc)
       kptr=kptr+2*nlev
       call edgeVunpack(edge, elem(ie)%state%T(:,:,:,1),nlev,kptr,elem(ie)%desc)
       kptr=kptr+nlev
       call edgeVunpack(edge, elem(ie)%state%Q(:,:,:,:),nlev*pcnst,kptr,elem(ie)%desc)
    end do

!$omp parallel do private(ie, t, m_cnst)
    do ie=1,nelemd
       do t=2,3
          elem(ie)%state%ps_v(:,:,t)=elem(ie)%state%ps_v(:,:,1)
          elem(ie)%state%v(:,:,:,:,t)=elem(ie)%state%v(:,:,:,:,1)
          elem(ie)%state%T(:,:,:,t)=elem(ie)%state%T(:,:,:,1)
       end do
       call shr_vmath_log(elem(ie)%state%ps_v,elem(ie)%state%lnps,size(elem(ie)%state%lnps))
    end do

    if(iam < par%nprocs) then
       call FreeEdgeBuffer(edge)
    end if

    !
    ! This subroutine is used to create nc_topo files, if requested
    ! 

    call nctopo_util_inidat(fh_topo, elem)

    ! Cleanup
    deallocate(tmp)
    deallocate(qtmp)
    if (associated(ldof)) then
      deallocate(ldof)
      nullify(ldof)
    end if

  end subroutine read_inidat


end module inidat
