module filter_module
  use shr_kind_mod,only: r8 => shr_kind_r8
  use cam_logfile ,only: iulog
  use pmgrid      ,only: plat,plon,plev
  use fft9        ,only: fft999
  implicit none
!
! Coefficients and factors for fft. Sub setfft is called once per run from edyn_init
!
  integer,parameter :: ntrigs = 3*plon/2+1
  real(r8) :: trigs(ntrigs)
  integer :: ifax(13)
!--------------------------------------------------------------------------
!
! For filter1:
!
! This is used by TIEGCM for basic filtering (t,u,v, et.al.),
! when nlat=72 (2.5 deg res):
!
!      integer,parameter :: kut(nlat) =
!    |   (/1  ,1  ,2  ,2  ,4  ,4  ,8  ,8  ,10 ,10 ,12 ,12,
!    |     15 ,15 ,18 ,18 ,22 ,22 ,26 ,26 ,30 ,30 ,32 ,32,
!    |     34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34,
!    |     34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34,
!    |     32 ,32 ,30 ,30 ,26 ,26 ,22 ,22 ,18 ,18 ,15 ,15,
!    |     12 ,12 ,10 ,10 ,8  ,8  ,4  ,4  ,2  ,2  ,1  ,1/)

  integer,parameter :: kut1(plat) =                   &
    (/0  ,0  ,0  ,0  ,1  ,1  ,1  ,1  ,2  ,2  ,2  ,2 , &
      3  ,3  ,3  ,3  ,4  ,4  ,4  ,4  ,6  ,6  ,6  ,6 , &
      8  ,8  ,8  ,8  ,10 ,10 ,10 ,10 ,12 ,12 ,12 ,12, &
     16  ,16 ,18 ,18 ,20 ,20 ,22 ,22 ,24 ,24 ,26 ,26, &
     26  ,26 ,24 ,24 ,22 ,22 ,20 ,20 ,18 ,18 ,16 ,16, &
     12  ,12 ,12 ,12 ,10 ,10 ,10 ,10 ,8  ,8  ,8  ,8 , &
      6  ,6  ,6  ,6  ,4  ,4  ,4  ,4  ,3  ,3  ,3  ,3 , &
      2  ,2  ,2  ,2  ,1  ,1  ,1  ,1  ,0  ,0  ,0  ,0  /)
!--------------------------------------------------------------------------
!
! For filter2:
!
! This is used by TIEGCM for O+ filtering when nlat=72 (2.5 deg res):
!
!      kut2=(/0, 0, 1, 2, 4, 4, 6, 6, 8, 8,10,10,12,12,15,15,18,18,
!    |      20,20,20,20,18,18,15,12, 8, 8, 4, 4, 4, 4, 2, 2, 1, 1,
!    |       1, 1, 2, 2, 4, 4, 4, 4, 8, 8,12,15,18,18,20,20,20,20,
!    |      18,18,15,15,12,12,10,10, 8, 8, 6, 6, 4, 4, 2, 1, 0, 0/) ! 2.5 deg
!
!     nn=(/90,90,40,40,22,22,14,14,10,10, 8, 8, 6, 6, 4, 4, 2, 2,
!    |      1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
!    |      1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,
!    |      2, 2, 4, 4, 6, 6, 8, 8,10,10,14,14,22,22,40,40,90,90/) ! 2.5 deg
!
! At 1.9 deg resolution, plat==96
!
  integer,parameter :: kut2(plat) =                               & 
    (/0, 0, 1, 1, 1, 1, 2, 2, 2, 2, 3, 3, 3, 3, 4, 4, 4, 4, 6, 6, &
      8, 8,10,10,12,12,15,15,18,18,20,20,20,20,18,18,15,12, 8, 8, &
      4, 4, 4, 4, 2, 2, 1, 1, 1, 1, 2, 2, 4, 4, 4, 4, 8, 8,12,15, &
      18,18,20,20,20,20,18,18,15,15,12,12,10,10,8, 8, 6, 6, 4, 4, &
      4, 4, 3, 3, 3, 3, 2, 2, 2, 2, 1, 1, 1, 1, 0, 0 /) 
!
  integer,parameter :: nn(plat) =                             &
    (/90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90,         &
      90,90,40,40,22,22,14,14,10,10, 8, 8, 6, 6, 4, 4, 2, 2,  &
       1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  &
       1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1, 1,  &
       2, 2, 4, 4, 6, 6, 8, 8,10,10,14,14,22,22,40,40,90,90,  &
      90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90, 90 /)
!
  contains
!-----------------------------------------------------------------------
  subroutine filter1(f,lev0,lev1,kutj,lat)
!
! Remove longitudinal waves of prognostic variables with global fft.
! Remove wave numbers greater than kut(nlat).  This is called after 
! mp_gatherlons, and only by tasks with mytidi==0. On entry, task must 
! have global longitude data defined (mp_gatherlons).
!
! Args:
  integer,intent(in) :: lev0,lev1,lat,kutj
  real(r8),intent(inout) :: f(plon,lev0:lev1)
!
! Local:
  integer :: n1,n2,k,i,nlevs,nx,nw
!
! For FFT999:
! ARGUMENT     A(M*(N+2)), WORK(M*(N+1)), TRIGS(3*N/2+1), IFAX(13)
!              N  THE LENGTH OF EACH TRANSFORM 
!              M  THE NUMBER OF TRANSFORMS TO BE DONE SIMULTANEOUSLY.
!              a(plon+2),nlevs), work(plon+1,nlevs)
!
  real(r8) :: fx(plon+2,lev1-lev0+1), wfft(plon+1,lev1-lev0+1)
 
  nlevs = lev1-lev0+1
  n1 = 2*kutj+3 ! nyquist freq (?)
  n2 = plon+2
  if (n1 > n2) then
    write(iulog,"('filter1: lat=',i2,' kutj=',i2,' n1,2=',2i3,' n1 > n2')") &
      lat,kutj,n1,n2
    return
  endif
!
! Load fx from f for the fft:
  fx(:,:) = 0._r8
  do k=lev0,lev1
    do i=1,plon
      fx(i,k) = f(i,k)
    enddo
  enddo
!
! Forward transform gridpoint to spectral:
!     SUBROUTINE FFT999(A,na,WORK,nw,TRIGS,ntrigs,IFAX,INC,JUMP,N,LOT,ISIGN)
!     real(r8) :: A(na),WORK(nw),TRIGS(ntrigs)
!
  nx = (plon+2)*nlevs
  nw = (plon+1)*nlevs
  call fft999(fx,nx,wfft,nw,trigs,ntrigs,ifax,1,plon+2,plon,nlevs,-1) ! fft9.F from TIEGCM
!
! Remove wave numbers greater than kut(lat)
  do k = 1,nlevs
    do i=n1,n2
      fx(i,k) = 0.0_r8
    enddo
  enddo
!
! Inverse transform fourier back to gridpoint:
!
  call fft999(fx,nx,wfft,nw,trigs,ntrigs,ifax,1,plon+2,plon,nlevs,1)    ! fft9.F from TIEGCM
!
! Redefine f from fx:
  do k=lev0,lev1
    do i=1,plon
      f(i,k) = fx(i,k)
    enddo
  enddo
  end subroutine filter1
!-----------------------------------------------------------------------
  subroutine filter2(f,lev0,lev1,kutj,lat)
    use edyn_geogrid,only : dlamda
!
! Remove longitudinal waves of prognostic variables with global fft.
! Remove wave numbers greater than kut2(nlat).  This is called after 
! mp_gatherlons, and only by tasks with mytidi==0. On entry, task must 
! have global longitude data defined (mp_gatherlons).
!
! Args:
    integer,intent(in) :: lev0,lev1,lat,kutj
    real(r8),intent(inout) :: f(plon,lev0:lev1)
!
! Local:
    integer :: n1,k,i,nlevs,nx,nw
    real(r8) :: fx(plon+2,lev1-lev0+1), wfft(plon+1,lev1-lev0+1)
    real(r8) :: smoothfunc,coslon
!
    nlevs = lev1-lev0+1
!
! Load local fx from inout f subdomain for the fft:
!
    fx(:,:) = 0._r8
    do k=lev0,lev1
      do i=1,plon
        fx(i,k) = f(i,k)
      enddo
    enddo
!
! Forward transform gridpoint to spectral:
!
    nx = (plon+2)*nlevs
    nw = (plon+1)*nlevs
!
! FFT999:
! ARGUMENT     A(M*(N+2)), WORK(M*(N+1)), TRIGS(3*N/2+1), IFAX(13)
!              N  THE LENGTH OF EACH TRANSFORM 
!              M  THE NUMBER OF TRANSFORMS TO BE DONE SIMULTANEOUSLY.
!              a(plon+2),nlevs), work(plon+1,nlevs)
! 
    call fft999(fx,nx,wfft,nw,trigs,ntrigs,ifax,1,plon+2,plon,nlevs,-1)    ! fft9.F from TIEGCM
!
! Wenbin's comments from TIEGCM:
! Change filters so that it does not over filtering at high latitudes, it will be the
! same as filter for low wavenumer, but wrapping up smoothly for large wavenumers, not a
! sharp transition, so there is still filtering effect in the lower latitudes
!                                                       Wenbin Wang  06/11/13
    n1=2*(kutj-1)+3
!
! Multiply by smoothing function:
! Test coslon to avoid underflow in smoothfunc at the poles
!
    do k=lev0,lev1
      do i=n1,plon
        coslon = cos(((i-n1)/2._r8)*dlamda/2._r8)
        if (coslon >= 0.1_r8) then
          smoothfunc = coslon**(2*nn(lat))
          fx(i,k) = fx(i,k)*smoothfunc
        else
          fx(i,k) = 0._r8
        endif
      enddo ! i=n1,plon
    enddo ! k=lev0,lev1
!
! Inverse transform fourier back to gridpoint:
!
    call fft999(fx,nx,wfft,nw,trigs,ntrigs,ifax,1,plon+2,plon,nlevs,1) ! fft9.F from TIEGCM
!
! Redefine f from fx:
    do k=lev0,lev1
      do i=1,plon
        f(i,k) = fx(i,k)
      enddo
    enddo
  end subroutine filter2
!-----------------------------------------------------------------------
end module filter_module
