module savefield_t3d
!
! Save 3d fields to netcdf file(s).
! B.Foster Nov, 2014.
!
  use shr_kind_mod ,only: r8 => shr_kind_r8
  use cam_logfile  ,only: iulog
  use cam_abortutils   ,only: endrun
  use netcdf
  implicit none
  contains
!-----------------------------------------------------------------------
  subroutine savefld_t3d(filepath,fname,fname_long,units,fdata,idims,&
    dimnames,action,istep)
!
! Save input field to netcdf file. Multiple files can be written in
! a single model run (multiple calls), but each file supports only 
! 3 dimensions and one or more 3d variables dimensioned 
! (idims(1),idims(2),idims(3)).
!
! Dimensions may be different lengths and order between files.
!
! On input: 
!   filepath: 
!     Path to netcdf file (file may or may not exist)
!   fname: 
!     Field short name (name of variable on the file)
!   fname_long: 
!     Field long name (variable attribute)
!   units: 
!     Field units (variable attribute)
!   fdata: 
!     Input array of field data, dimensioned (idims(1),idims(2),idims(3))
!   idims(3): 
!     The 3 dimension sizes, corresponding to the input array shape
!   dimnames(3): 
!     Names for the 3 dimensions on the file
!   action:
!     If action=='create', a new file is created and dimensions defined. 
!     If action=='append', the file must exist, and is reopened for appending.
!   istep: 
!     The current timestep from the model. This will increment in some 
!     interval of model timesteps. The unlimited dimension index on the
!     current file (itimes(ifile)) will be incremented by one when
!     istep changes.
!
! On output:
!   File has either been created, or opened for appending.
!   Variable has been written, and the file is closed.
!
! Args:
    character(len=*),intent(in) :: &
      filepath,   & ! path to netcdf dataset (may or may not exist)
      fname,      & ! field short name (of variable on file)
      fname_long, & ! field long name (variable attribute)
      units,      & ! units of the field (variable attribute)
      dimnames(3),& ! dimension names
      action        ! either 'create' or 'append'
    integer,intent(in) :: idims(3),istep
    real(r8),intent(in) :: fdata(idims(1),idims(2),idims(3))
!
! Local:
    logical :: exists
    integer :: istat,id,idvar,id1,id2,id3,id_unlim,ncid,ids3(3)
    character(len=1024) :: char1024
    integer :: i,dimlen,ifile
    character(len=1024) :: dimname
    integer,parameter :: mxfiles=50
    character(len=1024) :: filepaths(mxfiles)=' '
    integer,save :: itimes(mxfiles)=0
    integer,save :: isteps(mxfiles)=0

!   write(iulog,"(/,'Enter savefld_t3d: file=',a,' fname=',a,' istep=',i5,' action=',a)") &
!     trim(filepath),trim(fname),istep,action

    char1024=' '
!
    inquire(file=trim(filepath),exist=exists) 
!
! Create new dataset, and define dimensions 
! Dimension id's are saved between calls, and not redefined until
!   a new create call, so local dimension and variable id's, and
!   ncid logical unit are valid for the currently open file.
!
    if (action=='create') then
      istat = nf90_create(trim(filepath),NF90_CLOBBER,ncid)
      if (istat /= NF90_NOERR) then
        write(char1024,"('savefld_t3d: Error creating dataset ',a)") trim(filepath)
        call handle_ncerr(istat,char1024,1)
      endif
      ifile = addfile(filepath,filepaths,mxfiles)

      istat = nf90_def_dim(ncid,'time',NF90_UNLIMITED,id_unlim)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'savefld_t3d: Error defining unlimited dim',1)
      istat = nf90_def_dim(ncid,dimnames(1),idims(1),id1)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'savefld_t3d: Error defining idim1',1)
      istat = nf90_def_dim(ncid,dimnames(2),idims(2),id2)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'savefld_t3d: Error defining idim2',1)
      istat = nf90_def_dim(ncid,dimnames(3),idims(3),id3)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'savefld_t3d: Error defining idim3',1)
      write(iulog,"('savefld_t3d: Created nc file ',a)") trim(filepath)
!
! Open existing file for appending:
!
    elseif (action=='append'.and.exists) then
      istat = nf90_open(trim(filepath),NF90_WRITE,ncid)
      if (istat /= NF90_NOERR) then
        write(char1024,"('savefld_t3d: Error opening dataset ',a)") trim(filepath)
        call handle_ncerr(istat,char1024,1)
      endif
!     write(iulog,"('savefld_t3d: Opened nc file ',a,' for appending.')") trim(filepath)

      istat = nf90_inq_dimid(ncid,dimnames(1),id1)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'Error getting id for idim1',1)
      istat = nf90_inq_dimid(ncid,dimnames(2),id2)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'Error getting id for idim2',1)
      istat = nf90_inq_dimid(ncid,dimnames(3),id3)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'Error getting id for idim3',1)
      istat = nf90_inq_dimid(ncid,'time',id_unlim)
      if (istat /= NF90_NOERR) call handle_ncerr(istat,'Error getting id for id_unlim',1)

      istat = nf90_redef(ncid) ! open for redefining or changing
      if (istat /= NF90_NOERR) &
        call handle_ncerr(istat,'savefld_t3d: Error return from nf90_redef',1)

      ifile = findpath(filepath,filepaths,mxfiles)
!
! Error: open non-existent file.
!
    elseif (action=='append'.and..not.exists) then
      write(iulog,"('>>> savefld_t3d: action=',a,' but file ',a,' does not exist.')") &
        action,trim(filepath)
      call endrun
    endif
!
! Increment time index for current file if this is a new model timestep: 
!
    if (istep /= isteps(ifile)) then   ! new timestep
      itimes(ifile) = itimes(ifile)+1
      isteps(ifile) = istep
    endif
!
! Confirm that input dims match dims on the file:
    ids3(1)=id1 ; ids3(2)=id2 ; ids3(3)=id3
    do i=1,3
      istat = nf90_inquire_dimension(ncid,ids3(i),dimname,dimlen) 
      if (trim(dimname) /= trim(dimnames(i)) .or. dimlen /= idims(i)) then
        write(iulog,"('>>> savefld_t3d: input dim ',i2,' does not match dim on file:')") i
        write(iulog,"('>>> i=',i2,' input dimnames(i)=',a,' dimname on file=',a)") &
          i,trim(dimnames(i)),trim(dimname)
        write(iulog,"('>>> i=',i2,' input idims(i)=',a,' len dim on file=',a)") &
          i,idims(i),dimlen
        call endrun
      endif
    enddo
!   write(iulog,"('savefld_t3d: file=',a,' dimnames=',3a8,' idims=',3i5,' id_unlim=',i5)") &
!     trim(filepath),dimnames,idims,id_unlim
!
! Define current variable, if not already defined:
!
    if (nf90_inq_varid(ncid,trim(fname),idvar) /= NF90_NOERR) then ! variable not yet defined
      istat = nf90_def_var(ncid,trim(fname),NF90_DOUBLE,(/id1,id2,id3,id_unlim/),idvar)
      if (istat /= NF90_NOERR) then
        write(char1024,"('savefld_t3d: Error defining var ',a,' on file ',a)") &
          trim(fname),trim(filepath)
        call handle_ncerr(istat,char1024,1)
      else
!       write(iulog,"('savefld_t3d: Defined var ',a,' on file ',a,' idvar=',i4)") &
!         trim(fname),trim(filepath),idvar
      endif
    endif
!
! Reset long_name and units (long_name may change)
    istat = nf90_inq_varid(ncid,fname,idvar)
    istat = nf90_put_att(ncid,idvar,'long_name',fname_long)
    istat = nf90_put_att(ncid,idvar,'units',units)
!
! Take out of define mode, and into data mode:
!
    istat = nf90_enddef(ncid)
!
! Write data to the file at current unlimited time index:
!
    istat = nf90_inq_varid(ncid,fname,idvar)
    if (idvar > 0) then ! this should always be true
      istat = nf90_put_var(ncid,idvar,fdata,&
        (/1,1,1,itimes(ifile)/),(/idims(1),idims(2),idims(3),1/)) ! start,count
      if (istat /= NF90_NOERR) then
        write(char1024,"('savefld_t3d: Error writing var ',a)") trim(fname)
        call handle_ncerr(istat,char1024,1)
      endif
!     write(iulog,"('savefld_t3d: Wrote var ',a,' to file ',a,' ifile=',i4,' itimes(ifile)=',i5,' istep=',i4,' min,max=',2(1pe12.4))") &
!       fname,trim(filepath),ifile,itimes(ifile),istep,minval(fdata),maxval(fdata)
    endif
!
! Close the file:
!
    istat = nf90_close(ncid) 
  end subroutine savefld_t3d
!-----------------------------------------------------------------------
  subroutine handle_ncerr(istat,msg,ifatal)
    implicit none
!
! Handle a netcdf lib error:
!
    integer,intent(in) :: istat,ifatal
    character(len=*),intent(in) :: msg
!
    write(iulog,"(/72('-'))")
    write(iulog,"('>>> Error from netcdf library:')")
    write(iulog,"(a)") trim(msg)
    write(iulog,"('istat=',i5)") istat
    write(iulog,"(a)") nf90_strerror(istat)
    write(iulog,"(72('-')/)")
    if (ifatal > 0) call endrun
  end subroutine handle_ncerr
!-----------------------------------------------------------------------
  integer function addfile(filepath,filepaths,mxfiles)
    implicit none
!
! Args:
    integer,intent(in) :: mxfiles
    character(len=*),intent(in) :: filepath
    character(len=*),intent(inout) :: filepaths(mxfiles)
!
! Local:
    integer :: i

    if (len_trim(filepath)==0) then
      write(iulog,"('>>> WARNING addfile: Empty filepath.')") 
      call endrun
    endif
    addfile = 0
    do i=1,mxfiles
      if (len_trim(filepaths(i))==0) then
        filepaths(i) = trim(filepath)
        addfile = i
        return
      endif
    enddo
    write(iulog,"('>>> WARNING addfile: Could not add filepath ',a,': need to increment mxfiles')") &
      trim(filepath)
    call endrun

  end function addfile
!-----------------------------------------------------------------------
  integer function findpath(filepath,filepaths,mxfiles)
!
! Args:
    integer,intent(in) :: mxfiles
    character(len=*),intent(in) :: filepath
    character(len=*),intent(in) :: filepaths(mxfiles)
!
! Local:
    integer :: i

    if (len_trim(filepath)==0) then
      write(iulog,"('>>> WARNING findpath: Empty filepath.')") 
      call endrun
    endif
    findpath = 0
    do i=1,mxfiles
      if (trim(filepaths(i))==trim(filepath)) then
        findpath = i
        return
      endif
    enddo
    write(iulog,"('>>> WARNING findpath: Could not find filepath ',a)") &
      trim(filepath)
    call endrun

  end function findpath
!-----------------------------------------------------------------------
end module savefield_t3d
