module check_for_nans
  use shr_kind_mod ,only: r8 => shr_kind_r8
  use cam_logfile  ,only: iulog
  use cam_abortutils   ,only: endrun
  contains
!-----------------------------------------------------------------------
      subroutine check_nans(f,id1,id2,id3,name,n_total,&
        ireplace,replace,iprint,ifatal)
!
! Check for existence of +/-INF and NaN's in field f(id1,id2,id3).
!   If iprint==1  -> print warnings only if INF or NaNs are found
!   If iprint==2  -> always print number of INF and NaNs found
!   If ifatal > 0 -> stop program when first INF or NaNs are found
! Note: Can distinguish between +/-INF (not really NaNs), but cannot 
!       distinguish between types of actual NaNs (+/-NaNQ and NaNS).
! IBM only. See pp318-319 User's Guide Version 8 XL Fortran for AIX
!
      implicit none
!
! Args:
      integer,intent(in) :: id1,id2,id3,iprint,ifatal,ireplace
      integer,intent(out) :: n_total ! total number of +/-INF+NaNs
      real(r8),intent(inout) :: f(id1,id2,id3)
      character(len=*),intent(in) :: name 
      real(r8),intent(in) :: replace
!
! Local:
      real(r8) :: plus_inf,minus_inf,plus_nanq,minus_nanq,sig_nan
!
! For double precision 8-byte reals (-qrealsize=8):
      data plus_inf   /z'7ff0000000000000'/  ! INF   (overflow)
      data minus_inf  /z'fff0000000000000'/  ! -INF  (underflow)
      data plus_nanq  /z'7ff8000000000000'/  ! NaNQ  (plus quiet NaN)
      data minus_nanq /z'fff8000000000000'/  ! -NaNQ (minus quiet NaN)
      data sig_nan    /z'7ff0000000000001'/  ! NaNS  (signalling NaN)
!
! For single precision (4-byte) reals:
!     data plus_inf   /z'7f800000'/  ! INF   (overflow)
!     data minus_inf  /z'ff800000'/  ! -INF  (underflow)
!     data plus_nanq  /z'7fc00000'/  ! NaNQ  (plus quiet NaN)
!     data minus_nanq /z'ffc00000'/  ! -NaNQ (minus quiet NaN)
!     data sig_nan    /z'7f800001'/  ! NaNS  (signalling NaN)
!
      integer :: i1,i2,i3
      integer :: &
        n_plus_inf,   & ! number of INF
        n_minus_inf,  & ! number of -INF
        n_nan           ! total number of NaNs (+/-NaNQ and NaNS)
!
! Init:
      n_plus_inf = 0
      n_minus_inf = 0
      n_nan = 0
      n_total = 0
!
! Scan array:
      do i3=1,id3
        do i2=1,id2
!
! +/-INF are detected by simple comparison:
          n_plus_inf   = n_plus_inf   + count(f(:,i2,i3)==plus_inf) 
          n_minus_inf  = n_minus_inf  + count(f(:,i2,i3)==minus_inf) 
!
! NaNs (NaNQ or NaNS) are detected by (a /= a):
          n_nan        = n_nan        + count(f(:,i2,i3)/=f(:,i2,i3))
          n_total = n_plus_inf+n_minus_inf+n_nan
!
!         write(iulog,"('i3=',i3,' i2=',i3,' n_plus_inf=',i8,' n_minus_inf=',i8,' n_nan=',i8,' n_total=',i8)") &
!           i3,i2,n_plus_inf,n_minus_inf,n_nan,n_total
!
! Fatal when first INF or NaN is found:
          if (ifatal > 0 .and. n_total > 0) then
            write(iulog,"(/,'>>> FATAL: Found INF and/or NaNs in field ',a)") name
            write(iulog,"('  Dimensions id1,id2,id3=',3i4)") id1,id2,id3
            write(iulog,"('  First INF or NaN found at id2=',i4,', id3=',i4)") i2,i3
            write(iulog,"('  n_plus_inf   = ',i6)") n_plus_inf
            write(iulog,"('  n_minus_inf  = ',i6)") n_minus_inf
            write(iulog,"('  n_nan (NaNS or NaNQ) = ',i6)") n_nan
            write(iulog,"('  data(:,',i3,',',i3,') = ',/,(6e12.4))") i2,i3,f(:,i2,i3)
            call endrun('check_nans')
          endif ! ifatal > 0
!
! Replace any INF or NaNs with given value:
          if (ireplace > 0 .and. n_total > 0) then
            do i1=1,id1
              if (f(i1,i2,i3)==plus_inf.or.f(i1,i2,i3)==minus_inf.or. &
                  f(i1,i2,i3)/=f(i1,i2,i3)) f(i1,i2,i3) = replace
            enddo
            if (iprint > 0) then
              write(iulog,"('>>> check_nans replaced ',i8,' NaNs in ',a,' with ',1pe12.4)") &
                n_total,name,replace
            endif
          endif
        enddo ! i2=1,id2
      enddo ! i3=1,id3
!
! Print level 1 (print warnings only if INF or NaNs are found):
      if (iprint > 0) then
        if (n_plus_inf > 0) write(6,"('>>> WARNING: found ',i6,' INF values in field ',a,' (id1,2,3=',3i4,')')") &
          n_plus_inf,name,id1,id2,id3
        if (n_minus_inf > 0) write(6,"('>>> WARNING: found ',i6,' -INF values in field ',a,' (id1,2,3=',3i4,')')") &
          n_minus_inf,name,id1,id2,id3
        if (n_nan > 0) write(6,"('>>> WARNING: found ',i6,' NaNS or NaNQ values in field ',a,' (id1,2,3=',3i4,')')") &
          n_nan,name,id1,id2,id3
      endif
!
      end subroutine check_nans
!-------------------------------------------------------------------
end module check_for_nans

