module dp_coupling
!BOP
!
! !MODULE: dp_coupling --- dynamics-physics coupling module
!
   use shr_kind_mod,      only: r8 => shr_kind_r8
   use rgrid,             only: nlon
   use ppgrid,            only: pcols, pver, pverp
   use phys_grid
   use physics_buffer,    only: physics_buffer_desc, pbuf_get_chunk, &
                            pbuf_get_field, pbuf_get_index
   
   use physics_types,     only: physics_state, physics_tend
   use constituents,      only: pcnst, qmin
   use physconst,         only: cpair, gravit, rair, zvir, cpairv, rairv
   use geopotential,      only: geopotential_t
   use check_energy,      only: check_energy_timestep_init
   use dynamics_vars,     only: T_FVDYCORE_GRID
   use dyn_comp,          only: dyn_import_t, dyn_export_t
   use cam_abortutils,    only: endrun
#if defined ( SPMD )
   use spmd_dyn,          only: local_dp_map, block_buf_nrecs, chunk_buf_nrecs
#endif
   use perf_mod
   use cam_logfile,       only: iulog
   use cam_history,       only: outfld
   use phys_control,      only: waccmx_is

!--------------------------------------------
!  Variables needed for WACCM-X
!--------------------------------------------
   use constituents,  only: cnst_get_ind, cnst_mw  !Needed to access constituent molecular weights
   use shr_const_mod, only: shr_const_rgas         !Gas constant
   use mo_jeuv,       only: nIonRates              ! Number of ionization rates (11 rates)
   use mo_photo,      only: ion_rates_idx          ! Index to IonRates
   use dpie_coupling, only: d_pie_coupling         ! WACCM-X ionosphere/electrodynamics coupling
   use edyn_init,     only: use_dynamo             ! logical flag
   use edyn_init,     only: use_dynamo_drifts      ! logical flag
   use short_lived_species, only : slvd_index,slvd_pbf_ndx => pbf_idx ! Routines to access short lived species in storage and pbuf
   use edyn_init,     only: iamie
!
! !PUBLIC MEMBER FUNCTIONS:
      PUBLIC d_p_coupling, p_d_coupling

!
! !DESCRIPTION:
!
!      This module provides 
!
!      \begin{tabular}{|l|l|} \hline \hline
!        d\_p\_coupling    &  dynamics output to physics input \\ \hline
!        p\_d\_coupling    &  physics output to dynamics input \\ \hline 
!                                \hline
!      \end{tabular}
!
! !REVISION HISTORY:
!   00.06.01   Boville    Creation
!   01.10.01   Lin        Various revisions
!   01.03.26   Sawyer     Added ProTeX documentation
!   01.06.27   Mirin      Separate noncoupling coding into new routines
!   01.07.13   Mirin      Some support for multi-2D decompositions
!   02.03.01   Worley     Support for nontrivial physics remapping
!   03.03.28   Boville    set all physics_state elements, add check_energy_timestep_init
!   03.08.13   Sawyer     Removed ghost N1 region in u3sxy
!   05.06.28   Sawyer     Simplified interfaces -- only XY decomposition 
!   05.10.25   Sawyer     Extensive refactoring, dyn_interface
!   05.11.10   Sawyer     Now using dyn_import/export_t containers
!   06.07.01   Sawyer     Transitioned constituents to T_TRACERS
!
!EOP
!-----------------------------------------------------------------------

   private 

!----------WACCM-X-----------
! Photoelectron production rate is private dp_coupling module data. 
! It is allocated in sub d_p_coupling, and assigned value in sub 
! p_d_coupling (using IonRates), and passed to sub d_pie_coupling 
! (dpie_coupling.F90) by sub d_p_coupling.

   real(r8),allocatable,save :: IonRates_blck(:,:,:,:) ! ionization rates (blocks)
   real(r8),allocatable,save :: ion_OpO2_blck(:,:,:)   ! Op+O2 rate (blocks)
   real(r8),allocatable,save :: ion_OpN2_blck(:,:,:)   ! Op+N2 rate (blocks)
   real(r8),allocatable,save :: o2pmmr_blck(:,:,:)     ! O2+ (blocks)
   real(r8),allocatable,save :: nopmmr_blck(:,:,:)     ! NO+ (blocks)
   real(r8),allocatable,save :: n2pmmr_blck(:,:,:)     ! N2+ (blocks)
   real(r8),allocatable,save :: diag_ik(:,:)           ! (i,k) for outfld calls
!
   integer :: sIndxOp, sIndxO2p, sIndxNOp, sIndxN2p    ! indices for accessing ions in pbuf when non-advected

!----------------------------

   real(r8), parameter ::  D0_5                    =  0.5_r8
   real(r8), parameter ::  D1_0                    =  1.0_r8

CONTAINS

!-----------------------------------------------------------------------
!BOP
! !IROUTINE: d_p_coupling --- convert dynamics output to physics input
!
! !INTERFACE: 
  subroutine d_p_coupling(grid, phys_state, phys_tend,  pbuf2d, dyn_out)

! !USES:
!    use physics_buffer, only: physics_buffer_desc, pbuf_get_chunk, &
!                              pbuf_get_field, pbuf_get_index
    use constituents,   only: cnst_get_type_byind, qmin
    use physics_types,  only: set_state_pdry, set_wet_to_dry

    use pmgrid,         only: plev, plevp
    use ctem,           only: ctem_diags, do_circulation_diags
    use gravity_waves_sources, only: gws_src_fnct
    use physconst,      only: physconst_update
    use shr_const_mod,  only: shr_const_rwv
    use dyn_comp,       only: frontgf_idx, frontga_idx, uzm_idx
    use qbo,            only: qbo_use_forcing
    use phys_control,   only: use_gw_front, use_gw_front_igw
    use zonal_mean,     only: zonal_mean_3D
    use dyn_comp,       only: ui_idx, vi_idx, wi_idx ! for WACCM-X
    use mo_aurora,      only: indxAMIEefxg, indxAMIEkevg
    use time_manager,   only: is_first_step, is_first_restart_step
    use chem_mods,      only : adv_mass      ! Array holding mass values for short lived species
    use mo_chem_utls,   only : get_spc_ndx   ! Routine to get index of adv_mass array for short lived species

!-----------------------------------------------------------------------
    implicit none
!-----------------------------------------------------------------------
! !INPUT PARAMETERS:
!
    type(T_FVDYCORE_GRID), intent(in) :: grid ! FV Dynamics grid
    type(physics_buffer_desc), pointer :: pbuf2d(:,:)
    type(dyn_export_t), intent(in)    :: dyn_out  ! dynamics export 

! !OUTPUT PARAMETERS:

    type(physics_state), intent(inout), dimension(begchunk:endchunk) :: phys_state
    type(physics_tend ), intent(inout), dimension(begchunk:endchunk) :: phys_tend
    

! !DESCRIPTION:
!
!   Coupler for converting dynamics output variables into physics 
!   input variables
!
! !REVISION HISTORY:
!   00.06.01   Boville    Creation
!   01.07.13   AAM        Some support for multi-2D decompositions
!   02.03.01   Worley     Support for nontrivial physics remapping
!   02.05.02   Sawyer     u3s made inout due to ghosting in d2a3dikj
!   03.08.05   Sawyer     Removed pe11k, pe11kln (for defunct Rayl fric)
!   04.08.29   Eaton      Added lat, lon coords to physics_state type
!   05.06.28   Sawyer     Simplified interface -- on XY decomp vars.
!   05.07.06   Sawyer     Added dyn_state as argument
!   05.10.31   Sawyer     Refactoring, replaced dyn_state by dyn_interface
!
!EOP
!-----------------------------------------------------------------------
!BOC
! !LOCAL VARIABLES:

! Variables from dynamics export container
    real(r8), pointer :: phisxy(:,:)              ! surface geopotential
    real(r8), pointer :: psxy (:,:)               ! surface pressure
    real(r8), pointer :: u3sxy(:,:,:)             ! u-wind on d-grid
    real(r8), pointer :: v3sxy(:,:,:)             ! v-wind on d-grid
    real(r8), pointer :: ptxy (:,:,:)             ! Virtual pot temp
    real(r8), pointer :: tracer(:,:,:,:)          ! constituents
    real(r8), pointer :: omgaxy(:,:,:)            ! vertical velocity
    real(r8), pointer :: pexy  (:,:,:)            ! edge pressure
    real(r8), pointer :: pelnxy(:,:,:)            ! log(pe)
    real(r8), pointer :: pkxy  (:,:,:)            ! pe**cappa
    real(r8), pointer :: pkzxy (:,:,:)            ! f-v mean of pk

!------------WACCM-X---------------
    real(r8),pointer :: opmmr_blck(:,:,:)         ! O+ (blocks)
    real(r8),pointer :: sigma_ped_blck(:,:,:)     ! Pederson Conductivity (blocks)
    real(r8),pointer :: sigma_hall_blck(:,:,:)    ! Hall Conductivity (blocks)
    real(r8),pointer :: optm1_blck(:,:,:)         ! O+ at previous time step(blocks)
    real(r8),pointer :: ti_blck(:,:,:)            ! Ion temp (blocks)
    real(r8),pointer :: te_blck(:,:,:)            ! Elec temp (blocks)
    real(r8),pointer :: zi_blck(:,:,:)            ! zi (blocks) (geopoten ht above surface (interfaces))
    real(r8),pointer :: phis_blck(:,:)            ! surface geopotential (blocks) 

    real(r8),pointer :: ui_blck(:,:,:)            ! zonal ion drift   
    real(r8),pointer :: vi_blck(:,:,:)            ! meridional ion drift   
    real(r8),pointer :: wi_blck(:,:,:)            ! vertical ion drift   

    real(r8),allocatable :: amie_efxg_blck(:,:)     ! AMIE energy flux   
    real(r8),allocatable :: amie_kevg_blck(:,:)     ! AMIE mean energy   
!----------------------------------

    integer :: i,ib,j,k,m,lchnk      ! indices
    integer :: ncol                  ! number of columns in current chunk
    integer :: lats(pcols)           ! array of latitude indices
    integer :: lons(pcols)           ! array of longitude indices
    integer :: blksiz                ! number of columns in 2D block
    integer :: tsize                 ! amount of data per grid point passed to physics
    integer, allocatable, dimension(:,:) :: bpter
                                     ! offsets into block buffer for packing data
    integer :: cpter(pcols,0:pver)   ! offsets into chunk buffer for unpacking data

    real(r8) :: rlat(pcols)          ! array of latitudes (radians)
    real(r8) :: rlon(pcols)          ! array of longitudes (radians)
    real(r8) :: qmavl                ! available q at level pver-1
    real(r8) :: dqreq                ! q change at pver-1 required to remove q<qmin at pver
    real(r8) :: qbot                 ! bottom level q before change
    real(r8) :: qbotm1               ! bottom-1 level q before change
    real(r8) :: pic(pcols)           ! ps**cappa
    real(r8) :: fraction
    real(r8), allocatable :: u3(:, :, :)       ! u-wind on a-grid (i,k,j)
    real(r8), allocatable :: v3(:, :, :)       ! v-wind on a-grid (i,k,j)
    real(r8), allocatable :: wuxy(:, :, :)     ! u-wind on a-grid for dpie_coupling (i,j,k)
    real(r8), allocatable :: wvxy(:, :, :)     ! v-wind on a-grid for dpie_coupling (i,j,k)
    real(r8), allocatable, dimension(:) :: bbuffer, cbuffer
                                     ! transpose buffers

    real(r8) :: rmassO2    ! O2 molecular weight kg/kmol
    real(r8) :: rmassO1    ! O atomic weight kg/kmol
    real(r8) :: rmassH     ! H atomic weight kg/kmol
    real(r8) :: rmassN2    ! N2 molecular weight kg/kmol
    real(r8) :: rmassN     ! N molecular weight kg/kmol
    real(r8) :: rmassO2p   ! O2+ molecular weight kg/kmol
    real(r8) :: rmassNOp   ! NO+ molecular weight kg/kmol
    real(r8) :: rmassN2p   ! N2+ molecular weight kg/kmol
    real(r8) :: rmassOp    ! O+ molecular weight kg/kmol

    real(r8) :: zvirv(pcols,pver)    ! Local zvir array pointer

    real(r8) :: uzm(plev,grid%jfirstxy:grid%jlastxy) ! Zonal mean zonal wind

    integer  :: im, jm, km, kmp1, iam
    integer  :: ifirstxy, ilastxy, jfirstxy, jlastxy
    integer  :: ic, jc
    integer  :: astat
    integer  :: boff
    logical, save :: debug_adjust_print = .true. ! true => print out tracer adjustment msgs

    ! frontogenesis function for gravity wave drag
    real(r8), allocatable :: frontgf(:,:,:)
    real(r8), pointer :: pbuf_frontgf(:,:)
    ! frontogenesis angle for gravity wave drag
    real(r8), allocatable :: frontga(:,:,:)
    real(r8), pointer :: pbuf_frontga(:,:)
    ! needed for qbo
    real(r8), pointer :: pbuf_uzm(:,:)

!--------------------------------------------
!  Variables needed for WACCM-X
!--------------------------------------------
    integer :: ixo, ixo2, ixh, ixh2, ixn, &  ! indices in state structure for O, O2, H, H2, and N mmr
               ixo2p,ixnop,ixn2p,&           ! indices in state structure for O2+, NO+, N2+ mmr
               ixop                          ! index in state structure for O+ mmr 
    integer :: sIndx                         ! index for getting ion mass
    real(r8) :: mmrSum_O_O2_H                ! Sum of mass mixing ratios for O, O2, and H
    real(r8), parameter :: mmrMin=1.e-20_r8  ! lower limit of o2, o, and h mixing ratios
    real(r8), parameter :: N2mmrMin=1.e-6_r8 ! lower limit of o2, o, and h mixing ratios

    integer,parameter :: nIons = 4     ! Number of ions needed from tracer or pbuf

    real(r8), allocatable :: h2ommr_blck(:, :, :)       ! H2O mmr on grid (i,j,k)
    real(r8), allocatable :: o2mmr_blck(:, :, :)        ! O2 mmr on grid (i,j,k)
    real(r8), allocatable :: o1mmr_blck(:, :, :)        ! O mmr on grid (i,j,k)
    real(r8), allocatable :: h1mmr_blck(:, :, :)        ! H mmr on grid (i,j,k)
    real(r8), allocatable :: n1mmr_blck(:, :, :)        ! N mmr on grid (i,j,k) 
       
    ! Dynamo output ion drift velocities, O+, and AMIE output in physics buffer:
    real(r8), pointer :: pbuf_ui(:,:), pbuf_vi(:,:), pbuf_wi(:,:)
    real(r8), pointer :: pbuf_op(:,:)     ! Pointer to access O+ in pbuf
    real(r8), pointer :: pbuf_amie_efxg(:)     ! Pointer to access AMIE energy flux in pbuf
    real(r8), pointer :: pbuf_amie_kevg(:)     ! Pointer to access AMIE mean energy in pbuf
    
    !--------------------------------------------
    
#if (! defined SPMD)
    integer  :: block_buf_nrecs = 0
    integer  :: chunk_buf_nrecs = 0
    logical  :: local_dp_map=.true. 
#endif
    type(physics_buffer_desc), pointer :: pbuf_chnk(:)

!---------------------------End Local workspace-------------------------

    if (use_gw_front .or. use_gw_front_igw) then

       allocate(frontgf(grid%ifirstxy:grid%ilastxy,plev,grid%jfirstxy:grid%jlastxy), stat=astat)
       if( astat /= 0 ) then
          write(iulog,*) 'd_p_coupling: failed to allocate frontgf; error = ',astat
          call endrun
       end if

       allocate(frontga(grid%ifirstxy:grid%ilastxy,plev,grid%jfirstxy:grid%jlastxy), stat=astat)
       if( astat /= 0 ) then
          write(iulog,*) 'd_p_coupling: failed to allocate frontga; error = ',astat
          call endrun
       end if

    end if

    nullify(pbuf_chnk)
    nullify(pbuf_frontgf)
    nullify(pbuf_frontga)
    nullify(pbuf_uzm)

    nullify(pbuf_ui)
    nullify(pbuf_vi)
    nullify(pbuf_wi)
    nullify(pbuf_op)
    nullify(pbuf_amie_efxg)
    nullify(pbuf_amie_kevg)

    fraction = 0.1_r8

    phisxy   => dyn_out%phis
    psxy     => dyn_out%ps
    u3sxy    => dyn_out%u3s
    v3sxy    => dyn_out%v3s
    ptxy     => dyn_out%pt
    tracer   => dyn_out%tracer

    omgaxy   => dyn_out%omga
    pexy     => dyn_out%pe
    pelnxy   => dyn_out%peln
    pkxy     => dyn_out%pk
    pkzxy    => dyn_out%pkz


!------------WACCM-X---------------
! Define pointers (i,j,k):
    opmmr_blck      => dyn_out%opmmr
    optm1_blck      => dyn_out%optm1
    sigma_ped_blck  => dyn_out%pedconduct
    sigma_hall_blck => dyn_out%hallconduct
    ti_blck         => dyn_out%iontemp
    te_blck         => dyn_out%electemp
    zi_blck         => dyn_out%zi
    phis_blck       => dyn_out%phis

    ui_blck         => dyn_out%ui
    vi_blck         => dyn_out%vi
    wi_blck         => dyn_out%wi

    im       = grid%im
    jm       = grid%jm
    km       = grid%km
    kmp1     = km + 1

    ifirstxy = grid%ifirstxy
    ilastxy  = grid%ilastxy
    jfirstxy = grid%jfirstxy
    jlastxy  = grid%jlastxy

    iam      = grid%iam
!-----------------------------------------------------------------------
! Transform dynamics staggered winds to physics grid (D=>A)
!-----------------------------------------------------------------------

    call t_startf ('d2a3dikj')
    allocate (u3(ifirstxy:ilastxy, km, jfirstxy:jlastxy))
    allocate (v3(ifirstxy:ilastxy, km, jfirstxy:jlastxy))

    if (waccmx_is('ionosphere')) then
      allocate (wuxy(ifirstxy:ilastxy, jfirstxy:jlastxy, km))
      allocate (wvxy(ifirstxy:ilastxy, jfirstxy:jlastxy, km))
    endif

    if (iam .lt. grid%npes_xy) then
       call d2a3dikj( grid, u3sxy,  v3sxy, u3, v3 )
    end if  ! (iam .lt. grid%npes_xy)

    call t_stopf  ('d2a3dikj')

    if ( do_circulation_diags ) then
       call t_startf('DP_CPLN_ctem')
       call ctem_diags( u3, v3, omgaxy, ptxy(:,jfirstxy:jlastxy,:), tracer(:,jfirstxy:jlastxy,:,1), &
                         psxy, pexy, grid )
       call t_stopf('DP_CPLN_ctem')
    endif
    if (use_gw_front .or. use_gw_front_igw) then
       call t_startf('DP_CPLN_gw_sources')
       call gws_src_fnct (u3,v3,ptxy,  tracer(:,jfirstxy:jlastxy,:,1), pexy, grid, frontgf, frontga)
       call t_stopf('DP_CPLN_gw_sources')
    end if
    if (qbo_use_forcing) then
       call zonal_mean_3D(grid, plev, u3, uzm)
    end if
   
    !-----------------------------------------------------------------------
    !  Get indices for neutrals to get mixing ratios from state%q and masses
    !-----------------------------------------------------------------------
    call cnst_get_ind('O2' ,ixo2,  abort=.true.)
    call cnst_get_ind('O'  ,ixo,   abort=.true.)
    call cnst_get_ind('H'  ,ixh,   abort=.true.)
    call cnst_get_ind('N'  ,ixn,   abort=.true.)

!------------------ WACCMX ----------------------

    if (waccmx_is('ionosphere')) then
      !
      ! Save u3,v3 winds on a-grid for dpie_coupling:
      do k=1,km
  	do j=jfirstxy,jlastxy
  	  do i=ifirstxy,ilastxy
  	    wuxy(i,j,k) = u3(i,k,j)
  	    wvxy(i,j,k) = v3(i,k,j)
  	  enddo
  	enddo
      enddo

      if (.not.allocated(IonRates_blck)) then
  	allocate(IonRates_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,km,nIonRates),stat=astat)
  	if (astat /= 0) then
  	  write(iulog,*) 'ERROR: d_p_coupling: failed to allocate IonRates_blck; error = ',astat
  	  call endrun
  	endif
  	IonRates_blck = 0._r8
      endif

      if (.not.allocated(ion_OpO2_blck)) then
  	allocate(ion_OpO2_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,km),stat=astat)
  	if (astat /= 0) then
  	  write(iulog,*) 'ERROR: d_p_coupling: failed to allocate ion_OpO2_blck; error = ',astat
  	  call endrun
  	endif
  	ion_OpO2_blck = 0._r8
      endif

      if (.not.allocated(ion_OpN2_blck)) then
  	allocate(ion_OpN2_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,km),stat=astat)
  	if (astat /= 0) then
  	  write(iulog,*) 'ERROR: d_p_coupling: failed to allocate ion_OpN2_blck; error = ',astat
  	  call endrun
  	endif
  	ion_OpN2_blck = 0._r8
      endif

      if (.not.allocated(o2pmmr_blck)) then
  	allocate(o2pmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,km),stat=astat)
  	if (astat /= 0) call endrun('d_p_coupling: failed to allocate o2pmmr_blck')
  	o2pmmr_blck = 0._r8
      endif

      if (.not.allocated(nopmmr_blck)) then
  	allocate(nopmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,km),stat=astat)
  	if (astat /= 0) call endrun('d_p_coupling: failed to allocate nopmmr_blck')
  	nopmmr_blck = 0._r8
      endif

      if (.not.allocated(n2pmmr_blck)) then
  	allocate(n2pmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,km),stat=astat)
  	if (astat /= 0) call endrun('d_p_coupling: failed to allocate n2pmmr_blck')
  	n2pmmr_blck = 0._r8
      endif 
      
      !--------------------------------
      !  Allocate neutrals from tracer
      !--------------------------------
      allocate (h2ommr_blck(ifirstxy:ilastxy, jfirstxy:jlastxy, km),stat=astat)
      if (astat /= 0) call endrun('d_p_coupling: failed to allocate h2ommr_blck')
      allocate (o2mmr_blck(ifirstxy:ilastxy, jfirstxy:jlastxy, km),stat=astat)
      if (astat /= 0) call endrun('d_p_coupling: failed to allocate o2mmr_blck')
      allocate (o1mmr_blck(ifirstxy:ilastxy, jfirstxy:jlastxy, km),stat=astat)
      if (astat /= 0) call endrun('d_p_coupling: failed to allocate o1mmr_blck')
      allocate (h1mmr_blck(ifirstxy:ilastxy, jfirstxy:jlastxy, km),stat=astat)
      if (astat /= 0) call endrun('d_p_coupling: failed to allocate h1mmr_blck')
      allocate (n1mmr_blck(ifirstxy:ilastxy, jfirstxy:jlastxy, km),stat=astat)       
      if (astat /= 0) call endrun('d_p_coupling: failed to allocate n1mmr_blck')
      
      !-------------------------------------------------------------------------------------------
      !  Set dpie_coupling input ions to those from pbuf or state%q.
      !-------------------------------------------------------------------------------------------
      call cnst_get_ind('O2p',ixo2p, abort=.false.)
      if (ixo2p > 0) then
  	o2pmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km) = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixo2p)
  	rMassO2p = cnst_mw(ixo2p)
      else
  	sIndxO2p  = slvd_index( 'O2p' )
  	if (sIndxO2p > 0) then
  	  sIndx = get_spc_ndx( 'O2p' )
  	  rmassO2p = adv_mass(sIndx)
  	else
  	  call endrun('dp_coupling: Cannot find state or pbuf index for O2p in ionos_tend')		    
  	endif
      endif
      call cnst_get_ind('NOp',ixnop, abort=.false.)
      if (ixnop > 0) then
  	nopmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km) = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixnop)
  	rMassNOp = cnst_mw(ixnop)
      else
  	sIndxNOp  = slvd_index( 'NOp' )
  	if (sIndxNOp > 0) then
  	  sIndx = get_spc_ndx( 'NOp' )
  	  rmassNOp = adv_mass(sIndx)
  	else
  	  call endrun('dp_coupling: Cannot find state or pbuf index for NOp in ionos_tend')		    
  	endif
      endif
      call cnst_get_ind('N2p',ixn2p, abort=.false.)
      if (ixn2p > 0) then
  	n2pmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km) = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixn2p)
  	rMassN2p = cnst_mw(ixn2p)
      else
  	sIndxN2p  = slvd_index( 'N2p' )
  	if (sIndxN2p > 0) then
  	  sIndx = get_spc_ndx( 'N2p' )
  	  rmassN2p = adv_mass(sIndx)
  	else
  	  call endrun('dp_coupling: Cannot find state or pbuf index for N2p in ionos_tend')		    
  	endif
      endif
      call cnst_get_ind('Op',ixop, abort=.false.)
      !
      ! O+ is not in tracer if we are doing only ion transport (not neutral transport)
      !
      if (ixop > 0) then
  	opmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km) = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixop)
  	rMassOp = cnst_mw(ixop)
      else
  	sIndxOp  = slvd_index( 'Op' )
  	if (sIndxOp > 0) then
  	  sIndx = get_spc_ndx( 'Op' )
  	  rmassOp = adv_mass(sIndx)
  	else
  	  call endrun('dp_coupling: Cannot find state or pbuf index for Op in ionos_tend')		    
  	endif
      endif

      if (.not.allocated(amie_efxg_blck)) then
  	allocate(amie_efxg_blck(ifirstxy:ilastxy,jfirstxy:jlastxy),stat=astat)
  	if (astat /= 0) then
  	  write(iulog,*) 'ERROR: d_p_coupling: failed to allocate amie_efxg_blck; error = ',astat
  	  call endrun
  	endif
  	amie_efxg_blck = 0._r8
      endif

      if (.not.allocated(amie_kevg_blck)) then
  	allocate(amie_kevg_blck(ifirstxy:ilastxy,jfirstxy:jlastxy),stat=astat)
  	if (astat /= 0) then
  	  write(iulog,*) 'ERROR: d_p_coupling: failed to allocate amie_kevg_blck; error = ',astat
  	  call endrun
  	endif
  	amie_kevg_blck = 0._r8
      endif

      !------------------------------------
      ! Get neutrals from state%q
      !------------------------------------
      h2ommr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km) = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,1)
      o2mmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km)  = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixo2)
      o1mmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km)  = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixo)
      h1mmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km)  = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixh)
      n1mmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km)  = tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixn)
 
      !------------------------------------
      ! Get neutral molecular weights
      !------------------------------------
      rmassO2 = cnst_mw(ixo2)
      rmassO1 = cnst_mw(ixo)
      rmassH  = cnst_mw(ixh)
      rmassN  = cnst_mw(ixn)
      rmassN2 = 28._r8
      
      !
      !	Call d_pie_coupling (dpie_coupling.F90)
      !	subroutine d_pie_coupling(omega,pe,z,u,v,pt,tracer,ionrates,sigma_ped,sigma_hall,i0,i1,j0,j1)
      !
!    if (.not. is_first_step() .and. .not. is_first_restart_step() .and.use_dynamo) then 
!      if (waccmx_is('ionosphere') .and. use_dynamo .and. .not. is_first_step()) then 
      if (use_dynamo .and. .not. is_first_step()) then 
        call t_startf('d_pie_coupling')
       !
       !   Make geopotential height (m) for d_pie_coupling.
       !   (zi_blck is geopotential height above surface on interfaces, and  was obtained via p_d_coupling)
        !
        do k=1,km
          do j=jfirstxy,jlastxy
            do i=ifirstxy,ilastxy
             zi_blck(i,j,k) = zi_blck(i,j,k)+phis_blck(i,j)/gravit ! phis_blck is redundant in k
            enddo
          enddo
        enddo
        !
        !   Most args are (i,j,k) blocked subdomains:
        !
        call d_pie_coupling(omgaxy,pexy,zi_blck,wuxy,wvxy,ptxy,IonRates_blck,ion_OpO2_blck,&
          ion_OpN2_blck,sigma_ped_blck,sigma_hall_blck,te_blck,ti_blck,h2ommr_blck,	 &
          o2mmr_blck,o1mmr_blck,h1mmr_blck,n1mmr_blck,o2pmmr_blck,nopmmr_blck,n2pmmr_blck, &
          opmmr_blck,optm1_blck,ui_blck,vi_blck,wi_blck,amie_efxg_blck, amie_kevg_blck,    &
          rmassO2,rmassO1,rmassH,rmassN,rmassN2,rmassO2p,rmassNOp,rmassN2p, rmassOp,	 &
          ifirstxy,ilastxy, jfirstxy,jlastxy)

        call t_stopf ('d_pie_coupling')

!
        !----------------------------------------
        !  Put data back in to state%q or pbuf
        !----------------------------------------
        if (ixop > 0) then
          tracer(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km,ixop) = opmmr_blck(ifirstxy:ilastxy,jfirstxy:jlastxy,1:km)           
        endif
	 
      endif !d_pie_coupling

    endif ! waccmx_is('ionosphere')

!-----------------------------------------------------------------------
! Copy data from dynamics data structure to physics data structure
!-----------------------------------------------------------------------
has_local_map : &
    if (local_dp_map) then

! This declaration is too long; this parallel section needs some stuff
! pulled out into routines.
!
!$omp parallel do private (lchnk, ncol, i, k, m, ic, jc, lons, lats, pic, pbuf_chnk, pbuf_uzm, pbuf_frontgf,pbuf_frontga,pbuf_ui,pbuf_vi,pbuf_wi,pbuf_op,pbuf_amie_efxg,pbuf_amie_kevg)
chnk_loop1 : &
       do lchnk = begchunk,endchunk
          ncol = phys_state(lchnk)%ncol
          call get_lon_all_p(lchnk, ncol, lons)
          call get_lat_all_p(lchnk, ncol, lats)

          pbuf_chnk => pbuf_get_chunk(pbuf2d, lchnk)

          if (use_gw_front .or. use_gw_front_igw) then
             call pbuf_get_field(pbuf_chnk, frontgf_idx, pbuf_frontgf)
             call pbuf_get_field(pbuf_chnk, frontga_idx, pbuf_frontga)
          end if

          if (waccmx_is('ionosphere')) then
            call pbuf_get_field(pbuf_chnk, ui_idx, pbuf_ui)
            call pbuf_get_field(pbuf_chnk, vi_idx, pbuf_vi)
            call pbuf_get_field(pbuf_chnk, wi_idx, pbuf_wi)
            if (sIndxOp > 0) &
              call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, pbuf_op, start=(/1,1,sIndxOp/), kount=(/pcols,pver,1/) )
          endif
          if (iamie == 1) then
	    call pbuf_get_field(pbuf_chnk, indxAMIEefxg, pbuf_amie_efxg)
	    if (is_first_step()) pbuf_amie_efxg = 0._r8
	    call pbuf_get_field(pbuf_chnk, indxAMIEkevg, pbuf_amie_kevg)
	    if (is_first_step()) pbuf_amie_kevg = 0._r8
	  endif

          if (qbo_use_forcing) then
             call pbuf_get_field(pbuf_chnk, uzm_idx, pbuf_uzm)
          end if

          do i=1,ncol
             ic = lons(i)
             jc = lats(i)
             phys_state(lchnk)%ps(i)   = psxy(ic,jc)
             phys_state(lchnk)%phis(i) = phisxy(ic,jc)
             pic(i) = pkxy(ic,jc,pver+1)
          enddo
          do k=1,km
             do i=1,ncol
                ic = lons(i)
                jc = lats(i)
                phys_state(lchnk)%u    (i,k) = u3(ic,k,jc)
                phys_state(lchnk)%v    (i,k) = v3(ic,k,jc)
                phys_state(lchnk)%omega(i,k) = omgaxy(ic,k,jc)
                phys_state(lchnk)%t    (i,k) = ptxy(ic,jc,k) / (D1_0 + zvir*tracer(ic,jc,k,1))
                phys_state(lchnk)%exner(i,k) = pic(i) / pkzxy(ic,jc,k)

                if (use_gw_front .or. use_gw_front_igw) then
                   pbuf_frontgf(i,k) = frontgf(ic,k,jc)
                   pbuf_frontga(i,k) = frontga(ic,k,jc)
                endif

! WACCM-X: define ion drift velocities in pbuf from blocked dynamo output:
                if (waccmx_is('ionosphere')) then
                  pbuf_ui(i,k) = ui_blck(ic,jc,k)
                  pbuf_vi(i,k) = vi_blck(ic,jc,k)
                  pbuf_wi(i,k) = wi_blck(ic,jc,k)
                  if (sIndxOp > 0) pbuf_op(i,k) = opmmr_blck(ic,jc,k)
                endif

                if (iamie == 1 .and. k == 1) then
		  pbuf_amie_efxg(i) = amie_efxg_blck(ic,jc)
		  pbuf_amie_kevg(i) = amie_kevg_blck(ic,jc)
		endif

                if (qbo_use_forcing) then
                   pbuf_uzm(i,k)     = uzm(k,jc)
                end if

             end do
          end do

          do k=1,kmp1
             do i=1,ncol
!
! edge-level pressure arrays: copy from the arrays computed by dynpkg
!
                ic = lons(i)
                jc = lats(i)
                phys_state(lchnk)%pint  (i,k) = pexy  (ic,k,jc)
                phys_state(lchnk)%lnpint(i,k) = pelnxy(ic,k,jc)
             end do
          end do

!
! Copy constituents
! Dry types converted from moist to dry m.r. at bottom of this routine
!
          do m=1,pcnst
             do k=1,km
                do i=1,ncol
                   phys_state(lchnk)%q(i,k,m) = &
                      tracer(lons(i),lats(i),k,m)
                end do
             end do
          end do
 
       end do chnk_loop1

    else has_local_map

       boff  = 6
       if (use_gw_front .or. use_gw_front_igw) boff  = boff+2
       if (qbo_use_forcing)  boff  = boff+1

       if (waccmx_is('ionosphere')) then
         if (sIndxOp > 0) then
           boff = boff+4 ! for ui,vi,wi,op
         else
           boff = boff+3 ! for ui,vi,wi
         endif
       endif

       tsize = boff + 1 + pcnst

       blksiz = (jlastxy-jfirstxy+1)*(ilastxy-ifirstxy+1)
       allocate( bpter(blksiz,0:km),stat=astat )
       if( astat /= 0 ) then
          write(iulog,*) 'd_p_coupling: failed to allocate bpter; error = ',astat
          call endrun
       end if
       allocate( bbuffer(tsize*block_buf_nrecs),stat=astat )
       write(6,"('bbuffer: tsize,block_buf_nrecs = ',2i5)") tsize,block_buf_nrecs 
       if( astat /= 0 ) then
          write(iulog,*) 'd_p_coupling: failed to allocate bbuffer; error = ',astat
          call endrun
       end if
       allocate( cbuffer(tsize*chunk_buf_nrecs),stat=astat )
       if( astat /= 0 ) then
          write(iulog,*) 'd_p_coupling: failed to allocate cbuffer; error = ',astat
          call endrun
       end if

       if (iam .lt. grid%npes_xy) then
          call block_to_chunk_send_pters(iam+1,blksiz,kmp1,tsize,bpter)
       endif

!$omp parallel do private (j, i, ib, k, m)
!dir$ concurrent
       do j=jfirstxy,jlastxy
!dir$ concurrent
          do i=ifirstxy,ilastxy
             ib = (j-jfirstxy)*(ilastxy-ifirstxy+1) + (i-ifirstxy+1)

             bbuffer(bpter(ib,0)+4:bpter(ib,0)+boff+pcnst) = 0.0_r8

             bbuffer(bpter(ib,0))   = pexy(i,kmp1,j)
             bbuffer(bpter(ib,0)+1) = pelnxy(i,kmp1,j)
             bbuffer(bpter(ib,0)+2) = psxy(i,j)
             bbuffer(bpter(ib,0)+3) = phisxy(i,j)

             if (iamie == 1) then 		  
               bbuffer(bpter(ib,0)+4) = amie_efxg_blck(i,j)
               bbuffer(bpter(ib,0)+5) = amie_kevg_blck(i,j)
             endif  

!dir$ concurrent
             do k=1,km

                bbuffer(bpter(ib,k))   = pexy(i,k,j)
                bbuffer(bpter(ib,k)+1) = pelnxy(i,k,j)
                bbuffer(bpter(ib,k)+2) = u3    (i,k,j)
                bbuffer(bpter(ib,k)+3) = v3    (i,k,j)
                bbuffer(bpter(ib,k)+4) = omgaxy(i,k,j)
                bbuffer(bpter(ib,k)+5) = ptxy(i,j,k) / (D1_0 + zvir*tracer(i,j,k,1))
                bbuffer(bpter(ib,k)+6) = pkxy(i,j,pver+1) / pkzxy(i,j,k) 

                if (use_gw_front .or. use_gw_front_igw) then
                   bbuffer(bpter(ib,k)+7) = frontgf(i,k,j)
                   bbuffer(bpter(ib,k)+8) = frontga(i,k,j)
                end if

                if (qbo_use_forcing) then
                   bbuffer(bpter(ib,k)+9) = uzm(k,j)
                end if
!
! WACCM-X: Dynamo ion drifts:
                if (waccmx_is('ionosphere')) then
                  bbuffer(bpter(ib,k)+10) = ui_blck(i,j,k)
                  bbuffer(bpter(ib,k)+11) = vi_blck(i,j,k)
                  bbuffer(bpter(ib,k)+12) = wi_blck(i,j,k)
                  if (sIndxOp > 0) bbuffer(bpter(ib,k)+13) = opmmr_blck(i,j,k)
                endif
		  
                do m=1,pcnst
                   bbuffer(bpter(ib,k)+boff+m) = tracer(i,j,k,m)
                end do

             end do
          end do
       end do

       call t_barrierf('sync_blk_to_chk', grid%commxy)
       call t_startf ('block_to_chunk')
       call transpose_block_to_chunk(tsize, bbuffer, cbuffer)
       call t_stopf  ('block_to_chunk')

!$omp parallel do private (lchnk, ncol, i, k, m, cpter, pbuf_chnk, pbuf_uzm, pbuf_frontgf, pbuf_frontga, pbuf_ui,pbuf_vi,pbuf_wi,pbuf_op,pbuf_amie_efxg,pbuf_amie_kevg)
chnk_loop2 : &
       do lchnk = begchunk,endchunk
          ncol = phys_state(lchnk)%ncol

          pbuf_chnk => pbuf_get_chunk(pbuf2d, lchnk)

          if (use_gw_front .or. use_gw_front_igw) then
             call pbuf_get_field(pbuf_chnk, frontgf_idx, pbuf_frontgf)
             call pbuf_get_field(pbuf_chnk, frontga_idx, pbuf_frontga)
          end if

          if (qbo_use_forcing) then
             call pbuf_get_field(pbuf_chnk, uzm_idx, pbuf_uzm)
          end if

          if (waccmx_is('ionosphere')) then
            call pbuf_get_field(pbuf_chnk, ui_idx, pbuf_ui)
            call pbuf_get_field(pbuf_chnk, vi_idx, pbuf_vi)
            call pbuf_get_field(pbuf_chnk, wi_idx, pbuf_wi)
            if (sIndxOp > 0) &
              call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, pbuf_op, start=(/1,1,sIndxOp/), kount=(/pcols,pver,1/) )
          endif

        if (iamie == 1) then
	    call pbuf_get_field(pbuf_chnk, indxAMIEefxg, pbuf_amie_efxg)
	    if (is_first_step()) pbuf_amie_efxg = 0._r8
	    call pbuf_get_field(pbuf_chnk, indxAMIEkevg, pbuf_amie_kevg)
	    if (is_first_step()) pbuf_amie_kevg = 0._r8
	  endif

          call block_to_chunk_recv_pters(lchnk,pcols,pver+1,tsize,cpter)

          do i=1,ncol

             phys_state(lchnk)%pint  (i,pver+1) = cbuffer(cpter(i,0))
             phys_state(lchnk)%lnpint(i,pver+1) = cbuffer(cpter(i,0)+1)
             phys_state(lchnk)%ps(i)            = cbuffer(cpter(i,0)+2)
             phys_state(lchnk)%phis(i)          = cbuffer(cpter(i,0)+3)

             if (iamie == 1) then
	       pbuf_amie_efxg(i) = cbuffer(cpter(i,0)+4)
	       pbuf_amie_kevg(i) = cbuffer(cpter(i,0)+5)
	     endif

             do k=1,km

                phys_state(lchnk)%pint  (i,k) = cbuffer(cpter(i,k))
                phys_state(lchnk)%lnpint(i,k) = cbuffer(cpter(i,k)+1)
                phys_state(lchnk)%u     (i,k) = cbuffer(cpter(i,k)+2)
                phys_state(lchnk)%v     (i,k) = cbuffer(cpter(i,k)+3)
                phys_state(lchnk)%omega (i,k) = cbuffer(cpter(i,k)+4)
                phys_state(lchnk)%t     (i,k) = cbuffer(cpter(i,k)+5)
                phys_state(lchnk)%exner (i,k) = cbuffer(cpter(i,k)+6)

                if (use_gw_front .or. use_gw_front_igw) then
                   pbuf_frontgf(i,k)  = cbuffer(cpter(i,k)+7)
                   pbuf_frontga(i,k)  = cbuffer(cpter(i,k)+8)
                end if

                if (qbo_use_forcing) then
                   pbuf_uzm(i,k) = cbuffer(cpter(i,k)+9)
                end if
!
! WACCM-X dynamo ion drifts in chunks:
                if (waccmx_is('ionosphere')) then
                  pbuf_ui(i,k) = cbuffer(cpter(i,k)+10)
                  pbuf_vi(i,k) = cbuffer(cpter(i,k)+11)
                  pbuf_wi(i,k) = cbuffer(cpter(i,k)+12)
                  if (sIndxOp > 0) pbuf_op(i,k) = cbuffer(cpter(i,k)+13)
                endif

                ! dry type constituents converted from moist to dry at bottom of routine
                do m=1,pcnst
                   phys_state(lchnk)%q(i,k,m) = cbuffer(cpter(i,k)+boff+m)
                end do 
             end do ! k=1,km
          end do ! i=1,ncol
 
          if (waccmx_is('ionosphere')) then
            call outfld ( 'UI', pbuf_ui, pcols, lchnk )
            call outfld ( 'VI', pbuf_vi, pcols, lchnk )
            call outfld ( 'WI', pbuf_wi, pcols, lchnk )
          endif

       end do chnk_loop2

       deallocate(bpter)
       deallocate(bbuffer)
       deallocate(cbuffer)

    endif has_local_map

!------------------------------------------------------
!  Get indices to access O, O2, H, H2, and N species
!------------------------------------------------------
    if ( waccmx_is('ionosphere') .or. waccmx_is('neutral') ) then 
      call cnst_get_ind('H2', ixh2, abort=.true.)
    endif
!
! Evaluate derived quantities
!
    call t_startf ('derived_fields')
!$omp parallel do private (lchnk, ncol, i, k, m, qmavl, dqreq, qbot, qbotm1, zvirv, pbuf_chnk, mmrSum_O_O2_H)
    do lchnk = begchunk,endchunk
       ncol = phys_state(lchnk)%ncol
       do k=1,km
          do i=1,ncol
             phys_state(lchnk)%pdel (i,k) = phys_state(lchnk)%pint(i,k+1) - phys_state(lchnk)%pint(i,k)
             phys_state(lchnk)%rpdel(i,k) = D1_0/phys_state(lchnk)%pdel(i,k)
             phys_state(lchnk)%pmid (i,k) = D0_5*(phys_state(lchnk)%pint(i,k) + phys_state(lchnk)%pint(i,k+1))
             phys_state(lchnk)%lnpmid(i,k) = log(phys_state(lchnk)%pmid(i,k))
          end do
       end do

! Attempt to remove negative constituents in bottom layer only by moving from next level
! This is a BAB kludge to avoid masses of warning messages for cloud water and ice, since
! the vertical remapping operator currently being used for cam is not strictly monotonic 
! at the endpoints.
       do m=1,pcnst
          do i=1,ncol
             if (phys_state(lchnk)%q(i,pver,m) < qmin(m)) then
! available q in 2nd level
                qmavl = phys_state(lchnk)%q (i,pver-1,m) - qmin(m)
! required q change in bottom level rescaled to mass fraction in 2nd level
                dqreq = (qmin(m) - phys_state(lchnk)%q(i,pver,m))                         &
                      * phys_state(lchnk)%pdel(i,pver) / phys_state(lchnk)%pdel(i,pver-1)
                qbot   = phys_state(lchnk)%q(i,pver  ,m)
                qbotm1 = phys_state(lchnk)%q(i,pver-1,m)
                if (dqreq < qmavl) then
                   phys_state(lchnk)%q(i,pver  ,m) = qmin(m)
                   phys_state(lchnk)%q(i,pver-1,m) = phys_state(lchnk)%q(i,pver-1,m) - dqreq
                   ! Comment out these log messages since they can make the log files so
                   ! large that they're unusable.
                   !if (dqreq>1.e-14_r8 .and. debug_adjust_print) write(iulog,*) 'dpcoup dqreq', m, lchnk, i, qbot, qbotm1, dqreq
                    if (dqreq>qmin(m) .and. dqreq>fraction*qbotm1 .and. debug_adjust_print) &
                                                                write(iulog,*) 'dpcoup dqreq', m, lchnk, i, qbot, qbotm1, dqreq
                else 
                   ! Comment out these log messages since they can make the log files so
                   ! large that they're unusable.
                   !if (debug_adjust_print) write(iulog,*) 'dpcoup cant adjust', m, lchnk, i, qbot, qbotm1, dqreq
                    if (dqreq>qmin(m) .and. debug_adjust_print) write(iulog,*) 'dpcoup cant adjust', m, lchnk, i, &
                         qbot, qbotm1, dqreq
                end if
             end if
          end do
       end do

!-----------------------------------------------------------------------------------------------------------------
! Ensure O2 + O + H (N2) mmr greater than one.  Check for unusually large H2 values and set to lower value
!-----------------------------------------------------------------------------------------------------------------
       if ( waccmx_is('ionosphere') .or. waccmx_is('neutral') ) then
          do i=1,ncol
             do k=1,pver

                if (phys_state(lchnk)%q(i,k,ixo) < mmrMin) phys_state(lchnk)%q(i,k,ixo) = mmrMin
                if (phys_state(lchnk)%q(i,k,ixo2) < mmrMin) phys_state(lchnk)%q(i,k,ixo2) = mmrMin

                mmrSum_O_O2_H = phys_state(lchnk)%q(i,k,ixo)+phys_state(lchnk)%q(i,k,ixo2)+phys_state(lchnk)%q(i,k,ixh)

                if ((1._r8-mmrMin-mmrSum_O_O2_H) < 0._r8) then
                   phys_state(lchnk)%q(i,k,ixo) = phys_state(lchnk)%q(i,k,ixo) * (1._r8 - N2mmrMin) / mmrSum_O_O2_H
                   phys_state(lchnk)%q(i,k,ixo2) = phys_state(lchnk)%q(i,k,ixo2) * (1._r8 - N2mmrMin) / mmrSum_O_O2_H
                   phys_state(lchnk)%q(i,k,ixh) = phys_state(lchnk)%q(i,k,ixh) * (1._r8 - N2mmrMin) / mmrSum_O_O2_H
                endif

                if(phys_state(lchnk)%q(i,k,ixh2) .gt. 6.e-5_r8) then
                   phys_state(lchnk)%q(i,k,ixh2) = 6.e-5_r8
                endif
	     
             end do
          end do
       endif

!-----------------------------------------------------------------------------
! Call physconst_update to compute cpairv, rairv, mbarv, and cappav as constituent dependent variables
! and compute molecular viscosity(kmvis) and conductivity(kmcnd) 
!-----------------------------------------------------------------------------
       if ( waccmx_is('ionosphere') .or. waccmx_is('neutral') ) then
         call physconst_update(phys_state(lchnk)%q, phys_state(lchnk)%t, &
                                cnst_mw(ixo), cnst_mw(ixo2), cnst_mw(ixh), cnst_mw(ixn), &
                                                           ixo, ixo2, ixh, pcnst, lchnk, ncol)
       endif

!------------------------------------------------------------------------
! Fill local zvirv variable; calculated for WACCM-X
!------------------------------------------------------------------------
       if ( waccmx_is('ionosphere') .or. waccmx_is('neutral') ) then 
         zvirv(:,:) = shr_const_rwv / rairv(:,:,lchnk) -1._r8
       else
         zvirv(:,:) = zvir    
       endif
!
! Compute initial geopotential heights
       call geopotential_t (phys_state(lchnk)%lnpint, phys_state(lchnk)%lnpmid  , phys_state(lchnk)%pint  , &
                            phys_state(lchnk)%pmid  , phys_state(lchnk)%pdel    , phys_state(lchnk)%rpdel , &
                            phys_state(lchnk)%t     , phys_state(lchnk)%q(:,:,1), rairv(:,:,lchnk), gravit, zvirv, &
                            phys_state(lchnk)%zi    , phys_state(lchnk)%zm      , ncol                )
!
! Compute initial dry static energy, include surface geopotential
       do k = 1, pver
          do i=1,ncol
             phys_state(lchnk)%s(i,k) = cpairv(i,k,lchnk)*phys_state(lchnk)%t(i,k) &
                                      + gravit*phys_state(lchnk)%zm(i,k) + phys_state(lchnk)%phis(i)
          end do
       end do

!
! Convert dry type constituents from moist to dry mixing ratio
!
       call set_state_pdry(phys_state(lchnk))    ! First get dry pressure to use for this timestep
       call set_wet_to_dry(phys_state(lchnk))    ! Dynamics had moist, physics wants dry.
!
! Ensure tracers are all positive
!
       call qneg3('D_P_COUPLING',lchnk  ,ncol    ,pcols   ,pver    , &
                  1, pcnst, qmin  ,phys_state(lchnk)%q)

! Compute energy and water integrals of input state

       pbuf_chnk => pbuf_get_chunk(pbuf2d, lchnk)
       call check_energy_timestep_init(phys_state(lchnk), phys_tend(lchnk), pbuf_chnk)

    end do
    call t_stopf('derived_fields')

    deallocate (u3)
    deallocate (v3)

    if (waccmx_is('ionosphere')) then
      deallocate (wuxy)
      deallocate (wvxy)
      deallocate (h2ommr_blck)
      deallocate (o2mmr_blck)
      deallocate (o1mmr_blck)
      deallocate (h1mmr_blck)
      deallocate (n1mmr_blck)
    endif
    
!EOC
  end subroutine d_p_coupling
!-----------------------------------------------------------------------

!-----------------------------------------------------------------------
!BOP
! !IROUTINE: p_d_coupling --- convert physics output to dynamics input
!
! !INTERFACE: 
  subroutine p_d_coupling(grid, phys_state, phys_tend, pbuf2d, &
                          dyn_in, dtime, zvir, cappa, ptop)

! !USES:
#if ( defined OFFLINE_DYN )
   use metdata,     only: get_met_fields
#endif
 
!-----------------------------------------------------------------------
    implicit none

! Variables ending in xy are xy-decomposition instanciations.

    type(T_FVDYCORE_GRID), intent(in) :: grid ! FV Dynamics grid

! !INPUT PARAMETERS:
    type(physics_state), intent(inout), dimension(begchunk:endchunk) :: phys_state
    type(physics_tend),  intent(inout), dimension(begchunk:endchunk) :: phys_tend
    type(dyn_import_t),  intent(inout)   :: dyn_in
    type(physics_buffer_desc), pointer   :: pbuf2d(:,:)

    real(r8), intent(in) :: dtime
    real(r8), intent(in) :: zvir
    real(r8), intent(in) :: cappa
    real(r8), intent(in) :: ptop

! !DESCRIPTION:
!
!   Coupler for converting physics output variables into dynamics input variables
!
! !REVISION HISTORY:
!   00.06.01   Boville    Creation
!   01.06.08   AAM        Compactified
!   01.07.13   AAM        Some support for multi-2D decompositions
!   02.03.01   Worley     Support for nontrivial physics remapping
!   02.08.06   Sawyer     T3 added -- updated to current temperature
!   05.07.12   Sawyer     Added dyn_state as argument
!   05.09.23   Sawyer     Transitioned to XY decomposition vars. only
!   05.10.31   Sawyer     Replaced dyn_state with dyn_interface
!
!EOP
!-----------------------------------------------------------------------
!BOC
! !LOCAL VARIABLES:

! Variables from the dynamics import container

    real(r8), pointer :: psxy(:,:)
    real(r8), pointer :: u3sxy(:,:,:)
    real(r8), pointer :: v3sxy(:,:,:)
    real(r8), pointer :: t3xy(:,:,:)                  !  Temperature
    real(r8), pointer :: ptxy(:,:,:)                  !  Virt. pot. temp.
    real(r8), pointer :: tracer(:,:,:,:)              !  Constituents

    real(r8), pointer :: pexy(:,:,:)
    real(r8), pointer :: delpxy(:,:,:)
    real(r8), pointer :: pkxy(:,:,:)
    real(r8), pointer :: pkzxy(:,:,:)

! Local workspace

    real(r8):: dudtxy(grid%ifirstxy:grid%ilastxy,&
                      grid%km,grid%jfirstxy:grid%jlastxy)
    real(r8):: dvdtxy(grid%ifirstxy:grid%ilastxy,&
                      grid%km,grid%jfirstxy:grid%jlastxy)
    real(r8):: dummy_pelnxy(grid%ifirstxy:grid%ilastxy,grid%km+1, &
                            grid%jfirstxy:grid%jlastxy)

    integer :: i, ib, k, m, j, lchnk  ! indices
    integer :: ncol                   ! number of columns in current chunk
    integer :: lats(pcols)            ! array of latitude indices
    integer :: lons(pcols)            ! array of longitude indices
    integer :: blksiz                 ! number of columns in 2D block
    integer :: tsize                  ! amount of data per grid point passed to physics
    integer, allocatable, dimension(:,:) :: bpter
                                     ! offsets into block buffer for unpacking data
    integer :: cpter(pcols,0:pver)   ! offsets into chunk buffer for packing data
    integer :: iqa, iqb, iqc, iqd, mq     ! used for tracer transpose grouping

    real(r8) :: dt5
    real(r8), allocatable, dimension(:) :: &
       bbuffer, cbuffer               ! transpose buffers
#if (! defined SPMD)
    integer  :: block_buf_nrecs = 0
    integer  :: chunk_buf_nrecs = 0
    logical  :: local_dp_map=.true. 
#endif
    integer  :: im, jm, km, ng_d, ng_s, iam
    integer  :: ifirstxy, ilastxy, jfirstxy, jlastxy 
    integer  :: jfirst, jlast, kfirst, klast

!----------WACCM-X-----------
    type(physics_buffer_desc), pointer :: pbuf_chnk(:)
    real(r8),pointer :: IonRates_phys(:,:,:) ! ionization rates from pbuf
    real(r8),pointer :: ion_OpO2_phys(:,:)   ! Op+O2 rate from pbuf
    real(r8),pointer :: ion_OpN2_phys(:,:)   ! Op+N2 rate from pbuf
    real(r8),pointer :: sigma_ped_phys(:,:)  ! physics pointer to Pedersen Conductivity
    real(r8),pointer :: sigma_hall_phys(:,:) ! physics pointer fo Hall Conductivity
    real(r8),pointer :: te_phys(:,:)         ! te from pbuf
    real(r8),pointer :: ti_phys(:,:)         ! ti from pbuf
    real(r8),pointer :: mmrPO2p_phys(:,:)    ! Pointer to access O2+ in pbuf
    real(r8),pointer :: mmrPNOp_phys(:,:)    ! Pointer to access NO+ in pbuf
    real(r8),pointer :: mmrPN2p_phys(:,:)    ! Pointer to access N2+ in pbuf
    real(r8),pointer :: mmrPOp_phys(:,:)     ! Pointer to access O+ in pbuf
!
! Empirical ion drifts from exbdrift (to be converted to blocked for dpie_coupling):
    real(r8),pointer :: ui_phys(:,:)         ! zonal ion drift from pbuf
    real(r8),pointer :: vi_phys(:,:)         ! meridional ion drift from pbuf
    real(r8),pointer :: wi_phys(:,:)         ! vertical ion drift from pbuf
!
! Empirical ion drifts converted to blocked subdomains:
    real(r8),pointer :: ui_blck(:,:,:)       ! zonal ion drift (redundant in vertical)
    real(r8),pointer :: vi_blck(:,:,:)       ! meridional ion drift (redundant in vertical)
    real(r8),pointer :: wi_blck(:,:,:)       ! vertical ion drift (redundant in vertical)
!
    real(r8),pointer :: opmmr_blck(:,:,:)      ! O+ (blocks)
    real(r8),pointer :: sigma_ped_blck(:,:,:)  ! Pederson Conductivity (blocks)
    real(r8),pointer :: sigma_hall_blck(:,:,:) ! Hall Conductivity (blocks)
    real(r8),pointer :: ti_blck(:,:,:)         ! Ion temp (blocks)
    real(r8),pointer :: te_blck(:,:,:)         ! Elec temp (blocks)
    real(r8),pointer :: zi_blck(:,:,:)         ! zi (blocks) (geopoten ht above surface (interfaces))
    real(r8),pointer :: phis_blck(:,:)         ! surface geopotential (blocks) (no vertical dimension)

    real(r8) :: zi_phys(pcols,pverp)
    real(r8) :: phis_phys(pcols)

    integer :: index_ped,index_hall,index_OpO2,index_OpN2,index_te,index_ti,&
               index_ui,index_vi,index_wi
    integer :: nSIons                        ! number of ions set to non-advected
    integer :: ibuffOp,ibuffO2p,ibuffNOp, ibuffN2p ! Buffer indices for non-advected ions

!---------------------------End Local workspace-------------------------

!----------WACCM-X-----------
    if (waccmx_is('ionosphere')) then
      nullify(pbuf_chnk)
      nullify(IonRates_phys)
      nullify(sigma_ped_phys)
      nullify(sigma_hall_phys)

      index_ped  = pbuf_get_index('PedConduct')
      index_hall = pbuf_get_index('HallConduct')

      index_OpO2 = pbuf_get_index('rxtOpO2')
      index_OpN2 = pbuf_get_index('rxtOpN2')

      index_te   = pbuf_get_index('ElecTemp')
      index_ti   = pbuf_get_index('IonTemp')
!
! pbuf indices to empirical ion drifts, to be passed to oplus_xport,
! if use_dynamo_drifts is false.
!
    index_ui   = pbuf_get_index('UI')
    index_vi   = pbuf_get_index('VI')
    index_wi   = pbuf_get_index('WI')

      sIndxOp  = slvd_index( 'Op' )
      sIndxO2p = slvd_index( 'O2p' )
      sIndxNOp = slvd_index( 'NOp' )
      sIndxN2p = slvd_index( 'N2p' )
    endif

!-----------------------------

! Pull the variables out of the dynamics export container

    psxy    => dyn_in%ps
    u3sxy   => dyn_in%u3s
    v3sxy   => dyn_in%v3s
    t3xy    => dyn_in%t3
    ptxy    => dyn_in%pt
    tracer  => dyn_in%tracer

    pexy    => dyn_in%pe
    delpxy  => dyn_in%delp
    pkxy    => dyn_in%pk
    pkzxy   => dyn_in%pkz    

!-------WACCM-X---------
    opmmr_blck      => dyn_in%opmmr
    sigma_ped_blck  => dyn_in%pedconduct
    sigma_hall_blck => dyn_in%hallconduct
    ti_blck         => dyn_in%iontemp
    te_blck         => dyn_in%electemp
    zi_blck         => dyn_in%zi
    phis_blck       => dyn_in%phis

! Empirical ion drifts are 3d arrays, but redundant in vertical.
    ui_blck         => dyn_in%ui
    vi_blck         => dyn_in%vi
    wi_blck         => dyn_in%wi

    im   = grid%im
    jm   = grid%jm
    km   = grid%km

    ifirstxy = grid%ifirstxy
    ilastxy  = grid%ilastxy
    jfirstxy = grid%jfirstxy
    jlastxy  = grid%jlastxy

    jfirst   = grid%jfirst
    jlast    = grid%jlast
    kfirst   = grid%kfirst
    klast    = grid%klast

    ng_d     = grid%ng_d
    ng_s     = grid%ng_s

    iam      = grid%iam

!---------------------------End Local workspace-------------------------

#if ( defined OFFLINE_DYN )
!
! set the dyn flds to offline meteorological data
!
      call get_met_fields( phys_state, phys_tend, dtime )
#endif
! -------------------------------------------------------------------------
! Copy temperature, tendencies and constituents to dynamics data structures
! -------------------------------------------------------------------------

! -------------------------------------------------------------------------
! Copy onto xy decomposition, then transpose to yz decomposition
! -------------------------------------------------------------------------

       if (local_dp_map) then

!$omp parallel do private(lchnk, i, k, ncol, m, lons, lats, pbuf_chnk, IonRates_phys, sigma_ped_phys, sigma_hall_phys, ion_OpO2_phys, ion_OpN2_phys, te_phys, ti_phys, mmrPO2p_phys, mmrPNOp_phys, mmrPN2p_phys, mmrPOp_phys, ui_phys,vi_phys,wi_phys)

          do lchnk = begchunk,endchunk
             ncol = get_ncols_p(lchnk)
             call get_lon_all_p(lchnk, ncol, lons)
             call get_lat_all_p(lchnk, ncol, lats)

             do k = 1, km
                do i = 1, ncol
                   dvdtxy(lons(i),k,lats(i)) = phys_tend(lchnk)%dvdt(i,k)
                   dudtxy(lons(i),k,lats(i)) = phys_tend(lchnk)%dudt(i,k)
                   ptxy  (lons(i),lats(i),k) = phys_state(lchnk)%t(i,k)
                   delpxy(lons(i),lats(i),k) = phys_state(lchnk)%pdel(i,k)
                enddo
             enddo

             do m=1,pcnst
                do k=1,km
                   do i=1,ncol
                      tracer(lons(i),lats(i),k,m) = &
                         phys_state(lchnk)%q(i,k,m)
                   end do
                end do
             end do

!------WACCM-X: Photo electron production rate--------
             if (waccmx_is('ionosphere')) then
             pbuf_chnk => pbuf_get_chunk(pbuf2d, lchnk)
!
! Get ion rates from physics buffer:
             call pbuf_get_field(pbuf_chnk, ion_rates_idx, IonRates_phys)
             do m=1,nIonRates
               do k=1,km
                 do i=1,ncol
                   IonRates_blck(lons(i),lats(i),k,m) = IonRates_phys(i,k,m)
                 end do
               enddo
             enddo
!
! Get Pedersen and Hall conductivities:
             call pbuf_get_field(pbuf_chnk, index_ped,  sigma_ped_phys)
             call pbuf_get_field(pbuf_chnk, index_hall, sigma_hall_phys)
             do k=1,km
               do i=1,ncol
                 sigma_ped_blck(lons(i),lats(i),k) = sigma_ped_phys(i,k)
                 sigma_hall_blck(lons(i),lats(i),k) = sigma_hall_phys(i,k)
               end do
             enddo
!
! Get O+ loss rates Op+O2 and Op+N2:
             call pbuf_get_field(pbuf_chnk, index_OpO2,  ion_OpO2_phys)
             call pbuf_get_field(pbuf_chnk, index_OpN2,  ion_OpN2_phys)
             do k=1,km
               do i=1,ncol
                 ion_OpO2_blck(lons(i),lats(i),k) = ion_OpO2_phys(i,k)
                 ion_OpN2_blck(lons(i),lats(i),k) = ion_OpN2_phys(i,k)
               end do
             enddo
!
! Get te,ti:
             call pbuf_get_field(pbuf_chnk, index_te, te_phys)
             call pbuf_get_field(pbuf_chnk, index_ti, ti_phys)
             do k=1,km
               do i=1,ncol
                 te_blck(lons(i),lats(i),k) = te_phys(i,k)
                 ti_blck(lons(i),lats(i),k) = ti_phys(i,k)
               end do
             enddo

             call pbuf_get_field(pbuf_chnk, index_ui, ui_phys)
             call pbuf_get_field(pbuf_chnk, index_vi, vi_phys)
             call pbuf_get_field(pbuf_chnk, index_wi, wi_phys)
             do k=1,km
               do i=1,ncol
                 ui_blck(lons(i),lats(i),k) = ui_phys(i,k)
                 vi_blck(lons(i),lats(i),k) = vi_phys(i,k)
                 wi_blck(lons(i),lats(i),k) = wi_phys(i,k)
               enddo
             enddo
 
             !--------------------------------------------------------
             ! Get ions from physics buffer if non-advected
             !--------------------------------------------------------
             if (sIndxO2p > 0) then
               call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPO2p_phys, start=(/1,1,sIndxO2p/), kount=(/pcols,pver,1/) )
               do k=1,km
                 do i=1,ncol
                   o2pmmr_blck(lons(i),lats(i),k) = mmrPO2p_phys(i,k)
                 end do
               enddo
             endif
             if (sIndxNOp > 0) then
               call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPNOp_phys, start=(/1,1,sIndxNOp/), kount=(/pcols,pver,1/) )
               do k=1,km
                 do i=1,ncol
                   nopmmr_blck(lons(i),lats(i),k) = mmrPNOp_phys(i,k)
                 end do
               enddo
             endif
             if (sIndxN2p > 0) then
               call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPN2p_phys, start=(/1,1,sIndxN2p/), kount=(/pcols,pver,1/) )
               do k=1,km
                 do i=1,ncol
                   n2pmmr_blck(lons(i),lats(i),k) = mmrPN2p_phys(i,k)
                 end do
               enddo
             endif
             if (sIndxOp > 0) then
               call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPOp_phys, start=(/1,1,sIndxOp/), kount=(/pcols,pver,1/) )
               do k=1,km
                 do i=1,ncol
                   opmmr_blck(lons(i),lats(i),k) = mmrPOp_phys(i,k)
                 end do
               enddo
             endif
             endif !waccmx_is('ionosphere')	     
!-----------------------------------------------------

          enddo ! do lchnk = begchunk,endchunk

       else ! local_dp_map == false

!--------WACCM-X--------------
          if (waccmx_is('ionosphere')) then

            tsize = 15 + pcnst + nIonRates

            nSIons = 0
            if (sIndxOp > 0)  then 
  	      ibuffOp = tsize + nSIons
  	      nSIons = nSIons + 1
            endif
            if (sIndxO2p > 0) then
	      ibuffO2p = tsize + nSIons
	      nSIons = nSIons + 1
	    endif
            if (sIndxNOp > 0) then
	      ibuffNOp = tsize + nSIons
	      nSIons = nSIons + 1
            endif	  
            if (sIndxN2p > 0) then
	      ibuffN2p = tsize + nSIons
	      nSIons = nSIons + 1
            endif	 
	    tsize = tsize + nSIons
          else
            tsize = 4 + pcnst
          endif !waccmx_is('ionosphere')
!-----------------------------

          blksiz = (jlastxy-jfirstxy+1)*(ilastxy-ifirstxy+1)
          allocate(bpter(blksiz,0:km))
          allocate(bbuffer(tsize*block_buf_nrecs))
          allocate(cbuffer(tsize*chunk_buf_nrecs))

!$omp parallel do private (lchnk, ncol, i, k, m, cpter, pbuf_chnk, IonRates_phys, sigma_ped_phys, sigma_hall_phys, ion_OpO2_phys, ion_OpN2_phys, te_phys, ti_phys, mmrPO2p_phys, mmrPNOp_phys, mmrPN2p_phys, mmrPOp_phys, zi_phys, phis_phys, ui_phys, vi_phys, wi_phys)
          do lchnk = begchunk,endchunk
             ncol = get_ncols_p(lchnk)

             if (waccmx_is('ionosphere')) then

             pbuf_chnk => pbuf_get_chunk(pbuf2d, lchnk)

!---------WACCM-X------------
!
! Get ion rates from physics buffer:
             call pbuf_get_field(pbuf_chnk, ion_rates_idx, IonRates_phys)
!
! Get Pedersen and Hall conductivities:
             call pbuf_get_field(pbuf_chnk, index_ped,  sigma_ped_phys)
             call pbuf_get_field(pbuf_chnk, index_hall, sigma_hall_phys)
!
! Get O+ loss rates Op+O2 and Op+N2:
             call pbuf_get_field(pbuf_chnk, index_OpO2,  ion_OpO2_phys)
             call pbuf_get_field(pbuf_chnk, index_OpN2,  ion_OpN2_phys)
!
! Get te,ti:
             call pbuf_get_field(pbuf_chnk, index_te,  te_phys)
             call pbuf_get_field(pbuf_chnk, index_ti,  ti_phys)
!
! Get 2d empirical or edynamo 3d ion drifts:
             call pbuf_get_field(pbuf_chnk, index_ui,  ui_phys)
             call pbuf_get_field(pbuf_chnk, index_vi,  vi_phys)
             call pbuf_get_field(pbuf_chnk, index_wi,  wi_phys)
!
! Get geopotential height on interfaces and surface geopotential from physics state:
             zi_phys(1:ncol,:) = phys_state(lchnk)%zi(1:ncol,:)
             phis_phys(1:ncol) = phys_state(lchnk)%phis(1:ncol) ! surface geopotential
 
             !--------------------------------------------------------
             ! Get ions from physics buffer if non-advected
             !--------------------------------------------------------

             if (sIndxOp > 0)  call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPOp_phys,  &
               start=(/1,1,sIndxOp/), kount=(/pcols,pver,1/) )
             if (sIndxO2p > 0) call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPO2p_phys, &
               start=(/1,1,sIndxO2p/), kount=(/pcols,pver,1/) )	     
             if (sIndxNOp > 0) call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPNOp_phys, &
               start=(/1,1,sIndxNOp/), kount=(/pcols,pver,1/) )	     
             if (sIndxN2p > 0) call pbuf_get_field(pbuf_chnk, slvd_pbf_ndx, mmrPN2p_phys, &
               start=(/1,1,sIndxN2p/), kount=(/pcols,pver,1/) )
        
	     endif
!----------------------------

             call chunk_to_block_send_pters(lchnk,pcols,km+1,tsize,cpter)

!---------WACCM-X------------
!
! Zero c-buffer:
             do i=1,ncol
               cbuffer(cpter(i,0):cpter(i,0)+tsize-1) = 0.0_r8
             enddo
!----------------------------

!dir$ concurrent
             do k=1,km
!dir$ concurrent
                do i=1,ncol

                   cbuffer(cpter(i,k))   = phys_tend(lchnk)%dvdt(i,k)
                   cbuffer(cpter(i,k)+1) = phys_tend(lchnk)%dudt(i,k)
                   cbuffer(cpter(i,k)+2) = phys_state(lchnk)%t(i,k)
                   cbuffer(cpter(i,k)+3) = phys_state(lchnk)%pdel(i,k)

                   do m=1,pcnst
                      cbuffer(cpter(i,k)+3+m) = phys_state(lchnk)%q(i,k,m)
                   end do

!-----------WACCM-X----------
                   if (waccmx_is('ionosphere')) then
                     do m=1,nIonRates
                       cbuffer(cpter(i,k)+3+pcnst+m) = IonRates_phys(i,k,m)
                     enddo
                     cbuffer(cpter(i,k)+4+pcnst+nIonRates) = sigma_ped_phys(i,k)
                     cbuffer(cpter(i,k)+5+pcnst+nIonRates) = sigma_hall_phys(i,k)

                     cbuffer(cpter(i,k)+6+pcnst+nIonRates) = ion_OpO2_phys(i,k)
                     cbuffer(cpter(i,k)+7+pcnst+nIonRates) = ion_OpN2_phys(i,k)
                     cbuffer(cpter(i,k)+8+pcnst+nIonRates) = te_phys(i,k)
                     cbuffer(cpter(i,k)+9+pcnst+nIonRates) = ti_phys(i,k)
                     cbuffer(cpter(i,k)+10+pcnst+nIonRates) = zi_phys(i,k)
                     cbuffer(cpter(i,k)+11+pcnst+nIonRates) = phis_phys(i) ! redundant in k

                     cbuffer(cpter(i,k)+12+pcnst+nIonRates) = ui_phys(i,k)
                     cbuffer(cpter(i,k)+13+pcnst+nIonRates) = vi_phys(i,k)
                     cbuffer(cpter(i,k)+14+pcnst+nIonRates) = wi_phys(i,k)
		   
		     if (sIndxO2p > 0)cbuffer(cpter(i,k)+ibuffO2p) = mmrPO2p_phys(i,k)
		     if (sIndxNOp > 0)cbuffer(cpter(i,k)+ibuffNOp) = mmrPNOp_phys(i,k)
		     if (sIndxN2p > 0)cbuffer(cpter(i,k)+ibuffN2p) = mmrPN2p_phys(i,k)
		     if (sIndxOp > 0) cbuffer(cpter(i,k)+ibuffOp) = mmrPOp_phys(i,k)
                   endif

                end do
             end do
          end do

          call t_barrierf('sync_chk_to_blk', grid%commxy)
          call t_startf ('chunk_to_block')
          call transpose_chunk_to_block(tsize, cbuffer, bbuffer)
          call t_stopf  ('chunk_to_block')

          if (iam .lt. grid%npes_xy) then
             call chunk_to_block_recv_pters(iam+1,blksiz,km+1,tsize,bpter)
          endif

!$omp parallel do private (j, i, ib, k, m)
!dir$ concurrent
          do j=jfirstxy,jlastxy
!dir$ concurrent
             do k=1,km
!dir$ concurrent
                do i=ifirstxy,ilastxy
                   ib = (j-jfirstxy)*(ilastxy-ifirstxy+1) + (i-ifirstxy+1)

                   dvdtxy(i,k,j) = bbuffer(bpter(ib,k))
                   dudtxy(i,k,j) = bbuffer(bpter(ib,k)+1)
                   ptxy  (i,j,k) = bbuffer(bpter(ib,k)+2)
                   delpxy(i,j,k) = bbuffer(bpter(ib,k)+3)

                   do m=1,pcnst
                      tracer(i,j,k,m) = bbuffer(bpter(ib,k)+3+m)
                   end do

!----------WACCM-X-------------
                   if (waccmx_is('ionosphere')) then
                     do m=1,nIonRates
                       IonRates_blck(i,j,k,m) = bbuffer(bpter(ib,k)+3+pcnst+m)
                     enddo
                     sigma_ped_blck(i,j,k)  = bbuffer(bpter(ib,k)+4+pcnst+nIonRates)
                     sigma_hall_blck(i,j,k) = bbuffer(bpter(ib,k)+5+pcnst+nIonRates)

                     ion_OpO2_blck(i,j,k) = bbuffer(bpter(ib,k)+6+pcnst+nIonRates)
                     ion_OpN2_blck(i,j,k) = bbuffer(bpter(ib,k)+7+pcnst+nIonRates)

                     te_blck(i,j,k) = bbuffer(bpter(ib,k)+8+pcnst+nIonRates)
                     ti_blck(i,j,k) = bbuffer(bpter(ib,k)+9+pcnst+nIonRates)

                   zi_blck(i,j,k) = bbuffer(bpter(ib,k)+10+pcnst+nIonRates)
                   phis_blck(i,j) = bbuffer(bpter(ib,k)+11+pcnst+nIonRates)

                   ui_blck(i,j,k) = bbuffer(bpter(ib,k)+12+pcnst+nIonRates)
                   vi_blck(i,j,k) = bbuffer(bpter(ib,k)+13+pcnst+nIonRates)
                   wi_blck(i,j,k) = bbuffer(bpter(ib,k)+14+pcnst+nIonRates)
		   
		     if (sIndxO2p > 0) o2pmmr_blck(i,j,k) = bbuffer(bpter(ib,k)+ibuffO2p)
		     if (sIndxNOp > 0) nopmmr_blck(i,j,k) = bbuffer(bpter(ib,k)+ibuffNOp)
		     if (sIndxN2p > 0) n2pmmr_blck(i,j,k) = bbuffer(bpter(ib,k)+ibuffN2p)
		     if (sIndxOp > 0)  opmmr_blck(i,j,k) = bbuffer(bpter(ib,k)+ibuffOp)
                   endif
!--------------------------------

                enddo
             enddo
          enddo

          deallocate(bpter)
          deallocate(bbuffer)
          deallocate(cbuffer)

       endif ! logical local_dp_map

! WS: 02.08.06: Update t3 to temperature
!$omp parallel do private(i,j,k)
!dir$ concurrent
       do k=1,km
          do j = jfirstxy,jlastxy
             do i = ifirstxy,ilastxy
                t3xy(i,j,k) = ptxy(i,j,k)
             enddo
          enddo
       enddo

! -------------------------------------------------------------------------
! Update u3s and v3s from tendencies dudt and dvdt.
! -------------------------------------------------------------------------
       dt5 = D0_5*dtime

       call t_barrierf('sync_uv3s_update', grid%commxy)
       call t_startf('uv3s_update')
       if (iam .lt. grid%npes_xy) then
          call uv3s_update( grid, dudtxy, u3sxy, dvdtxy, v3sxy, dt5 )
       end if  ! (iam .lt. grid%npes_xy)
       call t_stopf('uv3s_update')

! -------------------------------------------------------------------------
! Compute pt, q3, pe, delp, ps, peln, pkz and pk.
! For 2-D decomposition, delp is transposed to delpxy, pexy is computed
!  from delpxy (and ptop), and pexy is transposed back to pe.
! Note that pt, q3, delp and pe are input parameters as well.
! -------------------------------------------------------------------------
    call t_barrierf('sync_p_d_adjust', grid%commxy)
    call t_startf ('p_d_adjust')
    if (iam .lt. grid%npes_xy) then
       call p_d_adjust(grid, tracer, dummy_pelnxy, pkxy, pkzxy, zvir,  cappa, &
                       delpxy, ptxy, pexy, psxy, ptop)
    end if  ! (iam .lt. grid%npes_xy)
    call t_stopf  ('p_d_adjust')

!EOC
  end subroutine p_d_coupling
!-----------------------------------------------------------------------
end module dp_coupling
