module dpie_coupling
!
! Dynamics/Physics Ionosphere/Electrodynamics coupler.
! B. Foster, 2015.
!
  use shr_kind_mod    ,only: r8 => shr_kind_r8
  use cam_logfile     ,only: iulog
  use cam_history     ,only: outfld ! CAM routine to output fields to history files
  use edyn_output     ,only: write_output ! for edynamo output file
  use cam_abortutils  ,only: endrun
  use spmd_utils      ,only: masterproc
  use edyn_init       ,only: nstep_savefld_edyn
  use savefield_waccm ,only: savefld_waccm
  use edyn_mpi        ,only: array_ptr_type
  use amie_module     ,only: getamie
  use edyn_init,     only: iamie,amienh,amiesh ! amie NH and SH inputs
!  use edyn_init,     only: amie_efxg, amie_kevg
  use edyn_solve,    only: crad, phihm, amie_efxm, amie_kevm

  implicit none
!
! Weimer 2005 data file name (this should be in a namelist file):
! 
  character(len=256) :: wei05sc_ncfile = 'wei05sc.nc'

  private
  public d_pie_coupling

contains
!-----------------------------------------------------------------------
  subroutine d_pie_coupling(omega,pe,zgi,u,v,pt,ionrates,ion_OpO2,ion_OpN2,    &
    sigma_ped,sigma_hall,te,ti,h2ommr,o2mmr,o1mmr,h1mmr,n1mmr,o2pmmr,          &
    nopmmr,n2pmmr,opmmr,optm1,ui,vi,wi,efxg,kevg,                              &
    rmassO2,rmassO1,rmassH,rmassN,rmassN2,rmassO2p, rmassNOp,rmassN2p,rmassOp, &
    i0,i1,j0,j1)
!
! Call dynamo to calculate electric potential, electric field, and ion drifts.
! Then call oplus_xport to transport O+, which is passed back to physics.
!
! This routine is called from p_d_coupling (dynamics/fv/dp_coupling.F90) when
! nstep > 0.
!
    use pmgrid,       only: &
      plev,                 & ! number of midpoint levels
      plevp                   ! number of interface levels
    use constituents, only: &
      cnst_get_ind,         & ! Get field indices
      cnst_mw,              & ! Needed to access constituent molecular weights
      pcnst                   ! Number of constituents
    use physconst,    only: zvir
    use shr_const_mod,only :     &
      grav   => shr_const_g,     &   ! gravitational constant (m/s^2)
      kboltz => shr_const_boltz      ! Boltzmann constant
    use time_manager, only: get_nstep
    use mo_jeuv,      only : nIonRates        ! Number of ionization rates (11 rates)
    use time_manager, only: get_curr_date,get_curr_calday
    use mag_parms,    only: get_mag_parms
    use heelis,       only: heelis_model
    use edynamo,      only: dynamo,ed1,ed2,ed1_glb,ed2_glb
    use savefield_edyn,only: savefld_edyn           ! save field to edynamo.nc
    use edyn_geogrid,  only: nglblon=>nlon, nglblat=>nlat
    use edyn_maggrid,  only: nmlat,nmlon,nmlonp1
    use edyn_mpi,      only: mlat0,mlat1,mlon0,mlon1
    use edyn_init,     only: use_time3d,       & ! logical flag to toggle time3d off/on
                             use_time3d_integ, & ! if true, use time3d conductances in edynamo
                             use_dynamo_drifts,&
                             use_time3d_output,& ! if true, time3d output O+ is passed back into waccm
                             use_tiegcm_oplus    ! if true, call oplus_xport for O+ transport
    use edyn_init,     only: highlat_potential_model ! either 'heelis' or 'weimer' or 'amie'
    use edyn_mpi,      only: switch_model_format ! routine to switch between "model formats"
    use time3d,        only: time3d_drv          ! time3d model driver
    use time3d,        only: prep_time3d_input   ! prepare fields for input to time3d
    use oplus,         only: oplus_xport,kbot_op=>kbot
    use fv_control_mod,only: nsplit
    use mo_solar_parms,only: solar_parms_get,solar_parms_file
    use wei05sc,       only: weimer05            ! driver for weimer high-lat convection model
    use edyn_solve,    only: pfrac,crit    ! NH fraction of potential (nmlonp1,nmlat0)
    use edyn_params   ,only: dtr
!
! Args:
!
    integer,intent(in) :: &
      i0,                 & ! grid%ifirstxy
      i1,                 & ! grid%ilastxy
      j0,                 & ! grid%jfirstxy
      j1                    ! grid%jlastxy

    real(r8),intent(in) :: omega  (i0:i1,plev,j0:j1)    ! pressure velocity on midpoints (Pa/s) (i,k,j)
    real(r8),intent(in) :: pe     (i0:i1,plevp,j0:j1)   ! interface pressure (Pa)  (note i,k,j dims)
    real(r8),intent(in) :: zgi    (i0:i1,j0:j1,plevp)   ! geopotential height (on interfaces) (m)
    real(r8),intent(in) :: u      (i0:i1,j0:j1,plev)    ! U-wind (m/s)
    real(r8),intent(in) :: v      (i0:i1,j0:j1,plev)    ! V-wind (m/s)
    real(r8),intent(in) :: pt     (i0:i1,j0:j1,plev)    ! potential temperature
    real(r8),intent(in) :: ionrates(i0:i1,j0:j1,plev,nIonRates) ! EUV ion rates from mo_photo
    real(r8),intent(in) :: ion_OpO2(i0:i1,j0:j1,plev)   ! Op+O2 rate (s-1/cm^3)
    real(r8),intent(in) :: ion_OpN2(i0:i1,j0:j1,plev)   ! Op+N2 rate (s-1/cm^3)
    real(r8),intent(in) :: sigma_ped (i0:i1,j0:j1,plev) ! Pedersen conductivity
    real(r8),intent(in) :: sigma_hall(i0:i1,j0:j1,plev) ! Hall conductivity
    real(r8),intent(in) :: te(i0:i1,j0:j1,plev)         ! electron temperature
    real(r8),intent(in) :: ti(i0:i1,j0:j1,plev)         ! ion temperature
    real(r8),intent(in) :: h2ommr(i0:i1,j0:j1,plev)     ! H2O mass mixing ratio (for oplus)
    real(r8),intent(in) :: o2mmr(i0:i1,j0:j1,plev)      ! O2 mass mixing ratio (for oplus)
    real(r8),intent(in) :: o1mmr(i0:i1,j0:j1,plev)      ! O mass mixing ratio (for oplus)
    real(r8),intent(in) :: h1mmr(i0:i1,j0:j1,plev)      ! H mass mixing ratio (for oplus)
    real(r8),intent(in) :: n1mmr(i0:i1,j0:j1,plev)      ! N mass mixing ratio (for oplus)
    real(r8),intent(in) :: o2pmmr(i0:i1,j0:j1,plev)     ! O2+ mass mixing ratio (for oplus)
    real(r8),intent(in) :: nopmmr(i0:i1,j0:j1,plev)     ! NO+ mass mixing ratio (for oplus)
    real(r8),intent(in) :: n2pmmr(i0:i1,j0:j1,plev)     ! N2+ mass mixing ratio (for oplus)
    real(r8),intent(inout) :: opmmr(i0:i1,j0:j1,plev)   ! O+ mass mixing ratio (oplus_xport output)
    real(r8),intent(inout) :: optm1(i0:i1,j0:j1,plev)   ! O+ previous time step (oplus_xport output)
    real(r8),intent(inout) :: ui(i0:i1,j0:j1,plev)      ! zonal ion drift (edynamo or empirical)
    real(r8),intent(inout) :: vi(i0:i1,j0:j1,plev)      ! meridional ion drift (edynamo or empirical)
    real(r8),intent(inout) :: wi(i0:i1,j0:j1,plev)      ! vertical ion drift (edynamo or empirical)
    real(r8),intent(inout) :: efxg(i0:i1,j0:j1)         ! energy flux from AMIE
    real(r8),intent(inout) :: kevg(i0:i1,j0:j1)         ! characteristic mean energy from AMIE
    real(r8),intent(in) :: rmassO2      	        ! O2 molecular weight kg/kmol
    real(r8),intent(in) :: rmassO1      	        ! O atomic weight kg/kmol
    real(r8),intent(in) :: rmassH       	        ! H atomic weight kg/kmol
    real(r8),intent(in) :: rmassN2      	        ! N2 molecular weight kg/kmol
    real(r8),intent(in) :: rmassN       	        ! N molecular weight kg/kmol
    real(r8),intent(in) :: rmassO2p     	        ! O2+ molecular weight kg/kmol
    real(r8),intent(in) :: rmassNOp     	        ! NO+ molecular weight kg/kmol
    real(r8),intent(in) :: rmassN2p     	        ! N2+ molecular weight kg/kmol
    real(r8),intent(in) :: rmassOp 			! O+ molecular weight kg/kmol
!
! Local:
!
    integer :: i,j,k,n,iField
    integer :: nstep,nlons
    integer :: nfields             ! Number of fields for multi-field calls
    integer :: iyear,imo,iday,tod  ! tod is time-of-day in seconds
    integer :: calday              ! calendar day (day of the year)
    integer :: nspltop             ! local nsplit for oplus_xport
    integer :: isplit              ! loop index
    integer :: iprint,amie_ibkg

    real(r8) :: secs           ! time of day in seconds
    real(r8) :: ctpoten        ! Cross-tail potential from get_mag_parms method

    real(r8), parameter :: n2min = 1.e-6_r8  ! lower limit of N2 mixing ratios
    real(r8), parameter :: small = 1.e-25_r8 ! for fields not currently available
    real(r8) :: tn   (i0:i1,j0:j1,plev) ! neutral temperature (deg K)
    real(r8) :: zht  (i0:i1,j0:j1,plev) ! geometric height (m) 
    real(r8) :: wn   (i0:i1,j0:j1,plev) ! vertical velocity (from omega)
    real(r8) :: mbar (i0:i1,j0:j1,plev) ! mean molecular weight
    real(r8) :: n2mmr(i0:i1,j0:j1,plev) ! N2 mass mixing ratio (for oplus)
    real(r8) :: pmid_inv(plev)          ! pressure at midpoints (Pa) (arbitrary i,j for oplus)
    real(r8) :: pmid(i0:i1,plev,j0:j1)  ! pressure at midpoints (Pa) (global i,j)
    real(r8) :: re = 6.370e6            ! earth radius (m)

    real(r8),dimension(i0:i1,j0:j1,plev) :: & ! species number densities (m^3)
      o2,o1,n2,n1,he
    real(r8),dimension(i0:i1,j0:j1,plev) :: & ! ion number densities (m^3)
      o2p,nop,n2p,op,ne
    real(r8),dimension(i0:i1,j0:j1,plev) :: & ! production rates
      qop, qhep, qhp, qep                     ! O+, He+, H+, photoelectron
    real(r8),dimension(i0:i1,j0:j1,plev) :: & ! loss rates
      lop, lhep, lhp                          ! O+, He+, H+

    real(r8),dimension(plev,i0:i1,j0:j1) :: &
      op_kij,opmmr_kij, qop_kij, qep_kij, ped_kij, hall_kij, opo2_kij, opn2_kij, lop_kij
    real(r8) :: ionPRates(i0:i1,j0:j1,plev,nIonRates) ! ion production rates m^3
    real(r8) :: diag_ik(i0:i1,plev)                   ! for outfld calls
    character(len=16) :: ionrate_names(nIonRates)
!
! Args for dynamo:
    real(r8),target :: edyn_tn   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_un   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_vn   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_wn   (plev,i0:i1,j0:j1) ! vertical wind (cm/s)
    real(r8),target :: edyn_zht  (plev,i0:i1,j0:j1) ! geometric height (cm)
    real(r8),target :: edyn_mbar (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_ped  (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_hall (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_ui   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_vi   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_wi   (plev,i0:i1,j0:j1)
!
! Additional fields needed by oplus_xport:
    real(r8),target :: edyn_te   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_ti   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_o2   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_o1   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_n2   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_op   (plev,i0:i1,j0:j1)
    real(r8),target :: edyn_optm1(plev,i0:i1,j0:j1)
    real(r8),target :: edyn_om   (plev,i0:i1,j0:j1) ! omega vertical motion (1/s)
    real(r8),target :: edyn_zgi  (plev,i0:i1,j0:j1) ! geopotential height (cm) (interfaces)
    real(r8),target :: op_out    (plev,i0:i1,j0:j1) ! oplus_xport output
    real(r8),target :: opnm_out  (plev,i0:i1,j0:j1) ! oplus_xport output at time n-1
    real(r8),target :: edyn_ne   (plev,i0:i1,j0:j1) ! electron density diagnostic
!
! AMIE fields (extra dimension added for longitude switch)
!
    real(r8),target :: amie_efxg (plev,i0:i1,j0:j1) ! AMIE energy flux
    real(r8),target :: amie_kevg (plev,i0:i1,j0:j1) ! AMIE characteristic mean energy
!
! Solar parameter(read-only):
    real(r8) :: f107d,f107a,ap,kp,hp
! AMIE parameters
    real(r8) :: c35,c25
!    real(r8),dimension(i0:i1,j0:j1) :: efxg,kevg
!
! IMF and solar wind scalar parameters for Weimer model.
! (These could go into a namelist input file at some point,
!  made time-dependent, or OMNI nc data files could be 
!  imported, as in TIEGCM, LFM, etc)
!
    real(r8) :: &
      byimf,     &  ! BY component of IMF
      bzimf,     &  ! BZ component of IMF in nT
      swvel,     &  ! Solar wind velocity in km/s
      swden         ! Solar wind density in #/cm3
    real(r8) :: sunlons(nglblat)
    real(r8) :: ctpoten_weimer ! Cross-tail potential from Weimer model
!
    logical :: do_integrals
!
    real(r8),dimension(nmlonp1,nmlat,1) :: &
      ed1_glb_ij,ed2_glb_ij                    ! 2d (lon,lat) efield on mag grid
!
! Pointers for multiple-field calls:
    type(array_ptr_type),allocatable :: ptrs(:)

    nstep = get_nstep()
    call get_curr_date(iyear,imo,iday,tod) ! tod is integer time-of-day in seconds
    secs = tod ! integer to float
!
! If dtime=300 (5 minutes) and nspltop=5, then oplus_xport will be
! called with 60-sec step (300/5)
!
    nspltop = 5
    write(iulog,"('Enter d_pie_coupling: nstep=',i8,' iyear,imo,iday=',3i5,' ut (hrs)=',f6.2)") &
      nstep,iyear,imo,iday,secs/3600._r8
!
! Get solar parameters (solar_parms_file can be specified in user_nl_cam)
! (default location of solar parms files is /glade/p/cesmdata/cseg/inputdata/atm/waccm/phot/) 
!
    call solar_parms_get(f107d,f107a,ap,kp,hp)
    if (nstep==1) then
      write(iulog,"('d_pie_coupling: nsplit=',i3,' nspltop=',i3)") nsplit,nspltop
      write(iulog,"('d_pie_coupling: solar_parms_file=',a)") trim(solar_parms_file)
      write(iulog,"('d_pie_coupling: f107d=',f8.2,' f107a=',f8.2,' ap=',f8.2,' kp=',f8.2,' hp=',f8.2)") &
        f107d,f107a,ap,kp,hp
    endif
!
! Get pressure at midpoints from pe (note pe is vertical dimension is plevp):
!
    do k=1,plev
      pmid(i0:i1,k,j0:j1) = 0.5_r8*(pe(i0:i1,k,j0:j1)+pe(i0:i1,k+1,j0:j1))
    enddo
!
! Convert geopotential z to geometric height zht (m):
!
!   zht = z/grav                ! geopotential height (grav is in m/s^2)
!   zht = zht * (1._r8+zht/re)  ! geometric height

    zht(:,:,1:plev) = zgi(:,:,1:plev) * (1._r8+zgi(:,:,1:plev)/re) ! geometric height (interfaces)
!
! Convert virtual potential temperature to temperature and compute mean molecular weight:
!
    do k=1,plev
      do j=j0,j1
        do i=i0,i1
          tn(i,j,k) = pt(i,j,k) / (1.0_r8 + zvir*h2ommr(i,j,k))
          n2mmr(i,j,k) = max(1.0_r8-(o1mmr(i,j,k)+o2mmr(i,j,k)+h1mmr(i,j,k)),n2min)
          mbar(i,j,k) = 1.0_r8/(o1mmr(i,j,k)/rmassO1+o2mmr(i,j,k)/rmassO2 &
                               +h1mmr(i,j,k)/rmassH+n2mmr(i,j,k)/rmassN2)
        enddo
      enddo
    enddo
!
! Save input omega (Pa/s) and mbar.
    do j=j0,j1
!     call outfld('DPIE_OMEGA',omega(i0:i1,1:plev,j),i1-i0+1,j)
!     call outfld('DPIE_MBAR' ,mbar (i0:i1,j,1:plev),i1-i0+1,j)
    enddo
!
! Calculate vertical neutral wind velocity wn(i,j,k).
! (omega is input Pa/s, grav is m/s^2, tn and mbar are calculated above)
!
    call calc_wn(tn,omega,pmid,mbar,grav,wn,i0,i1,j0,j1,plev) ! wn is output (m/s)
!
! Convert from mmr to number densities (m^3):
!
    do k=1,plev
      do j=j0,j1
        do i=i0,i1
! O2, O, N2, N:
          o2(i,j,k) = o2mmr(i,j,k) * mbar(i,j,k) / rmassO2 * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
          o1(i,j,k) = o1mmr(i,j,k) * mbar(i,j,k) / rmassO1  * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
          n2(i,j,k) = n2mmr(i,j,k) *      mbar(i,j,k) / rmassN2  * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
          n1(i,j,k) = n1mmr(i,j,k) * mbar(i,j,k) / rmassN   * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
! O2+, NO+, N2+, O+:
          o2p(i,j,k) = o2pmmr(i,j,k) * mbar(i,j,k) / rmassO2p * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
          nop(i,j,k) = nopmmr(i,j,k) * mbar(i,j,k) / rmassNOp * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
          n2p(i,j,k) = n2pmmr(i,j,k) * mbar(i,j,k) / rmassN2p * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
          op(i,j,k)  = opmmr(i,j,k)  * mbar(i,j,k) / rmassOp  * &
            pmid(i,k,j) / (kboltz * tn(i,j,k))
        enddo
      enddo
    enddo ! k=1,plev
!
! Save input ions to waccm history (m^3):
    do j=j0,j1
!     call outfld('DPIE_O2P',o2p(i0:i1,j,1:plev),i1-i0+1,j)
!     call outfld('DPIE_NOP',nop(i0:i1,j,1:plev),i1-i0+1,j)
!     call outfld('DPIE_N2P',n2p(i0:i1,j,1:plev),i1-i0+1,j)
!     call outfld('DPIE_OP' ,op (i0:i1,j,1:plev),i1-i0+1,j)
      do k=1,plev
        do i=i0,i1
          ne(i,j,k) = o2p(i,j,k)+nop(i,j,k)+n2p(i,j,k)+op(i,j,k)
        enddo
      enddo
!     call outfld('DPIE_NE' ,ne (i0:i1,j,1:plev),i1-i0+1,j)
    enddo ! j=j0,j1
!
! Some fields are not available, and are set to small values for now:
    he   = small ! Helium number density
    qhep = small ! He+ production rate
    qhp  = small ! H+ production rate
!   lhep = small ! He+ loss rate
!   lhp  = small ! H+ loss rate
!
! Convert ion rates to m^3 using number densities from above.
! (see also ionosphere.F90)
!
    do n=1,nIonRates
      do k=1,plev
        do j=j0,j1
          if (n <= 3) ionPRates(:,j,k,n) = ionrates(:,j,k,n) * o1(:,j,k)
          if (n == 4) ionPRates(:,j,k,n) = ionrates(:,j,k,n) * n1(:,j,k)
          if ((n == 5).or.(n >= 7 .and. n <= 9)) &
            ionPRates(:,j,k,n) = ionrates(:,j,k,n) * o2(:,j,k) 
          if (n==6 .or. n==10 .or. n==11) &
            ionPRates(:,j,k,n) = ionrates(:,j,k,n) * n2(:,j,k) 
        enddo ! j=j0,j1
      enddo ! k=1,plev
    enddo ! n=1,nIonRates
!
! Photo-Electron and O+ Production rates (for input to time3d):
! (see also outfld calls in mo_setext.F90 for P_Op,P_O2p...P_IONS)
!
! O+ production rate qop is sum of these (see mo_jeuv.F90):
! ionPRates(...1) =  O + hv --> O+ (4S) + e*
! ionPRates(...2) =  O + hv --> O+ (2D) + e*
! ionPRates(...3) =  O + hv --> O+ (2P) + e*
! ionPRates(...7) =  O2 + hv --> O + O+(4S) + e*
! ionPRates(...8) =  O2 + hv --> O + O+(2D) + e*
! ionPRates(...9) =  O2 + hv --> O + O+(2P) + e*
!
    qep = 0._r8   ! init photo-electron production
    qop = 0._r8   ! init O+ production rate
    do k=1,plev
      do j=j0,j1
        do i=i0,i1
          qep(i,j,k) = sum(ionPRates(i,j,k,1:nIonRates))
          qop(i,j,k) = &
            ionPRates(i,j,k,1)+ionPRates(i,j,k,2)+ionPRates(i,j,k,3)+ &
            ionPRates(i,j,k,7)+ionPRates(i,j,k,8)+ionPRates(i,j,k,9)
        enddo
      enddo
    enddo
!
! Save fields to waccm history:
! (must be transformed from (i,j,k) to (k,i,j))
!
    do j=j0,j1
      do i=i0,i1
        ped_kij (:,i,j) = sigma_ped(i,j,:)
        hall_kij(:,i,j) = sigma_hall(i,j,:)
        op_kij  (:,i,j) = op(i,j,:)
        opmmr_kij(:,i,j) = opmmr(i,j,:)
        qep_kij (:,i,j) = qep(i,j,:)
        qop_kij (:,i,j) = qop(i,j,:)
        opo2_kij(:,i,j) = ion_OpO2(i,j,:)
        opn2_kij(:,i,j) = ion_OpN2(i,j,:)
      enddo
    enddo
!
!   call savefld_waccm(ped_kij  ,'SIGMA_PED' ,plev,i0,i1,j0,j1)
!   call savefld_waccm(hall_kij ,'SIGMA_HALL',plev,i0,i1,j0,j1)
!   call savefld_waccm(op_kij   ,'DPIE_OP'   ,plev,i0,i1,j0,j1) ! m^3
!   call savefld_waccm(opmmr_kij,'DPIE_OPMMR',plev,i0,i1,j0,j1) ! mmr
!   call savefld_waccm(qep_kij  ,'QEP'       ,plev,i0,i1,j0,j1)
!   call savefld_waccm(qop_kij  ,'QOP'       ,plev,i0,i1,j0,j1)
!   call savefld_waccm(opo2_kij ,'OpO2'      ,plev,i0,i1,j0,j1)
!   call savefld_waccm(opn2_kij ,'OpN2'      ,plev,i0,i1,j0,j1)
!
! O+ loss rates:
! Input loss rates:
!   real(r8),intent(in) :: ion_OpO2(i0:i1,j0:j1,plev) ! Op+O2 rate
!   real(r8),intent(in) :: ion_OpN2(i0:i1,j0:j1,plev) ! Op+N2 rate
! Output (for input to time3d):
!   real(r8),dimension(i0:i1,j0:j1,plev) :: lop
! Multiply input loss rates coefficients by number density (m^3) of neutral densities
!
    lop = 0._r8
    do k=1,plev
      do j=j0,j1
        do i=i0,i1
          lop(i,j,k) = (ion_OpO2(i,j,k)*o2(i,j,k)+ion_OpN2(i,j,k)*n2(i,j,k))*1.e-6
        enddo
      enddo
    enddo

    do j=j0,j1
      do i=i0,i1
        lop_kij(:,i,j) = lop(i,j,:)
      enddo
    enddo
!   call savefld_waccm(lop_kij,'LOP',plev,i0,i1,j0,j1)
!
! Get high-latitude potential from Heelis empirical model:
! (sub heelis_model is in heelis.F90)
!
    calday = get_curr_calday()             ! day of year
    call get_curr_date(iyear,imo,iday,tod) ! tod is integer time-of-day in seconds
    secs = tod                             ! should promote from int to real(r8)
    call get_mag_parms( ctpoten = ctpoten )
!
! use_time3d and use_tiegcm_oplus are mutually exclusive:
!
    if (use_time3d.and.use_tiegcm_oplus) then
      write(iulog,"('>>> dpie_coupling: cannot have both use_time3d and use_tiegcm_oplus true.')")
      call endrun
    endif
!
! Optionally invoke time3d model (use_time3d flag is in edyn_init.F90):
!
    if (use_time3d) then
!
! Prepare edynamo and waccm fields for input to time3d (time3d.F90): 
! (z must be the first field)
!
      call prep_time3d_input(zht,tn,u,v,wn,o2,o1,n2,n1,o2p,nop,n2p,op, &
        sigma_ped,sigma_hall,qep,qop,lop,i0,i1,j0,j1,plev,18, &
        (/'z','t','u','v','w','o2','o','n2','n','o2p','nop','n2p','op',&
          'Ped','Hall','Qep','Qop','OpL'/))
!
! All tasks make this call, but only the root task will execute time3d
! If use_time3d_output=T then op(i,j,k) (O+ on subdomains) is returned 
! by time3d_drv, otherwise it is unchanged.
!
      call time3d_drv(op,zht,i0,i1,j0,j1,plev)
!
! If O+ output from time3d is to be passed to waccm, then convert 
! from m^3 to mmr.
!
      if (use_time3d_output) then
        do k=1,plev
          do j=j0,j1
            do i=i0,i1
              opmmr(i,j,k) = op(i,j,k) * rmassOp / mbar(i,j,k) * &
                (kboltz * tn(i,j,k)) / pmid(i,k,j)
            enddo
          enddo
        enddo
      endif
    endif ! use_time3d
!
! Get sun's longitude at latitudes (geographic):
!
    call sunloc(iday,secs,sunlons) ! sunlons(nglblat) is returned
!
! Get high-latitude convection from empirical model (heelis or weimer).
! High-latitude potential phihm (edyn_solve) is defined for edynamo.
! Heelis takes ctpoten as input, Weimer returns ctpoten_weimer as output.
!
    if (trim(highlat_potential_model) == 'heelis') then
      write(iulog,"('dpie_coupling call heelis: nstep=',i5,' ctpoten=',f8.2)") &
        nstep,ctpoten
      call heelis_model(calday,secs,ctpoten,sunlons) ! heelis.F90
      call calc_pfrac(sunlons(1),ctpoten,pfrac) ! returns pfrac for dynamo (edyn_solve)
    elseif (trim(highlat_potential_model) == 'weimer') then
!     byimf = 0._r8   ! nT
!     bzimf = -2.0_r8 ! nT
      swvel = 400._r8 ! km/s
      swden = 10._r8  ! per cm3
!
! get_mag_parms will return by=0, and will calculate bz from kp.
      call get_mag_parms(by=byimf, bz=bzimf)
!
! ctpoten_weimer is returned by Weimer model (weisc05.F90).
! If get_mag_parms is called for ctpoten, it will return ctpoten_weimer.
!
      call weimer05(byimf,bzimf,swvel,swden,sunlons,wei05sc_ncfile,ctpoten_weimer)
      call calc_pfrac(sunlons(1),ctpoten_weimer,pfrac) ! returns pfrac for dynamo (edyn_solve)
      write(iulog,"('dpie_coupling call weimer05: byimf,bzimf=',2f8.2,' swvel,swden=',2f8.2,' ctpoten_weimer=',f8.2)") &
        byimf,bzimf,swvel,swden,ctpoten_weimer
    else
      call endrun('dpie_coupling: Unknown highlat_potential_model') 
    endif
!
! Get AMIE data input
!
    if (iamie > 0) then
      iprint = 1
      amie_ibkg = 0
      iamie = 1
      if (iprint>0) write(6,"('Calling getamie >>> iamie=',i2)") iamie

      call getamie(iyear,imo,iday,int(secs),amie_ibkg,iprint,iamie,amienh,amiesh,phihm,amie_efxm,amie_kevm,crad,efxg,kevg)
      
      if (iprint>0) write(6,"('After Calling getamie >>> iamie=',i2)") iamie
      call savefld_waccm(phihm,'phihm',1,mlon0,mlon1,mlat0,mlat1) 
      call savefld_waccm(amie_efxm,'amie_efxm',1,mlon0,mlon1,mlat0,mlat1) 
      call savefld_waccm(amie_kevm,'amie_kevm',1,mlon0,mlon1,mlat0,mlat1) 

      call savefld_waccm(efxg,'amie_efxg',1,i0,i1,j0,j1) 
      call savefld_waccm(kevg,'amie_kevg',1,i0,i1,j0,j1) 

      c35 = min(30.,(crad(1)+crad(2))*0.5/dtr + 5.0)
      c25 = max(35.,c35)
      crit(1) = c25 * dtr
      crit(2) = 40. * dtr
      call calc_pfrac_amie(sunlons(1),pfrac,crit) ! returns pfrac for dynamo (edyn_solve)
    endif
!
! Prepare inputs to edynamo and oplus_xport:
!
    do k = 1,plev
      edyn_tn   (k,i0:i1,j0:j1) = tn        (i0:i1,j0:j1,k)
      edyn_un   (k,i0:i1,j0:j1) = u         (i0:i1,j0:j1,k) * 100._r8 ! m/s -> cm/s
      edyn_vn   (k,i0:i1,j0:j1) = v         (i0:i1,j0:j1,k) * 100._r8 ! m/s -> cm/s
      edyn_wn   (k,i0:i1,j0:j1) = wn        (i0:i1,j0:j1,k) * 100._r8 ! m/s -> cm/s
      edyn_zgi  (k,i0:i1,j0:j1) = zgi       (i0:i1,j0:j1,k) * 100._r8 ! m -> cm
      edyn_zht  (k,i0:i1,j0:j1) = zht       (i0:i1,j0:j1,k) * 100._r8 ! m -> cm
      edyn_mbar (k,i0:i1,j0:j1) = mbar      (i0:i1,j0:j1,k)
      edyn_ped  (k,i0:i1,j0:j1) = sigma_ped (i0:i1,j0:j1,k)
      edyn_hall (k,i0:i1,j0:j1) = sigma_hall(i0:i1,j0:j1,k)
      edyn_ui   (k,i0:i1,j0:j1) = ui        (i0:i1,j0:j1,k) * 100._r8 ! zonal ion drift (m/s -> cm/s)
      edyn_vi   (k,i0:i1,j0:j1) = vi        (i0:i1,j0:j1,k) * 100._r8 ! meridional ion drift (m/s -> cm/s)
      edyn_wi   (k,i0:i1,j0:j1) = wi        (i0:i1,j0:j1,k) * 100._r8 ! vertical ion drift (m/s -> cm/s)
!
! Additional fields for oplus:
!
      edyn_te   (k,i0:i1,j0:j1) = te     (i0:i1,j0:j1,k)
      edyn_ti   (k,i0:i1,j0:j1) = ti     (i0:i1,j0:j1,k)
      edyn_o2   (k,i0:i1,j0:j1) = o2mmr  (i0:i1,j0:j1,k)
      edyn_o1   (k,i0:i1,j0:j1) = o1mmr  (i0:i1,j0:j1,k)
      edyn_n2   (k,i0:i1,j0:j1) = n2mmr  (i0:i1,j0:j1,k)
      edyn_om   (k,i0:i1,j0:j1) = -(omega(i0:i1,k,j0:j1) / pmid(i0:i1,k,j0:j1)) ! Pa/s -> 1/s
      edyn_op   (k,i0:i1,j0:j1) = op     (i0:i1,j0:j1,k) / 1.e6_r8  ! m^3 -> cm^3
    enddo
!
! At first timestep, allocate optm1 module data, and initialize local
! edyn_optm1 to op from physics. This will be input to oplus_xport.
! After oplus_xport, optm1 will be updated from local oplus_xport output.
! After first timestep, simply update edyn_optm1 from optm1.
! optm1 is m^3 for waccm, whereas edyn_optm1 is cm^3 for oplus_xport.
!
! At this point, everything is in waccm format. The locals edyn_op and
! edyn_optm1 will be converted to tiegcm format for the call to oplus_xport,
! then oplus_xport output (opnm_out) will be converted back to waccm format
! before using it to update optm1 module data.
!
    if (nstep==1) then
      optm1 = 0._r8
      do k=1,plev
        edyn_optm1(k,i0:i1,j0:j1) = op(i0:i1,j0:j1,k) / 1.e6_r8  ! m^3 -> cm^3
      enddo
!
! After the first step, edyn_optm1 input is updated from the module data
! (note edyn_optm1 will be converted to TIEGCM format before being
! passed in to oplus_xport)
!
    else ! nstep > 1
      do k=1,plev
        edyn_optm1(k,i0:i1,j0:j1) = optm1(i0:i1,j0:j1,k) / 1.e6_r8 ! m^3 -> cm^3
      enddo
    endif
!
! These are in WACCM format, and most are in CGS units (see above):
! (units are specified in addfld calls, edyn_init.F90)
!
!   call savefld_waccm(edyn_tn   ,'DPIE_TN'  ,plev,i0,i1,j0,j1)  ! deg K
!   call savefld_waccm(edyn_un   ,'DPIE_UN'  ,plev,i0,i1,j0,j1)  ! cm/s
!   call savefld_waccm(edyn_vn   ,'DPIE_VN'  ,plev,i0,i1,j0,j1)  ! cm/s
!   call savefld_waccm(edyn_wn   ,'DPIE_WN'  ,plev,i0,i1,j0,j1)  ! cm/s
!   call savefld_waccm(edyn_om   ,'DPIE_OM'  ,plev,i0,i1,j0,j1)  ! omega on midpoints (1/s)
!   call savefld_waccm(edyn_zht  ,'DPIE_ZHT' ,plev,i0,i1,j0,j1)  ! geometric height (cm)
!   call savefld_waccm(edyn_zgi  ,'DPIE_ZGI'  ,plev,i0,i1,j0,j1) ! geopotential height on interfaces (cm)
!   call savefld_waccm(edyn_mbar ,'DPIE_BARM',plev,i0,i1,j0,j1)  ! mean mass
!   call savefld_waccm(edyn_ped  ,'DPIE_PED' ,plev,i0,i1,j0,j1)  ! pedersen conductivity
!   call savefld_waccm(edyn_hall ,'DPIE_HALL',plev,i0,i1,j0,j1)  ! hall conductivity
!   call savefld_waccm(edyn_ui   ,'DPIE_UI'  ,plev,i0,i1,j0,j1)  ! zonal ion drift (cm/s)
!   call savefld_waccm(edyn_vi   ,'DPIE_VI'  ,plev,i0,i1,j0,j1)  ! meridional ion drift (cm/s)
!   call savefld_waccm(edyn_wi   ,'DPIE_WI'  ,plev,i0,i1,j0,j1)  ! vertical ion drift (cm/s)
!   call savefld_waccm(edyn_o2   ,'DPIE_O2'  ,plev,i0,i1,j0,j1)  ! cm^3
!   call savefld_waccm(edyn_o1   ,'DPIE_O'   ,plev,i0,i1,j0,j1)  ! cm^3
!   call savefld_waccm(edyn_n2   ,'DPIE_N2'  ,plev,i0,i1,j0,j1)  ! cm^3
!   call savefld_waccm(edyn_te   ,'DPIE_TE'  ,plev,i0,i1,j0,j1)
!   call savefld_waccm(edyn_ti   ,'DPIE_TI'  ,plev,i0,i1,j0,j1)
!
! Save electron density to TIEGCM-format file (edynamo.nc):
! (ne(i,j,k) was calculated in m^3 above, save here in cm^3)
!
    do j=j0,j1
      do i=i0,i1
        do k=1,plev 
          edyn_ne(k,i,j) = ne(i,j,k)*1.e-6 ! m^3 -> cm^3
        enddo
      enddo
    enddo
!
! Convert input fields from "WACCM format" to "TIEGCM format" 
! (phase shift longitude data and invert the vertical dimension).
!
    nfields = 21
    allocate(ptrs(nfields))
!
! Fields needed for edynamo:
    ptrs(1)%ptr => edyn_tn    ; ptrs(2)%ptr => edyn_un   ; ptrs(3)%ptr => edyn_vn 
    ptrs(4)%ptr => edyn_wn    ; ptrs(5)%ptr => edyn_zht  ; ptrs(6)%ptr => edyn_zgi
    ptrs(7)%ptr => edyn_mbar  ; ptrs(8)%ptr => edyn_ped  ; ptrs(9)%ptr => edyn_hall
!
! Additional fields needed for oplus (and Ne for diag):
    ptrs(10)%ptr => edyn_te  ; ptrs(11)%ptr => edyn_ti    ; ptrs(12)%ptr => edyn_o2   
    ptrs(13)%ptr => edyn_o1  ; ptrs(14)%ptr => edyn_n2    ; ptrs(15)%ptr => edyn_om   
    ptrs(16)%ptr => edyn_op  ; ptrs(17)%ptr => edyn_optm1 ; ptrs(18)%ptr => edyn_ne
    ptrs(19)%ptr => edyn_ui  ; ptrs(20)%ptr => edyn_vi    ; ptrs(21)%ptr => edyn_wi
!
! Convert from WACCM to TIEGCM format:
    call switch_model_format(ptrs,1,plev,i0,i1,j0,j1,nfields)
    deallocate(ptrs)
!
! Save fields to edynamo.nc in TIEGCM format:
!
!   do j=j0,j1
!     call savefld_edyn('DPIE_TN' ,'DPIE_TN' ,'deg K',edyn_tn  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_UN' ,'DPIE_UN' ,'cm/s' ,edyn_un  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_VN' ,'DPIE_VN' ,'cm/s' ,edyn_vn  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_WN' ,'DPIE_WN' ,'cm/s' ,edyn_wn  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_OM' ,'DPIE_OM' ,'1/s'  ,edyn_om  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_PED','DPIE_PED','S/m'  ,edyn_ped (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_HAL','DPIE_HAL','S/m'  ,edyn_hall(:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_UI' ,'DPIE_UI' ,'   '  ,edyn_ui  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_VI' ,'DPIE_VI' ,'   '  ,edyn_vi  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_WI' ,'DPIE_WI' ,'   '  ,edyn_wi  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_ZHT','DPIE_ZHT','cm'   ,edyn_zht (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_ZG' ,'DPIE_ZG' ,'cm'   ,edyn_zgi (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j) ! same as Z
!     call savefld_edyn('DPIE_O2' ,'DPIE_O2' ,'mmr'  ,edyn_o2  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_O'  ,'DPIE_O'  ,'mmr'  ,edyn_o1  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_N2' ,'DPIE_N2' ,'mmr'  ,edyn_n2  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_TE' ,'DPIE_TE' ,'deg K',edyn_te  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_TI' ,'DPIE_TI' ,'deg K',edyn_ti  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!     call savefld_edyn('DPIE_NE' ,'DPIE_NE' ,'cm^3' ,edyn_ne  (:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!
! Z should be written to the history if you are planning to use the tgcmproc_idl 
! post-processor to look at fields on geo grid (ZMAG is saved elsewhere).
! Note that when tgcmproc_idl sees "Z", it converts it to km, so am leaving
! units blank in this call:
!
!     call savefld_edyn('Z','geopotential height (interfaces)','cm',edyn_zgi(:,i0:i1,j),&
!       'lev',1,plev,'lon',i0,i1,j)
!   enddo
!
! Call electrodynamo (edynamo.F90)
! If using time3d conductances, tell dynamo to *not* do fieldline 
! integrations (i.e., do_integrals == false). In this case, edynamo
! conductances zigmxx,rim1,2 from time3d will be set by subroutine
! transform_glbin in time3d module.
!
    do_integrals = .true.
    if (use_time3d.and.use_time3d_integ) do_integrals = .false.
!
! If use_dynamo_drifts=false, then empirical ion drifts were passed in from physics,
! otherwise dynamo calculates them here, and they will be passed to physics.
!
    if (use_dynamo_drifts) then

      call dynamo(edyn_tn, edyn_un, edyn_vn, edyn_wn, edyn_zgi,   &
                  edyn_ped, edyn_hall, edyn_ui, edyn_vi, edyn_wi, &
                  1,plev,i0,i1,j0,j1,do_integrals)

      write(iulog,"('dpie_coupling after dynamo: nstep=',i8)") nstep
      write(iulog,"('  ui min,max (cm/s)=',2es12.4)") minval(edyn_ui),maxval(edyn_ui)
      write(iulog,"('  vi min,max (cm/s)=',2es12.4)") minval(edyn_vi),maxval(edyn_vi)
      write(iulog,"('  wi min,max (cm/s)=',2es12.4)") minval(edyn_wi),maxval(edyn_wi)
    else
      write(iulog,"('dpie_coupling (dynamo NOT called): nstep=',i8)") nstep
      write(iulog,"('  empirical ExB ui min,max (cm/s)=',2es12.4)") minval(ui),maxval(ui)
      write(iulog,"('  empirical ExB vi min,max (cm/s)=',2es12.4)") minval(vi),maxval(vi)
      write(iulog,"('  empirical ExB wi min,max (cm/s)=',2es12.4)") minval(wi),maxval(wi)
    endif
!
! Call O+ transport routine.  Now all inputs to oplus_xport should be in 
! tiegcm-format wrt longitude (-180->180), vertical (bot2top), and units (CGS).
! (Composition is mmr, ne is cm^3, winds are cm/s)
! Output op_out and opnm_out will be in cm^3, converted to mmr below.
!
    if (use_tiegcm_oplus) then
      pmid_inv(1:plev) = pmid(i0,plev:1:-1,j0) ! invert local pmid as in tiegcm
!
! Transport O+ (all args in 'TIEGCM format')
! Subcycle oplus_xport nspltop times.
!
      do isplit=1,nspltop
        if (isplit > 1) then
          edyn_op = op_out
          edyn_optm1 = opnm_out
        endif

        call oplus_xport(edyn_tn,edyn_te,edyn_ti,edyn_un,edyn_vn,edyn_om,            &
                         edyn_zgi,edyn_o2,edyn_o1,edyn_n2,edyn_op,edyn_optm1,        &
                         edyn_mbar,edyn_ui,edyn_vi,edyn_wi,pmid_inv,op_out,opnm_out, &
                         i0,i1,j0,j1,nspltop,isplit)

      enddo ! isplit=1,nspltop
      write(iulog,"('dpie_coupling after subcycling oplus_xport: nstep=',i8,' nspltop=',i3)") &
        nstep,nspltop
      write(iulog,"('  op_out   min,max (cm^3)=',2es12.4)") minval(op_out)  ,maxval(op_out)
      write(iulog,"('  opnm_out min,max (cm^3)=',2es12.4)") minval(opnm_out),maxval(opnm_out)
    endif ! use_tiegcm_oplus
!
! Save transported O+ to edynamo.nc:
!
!   do j=j0,j1
!     call savefld_edyn('OPLUS','O+ (oplus_xport output)','cm^3', &
!       op_out(:,i0:i1,j),'lev',1,plev,'lon',i0,i1,j)
!   enddo
!
!
! Write to edyn netcdf output file (fields and coords are in "TIEGCM format"),
! every nstep_savefld_edyn timesteps (note d_pie_coupling is not called when 
! nstep==0)
!
!   if (mod(nstep,nstep_savefld_edyn)==0) then
!     call write_output
!   endif
!
! Load AMIE fields into pointers for TIE-GCM to WACCM longitude swap
!
    do i=i0,i1
      do j=j0,j1
   	amie_efxg(1,i,j) = efxg(i,j)
   	amie_kevg(1,i,j) = kevg(i,j)
      enddo
    enddo
!
! Convert ion drifts, O+ output, and AMIE fields from TIEGCM to WACCM format:
!
    nfields = 7 ! ui,vi,wi,op,opnm,efxg,kevg
    allocate(ptrs(nfields))
    ptrs(1)%ptr => edyn_ui ; ptrs(2)%ptr => edyn_vi ; ptrs(3)%ptr => edyn_wi
    ptrs(4)%ptr => op_out  ; ptrs(5)%ptr => opnm_out ; ptrs(6)%ptr => amie_efxg 
    ptrs(7)%ptr => amie_kevg
    call switch_model_format(ptrs,1,plev,i0,i1,j0,j1,nfields)
    deallocate(ptrs)
!
    if (use_tiegcm_oplus) then
      call savefld_waccm(op_out,'OPLUS',plev,i0,i1,j0,j1) ! cm^3
!
! Save electron density Ne for diagnostics (m^3):
      do k=1,plev
        do j=j0,j1
          do i=i0,i1
            edyn_ne(k,i,j) = op_out(k,i,j)*1.e6 + o2p(i,j,k) + nop(i,j,k) + n2p(i,j,k)
          enddo
        enddo
      enddo
      call savefld_waccm(edyn_ne,'WACCM_NE',plev,i0,i1,j0,j1) ! m^3
!
! Update O+ at time-1 (optm1 is module data in waccm format)
!
      do k=1,plev
        optm1(i0:i1,j0:j1,k) = opnm_out(k,i0:i1,j0:j1)*1.e6 ! cm^3 to m^3
      enddo
!
! Pass new O+ back to physics (convert from cm^3 to m^3 and back to mmr).
!
      do k=1,plev
        do j=j0,j1
          do i=i0,i1
            opmmr(i,j,k) = op_out(k,i,j)*1.e6_r8 * rmassOp / mbar(i,j,k) * &
              (kboltz * tn(i,j,k)) / pmid(i,k,j)
            op_out(k,i,j) = opmmr(i,j,k) ! for save to waccm hist in mmr
          enddo
        enddo
      enddo
!
! Save transported O+ to waccm history (mmr):
      call savefld_waccm(op_out,'WACCM_OP',plev,i0,i1,j0,j1) ! mmr
    endif ! use_tiegcm_oplus
!
! Convert ion drifts from cm/s to m/s for WACCM physics and history files.
!   real(r8),intent(inout) :: ui(i0:i1,j0:j1,plev)      ! zonal ion drift (edynamo or empirical)
!
    do k=1,plev
      do j=j0,j1
        do i=i0,i1
          ui(i,j,k) = edyn_ui(k,i,j)/100._r8
          vi(i,j,k) = edyn_vi(k,i,j)/100._r8
          wi(i,j,k) = edyn_wi(k,i,j)/100._r8
        enddo
      enddo
    enddo
    call savefld_waccm(edyn_ui/100._r8,'WACCM_UI',plev,i0,i1,j0,j1)
    call savefld_waccm(edyn_vi/100._r8,'WACCM_VI',plev,i0,i1,j0,j1)
    call savefld_waccm(edyn_wi/100._r8,'WACCM_WI',plev,i0,i1,j0,j1)
!    
!  Load up AMIE fields to pass back
!
    do i=i0,i1
      do j=j0,j1
   	efxg(i,j) = amie_efxg(plev,i,j)
   	kevg(i,j) = amie_kevg(plev,i,j)
      enddo
    enddo
  end subroutine d_pie_coupling
!-----------------------------------------------------------------------
  subroutine calc_wn(tn,omega,pmid,mbar,grav,wn,i0,i1,j0,j1,plev)
    use shr_const_mod,only : shr_const_rgas ! Universal gas constant
!
! Calculate neutral vertical wind on midpoints (m/s)
!
! Inputs:
    integer,intent(in) :: i0,i1,j0,j1,plev
    real(r8),dimension(i0:i1,j0:j1,plev),intent(in) :: &
      tn,   &  ! neutral temperature (deg K)
      mbar     ! mean molecular weight
    real(r8),dimension(i0:i1,plev,j0:j1),intent(in) :: &
      omega,&  ! pressure velocity (Pa/s)
      pmid     ! pressure at midpoints (Pa)
    real(r8),intent(in) :: grav ! m/s^2
!
! Output:
    real(r8),intent(out) :: wn(i0:i1,j0:j1,plev)    ! vertical velocity output (m/s)
!
! Local:
    integer :: i,j,k
    real(r8) :: scheight(i0:i1,j0:j1,plev) ! dimensioned for vectorization

    do k=1,plev
      do j=j0,j1
        do i=i0,i1
          scheight(i,j,k) = shr_const_rgas*tn(i,j,k)/(mbar(i,j,k)*grav)
          wn(i,j,k) = -omega(i,k,j)*scheight(i,j,k)/pmid(i,k,j)
        enddo
      enddo
    enddo
  end subroutine calc_wn
!-----------------------------------------------------------------------
  subroutine calc_pfrac(sunlon,ctpoten,pfrac)
!
! Calculate pfrac fractional presence of dynamo equation using critical
!  convection colatitudes crit(2).
!
    use edyn_maggrid  ,only: nmlonp1,ylonm,ylatm
    use edyn_solve    ,only: nmlat0
    use edyn_solve    ,only: crit
    use edyn_params   ,only: dtr
    implicit none
!
! Args:
    real(r8),intent(in) :: sunlon  ! Sun's longitude in dipole coordinates
    real(r8),intent(in) :: ctpoten ! Cross-cap potential
!
! Output: fractional presence of dynamo equation using critical colatitudes
!
    real(r8),intent(out) :: pfrac(nmlonp1,nmlat0) ! NH fraction of potential
!
! Local:
    integer :: j,i
    real(r8) :: offc(2),dskofc(2),theta0(2)
    real(r8),dimension(nmlonp1,nmlat0) :: colatc
    real(r8) :: sinlat,coslat,aslonc,ofdc,cosofc,sinofc 
!   real(r8) :: crit(2)  ! colatitude limits of high-lat potential (radians)

!   write(iulog,"('Enter calc_pfrac: sunlon=',es12.4,' ctpoten=',f8.2)") &
!     sunlon,ctpoten
!
! As in TIEGCM aurora:
    offc(:)   =  1._r8*dtr  ! offset of auroral circle coord from mag coords
    dskofc(:) =  0._r8
    theta0(:) = (-3.80_r8+8.48_r8*(ctpoten**0.1875_r8))*dtr
!
! Critical colatitudes:
!   crit      = (/0.261799387_r8, 0.523598775_r8/)
!    
! offc(2), dskofc(2) are for northern hemisphere aurora 
!
    ofdc = sqrt(offc(2)**2+dskofc(2)**2)
    cosofc = cos(ofdc)
    sinofc = sin(ofdc)
    aslonc = asin(dskofc(2)/ofdc)
!
! Define colatc with northern convection circle coordinates
!
    do j=1,nmlat0
      sinlat = sin(abs(ylatm(j+nmlat0-1)))
      coslat = cos(    ylatm(j+nmlat0-1))
      do i=1,nmlonp1
        colatc(i,j) = cos(ylonm(i)-sunlon+aslonc)
        colatc(i,j) = acos(cosofc*sinlat-sinofc*coslat*colatc(i,j))
      enddo ! i=1,nmlonp1
!
! Calculate fractional presence of dynamo equation at each northern
! hemisphere geomagnetic grid point. Output in pfrac(nmlonp1,nmlat0)
!
      do i=1,nmlonp1
        pfrac(i,j) = (colatc(i,j)-crit(1))/(crit(2)-crit(1))
        if (pfrac(i,j) < 0._r8) pfrac(i,j) = 0._r8
        if (pfrac(i,j) >= 1._r8) pfrac(i,j) = 1._r8
      enddo ! i=1,nmlonp1
!     write(iulog,"('calc_pfrac: j=',i3,' pfrac(:,j)=',/,(6e12.4))") &
!       j,pfrac(:,j)
    enddo ! j=1,nmlat0
  end subroutine calc_pfrac
!-----------------------------------------------------------------------
  subroutine calc_pfrac_amie(sunlon,pfrac,crit)
!
! Calculate pfrac fractional presence of dynamo equation using critical
!  convection colatitudes crit(2).
!
    use edyn_maggrid  ,only: nmlonp1,ylonm,ylatm
    use edyn_solve    ,only: nmlat0
    use edyn_params   ,only: dtr
    implicit none
!
! Args:
    real(r8),intent(in) :: sunlon  ! Sun's longitude in dipole coordinates
!   real(r8),intent(in) :: ctpoten ! Cross-cap potential
    real(r8),intent(in) :: crit(2)    ! Crit parameter from AMIE 
!
! Output: fractional presence of dynamo equation using critical colatitudes
!
    real(r8),intent(out) :: pfrac(nmlonp1,nmlat0) ! NH fraction of potential
!
! Local:
    integer :: j,i
    real(r8) :: offc(2),dskofc(2),theta0(2)
    real(r8),dimension(nmlonp1,nmlat0) :: colatc
    real(r8) :: sinlat,coslat,aslonc,ofdc,cosofc,sinofc 

!   write(iulog,"('Enter calc_pfrac: sunlon=',es12.4,' ctpoten=',f8.2)") &
!     sunlon,ctpoten
!
! As in TIEGCM aurora:
    offc(:)   =  1._r8*dtr  ! offset of auroral circle coord from mag coords
    dskofc(:) =  0._r8
!   theta0(:) = (-3.80_r8+8.48_r8*(ctpoten**0.1875_r8))*dtr
    theta0(:) = crit(:)
!
! Critical colatitudes:
!   crit      = (/0.261799387_r8, 0.523598775_r8/)
!    
! offc(2), dskofc(2) are for northern hemisphere(-3.80_r8+8.48_r8*(ctpoten**0.1875_r8))*dtr aurora 
!
    ofdc = sqrt(offc(2)**2+dskofc(2)**2)
    cosofc = cos(ofdc)
    sinofc = sin(ofdc)
    aslonc = asin(dskofc(2)/ofdc)
!
! Define colatc with northern convection circle coordinates
!
    do j=1,nmlat0
      sinlat = sin(abs(ylatm(j+nmlat0-1)))
      coslat = cos(    ylatm(j+nmlat0-1))
      do i=1,nmlonp1
        colatc(i,j) = cos(ylonm(i)-sunlon+aslonc)
        colatc(i,j) = acos(cosofc*sinlat-sinofc*coslat*colatc(i,j))
      enddo ! i=1,nmlonp1
!
! Calculate fractional presence of dynamo equation at each northern
! hemisphere geomagnetic grid point. Output in pfrac(nmlonp1,nmlat0)
!
      do i=1,nmlonp1
        pfrac(i,j) = (colatc(i,j)-crit(1))/(crit(2)-crit(1))
        if (pfrac(i,j) < 0._r8) pfrac(i,j) = 0._r8
        if (pfrac(i,j) >= 1._r8) pfrac(i,j) = 1._r8
      enddo ! i=1,nmlonp1
!     write(iulog,"('calc_pfrac: j=',i3,' pfrac(:,j)=',/,(6e12.4))") &
!       j,pfrac(:,j)
    enddo ! j=1,nmlat0
  end subroutine calc_pfrac_amie
!-----------------------------------------------------------------------
  subroutine sunloc(iday,secs,sunlons)
!
! Given day of year and ut, return sun's longitudes in dipole coordinates 
! in sunlons(nlat)
!
    use getapex       ,only: alonm     ! (nlonp1,0:nlatp1)
    use edyn_geogrid  ,only: nlon,nlat
    use edyn_params   ,only: pi
!
! Args:
    integer,intent(in)   :: iday          ! day of year
    real(r8),intent(in)  :: secs          ! ut in seconds
    real(r8),intent(out) :: sunlons(nlat) ! output
!
! Local:
    integer :: j,i,ii,isun,jsun
    real(r8) :: glats,glons,pisun,pjsun,sndlons,csdlons
    real(r8) :: dphi,dlamda
    real(r8),allocatable,save :: rlonm(:,:) ! (nlon+4,nlat)
    real(r8) :: r8_nlat, r8_nlon
    real(r8) :: r8_isun, r8_jsun

    logical,save :: first = .true.
!
! Sun's geographic coordinates:
    r8_nlat = real(nlat)
    r8_nlon = real(nlon)
    glats   = asin(.398749_r8*sin(2._r8*pi*(iday-80)/365._r8))
    glons   = pi*(1._r8-2._r8*secs/86400._r8)
    dphi    = pi/r8_nlat
    dlamda  = 2._r8*pi/r8_nlon

    if (first) then
      allocate(rlonm(nlon+4,nlat))
      do j=1,nlat
        do i=1,nlon
          ii = i+2
          rlonm(ii,j) = alonm(i,j)
        enddo
      enddo
      do i=1,2
        do j=1,nlat
          rlonm(i,j) = rlonm(i+nlon,j)
          rlonm (i+nlon+2,j) = rlonm (i+2,j)
        enddo
      enddo
    endif ! first call

    pisun = (glons+pi)/dlamda+1._r8
    pjsun = (glats+.5_r8*(pi-dphi))/dphi+1._r8
    isun = int(pisun)
    jsun = int(pjsun)
    r8_isun = real(isun)
    r8_jsun = real(jsun)
    pisun = pisun-r8_isun
    pjsun = pjsun-r8_jsun

    sndlons = (1._r8-pisun)*(1._r8-pjsun)*sin(rlonm(isun+2,jsun))+   &
               pisun*(1._r8-pjsun)       *sin(rlonm(isun+3,jsun))+   &
               pisun*pjsun               *sin(rlonm(isun+3,jsun+1))+ &
               (1._r8-pisun)*pjsun       *sin(rlonm(isun+2,jsun+1))
    csdlons = (1._r8-pisun)*(1._r8-pjsun)*cos(rlonm(isun+2,jsun))+   &
               pisun*(1._r8-pjsun)       *cos(rlonm(isun+3,jsun))+   &
               pisun*pjsun               *cos(rlonm(isun+3,jsun+1))+ &
               (1._r8-pisun)*pjsun       *cos(rlonm(isun+2,jsun+1))
    sunlons(1) = atan2(sndlons,csdlons)
    do j = 2,nlat
      sunlons(j) = sunlons(1)
    enddo

    if (first) first = .false.
  end subroutine sunloc
!-----------------------------------------------------------------------
end module dpie_coupling
