   module edyn_init
!
! Initialize edynamo and time3d.
!
  use shr_kind_mod   ,only: r8 => shr_kind_r8 ! 8-byte reals
  use cam_logfile    ,only: iulog
  use edyn_params    ,only: set_cons,pi
  use edyn_geogrid,only : nlon,nlat,nlev,nilev,glon,glat,zlev,zilev,&
                          nlonp1,nlonp2,nlatp1,jspole,jnpole,dlatg,dlong,&
                          ylatg,ylong,dphi,dlamda,cs,expz,zp
  use cam_abortutils   ,only: endrun
  use time3d_flds  ,only: time3d_alloc
  use time3d_grid  ,only: time3d_alloc_neutrals
  use filter_module,only: trigs,ifax
  use amie_module  ,only: init_amie

  implicit none
!
! Precedence of logical "use" flags:
!   use_dynamo: if false, use_time3d and use_time3d_integ are ignored.
!   use_time3d: if false, use_time3d_integ is ignored.
! Both use_dynamo and use_dynamo_drifts must be true for modules
!   ionosphere.F90 and iondrag.F90 to use ion drifts from edynamo or time3d
!
  logical :: use_dynamo=.true.         ! if true, call d_pie_coupling from dp_coupling
  logical :: use_time3d=.false.        ! if true, call time3d from d_pie_coupling
  logical :: use_time3d_integ=.true.   ! if true, edyn will use integrated conductances from t3d
  logical :: use_time3d_gcmsim=.false. ! if true, use empirical fields as input to time3d
                                       ! if false, use waccm fields as input to time3d
  logical :: time3d_use_edyn_geogrid=.false. ! if true, time3d will use edyn geogrid 
                                             ! (true does not work, must be false)
  logical :: time3d_use_edyn_maggrid=.true.  ! if true, time3d will use edyn maggrid
  logical :: use_time3d_output=.false.       ! if true, time3d output will be input to waccm
                                             ! (O+,Te,Ti) (does not affect use_time3d_integ)
!
! use_dynamo_drifts controls whether ionosphere/iondrag modules
! use ion drifts from edynamo or from time3d (both use_dynamo and 
! use_dynamo_drifts must be true for the drifts to be used in the 
! ionosphere code). If time3d is not called and use_dynamo_drifts==false,
! then empirical ion drifts calculated in exbdrift (physics) will be used.
!
  logical :: use_dynamo_drifts=.true. ! if true, use ion drifts from edynamo
                                      ! if false, use empirical drifts from exbdrift

  logical :: use_tiegcm_oplus=.true.  ! if true, call sub oplus (based on tiegcm oplus.F)
                                      ! (this and use_time3d are mutually exclusive)
!
! nstep_savefld_edyn is number of steps between writes to history files edynamo_nnn.nc 
!
  integer,parameter :: nstep_savefld_edyn = 12  ! hourly, if step=5 mins
! integer,parameter :: nstep_savefld_edyn = 3   ! every 15 minutes (step=5 min)
! integer,parameter :: nstep_savefld_edyn = 4   ! every 20 minutes (step=5 min)
! integer,parameter :: nstep_savefld_edyn = 6   ! every 30 minutes (step=5 min)
! integer,parameter :: nstep_savefld_edyn = 12  ! every hour (step=5 min)
! integer,parameter :: nstep_savefld_edyn = 15  ! every 15 minutes (step=1 min)
! integer,parameter :: nstep_savefld_edyn = 30  ! every 30 minutes (step=1 min)
! integer,parameter :: nstep_savefld_edyn = 1   ! every step
!
! nstep_savefld_t3d is number of steps between writes to global time3d history files
! (see module savefield_t3d)
!
  integer,parameter :: nstep_savefld_t3d = 12 ! hourly, if step=5 mins
! integer,parameter :: nstep_savefld_t3d = 3  ! every 15 minutes (step=5)
! integer,parameter :: nstep_savefld_t3d = 1
!
! Empirical model to call for high-latitude electric potential
! Must be either 'heelis' or 'weimer'
! This is ignored if use_dynamo_drifts=false (in this case, efield will
! call weimer96 model for high-latitude potential).
!
! character(len=16) :: highlat_potential_model = 'weimer'
  character(len=16) :: highlat_potential_model = 'heelis'
! character(len=16) :: highlat_potential_model = 'amie'
  character(len=256) :: amienh = '/glade/scratch/ganglu/tiegcm_amie/tiegcm_amie-ys/apr01_10_2010_nh_ssusi.nc'
  character(len=256) :: amiesh = '/glade/scratch/ganglu/tiegcm_amie/tiegcm_amie-ys/apr01_10_2010_sh_ssusi.nc'
  integer :: iamie
  real(r8),allocatable :: amie_efxg(:,:), amie_kevg(:,:) ! (nlonp1,0:nlatp1) (auroral energy flux)

  private
  public edynamo_init,use_dynamo,use_time3d,use_time3d_integ,use_dynamo_drifts,&
    lonshift_global,time3d_use_edyn_geogrid,time3d_use_edyn_maggrid,use_time3d_gcmsim,&
    use_time3d_output,nstep_savefld_edyn,nstep_savefld_t3d,use_tiegcm_oplus,&
    highlat_potential_model, iamie, amienh, amiesh, amie_efxg, amie_kevg


  contains
!-----------------------------------------------------------------------
  subroutine edynamo_init(dyn_state)
!
! One-time initialization, called from inital.F90 after dyn_init and initcom.
!
    use dynamics_vars ,only: t_fvdycore_state, t_fvdycore_grid
    use time_manager  ,only: get_curr_date
    use edyn_maggrid  ,only: set_maggrid
    use edyn_mpi      ,only: mp_init,mp_distribute_geo,mp_distribute_mag,&
                             mp_exchange_tasks
    use getapex       ,only: get_apex,magfield
    use edynamo       ,only: alloc_edyn,ed1_glb,ed2_glb
    use edyn_esmf     ,only: esmf_init  ! initialize ESMF 
    use edyn_output   ,only: output_init
    use time3d_gcmsim ,only: pregrid
!
! Args:
    type (t_fvdycore_state),target :: dyn_state
!
! Local:
    type (t_fvdycore_grid),pointer :: grid
    integer :: iyear,imo,iday,tod
    real(r8) :: epoch

    write(iulog,"('Enter edynamo_init:')")
    write(iulog,"('  use_dynamo            = ',l1)") use_dynamo
    write(iulog,"('  use_time3d            = ',l1)") use_time3d
    write(iulog,"('  use_time3d_integ      = ',l1)") use_time3d_integ
    write(iulog,"('  use_time3d_gcmsim     = ',l1)") use_time3d_gcmsim
    write(iulog,"('  use_time3d_output     = ',l1)") use_time3d_output
    write(iulog,"('  use_dynamo_drifts     = ',l1)") use_dynamo_drifts
    write(iulog,"('  use_tiegcm_oplus      = ',l1)") use_tiegcm_oplus
    write(iulog,"('  time3d_use_edyn_geogrid = ',l1)") time3d_use_edyn_geogrid
    write(iulog,"('  time3d_use_edyn_maggrid = ',l1)") time3d_use_edyn_maggrid
    write(iulog,"('  highlat_potential_model = ',a)") trim(highlat_potential_model)

    grid => dyn_state%grid

    call mp_init      ! get ntask,mytid
    call set_cons     ! edyn_params.F90
    call set_geogrid  ! set global geographic grid from dynamics (commap)
    call set_maggrid  ! set parameter-based global magnetic grid
!
! set time3d grids (either original time3d, or edynamo, see use flags above)
! (see also call to pregrid in time3d.F90)
    call pregrid(time3d_use_edyn_geogrid,time3d_use_edyn_maggrid)

    call mp_distribute_geo(grid)
    call mp_distribute_mag
    call register_maggrid
    call mp_exchange_tasks(0) ! single arg is iprint
!
! Get apex coordinates.
! subroutine get_apex(epoch,nlon,nlat,jspole,jnpole)
!
    call get_curr_date(iyear,imo,iday,tod)
    epoch = iyear

!   write(iulog,"('edynamo_init call get_apex: epoch=',f8.1,' nlon,nlat=',2i4,' jspole,jnpole=',2i4)") &
!     epoch,nlon,nlat,jspole,jnpole

    call get_apex(epoch,nlon,nlat,jspole,jnpole) ! get apex coordinates
    call magfield                                ! calculate magnetic field parameters

    call alloc_edyn      ! allocate dynamo arrays
    call esmf_init       ! initialize ESMF
    call add_fields      ! add fields to WACCM history master list
    call output_init     ! initialize fields for edynamo.nc history file

! SUBROUTINE SET999 (TRIGS, IFAX, ntrigs, N) ! called in TIEGCM (fft9.F)
!   call set999(trigs,ifax,ntrigs,plon)

! WACCM calls set99 (cam/src/fv/fft99.F90)
    call set99(trigs,ifax,nlon) ! initialize fft for O+ polar filtering
!   write(iulog,"('edyn_init after set99: nlon=',i4,' ifax=',13i4)") nlon,ifax
!   write(iulog,"('edyn_init after set99: trigs=',/,(6es12.4))") trigs

    if (use_time3d) then ! pregrid must be called above
      call time3d_alloc()
      call time3d_alloc_neutrals()
    endif
!
! Use AMIE output
!   if (trim(highlat_potential_model) == 'amie') then
    if (len_trim(amienh) > 5) then
      iamie = 1
      call init_amie(amienh,amiesh)
    endif
!
! Init for inputs to time3d (could go inside above conditional): 
    ed1_glb = 0._r8
    ed2_glb = 0._r8

  end subroutine edynamo_init
!-----------------------------------------------------------------------
  subroutine set_geogrid
    use pmgrid,    only : plon, plat, plev, plevp
    use commap,    only : &
      latdeg,             & ! global latitude (nlat)
      londeg                ! global longitude (nlon)
!   use edyn_solve,only: amie_efxg,amie_kevg
!   use amie_module,only: amie_efxg,amie_kevg

!    use hycoef,    only : & ! Hybrid level definitions:
!      alev,               & ! target alev(plev) midpoint levels coord
!      ailev                 ! target ailev(plevp) interface levels coord
    use ref_pres,  only : & ! Hybrid level definitions:
      pref_mid,           & ! target alev(plev) midpoint levels coord
      pref_edge             ! target ailev(plevp) interface levels coord

!   use physics_types,only: physics_state
!
! Local:
    integer :: i,j,js,k
    real(r8) :: real8,pi,phi
    real(r8),parameter :: eps = 1.e-6_r8
!   type(physics_state), pointer :: phys_state(:) => null()

    nlon = plon
    nlat = plat
    nlev = plev
    nilev = plevp

    nlonp1 = nlon+1
    nlonp2 = nlon+2
    nlatp1 = nlat+1

    jspole = 1
    jnpole = nlat

    pi  = 4._r8*atan(1._r8)
    dphi   = pi/float(plat)
    dlamda = 2._r8*pi/float(plon)

!   write(iulog,"('set_geogrid: nlon=',i4,' nlat=',i4,' nlev=',i4,' nilev=',i4)") &
!     nlon,nlat,nlev,nilev
!
! Allocate coordinate variables:
!
    allocate(glon(nlon))
    allocate(glat(nlat))
    allocate(zlev(nlev))
    allocate(zilev(nilev))
!
! zp and expz are not set until oplus is called from dpie_coupling.
    allocate(zp(nlev))      ! log pressure (as in TIEGCM)
    allocate(expz(nlev))    ! exp(-zp)
!
! Global lat x lon x lev grid from commap module:
! latdeg(plat), londeg(plon,plat)
!
    glat = latdeg
    glon = londeg(:,1) ! assumes longitudes are redundant in latitude
    call lonshift_global(glon,nlon,'-180to180',.true.) ! shift to +/-180
!
! Hybrid-sigma levels from hycoef module:
!
!    zlev(:)  = alev(:)   ! midpoints vertical coord (top down)
!    zilev(:) = ailev(:)  ! interfaces vertical coord
!
! Hybrid-sigma levels from ref_pres module:
!
    zlev(:)  = pref_mid(:)   ! midpoints vertical coord (top down)
    zilev(:) = pref_edge(:)  ! interfaces vertical coord
!
! Set horizontal geographic grid in radians (for apex code):
!
    allocate(ylatg(nlat))   ! waccm grid includes poles
    allocate(ylong(nlonp1)) ! single periodic point
    real8 = real(nlat) ; dlatg = pi/real8
    real8 = real(nlon) ; dlong = 2._r8*pi/real8
    ylatg(1)    = -pi/2._r8+eps ! south pole
    ylatg(nlat) =  pi/2._r8-eps ! north pole
    do j=2,nlat-1
      real8 = real(j-1)
      ylatg(j) = -0.5_r8*(pi-dlatg)+real8*dlatg
    enddo
    do i=1,nlonp1
      real8 = real(i-1)
      ylong(i) = -pi+real8*dlong
    enddo
!
! QUESTION: taking cs(0:nlat+1) from TIEGCM cons.F
!   Not sure if this will work for all resolutions of plat
!
    allocate(cs(0:nlat+1))
    js = -(nlat/2)
    do j=1,nlat
      phi = (j+js-.5_r8)*dphi
      cs(j) = cos(phi)
    enddo
    cs(0) = -cs(1)
    cs(nlat+1) = -cs(nlat)
!
! Allocate amie arrays on geographic grid:
    allocate(amie_efxg(nlonp1,0:nlatp1))
    allocate(amie_kevg(nlonp1,0:nlatp1))
!
!   write(iulog,"('set_geogrid: nlat=',i3,' glat=',/,(8f8.2))") nlat,glat
!   write(iulog,"('set_geogrid: nlon=',i3,' glon=',/,(8f8.2))") nlon,glon
!   write(iulog,"('set_geogrid: nlev=',i3,' zlev=',/,(6e12.4))") nlev,zlev
!   write(iulog,"('set_geogrid: nilev=',i3,' zilev=',/,(6e12.4))") nilev,zilev

!   write(iulog,"('set_geogrid: nlat=',i3,' ylatg (radians)=',/,(8f8.2))") nlat,ylatg
!   write(iulog,"('set_geogrid: nlon=',i3,' ylong (radians)=',/,(8f8.2))") nlon,ylong

!   write(iulog,"('set_geogrid: nlat=',i3,' cs(0:nlat+1)=',/,(8f8.2))") nlat,cs

  end subroutine set_geogrid
!-----------------------------------------------------------------------
  subroutine lonshift_global(f,nlon,lonseq,iscoord)
!
! Shift longitude vector f(nlon) forward 180 degrees according to input
! string lonseq. Input f can be either arbitrary field values or
! the coordinate array itself. Shift f in the 'lonseq' manner, as follows:
!
! If lonseq='-180to180', then shift from 0->360 to -180->+180
! If lonseq='zeroto360', then shift from -180->+180 to 0->360
!
! WARNING: This routine works with WACCM-X history files, where nlon=144, 72, or 80
!          It has not been tested with other models or resolutions.
!          (e.g., there is no test for center point, its assumed to be nlon/2)
!
! Args:
    integer,intent(in) :: nlon
    real(r8),intent(inout) :: f(nlon)
    character(len=*),intent(in) :: lonseq
    logical,intent(in) :: iscoord ! if true, f is a coordinate, otherwise it is data
!
! Local:
    character(len=80) :: msg
    integer :: ihalf,i

    if (lonseq /= '-180to180'.and.lonseq /= 'zeroto360') then
      write(msg,"('shift_lon: bad lonseq=',a,' must be either ''-180to180'' or ''zeroto360''')") &
        lonseq
      call endrun
    endif

    ihalf = nlon/2
    if (lonseq == '-180to180') then ! shift to -180 -> +180
      f = cshift(f,ihalf)           ! cshift is circular shift intrinsic
      if (iscoord) then
        do i=1,ihalf
          f(i) = f(i)-360._r8
        enddo
      endif
    else                           ! shift to 0 -> 360
      f = cshift(f,ihalf)          ! cshift is circular shift intrinsic
      if (iscoord) then
        do i=ihalf+1,nlon
          f(i) = f(i)+360._r8
        enddo
      endif
    endif
  end subroutine lonshift_global
!-----------------------------------------------------------------------
  subroutine add_fields
    use cam_history,  only: addfld, add_default, horiz_only
    use pmgrid,       only: plev
    use mo_jeuv,      only : nIonRates            ! Number of ionization rates (11 rates)

    integer :: m
    character(len=16) :: ionrate_names(nIonRates)
    character(len=64) :: ionrate_longnames(nIonRates) =        &
      (/"Ion Reaction Rates 01: O + hv --> O+ (4S) + e*     ", &    ! J1
        "Ion Reaction Rates 02: O + hv --> O+ (2D) + e*     ", &    ! J2
        "Ion Reaction Rates 03: O + hv --> O+ (2P) + e*     ", &    ! J3
        "Ion Reaction Rates 04: N (4S) + hv --> N+ + e*     ", &    ! J4
        "Ion Reaction Rates 05: O2 + hv --> O2+ + e*        ", &    ! J5
        "Ion Reaction Rates 06: N2 + hv --> N2+ + e*        ", &    ! J6
        "Ion Reaction Rates 07: O2 + hv --> O + O+(4S) + e* ", &    ! J7
        "Ion Reaction Rates 08: O2 + hv --> O + O+(2D) + e* ", &    ! J8
        "Ion Reaction Rates 09: O2 + hv --> O + O+(2P) + e* ", &    ! J9
        "Ion Reaction Rates 10: N2 + hv --> N (4S) + N+ + e*", &    ! J10
        "Ion Reaction Rates 11: N2 + hv --> N (2D) + N+ + e*"/)     ! J11

! Geomagnetic fields are in waccm format, in CGS units):
    call addfld ('PED_MAG'   ,(/ 'lev' /), 'I', 'S/m  ','Pedersen Conductivity'            ,gridname='gmag_grid')
    call addfld ('HAL_MAG'   ,(/ 'lev' /), 'I', 'S/m  ','Hall Conductivity'                ,gridname='gmag_grid')
    call addfld ('ZMAG'      ,(/ 'lev' /), 'I', 'cm   ','ZMAG: Geopotential'               ,gridname='gmag_grid')
    call addfld ('PHIM2D'    , horiz_only, 'I', 'VOLTS','PHIM2D: Electric Potential'       ,gridname='gmag_grid')
    call addfld ('ED1'       , horiz_only, 'I', 'V/m  ','ED1: Eastward Electric Field'     ,gridname='gmag_grid')
    call addfld ('ED2'       , horiz_only, 'I', 'V/m  ','ED2: Equatorward Electric Field'  ,gridname='gmag_grid')
    call addfld ('PHIM3D'    ,(/ 'lev' /), 'I', 'VOLTS','PHIM3D: 3d Electric Potential'    ,gridname='gmag_grid')
    call addfld ('ED13D'     ,(/ 'lev' /), 'I', 'V/m  ','ED13D: Eastward Electric Field'   ,gridname='gmag_grid')
    call addfld ('ED23D'     ,(/ 'lev' /), 'I', 'V/m  ','ED23D: Equatorward Electric Field',gridname='gmag_grid')
    call addfld ('ZPOT_MAG'  ,(/ 'lev' /), 'I', 'cm   ','Geopotential on mag grid (h0 min)',gridname='gmag_grid')
    call addfld ('ADOTV1_MAG',(/ 'lev' /), 'I', '     ','ADOTV1 on mag grid'               ,gridname='gmag_grid')
    call addfld ('ADOTV2_MAG',(/ 'lev' /), 'I', '     ','ADOTV2 on mag grid'               ,gridname='gmag_grid')
    call addfld ('phihm'     , horiz_only, 'I', 'VOLTS','AMIE Electric Potential-mag grid' ,gridname='gmag_grid')
    call addfld ('amie_efxm'  , horiz_only, 'I','mW/m2','AMIE energy flux on mag grid'     ,gridname='gmag_grid')
    call addfld ('amie_kevm'  , horiz_only, 'I','keV  ','AMIE mean energy on mag grid'     ,gridname='gmag_grid')
    call addfld ('amie_efxg'  , horiz_only, 'I','mW/m2','AMIE energy flux on geo grid'     ,gridname='fv_centers')
    call addfld ('amie_kevg'  , horiz_only, 'I','keV  ','AMIE mean energy on geo grid'     ,gridname='fv_centers')

! Dynamo inputs (called from dpie_coupling. Fields are in waccm format, in CGS units):
    call addfld ('DPIE_OMEGA',(/ 'lev' /), 'I', 'Pa/s    ','OMEGA input to DPIE coupling', gridname='fv_centers')
    call addfld ('DPIE_MBAR' ,(/ 'lev' /), 'I', '        ','MBAR Mean Mass from dpie_coupling', gridname='fv_centers')
    call addfld ('DPIE_TN   ',(/ 'lev' /), 'I', 'deg K   ','DPIE_TN'   , gridname='fv_centers')
    call addfld ('DPIE_UN   ',(/ 'lev' /), 'I', 'cm/s    ','DPIE_UN'   , gridname='fv_centers')
    call addfld ('DPIE_VN   ',(/ 'lev' /), 'I', 'cm/s    ','DPIE_VN'   , gridname='fv_centers')
    call addfld ('DPIE_WN   ',(/ 'lev' /), 'I', 'cm/s    ','DPIE_WN'   , gridname='fv_centers')
    call addfld ('DPIE_OM   ',(/ 'lev' /), 'I', 's-1     ','DPIE_OM'   , gridname='fv_centers')
    call addfld ('DPIE_ZHT  ',(/ 'lev' /), 'I', 'cm      ','DPIE_ZHT (geometric height)', gridname='fv_centers')
    call addfld ('DPIE_ZGI  ',(/ 'lev' /), 'I', 'cm      ','DPIE_ZGI (geopotential height on interfaces)', gridname='fv_centers')
    call addfld ('DPIE_BARM ',(/ 'lev' /), 'I', '        ','DPIE_BARM' , gridname='fv_centers')
    call addfld ('DPIE_PED  ',(/ 'lev' /), 'I', 'S/m     ','DPIE_PED'  , gridname='fv_centers')
    call addfld ('DPIE_HALL ',(/ 'lev' /), 'I', 'S/m     ','DPIE_HALL' , gridname='fv_centers')
    call addfld ('DPIE_UI   ',(/ 'lev' /), 'I','         ','DPIE_UI'   , gridname='fv_centers')
    call addfld ('DPIE_VI   ',(/ 'lev' /), 'I','         ','DPIE_VI'   , gridname='fv_centers')
    call addfld ('DPIE_WI   ',(/ 'lev' /), 'I','         ','DPIE_WI'   , gridname='fv_centers')
    call addfld ('DPIE_O2   ',(/ 'lev' /), 'I', 'mmr     ','DPIE_O2'   , gridname='fv_centers')
    call addfld ('DPIE_O    ',(/ 'lev' /), 'I', 'mmr     ','DPIE_O'    , gridname='fv_centers')
    call addfld ('DPIE_N2   ',(/ 'lev' /), 'I', 'mmr     ','DPIE_N2'   , gridname='fv_centers')
    call addfld ('DPIE_TE   ',(/ 'lev' /), 'I', 'deg K   ','DPIE_TE'   , gridname='fv_centers')
    call addfld ('DPIE_TI   ',(/ 'lev' /), 'I', 'deg K   ','DPIE_TI'   , gridname='fv_centers')
!
! Dynamo inputs from sub dynamo_input (edynamo.F90):
    call addfld ('EDYN_TN   ',(/ 'lev' /), 'I', 'deg K   ','EDYN_TN'   , gridname='fv_centers') 
    call addfld ('EDYN_UN   ',(/ 'lev' /), 'I', 'cm/s    ','EDYN_UN'   , gridname='fv_centers')
    call addfld ('EDYN_VN   ',(/ 'lev' /), 'I', 'cm/s    ','EDYN_VN'   , gridname='fv_centers')
    call addfld ('EDYN_OMG  ',(/ 'lev' /), 'I', 's-1     ','EDYN_OMG'  , gridname='fv_centers')
    call addfld ('EDYN_Z    ',(/ 'lev' /), 'I', 'cm      ','EDYN_ZHT'  , gridname='fv_centers')
    call addfld ('EDYN_BARM ',(/ 'lev' /), 'I', '        ','EDYN_MBAR' , gridname='fv_centers')
    call addfld ('EDYN_PED  ',(/ 'lev' /), 'I', 'S/m     ','EDYN_PED'  , gridname='fv_centers')
    call addfld ('EDYN_HALL ',(/ 'lev' /), 'I', 'S/m     ','EDYN_HALL' , gridname='fv_centers')

    call addfld ('EDYN_SCHT   ',(/ 'lev' /), 'I', '        ','EDYN_SCHT  ' , gridname='fv_centers')
    call addfld ('EDYN_WN     ',(/ 'lev' /), 'I', 'm/s     ','EDYN_WN    ' , gridname='fv_centers')
    call addfld ('EDYN_ADOTV1 ',(/ 'lev' /), 'I', '        ','EDYN_ADOTV1' , gridname='fv_centers')
    call addfld ('EDYN_ADOTV2 ',(/ 'lev' /), 'I', '        ','EDYN_ADOTV2' , gridname='fv_centers')
!
! 2d dynamo input fields on geo grid (edynamo.F90):
    call addfld ('EDYN_SINI   ', horiz_only , 'I', '        ','EDYN_SINI'   , gridname='fv_centers')
    call addfld ('EDYN_ADOTA1 ', horiz_only , 'I', '        ','EDYN_ADOTA1' , gridname='fv_centers')
    call addfld ('EDYN_ADOTA2 ', horiz_only , 'I', '        ','EDYN_ADOTA2' , gridname='fv_centers')
    call addfld ('EDYN_A1DTA2 ', horiz_only , 'I', '        ','EDYN_A1DTA2' , gridname='fv_centers')
    call addfld ('EDYN_BE3    ', horiz_only , 'I', '        ','EDYN_BE3'    , gridname='fv_centers')
!
! 3d ion drifts and 2d conductances at end of dpie_coupling 
! (from either edynamo or time3d):
!
!       call addfld ('TIME3D_ZIGM11',horiz_only,'I',' ','TIME3D_ZIGM11',gridname='gmag_grid)
!       call addfld ('TIME3D_ZIGM22',horiz_only,'I',' ','TIME3D_ZIGM22',gridname='gmag_grid)
!       call addfld ('TIME3D_ZIGMC' ,horiz_only,'I',' ','TIME3D_ZIGMC' ,gridname='gmag_grid)
!       call addfld ('TIME3D_ZIGM2' ,horiz_only,'I',' ','TIME3D_ZIGM2' ,gridname='gmag_grid)
!       call addfld ('TIME3D_RIM1'  ,horiz_only,'I',' ','TIME3D_RIM1'  ,gridname='gmag_grid)
!       call addfld ('TIME3D_RIM2'  ,horiz_only,'I',' ','TIME3D_RIM2'  ,gridname='gmag_grid)

!       call addfld ('TIME3D_UI',(/ 'lev' /),'I',' ','TIME3D_UI')
!       call addfld ('TIME3D_VI',(/ 'lev' /),'I',' ','TIME3D_VI')
!       call addfld ('TIME3D_WI',(/ 'lev' /),'I',' ','TIME3D_WI')

!       call addfld ('T3D_OP_2WACCM',(/ 'lev' /),'I',' ','T3D_OP_2WACCM')
!       call addfld ('DPIE_OP',(/ 'lev' /),'I',' ','DPIE_OP') ! this is also below

        call addfld ('QEP',(/ 'lev' /), 'I', 'm^3/s'   ,'Photo-Electron Production', gridname='fv_centers')
        call addfld ('QOP',(/ 'lev' /), 'I', 'm^3/s'   ,'O+ Production Rate'       , gridname='fv_centers')
        call addfld ('OpO2',(/ 'lev' /), 'I', 'cm^3/s' ,'Op+O2 Loss Rate'          , gridname='fv_centers')
        call addfld ('OpN2',(/ 'lev' /), 'I', 'cm^3/s' ,'Op+N2 Loss Rate'          , gridname='fv_centers')
        call addfld ('LOP',(/ 'lev' /), 'I', 'cm^3/s'  ,'O+ Loss Rate'             , gridname='fv_centers')
        call addfld ('SIGMA_PED' ,(/ 'lev' /), 'I', ' ','Pederson Conductivity'    , gridname='fv_centers')
        call addfld ('SIGMA_HALL',(/ 'lev' /), 'I', ' ','Hall Conductivity'        , gridname='fv_centers')
!
!       call addfld ('EDYN_ZIGM11', horiz_only, 'I', ' ','EDYN_ZIGM11',gridname='gmag_grid)
!       call addfld ('EDYN_ZIGM22', horiz_only, 'I', ' ','EDYN_ZIGM22',gridname='gmag_grid)
!       call addfld ('EDYN_ZIGMC' , horiz_only, 'I', ' ','EDYN_ZIGMC' ,gridname='gmag_grid)
!       call addfld ('EDYN_ZIGM2' , horiz_only, 'I', ' ','EDYN_ZIGM2' ,gridname='gmag_grid)
!       call addfld ('EDYN_RIM1'  , horiz_only, 'I', ' ','EDYN_RIM1'  ,gridname='gmag_grid)
!       call addfld ('EDYN_RIM2'  , horiz_only, 'I', ' ','EDYN_RIM2'  ,gridname='gmag_grid)

        call addfld ('EDYN_UI',(/ 'lev' /), 'I', 'cm/s','EDYN_UI', gridname='fv_centers')
        call addfld ('EDYN_VI',(/ 'lev' /), 'I', 'cm/s','EDYN_VI', gridname='fv_centers')
        call addfld ('EDYN_WI',(/ 'lev' /), 'I', 'cm/s','EDYN_WI', gridname='fv_centers')

        call addfld ('POTEN'  ,(/ 'lev' /), 'I', 'Volts','POTEN: Electric Potential',&
          gridname='fv_centers')
        call addfld ('EX'     ,(/ 'lev' /), 'I', 'V/m'  ,'EX: Zonal component of Electric Field',&
          gridname='fv_centers')
        call addfld ('EY'     ,(/ 'lev' /), 'I', 'V/m'  ,'EY: Meridional component of Electric Field',&
          gridname='fv_centers')
        call addfld ('EZ'     ,(/ 'lev' /), 'I', 'V/m'  ,'EZ: Vertical component of Electric Field',&
          gridname='fv_centers')
!
! Save fields from oplus module:
!
        call addfld ('OPLUS_Z'    ,(/ 'lev' /), 'I', 'cm   ','OPLUS_Z'     , gridname='fv_centers')
        call addfld ('OPLUS_TN'   ,(/ 'lev' /), 'I', 'deg K','OPLUS_TN'    , gridname='fv_centers')
        call addfld ('OPLUS_TE'   ,(/ 'lev' /), 'I', 'deg K','OPLUS_TE'    , gridname='fv_centers')
        call addfld ('OPLUS_TI'   ,(/ 'lev' /), 'I', 'deg K','OPLUS_TI'    , gridname='fv_centers')
        call addfld ('OPLUS_UN'   ,(/ 'lev' /), 'I', 'cm/s' ,'OPLUS_UN'    , gridname='fv_centers')
        call addfld ('OPLUS_VN'   ,(/ 'lev' /), 'I', 'cm/s' ,'OPLUS_VN'    , gridname='fv_centers')
        call addfld ('OPLUS_OM'   ,(/ 'lev' /), 'I', 'Pa/s' ,'OPLUS_OM'    , gridname='fv_centers')
        call addfld ('OPLUS_O2'   ,(/ 'lev' /), 'I', 'mmr'  ,'OPLUS_O2'    , gridname='fv_centers')
        call addfld ('OPLUS_O1'   ,(/ 'lev' /), 'I', 'mmr'  ,'OPLUS_O1'    , gridname='fv_centers') 

        call addfld ('OPLUS_N2'   ,(/ 'lev' /), 'I', 'mmr'  ,'OPLUS_N2'    , gridname='fv_centers')
        call addfld ('OPLUS_OP'   ,(/ 'lev' /), 'I', 'cm^3' ,'OPLUS_OP'    , gridname='fv_centers')
        call addfld ('OPLUS_UI'   ,(/ 'lev' /), 'I', 'm/s'  ,'OPLUS_UI'    , gridname='fv_centers')
        call addfld ('OPLUS_VI'   ,(/ 'lev' /), 'I', 'm/s'  ,'OPLUS_VI'    , gridname='fv_centers')
        call addfld ('OPLUS_WI'   ,(/ 'lev' /), 'I', 'm/s'  ,'OPLUS_WI'    , gridname='fv_centers')
        call addfld ('OPLUS_MBAR' ,(/ 'lev' /), 'I', ' '    ,'OPLUS_MBAR'  , gridname='fv_centers')
        call addfld ('OPLUS_TR'   ,(/ 'lev' /), 'I', ' '    ,'OPLUS_TR'    , gridname='fv_centers')
        call addfld ('OPLUS_TP0'  ,(/ 'lev' /), 'I', ' '    ,'OPLUS_TP0'   , gridname='fv_centers')
        call addfld ('OPLUS_TP1'  ,(/ 'lev' /), 'I', ' '    ,'OPLUS_TP1'   , gridname='fv_centers')
!       call addfld ('OPLUS_TP2'  ,(/ 'lev' /), 'I', ' '    ,'OPLUS_TP2'   , gridname='fv_centers')
        call addfld ('OPLUS_DJ'   ,(/ 'lev' /), 'I', ' '    ,'OPLUS_DJ'    , gridname='fv_centers')
        call addfld ('OPLUS_HJ'   ,(/ 'lev' /), 'I', ' '    ,'OPLUS_HJ'    , gridname='fv_centers')
        call addfld ('OPLUS_BVEL' ,(/ 'lev' /), 'I', ' '    ,'OPLUS_BVEL'  , gridname='fv_centers')
        call addfld ('OPLUS_DIFFJ',(/ 'lev' /), 'I', ' '    ,'OPLUS_DIFFJ' , gridname='fv_centers')
        call addfld ('OPLUS_OPNM' ,(/ 'lev' /), 'I', ' '    ,'OPLUS_OPNM'  , gridname='fv_centers')
        call addfld ('OPNM_SMOOTH',(/ 'lev' /), 'I', ' '    ,'OPNM_SMOOTH' , gridname='fv_centers')
        call addfld ('BDOTDH_OP'  ,(/ 'lev' /), 'I', ' '    ,'BDOTDH_OP'   , gridname='fv_centers')
        call addfld ('BDOTDH_OPJ' ,(/ 'lev' /), 'I', ' '    ,'BDOTDH_OPJ'  , gridname='fv_centers')
        call addfld ('BDOTDH_DIFF',(/ 'lev' /), 'I', ' '    ,'BDOTDH_DIFF' , gridname='fv_centers')
        call addfld ('BDZDVB_OP'  ,(/ 'lev' /), 'I', ' '    ,'BDZDVB_OP'   , gridname='fv_centers') 
        call addfld ('EXPLICIT0'  ,(/ 'lev' /), 'I', ' '    ,'EXPLICIT0'   , gridname='fv_centers')

        call addfld ('EXPLICITa'  ,(/ 'lev' /), 'I', ' '    ,'EXPLICITa'   , gridname='fv_centers') ! part a
        call addfld ('EXPLICITb'  ,(/ 'lev' /), 'I', ' '    ,'EXPLICITb'   , gridname='fv_centers') ! part b
        call addfld ('EXPLICIT1'  ,(/ 'lev' /), 'I', ' '    ,'EXPLICIT1'   , gridname='fv_centers') ! complete
        call addfld ('EXPLICIT'   ,(/ 'lev' /), 'I', ' '    ,'EXPLICIT'    , gridname='fv_centers') ! final w/ poles

        call addfld ('EXPLICIT2'  ,(/ 'lev' /), 'I', ' '    ,'EXPLICIT2'   , gridname='fv_centers')
        call addfld ('EXPLICIT3'  ,(/ 'lev' /), 'I', ' '    ,'EXPLICIT3'   , gridname='fv_centers')
        call addfld ('TPHDZ0'     ,(/ 'lev' /), 'I', ' '    ,'TPHDZ0'      , gridname='fv_centers')
        call addfld ('TPHDZ1'     ,(/ 'lev' /), 'I', ' '    ,'TPHDZ1'      , gridname='fv_centers')
        call addfld ('DIVBZ'      ,(/ 'lev' /), 'I', ' '    ,'DIVBZ'       , gridname='fv_centers')
        call addfld ('HDZMBZ'     ,(/ 'lev' /), 'I', ' '    ,'HDZMBZ'      , gridname='fv_centers')
        call addfld ('HDZPBZ'     ,(/ 'lev' /), 'I', ' '    ,'HDZPBZ'      , gridname='fv_centers')
        call addfld ('P_COEFF0'   ,(/ 'lev' /), 'I', ' '    ,'P_COEFF0'    , gridname='fv_centers')
        call addfld ('Q_COEFF0'   ,(/ 'lev' /), 'I', ' '    ,'Q_COEFF0'    , gridname='fv_centers')
        call addfld ('R_COEFF0'   ,(/ 'lev' /), 'I', ' '    ,'R_COEFF0'    , gridname='fv_centers')
        call addfld ('P_COEFF0a'  ,(/ 'lev' /), 'I', ' '    ,'P_COEFF0a'   , gridname='fv_centers')
        call addfld ('Q_COEFF0a'  ,(/ 'lev' /), 'I', ' '    ,'Q_COEFF0a'   , gridname='fv_centers')
        call addfld ('DJINT'      ,(/ 'lev' /), 'I', ' '    ,'DJINT'       , gridname='fv_centers')
        call addfld ('BDOTU'      ,(/ 'lev' /), 'I', ' '    ,'BDOTU'       , gridname='fv_centers')
        call addfld ('R_COEFF0a'  ,(/ 'lev' /), 'I', ' '    ,'R_COEFF0a'   , gridname='fv_centers')
        call addfld ('Q_COEFF1'   ,(/ 'lev' /), 'I', ' '    ,'Q_COEFF1'    , gridname='fv_centers')
        call addfld ('R_COEFF1'   ,(/ 'lev' /), 'I', ' '    ,'R_COEFF1'    , gridname='fv_centers')
        call addfld ('P_COEFF2'   ,(/ 'lev' /), 'I', ' '    ,'P_COEFF2'    , gridname='fv_centers')
        call addfld ('Q_COEFF2'   ,(/ 'lev' /), 'I', ' '    ,'Q_COEFF2'    , gridname='fv_centers')
        call addfld ('R_COEFF2'   ,(/ 'lev' /), 'I', ' '    ,'R_COEFF2'    , gridname='fv_centers')

        call addfld ('P_COEFF'    ,(/ 'lev' /), 'I', ' '    ,'P_COEFF'     , gridname='fv_centers') ! final w/ poles
        call addfld ('Q_COEFF'    ,(/ 'lev' /), 'I', ' '    ,'Q_COEFF'     , gridname='fv_centers') ! final w/ poles
        call addfld ('R_COEFF'    ,(/ 'lev' /), 'I', ' '    ,'R_COEFF'     , gridname='fv_centers') ! final w/ poles

        call addfld ('OP_SOLVE'   ,(/ 'lev' /), 'I', ' '    ,'OP_SOLVE'    , gridname='fv_centers')
        call addfld ('OP_OUT'     ,(/ 'lev' /), 'I', 'cm^3' ,'OPLUS (oplus_xport output)', gridname='fv_centers')
        call addfld ('OPNM_OUT'   ,(/ 'lev' /), 'I', 'cm^3' ,'OPNM_OUT'    , gridname='fv_centers')
!       call addfld ('DPIE_OP'    ,(/ 'lev' /), 'I', 'cm^3' ,'DPIE_OP'     , gridname='fv_centers')
        call addfld ('DPIE_OPMMR' ,(/ 'lev' /), 'I', 'mmr'  ,'DPIE_OPMMR'  , gridname='fv_centers')
        call addfld ('BMOD2'      ,(/ 'lev' /), 'I', ' '    ,'BMOD2'       , gridname='fv_centers')

        call addfld ('WACCM_OP'   ,(/ 'lev' /), 'I', 'mmr'  ,'WACCM_OP (dpie output)', gridname='fv_centers')
        call addfld ('WACCM_UI'   ,(/ 'lev' /), 'I', 'm/s'  ,'WACCM_UI (dpie output)', gridname='fv_centers')
        call addfld ('WACCM_VI'   ,(/ 'lev' /), 'I', 'm/s'  ,'WACCM_VI (dpie output)', gridname='fv_centers')
        call addfld ('WACCM_WI'   ,(/ 'lev' /), 'I', 'm/s'  ,'WACCM_WI (dpie output)', gridname='fv_centers')
        call addfld ('WACCM_NE'   ,(/ 'lev' /), 'I', 'm^3'  ,'WACCM_NE (dpie output)', gridname='fv_centers')

        call addfld ('OPLUS_FLUX', horiz_only , 'I', ' ','OPLUS_FLUX', gridname='fv_centers')
        call addfld ('OPLUS_DIVB', horiz_only , 'I', ' ','OPLUS_DIVB', gridname='fv_centers')
        call addfld ('OPLUS_BX'  , horiz_only , 'I', ' ','OPLUS_BX'  , gridname='fv_centers')
        call addfld ('OPLUS_BY'  , horiz_only , 'I', ' ','OPLUS_BY'  , gridname='fv_centers')
        call addfld ('OPLUS_BZ'  , horiz_only , 'I', ' ','OPLUS_BZ'  , gridname='fv_centers')
        call addfld ('OPLUS_BMAG', horiz_only , 'I', ' ','OPLUS_BMAG', gridname='fv_centers')

        call addfld ('DPIE_O2P',(/ 'lev' /), 'I', 'm^3','DPIE_O2P(dpie input)', gridname='fv_centers')
        call addfld ('DPIE_NOP',(/ 'lev' /), 'I', 'm^3','DPIE_NOP(dpie input)', gridname='fv_centers')
        call addfld ('DPIE_N2P',(/ 'lev' /), 'I', 'm^3','DPIE_N2P(dpie input)', gridname='fv_centers')
        call addfld ('DPIE_OP' ,(/ 'lev' /), 'I', 'm^3','DPIE_OP (dpie input)', gridname='fv_centers')
        call addfld ('DPIE_NE' ,(/ 'lev' /), 'I', 'm^3','DPIE_NE (sum of O2+,NO+,N2+,O+)', gridname='fv_centers')
        call addfld ('OPLUS'   ,(/ 'lev' /), 'I', 'cm^3','OPLUS (oplus_xport output)'    , gridname='fv_centers')

    ionrate_names = ' '
    do m=1,nIonRates
      write(ionrate_names(m),"('IonRates-',i2.2)") m
      call addfld (ionrate_names(m) ,(/ 'lev' /), 'I', 'm^3     ',trim(ionrate_longnames(m)), gridname='fv_centers')
    enddo

    call addfld ('ZEDYN360   ' ,(/ 'lev' /), 'I', 'm     ','Geopotential 0 to 360 lon grid', gridname='fv_centers')
    call addfld ('ZEDYN180   ',(/ 'lev' /), 'I', 'm     ','Geopotential -180 to 180 lon grid', gridname='fv_centers')

  end subroutine add_fields
!-----------------------------------------------------------------------

  subroutine register_maggrid

  use cam_grid_support, only: horiz_coord_t, horiz_coord_create, iMap, cam_grid_register
  use edyn_mpi,         only: mlat0,mlat1,mlon0,omlon1
  use edyn_maggrid,     only: gmlat, gmlon, nmlat, nmlon
  integer, parameter :: mag_decomp         = 111 !arbitrary value

  type(horiz_coord_t), pointer :: lat_coord
  type(horiz_coord_t), pointer :: lon_coord
  integer(iMap),       pointer :: grid_map(:,:)
  integer(iMap),       pointer :: coord_map(:)
  integer                      :: i,j,ind

  allocate(grid_map(4, ((omlon1 - mlon0 + 1) * (mlat1 - mlat0 + 1))))
  ind = 0
  do i = mlat0, mlat1
    do j = mlon0, omlon1
      ind = ind + 1
      grid_map(1, ind) = j
      grid_map(2, ind) = i
      grid_map(3, ind) = j
      grid_map(4, ind) = i
    end do
  end do

  allocate(coord_map(mlat1 - mlat0 + 1))
  coord_map = (/ (i, i = mlat0, mlat1) /)
  lat_coord => horiz_coord_create('mlat', '', nmlat, 'latitude',                &
       'degrees_north', mlat0, mlat1, gmlat(mlat0:mlat1),                      &
       map=coord_map)
  nullify(coord_map)

    allocate(coord_map(omlon1 - mlon0 + 1))
    coord_map = (/ (i, i = mlon0, omlon1) /)
    lon_coord => horiz_coord_create('mlon', '', nmlon, 'longitude',             &
         'degrees_east', mlon0, omlon1, gmlon(mlon0:omlon1),                   &
         map=coord_map)
    deallocate(coord_map)
    nullify(coord_map)

  call cam_grid_register('gmag_grid', mag_decomp, lat_coord, lon_coord,      &
       grid_map, unstruct=.false.)
  nullify(grid_map)

  end subroutine register_maggrid

!-----------------------------------------------------------------------
end module edyn_init
