module edyn_output
!
! Write fields in "TIEGCM format" to netcdf history files.
! See also savefield_edyn.F90
!
  use shr_kind_mod  ,only: r8 => shr_kind_r8
  use cam_logfile   ,only: iulog
  use edyn_mpi      ,only: mytid
  use edyn_geogrid  ,only: nlat,nlev,nilev,nlon,glat
  use edyn_maggrid  ,only: nmlat,nmlev,nmlon,nmlonp1,gmlat,gmlon
  use time_manager  ,only: get_curr_date
  use mag_parms     ,only: get_mag_parms
  use mo_solar_parms,only: solar_parms_get
  use cam_abortutils,only: endrun
  use commap       ,only: latdeg,londeg
! use hycoef       ,only: alev,ailev
  use ref_pres     ,only: pref_mid, pref_edge 
  use spmd_utils   ,only: masterproc
  use netcdf
  implicit none
  save
  private

  real(r8),allocatable :: geolat(:),geolon(:),zlev(:),zilev(:)
  real(r8),allocatable :: maglat(:),maglon(:),zmlev(:)

  integer,parameter :: max_fields = 200      ! max number of fields
  integer,parameter :: max_chars = 1024      ! max char length (e.g., long_name)
  real(r8),parameter :: fillvalue = 1.e36_r8 ! default fill value

  character(len=max_chars) :: base_filename='edynamo_'
  integer,parameter :: edyn_nfill=6 ! number of histories to fill a file
!
! Define field structure for edynamo output:
!
  type field
    character(len=max_chars) :: name
    character(len=max_chars) :: long_name
    character(len=max_chars) :: units
    character(len=8)         :: dimnames(3)
    integer                  :: dimsizes(3)
    integer                  :: ndims
    logical                  :: geo,mag
    logical                  :: task0_only
    real(r8)                 :: fillvalue
    real(r8),pointer         :: data(:,:,:)
  end type field
  type(field) :: fields_out(max_fields)

  integer :: &
    idlon,idlat,idlev,idilev,             & ! geographic dimension ids
    idmlon,idmlat,idmlev,                 & ! magnetic dimension ids
    idunlim,idmtime                         ! time dimension ids
  integer :: &
    idv_lon,idv_lat,idv_lev,idv_ilev,idv_levsp, & ! geographic coord var ids
    idv_mlon,idv_mlat,idv_mlev,                 & ! magnetic coord var ids
    idv_time,idv_mtime,idv_year,idv_day,idv_ut, & ! time variable ids
    idv_nsteph,idv_ctpoten,idv_date,idv_datesec,&
    idv_ndcur,idv_nscur,idv_f107,idv_f107a,idv_kp,idv_ap,idv_hp

  character(len=1024) :: msg ! error message
  character(len=8) :: model_name="TIMEGCM "
  integer :: nfields,mtime(3)
  real(r8) :: ctpoten  ! Cross-tail potential from get_mag_parms method
  real(r8),parameter :: p0=5.e-5  ! ref pressure used to convert vertical to zp
  public output_init,write_output,max_fields,fields_out,nfields,fillvalue
  contains
!-----------------------------------------------------------------------
  subroutine output_init
!
! Local:
    integer :: n

    do n=1,max_fields
      call field_init(fields_out(n))
    enddo
  end subroutine output_init
!-----------------------------------------------------------------------
  subroutine field_init(f)
! 
! Args:
    type(field),intent(out) :: f
    
    f%name       = ' '
    f%long_name  = ' '  
    f%units      = ' '    
    f%dimnames   = ' '
    f%dimsizes   = 0
    f%ndims      = 0
    f%geo        = .false.
    f%mag        = .true.
    f%task0_only = .false.
    f%data       => NULL()
    f%fillvalue  = fillvalue
  
  end subroutine field_init
!-----------------------------------------------------------------------
  subroutine write_output
!
! Write a history for current timestep to output file:
!
! Local:
    integer :: istat,n
    integer,save :: ncid            ! netcdf file id
    logical,save :: first = .true.  ! true if first call
    character(len=max_chars),save :: ncfile ! output file path
    integer,save :: nfiles=1 ! number of current file (flnm_001.nc, flnm_002.nc, etc)
    integer,save :: nhist=0  ! number of histories on current file
!
! If first call, master task creates a new dataset and defines dimensions, 
! variables, etc. If not the first call, reopen existing dataset for appending.
! If current file was filled by previous write, open new file in the series.
!
    if (masterproc) then
      if (first) then
        write(ncfile,"(a,i3.3,'.nc')") trim(base_filename),nfiles
        istat = nf90_create(ncfile,NF90_CLOBBER,ncid)
        if (istat /= NF90_NOERR) then
          write(msg,"('Error opening output ncfile ',a)") trim(ncfile)
          call handle_ncerr(istat,msg,1)
        else
          write(iulog,"('Created output file ',a,' nfiles=',i3)") trim(ncfile),nfiles
        endif
        call define_ncfile(ncid)
!
! Current file is full -> create new file.
!
      elseif (nhist == edyn_nfill) then ! previous write filled current file
        nfiles = nfiles+1
        if (nfiles > 999) call endrun('>>> write_output: too many edynamo history files.')
        write(ncfile,"(a,i3.3,'.nc')") trim(base_filename),nfiles
        istat = nf90_create(ncfile,NF90_CLOBBER,ncid)
        if (istat /= NF90_NOERR) then
          write(msg,"('Error opening output ncfile ',a)") trim(ncfile)
          call handle_ncerr(istat,msg,1)
        else
          write(iulog,"('Created output file ',a,' nfiles=',i3)") trim(ncfile),nfiles
        endif
        call define_ncfile(ncid)
        nhist = 0
      else           ! reopen for appending new history
        istat = nf90_open(ncfile,NF90_WRITE,ncid)
      endif
    endif ! masterproc
!
! Gather fields to root task and write to file:
!
    call gather_fields ! everybody calls gather_fields
!
! Write fields to output file, and close the file (masterproc only):
!
    if (masterproc) then
      nhist = nhist+1
      call write_fields(ncid,nhist==1)
      istat = nf90_close(ncid)
!     write(iulog,"('Closed output file ',a,' nfiles=',i3,' nhist=',i3,' edyn_nfill=',i3)") &
!       trim(ncfile),nfiles,nhist,edyn_nfill
    endif
    first = .false.
  end subroutine write_output
!-----------------------------------------------------------------------
  subroutine define_ncfile(ncid)
!
! Define dimensions and coordinate variables on netcdf output file.
! This is called once per run from write_output.
! Data fields will be defined in the first call to write_fields.
!
! Args:
    integer,intent(in) :: ncid
!
! Local:
    integer :: n,k,istat,j
!
! Define dimensions:
!
    istat = nf90_def_dim(ncid,'time',NF90_UNLIMITED,idunlim)

    istat = nf90_def_dim(ncid,'lon' ,nlon,  idlon)
    istat = nf90_def_dim(ncid,'lat' ,nlat,  idlat)

    istat = nf90_def_dim(ncid,'lev' ,nlev  ,idlev)
    istat = nf90_def_dim(ncid,'ilev',nilev ,idilev)

    istat = nf90_def_dim(ncid,'mlon' ,nmlon  ,idmlon)
    istat = nf90_def_dim(ncid,'mlat' ,nmlat  ,idmlat)
    istat = nf90_def_dim(ncid,'mlev' ,nmlev  ,idmlev)

    istat = nf90_def_dim(ncid,'mtimedim',3,  idmtime)
!
! Define coordinates and other vars:
!
    istat = nf90_def_var(ncid,'lon',NF90_REAL8,(/idlon/),idv_lon)
    istat = nf90_put_att(ncid,idv_lon,"long_name","geographic longitude (-west, +east)")
    istat = nf90_put_att(ncid,idv_lon,"units","degrees_east")

    istat = nf90_def_var(ncid,'lat',NF90_REAL8,(/idlat/),idv_lat)
    istat = nf90_put_att(ncid,idv_lat,"long_name","geographic latitude (-south, +north)")
    istat = nf90_put_att(ncid,idv_lat,"units","degrees_north")

    istat = nf90_def_var(ncid,'lev',NF90_REAL8,(/idlev/),idv_lev)
    istat = nf90_put_att(ncid,idv_lev,"long_name","midpoint levels")
    istat = nf90_put_att(ncid,idv_lev,"units","")
    istat = nf90_put_att(ncid,idv_lev,"positive","up")
    istat = nf90_put_att(ncid,idv_lev,"standard_name","atmosphere_ln_pressure_coordinate")
    istat = nf90_put_att(ncid,idv_lev,"formula_terms","p0:p0 lev:lev")
    istat = nf90_put_att(ncid,idv_lev,"formula","p(k) = p0 * exp(-lev(k))")

    istat = nf90_def_var(ncid,'ilev',NF90_REAL8,(/idilev/),idv_ilev)
    istat = nf90_put_att(ncid,idv_ilev,"long_name","interface levels")
    istat = nf90_put_att(ncid,idv_ilev,"units","")
    istat = nf90_put_att(ncid,idv_ilev,"positive","up")
    istat = nf90_put_att(ncid,idv_ilev,"standard_name","atmosphere_ln_pressure_coordinate")
    istat = nf90_put_att(ncid,idv_ilev,"formula_terms","p0:p0 lev:lev")
    istat = nf90_put_att(ncid,idv_ilev,"formula","p(k) = p0 * exp(-lev(k))")

    istat = nf90_def_var(ncid,'levsp',NF90_REAL8,(/idlev/),idv_levsp)
    istat = nf90_put_att(ncid,idv_levsp,"long_name","midpoint sigma pressure levels")
    istat = nf90_put_att(ncid,idv_levsp,"units","hPa")
    istat = nf90_put_att(ncid,idv_levsp,"standard_name","atmosphere_sigma_pressure_coordinate")

    istat = nf90_def_var(ncid,'mlon',NF90_REAL8,(/idmlon/),idv_mlon)
    istat = nf90_put_att(ncid,idv_mlon,"long_name","magnetic longitude (-west, +east)")
    istat = nf90_put_att(ncid,idv_mlon,"units","degrees_east")

    istat = nf90_def_var(ncid,'mlat',NF90_REAL8,(/idmlat/),idv_mlat)
    istat = nf90_put_att(ncid,idv_mlat,"long_name","magnetic latitude (-south, +north)")
    istat = nf90_put_att(ncid,idv_mlat,"units","degrees_north")

    istat = nf90_def_var(ncid,'mlev',NF90_REAL8,(/idmlev/),idv_mlev)
    istat = nf90_put_att(ncid,idv_mlev,"long_name","magnetic midpoint levels")
    istat = nf90_put_att(ncid,idv_mlev,"units","")
    istat = nf90_put_att(ncid,idv_mlev,"positive","up")
    istat = nf90_put_att(ncid,idv_mlev,"standard_name","atmosphere_ln_pressure_coordinate")
    istat = nf90_put_att(ncid,idv_mlev,"formula_terms","p0:p0 mlev:mlev")
    istat = nf90_put_att(ncid,idv_mlev,"formula","p(k) = p0 * exp(-mlev(k))")

    istat = nf90_def_var(ncid,'mtime',NF90_INT,(/idmtime,idunlim/),idv_mtime)
    istat = nf90_put_att(ncid,idv_mtime,"long_name","model times (day, hour, minute)")
    istat = nf90_put_att(ncid,idv_mtime,"units","day, hour, minute")

    istat = nf90_def_var(ncid,'time',NF90_REAL8,(/idunlim/),idv_time)
    istat = nf90_put_att(ncid,idv_time,"long_name","model time")
    istat = nf90_put_att(ncid,idv_time,"units","decimal julian day")

    istat = nf90_def_var(ncid,'year',NF90_INT,(/idunlim/),idv_year)
    istat = nf90_put_att(ncid,idv_year,"long_name","calendar year")

    istat = nf90_def_var(ncid,'day',NF90_INT,(/idunlim/),idv_day)
    istat = nf90_put_att(ncid,idv_day,"long_name","calendar day")

    istat = nf90_def_var(ncid,'ut',NF90_REAL8,(/idunlim/),idv_ut)
    istat = nf90_put_att(ncid,idv_ut,"long_name",&
      "universal time (from model time hour and minute)")
    istat = nf90_put_att(ncid,idv_ut,"units","hours")

    istat = nf90_def_var(ncid,'ctpoten',NF90_REAL8,(/idunlim/),idv_ctpoten)
    istat = nf90_put_att(ncid,idv_ctpoten,"long_name","Cross-tail potential")
    istat = nf90_put_att(ncid,idv_ctpoten,"units","Volts")

    istat = nf90_def_var(ncid,'f107' ,NF90_REAL8,(/idunlim/),idv_f107)
    istat = nf90_put_att(ncid,idv_f107,"long_name",&
      "10.7 cm solar radio flux (F10.7)")
    istat = nf90_put_att(ncid,idv_f107,"units","10^-22 W m^-2 Hz^-1")

    istat = nf90_def_var(ncid,'f107a',NF90_REAL8,(/idunlim/),idv_f107a)
    istat = nf90_put_att(ncid,idv_f107a,"long_name",&
      "81-day centered mean of 10.7 cm solar radio flux (F10.7)")
    istat = nf90_put_att(ncid,idv_f107a,"units","10^-22 W m^-2 Hz^-1")

    istat = nf90_def_var(ncid,'kp',NF90_REAL8,(/idunlim/),idv_kp)
    istat = nf90_put_att(ncid,idv_kp,"long_name",&
      "Daily planetary K geomagnetic index")

    istat = nf90_def_var(ncid,'ap',NF90_REAL8,(/idunlim/),idv_ap)
    istat = nf90_put_att(ncid,idv_ap,"long_name",&
      "Daily planetary A geomagnetic index")

    istat = nf90_def_var(ncid,'hp',NF90_REAL8,(/idunlim/),idv_hp)
    istat = nf90_put_att(ncid,idv_hp,"long_name","Hemispheric Power")
    istat = nf90_put_att(ncid,idv_hp,"units","GW")
!
! Take out of define mode:
!
    istat = nf90_enddef(ncid)
!
! Coordinate vars for edyn history files are in "TIMEGCM format"
! (i.e., lon is +/-180, column k=1 is at the bottom, k=nlev is at the top)
!
! Set geographic coordinates:
!
    if (.not.allocated(geolon)) allocate(geolon(nlon))
    if (.not.allocated(geolat)) allocate(geolat(nlat))
    if (.not.allocated(zlev))   allocate(zlev(nlev))
    if (.not.allocated(zilev))  allocate(zilev(nilev))

    geolat = glat                 ! same as waccm (includes poles)
    geolon(:) = londeg(:,1)       ! first set to waccm longitudes (0->360)
!
! Shift longitude coordinate to TIMEGCM format (data is already shifted):
    call lonshift_global(geolon,nlon,'-180to180',.true.) ! shift to +/-180
!
! Invert vertical coordinate to TIMEGCM format (data already inverted):
!    zlev(:)  = alev(:)            ! waccm midpoints vertical coord (top down)
!    zilev(:) = ailev(:)           ! waccm interfaces vertical coord
    zlev(:)  = pref_mid(:)        ! waccm midpoints vertical coord (top down)
    zilev(:) = pref_edge(:)       ! waccm interfaces vertical coord
    call reverse_vec(zlev,nlev)   ! invert to "bottom2top"
    call reverse_vec(zilev,nilev) ! invert to "bottom2top"

    istat = nf90_put_var(ncid,idv_levsp,zlev)
!
! Set magnetic coordinates:
!
    if (.not.allocated(maglon)) allocate(maglon(nmlon))
    if (.not.allocated(maglat)) allocate(maglat(nmlat))
    if (.not.allocated(zmlev))  allocate(zmlev(nmlev))

    maglon(:) = gmlon(1:nmlon)
    maglat = gmlat

!     write(iulog,"('define_ncfile: nmlat=',i4,' gmlat=',/,(8f10.3))") nmlat,gmlat
!     do j=1,nmlat
!       if (j > 1) then
!         write(iulog,"('define_ncfile: nmlat=',i4,' j=',i4,' gmlat(j)=',f10.3,' dlat=',f10.3)") &
!           nmlat,j,gmlat(j),abs(abs(gmlat(j))-abs(gmlat(j-1)))
!       else
!         write(iulog,"('define_ncfile: nmlat=',i4,' j=',i4,' gmlat(j)=',f10.3)") &
!           nmlat,j,gmlat(j)
!       endif
!     enddo

    zmlev = zlev ! (already inverted above)
!
! Convert column coordinate from WACCM pressure (hPa) to TIMEGCM zp:
    do k=1,nlev
      zlev(k) = log(p0/zlev(k))
    enddo
    do k=1,nilev
      zilev(k) = log(p0/zilev(k))
    enddo
    do k=1,nmlev
      zmlev(k) = log(p0/zmlev(k))
    enddo
!   write(iulog,"('define_ncfile: nlev=',i4,' zlev=',/,(8f9.2))") nlev,zlev
!   write(iulog,"('define_ncfile: nilev=',i4,' zilev=',/,(8f9.2))") nilev,zilev
!   write(iulog,"('define_ncfile: nmlev=',i4,' zmlev=',/,(8f9.2))") nilev,zmlev
!
    istat = nf90_put_var(ncid,idv_lon,geolon)
    istat = nf90_put_var(ncid,idv_lat,geolat)
    istat = nf90_put_var(ncid,idv_lev,zlev)
    istat = nf90_put_var(ncid,idv_ilev,zilev)
    istat = nf90_put_var(ncid,idv_mlon,maglon)
    istat = nf90_put_var(ncid,idv_mlat,maglat)
    istat = nf90_put_var(ncid,idv_mlev,zmlev)

  end subroutine define_ncfile
!-----------------------------------------------------------------------
  subroutine gather_fields
!
! Gather output fields to the root task prior to writing to output file.
! Fields may be 2d (lon,lat), or 3d (lon,lat,lev), and on geo or mag grids.
!
  use edyn_mpi,only: lon0,lon1,lat0,lat1,mlon0,mlon1,mlat0,mlat1,mp_gather2root
!
! Local
    integer :: n,k,j
    real(r8) :: fsub(lon0:lon1,lat0:lat1,nlev)

    do n=1,nfields
!     write(iulog,"('Enter gather_fields: n=',i3,' nfields=',i3,' field ',a,' ndims=',i3,' geo=',l1)") &
!       n,nfields,trim(fields_out(n)%name),fields_out(n)%ndims,fields_out(n)%geo
!
! If field is already global (savefld was called by root task only),
! then the gather is not necessary.
!
      if (fields_out(n)%task0_only) cycle

      if (fields_out(n)%ndims == 2) then ! 2d field
        if (fields_out(n)%geo) then      ! 2d field on geo grid
          call mp_gather2root(fields_out(n)%data(lon0:lon1,lat0:lat1,1),&
            lon0,lon1,lat0,lat1,fields_out(n)%data,nlon,nlat,1,  &
            fields_out(n)%geo,0)
        else                             ! 2d field on mag grid
          call mp_gather2root(fields_out(n)%data(mlon0:mlon1,mlat0:mlat1,1),&
            mlon0,mlon1,mlat0,mlat1,fields_out(n)%data,nmlonp1,nmlat,1,  &
            fields_out(n)%geo,0)
        endif
      else                               ! 3d field
        if (fields_out(n)%geo) then      ! 3d field on geo grid
          fsub(lon0:lon1,lat0:lat1,:) = fields_out(n)%data(lon0:lon1,lat0:lat1,:)
          call mp_gather2root(fsub, &
            lon0,lon1,lat0,lat1,fields_out(n)%data,nlon,nlat,nlev,  &
            fields_out(n)%geo,0)
        else                             ! 3d field on mag grid
          call mp_gather2root(fields_out(n)%data(mlon0:mlon1,mlat0:mlat1,:), &
            mlon0,mlon1,mlat0,mlat1,fields_out(n)%data,nmlonp1,nmlat,nmlev,  &
            fields_out(n)%geo,0)
!
! Set mag periodic point:
!         if (mytid==0) then
!           fields_out(n)%data(nmlonp1,:,:) = fields_out(n)%data(1,:,:)
!         endif
        endif
      endif
    enddo ! n=1,nfields
  end subroutine gather_fields
!-----------------------------------------------------------------------
  subroutine write_fields(ncid,newfile)
!
! Define and write fields to an open netcdf output file. These are the 2d
! or 3d fields added with calls to sub savefld. This routine is called at 
! every timestep by the root task only. If this is first call, the fields
! are defined on the file before writing the values.
!
! Field data has been gathered to the root task prior to this call.
! Fields will be 3d or 4d on the file (including unlimited time dimension),
! and on either geographic or magnetic grids.
!
! Args:
    integer,intent(in) :: ncid
    logical,intent(in) :: newfile
!
! Local:
    integer :: i,i0,i1,j,k,n,istat,iddims(4),varid
    integer :: start(4),count(4)
    integer :: iyear_w,imo_w,iday_w,tod_w
    integer,save :: ntime ! number of times (unlim dim) on the file
    real(r8) :: time,ut,f107,f107a,kp,ap,hp
!
! If this is a new file, define the fields on the file:
!
    if (newfile) then
      ntime = 0
      istat = nf90_redef(ncid)
      do n=1,nfields
        do i=1,3
          if (index(fields_out(n)%dimnames(i),'lon' ) > 0) iddims(i) = idlon
          if (index(fields_out(n)%dimnames(i),'lat' ) > 0) iddims(i) = idlat
          if (index(fields_out(n)%dimnames(i),'lev' ) > 0) iddims(i) = idlev
          if (index(fields_out(n)%dimnames(i),'ilev') > 0) iddims(i) = idilev

          if (index(fields_out(n)%dimnames(i),'mlon')  > 0) iddims(i) = idmlon
          if (index(fields_out(n)%dimnames(i),'mlat')  > 0) iddims(i) = idmlat
          if (index(fields_out(n)%dimnames(i),'mlev')  > 0) iddims(i) = idmlev
        enddo
        if (fields_out(n)%ndims == 2) then ! 2d field (3d with time on the file)
          iddims(3) = idunlim
          istat = nf90_def_var(ncid,fields_out(n)%name,NF90_DOUBLE,iddims(1:3),varid)
          if (istat /= NF90_NOERR) then
            write(msg,"('Defining 2d var ',a)") trim(fields_out(n)%name)
            call handle_ncerr(istat,msg,1)
          endif
        else                               ! 3d field (4d with time on the file)
          iddims(4) = idunlim
          istat = nf90_def_var(ncid,fields_out(n)%name,NF90_DOUBLE,iddims,varid)
          if (istat /= NF90_NOERR) then
            write(msg,"('Defining 3d var ',a)") trim(fields_out(n)%name)
            call handle_ncerr(istat,msg,1)
          endif
        endif
!
! Add long_name and units attributes for each field:
!
        istat = nf90_put_att(ncid,varid,"long_name",fields_out(n)%long_name)
        istat = nf90_put_att(ncid,varid,"units",fields_out(n)%units)
      enddo ! n=1,nfields
!
! Add global file attributes:
!
      istat = nf90_put_att(ncid,NF90_GLOBAL,"author"       ,"Ben Foster (foster@ucar.edu)")
      istat = nf90_put_att(ncid,NF90_GLOBAL,"model_name"   ,"WACCM-X")
      istat = nf90_put_att(ncid,NF90_GLOBAL,"model_version","WACCM-X") ! tgcmproc_idl needs this
      istat = nf90_put_att(ncid,NF90_GLOBAL,"missing_value",fillvalue)
!
! Take out of define mode:
!
      istat = nf90_enddef(ncid)
    endif ! newfile

    i0 = 1 ; i1 = nlon
    ntime = ntime+1
!
! Write field values to the file for current time step:
!
    do n=1,nfields
      istat = nf90_inq_varid(ncid,fields_out(n)%name,varid)
      if (istat /= NF90_NOERR) then
        call handle_ncerr(istat,'nf90_inq_varid',1)
      endif
      count = 0 ; start = 0
!
! Write 2d fields (i,j):
      if (fields_out(n)%ndims == 2) then ! 2d field (3d with time on the file)
        start(1:2) = 1
        start(3) = ntime
        count(2) = fields_out(n)%dimsizes(2)
        count(3) = 1
!
! Write 2d field on geographic grid (lon,lat):
        if (fields_out(n)%geo) then
          count(1) = nlon
          istat = nf90_put_var(ncid,varid,fields_out(n)%data(i0:i1,:,1), &
                               start(1:3),count(1:3))
          if (istat /= NF90_NOERR) &
            call handle_ncerr(istat,'nf90_put_var for 2d geo field',1)
!
! Write 2d field on magnetic grid (does not include periodic point):
        else
          count(1) = nmlon
          istat = nf90_put_var(ncid,varid,fields_out(n)%data(1:nmlon,:,1), &
                               start(1:3),count(1:3))
          if (istat /= NF90_NOERR) &
            call handle_ncerr(istat,'nf90_put_var for 2d mag field',1)
        endif
!
! Write 3d fields (lon,lat,lev):
      else                               ! 3d field (4d with time on the file)
        start(1:3) = 1
        start(4) = ntime
        count(2) = fields_out(n)%dimsizes(2)
        count(3) = fields_out(n)%dimsizes(3)
        count(4) = 1
!
! Write 3d field on geographic grid:
        if (fields_out(n)%geo) then
          count(1) = nlon
!          
! i0:i1 == 1:nlon
          istat = nf90_put_var(ncid,varid,fields_out(n)%data(i0:i1,:,:),start,count)
          if (istat /= NF90_NOERR) &
            call handle_ncerr(istat,'nf90_put_var for 3d geo field',1)
!
! Write 3d field on magnetic grid (does not include periodic point)
        else
          count(1) = nmlon
          istat = nf90_put_var(ncid,varid,fields_out(n)%data(1:nmlon,:,:),start,count)
          if (istat /= NF90_NOERR) &
            call handle_ncerr(istat,'nf90_put_var for 3d mag field',1)
        endif
      endif ! 2d or 3d field
    enddo ! n=1,nfields
!
! mtime is integer triplet model time (day,hour,minute)
!
    call get_curr_date(iyear_w,imo_w,iday_w,tod_w) ! tod is integer time-of-day in seconds

    mtime(1) = iday_w
    mtime(2) = int(real(tod_w)/3600._r8)
    mtime(3) = (tod_w-mtime(2)*3600)/60
!   write(iulog,"('write_fields: mtime=',3i4)") mtime

    istat = nf90_put_var(ncid,idv_mtime,mtime,(/1,ntime/),(/3,1/))
!
! Time is decimal day, calculated from mtime (day,hour,minute).
!
    time = real(mtime(1))+real(mtime(2))/24._r8+real(mtime(3))/(60._r8*24._r8) 
    ut = real(tod_w)/3600._r8
    call get_mag_parms( ctpoten = ctpoten )
    call solar_parms_get(f107,f107a,ap,kp,hp)

    istat = nf90_put_var(ncid,idv_time ,(/time/)     ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_year ,(/iyear_w/)  ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_day  ,(/iday_w/)   ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_ut   ,(/ut/)       ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_f107 ,(/f107/)     ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_f107a,(/f107a/)    ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_kp   ,(/kp/)       ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_ap   ,(/ap/)       ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_hp   ,(/hp/)       ,(/ntime/),(/1/))
    istat = nf90_put_var(ncid,idv_ctpoten,(/ctpoten/),(/ntime/),(/1/))

  end subroutine write_fields
!-----------------------------------------------------------------------
  subroutine shift_lon(f,nlon,lonseq,iscoord)
!
! Shift longitude vector f(nlon) forward 180 degrees according to input 
! string lonseq. Input f can be either arbitrary field values or 
! the coordinate array itself. Shift f in the 'lonseq' manner, as follows:
!
! If lonseq='-180to180', then shift from 0->360 to -180->+180
! If lonseq='zeroto360', then shift from -180->+180 to 0->360
!
! WARNING: This routine works with WACCM-X history files, where nlon=144, 72, or 80
!          It has not been tested with other models or resolutions.
!          (e.g., there is no test for center point, its assumed to be nlon/2)
!
! Args:
    integer,intent(in) :: nlon
    real(r8),intent(inout) :: f(nlon)
    character(len=*),intent(in) :: lonseq 
    logical,intent(in) :: iscoord ! if true, f is a coordinate, otherwise it is data
!
! Local:
    character(len=80) :: msg
    integer :: i,ihalf

    if (lonseq /= '-180to180'.and.lonseq /= 'zeroto360') then
      write(msg,"('shift_lon: bad lonseq=',a,' must be either ''-180to180'' or ''zeroto360''')") &
        lonseq
      call endrun
    endif

    ihalf = nlon/2
    if (lonseq == '-180to180') then ! shift to -180 -> +180
      f = cshift(f,ihalf)           ! cshift is circular shift intrinsic
      if (iscoord) then
        do i=1,ihalf
          f(i) = f(i)-360._r8
        enddo
      endif
    else                           ! shift to 0 -> 360
      f = cshift(f,ihalf)          ! cshift is circular shift intrinsic
      if (iscoord) then
        do i=ihalf+1,nlon
          f(i) = f(i)+360._r8
        enddo
      endif
    endif 
  end subroutine shift_lon
!-----------------------------------------------------------------------
  subroutine handle_ncerr(istat,msg,ifatal)
    implicit none
!
! Handle a netcdf lib error:
!
    integer,intent(in) :: istat,ifatal
    character(len=*),intent(in) :: msg
!
    write(iulog,"(/72('-'))")
    write(iulog,"('>>> Error from netcdf library:')")
    write(iulog,"(a)") trim(msg)
    write(iulog,"('istat=',i5)") istat
    write(iulog,"(a)") nf90_strerror(istat)
    write(iulog,"(72('-')/)")
    if (ifatal > 0) call endrun
  end subroutine handle_ncerr
!-----------------------------------------------------------------------
  subroutine lonshift_global(f,nlon,lonseq,iscoord)
!
! Shift longitude vector f(nlon) forward 180 degrees according to input
! string lonseq. Input f can be either arbitrary field values or
! the coordinate array itself. Shift f in the 'lonseq' manner, as follows:
!
! If lonseq='-180to180', then shift from 0->360 to -180->+180
! If lonseq='zeroto360', then shift from -180->+180 to 0->360
!
! WARNING: This routine works with WACCM-X history files, where nlon=144, 72, or 80
!          It has not been tested with other models or resolutions.
!          (e.g., there is no test for center point, its assumed to be nlon/2)
!
! Args:
    integer,intent(in) :: nlon
    real(r8),intent(inout) :: f(nlon)
    character(len=*),intent(in) :: lonseq
    logical,intent(in) :: iscoord ! if true, f is a coordinate, otherwise it is data
!
! Local:
    character(len=80) :: msg
    integer :: ihalf,i

    if (lonseq /= '-180to180'.and.lonseq /= 'zeroto360') then
      write(msg,"('shift_lon: bad lonseq=',a,' must be either ''-180to180'' or ''zeroto360''')") &
        lonseq
      call endrun
    endif

    ihalf = nlon/2
    if (lonseq == '-180to180') then ! shift to -180 -> +180
      f = cshift(f,ihalf)           ! cshift is circular shift intrinsic
      if (iscoord) then
        do i=1,ihalf
          f(i) = f(i)-360._r8
        enddo
      endif
    else                           ! shift to 0 -> 360
      f = cshift(f,ihalf)          ! cshift is circular shift intrinsic
      if (iscoord) then
        do i=ihalf+1,nlon
          f(i) = f(i)+360._r8
        enddo
      endif
    endif
  end subroutine lonshift_global
!-----------------------------------------------------------------------
  subroutine reverse_vec(vec,n)
!
! Reverse order of elements in vector vec(n)
!
! Args:
    integer,intent(in) :: n
    real(r8),intent(inout) :: vec(n)
!
! Local:
    real(r8) :: tmp(n)
    integer :: i

    do i=1,n
      tmp(i) = vec(n-i+1)
    enddo
    vec = tmp
  end subroutine reverse_vec
!-----------------------------------------------------------------------
end module edyn_output
