module gas_params_mod
  use shr_kind_mod,  only: r8 => shr_kind_r8
  use shr_const_mod, only: shr_const_rgas
  use physconst,     only: rair, cpair
  use constituents,  only: cnst_get_ind, cnst_mw
  use dynamics_vars, only: T_FVDYCORE_GRID
  use spmd_utils,    only: masterproc
  use cam_logfile,   only: iulog

  implicit none
  private
  public :: gas_params_init
  public :: gas_params_calc

  real(r8) :: o2_mwi, o_mwi, h_mwi, n2_mwi ! inverse molecular weights
  integer :: ixo2,ixo,ixh,ixn ! constituent indexes
  logical :: variable_phys

  interface gas_params_calc
     module procedure calc_params
     module procedure intr_params
  end interface gas_params_calc

contains
  !------------------------------------------------------------------------------
  !------------------------------------------------------------------------------ 
  subroutine gas_params_init(high_alt)

    logical, intent(in) :: high_alt

    call cnst_get_ind('O2',ixo2,abort=.false.)
    call cnst_get_ind('O' ,ixo, abort=.false.)
    call cnst_get_ind('H' ,ixh, abort=.false.)
    call cnst_get_ind('N' ,ixn, abort=.false.)

    variable_phys = ixo2>0 .and. ixo>0 .and. ixh>0 .and. ixn>0 .and. high_alt

    if (variable_phys) then
       o2_mwi = 1.0_r8/cnst_mw(ixo2)
       o_mwi  = 1.0_r8/cnst_mw(ixo)
       n2_mwi = 0.5_r8/cnst_mw(ixn)
       h_mwi  = 1.0_r8/cnst_mw(ixh)
    endif

   if (masterproc) then
      write(iulog,*) 'gas_params_init : variable_phys = ', variable_phys
   endif

  end subroutine gas_params_init

  !------------------------------------------------------------------------------
  !------------------------------------------------------------------------------ 
  subroutine calc_params( i0,i1,j0,j1,k0,k1,ntotq, tracer, cpv, rairv, kappav )

    ! args
    integer,  intent(in) :: i0,i1,j0,j1,k0,k1, ntotq
    real(r8), intent(in) :: tracer(i0:i1,j0:j1,k0:k1,ntotq) ! Tracer array
    real(r8), optional, intent(out) :: cpv(i0:i1,j0:j1,k0:k1) 
    real(r8), optional, intent(out) :: rairv(i0:i1,j0:j1,k0:k1) 
    real(r8), optional, intent(out) :: kappav(i0:i1,j0:j1,k0:k1)

    ! local vars
    integer :: i,j,k
    real(r8),  dimension(i0:i1,j0:j1,k0:k1) :: &
         kap_var, mbar_var, rgas_var, cp_var, mmro, mmro2, mmrh, mmrn2
    real(r8) :: cap

    real(r8), parameter ::  dof1 = 5.0_r8 ! Degrees of freedom for cpair3v calculation
    real(r8), parameter ::  dof2 = 7.0_r8 ! Degrees of freedom for cpair3v calculation

    if (variable_phys) then

       !-----------------------------------------------------------------------
       !  Calculate constituent dependent specific heat, gas constant and cappa
       !-----------------------------------------------------------------------
!$omp parallel do private(i,j,k)
       do k = k0,k1
          do j = j0,j1
             do i = i0,i1
                mmro(i,j,k)  = tracer(i,j,k,ixo)
                mmro2(i,j,k) = tracer(i,j,k,ixo2)
                mmrh(i,j,k)  = tracer(i,j,k,ixh)
                mmrn2(i,j,k) = 1._r8-mmro(i,j,k)-mmro2(i,j,k)-mmrh(i,j,k)

                mbar_var(i,j,k) = 1._r8/( mmro (i,j,k)*o_mwi + &
                                          mmro2(i,j,k)*o2_mwi +	&
                                          mmrn2(i,j,k)*n2_mwi + &
                                          mmrh (i,j,k)*h_mwi )

                rgas_var(i,j,k) = shr_const_rgas / mbar_var(i,j,k)

                if (present(rairv)) then
                   rairv(i,j,k) = rgas_var(i,j,k)
                endif

                cp_var(i,j,k) = 0.5_r8*shr_const_rgas &
                              * ( dof1*mmro (i,j,k)*o_mwi + &
                                  dof2*mmro2(i,j,k)*o2_mwi +	&
                                  dof2*mmrn2(i,j,k)*n2_mwi + &
                                  dof1*mmrh (i,j,k)*h_mwi )

                if (present(cpv)) then
                   cpv(i,j,k) = cp_var(i,j,k)
                endif

                kap_var(i,j,k) = rgas_var(i,j,k)/cp_var(i,j,k)

                if (present(kappav)) then
                   kappav(i,j,k) = kap_var(i,j,k)
                endif
             enddo
          enddo
       enddo
    else
       cap = rair/cpair
       if (present(cpv)) cpv(:,:,:) = cpair
       if (present(rairv)) rairv(:,:,:) = rair
       if (present(kappav)) kappav(:,:,:) = cap
    endif

  end subroutine calc_params
  !------------------------------------------------------------------------------
  !------------------------------------------------------------------------------ 
  subroutine intr_params( grid, kappav, kappavi )

    type (T_FVDYCORE_GRID), intent(in) :: grid    ! grid for XY decomp
    real(r8), intent(in) :: &
         kappav( grid%ifirstxy:grid%ilastxy, grid%jfirstxy:grid%jlastxy, grid%km )    ! kappa at level centers
    real(r8), intent(out) :: &
         kappavi(grid%ifirstxy:grid%ilastxy, grid%jfirstxy:grid%jlastxy, grid%km+1)  ! kappa at level interfaces

    integer :: i,j,k

    if (variable_phys) then
!$omp parallel do private(i,j,k)
       do k=2,grid%km
          do j=grid%jfirstxy,grid%jlastxy
             do i=grid%ifirstxy,grid%ilastxy
                kappavi(i,j,k) = 0.5_r8*(kappav(i,j,k-1)+kappav(i,j,k))
             enddo
          enddo
       enddo
       kappavi(:,:,1) = 1.5_r8 * kappav(:,:,1) - 0.5_r8 * kappav(:,:,2)
       kappavi(:,:,grid%km+1) = 1.5_r8 * kappav(:,:,grid%km) - 0.5_r8 * kappav(:,:,grid%km-1)
    else
       kappavi(:,:,:) = kappav(grid%ifirstxy,grid%jfirstxy,1)
    endif

  end subroutine intr_params

end module gas_params_mod
