!--------------------------------------------------------------------------------
! module where input data utility classes reside
!
! classes:
!     time_coordinate -- manages the time coordinate of input data sets
!--------------------------------------------------------------------------------
module input_data_utils
  use shr_kind_mod,   only : r8 => shr_kind_r8, cs => shr_kind_cs, cl=> shr_kind_cl
  use cam_abortutils, only : endrun
  use cam_logfile,    only : iulog
  use pio,            only : file_desc_t, pio_inq_dimid, pio_inq_dimlen, pio_get_att
  use pio,            only : pio_seterrorhandling, pio_get_var, pio_inq_varid
  use pio,            only : PIO_NOWRITE, PIO_BCAST_ERROR, PIO_INTERNAL_ERROR, PIO_NOERR
  use time_manager,   only : timemgr_get_calendar_cf, set_time_float_from_date, get_curr_date

  implicit none

  private
  public :: time_coordinate

  type :: time_coordinate
     integer :: ntimes
     real(r8) :: wghts(2)
     integer :: indxs(2)
     real(r8), allocatable :: times(:)
     real(r8), allocatable :: time_bnds(:,:)
     logical :: time_interp = .true.
     logical :: fixed = .false.
     integer :: fixed_ymd, fixed_tod
     character(len=cl) :: filename
   contains
     procedure :: initialize
     procedure :: advance
     procedure :: read_more
     procedure :: copy
     procedure :: destroy
  end type time_coordinate

contains

  !-----------------------------------------------------------------------------
  ! initializer
  !-----------------------------------------------------------------------------
  subroutine initialize( this, filepath, fixed, fixed_ymd, fixed_tod, force_time_interp, set_weights, try_dates )
    use ioFileMod,      only : getfil
    use cam_pio_utils,  only : cam_pio_openfile, cam_pio_closefile
    use string_utils,   only : to_upper

    class(time_coordinate), intent(inout) :: this
    character(len=*), intent(in) :: filepath
    logical, optional,intent(in) :: fixed
    integer, optional,intent(in) :: fixed_ymd
    integer, optional,intent(in) :: fixed_tod
    logical, optional,intent(in) :: force_time_interp
    logical, optional,intent(in) :: set_weights
    logical, optional,intent(in) :: try_dates

    character(len=cl) :: filen
    character(len=cl) :: time_units, err_str
    character(len=cs) :: time_calendar, model_calendar
    character(len=4) :: yr_str
    character(len=2) :: mon_str, day_str, hr_str, min_str, sec_str
    integer :: ref_yr, ref_mon, ref_day, ref_hr, ref_min, ref_sec, tod
    integer :: varid, ierr
    real(r8) :: ref_time

    integer,  allocatable :: dates(:)
    integer,  allocatable :: datesecs(:)
    type(file_desc_t) :: fileid
    logical :: force_interp
    logical :: set_wghts
    logical :: use_time
    integer :: i

    if (present(fixed)) this%fixed = fixed
    if (present(fixed_ymd)) this%fixed_ymd = fixed_ymd
    if (present(fixed_tod)) this%fixed_tod = fixed_tod

    if (present(force_time_interp)) then
       force_interp = force_time_interp
    else
       force_interp = .false.
    endif

    if (present(set_weights)) then
       set_wghts = set_weights
    else
       set_wghts = .true.
    endif

    this%filename = trim(filepath)

    call getfil( filepath, filen, 0 )
    call cam_pio_openfile( fileid, filen, PIO_NOWRITE )

    call pio_seterrorhandling( fileid, PIO_BCAST_ERROR)

    call get_dimension( fileid, 'time', this%ntimes )
    allocate ( this%times( this%ntimes ) )

    ierr =  pio_inq_varid( fileid, 'time', varid )
    use_time = ierr.eq.PIO_NOERR
    ierr = pio_get_att( fileid, varid, 'calendar', time_calendar)
    use_time = ierr.eq.PIO_NOERR .and. use_time
    ierr = pio_get_att( fileid, varid, 'units', time_units)
    use_time = ierr.eq.PIO_NOERR .and. use_time
    if (use_time) then
       use_time = time_units(1:10).eq.'days since'
    endif

    if (use_time) then
       ! check the calendar attribute - must match model calendar
       model_calendar = timemgr_get_calendar_cf()

       if (this%ntimes>2) then 
          ! if only 2 time records then it is assumed that the input has 2 identical time records
          !  -- climatological or solar-cycle avaraged
          if (to_upper(time_calendar(1:6)) .ne. to_upper(model_calendar(1:6))) then
             err_str = 'time_coordinate%initialize: model calendar '//trim(model_calendar)// &
                  ' does not match input data calendar '//trim(time_calendar)
             write(iulog,*) err_str
             use_time = .false.
          end if
       end if
    endif

    if (present(try_dates)) then
       if (try_dates) then
          ierr = pio_inq_varid( fileid, 'date', varid  )
          use_time = ierr .ne. PIO_NOERR
       endif
    endif

    time_var_use: if (use_time) then

       ! parse out ref date and time
       !                         1         2         3
       !                123456789012345678901234567890
       !  time:units = "days since YYYY-MM-DD hh:mm:ss" ;

       yr_str  = time_units(12:15)
       mon_str = time_units(17:18)
       day_str = time_units(20:21)
       hr_str  = time_units(23:24)
       min_str = time_units(26:27)

       read( yr_str,  * ) ref_yr
       read( mon_str, * ) ref_mon
       read( day_str, * ) ref_day
       read( hr_str,  * ) ref_hr
       read( min_str, * ) ref_min
       if (len_trim(time_units).ge.30) then
          sec_str = time_units(29:30)
          read( sec_str, * ) ref_sec
       else
          ref_sec = 0
       endif

       tod = ref_hr*3600 + ref_min*60 + ref_sec
       call set_time_float_from_date( ref_time, ref_yr, ref_mon, ref_day, tod )

       ierr = pio_get_var( fileid, varid, this%times )
       if (ierr.ne.PIO_NOERR) then
          call endrun('time_coordinate%initialize: not able to read times')
       endif

       this%times = this%times + ref_time

       ierr =  pio_inq_varid( fileid, 'time_bnds', varid )

       if(ierr==PIO_NOERR .and. .not.force_interp) then
          allocate ( this%time_bnds( 2, this%ntimes ) )
          ierr =  pio_get_var( fileid, varid, this%time_bnds )
          this%time_bnds = this%time_bnds + ref_time
          this%time_interp = .false.
          do i = 1,this%ntimes
            if (.not. (this%time_bnds(1,i)<this%times(i) &
                 .and. this%time_bnds(2,i)>this%times(i)) ) then
                write(err_str,*) 'incorrect time_bnds -- time index: ',i,' file: '//trim(filepath)
                call endrun(err_str)
            endif
          enddo
       else
          this%time_interp = .true.
       endif

    else ! no time coordinate variable 

       ! try using date and datesec
       allocate(dates(this%ntimes), stat=ierr )
       if( ierr /= 0 ) then
          write(iulog,*) 'time_coordinate%initialize: failed to allocate dates; error = ',ierr
          call endrun('time_coordinate%initialize: failed to allocate dates')
       end if

       allocate(datesecs(this%ntimes), stat=ierr )
       if( ierr /= 0 ) then
          write(iulog,*) 'time_coordinate%initialize: failed to allocate datesecs; error = ',ierr
          call endrun('time_coordinate%initialize: failed to allocate datesecs')
       end if

       ierr = pio_inq_varid( fileid, 'date', varid  )
       if (ierr/=PIO_NOERR) then
          call endrun('time_coordinate%initialize: input file must contain time or date variable '//trim(filepath))
       endif
       ierr = pio_get_var( fileid, varid, dates )
       ierr = pio_inq_varid( fileid, 'datesec', varid )
       if (ierr==PIO_NOERR) then
          ierr = pio_get_var( fileid, varid, datesecs )
       else
          datesecs(:) = 0
       endif

       call convert_dates( dates, datesecs, this%times )

       deallocate( dates, datesecs )

       this%time_interp = .true.

    endif time_var_use

    call pio_seterrorhandling(fileid, PIO_INTERNAL_ERROR)

    call cam_pio_closefile(fileid)

    this%indxs(1)=1
    if (set_wghts) call set_wghts_indices(this)

  end subroutine initialize

  !-----------------------------------------------------------------------------
  ! advance the time coordinate
  !-----------------------------------------------------------------------------
  subroutine advance( this )
    class(time_coordinate) :: this

    if (.not.this%fixed) call set_wghts_indices(this)

  end subroutine advance

  !-----------------------------------------------------------------------------
  ! determine if need to read more data from input data set
  !-----------------------------------------------------------------------------
  function read_more(this) result(check)
    class(time_coordinate), intent(in) :: this
    logical :: check

    real(r8) :: model_time

    model_time = get_model_time()

    if (.not.this%fixed) then
       if (allocated(this%time_bnds)) then
          check = model_time > this%time_bnds(2,this%indxs(1))
       else
          check = model_time > this%times(this%indxs(2))
       endif
    else
       check = .false.
    endif

  end function read_more

  !-----------------------------------------------------------------------------
  ! destroy method -- deallocate memory and revert to default settings
  !-----------------------------------------------------------------------------
  subroutine destroy( this )
        class(time_coordinate), intent(inout) :: this
    
    if (allocated(this%times)) deallocate(this%times)
    if (allocated(this%time_bnds)) deallocate(this%time_bnds)
    this%ntimes = 0
    this%filename='NONE'

  end subroutine destroy
    
  !-----------------------------------------------------------------------------
  ! produce a duplicate time coordinate object
  !-----------------------------------------------------------------------------
  subroutine copy( this, obj )
    class(time_coordinate), intent(inout) :: this
    class(time_coordinate), intent(in) :: obj

    call this%destroy()

    this%ntimes = obj%ntimes
    this%fixed  = obj%fixed
    this%fixed_ymd = obj%fixed_ymd
    this%fixed_tod = obj%fixed_tod

    allocate ( this%times( this%ntimes ) )
    this%times = obj%times

    if (allocated( obj%time_bnds )) then
       allocate ( this%time_bnds( 2, this%ntimes ) )
       this%time_bnds = obj%time_bnds
    endif
    this%filename = obj%filename

  end subroutine copy

! private methods

  !-----------------------------------------------------------------------
  ! set time interpolation weights
  !-----------------------------------------------------------------------
  subroutine set_wghts_indices(obj)

    class(time_coordinate), intent(inout) :: obj

    real(r8) :: model_time
    real(r8) :: datatm, datatp
    integer :: yr, mon, day
    integer :: index, i
    character(len=cl) :: errmsg

    ! set time indices and time-interpolation weights 
    fixed_time: if (obj%fixed) then
       yr = obj%fixed_ymd/10000
       mon = (obj%fixed_ymd-yr*10000) / 100
       day = obj%fixed_ymd-yr*10000-mon*100
       call set_time_float_from_date( model_time, yr, mon, day, obj%fixed_tod )
    else
       model_time = get_model_time()
    endif fixed_time

    index = -1

    findtimes: do i = obj%indxs(1), obj%ntimes
       if (allocated(obj%time_bnds)) then
          datatm = obj%time_bnds(1,i)
          datatp = obj%time_bnds(2,i)
       else
          datatm = obj%times(i)
          datatp = obj%times(i+1)
       endif
       if ( model_time .ge. datatm .and. model_time .le. datatp ) then
           index = i
           obj%indxs(1) = i
           obj%indxs(2) = i+1
           exit findtimes
        endif
    enddo findtimes

    if (.not.(index>0.and.index<obj%ntimes)) then
       errmsg = 'input_data_utils::set_wghts_indices cannot not find time indices for input file: '&
            // trim(obj%filename)
       write(iulog,*) trim(errmsg)
       call endrun(trim(errmsg))
    endif

    if (obj%time_interp) then
       obj%wghts(2) = ( model_time - obj%times(index) ) / ( obj%times(index+1) - obj%times(index) )
       obj%wghts(1) = 1._r8 - obj%wghts(2)
    else
       obj%wghts(1) = 1._r8
       obj%wghts(2) = 0._r8       
    endif

  end subroutine set_wghts_indices

  !-----------------------------------------------------------------------
  ! returns dimension size
  !-----------------------------------------------------------------------
  subroutine get_dimension( fid, dname, dsize )
    type(file_desc_t), intent(in) :: fid
    character(*), intent(in) :: dname
    integer, intent(out) :: dsize

    integer :: dimid, ierr

    ierr = pio_inq_dimid( fid, dname, dimid )
    ierr = pio_inq_dimlen( fid, dimid, dsize )

  end subroutine get_dimension

  !-----------------------------------------------------------------------
  ! returns a real which represents the current model time
  !-----------------------------------------------------------------------
  function get_model_time() result(time)

    real(r8) :: time

    integer yr, mon, day, ncsec  ! components of a date

    call get_curr_date(yr, mon, day, ncsec)

    call set_time_float_from_date( time, yr, mon, day, ncsec )

  end function get_model_time

  !---------------------------------------------------------------------------
  ! convert a collection of dates and times to reals
  !---------------------------------------------------------------------------
  subroutine convert_dates( dates, secs, times )

    use time_manager, only: set_time_float_from_date

    integer,  intent(in)  :: dates(:)
    integer,  intent(in)  :: secs(:)

    real(r8), intent(out) :: times(:)

    integer :: year, month, day, sec,n ,i

    n = size( dates ) 

    do i=1,n
       year = dates(i)/10000
       month = (dates(i)-year*10000)/100
       day = dates(i)-year*10000-month*100
       sec = secs(i)
       call set_time_float_from_date( times(i), year, month, day, sec )
    enddo

  end subroutine convert_dates

end module input_data_utils
