/**
 * @file
 * PIO functions to get data (excluding varm functions).
 *
 * @author Ed Hartnett
 * @date  2016
 *
 * @see http://code.google.com/p/parallelio/
 */

#include <config.h>
#include <pio.h>
#include <pio_internal.h>

/**
 * Get strided, muti-dimensional subset of a text variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_text(int ncid, int varid, const PIO_Offset *start, const PIO_Offset *count,
                       const PIO_Offset *stride, char *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_CHAR, buf);
}

/**
 * Get strided, muti-dimensional subset of an unsigned char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_uchar(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, const PIO_Offset *stride, unsigned char *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_UBYTE, buf);
}

/**
 * Get strided, muti-dimensional subset of a signed char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_schar(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, const PIO_Offset *stride, signed char *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_BYTE, buf);
}

/**
 * Get strided, muti-dimensional subset of an unsigned 16-bit integer
 * variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_ushort(int ncid, int varid, const PIO_Offset *start,
                         const PIO_Offset *count, const PIO_Offset *stride, unsigned short *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_USHORT, buf);
}

/**
 * Get strided, muti-dimensional subset of a 16-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_short(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, const PIO_Offset *stride, short *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_SHORT, buf);
}

/**
 * Get strided, muti-dimensional subset of an unsigned integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_uint(int ncid, int varid, const PIO_Offset *start,
                       const PIO_Offset *count, const PIO_Offset *stride, unsigned int *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_UINT, buf);
}

/**
 * Get strided, muti-dimensional subset of an integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_int(int ncid, int varid, const PIO_Offset *start, const PIO_Offset *count,
                      const PIO_Offset *stride, int *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_INT, buf);
}

/**
 * Get strided, muti-dimensional subset of a 64-bit int variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_long(int ncid, int varid, const PIO_Offset *start,
                       const PIO_Offset *count, const PIO_Offset *stride, long *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, PIO_LONG_INTERNAL, buf);
}

/**
 * Get strided, muti-dimensional subset of a floating point variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_float(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, const PIO_Offset *stride, float *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_FLOAT, buf);
}

/**
 * Get strided, muti-dimensional subset of a 64-bit floating point
 * variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_double(int ncid, int varid, const PIO_Offset *start,
                         const PIO_Offset *count, const PIO_Offset *stride, double *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_DOUBLE, buf);
}

/**
 * Get strided, muti-dimensional subset of an unsigned 64-bit int
 * variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_ulonglong(int ncid, int varid, const PIO_Offset *start,
                            const PIO_Offset *count, const PIO_Offset *stride,
                            unsigned long long *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, PIO_LONG_INTERNAL, buf);
}

/**
 * Get strided, muti-dimensional subset of a 64-bit int variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars_longlong(int ncid, int varid, const PIO_Offset *start,
                           const PIO_Offset *count, const PIO_Offset *stride, long long *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_INT64, buf);
}

/**
 * Get a muti-dimensional subset of a text variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_text(int ncid, int varid, const PIO_Offset *start,
                       const PIO_Offset *count, char *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_CHAR, buf);
}

/**
 * Get a muti-dimensional subset of an unsigned char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_uchar(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, unsigned char *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_UBYTE, buf);
}

/**
 * Get a muti-dimensional subset of a signed char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_schar(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, signed char *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_BYTE, buf);
}

/**
 * Get a muti-dimensional subset of an unsigned 16-bit integer
 * variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_ushort(int ncid, int varid, const PIO_Offset *start,
                         const PIO_Offset *count, unsigned short *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_USHORT, buf);
}

/**
 * Get a muti-dimensional subset of a 16-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_short(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, short *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_SHORT, buf);
}

/**
 * Get a muti-dimensional subset of a 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_long(int ncid, int varid, const PIO_Offset *start,
                       const PIO_Offset *count, long *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, PIO_LONG_INTERNAL, buf);
}

/**
 * Get a muti-dimensional subset of an unsigned integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_uint(int ncid, int varid, const PIO_Offset *start,
                       const PIO_Offset *count, unsigned int *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_UINT, buf);
}

/**
 * Get a muti-dimensional subset of an integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_int(int ncid, int varid, const PIO_Offset *start,
                      const PIO_Offset *count, int *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_INT, buf);
}

/**
 * Get a muti-dimensional subset of a floating point variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_float(int ncid, int varid, const PIO_Offset *start,
                        const PIO_Offset *count, float *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_FLOAT, buf);
}

/**
 * Get a muti-dimensional subset of a 64-bit floating point variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_double(int ncid, int varid, const PIO_Offset *start,
                         const PIO_Offset *count, double *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_DOUBLE, buf);
}

/**
 * Get a muti-dimensional subset of an unsigned 64-bit integer
 * variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_ulonglong(int ncid, int varid, const PIO_Offset *start,
                            const PIO_Offset *count, unsigned long long *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_UINT64, buf);
}

/**
 * Get a muti-dimensional subset of a 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara_longlong(int ncid, int varid, const PIO_Offset *start,
                           const PIO_Offset *count, long long *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_INT64, buf);
}

/**
 * Get all data of a text variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_text(int ncid, int varid, char *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_CHAR, buf);
}

/**
 * Get all data of an unsigned char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_uchar(int ncid, int varid, unsigned char *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_UBYTE, buf);
}

/**
 * Get all data of a signed char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_schar(int ncid, int varid, signed char *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_BYTE, buf);
}

/**
 * Get all data of an unsigned 16-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_ushort(int ncid, int varid, unsigned short *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_USHORT, buf);
}

/**
 * Get all data of a 16-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_short(int ncid, int varid, short *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_SHORT, buf);
}

/**
 * Get all data of an unsigned integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_uint(int ncid, int varid, unsigned int *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_UINT, buf);
}

/**
 * Get all data of an integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_int(int ncid, int varid, int *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_INT, buf);
}

/**
 * Get all data of a 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_long (int ncid, int varid, long *buf)
{
    return PIOc_get_var_tc(ncid, varid, PIO_LONG_INTERNAL, buf);
}

/**
 * Get all data of a floating point variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_float(int ncid, int varid, float *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_FLOAT, buf);
}

/**
 * Get all data of a 64-bit floating point variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_double(int ncid, int varid, double *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_DOUBLE, buf);
}

/**
 * Get all data of an unsigned 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_ulonglong(int ncid, int varid, unsigned long long *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_UINT64, buf);
}

/**
 * Get all data of a 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var_longlong(int ncid, int varid, long long *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_INT64, buf);
}

/**
 * Get one value of a text variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_text(int ncid, int varid, const PIO_Offset *index, char *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_CHAR, buf);
}

/**
 * Get one value of an unsinged char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_uchar (int ncid, int varid, const PIO_Offset *index, unsigned char *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_UBYTE, buf);
}

/**
 * Get one value of a signed char variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_schar(int ncid, int varid, const PIO_Offset *index, signed char *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_BYTE, buf);
}

/**
 * Get one value of an unsigned 16-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_ushort(int ncid, int varid, const PIO_Offset *index, unsigned short *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_USHORT, buf);
}

/**
 * Get one value of a 16-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_short(int ncid, int varid, const PIO_Offset *index, short *buf)
{
    int ret = PIOc_get_var1_tc(ncid, varid, index, NC_SHORT, buf);
    LOG((1, "PIOc_get_var1_short returned %d", ret));
    return ret;
}

/**
 * Get one value of an unsigned integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_uint(int ncid, int varid, const PIO_Offset *index, unsigned int *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_UINT, buf);
}

/**
 * Get one value of a 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_long (int ncid, int varid, const PIO_Offset *index, long *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, PIO_LONG_INTERNAL, buf);
}

/**
 * Get one value of an integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_int(int ncid, int varid, const PIO_Offset *index, int *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_INT, buf);
}

/**
 * Get one value of a floating point variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_float(int ncid, int varid, const PIO_Offset *index, float *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_FLOAT, buf);
}

/**
 * Get one value of a 64-bit floating point variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_double (int ncid, int varid, const PIO_Offset *index, double *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_DOUBLE, buf);
}

/**
 * Get one value of an unsigned 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_ulonglong (int ncid, int varid, const PIO_Offset *index,
                             unsigned long long *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_INT64, buf);
}

/**
 * Get one value of a 64-bit integer variable.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1_longlong(int ncid, int varid, const PIO_Offset *index,
                           long long *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_INT64, buf);
}

/**
 * Get all data from a variable the same type as the variable in the
 * file.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var(int ncid, int varid, void *buf)
{
    return PIOc_get_var_tc(ncid, varid, NC_NAT, buf);
}

/**
 * Get one value from a variable the same type as the variable in the
 * file.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param index an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_var1(int ncid, int varid, const PIO_Offset *index, void *buf)
{
    return PIOc_get_var1_tc(ncid, varid, index, NC_NAT, buf);
}

/**
 * Get a muti-dimensional subset of a variable the same type
 * as the variable in the file.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vara(int ncid, int varid, const PIO_Offset *start, const PIO_Offset *count,
                  void *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, NULL, NC_NAT, buf);
}

/**
 * Get strided, muti-dimensional subset of a variable of the same type
 * as the variable in the file.
 *
 * This routine is called collectively by all tasks in the
 * communicator ios.union_comm.
 *
 * @param ncid identifies the netCDF file
 * @param varid the variable ID number
 * @param start an array of start indicies (must have same number of
 * entries as variable has dimensions). If NULL, indices of 0 will be
 * used.
 * @param count an array of counts (must have same number of entries
 * as variable has dimensions). If NULL, counts matching the size of
 * the variable will be used.
 * @param stride an array of strides (must have same number of
 * entries as variable has dimensions). If NULL, strides of 1 will be
 * used.
 * @param buf pointer that will get the data.
 * @return PIO_NOERR on success, error code otherwise.
 */
int PIOc_get_vars(int ncid, int varid, const PIO_Offset *start, const PIO_Offset *count,
                  const PIO_Offset *stride, void *buf)
{
    return PIOc_get_vars_tc(ncid, varid, start, count, stride, NC_NAT, buf);
}
