
#include <misc.h>

      module mo_tgcm_ubc
!---------------------------------------------------------------
!	... tgcm upper bndy values
!---------------------------------------------------------------

      use shr_kind_mod, only : r8 => shr_kind_r8
      use pmgrid,       only : plon, plat
      use chem_mods,    only : gas_pcnst
      use constituents, only : ppcnst, cnst_fixed_ubc

      use abortutils, only: endrun

      implicit none

      private
      public  :: tgcm_ubc_inti, set_tgcm_ubc, tgcm_timestep_init, klower

      save

      integer, parameter :: klower   = 1
      integer :: ub_nlevs, &                                          ! # of levs in tgcm ubc file
                 ub_nlat, &                                           ! # of lats in tgcm ubc file
                 ub_nlon, &                                           ! # of lons in tgcm ubc file
                 ub_nspecies, &                                       ! # of requested species in tgcm ubc file
                 ub_ndays, &                                          ! # of days in tgcm ubc file
                 last, next, &                                        ! day indicies
                 kmin, kmax
      integer :: tgcm_frq                                             ! frequency of tgcm update (timesteps)
      integer :: stepsize                                             ! model timestep (s)
      integer, allocatable                  :: kl(:,:,:)
      integer, allocatable                  :: ku(:,:,:)
      integer, allocatable                  :: map(:,:)               ! species indices for UBC species
      real(r8)                              :: dels
      real(r8), allocatable                 :: days(:)                ! day of year for input data
      real(r8), allocatable                 :: tgcm_ub_mmr(:,:,:,:,:) ! module array for tgcm ub values (kg/kg)
      real(r8), allocatable                 :: delp(:,:,:)
      logical :: ubc_from_tgcm(ppcnst) = .false.
      logical :: zonal_average         = .false.                      ! use zonal averaged tgcm values

      contains

      subroutine tgcm_ubc_inti(tgcm_ubc_file, zonal_avg, freq)
!------------------------------------------------------------------
!	... initialize upper boundary values
!------------------------------------------------------------------

      use m_spc_id
      use ioFileMod,       only : getfil
      use phys_grid,       only : clat_p, clon_p
      use mo_regrider,     only : regrid_inti, regrid_2d, regrid_lat_limits, regrid_diagnostics
      use mo_tracname,     only : solsym
      use ppgrid,          only : pver, pverp, pcols, begchunk, endchunk
      use pmgrid,          only : masterproc, plev, plevp
      use phys_grid,       only : get_ncols_p, get_lon_all_p, get_lat_all_p
      use time_manager,    only : get_step_size
      use constituents,    only : cnst_get_ind, cnst_name
      use shr_const_mod,   only : pi => shr_const_pi
#ifdef SPMD
      use mpishorthand,    only : mpicom, mpiint, mpir8
#endif

      implicit none

!------------------------------------------------------------------
!	... dummy args
!------------------------------------------------------------------
      integer, intent(in) :: &
        freq                 ! frequency of update
      character(len=*), intent(in) :: &
        tgcm_ubc_file
      logical, intent(in) :: &
        zonal_avg            ! zonal averaging switch        

!------------------------------------------------------------------
!	... local variables
!------------------------------------------------------------------
      integer :: c, i, ip, k, k1, m, nchar
      integer :: j, jl, ju, ncol
      integer :: ic, jc
      integer :: spcno, lev, day, ierr
      integer :: ncid, vid
      integer :: dimid_lat, dimid_lon, dimid_lev, dimid_species, dimid_time, ndims
      integer :: gndx = 0
      integer :: jlim_in(2)
      integer :: start(4)
      integer :: cnt(4)
      integer :: dimid(4)
      integer :: latndx(pcols)
      integer :: lonndx(pcols)
      integer :: kl_g(plon,klower,plat)
      integer :: ku_g(plon,klower,plat)
      real(r8)              :: d2r
      real(r8)              :: p0, wrk, pinterp
      real(r8)              :: pint(pverp)
      real(r8)              :: delp_g(plon,klower,plat)
      real(r8), allocatable :: mr_ub_in(:,:,:,:,:)
      real(r8), allocatable :: mr_ub(:,:,:,:,:)       ! wrk array
      real(r8), allocatable :: lat(:), lon(:)
      real(r8), allocatable :: ub_plevs(:)            ! pascals
      character(len=80) :: attribute
      character(len=8)  :: ub_species_name
      character(len=256) :: locfn

#include <netcdf.inc>
#include <comhyb.h>

      zonal_average = zonal_avg
      tgcm_frq      = max( freq, 1 )
Masterproc_only : &
      if( masterproc ) then
	 d2r = pi/180._r8
         call cnst_get_ind( 'H2', vid, abort=.false. )
         if( vid > 0 ) then
            if( cnst_fixed_ubc(vid) ) then
               ubc_from_tgcm(vid) = .true.
            end if
         end if
         call cnst_get_ind( 'CO', vid, abort=.false. )
         if( vid > 0 ) then
            if( cnst_fixed_ubc(vid) ) then
               ubc_from_tgcm(vid) = .true.
            end if
         end if
         call cnst_get_ind( 'CO2', vid, abort=.false. )
         if( vid > 0 ) then
            if( cnst_fixed_ubc(vid) ) then
               ubc_from_tgcm(vid) = .true.
            end if
         end if
         ub_nspecies = count( ubc_from_tgcm )
has_tgcm_ubc : &
         if( ub_nspecies /= 0 ) then
!-----------------------------------------------------------------------
!       ... open netcdf file
!-----------------------------------------------------------------------
            call getfil (tgcm_ubc_file, locfn, 0)
            call wrap_open (trim(locfn), NF_NOWRITE, ncid)
!-----------------------------------------------------------------------
!       ... get tgcm dimensions
!-----------------------------------------------------------------------
            call wrap_inq_dimid( ncid, 'lon', dimid_lon )
            call wrap_inq_dimlen( ncid, dimid_lon, ub_nlon )

            call wrap_inq_dimid( ncid, 'lat', dimid_lat )
            call wrap_inq_dimlen( ncid, dimid_lat, ub_nlat )

            call wrap_inq_dimid( ncid, 'lev', dimid_lev )
            call wrap_inq_dimlen( ncid, dimid_lev, ub_nlevs )

            call wrap_inq_dimid( ncid, 'time', dimid_time )
            call wrap_inq_dimlen( ncid, dimid_time, ub_ndays )

!------------------------------------------------------------------
!	... form map from tgcm dataset to model species
!------------------------------------------------------------------
            allocate( map(ub_nspecies,2), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: map allocation error = ',ierr
               call endrun
            end if
            i        = 0
            map(:,:) = 0
species_loop : &
            do m = 1,ppcnst
	       if( ubc_from_tgcm(m) ) then
                  if( cnst_name(m) == 'N' ) then
                     ub_species_name = 'N4S'
                  else if( cnst_name(m) == 'O' ) then
                     ub_species_name = 'O1'
	          else
                     ub_species_name = cnst_name(m)
	          end if
!------------------------------------------------------------------
!	... check that requested species is in tgcm dataset
!------------------------------------------------------------------
	          ierr = nf_inq_varid( ncid, trim( ub_species_name ), vid )
	          if( ierr == NF_NOERR ) then
	             write(*,*) 'tgcm_ubc_inti : found ',trim(cnst_name(m)),' in tgcm ub dataset'
                  else
	             write(*,*) 'tgcm_ubc_inti : ',trim(cnst_name(m)),' not in tgcm ub dataset'
                     call endrun
	          end if
!------------------------------------------------------------------
!	... check dimensions for ubc variable
!------------------------------------------------------------------
                  ierr = nf_inq_varndims( ncid, vid, ndims )
	          if( ierr /= NF_NOERR ) then
	             write(*,*) 'tgcm_ubc_inti : failed to get # dimensions for species ',trim( ub_species_name )
		     call endrun
	          end if
                  if( ndims /= 4 )then
                     write(*,*) 'tgcm_ubc_inti: Error! variable vmr has ndims = ',ndims,', expecting 4'
                     call endrun
                  end if
                  ierr = nf_inq_vardimid( ncid, vid, dimid )
	          if( ierr /= NF_NOERR ) then
	             write(*,*) 'tgcm_ubc_inti : failed to get dimensions ids for species ',trim( ub_species_name )
		     call endrun
	          end if
                  if( dimid(1) /= dimid_lon .or. dimid(2) /= dimid_lat .or. &
                      dimid(3) /= dimid_lev .or. dimid(4) /= dimid_time ) then
                     write(*,*) 'tgcm_ubc_inti: Error! dimensions in wrong order for variable vmr expecting (lon,lat,day,lev)'
                     call endrun
                  end if
                  i = i + 1
                  map(i,1) = m
                  map(i,2) = vid
               end if
            end do species_loop

!-----------------------------------------------------------------------
!       ... get tgcm longitudes
!-----------------------------------------------------------------------
            allocate( lon(ub_nlon), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: lon allocation error = ',ierr
               call endrun
            end if
            call wrap_inq_varid( ncid, 'lon', vid )
            call wrap_get_var_realx( ncid, vid, lon )
            do i = 1,ub_nlon/2
               wrk    = lon(i)
	       lon(i) = lon(i+ub_nlon/2)
	       lon(i+ub_nlon/2) = wrk
            end do
            where( lon(:) < 0._r8 )
	       lon(:) = 360._r8 + lon(:)
            endwhere
            write(*,*) ' '
            write(*,*) '------------------------------------------------------------------------------'
            write(*,*) 'tgcm_ubc_inti: From lons'
            write(*,'(10f10.2)') lon(:)
            write(*,*) '------------------------------------------------------------------------------'
            lon(:ub_nlon) = lon(:ub_nlon) * d2r
!-----------------------------------------------------------------------
!       ... get tgcm latitudes
!-----------------------------------------------------------------------
            allocate( lat(ub_nlat), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: lat allocation error = ',ierr
               call endrun
            end if
            call wrap_inq_varid( ncid, 'lat', vid )
            call wrap_get_var_realx( ncid, vid, lat )
            write(*,*) ' '
            write(*,*) '------------------------------------------------------------------------------'
            write(*,*) 'tgcm_ubc_inti: From lats'
            write(*,'(10f10.2)') lat(:)
            write(*,*) '------------------------------------------------------------------------------'
            lat(:ub_nlat) = lat(:ub_nlat) * d2r
!-----------------------------------------------------------------------
!       ... setup tgcm dataset regrid to model
!-----------------------------------------------------------------------
            gndx = regrid_inti( ub_nlat, plat, &
                                ub_nlon, plon, &
                                lon,  clon_p(:,1), &
                                lat,  clat_p, &
                                0, &
                                do_lons=.true.,do_lats=.true. )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: regrider failed; error = ',gndx
               call endrun
            end if
            jl = 1
            ju = plat
            jlim_in(:) = regrid_lat_limits( gndx )
            call regrid_diagnostics( gndx )
            deallocate( lat, lon, stat=ierr )

            write(*,'(''tgcm_ubc_inti: gndx='',I2,'', grid limits = '',2i4,'', jl,ju='',2i4)') &
               gndx, jlim_in, jl, ju

!-----------------------------------------------------------------------
!       ... get vertical levels
!-----------------------------------------------------------------------
            call wrap_inq_dimid( ncid, 'lev', dimid_lev )
            call wrap_inq_dimlen( ncid, dimid_lev, ub_nlevs )
            allocate( ub_plevs(ub_nlevs), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: ub_plevs allocation error = ',ierr
               call endrun
            end if
            call wrap_inq_varid( ncid, 'lev', vid )
            call wrap_get_var_realx( ncid, vid, ub_plevs )
            call wrap_inq_varid( ncid, 'p0', vid )
            call wrap_get_var_realx( ncid, vid, p0 )
!-----------------------------------------------------------------------
!	... convert incoming ln p values to pascals
!-----------------------------------------------------------------------
            ub_plevs(:) = p0 * exp( -ub_plevs(:) ) * 1.e-1_r8
            do k = 1,ub_nlevs/2
               wrk         = ub_plevs(k)
	       ub_plevs(k) = ub_plevs(ub_nlevs-k+1)
	       ub_plevs(ub_nlevs-k+1) = wrk
            end do
            write(*,*) '------------------------------------------------------------------------------'
            write(*,*) 'tgcm_ubc_inti: plevs( Pa )'
            write(*,'(1p,5e12.5)') ub_plevs(:)
            write(*,*) '------------------------------------------------------------------------------'
!-----------------------------------------------------------------------
!       ... get times (days of year)
!-----------------------------------------------------------------------
            allocate( days(ub_ndays),stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: days allocation error = ',ierr
               call endrun
            end if
            call wrap_inq_varid( ncid, 'day', vid )
            call wrap_get_var_realx( ncid, vid, days )

!------------------------------------------------------------------
!	... allocate and read tgcm ubc array
!------------------------------------------------------------------
            allocate( mr_ub_in(ub_nlon,jlim_in(1):jlim_in(2),ub_nlevs,ub_ndays,ub_nspecies), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: mr_ub_in allocation error = ',ierr
               call endrun
            end if
            do m = 1,ub_nspecies
!------------------------------------------------------------------
!	... read in the tgcm ub mixing ratio values
!------------------------------------------------------------------
              start = (/ 1, jlim_in(1), 1, 1 /)
              cnt   = (/ ub_nlon,jlim_in(2) - jlim_in(1) + 1, ub_nlevs, ub_ndays /)
	      write(*,*) 'tgcm_ubc_inti: Reading ubc for ',trim(cnst_name(map(m,1)))
	      call wrap_get_vara_realx( ncid, map(m,2), start, cnt, mr_ub_in(1,jlim_in(1),1,1,m) )
            end do
            write(*,*) 'tgcm_ubc_inti: map = ',map(:,1)
            call wrap_close( ncid )

!--------------------------------------------------------------------
!	... regrid
!--------------------------------------------------------------------
!	... first the longitude reorder
!--------------------------------------------------------------------
         do spcno = 1,ub_nspecies
            do day = 1,ub_ndays
               do lev = 1,ub_nlevs
                  do j = jlim_in(1),jlim_in(2)
                     do i = 1,ub_nlon/2
                        wrk                         = mr_ub_in(i,j,lev,day,spcno)
                        mr_ub_in(i,j,lev,day,spcno) = mr_ub_in(i+ub_nlon/2,j,lev,day,spcno)
	                mr_ub_in(i+ub_nlon/2,j,lev,day,spcno) = wrk
                     end do
	          end do
	       end do
	    end do
!--------------------------------------------------------------------
!	... then the level reorder
!--------------------------------------------------------------------
            do day = 1,ub_ndays
               do j = jlim_in(1),jlim_in(2)
                  do i = 1,ub_nlon
                     do lev = 1,ub_nlevs/2
                        wrk                         = mr_ub_in(i,j,lev,day,spcno)
                        mr_ub_in(i,j,lev,day,spcno) = mr_ub_in(i,j,ub_nlevs-lev+1,day,spcno)
	                mr_ub_in(i,j,ub_nlevs-lev+1,day,spcno) = wrk
                     end do
	          end do
	       end do
	    end do
         end do

!--------------------------------------------------------------------
!	... setup the pressure interpolation
!--------------------------------------------------------------------
            do k = 1,pverp
	       pint(k) = ps0 * hyai(k)
            end do
            do j = 1,plat
               do k = 1,klower
	          do i = 1,plon
	             if( pint(k) <= ub_plevs(1) ) then
	                kl_g(i,k,j) = 1
	                ku_g(i,k,j) = 1
	                delp_g(i,k,j) = 0._r8
	             else if( pint(k) >= ub_plevs(ub_nlevs) ) then
	                kl_g(i,k,j) = ub_nlevs
	                ku_g(i,k,j) = ub_nlevs
	                delp_g(i,k,j) = 0._r8
	             else
	                pinterp = pint(k)
	                do k1 = 2,ub_nlevs
	                   if( pinterp <= ub_plevs(k1) ) then
		              ku_g(i,k,j) = k1
		              kl_g(i,k,j) = k1 - 1
		              delp_g(i,k,j) = log( pinterp/ub_plevs(k1) ) / log( ub_plevs(k1-1) /ub_plevs(k1) )
		              exit
		           end if
	                end do
	             end if
	          end do
               end do
            end do

            kmin = minval(kl_g)
            kmax = maxval(ku_g)
            write(*,*) ' '
            write(*,*) 'tgcm_ubc_inti: min kl, max ku = ',kmin,kmax
            write(*,*) ' '

!------------------------------------------------------------------
!	... allocate tgcm ubc array on waccm grid
!------------------------------------------------------------------
            allocate( mr_ub(plon,kmin:kmax,ub_nspecies,plat,ub_ndays), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: mr_ub allocation error = ',ierr
               write(*,*) '         plon,ub_nspecies,plat,ub_ndays = ', plon,ub_nspecies,plat,ub_ndays
               call endrun
            end if

!--------------------------------------------------------------------
!	... do horizontal regridding
!--------------------------------------------------------------------
            do day = 1,ub_ndays
               do spcno = 1,ub_nspecies
                  do lev = kmin,kmax
	             call regrid_2d( mr_ub_in(:,jlim_in(1):jlim_in(2),lev,day,spcno), mr_ub(:,lev,spcno,:,day), &
	                             gndx, jl, ju, do_poles=.true. ) 
	          end do
	       end do
            end do
            deallocate( mr_ub_in )
            deallocate( ub_plevs )
         end if has_tgcm_ubc
      end if Masterproc_only
#ifdef SPMD
!--------------------------------------------------------------------
!	... broadcast variables
!--------------------------------------------------------------------
      call mpibcast( ub_nspecies, 1, mpiint, 0, mpicom )
      if( ub_nspecies /= 0 ) then
         call mpibcast( ub_ndays, 1, mpiint, 0, mpicom )
         if( .not. masterproc ) then
            allocate( days(ub_ndays),stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: days allocation error = ',ierr
               call endrun
            end if
            allocate( map(ub_nspecies,2), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: map allocation error = ',ierr
               call endrun
            end if
         end if
         call mpibcast( days, ub_ndays, mpir8, 0, mpicom )
         call mpibcast( map, 2*ub_nspecies, mpiint, 0, mpicom )
         call mpibcast( ub_nlevs, 1, mpiint, 0, mpicom )
         call mpibcast( kmin, 1, mpiint, 0, mpicom )
         call mpibcast( kmax, 1, mpiint, 0, mpicom )
         if( .not. masterproc ) then
            allocate( mr_ub(plon,kmin:kmax,ub_nspecies,plat,ub_ndays), stat=ierr )
            if( ierr /= 0 ) then
               write(*,*) 'tgcm_ubc_inti: mr_ub allocation error = ',ierr
               call endrun
            end if
         end if
         call mpibcast( mr_ub, plon*(kmax-kmin+1)*ub_nspecies*plat*ub_ndays, mpir8, 0, mpicom )
         call mpibcast( kl_g, plon*klower*plat, mpiint, 0, mpicom )
         call mpibcast( ku_g, plon*klower*plat, mpiint, 0, mpicom )
         call mpibcast( delp_g, plon*klower*plat, mpir8, 0, mpicom )
      end if
#endif
      if( ub_nspecies /= 0 ) then
!--------------------------------------------------------------------
!	... move from wrk array to module array
!--------------------------------------------------------------------
         allocate( tgcm_ub_mmr(pcols,kmin:kmax,ub_nspecies,ub_ndays,begchunk:endchunk),stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'tgcm_ubc_inti: tgcm_ub_mmr allocation error = ',ierr
            call endrun
         end if
         allocate( kl(pcols,klower,begchunk:endchunk), &
                   ku(pcols,klower,begchunk:endchunk), &
                   delp(pcols,klower,begchunk:endchunk),stat=ierr )
         if( ierr /= 0 ) then
            write(*,*) 'tgcm_ubc_inti: kl,ku,delp allocation error = ',ierr
            call endrun
         end if
chunk_loop : &
         do c = begchunk,endchunk
            ncol = get_ncols_p( c )
            call get_lon_all_p( c, ncol, lonndx )
            call get_lat_all_p( c, ncol, latndx )
            if( zonal_average ) then
               do day = 1,ub_ndays
                  do spcno = 1,ub_nspecies
                     do lev = kmin,kmax
                        do i = 1,ncol
                           jc = latndx(i)
                           tgcm_ub_mmr(i,lev,spcno,day,c) = &
                                  sum( mr_ub(:plon,lev,spcno,jc,day) )/plon
                        end do
                     end do
                  end do
               end do
            else
               do day = 1,ub_ndays
                  do spcno = 1,ub_nspecies
                     do lev = kmin,kmax
                        do i = 1,ncol
                           ic = lonndx(i)
                           jc = latndx(i)
                           tgcm_ub_mmr(i,lev,spcno,day,c) = &
                                  mr_ub(ic,lev,spcno,jc,day)
                        end do
                     end do
                  end do
               end do
            end if
            do lev = 1,klower
               do i = 1,ncol
                  ic = lonndx(i)
                  jc = latndx(i)
                  kl(i,lev,c)   = kl_g(ic,lev,jc)
                  ku(i,lev,c)   = ku_g(ic,lev,jc)
                  delp(i,lev,c) = delp_g(ic,lev,jc)
               end do
            end do
         end do chunk_loop
         deallocate( mr_ub )
      end if

      stepsize = get_step_size()

      end subroutine tgcm_ubc_inti

      subroutine tgcm_timestep_init
!--------------------------------------------------------------------
!	... Check timing for ub values
!--------------------------------------------------------------------

      use time_manager, only : get_curr_calday, get_nstep, get_calday, &
                               get_curr_date, is_first_step, is_first_restart_step
      use pmgrid,       only : masterproc

      implicit none

!--------------------------------------------------------------------
!	... Local variables
!--------------------------------------------------------------------
      real(r8), parameter :: dayspy = 365._r8
      integer  ::  m, upper
      integer  ::  yr, mon, day, tod, nstep
      integer  ::  date
      integer  ::  offset
      real(r8) ::  numer, denom
      real(r8) ::  calday

      if( ub_nspecies /= 0 ) then
         nstep = get_nstep()
         if( is_first_step() .or. is_first_restart_step() .or. mod( nstep,tgcm_frq ) == 0 ) then
	    offset = mod( nstep,tgcm_frq )
	    if( offset /= 0 ) then
	       offset = -offset*stepsize
	    end if
            call get_curr_date( yr, mon, day, tod, offset )
	    date   = 10000*yr + 100*mon + day
            calday = get_calday( date, tod )
!--------------------------------------------------------
!	... setup the time interpolation
!--------------------------------------------------------
#ifdef TGCM_DIAGS
	    if( masterproc ) then
	       write(*,*) '===================================='
	       write(*,*) 'tgcm_timestep_init: diagnostics'
	       write(*,*) 'nstep,yr,mon,day,tod,date,calday,offset,tgcm_frq = ',nstep, yr, mon, day, tod, date, calday, offset, tgcm_frq
	       write(*,*) '===================================='
	    end if
#endif
            if( calday < days(1) ) then
	       next = 1
	       last = ub_ndays
            else
	       if( days(ub_ndays) < dayspy ) then
	          upper = ub_ndays
	       else
	          upper = ub_ndays - 1
	       end if
               do m = upper,1,-1
	          if( calday >= days(m) ) then
	             exit
	          end if
               end do
	       last = m
	       next = mod( m,ub_ndays ) + 1
            end if
            numer = calday - days(last)
            denom = days(next) - days(last)
            if( numer < 0. ) then
	       numer = dayspy + numer
            end if
            if( denom < 0. ) then
	          denom = dayspy + denom
            end if
            dels = max( min( 1._r8,numer/denom ),0._r8 )
         end if
      end if

      end subroutine tgcm_timestep_init

      subroutine set_tgcm_ubc( lchunk, ncol, mmr, mbartop )
!--------------------------------------------------------------------
!	... Set the upper boundary values h2o, h2, and h
!--------------------------------------------------------------------

      use m_spc_id
      use ppgrid,       only : pcols
      use constituents, only : cnst_get_ind, cnst_mw

      implicit none

!--------------------------------------------------------------------
!	... dummy args
!--------------------------------------------------------------------
      integer, intent(in)     :: lchunk            ! chunk id
      integer, intent(in)     :: ncol              ! columns in chunk
!+hi-waccm
      real(r8), intent(in)    :: mbartop(pcols)      ! composition dependent mean mass
!-hi-waccm
      real(r8), intent(inout) :: mmr(pcols,ppcnst)

!--------------------------------------------------------------------
!	... local variables
!--------------------------------------------------------------------
      real(r8), parameter ::  h2o_ubc_vmr = 2.e-8_r8            ! fixed ub h2o concentration (kg/kg)
      real(r8), parameter ::  ch4_ubc_vmr = 2.e-10_r8           ! fixed ub h2o concentration (kg/kg)
      integer  ::  m, i, k, k1, k2, n
      real(r8) ::  pint_vals(2)

!--------------------------------------------------------
!	... set the mixing ratios at upper boundary from tgcm
!--------------------------------------------------------
      if( ub_nspecies /= 0 ) then
         do k = 1,klower
            do m = 1,ub_nspecies
               n = map(m,1)
               do i = 1,ncol
	          k1 = kl(i,k,lchunk)
	          k2 = ku(i,k,lchunk)
	          pint_vals(1) = tgcm_ub_mmr(i,k1,m,last,lchunk) &
                               + delp(i,k,lchunk) * (tgcm_ub_mmr(i,k2,m,last,lchunk) - tgcm_ub_mmr(i,k1,m,last,lchunk))
	          pint_vals(2) = tgcm_ub_mmr(i,k1,m,next,lchunk) &
                               + delp(i,k,lchunk) * (tgcm_ub_mmr(i,k2,m,next,lchunk) - tgcm_ub_mmr(i,k1,m,next,lchunk))
	          mmr(i,n) = pint_vals(1) + dels * (pint_vals(2) - pint_vals(1))
               end do
            end do
         end do
      end if

!--------------------------------------------------------
!	... special section to set h2o and ch4 ub concentrations
!--------------------------------------------------------
!+hi-waccm
      mmr(:ncol,1) = cnst_mw(1)*h2o_ubc_vmr/mbartop(:ncol)
      call cnst_get_ind( 'CH4', m, abort=.false. )
      if( m > 0 ) then
         mmr(:ncol,m) = cnst_mw(m)*ch4_ubc_vmr/mbartop(:ncol)
      end if
!-hi-waccm

#ifdef TGCM_DIAGS
         call cnst_get_ind( 'H2', m, abort=.false. )
         if( m > 0 ) then
            write(*,*) 'set_ub_vals: diagnostics for chunk = ',lchunk
            write(*,*) 'last,next,dels = ',last,next,dels
            write(*,*) 'h2 mmr at level ',k
            write(*,'(1x,1p,10g12.5)') mmr(:ncol,m))
         end if
#endif

      end subroutine set_tgcm_ubc

      end module mo_tgcm_ubc
