module physpkg
!-----------------------------------------------------------------------
! Purpose:
!
! Provides the interface to CAM physics package
!
! Revision history:
! Aug  2005,  E. B. Kluzek,  Creation of module from physpkg subroutine
! 2005-10-17  B. Eaton       Add contents of inti.F90 to phys_init().  Add
!                            initialization of grid info in phys_state.
!-----------------------------------------------------------------------
  use shr_kind_mod,     only: r8 => shr_kind_r8
  use spmd_utils,       only: masterproc
  use physics_types,    only: physics_state, physics_tend, physics_state_set_grid
  use phys_grid,        only: get_ncols_p, get_lat_all_p, get_lon_all_p
  use ppgrid,           only: begchunk, endchunk, pcols
  use constituents,     only: pcnst, cnst_name, cnst_need_pdeldry, cnst_get_ind
  use camsrfexch_types, only: surface_state, srfflx_state
  use phys_buffer,      only: pbuf_size_max, pbuf_fld
  use cam_control_mod,  only: ideal_phys, adiabatic
  use scamMod,         only: single_column, scm_crm_mode
#ifdef SPMD
  use mpishorthand
#endif
   use perf_mod
#if ( defined WACCM_GHG || defined WACCM_MOZART )
   use qbo,             only: qbo_init
#endif

   implicit none
   private
   save

! Public methods
   public phys_init   ! Public initialization method
   public phys_run1   ! First phase of the public run method
   public phys_run2   ! Second phase of the public run method
   public phys_final  ! Public finalization method
!
! Private module data
!

!======================================================================= 
contains
!======================================================================= 

subroutine phys_inidat( surface_state2d )
  use abortutils, only : endrun
  use buffer, only : tpert, qpert,pblht
  use phys_buffer,      only: pbuf, pbuf_times, pbuf_get_fld_idx
  use startup_initialconds, only : initial_file_get_id
  use ppgrid,         only: pver, pverp
  use ncdio_atm,          only: infld
  use dycore,            only: dycore_is
  use camsrfexch_types, only: surface_state
  
  type(surface_state),         intent(inout) :: surface_state2d(begchunk:endchunk)
  integer :: lchnk, ncid_ini, m, n, i, k, ncol
  character(len=8) :: fieldname
  real(r8), pointer :: cldptr(:,:,:,:), convptr_3d(:,:,:,:)
  real(r8), pointer :: tptr(:,:), tptr3d(:,:,:), t3dtmp(:,:,:)

  character*11 :: subname='phys_inidat' ! subroutine name

  logical :: found=.false.
  integer :: ierr
  character(len=4) :: dim1name
  integer :: ixcldice, ixcldliq
  nullify(t3dtmp,tptr,tptr3d,cldptr,convptr_3d)
  ncid_ini=initial_file_get_id()

!   dynamics variables are handled in dyn_init - here we read variables needed for physics 
!   but not dynamics

  if(dycore_is('homme')) then  
     dim1name='ncol'
  else
     dim1name='lon'
  end if
  call infld('PBLH', ncid_ini, dim1name, 'lat', 1, pcols, begchunk, endchunk, &
       pblht, found, grid_map='PHYS')
  if(.not. found) then
     pblht(:,:) = 0._r8
     if (masterproc) write(6,*) 'PBLH initialized to 0.'
  end if

  call infld('TPERT', ncid_ini, dim1name, 'lat', 1, pcols, begchunk, endchunk, &
       tpert, found, grid_map='PHYS')
  if(.not. found) then
     tpert(:,:) = 0._r8
     if (masterproc) write(6,*) 'TPERT initialized to 0.'
  end if

  fieldname='QPERT'  
  qpert(:,:,:) = 0._r8
  allocate(tptr(1:pcols,begchunk:endchunk))
  call infld(fieldname, ncid_ini, dim1name, 'lat', 1, pcols, begchunk, endchunk, &
       tptr, found, grid_map='PHYS')
  if(found) then
     do lchnk=begchunk,endchunk
        qpert(:,1,lchnk) = tptr(:,lchnk)
     end do
  else
     if (masterproc) write(6,*) trim(fieldname), ' initialized to 0.'
  end if
  deallocate(tptr)
   
  fieldname='CUSH'
  m = pbuf_get_fld_idx('cush')
  tptr3d => pbuf(m)%fld_ptr(1,1:pcols,1,begchunk:endchunk,1:pbuf_times)
  call infld(fieldname, ncid_ini, dim1name, 'lat', 1, pcols, begchunk, endchunk, &
       tptr3d(:,:,1), found, grid_map='PHYS')
  if(.not.found) then
     if(masterproc) write(6,*) trim(fieldname), ' initialized to 1000.'
     tptr3d=1000._r8
  else
     do n=2,pbuf_times
        tptr3d(:,:,n)=tptr3d(:,:,1)
     end do
  end if

  fieldname='TBOT'
  allocate(tptr(1:pcols,begchunk:endchunk))
  call infld(fieldname, ncid_ini, dim1name, 'lat', 1, pcols, begchunk, endchunk, &
       tptr, found, grid_map='PHYS')
  if(.not.found) then
     allocate(t3dtmp(1:pcols,1:pver,begchunk:endchunk))
     call infld('T',ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
          t3dtmp, found, grid_map='PHYS',poleavg_in=(-1))
     if(masterproc) write(6,*) trim(fieldname), ' initialized to lowest level of T.'
     tptr(:,:)=t3dtmp(:,pver,:)
  end if

!
! 3-D fields
!
  fieldname='CLOUD'
  m = pbuf_get_fld_idx('CLD')
  if(associated(pbuf(m)%fld_ptr)) then
     cldptr => pbuf(m)%fld_ptr(1,:,:,begchunk:endchunk,:)
  else
     call endrun('pbuf not allocated in phys_inidat') 
  end if
  cldptr=0._r8
  call infld(fieldname,ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
       cldptr(:,:,:,1), found, grid_map='PHYS')
  if(.not. found .and. masterproc) write(6,*) trim(fieldname), ' initialized to 0.'
  do n = 2, pbuf_times
     cldptr(:,:,:,n) = cldptr(:,:,:,1)
  end do

  fieldname = 'QCWAT'
  m = pbuf_get_fld_idx(fieldname)
  if(associated(pbuf(m)%fld_ptr)) then
     cldptr => pbuf(m)%fld_ptr(1,:,:,begchunk:endchunk,:)
  else
     call endrun('pbuf not allocated in phys_inidat') 
  end if
  cldptr=0._r8
  call infld(fieldname,ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
       cldptr(:,:,:,1), found, grid_map='PHYS')
  if(.not.found) then
     call infld('Q',ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
          cldptr(:,:,:,1), found, grid_map='PHYS',poleavg_in=1)
     if (found) then
        if (masterproc) write(6,*) trim(fieldname), ' initialized with Q'
     else
        call endrun('  '//trim(subname)//' Error:  Q must be on Initial File')
     end if
  end if
  do n = 2, pbuf_times
     cldptr(:,:,:,n) = cldptr(:,:,:,1)
  end do 


  fieldname = 'LCWAT'
  m = pbuf_get_fld_idx(fieldname)

  if(associated(pbuf(m)%fld_ptr)) then
     cldptr => pbuf(m)%fld_ptr(1,:,:,begchunk:endchunk,:)
  else
     call endrun('pbuf not allocated in phys_inidat') 
  end if

  cldptr=0._r8
  call infld(fieldname,ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
       cldptr(:,:,:,1), found, grid_map='PHYS')
  if(.not. found) then
     allocate(tptr3d(pcols,pver,begchunk:endchunk))     
     call cnst_get_ind('CLDICE', ixcldice)
     call cnst_get_ind('CLDLIQ', ixcldliq)
     call infld('CLDICE',ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
          tptr3d, found, grid_map='PHYS',poleavg_in=ixcldice)
     if(found) then
        cldptr(:,:,:,1)=tptr3d(:,:,:)
     end if
     call infld('CLDLIQ',ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
          tptr3d, found, grid_map='PHYS',poleavg_in=ixcldliq)
     if(found) then
        cldptr(:,:,:,1)=cldptr(:,:,:,1)+tptr3d(:,:,:)
     end if
     if (masterproc) then
        if (found) then
           write(6,*) trim(fieldname), ' initialized with CLDICE + CLDLIQ'
        else
           write(6,*) trim(fieldname), ' initialized to 0.0'
        end if
     end if
     deallocate(tptr3d)
  end if
  do n = 2, pbuf_times
     cldptr(:,:,:,n) = cldptr(:,:,:,1)
  end do

  fieldname = 'TCWAT'
  m = pbuf_get_fld_idx(fieldname)
  if(associated(pbuf(m)%fld_ptr)) then
     cldptr => pbuf(m)%fld_ptr(1,:,:,begchunk:endchunk,:)
  else
     call endrun('pbuf not allocated in phys_inidat') 
  end if
  cldptr=0._r8
  call infld(fieldname,ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
       cldptr(:,:,:,1), found, grid_map='PHYS')
  if(.not. found) then
     if(associated(t3dtmp)) then
        cldptr(:,:,:,1)=t3dtmp(:,:,:)
     else
        call infld('T',ncid_ini,dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
             cldptr(:,:,:,1), found, grid_map='PHYS',poleavg_in=(-1))
     end if
     if (masterproc) write(6,*) trim(fieldname), ' initialized with T'
  end if
  do n = 2, pbuf_times
     cldptr(:,:,:,n) = cldptr(:,:,:,1)
  end do
  if(associated(t3dtmp)) deallocate(t3dtmp)

  fieldname = 'TKE'
  m = pbuf_get_fld_idx('tke')
  convptr_3d => pbuf(m)%fld_ptr(1,1:pcols,1:pverp,begchunk:endchunk,1:pbuf_times)
  call infld(fieldname, ncid_ini, dim1name, 'ilev', 'lat', 1, pcols, 1, pverp, begchunk, endchunk, &
       convptr_3d(:,:,:,1), found, grid_map='phys')
  if(.not. found) then
     convptr_3d(:,:,:,1) = 0.01_r8
     if (masterproc) write(6,*) trim(fieldname), ' initialized to 0.01'
  end if
  do n = 2, pbuf_times
     convptr_3d(:,:,:,n) = convptr_3d(:,:,:,1)
  end do

  fieldname = 'KVM'
  m = pbuf_get_fld_idx('kvm')
  convptr_3d => pbuf(m)%fld_ptr(1,1:pcols,1:pverp,begchunk:endchunk,1:pbuf_times)
  call infld(fieldname, ncid_ini, dim1name, 'ilev', 'lat', 1, pcols, 1, pverp, begchunk, endchunk, &
       convptr_3d(:,:,:,1), found, grid_map='phys')
  if(.not. found) then
     convptr_3d(:,:,:,1) = 0._r8
     if (masterproc) write(6,*) trim(fieldname), ' initialized to 0.'
  end if
  do n = 2, pbuf_times
     convptr_3d(:,:,:,n) = convptr_3d(:,:,:,1)
  end do

  fieldname = 'KVH'
  m = pbuf_get_fld_idx('kvh')
  convptr_3d => pbuf(m)%fld_ptr(1,1:pcols,1:pverp,begchunk:endchunk,1:pbuf_times)
  call infld(fieldname, ncid_ini, dim1name, 'ilev', 'lat', 1, pcols, 1, pverp, begchunk, endchunk, &
       convptr_3d(:,:,:,1), found, grid_map='phys')
  if(.not. found) then
     convptr_3d(:,:,:,1) = 0._r8
     if (masterproc) write(6,*) trim(fieldname), ' initialized to 0.'
  end if
  do n = 2, pbuf_times
     convptr_3d(:,:,:,n) = convptr_3d(:,:,:,1)
  end do

  fieldname = 'CONCLD'
  m = pbuf_get_fld_idx('CONCLD')
  convptr_3d => pbuf(m)%fld_ptr(1,1:pcols,1:pver,begchunk:endchunk,1:pbuf_times)
  call infld(fieldname, ncid_ini, dim1name, 'lev', 'lat', 1, pcols, 1, pver, begchunk, endchunk, &
       convptr_3d(:,:,:,1), found, grid_map='phys')
  if(.not. found) then
     convptr_3d(:,:,:,1) = 0.
     if (masterproc) write(6,*) trim(fieldname), ' initialized to 0.'
  end if
  do n = 2, pbuf_times
     convptr_3d(:,:,:,n) = convptr_3d(:,:,:,1)
  end do

end subroutine phys_inidat


subroutine phys_init( phys_state, phys_tend, surface_state2d )
!----------------------------------------------------------------------- 
! 
! Purpose: 
! Initialization of physics package.
! 
!-----------------------------------------------------------------------

   use shr_const_mod,      only: shr_const_zvir, shr_const_cpwv, shr_const_rwv
   use physconst,          only: rair, cpair, cpwv, gravit, stebol, epsilo, tmelt, &
                                 latvap, latice, rh2o, zvir, cpvir, rhoh2o, pstd,  &
                                 karman, rhodair 
   use hycoef,             only: hypi, hypm

   use aerosol_intr,       only: prognostic_aerosol_initialize
   use aerosol_radiation_interface, only: aerosol_radiation_init
   use aer_optics,         only: aer_optics_initialize
   use cam_control_mod,    only: nsrest  ! restart flag
   use check_energy,       only: check_energy_init
   use chemistry,          only: chem_init
   use cloud_fraction,     only: cldfrc_init
   use cloudsimulator,     only: doisccp, cloudsimulator_init
   use co2_cycle,          only: co2_init, co2_transport
   use convect_deep,  only: convect_deep_init
   use convect_shallow,    only: convect_shallow_init
   use cam_diagnostics,    only: diag_init
   use ghg_defaults,       only: ghg_defaults_init
   use gw_drag,            only: gw_inti
   use ozone_data,         only: ozone_data_init
   use param_cldoptics,    only: param_cldoptics_init
   use prescribed_aerosols,only: aerosol_initialize
   use radheat,            only: radheat_init
   use radiation,          only: radiation_init
   use stratiform,         only: stratiform_init
   use tracers,            only: tracers_init
   use vertical_diffusion, only: vertical_diffusion_init
   use water_tracers,      only: wtrc_init
   use dycore,             only: dycore_is

#if ( defined WACCM_GHG || defined WACCM_MOZART )
   use ctem,               only: ctem_inti
   use iondrag,            only: iondrag_init
#endif
#if ( defined OFFLINE_DYN )
   use metdata,            only: metdata_phys_init
#endif

   ! Input/output arguments
   type(physics_state), pointer :: phys_state(:)
   type(physics_tend ), pointer :: phys_tend(:)
   type(surface_state),         intent(inout) :: surface_state2d(begchunk:endchunk)

   ! local variables
   integer :: lchnk

!-----------------------------------------------------------------------
   allocate(phys_state(begchunk:endchunk))
   allocate(phys_tend(begchunk:endchunk))

   ! Set chunk id, number of columns, and coordinates
   do lchnk = begchunk,endchunk
      call physics_state_set_grid(lchnk, phys_state(lchnk))
   end do

   if(nsrest.eq.0) then
      call phys_inidat(surface_state2d) 
   end if

   ! Initialize prescripted aerosols, and aerosol-radiation interface
   call aerosol_initialize(phys_state(begchunk:endchunk))
   call aerosol_radiation_init

   ! Initialize prognostic aerosols
   call prognostic_aerosol_initialize(phys_state(begchunk:endchunk))

   ! Initialize physconst variables
   ! In adiabatic case, set zvir and cpvir explicitly to zero instead of 
   ! computing as (rh2o/rair - 1.) and (cpwv/cpair - 1.) respectively, in order 
   ! to guarantee an identical zero.
   if (adiabatic .or. ideal_phys) then
      rh2o  = rair
      zvir  = 0._r8
      cpwv  = cpair
      cpvir = 0._r8
   else
      rh2o  = shr_const_rwv
      zvir  = shr_const_zvir
      cpwv  = shr_const_cpwv
      cpvir = cpwv/cpair - 1._r8
   end if

   ! Water tracers
   call wtrc_init()

   ! Prognostic chemistry.
   call chem_init(phys_state(begchunk:endchunk))
 
   ! Initialize aerosol optical properties.
   ! ** N.B. ** This call moved to  be after the aerosol and chemistry initializations
   !            because the dust optics depends on which package supplies dust mass.
   call aer_optics_initialize()

   ! co2 cycle            
   if (co2_transport()) then
      call co2_init()
   end if

   ! Default distributions for CH4, N2O, CFC11 and CFC12.
   call ghg_defaults_init()

   ! Initialize reading of ozone dataset.
   call ozone_data_init(phys_state(begchunk:endchunk))

   call tracers_init

   call gw_inti (cpair   ,cpwv    ,gravit  ,rair    ,hypi    )

   call vertical_diffusion_init

   call tsinti  (tmelt   ,latvap  ,rair    ,stebol  ,latice  )

   call radiation_init

   call radheat_init(hypm)

   call esinti  (epsilo  ,latvap  ,latice  ,rh2o    ,cpair  , &
                 tmelt   )
   call convect_shallow_init(hypi)

   call cldfrc_init


   call convect_deep_init(hypi)

   call cldinti ()

   call stratiform_init

   call param_cldoptics_init

   call check_energy_init

   if (doisccp) call cloudsimulator_init

   call diag_init()

#if ( defined WACCM_GHG || defined WACCM_MOZART )
   call ctem_inti
   call iondrag_init( hypm )
   call qbo_init
#endif

#if ( defined OFFLINE_DYN )
   call metdata_phys_init()
#endif
 
end subroutine phys_init

!
!-----------------------------------------------------------------------
!

subroutine phys_run1(phys_state, etamid, gw, ztodt, phys_tend, pbuf, cam_in, cam_out)
!----------------------------------------------------------------------- 
! 
! Purpose: 
! First part of atmospheric physics package before updating of surface models
! 
!-----------------------------------------------------------------------
   use ppgrid,         only: pver
   use time_manager,   only: get_nstep
   use cam_diagnostics,only: diag_allocate, diag_physvar_ic
   use check_energy,   only: check_energy_gmean
   use phys_buffer,    only: pbuf_allocate
   use buffer,         only: pblht, tpert, qpert
#if (defined BFB_CAM_SCAM_IOP )
   use cam_history,    only: outfld
#endif
   use comsrf,         only: fsns, fsnt, flns, flnt, landm, fsds
!
! Input arguments
!
   real(r8), intent(in) :: etamid(pver)     ! vertical coords at midpoints
   real(r8), intent(in) :: gw(:)         ! Gaussian weights
   real(r8), intent(in) :: ztodt            ! physics time step unless nstep=0
!
! Input/Output arguments
!
   type(physics_state), intent(inout), dimension(begchunk:endchunk) :: phys_state
   type(physics_tend ), intent(inout), dimension(begchunk:endchunk) :: phys_tend
   type(pbuf_fld),      intent(inout), dimension(pbuf_size_max)     :: pbuf
   type(srfflx_state),                 dimension(begchunk:endchunk) :: cam_in
   type(surface_state),                dimension(begchunk:endchunk) :: cam_out
!-----------------------------------------------------------------------
!
!---------------------------Local workspace-----------------------------
!
   integer :: c                                 ! indices
   integer :: ncol                              ! number of columns
   integer :: nstep                             ! current timestep number
#if (! defined SPMD)
   integer  :: mpicom = 0
#endif

   if ( adiabatic .or. ideal_phys )then
      call phys_run1_adiabatic_or_ideal(phys_state, etamid, ztodt, phys_tend, cam_in )
   else
      call t_startf ('physpkg_st1')
      nstep = get_nstep()

      call pbuf_allocate('physpkg')
      call diag_allocate()

      ! Compute total energy of input state and previous output state
      call t_startf ('chk_en_gmean')
      call check_energy_gmean(phys_state, pbuf, ztodt, nstep)
      call t_stopf ('chk_en_gmean')

      !-----------------------------------------------------------------------
      ! Advance time information
      !-----------------------------------------------------------------------

      call advnce( phys_state )

      call t_stopf ('physpkg_st1')

#ifdef TRACER_CHECK
      call gavglook ('before tphysbc DRY', phys_state, gw)
#endif


      !-----------------------------------------------------------------------
      ! Tendency physics before flux coupler invocation
      !-----------------------------------------------------------------------
      !

#if (defined BFB_CAM_SCAM_IOP )
      do c=begchunk, endchunk
         call outfld('Tg',cam_in(c)%ts,pcols   ,c     )
      end do
#endif
      call t_barrierf('sync_bc_physics', mpicom)
      call t_startf ('bc_physics')
      call t_adj_detailf(+1)

!$OMP PARALLEL DO PRIVATE (C)

      do c=begchunk, endchunk
         !
         ! Output physics terms to IC file
         !
         call t_startf ('diag_physvar_ic')
         call diag_physvar_ic ( c, pbuf, cam_out(c), cam_in(c) )
         call t_stopf ('diag_physvar_ic')

         call tphysbc (ztodt, pblht(1,c), tpert(1,c), qpert(1,1,c),                      &
                       fsns(1,c), fsnt(1,c), flns(1,c), flnt(1,c), phys_state(c),        &
                       phys_tend(c), pbuf,  fsds(1,c), landm(1,c),                       &
                       cam_out(c), cam_in(c))

      end do
      call t_adj_detailf(-1)
      call t_stopf ('bc_physics')

! Don't call the rest in CRM mode
      if(single_column.and.scm_crm_mode) return

#ifdef TRACER_CHECK
      call gavglook ('between DRY', phys_state, gw)
#endif
   end if

end subroutine phys_run1

!
!-----------------------------------------------------------------------
!

subroutine phys_run1_adiabatic_or_ideal(phys_state, etamid, ztodt, &
                                        phys_tend, cam_in )
!----------------------------------------------------------------------- 
! 
! Purpose: 
! Physics for adiabatic or idealized physics case.
! 
!-----------------------------------------------------------------------
   use ppgrid,           only: pver
   use geopotential ,    only: geopotential_t
   use cam_diagnostics,  only: diag_phys_writeout
   use physconst,        only: zvir, rair, gravit

!
! Input arguments
!
   real(r8), intent(in) :: etamid(pver)     ! vertical coords at midpoints
   real(r8), intent(in) :: ztodt            ! physics time step unless nstep=0
!
! Input/Output arguments
!
   type(physics_state), intent(inout), dimension(begchunk:endchunk) :: phys_state
   type(physics_tend ), intent(inout), dimension(begchunk:endchunk) :: phys_tend
   type(srfflx_state),  intent(inout), dimension(begchunk:endchunk) :: cam_in
!-----------------------------------------------------------------------
!---------------------------Local workspace-----------------------------
!
  integer  :: i,k,lchnk      ! indices
  integer  :: ncol             ! number of columns
  real(r8) rpdel(pcols,pver)   ! 1./(pintm1(k+1)-pintm1(k))
!-----------------------------------------------------------------------

!$OMP PARALLEL DO PRIVATE (I, K, LCHNK, NCOL, RPDEL)
  do lchnk=begchunk, endchunk
     ncol = get_ncols_p(lchnk)
!
! Dump dynamics variables to H.T.
!
     do k=1,pver
        do i=1,ncol
           rpdel(i,k) = 1._r8/phys_state(lchnk)%pdel(i,k)
        end do
     end do
     call geopotential_t(                                                             &
          phys_state(lchnk)%lnpint  , phys_state(lchnk)%lnpmid, phys_state(lchnk)%pint    , &
          phys_state(lchnk)%pmid    , phys_state(lchnk)%pdel  , rpdel, phys_state(lchnk)%t, &
          phys_state(lchnk)%q(:,:,1), rair , gravit, zvir , phys_state(lchnk)%zi        , &
          phys_state(lchnk)%zm      , ncol )

     call diag_phys_writeout(phys_state(lchnk))

!
! Set tendencies to 0
!
     do k=1,pver
        do i=1,ncol
           phys_tend(lchnk)%dTdt(i,k) = 0._r8
           phys_tend(lchnk)%dudt(i,k) = 0._r8
           phys_tend(lchnk)%dvdt(i,k) = 0._r8
        end do
     end do
     do i=1,ncol
        phys_tend(lchnk)%flx_net(i) = 0._r8
     end do
!
     if ( ideal_phys )then
        call t_startf('tphysidl')
        call tphysidl ( ztodt, cam_in(lchnk)%wsx, cam_in(lchnk)%wsy, &
                        etamid, phys_state(lchnk), phys_tend(lchnk) )
        call t_stopf('tphysidl')
     end if
  end do

end subroutine phys_run1_adiabatic_or_ideal

!
!-----------------------------------------------------------------------
!

subroutine phys_run2(phys_state, gw, ztodt, phys_tend, pbuf, cam_out, &
                     cam_in )
!----------------------------------------------------------------------- 
! 
! Purpose: 
! Second part of atmospheric physics package after updating of surface models
! 
!-----------------------------------------------------------------------
   use buffer,         only: pblht, tpert, qpert
#if ( defined WACCM_MOZART || defined TROPCHEM )
   use mo_lightning,   only: lightning_no_prod
#endif
   use phys_buffer,    only: pbuf_update_tim_idx
   use phys_buffer,    only: pbuf_deallocate
   use cam_diagnostics,only: diag_deallocate, diag_surf
   use comsrf,         only: trefmxav, trefmnav, sgh, sgh30, fsds 
   use physconst,      only: stebol, latvap
!
! Input arguments
!
   real(r8), intent(in) :: gw(:)                    ! Gaussian weights
   real(r8), intent(in) :: ztodt                       ! physics time step unless nstep=0
!
! Input/Output arguments
!
   type(physics_state), intent(inout), dimension(begchunk:endchunk) :: phys_state
   type(physics_tend ), intent(inout), dimension(begchunk:endchunk) :: phys_tend
   type(pbuf_fld),      intent(inout), dimension(pbuf_size_max)     :: pbuf
   type(surface_state), intent(inout), dimension(begchunk:endchunk) :: cam_out
   type(srfflx_state),  intent(inout), dimension(begchunk:endchunk) :: cam_in
!
!-----------------------------------------------------------------------
!---------------------------Local workspace-----------------------------
!
   integer :: c                                 ! chunk index
   integer :: ncol                              ! number of columns
#if (! defined SPMD)
   integer  :: mpicom = 0
#endif
   !
   ! If exit condition just return
   !

   if(single_column.and.scm_crm_mode) return

   if ( adiabatic .or. ideal_phys ) return
   !-----------------------------------------------------------------------
   ! Tendency physics after coupler 
   ! Not necessary at terminal timestep.
   !-----------------------------------------------------------------------
   !
#if ( defined WACCM_MOZART || defined TROPCHEM )
   ! Set lightning production of NO
   call t_startf ('lightning_no_prod')
   call lightning_no_prod( phys_state, pbuf, cam_in )
   call t_stopf ('lightning_no_prod')
#endif
   call t_barrierf('sync_ac_physics', mpicom)
   call t_startf ('ac_physics')
   call t_adj_detailf(+1)

!$OMP PARALLEL DO PRIVATE (C, NCOL)

   do c=begchunk,endchunk
      ncol = get_ncols_p(c)
      !
      ! surface diagnostics for history files
      !
      call t_startf ('diag_surf')
      call diag_surf (cam_in(c), cam_out(c), cam_in(c)%icefrac, &
                         cam_in(c)%ocnfrac(1), cam_in(c)%landfrac(1), &
                      trefmxav(1,c), trefmnav(1,c) )
      call t_stopf ('diag_surf')

      call tphysac (ztodt, pblht(1,c), qpert(1,1,c), tpert(1,c), cam_in(c),        &
                    sgh(1,c), sgh30(1,c), cam_out(c),                              &
                    phys_state(c), phys_tend(c), pbuf, fsds(1,c))
   end do                    ! Chunk loop

   call t_adj_detailf(-1)
   call t_stopf('ac_physics')

#ifdef TRACER_CHECK
   call gavglook ('after tphysac FV:WET)', phys_state, gw )
#endif

   call t_startf ('physpkg_st2')
   call pbuf_deallocate('physpkg')
   call pbuf_update_tim_idx()
   call diag_deallocate()
   call t_stopf ('physpkg_st2')
end subroutine phys_run2

!
!----------------------------------------------------------------------- 
!

subroutine phys_final( phys_state, phys_tend )
  use chemistry, only : chem_final
!----------------------------------------------------------------------- 
! 
! Purpose: 
! Finalization of physics package
! 
!-----------------------------------------------------------------------
   ! Input/output arguments
   type(physics_state), pointer :: phys_state(:)
   type(physics_tend ), pointer :: phys_tend(:)

   deallocate(phys_state)
   deallocate(phys_tend)
   call chem_final
end subroutine phys_final


!
!-----------------------------------------------------------------------
!

subroutine gavglook (title, state, gw)
!-----------------------------------------------------------------------
!
! process info from state vectors. Only useful when data in all chunks are in sync
! e.g. before and after tphysac and tphysbc
!
!-----------------------------------------------------------------------
  use pmgrid,           only: plon, plat

  use ppgrid,         only: pver
  use physconst,      only: gravit
  use time_manager,   only: dtime

  ! arguments
  character(len=*), intent(in) :: title
  type(physics_state), intent(in), dimension(begchunk:endchunk) :: state
  real(r8), intent(in) :: gw(:)                    ! Gaussian weights
  
  ! local
  integer i, lat, c, lon, k
  integer :: lats(pcols)                       ! array of latitude indices
  integer :: lons(pcols)                       ! array of longitude indices
  integer m
  integer :: ncol                              ! number of columns
  real(r8) twodfld(plon,plat,pcnst)            ! summed at each grid point
  real(r8) twodfle(plon,plat,pcnst)            ! summed at each grid point
  real(r8) twodflx(plon,plat,pcnst)            ! summed at each grid point
  real(r8) twodfly(plon,plat,pcnst)            ! summed at each grid point
#ifdef SPMD                                     
  real(r8) :: twodfld_glob(plon,plat,pcnst)    ! global summed at each grid point
  real(r8) :: twodfle_glob(plon,plat,pcnst)    ! global summed at each grid point
  real(r8) :: twodflx_glob(plon,plat,pcnst)    ! global summed at each grid point
  real(r8) :: twodfly_glob(plon,plat,pcnst)    ! global summed at each grid point
#endif                                          
  real(r8) :: zonal(plat), zonalw(plat)        ! summed along each latitude
  real(r8) gavg, gavgw
  real(r8) col, wmin, wmax, colw

  !!--------------------------------------------------------


  ! operations on each processor
  twodfld(:,:,:) = 0._r8
  twodfle(:,:,:) = 0._r8
  twodflx(:,:,:) = 0._r8
  twodfly(:,:,:) = 0._r8
  do c=begchunk, endchunk
     ncol = get_ncols_p(c)
     call get_lat_all_p(c, ncol, lats)
     call get_lon_all_p(c, ncol, lons)
     do m = 1,pcnst
        do i=1,ncol
           lat = lats(i)
           lon = lons(i)
           col = 0._r8
           colw = 0._r8
!           fluxcol = 0.
           wmax = -1.e36_r8
           wmin = 1.e36_r8
           do k = 1,pver
              if ( cnst_need_pdeldry)  col  = col + state(c)%pdeldry(i,k)*state(c)%q(i,k,m)*gw(lats(i))
              colw = colw + state(c)%pdel(i,k)  *state(c)%q(i,k,m)*gw(lats(i))
              wmax = max(wmax,state(c)%q(i,k,m))
              wmin = min(wmin,state(c)%q(i,k,m))
           end do ! k
           if ( cnst_need_pdeldry)  col = col/gravit
           colw = colw/gravit
           if ( cnst_need_pdeldry) twodfld(lons(i),lats(i),m) = twodfld(lons(i),lats(i),m) + col
           twodfle(lons(i),lats(i),m) = twodfle(lons(i),lats(i),m) + colw
           twodflx(lons(i),lats(i),m) = twodflx(lons(i),lats(i),m) + wmin
           twodfly(lons(i),lats(i),m) = twodfly(lons(i),lats(i),m) + wmax
        enddo ! i
     enddo ! m
  end do ! c

  ! move data to masterproc
#ifdef SPMD
  call t_barrierf('sync_gavglook_data_mv', mpicom)
  if ( cnst_need_pdeldry ) call mpisum(twodfld, twodfld_glob, plon*plat*pcnst, mpir8, 0, mpicom)
  call mpisum(twodfle, twodfle_glob, plon*plat*pcnst, mpir8, 0, mpicom)
  call mpisum(twodflx, twodflx_glob, plon*plat*pcnst, mpir8, 0, mpicom)
  call mpisum(twodfly, twodfly_glob, plon*plat*pcnst, mpir8, 0, mpicom)
  if (masterproc) then
     if ( cnst_need_pdeldry ) twodfld(:,:,:) = twodfld_glob(:,:,:) 
     twodfle(:,:,:) = twodfle_glob(:,:,:) 
     twodflx(:,:,:) = twodflx_glob(:,:,:) 
     twodfly(:,:,:) = twodfly_glob(:,:,:) 
  endif
#endif

  ! process the data
  if (masterproc) then
     do m = 1,pcnst
        wmax = -1.e36_r8
        wmin = 1.e36_r8
        do lat=1,plat
           if ( cnst_need_pdeldry ) zonal(lat) = 0._r8
           zonalw(lat) = 0._r8
           do i=1,plon
              if ( cnst_need_pdeldry ) zonal(lat) = zonal(lat) + twodfld(i,lat,m)
              zonalw(lat) = zonalw(lat) + twodfle(i,lat,m)
              wmax = max(wmax,twodfly(i,lat,m))
              wmin = min(wmin,twodflx(i,lat,m))
           end do
        end do
        if ( cnst_need_pdeldry ) gavg = 0._r8
        gavgw = 0._r8
        do lat=1,plat
           if ( cnst_need_pdeldry ) gavg = gavg + zonal(lat)
           gavgw = gavgw + zonalw(lat)
        end do
        if ( cnst_need_pdeldry )  gavg = gavg/(2._r8*plon)
        gavgw = gavgw/(2._r8*plon)

        if ( .not. cnst_need_pdeldry ) then
             write (6,67) trim(title)//' m=',m,'name='//trim(cnst_name(m))//' gavg wet, min, max ' &
                  , gavgw,wmin,wmax
67           format (a24,i2,a36,1p,4g25.14)
          else
             write (6,66) trim(title)//' m=',m,'name='//trim(cnst_name(m))//' gavg dry, wet, min, max ' &
                  , gavg, gavgw,wmin,wmax
!66           format (a24,i2,a36,1p,4g25.14)
66           format (a24,i2,a36,1p,4e25.13)
     endif

     end do
  endif

end subroutine gavglook

end module physpkg
