
module vertical_diffusion

  !---------------------------------------------------------------------------------
  ! Module to compute vertical diffusion of momentum,  moisture, trace constituents
  ! and static energy. Separate modules compute 
  !   1. stresses associated with turbulent flow over orography
  !   2. eddy diffusivities, including nonlocal tranport terms
  !   3. molecular diffusivities
  !   4. coming soon... gravity wave drag
  ! Lastly, a implicit diffusion solver is called, and tendencies retrieved by 
  ! differencing the diffused and initial states.
  !
  ! calling sequence:
  !
  !  vertical_diffusion_init           initializes vertical diffustion constants and modules
  !        init_tms                      initializes turbulent mountain stress module
  !        init_hb_diff                  initializes eddy diffusivity module (includes PBL)
  !        init_molec_diff               initializes molecular diffusivity module 
  !        init_vdiff                    initializes diffusion solver module
  !  vertical_diffusion_ts_init        time step initialization (only used for upper boundary condition)
  !  vertical_diffusion_tend           computes vertical diffusion tendencies
  !        compute_tms                   computes turbulent mountain stresses
  !        compute_hb_diff               computes eddy diffusivities and countergradient terms
  !        compute_vdiff                 solves vertical diffusion equations (including molecular diffusivities)
  !
  !---------------------------Code history--------------------------------
  ! This is a reorganization of the original vertical diffusion module
  ! written by J. Rosinski in June 1992, and modified by many others.
  ! Initial coding for this version:  J. McCaa, September 2004.
  !---------------------------------------------------------------------------------

  use shr_kind_mod,     only: r8 => shr_kind_r8
  use ppgrid,           only: pcols, pver, pverp
  use constituents,     only: pcnst, qmin
  use diffusion_solver, only: vdiff_selector
  use abortutils,       only: endrun
  use physconst,        only: &
       cpair  , &     ! Specific heat of dry air
       gravit , &     ! Acceleration due to gravity
       rair   , &     ! Gas const for dry air
       zvir   , &     ! rh2o/rair - 1
       latvap , &     ! latent heat of vaporization
       latice , &     ! latent heat of fusion
       karman , &     ! von Karman constant
       mwdry  , &     ! molecular weight of dry air
       avogad , &     ! Avogadro's number
       boltz          ! Boltzman's constant
  use cam_history,      only : fieldname_len
  use perf_mod

  implicit none
  private          ! Make default type private to the module
  save
  
  !
  ! Public interfaces
  !
  public vd_register         ! Register multi-time-level variables with physics buffer
  public vertical_diffusion_init             ! Initialization
  public vertical_diffusion_ts_init          ! Time step initialization (only used for upper boundary condition)


  public vertical_diffusion_tend             ! Full routine
  ! 
  ! Private data
  !
  ! The following switches control the behavior of the vertical diffusion package
  !   eddy_scheme selects which eddy diffusivity scheme to apply
  !                      'HB'       = Holtslag and Boville (default)
  !                      'diag TKE' = diagnostic tke version of Grenier and Bretherton
  character(len=16) :: eddy_scheme  ! default set in phys_control.F90, use namelist to change
  character(len=16) :: shallow_scheme  !  check compatibility between eddy and shallow scheme
  integer, parameter :: nturb = 5         ! no. of iterations for solution (when using eddy_select=1)

  logical, parameter :: wstarent = .true. ! whether to use wstar entrainment (when using eddy_select=1)

  !   Local copies of physical constants
  !
  !  Other local data shared between routines
  logical :: do_molec_diff = .false.        ! switch for molecular diffusion
  logical :: do_tms        = .false.        ! switch for turbulent mountain stress
  type(vdiff_selector) :: fieldlist_wet     ! Logical switches for moist mixing ratio diffusion
  type(vdiff_selector) :: fieldlist_dry     ! Logical switches for dry mixing ratio diffusion
  integer :: ntop                           ! Top level to which vertical diffusion is applied (=1).
  integer :: nbot                           ! Bottom level to which vertical diffusion is applied (=pver).
  integer :: tke_idx,kvh_idx,kvm_idx        ! indices for fields in the physics buffer
  character(len=fieldname_len) :: vdiffnam(pcnst) ! names of v-diff tendencies
  integer :: ixcldice, ixcldliq             ! constituent indices for cloud liquid and ice water
  integer wgustd_index

  ! -------------------------------------------------------------- !
  ! Sungsu : Parameter for pseudo-conservative variables diffusion !
  ! -------------------------------------------------------------- !

  logical :: do_pseudocon_diff = .true. 

contains

  !===============================================================================
  subroutine vd_register()
    !-----------------------------------------------------------------------
    ! Register physics buffer fields and constituents
    !-----------------------------------------------------------------------
    use phys_buffer, only: pbuf_times, pbuf_add
    use phys_control, only: phys_getopts

    ! get eddy_scheme setting from phys_control
    call phys_getopts( eddy_scheme_out = eddy_scheme , shallow_scheme_out = shallow_scheme )

    ! Request physics buffer space for fields that persist across timesteps.
    call pbuf_add('tke', 'global', 1,pverp,pbuf_times, tke_idx) 
    call pbuf_add('kvh', 'global', 1,pverp,pbuf_times, kvh_idx) 
    call pbuf_add('kvm', 'global', 1,pverp,pbuf_times, kvm_idx) 

    ! Add wpert to the physics buffer
    ! It should serve as a prototype for tpert, qpert, and pblht
    call pbuf_add('wgustd','global',1,1,1,wgustd_index)



  end subroutine vd_register
  
!=========================================================================================
  

  subroutine vertical_diffusion_init()
    !-----------------------------------------------------------------------
    ! Initialization of time independent fields for vertical diffusion.
    ! Calls initialization routines for subsidiary modules
    !-----------------------------------------------------------------------
    use cam_history,      only: addfld, add_default, phys_decomp
    use eddy_diff,        only: init_eddy_diff
    use hb_diff,          only: init_hb_diff
    use molec_diff,       only: init_molec_diff
    use trb_mtn_stress,   only: init_tms
    use diffusion_solver, only: init_vdiff, vdiff_select
    use constituents,     only: cnst_get_ind, cnst_get_type_byind, cnst_name
    use spmd_utils,       only: masterproc
    
    ! include hypm (reference pressures at midpoints) 
    use hycoef, only : hypm

    !------------------------------Arguments--------------------------------
    ! none
    !---------------------------Local workspace-----------------------------
    integer :: ntop_eddy   ! Top level to which eddy vertical diffusion is applied.
    integer :: nbot_eddy   ! Bottom level to which eddy vertical diffusion is applied (=pver).
    integer :: ntop_molec  ! Top level to which molecular vertical diffusion is applied (=1).
    integer :: nbot_molec  ! Bottom level to which molecular vertical diffusion is applied.
    integer :: k           ! vertical loop index
    character(128) :: errstring                    ! error status for init_vdiff

    !-----------------------------------------------------------------------
    ! Get indices of cloud liquid and ice within the constituents array
    call cnst_get_ind('CLDLIQ', ixcldliq)
    call cnst_get_ind('CLDICE', ixcldice)

    ! Initialize molecular diffusivity module
    ! Molecular diffusion turned on above ~60 km (50 Pa) if model top is above ~90 km (.1 Pa).
    ! Note that computing molecular diffusivities is a trivial expense, but constituent
    ! diffusivities depend on their molecular weights. Decomposing the diffusion matric
    ! for each constituent is a needless expense unless the diffusivity is significant.
    ntop_molec = 1       ! should always be 1
    nbot_molec = 0       ! should be set below about 70 km
    if (hypm(1) .lt. 0.1_r8) then
       do_molec_diff = .true.
       do k = 1, pver
          if (hypm(k) .lt. 50._r8) nbot_molec  = k
       end do
       call init_molec_diff(r8, pcnst, rair, ntop_molec, nbot_molec, mwdry, &
            avogad, gravit, cpair, boltz)
       call addfld      ('TTPXMLC', 'K/S', 1, 'A','Top interf. temp. flux: molec. viscosity', phys_decomp)
       call add_default ('TTPXMLC', 1, ' ')
       if (masterproc) write (6,fmt='(a,i3,5x,a,i3)') 'NTOP_MOLEC =',ntop_molec, 'NBOT_MOLEC =',nbot_molec
    end if
    
    ! Initialize eddy diffusivity module
    ntop_eddy  = 1       ! no reason not to make this 1, if >1, must be <= nbot_molec
    nbot_eddy  = pver    ! should always be pver
    if(masterproc)write (6,fmt='(a,i3,5x,a,i3)') 'NTOP_EDDY  =',ntop_eddy,  'NBOT_EDDY  =',nbot_eddy

    select case (eddy_scheme)
    case ( 'diag TKE' ) 
       if (masterproc) &
	write(6,*)'vertical_diffusion_init: eddy_diffusivity scheme: diagnostic tke version of Grenier and Bretherton '

        ! Check compatibility of eddy and shallow scheme
        if ( shallow_scheme .ne. 'UW' ) then
           write(6,*)'ERROR: shallow convection scheme ',shallow_scheme,' is incompatible with eddy scheme ',eddy_scheme
           call endrun('convect_shallow_init: shallow_scheme and eddy_scheme are incompatible')
        endif

       call init_eddy_diff(r8, pver, gravit, cpair, rair, zvir, latvap, latice, &
            ntop_eddy, nbot_eddy,  hypm, karman )
       if(masterproc) write(6,*) 'vertical_diffusion: nturb, ntop_eddy, nbot_eddy ',nturb,ntop_eddy,nbot_eddy
    case ( 'HB' )
       if(masterproc) write(6,*)'vertical_diffusion_init: eddy_diffusivity scheme:  Holtslag and Boville'
       call init_hb_diff(gravit, cpair, rair, zvir, ntop_eddy, nbot_eddy, hypm, karman)
    end select
    
    ! The vertical diffusion solver must operate over the full range of molecular and eddy diffusion
    ntop = min(ntop_molec,ntop_eddy)
    nbot = max(nbot_molec,nbot_eddy)
    
    ! Initialize turbulent mountain stress module
    if ( do_tms ) then
       call init_tms(r8, 4.0_r8, karman, gravit, rair)
       call addfld ('TAUTMSX' ,'N/m2    ',1,    'A','Zonal turbulent mountain surface stress'  ,  phys_decomp)
       call addfld ('TAUTMSY' ,'N/m2    ',1,    'A','Meridional turbulent mountain surface stress', phys_decomp)
       call add_default ('TAUTMSX ', 1, ' ')
       call add_default ('TAUTMSY ', 1, ' ')
       if(masterproc)write(6,*)'Using turbulent mountain stress module'
    endif
    
    ! Initialize diffusion solver module
    call init_vdiff(r8, pcnst, rair, gravit, fieldlist_wet, fieldlist_dry, errstring)
    if (errstring.ne.'')call endrun(errstring)
    ! Use fieldlist_wet to select the fields which will be diffused using moist mixing ratios
    ! (all by default).  Use fieldlist_dry to select the fields which will be diffused using
    ! dry mixing ratios.
    if(vdiff_select(fieldlist_wet,'u').ne.'') call endrun(vdiff_select(fieldlist_wet,'u'))
    if(vdiff_select(fieldlist_wet,'v').ne.'') call endrun(vdiff_select(fieldlist_wet,'v'))
    if(vdiff_select(fieldlist_wet,'s').ne.'') call endrun(vdiff_select(fieldlist_wet,'s'))
    do k = 1, pcnst
       if (cnst_get_type_byind(k).eq.'wet') then
          if (vdiff_select(fieldlist_wet,'q',k).ne.'') call endrun(vdiff_select(fieldlist_wet,'q',k))
       else
          if (vdiff_select(fieldlist_dry,'q',k).ne.'') call endrun(vdiff_select(fieldlist_dry,'q',k))
       end if
    end do
    
    ! Diagnostic output fields
    do k = 1, pcnst
       vdiffnam(k) = 'VD'//cnst_name(k)
       if(k==1)vdiffnam(k) = 'VD01'    !**** compatibility with old code ****
       call addfld (vdiffnam(k),'kg/kg/s ',pver, 'A','Vertical diffusion of '//cnst_name(k),phys_decomp)
    end do
    call addfld('TKE'  , 'm2/s2', pverp, 'A', 'Turbulent Kinetic Energy'                          , phys_decomp)
    call addfld('PBLH' , 'm'    , 1    , 'A', 'PBL height'                                        , phys_decomp)
    call addfld('TPERT', 'K'    , 1    , 'A', 'Perturbation temperature (eddies in PBL)'          , phys_decomp)
    call addfld('QPERT', 'kg/kg', 1    , 'A', 'Perturbation specific humidity (eddies in PBL)'    , phys_decomp)
    call addfld('USTAR', 'm/s'  , 1    , 'A', 'Surface friction velocity'                         , phys_decomp)
    call addfld('KVH'  , 'm2/s' , pverp, 'A', 'Vertical diffusion diffusivities (heat/moisture)'  , phys_decomp)
    call addfld('KVM'  , 'm2/s' , pverp, 'A', 'Vertical diffusion diffusivities (momentum)'       , phys_decomp)
    call addfld('CGS'  , 's/m2' , pverp, 'A', 'Counter-gradient coeff on surface kinematic fluxes', phys_decomp)
    call addfld('DTVKE', 'K/s'  , pver , 'A', 'dT/dt vertical diffusion KE dissipation'           , phys_decomp)
    call addfld('DTV'  , 'K/s'  , pver , 'A', 'T vertical diffusion'                              , phys_decomp)
    call addfld('DUV'  , 'm/s2' , pver , 'A', 'U vertical diffusion'                              , phys_decomp)
    call addfld('DVV'  , 'm/s2' , pver , 'A', 'V vertical diffusion'                              , phys_decomp)
    call addfld('QT      ', 'kg/kg', pver, 'A', 'Total water mixing ratio'  ,  phys_decomp)
    call addfld('SL      ', 'J/kg', pver, 'A', 'Liquid water static energy'  ,  phys_decomp)
    call addfld('SLV     ', 'J/kg', pver, 'A', 'Liq wat virtual static energy'  ,  phys_decomp)
    call addfld('SLFLX   ', 'W/m2', pverp, 'A', 'Liquid static energy flux'  ,  phys_decomp) 
    call addfld('QTFLX   ', 'W/m2', pverp, 'A', 'Total water flux'  ,  phys_decomp) 
    call addfld('UFLX    ', 'W/m2', pverp, 'A', 'Zonal momentum flux'  ,  phys_decomp) 
    call addfld('VFLX    ', 'W/m2', pverp, 'A', 'Meridional momentm flux'  ,  phys_decomp) 
    call addfld('WGUSTD  ', 'm/s' ,1 , 'A','wind gusts from turbulence',phys_decomp)

    ! ------------------------------------------------------ !
    ! Below is for detailed analysis of PBL scheme by Sungsu !
    ! ------------------------------------------------------ !

    call addfld('qt_pre_PBL  ', 'kg/kg', pver, 'I', 'qt_prePBL'  ,  phys_decomp)
    call addfld('sl_pre_PBL  ', 'J/kg', pver, 'I', 'sl_prePBL'  ,  phys_decomp)
    call addfld('slv_pre_PBL ', 'J/kg', pver, 'I', 'slv_prePBL'  ,  phys_decomp)
    call addfld('u_pre_PBL   ', 'm/s', pver, 'I', 'u_prePBL'  ,  phys_decomp)
    call addfld('v_pre_PBL   ', 'm/s', pver, 'I', 'v_prePBL'  ,  phys_decomp)
    call addfld('qv_pre_PBL  ', 'kg/kg', pver, 'I', 'qv_prePBL'  ,  phys_decomp)
    call addfld('ql_pre_PBL  ', 'kg/kg', pver, 'I', 'ql_prePBL'  ,  phys_decomp)
    call addfld('qi_pre_PBL  ', 'kg/kg', pver, 'I', 'qi_prePBL'  ,  phys_decomp)
    call addfld('t_pre_PBL   ', 'K', pver, 'I', 't_prePBL'  ,  phys_decomp)
    call addfld('rh_pre_PBL  ', '%', pver, 'I', 'rh_prePBL'  ,  phys_decomp)

    call addfld('qt_aft_PBL  ', 'kg/kg', pver, 'I', 'qt_afterPBL'  ,  phys_decomp)
    call addfld('sl_aft_PBL  ', 'J/kg', pver, 'I', 'sl_afterPBL'  ,  phys_decomp)
    call addfld('slv_aft_PBL ', 'J/kg', pver, 'I', 'slv_afterPBL'  ,  phys_decomp)
    call addfld('u_aft_PBL   ', 'm/s', pver, 'I', 'u_afterPBL'  ,  phys_decomp)
    call addfld('v_aft_PBL   ', 'm/s', pver, 'I', 'v_afterPBL'  ,  phys_decomp)
    call addfld('qv_aft_PBL  ', 'kg/kg', pver, 'I', 'qv_afterPBL'  ,  phys_decomp)
    call addfld('ql_aft_PBL  ', 'kg/kg', pver, 'I', 'ql_afterPBL'  ,  phys_decomp)
    call addfld('qi_aft_PBL  ', 'kg/kg', pver, 'I', 'qi_afterPBL'  ,  phys_decomp)
    call addfld('t_aft_PBL   ', 'K', pver, 'I', 't_afterPBL'  ,  phys_decomp)
    call addfld('rh_aft_PBL  ', '%', pver, 'I', 'rh_afterPBL'  ,  phys_decomp)

    call addfld('slflx_PBL   ', 'J/m2/s', pverp, 'I', 'sl flux by PBL'  ,  phys_decomp) 
    call addfld('qtflx_PBL   ', 'kg/m2/s', pverp, 'I', 'qt flux by PBL'  ,  phys_decomp) 
    call addfld('uflx_PBL    ', 'kg/m/s2', pverp, 'I', 'u flux by PBL'  ,  phys_decomp) 
    call addfld('vflx_PBL    ', 'kg/m/s2', pverp, 'I', 'v flux by PBL'  ,  phys_decomp) 

    call addfld('qtten_PBL   ','kg/kg/s',pver,'I','qt tendency by PBL',phys_decomp)
    call addfld('slten_PBL   ','J/kg/s',pver,'I','sl tendency by PBL',phys_decomp)
    call addfld('uten_PBL    ','m/s2',pver,'I','u tendency by PBL',phys_decomp)
    call addfld('vten_PBL    ','m/s2',pver,'I','v tendency by PBL',phys_decomp)
    call addfld('qvten_PBL   ','kg/kg/s',pver,'I','qv tendency by PBL',phys_decomp)
    call addfld('qlten_PBL   ','kg/kg/s',pver,'I','ql tendency by PBL',phys_decomp)
    call addfld('qiten_PBL   ','kg/kg/s',pver,'I','qi tendency by PBL',phys_decomp)
    call addfld('tten_PBL    ','K/s',pver,'I','T tendency by PBL',phys_decomp)
    call addfld('rhten_PBL   ','%/s',pver,'I','RH tendency by PBL',phys_decomp)

    ! ------------- !
    ! End of Sungsu !
    ! ------------- !

    call add_default(vdiffnam(1), 1, ' ')
    call add_default('DTV'      , 1, ' ')
    call add_default('PBLH'     , 1, ' ')
 
    if ( eddy_scheme .eq. 'diag TKE' ) then    
       call addfld ('BPROD   ','M2/S3   ',pverp, 'A','Buoyancy Production',phys_decomp)
       call addfld ('SPROD   ','M2/S3   ',pverp, 'A','Shear Production',phys_decomp)
       call addfld ('SFI     ','FRACTION',pverp, 'A','Interface-layer sat frac',phys_decomp)       
       call add_default ('BPROD   ', 1, ' ')
       call add_default ('SPROD   ', 1, ' ')
       call add_default ('SFI     ', 1, ' ')

       call add_default ('TKE     ', 1, ' ')
       call add_default ('KVH     ', 1, ' ')
       call add_default ('KVM     ', 1, ' ')
       call add_default ('WGUSTD', 1, ' ')
       call add_default('QT      ', 1, ' ')
       call add_default('SL      ', 1, ' ')
       call add_default('SLV     ', 1, ' ')
       call add_default('SLFLX   ', 1, ' ')
       call add_default('QTFLX   ', 1, ' ')
       call add_default('UFLX    ', 1, ' ')
       call add_default('VFLX    ', 1, ' ')
    endif

#if ( defined WACCM_GHG || defined WACCM_MOZART )
    call add_default ('DUV'     , 1, ' ')
    call add_default ('DVV'     , 1, ' ')
#endif
    
  end subroutine vertical_diffusion_init
  
  !===============================================================================
  subroutine vertical_diffusion_ts_init
    !-----------------------------------------------------------------------
    ! timestep dependent setting, 
    ! at present only invokes upper bc code for molecular diffusion
    !-----------------------------------------------------------------------
    use molec_diff, only: init_timestep_molec_diff

    if (do_molec_diff) call init_timestep_molec_diff

  end subroutine vertical_diffusion_ts_init

  !===============================================================================
  subroutine vertical_diffusion_tend(                                     &
       ztodt    ,state    ,                               &
       taux     ,tauy     ,shflx    ,cflx     ,pblh     , &
       tpert    ,qpert    ,ustar    ,obklen   ,ptend    , &
       cldn     ,ocnfrac  ,landfrac ,sgh      ,pbuf       ) 
    !-----------------------------------------------------------------------
    ! interface routine for vertical diffusion
    !-----------------------------------------------------------------------
    use physics_types,  only: physics_state, physics_ptend
    use cam_history,    only: outfld
    use phys_buffer,    only: pbuf_size_max, pbuf_fld,pbuf_old_tim_idx,pbuf_times, pbuf_get_fld_idx
    use time_manager,   only: is_first_step
    use geopotential,   only: geopotential_dse
    ! The commented 'only' limiter from the following line acommodates broken pgf90 v.5.1.6
    use diffusion_solver ! , only: compute_vdiff, any, operator(.not.)
    use trb_mtn_stress, only: compute_tms
    use eddy_diff,   only: compute_eddy_diff
    use hb_diff, only: compute_hb_diff
    use wv_saturation, only: fqsatd, aqsat
    use molec_diff, only: compute_molec_diff , vd_lu_qdecomp
    use constituents, only: qmincg, qmin
    use infnan


    !------------------------------Arguments--------------------------------
    real(r8), intent(in) :: taux(pcols)            ! x surface stress (N/m2)
    real(r8), intent(in) :: tauy(pcols)            ! y surface stress (N/m2)
    real(r8), intent(in) :: shflx(pcols)           ! surface sensible heat flux (w/m2)
    real(r8), intent(in) :: cflx(pcols,pcnst)      ! surface constituent flux (kg/m2/s)
    real(r8), intent(in) :: ztodt                  ! 2 delta-t
    real(r8), intent(in) :: cldn(pcols,pver)       ! new cloud fraction
    real(r8), intent(in) :: ocnfrac(pcols)         ! Ocean fraction
    real(r8), intent(in) :: landfrac(pcols)        ! Land fraction
    real(r8), intent(in) :: sgh(pcols)             ! standard deviation of orography
    type(physics_state), intent(in)  :: state      ! Physics state variables
    
    type(physics_ptend), intent(inout)                      :: ptend ! indivdual parameterization tendencies
    type(pbuf_fld), intent(inout), dimension(pbuf_size_max) :: pbuf  ! physics buffer
    
    real(r8), intent(out) :: pblh(pcols)              ! planetary boundary layer height
    real(r8), intent(out) :: tpert(pcols)             ! convective temperature excess
    real(r8), intent(out) :: qpert(pcols)             ! convective humidity excess
    real(r8), intent(out) :: ustar(pcols)             ! surface friction velocity
    real(r8), intent(out) :: obklen(pcols)            ! Obukhov length
    !
    !---------------------------Local storage-------------------------------
    integer :: lchnk                               ! chunk identifier
    integer :: ncol                                ! number of atmospheric columns
    integer :: i,k,m                               ! longitude,level,constituent indices

    real(r8) :: dtk(pcols,pver)                    ! T tendency from KE dissipation
    real(r8) :: tke(pcols,pverp)                   ! turbulent kinetic energy
    real(r8) :: cgs(pcols,pverp)                   ! counter-gradient star (cg/flux)
    real(r8) :: cgh(pcols,pverp)                   ! counter-gradient term for heat
    real(r8) :: rztodt                             ! 1./ztodt
    real(r8) :: ksrftms(pcols)                     ! turbulent mountain stress surface drag coefficient
    real(r8) :: tautmsx(pcols)                     ! u component of turbulent mountain stress
    real(r8) :: tautmsy(pcols)                     ! v component of turbulent mountain stress
    real(r8) :: tautotx(pcols)                     ! u component of total surface stress
    real(r8) :: tautoty(pcols)                     ! v component of total surface stress

    integer :: time_index                          ! time level index for physics buffer access
    real(r8) :: kvh(pcols,pverp)                   ! eddy diffusivity for heat [m2/s]
    real(r8) :: kvm(pcols,pverp)                   ! eddy diffusivity for momentum [m2/s]
    real(r8) :: kvh_in(pcols,pverp)                ! kvh from previous timestep
    real(r8) :: kvm_in(pcols,pverp)                ! kvm from previous timestep
    real(r8) :: bprod(pcols,pverp)                 ! buoyancy production of tke
    real(r8) :: sprod(pcols,pverp)                 ! shear production of tke
    real(r8) :: sfi(pcols,pverp)                   ! saturation fraction at interfaces
    real(r8), dimension(pcols,pver) :: sl,qt,slv
    real(r8), dimension(pcols,pver) :: sl_prePBL,qt_prePBL,slv_prePBL,slten,qtten,slvten
    real(r8), dimension(pcols,pverp) :: slflx,qtflx,uflx,vflx
    real(r8) :: rhoair
    real(r8), pointer, dimension(:,:) :: qrl
    real(r8) :: kvq(pcols,pverp)                   ! diffusivity for constituents
    real(r8) :: th(pcols,pver)                     ! Potential temperature
    real(r8) :: topflx(pcols)                      ! molecular heat flux at top interface
    character(128) :: errstring                    ! error status for compute_vdiff
    logical :: kvinit                              ! tell compute_eddy_diff/ caleddy to initialize kvh, kvm (uses kvf)
    
    real(r8) wpert(pcols)                          ! turbulent wind gusts

    real(r8) ftem(pcols,pver)        ! saturation vapor pressure before PBL
    real(r8) ftem_prePBL(pcols,pver) ! saturation vapor pressure before PBL
    real(r8) ftem_aftPBL(pcols,pver) ! saturation vapor pressure after PBL
    real(r8) tem2(pcols,pver)        ! saturation specific humidity and RH
    real(r8) t_aftPBL(pcols,pver)    ! temperature after PBL diffusion
    real(r8) tten(pcols,pver)        ! temperature tendency by PBL diffusion
    real(r8) rhten(pcols,pver)       ! RH tendency by PBL diffusion
    real(r8) qv_aft_PBL(pcols,pver)  ! qv after PBL diffusion
    real(r8) ql_aft_PBL(pcols,pver)  ! ql after PBL diffusion
    real(r8) qi_aft_PBL(pcols,pver)  ! qi after PBL diffusion
    real(r8) s_aft_PBL(pcols,pver)   ! s after PBL diffusion

    ! Sungsu : Parameters for 'qt diffusion' in UW_PBL scheme
    real(r8) :: qv_pro, ql_pro, qi_pro
    real(r8) :: qv_evaptend, ql_evaptend, qi_evaptend, s_evaptend
    ! End by Sungsu of 'qt_diff'

    !-----------------------------------------------------------------------
    rztodt = 1._r8/ztodt
    lchnk = state%lchnk
    ncol  = state%ncol

    ! All variables are modified by vertical diffusion

    ptend%name  = "vertical diffusion"
    ptend%lq(:) = .TRUE.
    ptend%ls    = .TRUE.
    ptend%lu    = .TRUE.
    ptend%lv    = .TRUE.

    !-----------------------------------------------------------------------
    !    Computation of turbulent mountain stress
    !-----------------------------------------------------------------------

    if ( do_tms ) then
       ! compute the turbulent mountain stress
       call compute_tms(  pcols   , pver       , ncol        , &
            state%u     , state%v  , state%t , state%pmid , state%exner , &
            state%zm    , sgh      , ksrftms, tautmsx, tautmsy, landfrac)
       tautotx(:ncol) = taux(:ncol) + tautmsx(:ncol)
       tautoty(:ncol) = tauy(:ncol) + tautmsy(:ncol)
    else
       ksrftms(:ncol) = 0.0_r8
       tautotx(:ncol) = taux(:ncol)
       tautoty(:ncol) = tauy(:ncol)
    endif

    !----------------------------------------------------------------------- !
    !   Computation of eddy diffusivities - Select appropriate PBL scheme    !
    !----------------------------------------------------------------------- !

    select case (eddy_scheme)
    case ( 'diag TKE' ) 

       ! ---------------------------------------------------------------- !
       ! On first time step, have eddy_diff.F90:caleddy() use kvh=kvm=kvf !
       ! This has to be done in compute_eddy_diff after kvf is calculated !
       ! ---------------------------------------------------------------- !

       if (is_first_step()) then
          kvinit = .true.
       else
          kvinit = .false.
       endif

       ! ---------------------------------------------------- !
       ! Get longwave radiative heating out of physics buffer !
       ! ---------------------------------------------------- !

       qrl => pbuf(pbuf_get_fld_idx('QRL'))%fld_ptr(1,1:pcols,1:pver,lchnk,1)
       
       ! Retrieve eddy diffusivities for heat and momentum from physics buffer
       ! from last timestep  (if first timestep, has been initialized by inidat.F90)
       time_index = pbuf_old_tim_idx()
       kvm_in(:ncol,:) = pbuf(kvm_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,time_index)
       kvh_in(:ncol,:) = pbuf(kvh_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,time_index)

       call compute_eddy_diff(       lchnk      ,                                         &
            pcols    , pver        , ncol       , state%t    , state%q(:,:,1) , ztodt   , &
            state%q(:,:,ixcldliq)  , state%q(:,:,ixcldice)   ,                            &
            state%s  , state%rpdel , cldn       , qrl        ,                            &
            state%zm , state%zi    , state%pmid , state%pint , state%u        , state%v , &
            tautotx  , tautoty     , shflx      , cflx(:,1)  , wstarent       , nturb   , &
            ustar    , pblh        , kvm_in     , kvh_in     , kvm            , kvh     , &
            kvq      , cgh         ,                                                      &
            cgs      , tpert       , qpert      , wpert      , tke            , bprod   , &
            sprod    , sfi         , fqsatd     , kvinit )

       obklen(:ncol) = nan ! not provided by compute_eddy_diff

       ! ----------------------------------------------- !       
       ! Store tke in pbuf for use by shallow convection !
       ! ----------------------------------------------- !   

       pbuf(tke_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,time_index)=tke(:ncol,:)

       ! Store updated kvh, kvm in pbuf to use here on the next timestep 
          pbuf(kvh_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,time_index)=kvh(:ncol,:)
          pbuf(kvm_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,time_index)=kvm(:ncol,:)
          if (is_first_step()) then
             do i = 1, pbuf_times
                pbuf(kvh_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,i) = kvh(:ncol,:)
                pbuf(kvm_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,i) = kvm(:ncol,:)
             enddo
          endif

       ! write out fields that are only used by this scheme
       call outfld('BPROD   ', bprod(1,1),pcols,lchnk)
       call outfld('SPROD   ', sprod(1,1),pcols,lchnk)
       call outfld ('SFI     ',sfi,  pcols,lchnk)

    case ( 'HB' )
       th(:ncol,:pver) = state%t(:ncol,:pver) * state%exner(:ncol,:pver)
       call compute_hb_diff(ncol      ,                   &
            th      ,state%t ,state%q ,state%zm,state%zi, &
            state%pmid,state%u,state%v,tautotx ,tautoty , &
            shflx   ,cflx    ,obklen  ,ustar   ,pblh    , &
            kvm     ,kvh     ,kvq     ,cgh     ,cgs     , &
            tpert   ,qpert   ,cldn    ,ocnfrac ,tke     )

       wpert = 0  ! placeholder, should get this from the hb scheme,
                  ! not used in this scheme
       ! save kvh in physics buffer, used by gw_intr from tphysac
       time_index = pbuf_old_tim_idx()
       pbuf(kvm_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,time_index) =  kvm(:ncol,:)
       pbuf(kvh_idx)%fld_ptr(1,1:ncol,1:pverp,lchnk,time_index) =  kvh(:ncol,:)
    end select

    pbuf(wgustd_index)%fld_ptr(1,1:ncol,1,lchnk,1) = wpert(:ncol)
    call outfld('WGUSTD    ',wpert         ,pcols   ,lchnk   )

    !------------------------------------ ! 
    !    Application of diffusivities     !
    !------------------------------------ !

    ptend%q(:ncol,:,:) = state%q(:ncol,:,:)
    ptend%s(:ncol,:)   = state%s(:ncol,:)
    ptend%u(:ncol,:)   = state%u(:ncol,:)
    ptend%v(:ncol,:)   = state%v(:ncol,:)

    !------------------------------------------------------ !
    ! Write profile output before applying diffusion scheme !
    !------------------------------------------------------ !

    sl_prePBL(:ncol,:pver)  = ptend%s(:ncol,:pver) -   latvap           * ptend%q(:ncol,:pver,ixcldliq) &
                                                   - ( latvap + latice) * ptend%q(:ncol,:pver,ixcldice)
    qt_prePBL(:ncol,:pver)  = ptend%q(:ncol,:pver,1) + ptend%q(:ncol,:pver,ixcldliq) &
                                                     + ptend%q(:ncol,:pver,ixcldice)
    slv_prePBL(:ncol,:pver) = sl_prePBL(:ncol,:pver)*(1.+zvir*qt_prePBL(:ncol,:pver)) 

    call aqsat( state%t, state%pmid, tem2, ftem, pcols, ncol, pver, 1, pver )
    ftem_prePBL(:ncol,:) = state%q(:ncol,:,1)/ftem(:ncol,:)*100.

    call outfld('qt_pre_PBL   ', qt_prePBL,  pcols, lchnk)
    call outfld('sl_pre_PBL   ', sl_prePBL,  pcols, lchnk)
    call outfld('slv_pre_PBL  ', slv_prePBL, pcols, lchnk)
    call outfld('u_pre_PBL    ', state%u, pcols, lchnk)
    call outfld('v_pre_PBL    ', state%v, pcols, lchnk)
    call outfld('qv_pre_PBL   ', state%q(:ncol,:,1), pcols, lchnk)
    call outfld('ql_pre_PBL   ', state%q(:ncol,:,2), pcols, lchnk)
    call outfld('qi_pre_PBL   ', state%q(:ncol,:,3), pcols, lchnk)
    call outfld('t_pre_PBL    ', state%t, pcols, lchnk)
    call outfld('rh_pre_PBL   ', ftem_prePBL, pcols, lchnk)

    ! --------------------------------------------------------------------------------- !
    ! Call the diffusivity solver and solve diffusion equation                          !
    ! The final two arguments are optional function references to                       !
    ! constituent-independent and constituent-dependent moleculuar diffusivity routines !
    ! Their use allows the diffusion_solver module to be independent of CAM, and to be  !
    ! used, for instance, by the Grenier-Bretherton PBL module.                         !
    ! --------------------------------------------------------------------------------- !

    if (any(fieldlist_wet)) call compute_vdiff( state%lchnk , &
         pcols         , pver               , pcnst , ncol      , state%pmid , &
         state%pint    , state%rpdel        , state%t     , ztodt     , taux       , &
         tauy          , shflx              , cflx        , ntop      , nbot       , &
         kvh           , kvm                , kvq         , cgs       , cgh        , &
         state%zi      , ksrftms            , qmincg      , fieldlist_wet , &
         ptend%u       , ptend%v            , ptend%q     , ptend%s   , &
         tautmsx       , tautmsy            , dtk         , topflx    , errstring  , &
         do_molec_diff , compute_molec_diff , vd_lu_qdecomp )
    if (errstring.ne.'')call endrun(errstring)
 
    if (any(fieldlist_dry)) then
       if (do_molec_diff) then
          errstring = "Design flaw: dry vdiff not currently supported with molecular diffusion"
          call endrun(errstring)
       end if
       call compute_vdiff(  state%lchnk        , &
            pcols         , pver               , pcnst       , ncol       , state%pmiddry , &
            state%pintdry , state%rpdeldry     , state%t     , ztodt      , taux          , &       
            tauy          , shflx              , cflx        , ntop       , nbot          , &       
            kvh           , kvm                , kvq         , cgs        , cgh           , &   
            state%zi      , ksrftms            , qmincg      , fieldlist_dry , &
            ptend%u       , ptend%v            , ptend%q     , ptend%s    , &
            tautmsx       , tautmsy            , dtk         , topflx     , errstring     , &
            do_molec_diff , compute_molec_diff , vd_lu_qdecomp)
       if (errstring.ne.'')call endrun(errstring)
    end if
    
    ! -------------------------------------------------------- !
    ! Diagnostics and output writing after applying PBL scheme !  
    ! -------------------------------------------------------- !

    sl(:ncol,:pver)  = ptend%s(:ncol,:pver) -   latvap           * ptend%q(:ncol,:pver,ixcldliq) &
                                            - ( latvap + latice) * ptend%q(:ncol,:pver,ixcldice)
    qt(:ncol,:pver)  = ptend%q(:ncol,:pver,1) + ptend%q(:ncol,:pver,ixcldliq) &
                                              + ptend%q(:ncol,:pver,ixcldice)
    slv(:ncol,:pver) = sl(:ncol,:pver)*(1.+zvir*qt(:ncol,:pver)) 

    slflx(:ncol,1) = 0.
    qtflx(:ncol,1) = 0.
    uflx(:ncol,1)  = 0.
    vflx(:ncol,1)  = 0.
    do k = 2, pver
       do i = 1, ncol
          rhoair     = state%pint(i,k)/ (rair * ((0.5*(slv(i,k)+slv(i,k-1)) - gravit*state%zi(i,k))/cpair ) )
          slflx(i,k) = kvh(i,k) * &
               (- rhoair*(sl(i,k-1)-sl(i,k))/(state%zm(i,k-1)-state%zm(i,k)) &
               + sl(i,k)*cgh(i,k)/rair                     ) 
          qtflx(i,k) = kvh(i,k) * &
               (-rhoair*(qt(i,k-1)-qt(i,k))/(state%zm(i,k-1)-state%zm(i,k)) &
               + sl(i,k)*cgs(i,k)/rair                      )
          uflx(i,k)  = kvm(i,k)* &
               (-rhoair*(ptend%u(i,k-1)-ptend%u(i,k))/(state%zm(i,k-1)-state%zm(i,k)))
          vflx(i,k)  = kvm(i,k)* &
               (-rhoair*(ptend%v(i,k-1)-ptend%v(i,k))/(state%zm(i,k-1)-state%zm(i,k)))
       end do !i
    end do !k
    slflx(:ncol,pverp) = shflx(:ncol)
    qtflx(:ncol,pverp) = cflx(:ncol,1)
    ! Sep.14.2006. Sungsu corrected the signs of 'uflx' and 'vflx'.
    uflx(:ncol,pverp) = taux(:ncol)
    vflx(:ncol,pverp) = tauy(:ncol)

    ! --------------------------------------------------------------- !
    ! Convert the new profiles into vertical diffusion tendencies.    !
    ! Convert KE dissipative heat change into "temperature" tendency. !
    ! --------------------------------------------------------------- !

    ptend%s(:ncol,:)   = (ptend%s(:ncol,:)   - state%s(:ncol,:))   * rztodt
    ptend%u(:ncol,:)   = (ptend%u(:ncol,:)   - state%u(:ncol,:))   * rztodt
    ptend%v(:ncol,:)   = (ptend%v(:ncol,:)   - state%v(:ncol,:))   * rztodt
    ptend%q(:ncol,:pver,:) = (ptend%q(:ncol,:pver,:) - state%q(:ncol,:pver,:)) * rztodt
    slten(:ncol,:)     = (sl(:ncol,:) - sl_prePBL(:ncol,:)) * rztodt 
    qtten(:ncol,:)     = (qt(:ncol,:) - qt_prePBL(:ncol,:)) * rztodt     

    ! ----------------------------------------------------------- !
    ! In order to perform 'pseudo-conservative varible diffusion' !
    !                                                             !
    ! A.                                                          !
    !    Option.1. ( used in S026 global simulation )             !
    !      Re-set (1) 'qvten' by 'qtten', and 'qlten=qiten=0'     !
    !             (2) 'sten'  by 'slten', and                     !
    !             (3) 'qlten = qiten = 0'                         !
    !    Option.2. ( my best choice consistent with U.W. Cu )     !    
    !      Re-set (1) 'qv(ql,qi)ten' by 'qtten' in proportion     !
    !             (2) 'sten' from 'slten' and 'qv(ql,qi)ten'      !
    !                                                             !
    ! B. Check if there is any 'negative qv' existing in each     !
    !    layer. If then evaporate 'ql,qi' proportionally in the   !
    !    same layer to 'qv' and reduce dry static enegy 's'.      ! 
    !                                                             !
    ! ----------------------------------------------------------- !

    if ( eddy_scheme .eq. 'diag TKE' ) then    

    if ( do_pseudocon_diff ) then

         ! A. Option.1.

         ptend%q(:ncol,:pver,1) = qtten(:ncol,:pver)
         ptend%s(:ncol,:pver)   = slten(:ncol,:pver)
         ptend%q(:ncol,:pver,ixcldliq) = 0.         
         ptend%q(:ncol,:pver,ixcldice) = 0.         

         ! A. Option.2.

       ! ptend%q(:ncol,:pver,1) =  qtten(:ncol,:pver)*state%q(:ncol,:pver,1)/ &
       !                          (state%q(:ncol,:pver,1)+state%q(:ncol,:pver,2)+state%q(:ncol,:pver,3))
       ! ptend%q(:ncol,:pver,2) =  qtten(:ncol,:pver)*state%q(:ncol,:pver,2)/ &
       !                          (state%q(:ncol,:pver,1)+state%q(:ncol,:pver,2)+state%q(:ncol,:pver,3)) 
       ! ptend%q(:ncol,:pver,3) =  qtten(:ncol,:pver)*state%q(:ncol,:pver,3)/ &
       !                          (state%q(:ncol,:pver,1)+state%q(:ncol,:pver,2)+state%q(:ncol,:pver,3)) 
       ! ptend%s(:ncol,:pver)   =  slten(:ncol,:pver) + & 
       !                           latvap * ptend%q(:ncol,:pver,2) + &
       !                          (latvap + latice) * ptend%q(:ncol,:pver,3)

         ! B. Perform only for 'qv'

         do i = 1, ncol
            do k = 1, pver
               qv_pro = state%q(i,k,1) + ptend%q(i,k,1) * ztodt       
               ql_pro = state%q(i,k,ixcldliq) + ptend%q(i,k,ixcldliq) * ztodt
               qi_pro = state%q(i,k,ixcldice) + ptend%q(i,k,ixcldice) * ztodt              
               if ( qv_pro .lt. qmin(1) .and. (ql_pro+qi_pro).gt.1.e-16 ) then
                    qv_evaptend =   ( qmin(1) - qv_pro ) * rztodt
                    ql_evaptend = - qv_evaptend * ql_pro / ( ql_pro + qi_pro )
                    qi_evaptend = - qv_evaptend * qi_pro / ( ql_pro + qi_pro )
                    s_evaptend  =   latvap * ql_evaptend + ( latvap + latice ) * qi_evaptend
                    ptend%q(i,k,1) = ptend%q(i,k,1) + qv_evaptend
                    ptend%q(i,k,ixcldliq) = ptend%q(i,k,ixcldliq) + ql_evaptend
                    ptend%q(i,k,ixcldice) = ptend%q(i,k,ixcldice) + qi_evaptend
                    ptend%s(i,k) = ptend%s(i,k) + s_evaptend
                    write(6,*) ' Negative qv after UW PBL is reset to qvmin'
                    write(6,*) i, k, qv_pro, qv_pro + qv_evaptend * ztodt, qmin(1) 
                    write(6,*) qv_pro, ql_pro, qi_pro, qv_pro+ql_pro+qi_pro 
               end if
            end do
         end do

    end if

    end if

    ! ----------------------------------------------------- !
    ! Re-calculate diagnostic output variables after qtdiff !
    ! ----------------------------------------------------- !

    qv_aft_PBL(:ncol,:pver) = state%q(:ncol,:pver,1) + ptend%q(:ncol,:pver,1)*ztodt
    ql_aft_PBL(:ncol,:pver) = state%q(:ncol,:pver,ixcldliq) + ptend%q(:ncol,:pver,ixcldliq)*ztodt  
    qi_aft_PBL(:ncol,:pver) = state%q(:ncol,:pver,ixcldice) + ptend%q(:ncol,:pver,ixcldice)*ztodt  
    s_aft_PBL(:ncol,:pver)  = state%s(:ncol,:pver) + ptend%s(:ncol,:pver)*ztodt  
    t_aftPBL(:ncol,:pver) = (s_aft_PBL(:ncol,:pver) - gravit*state%zm(:ncol,:pver))/cpair 

    call aqsat( t_aftPBL, state%pmid, tem2, ftem, pcols, ncol, pver, 1, pver )
    ftem_aftPBL(:ncol,:pver) = qv_aft_PBL(:ncol,:pver)/ftem(:ncol,:pver)*100.

    tten(:ncol,:pver)      = (t_aftPBL(:ncol,:pver) - state%t(:ncol,:pver)) * rztodt     
    rhten(:ncol,:pver)     = (ftem_aftPBL(:ncol,:pver) - ftem_prePBL(:ncol,:pver)) * rztodt 

    ! -------------------------------------------------------------- !
    ! Writing state variables after PBL scheme for detailed analysis !
    ! -------------------------------------------------------------- !

    call outfld('sl_aft_PBL', sl, pcols, lchnk)
    call outfld('qt_aft_PBL', qt, pcols, lchnk)
    call outfld('slv_aft_PBL', slv, pcols, lchnk)
    call outfld('u_aft_PBL', ptend%u, pcols, lchnk)
    call outfld('v_aft_PBL', ptend%v, pcols, lchnk)
    call outfld('qv_aft_PBL', qv_aft_PBL, pcols, lchnk)
    call outfld('ql_aft_PBL', ql_aft_PBL, pcols, lchnk)
    call outfld('qi_aft_PBL', qi_aft_PBL, pcols, lchnk)
    call outfld('t_aft_PBL ', t_aftPBL, pcols, lchnk)
    call outfld('rh_aft_PBL', ftem_aftPBL, pcols, lchnk)

    ! -------------------------------------------------------------------- !
    ! Writing fluxes and tendencies after PBL scheme for detailed analysis !
    ! -------------------------------------------------------------------- !

    call outfld('slflx_PBL', slflx, pcols, lchnk)
    call outfld('qtflx_PBL', qtflx, pcols, lchnk)
    call outfld('uflx_PBL', uflx, pcols, lchnk)
    call outfld('vflx_PBL', vflx, pcols, lchnk)

    call outfld('slten_PBL', slten, pcols, lchnk)
    call outfld('qtten_PBL', qtten, pcols, lchnk)
    call outfld('uten_PBL', ptend%u(:ncol,:), pcols, lchnk)
    call outfld('vten_PBL', ptend%v(:ncol,:), pcols, lchnk)
    call outfld('qvten_PBL', ptend%q(:ncol,:,1), pcols, lchnk)
    call outfld('qlten_PBL', ptend%q(:ncol,:,2), pcols, lchnk)
    call outfld('qiten_PBL', ptend%q(:ncol,:,3), pcols, lchnk)
    call outfld('tten_PBL', tten, pcols, lchnk)
    call outfld('rhten_PBL', rhten, pcols, lchnk)

    ! ------------------------------------------- !
    ! Writing the other standard output variables !
    ! ------------------------------------------- !

    call outfld('QT      ', qt, pcols, lchnk)
    call outfld('SL      ', sl, pcols, lchnk)
    call outfld('SLV     ', slv, pcols,lchnk)
    call outfld('SLFLX   ', slflx, pcols, lchnk)
    call outfld('QTFLX   ', qtflx, pcols, lchnk)
    call outfld('UFLX    ', uflx, pcols, lchnk)
    call outfld('VFLX    ', vflx, pcols, lchnk)
    call outfld('TKE     ', tke,  pcols,lchnk)
    call outfld ('PBLH    ',pblh ,pcols,lchnk)
    call outfld ('TPERT   ',tpert,pcols,lchnk)
    call outfld ('QPERT   ',qpert,pcols,lchnk)
    call outfld ('USTAR   ',ustar,pcols,lchnk)
    call outfld ('KVH     ',kvh,pcols,lchnk)
    call outfld ('KVM     ',kvm,pcols,lchnk)
    call outfld ('CGS     ',cgs,pcols,lchnk)
    dtk(:ncol,:) = dtk(:ncol,:)/cpair                ! normalize heating for history
    call outfld ('DTVKE   ',dtk,pcols,lchnk)
    dtk(:ncol,:) = ptend%s(:ncol,:)/cpair            ! normalize heating for history using dtk
    call outfld ('DTV     ',dtk  ,pcols,lchnk)
    call outfld ('DUV     ',ptend%u,pcols,lchnk)
    call outfld ('DVV     ',ptend%v,pcols,lchnk)
    do m = 1, pcnst
       call outfld(vdiffnam(m),ptend%q(1,1,m),pcols,lchnk)
    end do
    if ( do_tms ) then
       call outfld ('TAUTMSX', tautmsx, pcols, lchnk)
       call outfld ('TAUTMSY', tautmsy, pcols, lchnk)
    end if
    if (do_molec_diff) then
       call outfld ('TTPXMLC',topflx,pcols,lchnk)
    end if

    return
  end subroutine vertical_diffusion_tend

end module vertical_diffusion
