;
;-----------------------------------------------------------------------
pro anim_event,event
;
; Event handler for animation window:
;
case event.action of
  "DONE": begin
    widget_control,event.top,/destroy
    return
  end
  else: print,'anim_event: Unknown action: ',event.action
endcase
return
end
;-----------------------------------------------------------------------
pro lons_event,event
widget_control,event.id,get_uvalue=widget
widget_control,event.top,get_uvalue=pinfo
info = *pinfo
lon = *info.plon
fields = *info.fields
field = fields[lon.ifield]
mag = 0 & if field.grid_type eq 'magnetic' then mag = 1
;
case widget of
  'CLOSE': widget_control,event.top,/destroy
  'SELECT_LON': begin
    lon.slon = event.value
    lon.sslt = fslt(lon.sslt,lon.ut,lon.slon,'getlt',mag=mag)
    widget_control,lon.wfsl_slt,set_value=lon.sslt
    if lon.zonalmean eq 1 then lon.zonalmean = 0 ; turn off zonal means
    data = *fields[lon.ifield].data
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'SELECT_SLT': begin
    lon.sslt = event.value
    lon.slon = fslt(lon.sslt,lon.ut,lon.slon,'getlon',mag=mag)
    if lon.zonalmean eq 1 then lon.zonalmean = 0 ; turn off zonal means
    lons = *fields[lon.ifield].lons
;
; If exact lon corresponding to slt is not found in lons array,
; set it to nearest lon within lons:
    if ixfind(lons,lon.slon,0.) lt 0 then begin
      lons = *fields[lon.ifield].lons
      dlon = lons[1]-lons[0]
      ilon = ixfind(lons,lon.slon,dlon)
      lon.slon = lons[ilon]
    endif else begin
;     print,'select_slt: found exact lon.slon=',lon.slon,' in lons'
    endelse
    widget_control,lon.wfsl_lon,set_value=lon.slon
    data = *fields[lon.ifield].data
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'COLOR': begin
;   xloadct,group=event.top,/modal
    clrtab = xcolor(info.colortab,event.top)
    info.colortab = clrtab
    pltlon,info
  end
  'LOG10': begin
    lon.log10 = 'density fields'
    data = *fields[lon.ifield].data
    deflondata,lon,data
;
; Reset min,max and contour interval:
    fminmax,*lon.data,fmin,fmax,lon.missing_value
    cmin=0. & cmax=0. & cint=0.
    *lon.clevels = mkclevels(fmin,fmax,cmin,cmax,cint)
    lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
    *info.plon = lon
    print,'Will plot log10 of density fields'
    pltlon,info
  end
  'LOG10_ALLFIELDS': begin
    lon.log10 = 'all fields'
    data = *fields[lon.ifield].data
    deflondata,lon,data
;
; Reset min,max and contour interval:
    fminmax,*lon.data,fmin,fmax,lon.missing_value
    cmin=0. & cmax=0. & cint=0.
    *lon.clevels = mkclevels(fmin,fmax,cmin,cmax,cint)
    lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
    *info.plon = lon
    print,'Will plot log10 of all fields'
    pltlon,info
  end
  'LINEAR': begin
    lon.log10 = 'none'
    data = *fields[lon.ifield].data
    deflondata,lon,data
;
; Reset min,max and contour interval:
    fminmax,*lon.data,fmin,fmax,lon.missing_value
    cmin=0. & cmax=0. & cint=0.
    *lon.clevels = mkclevels(fmin,fmax,cmin,cmax,cint)
    lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
    *info.plon = lon
    print,'Will plot linear fields (no log10)'
    pltlon,info
  end
  'ZONAL_MEAN_ON': begin
    print,'Plotting zonal mean..'
    lon.zonalmean = 1 ; deflondata and pltlon will check for this
    data = *fields[lon.ifield].data
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'ZONAL_MEAN_OFF': begin
    print,'Turning zonal mean off..'
    lon.zonalmean = 0 ; deflondata and pltlon will check for this
    data = *fields[lon.ifield].data
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'ANIM_TIME_FIXLON': begin
    if field.ntime eq 1 then begin
      print,'>>> Cannot animate in time: there is only one time on the file!'
      widget_control,event.id,sensitive=0
    endif else begin
      widget_control,event.id,/sensitive
      animate_lons,info,'time_fixlon'
    endelse
  end
  'ANIM_TIME_FIXSLT': begin
    if field.ntime eq 1 then begin
      print,'>>> Cannot animate in time: there is only one time on the file!'
      widget_control,event.id,sensitive=0
    endif else begin
      widget_control,event.id,/sensitive
      animate_lons,info,'time_fixslt'
    endelse
  end
  'ANIM_LON': begin
    animate_lons,info,'lon_fixut'
  end
  'SET_MINMAX': begin
    umin=lon.fmin & umax=lon.fmax
    setminmax,info,umin,umax
    lon.fmin=umin & lon.fmax=umax
    *info.plon = lon
    pltlon,info
  end
  'RESET_MINMAX': begin
    lon.fmin=0. & lon.fmax=0.
    print,'Lon slices: reset image min,max to full range.'
    *info.plon = lon
    pltlon,info
  end
  'SET_YAXIS': begin
; sets y-axis limits lon.levmin,max and updates *info.plon
    setaxis_vert,info,info.plon 
    lon = *info.plon
    data = *fields[lon.ifield].data
    lon.field = fields[lon.ifield]
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'RESET_YAXIS': begin
    lon.levmin = (*field.levs)[0] & lon.levmax = (*field.levs)[field.nlev-1]
    print,'Lon slices: reset y-axis (zp or ht) to full range.'
    data = *fields[lon.ifield].data
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'SET_XAXIS': begin
; sets x-axis limits lon.latmin,max and updates *info.plon
    setaxis_lat,info,info.plon 
    lon = *info.plon
    data = *fields[lon.ifield].data
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'RESET_XAXIS': begin
    lon.latmin = (*field.lats)[0] & lon.latmax = (*field.lats)[field.nlat-1]
    print,'Lon slices: reset x-axis (latitude) to full range.'
;   data = *fields[lon.ifield].data
    data = *field.data
    deflondata,lon,data
    *info.plon = lon
    pltlon,info
  end
  'ADD_YAX': begin
    lon.rhyaxis = 1 ; add extra right-hand y-axis
    if field.difftype ne '' then begin
      print,'>>> WARNING: this is a difference fields file: right-hand',$
            '    axis in height is not available...'
      lon.rhyaxis = 0
    endif else if mag eq 1 then begin
      print,'>>> WARNING: this is a magnetic gridded field: right-hand',$
            '    axis in height is not available...'
      lon.rhyaxis = 0
    endif else begin
      print,'Will add extra right-hand y-axis'
    endelse
    *info.plon = lon
    pltlon,info
  end
  'RM_YAX': begin
    lon.rhyaxis = 0 ; remove extra right-hand y-axis
    print,'Will remove extra right-hand y-axis'
    *info.plon = lon
    pltlon,info
  end
  'CUSTOM_CONTOUR': begin
    lon.setcontour = 'on'
    cmin=lon.fmin & cmax=lon.fmax & cint=lon.cint & clineclr = lon.clineclr
    custom_contour,info.tlb,cmin,cmax,cint,clineclr
    lon.clineclr = clineclr
    data = *fields[lon.ifield].data
    deflondata,lon,data
    data = *lon.data
    fminmax,data,fmin,fmax,lon.missing_value
    levels = mkclevels(fmin,fmax,cmin,cmax,cint)
    lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
    *lon.clevels = levels
    *info.plon = lon
    pltlon,info
  end
  'RESET_CONTOUR': begin
    lon.setcontour = 'off'
    cmin=0. & cmax=0. & cint=0.
    print,'Lon slices: reset contour min,max to full range.'
    data = *fields[lon.ifield].data
    deflondata,lon,data
    data = *lon.data
    fminmax,data,fmin,fmax,lon.missing_value
    levels = mkclevels(fmin,fmax,cmin,cmax,cint)
    lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
    *lon.clevels = levels
    *info.plon = lon
    pltlon,info
  end
;
; Save images in ps or bitmap file for export:
;
  'SAVE_PS': begin
    title='Select File for Postscript'
    psfile = dialog_pickfile(path=info.openpath,title=title,$
      group=event.top)
    if (psfile ne '') then begin
      widget_control,event.top,hourglass=1
      pson,filename=psfile,margin=1.0
      setclrtab,info.colortab,/ps
      pltlon,info,/ps
      psoff
      widget_control,event.top,hourglass=0
    endif
  end
  'SAVE_PNG': begin
    windowsave = !d.window
    window,xsize=lon.draw_xsize,ysize=lon.draw_ysize,/pixmap
    windo = !d.window
    wset,windo
    setclrtab,info.colortab
    pltlon,info,/png
    writeimage,info.openpath,event.top,'PNG'
    wset,windowsave
  end
  'SAVE_GIF': begin
    version = float(!version.release)
    if version le 5.3 then begin
      writeimage,info.openpath,event.top,'GIF'
    endif else begin 
      print,'>>> NEED IDL version 5.3 to make GIF'
    endelse
  end  
  'SAVE_BMP': writeimage,info.openpath,event.top,'BMP'
  'SAVE_JPEG': writeimage,info.openpath,event.top,'JPEG'
  'SAVE_TIFF': writeimage,info.openpath,event.top,'TIFF'
;
; Plot types
;
  'IMAGE_ONLY': begin
    if mag eq 1 then begin
      print,'>>> WARNING: cannot make image of magnetic field because of ',$
        'irregular latitude grid. Staying with plot type ',lon.plottype
    endif else begin
      lon.plottype = 'image_only'
      *info.plon = lon
      print,'Set plot type to ',lon.plottype
      pltlon,info
      widget_control,lon.minmaxmenu,sensitive=1   ; can set image min,max
      widget_control,lon.contourmenu,sensitive=0  ; not doing contours
    endelse
  end
  'IMAGE+CONTOURS': begin
    if mag eq 1 then begin
      print,'>>> WARNING: cannot make image of magnetic field because of ',$
        'irregular latitude grid. Staying with plot type ',lon.plottype
    endif else begin
      lon.plottype = 'image+contours'
      *info.plon = lon
      print,'Set plot type to ',lon.plottype
      pltlon,info
      widget_control,lon.minmaxmenu,sensitive=0   ; image minmax will be fmin,fmax
      widget_control,lon.contourmenu,sensitive=1  ; doing contours
    endelse
  end
  'MONOCHROME_CONTOURS': begin
    lon.plottype = 'monochrome_contours'
    *info.plon = lon
    print,'Set plot type to ',lon.plottype
    pltlon,info
    widget_control,lon.minmaxmenu,sensitive=0   ; no image
    widget_control,lon.contourmenu,sensitive=1  ; doing contours
  end
  'COLORFILL_CONTOURS': begin
    lon.plottype = 'colorfill_contours'
    *info.plon = lon
    print,'Set plot type to ',lon.plottype
    pltlon,info
    widget_control,lon.minmaxmenu,sensitive=0   ; no image
    widget_control,lon.contourmenu,sensitive=1  ; doing contours
  end
  else: print,'>>> lons_event: unknown widget ',widget
endcase
*pinfo = info
end
;-----------------------------------------------------------------------
pro field_lon_event,event
;
; User has selected a field. Read field from the file and update
; lon structure.
;
widget_control,event.id,get_uvalue=widget
widget_control,event.top,get_uvalue=pinfo
info = *pinfo
lon = *info.plon
lon.setcontour = 'off'
fields = *info.fields
gridtype_prev = fields[lon.ifield].grid_type
for i=0,info.nflds-1 do begin
  if (widget eq fields[i].name) then begin
    lon.ifield = i
    if ptr_valid(fields[i].data) then begin ; field has been read
      print,'Selected field ',fields[i].name
      fmin = min(*fields[i].data) & fmax = max(*fields[i].data)
      lon.field = fields[i]
    endif else begin
      widget_control,/hourglass
      varget,info,fields[i],ncdata
      field = fields[i]
      procfield,info,ncdata,field,info.z_hist
      fields[i] = field
      lon.field = field
      lon.levs = lon.field.levs
      lon.data = lon.field.data
      fmin = min(*fields[i].data) & fmax = max(*fields[i].data)
      rpt_minmax,info,fields[i],fmin,fmax
    endelse
    data = *fields[lon.ifield].data
;
; If this is a change in grid type, reset select-longitude slider:
;
    if fields[lon.ifield].grid_type ne gridtype_prev then begin
      mag = 0 & if fields[lon.ifield].grid_type eq 'magnetic' then mag = 1
;     print,'Switching longitude slider geo<->mag..'
      base = widget_info(lon.wfsl_lon,/parent)
      widget_control,lon.wfsl_lon,/destroy
      lons = *fields[lon.ifield].lons
      nlon = n_elements(lons)
      dlon = lons[1]-lons[0] ; new dlon
      ilon = ixfind(lons,lon.slon,dlon)
      lon.slon = lons(ilon)
      title = 'Select '+fields[lon.ifield].grid_type+' Longitude'
      lon.wfsl_lon = cw_fslide(base,minimum=lons[0],maximum=lons[nlon-1],$
        delta=dlon,xsize=200,uvalue="SELECT_LON",format='(f7.2)',$
        value=lon.slon,title=title)
;
; Now reset select-localtime slider lon.wfsl_slt:
;
;     print,'Switching slt slider geo<->mag..'
      base = widget_info(lon.wfsl_slt,/parent)
      widget_control,lon.wfsl_slt,/destroy
      title = 'Select '+fields[lon.ifield].grid_type+' Local time'

      sslt = fslt(sslt,lon.ut,lon.slon,'getlt',mag=mag)
      dslt = dlon/15.
;
; Set lon.slon to longitude nearest current local time:
;
      if ixfind(lons,lon.slon,0.) < 0 then begin ; lon.slon not in lons
        ilon = ixfind(lons,lon.slon,dlon)
        lon.slon = lons[ilon]
        print,'select_field: sslt=',sslt,' nearest lon.slon=',lon.slon
      endif else begin
;       print,'select_field: found exact lon.slon=',lon.slon,' for slt=',sslt
        widget_control,lon.wfsl_lon,set_value=lon.slon
      endelse
;
      lon.wfsl_slt = cw_fslide(base,minimum=0.,maximum=24.,$
        delta=dslt,xsize=200,uvalue="SELECT_SLT",format='(f5.2)',$
        value=sslt,title=title)
    endif
    deflondata,lon,data
    if (not ptr_valid(lon.data)) then begin 
;     print,'lons: invalid pointer returned from deflondata -- returning'
      return
    endif
;
; Reset min,max and contour interval:
;
    data = *lon.data
    fminmax,data,fmin,fmax,lon.missing_value
    cmin=0. & cmax=0. & cint=0.
    levels = mkclevels(fmin,fmax,cmin,cmax,cint)
    lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
    *lon.clevels = levels
  endif ; found field name
endfor
if fields[lon.ifield].grid_type eq 'magnetic' and $
  ((lon.plottype eq 'image_only') or (lon.plottype eq 'image+contours')) then begin
  print,'>>> WARNING: cannot make image of magnetic field because of ',$
    'irregular latitude grid. Switching to color-fill contour..'
  lon.plottype = 'colorfill_contours'
  widget_control,lon.minmaxmenu,sensitive=0
  widget_control,lon.contourmenu,sensitive=1
endif
;
; Update state info:
;
*info.fields = fields
*info.plon = lon
*pinfo = info
pltlon,info
end
;-----------------------------------------------------------------------
pro mtime_lon_event,event
;
; Respond to user selection of a model time:
;
widget_control,event.top,get_uvalue=pinfo
widget_control,event.id,get_uvalue=mtime_request
info = *pinfo
fields = *info.fields
lon = *info.plon
field = fields[lon.ifield]
mag = 0 & if field.grid_type eq 'magnetic' then mag = 1
file = *info.pfile
imtime = -1
for i=0,file.ntimes-1 do begin
  mtime_string = $
    strcompress(string(file.mtimes[0,i]),/remove_all)+','+ $
    strcompress(string(file.mtimes[1,i]),/remove_all)+','+ $
    strcompress(string(file.mtimes[2,i]),/remove_all)
  if mtime_string eq mtime_request then begin
    imtime = i
    goto, found
  endif
endfor
print,'>>> WARNING mtime_lon_event: could not find index to model time ',$
  mtime_request
return
found:
lon.imtime = imtime
lon.mtime = file.mtimes[*,imtime]
;
; Reset slt slider from new ut and current selected lon:
;
lon.ut = lon.mtime[1]+lon.mtime[2]/60.
lon.sslt = fslt(lon.sslt,lon.ut,lon.slon,"getlt",mag=mag)
widget_control,lon.wfsl_slt,set_value=lon.sslt
;
; Define longitude slice data:
;
data = *fields[lon.ifield].data
deflondata,lon,data
;
; Reset contour levels:
;
if lon.setcontour eq 'off' then begin 
  data = *lon.data
  fminmax,data,fmin,fmax,lon.missing_value
  cmin=0. & cmax=0. & cint=0.
  levels = mkclevels(fmin,fmax,cmin,cmax,cint)
  lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
  *lon.clevels = levels
endif
;
*info.plon = lon
pltlon,info
end ; mtime_lon_event
;-----------------------------------------------------------------------
pro lons,info
;
; Set up longitude slice widget:
;
file = *info.pfile     ; file info structure
fields = *info.fields  ; fields info structure
;
title = 'Lon Slices: '+info.file
;print,'lons: info.groupleader=',info.groupleader
;tlb = widget_base(column=1,mbar=mbar, title=title,group_leader=info.groupleader)
tlb = widget_base(column=1,mbar=mbar, title=title)
;
lon_ids = *info.plon_ids
for i=0,n_elements(lon_ids)-1 do begin
  if (lon_ids(i) le 0) then begin
    lon_ids(i) = tlb
;   print,'lons: added tlb ',tlb,' to lon_ids at i=',i
    goto,done
  endif
endfor
done:
lonbase = tlb
missing_value = file.missing_value
;
; Fields menu:
;
;fieldsmenu = widget_button(mbar,value='Field')
;for i=0,info.nflds-1 do begin
;  button = widget_button(fieldsmenu,value=fields[i].name,$
;    uvalue=fields[i].name,event_pro='field_lon_event')
;  if (i eq 0) then field = fields[i].name
;endfor
fieldsmenu = fields_menu(mbar,fields,'field_lon_event')
;
; Model times menu:
;
mtimesmenu = mtime_menu(mbar,file.mtimes,'mtime_lon_event')
mtime = file.mtimes[*,0]
imtime = 0
;
; ------------------------- Plot options menus -----------------
;
optsmenu = widget_button(mbar,value='PlotOptions')
;
; Plot types (submenu under plot options):
;
plottype = widget_button(optsmenu,value='Plot Type',/menu)
button = widget_button(plottype,$
  value='Image Only',uvalue='IMAGE_ONLY')
button = widget_button(plottype,$
  value='Image plus Contours',uvalue='IMAGE+CONTOURS')
button = widget_button(plottype,$
  value='Monochrome Contours',uvalue='MONOCHROME_CONTOURS')
button = widget_button(plottype,$
  value='Color-fill Contours',uvalue='COLORFILL_CONTOURS')
;
; Plotting log10 of the fields: there are 3 options:
;   lon.log10 = 'density fields'  -> plot log10 of density fields only
;   lon.log10 = 'all fields'      -> always plot log10
;   lon.log10 = 'none'            -> never plot log10
; (See pro lons_event above, and deflondata.pro and pltlon.pro)
; Log10 submenu under PlotOptions:
;
log10menu = widget_button(optsmenu,value='Plot log10 of field:',/menu)
button = widget_button(log10menu,value='Plot log10 (density fields only)',$
  uvalue='LOG10')
button = widget_button(log10menu,value='Plot log10 (all fields)',$
  uvalue='LOG10_ALLFIELDS')
button = widget_button(log10menu,value='Plot linear (all fields)',$
  uvalue='LINEAR')
;
; Add or remove extra right-hand y-axis (if left-hand y-axis is linear
; Zp then add non-linear right y-axis in height):
;
yaxmenu = widget_button(optsmenu,value='Extra right-hand y-axis:',/menu)
button = widget_button(yaxmenu,value='Add Right-hand Y-axis in Height',$
  uvalue='ADD_YAX')
button = widget_button(yaxmenu,value='Remove Right-hand Y-axis in Height',$
  uvalue='RM_YAX')
;
; Plot zonal means:
;
zmmenu = widget_button(optsmenu,value='Zonal Mean',/menu)
button = widget_button(zmmenu,value='Zonal Mean On',uvalue='ZONAL_MEAN_ON')
button = widget_button(zmmenu,value='Zonal Mean Off',uvalue='ZONAL_MEAN_OFF')
;
; User set y-axis range (zp):
;
yaxismenu = widget_button(optsmenu,value='Set y-axis range (zp or ht)',/menu)
button = widget_button(yaxismenu,value='Reset to full range',$
  uvalue='RESET_YAXIS')
button = widget_button(yaxismenu,value='Set vertical min,max...',$
  uvalue='SET_YAXIS')
;
; User set x-axis range (latitude):
;
xaxismenu = widget_button(optsmenu,value='Set x-axis range (latitude)',/menu)
button = widget_button(xaxismenu,value='Reset to full range',$
  uvalue='RESET_XAXIS')
button = widget_button(xaxismenu,value='Set latitude min,max...',$
  uvalue='SET_XAXIS')
;
; Custom Contour:
;
contourmenu = widget_button(optsmenu,value='Custom Contour',/menu)
button = widget_button(contourmenu,value='Reset to full range',$
  uvalue='RESET_CONTOUR')
button = widget_button(contourmenu,value='Set min,max,interval...',$
  uvalue='CUSTOM_CONTOUR')
;
; User set field min,max of image:
;
minmaxmenu = widget_button(optsmenu,value='Fix Image min,max',/menu)
button = widget_button(minmaxmenu,value='Reset to full range',$
  uvalue='RESET_MINMAX') 
button = widget_button(minmaxmenu,value='Set Image min,max...',$
  uvalue='SET_MINMAX') 
;
; Change color tables:
;
button = widget_button(optsmenu,value='Color...',uvalue='COLOR')
;
; ------------------------- Animation menus --------------------------
;
animmenu = widget_button(mbar,value='Animate')
button = widget_button(animmenu,value='Animate in time (fixed longitude)',$
  uvalue='ANIM_TIME_FIXLON')
button = widget_button(animmenu,value='Animate in time (fixed local time)',$
  uvalue='ANIM_TIME_FIXSLT')
animlon = widget_button(animmenu,value='Animate in longitude (fixed ut)',$
  uvalue='ANIM_LON')
;
; ------------------------- Save output files --------------------------
;
savemenu = widget_button(mbar,value='SaveFiles')
button = widget_button(savemenu,value='Postscript...',uvalue='SAVE_PS')
button = widget_button(savemenu,value='PNG...',uvalue='SAVE_PNG')
button = widget_button(savemenu,value='GIF...',uvalue='SAVE_GIF')
button = widget_button(savemenu,value='BMP...',uvalue='SAVE_BMP')
button = widget_button(savemenu,value='JPEG...',uvalue='SAVE_JPEG')
button = widget_button(savemenu,value='TIFF...',uvalue='SAVE_TIFF')
;
; ------------------ Sliders for selected lons, slt -------------------
;
; Base for 2 sliders that select longitude and local time:
;
sliderbase = widget_base(tlb,column=2,/frame)
;
; Select longitude:
;
lons = *fields[0].lons
nlon = n_elements(lons)
dlon = lons[1]-lons[0]
slon = -180.
slonbase = widget_base(sliderbase)
title = 'Select '+fields[0].grid_type+' Longitude'
wfsl_slon = cw_fslide(slonbase,minimum=lons[0],maximum=lons[nlon-1],$
  delta=dlon,xsize=230,uvalue="SELECT_LON",format='(f7.2)',value=slon,$
  title=title)
;
; Select local time:
;
mag = 0 & if fields[0].grid_type eq 'magnetic' then mag = 1
sltbase = widget_base(sliderbase)
ut = mtime[1]+mtime[2]/60.
;sslt = fslt(sslt,ut,slon,1)
sslt = fslt(sslt,ut,slon,"getlt",mag=mag)
dslt = dlon/15.
title='Select '+fields[0].grid_type+' Local Time'
wfsl_slt = cw_fslide(sltbase,minimum=0.,maximum=24.,$
  delta=dslt,xsize=230,uvalue="SELECT_SLT",format='(f5.2)',value=sslt,$
  title=title)
;
*info.plon_ids = lon_ids
draw_xsize = 500 & draw_ysize = 500
draw = widget_draw(tlb,xsize=draw_xsize,ysize=draw_ysize,uvalue='DRAW_LONS')
button = widget_button(tlb,value='Close',uvalue='CLOSE')
;
lats = *fields[0].lats
latmin = lats[0] & latmax = lats[n_elements(lats)-1]
levs = *fields[0].levs
levmin = levs[0]
;nlev = fields[0].nlev
nlev = size(levs,/n_elements)
levmax = levs[nlev-1]
;
; -------------------------- Lon slice structure ---------------------
;
lon = {lonstruct,                 $
  field:fields[0],                $ ; current field structure
  ifield:0,		          $ ; index to current field	
  slon:slon,                      $ ; selected longitude
  sslt:sslt,                      $ ; selected local time
  lon_type:'lon',                 $ ; 'lon','slt', or 'zm'
  mtime:mtime,                    $ ; model time (default first time on file)
  imtime:imtime,                  $ ; index to mtime (file.mtimes[imtime]) 
  ut:ut,                          $ ; ut (hrs)
  missing_value:missing_value,    $ ; missing data value
  lonbase:lonbase,                $ ; widget id of cw lev slider base
  rhyaxis:1,                      $ ; if > 0 then add extra right-hand y-axis
  wfsl_lon:wfsl_slon,             $ ; widget id of lon selection slider
  wfsl_slt:wfsl_slt,              $ ; widget id of slt selection slider
  log10:'density fields',         $ ; plot log10 (densities, current field, on none)
  draw:draw,                      $ ; draw widget id
  draw_xsize:draw_xsize,          $ ; x size of draw widget
  draw_ysize:draw_ysize,          $ ; y size of draw widget
  minmaxmenu:minmaxmenu,          $ ; menu widget for setting image min,max
  contourmenu:contourmenu,        $ ; menu widget for custom contour
  ftype:info.ftype,               $ ; Type of input file (TIME-GCM or WACCM presently)
  plottype:'image+contours',      $ ; 4 options (see plottype_menu)
  clineclr:!d.table_size-1,       $ ; contour line color
  zonalmean:0,                    $ ; plot zonal mean if zonalmean==1
  setcontour:'off',               $ ; tracks if contours have been set
  fmin:0.,fmax:0.,cint:0.,        $ ; field min,max and contour interval
  clevels:ptr_new(/allocate_heap),$ ; contour levels
  lats:ptr_new(/allocate_heap),   $ ; latitudes for x-axis
  latmin:latmin,latmax:latmax,    $ ; optional latitude range for x-axis
  levs:ptr_new(/allocate_heap),   $ ; vertical levels for y axis
  levmin:levmin,levmax:levmax,    $ ; min,max of lev y-axis
  data:ptr_new(/allocate_heap)    $ ; 2d data (lats,levs)
}
;
; Read data from first field:
;
if ptr_valid(fields[0].data) then begin ; field has been read
  deflondata,lon,*fields[0].data
endif else begin
  widget_control,/hourglass
  varget,info,fields[0],ncdata
  field = fields[0]
  procfield,info,ncdata,field,info.z_hist
  fields[0] = field
  lon.levmin = (*field.levs)[0] & lon.levmax = (*field.levs)[field.nlev-1]
  deflondata,lon,*fields[0].data
  lon.field = field
  lon.levs = lon.field.levs
endelse
;
; Set min,max for first field:
;
fminmax,*lon.data,fmin,fmax,lon.missing_value
rpt_minmax,info,fields[0],fmin,fmax
widget_control,lon.minmaxmenu,sensitive=0 ; because default plottype is image+contour
;
; Set min,max and contour interval:
;
cmin=0. & cmax=0. & cint=0.
levels = mkclevels(fmin,fmax,cmin,cmax,cint)
lon.fmin=cmin & lon.fmax=cmax & lon.cint=cint
*lon.clevels = levels
;
; Update state info, realize the widgets, plot the first slice:
;
*info.fields = fields
info.plon = ptr_new(lon)
widget_control,tlb,/realize
pinfo = ptr_new(info)
pltlon,info
widget_control,tlb,set_uvalue=pinfo
xmanager,'lons',tlb,/no_block
end
