module water_tracers
!-----------------------------------------------------------------------
!
! Provide core functionality for water tracers.
!
! This module works in tandem with "waterisotope" which specifically
! treats the istopic fractionation.
!
! All interface routine are identified by wtrc_*, etc.
!
! Indexing ASSUMES (as everywhere else in CAM), normal water vapour is
! m=1. Cloud liquid and ice are probably m=2 and m=3... but not
! assumed, as they are correctly registered.
!
! DEFAULT CONFIGURATION is for 3 additional tracers with each of 3 phases, to
! parallel the base CAM water prognosis. This default is invoked by setting the
! namelist variable "wisotope". If not set, don't know what to do, so 
! complain and crash.
!
! Note total vapouq (Q) is registered as wet, even though it is treated
! as dry. This means the PD coupling is diferent from the vertical
! diffusion. Do get around this we make all water species wet, but have
! spacial cased for vapour, as in main cam code.
!
!
! Code here based on chemistry.F90 and cldcond.F90.
!
! Author: David Noone <dcn@caltech.edu> - Sun Jun 29 15:22:48 MDT 2003
!
!-----------------------------------------------------------------------
#undef O17HTO		/* set this to tun on H217O and HTO */

! PWSPC sets the dimensions internally, and for the public parameter pwspc
#ifdef O17HTO		/* H2O, HDO, H218O, H217O, HTO, set pcnst = 18 */
#  define PWSPC 5
#else			/* H2O, HDO, H218O, set pcnst = 12 */
#  define PWSPC 3
#endif

!-----------------------------------------------------------------------
! Various debug flags
#undef NOCHECK          /* DEBUG: define to bypass conservation checks */
#define QCHKMESS         /* DEBUG: define to give "success" message for checks */
#define QCHKTERM         /* DEBUG: define to terminate when "qchk" failed */
!-----------------------------------------------------------------------

  use shr_kind_mod, only: r8 => shr_kind_r8
  use spmd_utils,   only: masterproc
  use ppgrid,       only: pcols, pver
  use constituents, only: pcnst
  use abortutils,   only: endrun
!
  use water_isotopes, only: wisotope

  implicit none
  private
  save

!------------------------ Module Interfaces -----------------------------
!
! Public interfaces
!
  public :: wtrc_defaultopts                     ! set default values of namelist variables
  public :: wtrc_setopts                         ! get namelist input
  public :: wtrc_init                            ! initialize module parameters
  public :: wtrc_register                        ! register constituents
  public :: wtrc_implements_cnst                 ! checks tracer implementation
  public :: wtrc_is_wtrc			 ! logical function for m = water tracer
  public :: wtrc_is_vap				 ! logical function for m = vapour
  public :: wtrc_is_liq				 ! logical function for m = cloud liquid
  public :: wtrc_is_ice				 ! logical function for m = cloud ice
  public :: wtrc_init_cnst                       ! sets data if not on IC file
  public :: wtrc_init_qpert			 ! initialize boundary layer perturbation
  public :: wtrc_setup_diag			 ! write tracer configuration 
  public :: wtrc_qchk1                           ! compare 1d tracer with prognostic
  public :: wtrc_qchk2                           ! compare 2d tracer with prognostic
  public :: wtrc_qchk3                           ! compare all 2d with prognostsic
  public :: wtrc_check			 	 ! checks tracer with prognostic
  public :: wtrc_chkdelta			 ! checks delta values
  public :: wtrc_ratio			  	 ! calulates ratio to precision
  public :: wtrc_ratio_all			 ! calulates ratio for all tracers
!  public :: wtrc_rescale			 ! scaler routine

!------------------- Module Variable Declarations -----------------------

! Namelist variable
  logical, public :: trace_water = .false.       ! set true to activate [off]

  logical, public :: lwtrcland   = .true.        ! set to use simple land model

! Tracer physics control flags
!
  logical, parameter, public :: lh2oadj    = .true.     ! adjust tracer H20 to Q
  logical, parameter, public :: lnomfix    = .true.     ! do not apply usual mass fixer (eul core)
  logical, parameter, public :: lwtrczmlin = .false.	! linear interp for zm midpoitns (else log)
!
! Choose if check should be terminal
!
  logical, parameter :: lcheck_warn_only = .true.       ! true for message only, no endrun

! Water tracer type identifiers
  integer, parameter :: iwtundef = 0     ! not water type 
  integer, parameter :: iwtvap   = 1     ! water type is vapour
  integer, parameter :: iwtliq   = 2     ! water type is liquid
  integer, parameter :: iwtice   = 3     ! water type is ice

! Water tracer module data .
! (this is the default set up when wisotope is set true)
  integer, parameter, public :: pwspc = PWSPC	! "N" species of water (h2o, hdo, o18, o17, hto, ...)
  integer, parameter :: pwtyp = 3		! 3 types of water (vap, liq, ice)
  integer, parameter :: ncnst = pwtyp*pwspc     ! number of constituents treated by 
						! his module (PWTYP types, PWSPC species)
!
! Tracer vapour names by me no more than 5 characters for history files
  character(len=8), dimension(ncnst), parameter :: & ! constituent names 
#ifdef O17HTO
     wtrc_names = (/'H2O   ', 'H2OL  ', 'H2OI  ', &     ! NCNST=3
                    'HDO   ', 'HDOL  ', 'HDOI  ', &     ! NCNST=6
                    'H218O ', 'H218OL', 'H218OI', &     ! NCNST=9
                    'H217O ', 'H217OL', 'H217OI', &     ! NCNST=12
                    'HTO   ', 'HTOL  ', 'HTOI  ' /)     ! NCNST=15
#else
     wtrc_names = (/'H2O   ', 'H2OL  ', 'H2OI  ', &     ! NCNST=3
                    'HDO   ', 'HDOL  ', 'HDOI  ', &     ! NCNST=6
                    'H218O ', 'H218OL', 'H218OI' /)     ! NCNST=9
#endif

  integer, public :: &
       ixh2oq  , ixh2ol  , ixh2oi  , &! H2O   vap, liq, ice tracer indicies
       ixhdoq  , ixhdol  , ixhdoi  , &! HDO   vap, liq, ice tracer indicies
       ixh218oq, ixh218ol, ixh218oi, &! H218O vap, liq, ice tracer indicies
       ixh217oq, ixh217ol, ixh217oi, &! H217O vap, liq, ice tracer indicies
       ixhtoq  , ixhtol  , ixhtoi     ! HTO   vap, liq, ice tracer indicies
!
! Configuration pointers/indicies
  integer, public :: iwater(pcnst)      ! flag for water type:
                                        ! 1 vapour, 2 liquid, 3 ice
                                        ! 0 not water
  integer, public :: iwspec(pcnst)      ! flag for water (isotope) species
                                        ! see water_isotopes for definitions
  integer, public :: iwtrip(pcnst)      ! intenx of watre triplet number
!
! Index arrays for all specified water tracers
!
  integer, public :: iavap(pwspc)  ! index arrays for vapour
  integer, public :: ialiq(pwspc)  ! index arrays for cloud liquid
  integer, public :: iaice(pwspc)  ! index arrays for cloud ice
!
  integer, public :: ixwti, ixwtx  ! lowest and highest index to search
  integer, public :: ntrip	   ! number of watre triplets

!-----------------------------------------------------------------------
!
! Default minimum difference to trigger check failure
!
!  real(r8) :: qchkmin = 1.e-15		! loss of 3 s.f.
  real(r8) :: qchkmin = 1.e-16_r8		! loss of 2 s.f.
!  real(r8) :: qchkmin = 1.e-17		! loss of 1 s.f.
!  real(r8) :: qchkmin = 1.e-18	  	! qmin (model sees anything less as zero
!  real(r8) :: qchkmin = 1.e-19
!  real(r8) :: qchkmin = 1.e-20
!  real(r8) :: qchkmin = 1.e-21		! very strict
!
!-----------------------------------------------------------------------
contains

!=======================================================================
  subroutine wtrc_defaultopts(trace_water_out)
!-----------------------------------------------------------------------
!
! Purpose: set default values of namelist variables in the runtime_opts
!          module
!
!-----------------------------------------------------------------------
     logical, intent(out), optional :: trace_water_out

     if ( present(trace_water_out) ) then
        trace_water_out = trace_water
     endif
     
  end subroutine wtrc_defaultopts

!=======================================================================
  subroutine wtrc_setopts(trace_water_in)
!-----------------------------------------------------------------------
!
! Purpose: set user defined values and do consistency checking of
!          namelist variables
!
!-----------------------------------------------------------------------
     logical, intent(in), optional :: trace_water_in

     if ( present(trace_water_in) ) then
        trace_water = trace_water_in
     endif
     
  end subroutine wtrc_setopts

!=======================================================================
  subroutine wtrc_init
!-----------------------------------------------------------------------
!
! Purpose: initialize water_tracer parameterizations and indexing
!          (declare additional history field)
!
! Method:
!   Set up water indexing scheme (which must be done AFTER tracers
!   have been registered). Also, set up indexing for the prognostic
!   waters, just for completeness, although are not used.
!   Calls initialization of water isotope module. 
!
! Author: David Noone <dcn@caltech.edu> - Sun Jun 29 18:01:52 MDT 2003
!
!-----------------------------------------------------------------------
  use water_isotopes, only: wiso_init
  use cam_history,    only: addfld, add_default, phys_decomp
  use constituents,   only: cnst_name, cnst_longname
!-----------------------------------------------------------------------
  integer m
!-----------------------------------------------------------------------
    if (.not. trace_water) return
!
! We must have the isotopes (or some type of definiton) set up 
!
    if (.not. wisotope) then
      write(6,*) 'WTRC_INIT: Water tracers require water isotopes.'
      call endrun
    end if
    write(6,*) 'WTRC_INIT: Initializing water tracers.'
!
! Initialize isotope module
!
     call wiso_init
!
! Add further diagnostics to history file
! (including snow pack, precipitation, bi-directional fluxes)
!
     do m = ixwti, ixwtx
        if (wtrc_is_wtrc(m)) then
          call addfld (cnst_name(m), 'kg/kg   ', pver, 'A', cnst_longname(m), phys_decomp)
          call add_default (cnst_name(m), 1, ' ')
        end if
     end do

    return
  end subroutine wtrc_init

!=======================================================================
  subroutine wtrc_register
!-----------------------------------------------------------------------
!
! Purpose: resister advected water tracer constituents
!
! Method:
!  Calls CAM constituent registration routines based on 
!  water tracer species and phase indexing.
!
! Author: David Noone <dcn@caltech.edu> - Sun Jun 29 15:31:56 MDT 2003
!
!-----------------------------------------------------------------------
    use physconst,      only: mwdry, cpair, mwh2o, cph2o
    use constituents,   only: cnst_get_ind
    use water_isotopes, only: ispundef, isph2o, isphdo, isph218o, isph217o, isphto

    integer itrip	 ! counter for number of triplets

    integer ixwprg       ! constituent index of prognostic
    integer flag
!-----------------------------------------------------------------------
!
      if (.not. trace_water) return
!
! Initialize all tracers as nonwater, with unknwon species
!
      ntrip = 0
      iwater(:) = iwtundef
      iwspec(:) = ispundef
!
! Set the species of the total water as H2O, but DONT set them 
! as water tracers, the they are prognostic (not "tracers")
!
     ixwti = 0
     ixwtx = 0
!
     call cnst_get_ind('Q     ', ixwprg)
     iwspec(ixwprg) = isph2o
     ixwti = max(ixwtx,ixwprg)
     call cnst_get_ind('CLDLIQ', ixwprg)
     iwspec(ixwprg) = isph2o
     ixwti = max(ixwtx,ixwprg)
     call cnst_get_ind('CLDICE', ixwprg)
     iwspec(ixwprg) = isph2o
     ixwti = max(ixwtx,ixwprg)
     ixwti = ixwti + 1		! (SHOULD BE 4 FOR CAM2.x)

! Set names of variable tendencies and declare them as history variables

    if (wisotope) then

! H2O
      call wtrc_cnst_add('H2O'   , iwtvap, isph2o, mwh2o, cph2o, 0._r8, &
            ixh2oq, longname='H2O tracer specific humidity')
      call wtrc_cnst_add('H2OL'  , iwtliq, isph2o, mwdry, cpair, 0._r8, &
            ixh2ol, longname='H2O tracer grid box avg. liquid condensate amount')
      call wtrc_cnst_add('H2OI'  , iwtice, isph2o, mwdry, cpair, 0._r8, &
            ixh2oi, longname='H2O tracer grid box avg. ice condensate amount')
!
      ntrip = ntrip + 1
      iavap(ntrip) = ixh2oq
      ialiq(ntrip) = ixh2ol
      iaice(ntrip) = ixh2oi
!HDO
      call wtrc_cnst_add('HDO'   , iwtvap, isphdo, mwh2o, cph2o, 0._r8, &
            ixhdoq, longname='HDO tracer specific humidity')
      call wtrc_cnst_add('HDOL'  , iwtliq, isphdo, mwdry, cpair, 0._r8, &
            ixhdol, longname='HDO tracer grid box avg. liquid condensate amount')
      call wtrc_cnst_add('HDOI'  , iwtice, isphdo, mwdry, cpair, 0._r8, &
            ixhdoi, longname='HDO tracer grid box avg. ice condensate amount')
!
      ntrip = ntrip + 1
      iavap(ntrip) = ixhdoq
      ialiq(ntrip) = ixhdol
      iaice(ntrip) = ixhdoi
!H218O 
      call wtrc_cnst_add('H218O' , iwtvap, isph218o, mwh2o, cph2o, 0._r8, &
            ixh218oq, longname='H218O tracer specific humidity')
      call wtrc_cnst_add('H218OL', iwtliq, isph218o, mwdry, cpair, 0._r8, &
            ixh218ol, longname='H218O tracer grid box avg. liquid condensate amount')
      call wtrc_cnst_add('H218OI', iwtice, isph218o, mwdry, cpair, 0._r8, &
            ixh218oi, longname='H218O tracer grid box avg. ice condensate amount')
!
      ntrip = ntrip + 1
      iavap(ntrip) = ixh218oq
      ialiq(ntrip) = ixh218ol
      iaice(ntrip) = ixh218oi

!H217O - (PCNST=15)
    if (ntrip < pwspc) then
    call wtrc_cnst_add('H217O' , iwtvap, isph217o, mwh2o, cph2o, 0._r8, &
          ixh217oq, longname='H217O tracer specific humidity')
    call wtrc_cnst_add('H217OL', iwtliq, isph217o, mwdry, cpair, 0._r8, &
          ixh217ol, longname='H217O tracer grid box avg. liquid condensate amount')
    call wtrc_cnst_add('H217OI', iwtice, isph217o, mwdry, cpair, 0._r8, &
          ixh217oi, longname='H217O tracer grid box avg. ice condensate amount')
!
      ntrip = ntrip + 1
      iavap(ntrip) = ixh217oq
      ialiq(ntrip) = ixh217ol
      iaice(ntrip) = ixh217oi
    end if

!HTO - (PCNST=18)
    if (ntrip < pwspc) then
    call wtrc_cnst_add('HTO' , iwtvap, isphto, mwh2o, cph2o, 0._r8, &
          ixhtoq, longname='HTO tracer specific humidity')
    call wtrc_cnst_add('HTOL', iwtliq, isphto, mwdry, cpair, 0._r8, &
          ixhtol, longname='HTO tracer grid box avg. liquid condensate amount')
    call wtrc_cnst_add('HTOI', iwtice, isphto, mwdry, cpair, 0._r8, &
          ixhtoi, longname='HTO tracer grid box avg. ice condensate amount')
!
      ntrip = ntrip + 1
      iavap(ntrip) = ixhtoq
      ialiq(ntrip) = ixhtol
      iaice(ntrip) = ixhtoi
    end if
!
    else
      write(6,*) 'WTRC_REGISTER: unknown water tracer configuration.'
      call endrun
    endif
!
! Check registry and modyule dimensions
!
    if (ntrip /= pwspc) then
      write(*,*) '(WTRC_REGISTER): Number of registered triplets differs from module dimensions.'
      write(*,*) 'NTRIP = ',ntrip,'   PWTYP=',pwtyp, 'PWSPC=',pwspc
      call endrun
    end if
!
! Set up referse mapping from constituent index to water triplet number
!
    iwtrip(:) = 0
    do itrip = 1, ntrip		! ntrip == pwspc
       iwtrip(iavap(itrip)) = itrip
       iwtrip(ialiq(itrip)) = itrip
       iwtrip(iaice(itrip)) = itrip
    end do
!
! Request space on physics buffer for variables that persist across time steps
!
!!    call pbuf_add()
!
!
! Once the registration is done, report what we actually have just to make sure
!
      call wtrc_setup_diag
      write(*,*) 'WATER TRACERS m=',ixwti,ixwtx
    write(6,*) 'WTRC_REGISTER: done.'
!
    return
  end subroutine wtrc_register

!=======================================================================
  function wtrc_is_wtrc(m)
!-----------------------------------------------------------------------
! Returns true if tracer is vapour
!-----------------------------------------------------------------------
  integer, intent(in) :: m              ! constituent index
  logical wtrc_is_wtrc
!-----------------------------------------------------------------------
!!    wtrc_is_wtrc = wtrc_is_vap(m) .or. wtrc_is_liq(m) .or.  wtrc_is_ice(m)
    wtrc_is_wtrc = .false.
    if (iwater(m) /= iwtundef) wtrc_is_wtrc = .true.
  return
  end function wtrc_is_wtrc

!=======================================================================
  function wtrc_is_vap(m)
!-----------------------------------------------------------------------
! Returns true if tracer is vapour
!-----------------------------------------------------------------------
  integer, intent(in) :: m		! constituent index
  logical wtrc_is_vap
!-----------------------------------------------------------------------
    wtrc_is_vap = .false.
    if (iwater(m) == iwtvap) wtrc_is_vap = .true.
  return
  end function wtrc_is_vap

!=======================================================================
  function wtrc_is_liq(m)
!-----------------------------------------------------------------------
! Returns true if tracer is cloud liquid
!-----------------------------------------------------------------------
  integer, intent(in) :: m		! constituent index
  logical wtrc_is_liq 
!-----------------------------------------------------------------------
    wtrc_is_liq = .false.
    if (iwater(m) == iwtliq) wtrc_is_liq = .true.
  return
  end function wtrc_is_liq

!=======================================================================
  function wtrc_is_ice(m)
!-----------------------------------------------------------------------
! Returns true if tracer is cloud ice
!-----------------------------------------------------------------------
  integer, intent(in) :: m		! constituent index
  logical wtrc_is_ice 
!-----------------------------------------------------------------------
    wtrc_is_ice = .false.
    if (iwater(m) == iwtice) wtrc_is_ice = .true.
  return
  end function wtrc_is_ice

!=======================================================================
  subroutine wtrc_cnst_add(name, iwt, isp, mwc, cpc, qminc, ind, &
                           longname, readiv, mixtype)
!-----------------------------------------------------------------------
! Purpose: provide a wrapper for cnst_add with added index condifuration
!          for more details registration of water tracers
! Author: David Noone <dcn@caltech.edu> - Sun Jun 29 21:02:25 MDT 2003
!-----------------------------------------------------------------------
  use constituents, only: cnst_add 
!---------------------------- Arguments --------------------------------
    character(len=*), intent(in) :: &
       name      ! constituent name for variable name in history file(8 char max)
    character(len=*), intent(in), optional :: &
       longname  ! long_name attribute in netcdf output (128 char max) [name]
    logical,          intent(in), optional :: &
       readiv    ! true => read initial values from initial file (default: true)
    character(len=*),         intent(in), optional :: &
       mixtype    ! mixing ratio type (dry, wet)

    integer, intent(in)    :: iwt    ! water type indicator
    integer, intent(in)    :: isp    ! water species indicator
    real(r8),intent(in)    :: mwc    ! const. molecular weight (kg/kmol)
    real(r8),intent(in)    :: cpc    ! const. spcfic heat  const press (J/kg/K)
    real(r8),intent(in)    :: qminc  ! minimum  mass mixing ratio (kg/kg)
!                                        normally 0., except water 1.E-12, for
!                                        radiation.

    integer, intent(out)   :: ind    ! global constituent index (in q array)

!-----------------------------------------------------------------------
!
! Pass aruments on to normal code
!
    call cnst_add(name, mwc, cpc, qminc, ind, longname, readiv, mixtype)
!
! Knowing the tracer index assign water type and species
!
    iwater(ind) = iwt
    iwspec(ind) = isp
    ixwtx = max(ixwtx,ind)
!
    return
  end subroutine wtrc_cnst_add

!=======================================================================
  function wtrc_implements_cnst(name)
!-----------------------------------------------------------------------
!
! Purpose: return true if specified constituent is implemented by this package
! Notice wtrc_names should be the same as hard coded calls to cnst_resister.
!  
! Author: David Noone <dcn@caltech.edu> - Sun Jun 29 16:10:29 MDT 2003
!
!-----------------------------------------------------------------------
     implicit none
!-----------------------------Arguments---------------------------------
     character(len=*), intent(in) :: name   ! constituent name
     logical :: wtrc_implements_cnst        ! return value
!---------------------------Local workspace-----------------------------
     integer :: m
!-----------------------------------------------------------------------
     wtrc_implements_cnst = .false.
     do m = 1, ncnst
        if (name == wtrc_names(m)) then
           wtrc_implements_cnst = .true.
           return
        end if
     end do
     return
  end function wtrc_implements_cnst

!=======================================================================
  subroutine wtrc_init_cnst(name, qwtrc_tmp, q)
!-----------------------------------------------------------------------
!
! Initializes water tracers if not read from initial conditions file.
! Assign as some standard mass fraction of the prognostic waters
! (which  can assumed are set correctly). If using water isotope
! Set the standard ration, else set zero
!
!
! Author: David Noone <dcn@caltech.edu> - Sun Jun 29 18:24:57 MDT 2003
!
!-----------------------------------------------------------------------
    use water_isotopes, only: wisotope, wiso_get_rstd
    use constituents, only: cnst_get_ind
!---------------------------- Arguments --------------------------------
    character(len=*),intent(in)  :: name                ! tracer name
    real(r8),        intent(in)  :: qwtrc_tmp(:,:,:,:) ! cam3
    real(r8),        intent(out) :: q(:,:,:)   ! mass mixing ratio
!------------------------- Local Variables -----------------------------
    integer ixwtrc              ! index of water tracer
    integer ixwprg              ! intext of water prognostic
    real(r8) rat                ! an isotope ratio
!-----------------------------------------------------------------------
!
! Retrieve the tracer index, and work out index of equivilent prognostic
!
   call cnst_get_ind(name, ixwtrc)      ! this SHOULD be m in calling routine
!
   if (.not. wtrc_is_wtrc(ixwtrc)) then
      call endrun( 'WTRC_INIT_CNST: non water tracer detected.')
   else if (wtrc_is_vap(ixwtrc)) then    ! vapour
     call cnst_get_ind('Q     ', ixwprg)
   else if (wtrc_is_liq(ixwtrc)) then    ! liquid
     call cnst_get_ind('CLDLIQ', ixwprg)
   else if (wtrc_is_ice(ixwtrc)) then    ! ice
     call cnst_get_ind('CLDICE', ixwprg)
   else
      call endrun('WTRC_INIT_CNST: water tracer set as unknown water type.')
   end if
!
! Assign tracer to be total, scaled by some standard ratio 
!
    if (wisotope) then
      rat = wiso_get_rstd(iwspec(ixwtrc))
    else
!      rat = 0.
      rat = 1._r8
    endif
!
    q(:,:,:) = rat*qwtrc_tmp(:,:,:,ixwprg)
!
    return
  end subroutine wtrc_init_cnst

!=======================================================================
   subroutine wtrc_init_qpert(qpert)
!-----------------------------------------------------------------------
! Initialize constituent perturbation to something (smow?)
!-----------------------------------------------------------------------
    use water_isotopes, only: wiso_get_rstd

    real(r8), intent(inout) :: qpert(pcols,pcnst)

    integer m
    real(r8) rat
!-----------------------------------------------------------------------
    do m = ixwti, ixwtx
       qpert(:,m) = 0._r8
      if (wtrc_is_vap(m)) then

        if (wisotope) then
          rat = wiso_get_rstd(iwspec(m))
        else
!          rat = 0.
          rat = 1._r8
        endif
!
        qpert(:,m) = rat*qpert(:,1)
 
      end if
     end do
!   
     return
   end subroutine wtrc_init_qpert

!=======================================================================
  subroutine  wtrc_setup_diag
!-----------------------------------------------------------------------
! Purpose: Writes configuration of water tracer scheme to standard output.
!-----------------------------------------------------------------------
    use constituents, only: cnst_name
    use water_isotopes, only: wiso_get_fisub, wiso_get_rstd
!------------------------- Local Variables -----------------------------
    integer m
!-----------------------------------------------------------------------

    write(6,*) ' ' 
    write(6,*) '---- Water isotopes tracer configurtaion ----'
    write(6,*) 'name      N  W  S  f     Rstd'
    do m = 1, pcnst
      if (wtrc_is_wtrc(m)) then
      write(6,1) cnst_name(m),iwtrip(m),iwater(m), iwspec(m),  &
             int(wiso_get_fisub(iwspec(m))), wiso_get_rstd(iwspec(m))
      end if
 1    format(a8,' ', i3,i3,i3,i3,e16.5)
    end do
    write(6,*) '---------------------------------------------'
    write(6,*) ' ' 
!
    return
  end subroutine wtrc_setup_diag

!=======================================================================
  subroutine wtrc_qchk3(subr, vname, ncol, q, qmag0)
!-----------------------------------------------------------------------
! Checks that all tracers areRstd*prognostic 
! (used for debuggin with no fractionation)
!-----------------------------------------------------------------------
    use water_isotopes, only: wisotope, wiso_get_rstd
    use constituents, only: pcnst, cnst_get_ind

!---------------------------- Arguments --------------------------------
    character(len=*),intent(in) :: subr   ! name of calling subroutine
    character(len=*),intent(in) :: vname  ! name of variable
    integer , intent(in) :: ncol          ! number of columns to scan
    real(r8), intent(in) :: q(pcols,pver,pcnst)   ! tarcers
    real(r8), intent(in), optional :: qmag0      ! minimum magnitude of qprg
!------------------------- Local Variables -----------------------------
    real(r8) rstd
    integer ixvap,ixliq,ixice
    integer mvap, mliq, mice
    integer m
!-----------------------------------------------------------------------

    call cnst_get_ind('Q'     , ixvap)
    call cnst_get_ind('CLDLIQ', ixliq)
    call cnst_get_ind('CLDICE', ixice)
!
    do m = ixwti,ixwtx
      if (wtrc_is_vap(m)) then
        mvap = m
        mliq = m + 1
        mice = m + 2
        rstd = wiso_get_rstd(iwspec(m))
        write(*,'(a40,3i6,g16.6)') 'WTRC_QCHK3 ('//trim(subr)//') - tracers:',mvap,mliq,mice, rstd
        call wtrc_qchk2(subr,trim(vname)//'_v',ncol,q(:,:,mvap),rstd*q(:,:,ixvap),qmag0)
        call wtrc_qchk2(subr,trim(vname)//'_l',ncol,q(:,:,mliq),rstd*q(:,:,ixliq),qmag0)
        call wtrc_qchk2(subr,trim(vname)//'_i',ncol,q(:,:,mvap),rstd*q(:,:,ixvap),qmag0)
      end if
    end do
!
    return
  end subroutine wtrc_qchk3

!=======================================================================
  subroutine wtrc_qchk2(subr,vname,ncol,qtrc,qprg,qmag0)
!-----------------------------------------------------------------------
! Purpose: Check the tracer water mass equal the prognostic
! Author: David Noone <dcn@caltech.edu> - Mon Jun 30 19:00:15 MDT 2003
!-----------------------------------------------------------------------

!---------------------------- Arguments --------------------------------
    character(len=*),intent(in) :: subr   ! name of calling subroutine
    character(len=*),intent(in) :: vname  ! name of variable
    integer , intent(in) :: ncol          ! number of columns to scan
    real(r8), intent(in) :: qtrc(pcols,pver)   ! tracer water
    real(r8), intent(in) :: qprg(pcols,pver)   ! prognostic water
    real(r8), intent(in), optional :: qmag0      ! minimum magnitude of qprg
!------------------------- Local Variables -----------------------------
    real(r8) etest                        ! test variable
    real(r8) qmag
    real(r8) qdw, etw                   ! worst values
    integer nbad                        ! number of bad values found
    integer i,k
!-----------------------------------------------------------------------
    nbad = 0
!    qmag = 0.
    qmag = qchkmin
    qdw = 0._r8
    etw = 0._r8
    if (present(qmag0)) qmag = qmag0
!
    do k = 1, pver
      do i = 1, ncol
       if (wtrc_qchk_one(qtrc(i,k),qprg(i,k),etest,qmag) > 0) then
#ifdef QCHKTERM
          write(6,1) 'WTRC_QCHK2: '//'('//trim(subr)//'.'//trim(vname)//')'// &
          ' q(m,1):',i,k,qtrc(i,k),qprg(i,k),etest
1       format(a36,2i4,2e12.3,e12.4)
#endif
         etw = max(etw, abs(etest))
         qdw = max(qdw, abs(qtrc(i,k)-qprg(i,k)))
         nbad = nbad + 1
       end if
      end do
    end do
!
    if (nbad /= 0) then
        write(6,*) 'WTRC_QCHK2: '//'('//trim(subr)//'.'//trim(vname)//')',&
              ' *** WARNING - chunk tracers /= Q =',nbad,etw,qdw
#ifdef QCHKTERM 
        call endrun('QCHK2 failed.')
#endif
    else

#ifdef QCHKMESS		/* print a sucess message */
        write(6,*) 'WTRC_QCHK2: '//'('//trim(subr)//'.'//trim(vname)//')',&
              ' All OK.'
#endif
    end if
!
    return
  end subroutine wtrc_qchk2

!=======================================================================
  subroutine wtrc_qchk1(subr,vname,ncol,qtrc,qprg,qmag0)
!-----------------------------------------------------------------------
! Purpose: Check the tracer water mass equal the prognostic
! Author: David Noone <dcn@caltech.edu> - Mon Jun 30 19:00:15 MDT 2003
!-----------------------------------------------------------------------
!---------------------------- Arguments --------------------------------
    character(len=*),intent(in) :: subr   ! name of calling subroutine
    character(len=*),intent(in) :: vname  ! name of variable
    integer , intent(in) :: ncol	  ! number of columns to scan
    real(r8), intent(in) :: qtrc(pcols)   ! tracer water
    real(r8), intent(in) :: qprg(pcols)   ! prognostic water
    real(r8), intent(in),optional :: qmag0 ! minimum q for fail test
!------------------------- Local Variables -----------------------------
    real(r8) etest                        ! test variable
    real(r8) qmag
    real(r8) qdw, etw                   ! worst values
    integer nbad 		    	  ! number of bad values found
    integer i
!-----------------------------------------------------------------------
    nbad = 0
!    qmag = 0.
    qmag = qchkmin
    qdw = 0._r8
    etw = 0._r8
    if (present(qmag0)) qmag = qmag0
!
    do i = 1, ncol
       if (wtrc_qchk_one(qtrc(i),qprg(i),etest,qmag) > 0) then
#ifdef QCHKTERM
          write(6,1) 'WTRC_QCHK1: '//'('//trim(subr)//'.'//trim(vname)//')'// &
          ' q(m,1):',i,qtrc(i),qprg(i),etest
1       format(a40,i4,'    ',2e12.4,e12.4)
#endif
         etw = max(etw, abs(etest))
         qdw = max(qdw, abs(qtrc(i)-qprg(i)))
         nbad = nbad + 1
       end if
    end do
!
    if (nbad /= 0) then
        write(6,*) 'WTRC_QCHK1: '//'('//trim(subr)//'.'//trim(vname)//')',&
              ' *** WARNING - chunk tracers /= Q =',nbad,etw,qdw
#ifdef QCHKTERM               /* terminate */
        call endrun('QCHK1 failed.')
#endif
    else
#ifdef QCHKMESS		/* print a sucess message */
        write(6,*) 'WTRC_QCHK1: '//'('//trim(subr)//'.'//trim(vname)//')',&
              ' All OK.'
#endif
    end if
!
    return
  end subroutine wtrc_qchk1


!=======================================================================
  function wtrc_qchk_one(qtrc,qprg,etest,qmag)
!-----------------------------------------------------------------------
! Purpose: Check the one tracer water mass equal the prognostic
! Author: David Noone <dcn@caltech.edu> - Mon Jun 30 19:00:15 MDT 2003
!-----------------------------------------------------------------------
!    real(r8), parameter :: elimit = 1.0e-16 ! precision required
!    real(r8), parameter :: elimit = 1.0e-14 ! precision required (q1q2 fails)
!    real(r8), parameter :: elimit = 1.0e-12 ! precision required
    real(r8), parameter :: elimit = 1.0e-10_r8 ! precision required
!    real(r8), parameter :: qmin = 1.0e-18 ! precision required
    real(r8), parameter :: qmin = 1.e-26_r8 ! precision required
!---------------------------- Arguments --------------------------------
    real(r8), intent(in) :: qtrc          ! tracer water
    real(r8), intent(in) :: qprg          ! prognostic water
    real(r8), intent(in) :: qmag          ! difference limit
    real(r8), intent(out) :: etest        ! test variable
    real(r8)              qdiff,qmabs
    integer wtrc_qchk_one		  ! return value
!-----------------------------------------------------------------------
    wtrc_qchk_one = 0

#ifdef NOCHECK
!
! By-pass all checking if running with fractionation
!
#else
    qmabs = max(abs(qprg),abs(qtrc))
    qdiff = abs(qtrc - qprg)
    if (qmabs > qmin .and. qdiff > qmag) then
      etest = qdiff / qmabs
      if (etest > elimit) then
         wtrc_qchk_one = 1
#ifdef QCHKTERM		/* if going to fail, write all diagnostics */
         write(*,'(a36,4e16.9)') '(WTRC_QCHK) FAILED TEST:',qtrc,qprg,qdiff,etest
#endif
      end if
    end if
#endif
!
    return
  end function wtrc_qchk_one

!=======================================================================
  subroutine wtrc_chkdelta(subr, ncol, q)
!-----------------------------------------------------------------------
! Checks the delta values of a 2d array (lon,lev)
!-----------------------------------------------------------------------
    use constituents, only: cnst_get_ind
    use water_isotopes, only: wiso_delta
!---------------------------- Arguments --------------------------------
    character(len=*), intent(in) :: subr	! name of calling routine/message
    integer , intent(in) :: ncol	  	! number of columns to scan
    real(r8), intent(in) :: q(pcols,pver,pcnst)	! tracer quantity
    real(r8) del, delbad
    integer i,k,m
    integer ixvap,ixliq,ixice		! prognostic water species
    integer mbase			! prognostic base for tracer m
    integer nbad
    real(r8) qbad
!-----------------------------------------------------------------------
!
  call cnst_get_ind('Q'     , ixvap)
  call cnst_get_ind('CLDLIQ', ixliq)
  call cnst_get_ind('CLDICE', ixice)
!
! Apply appropriate scaling
!
    do m = ixwti, ixwtx
!
      if (wtrc_is_vap(m)) then
         mbase = ixvap
         mbase = ixh2oq		! relative to tracer water
      else if (wtrc_is_liq(m)) then
         mbase = ixliq
         mbase = ixh2ol		! relative to tracer water
      else if (wtrc_is_ice(m)) then
         mbase = ixice
         mbase = ixh2oi		! relative to tracer water
      else
         write(*,*) '(WTRC_CHKDELTA) unknown tracer.'
         call endrun
      end if
!
     delbad = 0._r8
     qbad = 0._r8
     nbad = 0._r8
      do k = 1, pver
         do i = 1, ncol
            del = wiso_delta(iwspec(m), q(i,k,m), q(i,k,mbase))
            if (abs(del) > 1001._r8) then
              nbad = nbad + 1
              if (abs(del) > abs(delbad)) then
                 qbad = q(i,k,mbase)
                 delbad = del
              endif
!!              call endrun('(wtrc_chkdelta) Stopped.')
            end if
         end do
      end do

!!      if (nbad > 0) then
!!        write(*,*) trim(subr)//' Bad delta values for m=',m
!!        write(*,*) 'nbad = ',nbad, '  worst=',delbad,qbad
!!      end if
    end do
    return
  end subroutine wtrc_chkdelta

!=======================================================================
  subroutine wtrc_check(subr, ncol, q)
!-----------------------------------------------------------------------
! Checks H2O tracer (ice, liquid and vapour) is the same as the prognostic
! (optioanllly adjust)
!-----------------------------------------------------------------------
    use constituents, only: cnst_get_ind, qmin
    use water_isotopes, only: wiso_delta
!---------------------------- Arguments --------------------------------
    character(len=*), intent(in) :: subr        ! name of calling routine/message
    integer , intent(in) :: ncol                ! number of columns to scan
    real(r8), intent(inout) :: q(pcols,pver,pcnst) ! tracer quantity (optionally scaled)
    real(r8) dvap(pcols,pver)
    real(r8) dliq(pcols,pver)
    real(r8) dice(pcols,pver)
    integer i,k,m
    integer iw(2)			! indices of worst values
    integer ixvap,ixliq,ixice           ! prognostic water species
    integer mbase                       ! prognostic base for tracer m
    logical lerrors
!-----------------------------------------------------------------------
    real(r8) :: qtol = 1.e-12_r8		! qmin(Q) = 1.e-12, qmin(L,I) = 0.)
!    real(r8) :: qtol = 1.e-15		! qmin(Q) = 1.e-12, qmin(L,I) = 0.)
!    real(r8) :: qtol = 1.e-17		! Too strict for zm_evap (review?)
!-----------------------------------------------------------------------
!
    lerrors = .false.
    call cnst_get_ind('Q'     , ixvap)
    call cnst_get_ind('CLDLIQ', ixliq)
    call cnst_get_ind('CLDICE', ixice)
!
!  Get all differences
!
    do k = 1, pver
      do i = 1, ncol
        dvap(i,k) = q(i,k,ixvap) - q(i,k,ixh2oq)
        dliq(i,k) = q(i,k,ixliq) - q(i,k,ixh2ol)
        dice(i,k) = q(i,k,ixice) - q(i,k,ixh2oi)
      end do
    end do
!
! Send reports
!
    if (.not. lcheck_warn_only) then
      do k = 1, pver
        do i = 1, ncol
          if (abs(dvap(i,k)) > max(qtol,qmin(ixvap))) then
            write(*,1) i,k,q(i,k,ixvap), q(i,k,ixh2oq), dvap(i,k) 
1           format (2i3,3e20.10)
          end if
        end do
      end do
    end if
!
    if (count(abs(dvap(:ncol,:))>max(qtol,qmin(ixvap))) > 0) then
       write(*,*) '(wtrc_check) vapour differences: '//trim(subr)
       iw = maxloc(abs(dvap(:ncol,:)))
       write(*,2) count(abs(dvap(:ncol,:))>qmin(ixvap)),iw(1),iw(2),dvap(iw(1),iw(2)), q(iw(1),iw(2),ixh2oq)
       if (.not. lcheck_warn_only) &
           call endrun('wtrc_check: vapour check failed.')
    endif
    if (count(abs(dliq(:ncol,:))>max(qtol,qmin(ixliq))) > 0) then
       write(*,*) '(wtrc_check) liquid differences: '//trim(subr)
       iw = maxloc(abs(dliq(:ncol,:)))
       write(*,2) count(abs(dliq(:ncol,:))>qtol),iw(1),iw(2),dliq(iw(1),iw(2)), q(iw(1),iw(2),ixh2ol)
       if (.not. lcheck_warn_only) &
           call endrun('wtrc_check: cloud liquid check failed.')
    endif
    if (count(abs(dice(:ncol,:))>max(qtol,qmin(ixice))) > 0) then
       write(*,*) '(wtrc_check) ice differences: '//trim(subr)
       iw = maxloc(abs(dice(:ncol,:)))
       write(*,2) count(abs(dice(:ncol,:))>qtol),iw(1),iw(2),dice(iw(1),iw(2)), q(iw(1),iw(2),ixh2oi)
       if (.not. lcheck_warn_only) &
           call endrun('wtrc_check: cloud ice check failed.')
    endif
 2  format(i4,' point(s), worst (i,k):',2i5,2e16.6)
!
! Do any adjustments
!
    if (lh2oadj) then
!!        write(*,*) 'Applying rescaling to tracers to prohibit drift.'
        call wtrc_rescale(q,ncol)
    end if
!
#ifdef QCHKMESS
   write(*,*) '(wtrc_check) all OK: '//trim(subr)
#endif
!
    return
  end subroutine wtrc_check

!=======================================================================
  subroutine wtrc_rescale(q,ncol)
!-----------------------------------------------------------------------
! Purpose: Ensures tracer water mass is exactly the same as prognostic
! Author: David Noone <dcn@caltech.edu> - Mon Mar  8 16:22:30 PST 2004
!-----------------------------------------------------------------------
    use constituents,   only: cnst_get_ind
!---------------------------- Arguments --------------------------------
  integer, intent(in)     :: ncol
  real(r8), intent(inout) :: q(pcols,pver,pcnst)
!-----------------------------------------------------------------------
  real(r8) qerr(pcols,pver)
  real(r8) rat
  integer i,k,m
  integer ixvap,ixliq,ixice	! prognostic water species
  integer mbase			! tracer base for tracer m
  integer mprog			! prognostic base for tracer m
!-----------------------------------------------------------------------
!
  call cnst_get_ind('Q'     , ixvap)
  call cnst_get_ind('CLDLIQ', ixliq)
  call cnst_get_ind('CLDICE', ixice)
!
! Apply appropriate scaling
!
    do m = ixwti, ixwtx
!
      if (wtrc_is_vap(m)) then
         mprog = ixvap
         mbase = ixh2oq		! relative to tracer water
      else if (wtrc_is_liq(m)) then
         mprog = ixliq
         mbase = ixh2ol		! relative to tracer water
      else if (wtrc_is_ice(m)) then
         mprog = ixice
         mbase = ixh2oi		! relative to tracer water
      else
         write(*,*) '(WTRC_RESCALE) unknown traaver.'
         call endrun
      end if
!
! Compute the error
!
      qerr(:,:) = q(:,:,mbase) - q(:,:,mprog)
!
! Compute tracer ratio, consistent with tracers, then
! apply to total (prognostic) mass
!
      do k = 1, pver
        do i = 1, ncol
           rat = wtrc_ratio(q(i,k,m), q(i,k,mbase))
!!           q(i,k,m) =  rat*q(i,k,mprog) 		! direct scale
           q(i,k,m) =  q(i,k,m) - rat*qerr(i,k) 	! scale error 
        end do
      end do
!
    end do
!
    return
  end subroutine wtrc_rescale

!=======================================================================
  function wtrc_ratio(qtrc,qtot)
!-----------------------------------------------------------------------
! Purpose: Compute tracer ratio from masses, with numerical checks
! Author David Noone <dcn@colorado.edu> - Sat Jul  3 18:52:40 MDT 2004
!-----------------------------------------------------------------------
    real(r8),intent(in)  :: qtrc        ! tracer water mass
    real(r8),intent(in)  :: qtot        ! "base" water mass
    real(r8) :: wtrc_ratio              ! return value
!-----------------------------------------------------------------------
!    real(r8) :: qtiny = 1.e-16		! bigger makes scheme more stable
    real(r8) :: qtiny = 1.e-22_r8		! smaller makes scheme more accurate
!-----------------------------------------------------------------------
    if (abs(qtot) < qtiny) then
       wtrc_ratio = 0._r8
       return
    end if
!
    if (qtot > 0._r8) then
      wtrc_ratio = qtrc/(qtot+qtiny)
    else
      wtrc_ratio = qtrc/(qtot-qtiny)
    end if
    return
  end function wtrc_ratio

!=======================================================================
  subroutine wtrc_ratio_all(ncol,q,rat)
!-----------------------------------------------------------------------
! Computes ratios for all water tracers
!-----------------------------------------------------------------------
    use constituents,   only: cnst_get_ind
!---------------------------- Arguments --------------------------------
   integer , intent(in) :: ncol
   real(r8), intent(in) :: q(pcols,pver,pcnst)
   real(r8), intent(out) :: rat(pcols,pver,pcnst)
!-----------------------------------------------------------------------
   integer i,k,m
   integer ixvap,ixliq,ixice		! prognostic water species
   integer mbase			! prognostic base for tracer m
!-----------------------------------------------------------------------
   rat(:,:,:) = 0._r8
!
   call cnst_get_ind('Q'     , ixvap)
   call cnst_get_ind('CLDLIQ', ixliq)
   call cnst_get_ind('CLDICE', ixice)
!
   rat(:,:,ixvap) = 1._r8
   rat(:,:,ixliq) = 1._r8
   rat(:,:,ixice) = 1._r8
!
!  Compute ratios based on "parent"
!
    do m = ixwti, ixwtx
!
      if (wtrc_is_vap(m)) then
         mbase = ixvap
      else if (wtrc_is_liq(m)) then
         mbase = ixliq
      else if (wtrc_is_ice(m)) then
         mbase = ixice
      else
         write(*,*) '(WTRC_RAT_ALL) unknown traaver.'
         call endrun
      end if

      do k = 1, pver
        do i = 1, ncol
           rat(i,k,m) = wtrc_ratio(q(i,k,m), q(i,k,mbase))
        end do
      end do

    end do
!
    return
  end subroutine wtrc_ratio_all

!=======================================================================

end module water_tracers
