;-----------------------------------------------------------------------
pro bases_sensitivity,stateptr,sensitive
  state = *stateptr
  widget_control,state.tlb,sensitive=sensitive
  if (widget_info(state.utvert_tlb,/valid_id) gt 0) then $
    widget_control,state.utvert_tlb,sensitive=sensitive
  if (widget_info(state.utlat_tlb,/valid_id) gt 0) then $
    widget_control,state.utlat_tlb,sensitive=sensitive
end
;-----------------------------------------------------------------------
pro open_ps_file,stateptr
  state = *stateptr
;
; Save current ps_outfile (should be open):
  ps_outfile_save = state.ps_outfile
;
; Read new file name from text widget:
;
  id_ps_outfile = widget_info(state.tlb,find_by_uname='SET_PS_OUTFILE')
  widget_control,id_ps_outfile,get_value=file
  ps_outfile = strcompress(file,/remove_all)
;
; Get info on new ps_outfile from text widget:
  fileinfo = file_info(ps_outfile)
;
; If file from text widget is same as current file, then,
; if file does not exist (user may have deleted or renamed
; it), reopen it and set output device, otherwise return 
; silently (user toggled back from X display output):
;
  if (ps_outfile eq ps_outfile_save) then begin
    if (not fileinfo.exists) then begin
      print,format="('Opening new ps file ',a)",ps_outfile
      set_output_device,stateptr
    endif else return
  endif
;
; If directory does not exist, revert to current file.
; (no change in state, but reset ps file text widget)
;
  dir = strcompress(file_dirname(file),/remove_all)
  dirinfo = file_info(dir)
  if (dirinfo.exists eq 0) then begin
    print,'>>> Directory ',dir,' does not exist.'
    print,'Output will continue to ps file ',ps_outfile_save
    widget_control,id_ps_outfile,set_value=ps_outfile_save
    return
  endif
;
; Ok to open file if it does not exist or is empty:
;
  if (not fileinfo.exists or (fileinfo.exists and fileinfo.size eq 0)) $
    then begin
    state.ps_outfile = ps_outfile
    *stateptr = state
    print,'Closing ps file ',ps_outfile_save,$
      ' (this file cannot be reopened without truncation)' 
    set_output_device,stateptr
    return
  endif
;
; File exists and is non-empty -> ask user if its ok to open
; and truncate:
;
; I wanted to desensitize all bases prior to the truncate query,
; but if the user has closed one of the plot widget bases (mkutvert
; or mkutlat) using the close box in upper right of the window
; rather than my CANCEL button, then I don't know that its gone
; away, and bases_sensitivity stops w/ bad widget id. So will
; have to rely on modal aspect of the plot widget bases. The risk
; here is that the user might not know why none of the buttons
; are working.
;
  bases_sensitivity,stateptr,0
  file_truncate_query,stateptr,ps_outfile
  state = *stateptr
;
; User does not want to open/truncate, so revert to current file:
  if (state.ps_confirm_open eq 0) then begin
    widget_control,id_ps_outfile,set_value=ps_outfile_save
    print,'PS plots will continue to be sent to file ',state.ps_outfile
;
; User does want to open/truncate, so call set_output_device.
; Warn the user that the old file cannot be reopened w/o truncation.
; (should maybe do this warning w/ a pop-up, since they may not be
;  watching stdout window)
;
  endif else begin
    bases_sensitivity,stateptr,1
    print,'Closing ps file ',ps_outfile_save,$
      ' (this file cannot be reopened without truncation)' 
    set_output_device,stateptr
  endelse
  bases_sensitivity,stateptr,1
end ; open_ps_file
;-----------------------------------------------------------------------
function ixfind, z,val,del
;
; Find nearest index to val in z, with increment del
; If not found, return -1
;
nz = n_elements(z)
if del gt 0 then begin
  for i=0,nz-1 do begin
    if (val ge z(i)-0.5*del and val le z(i)+0.5*del) then return,i
  endfor
endif else begin
  for i=0,nz-1 do begin
    if (val eq z(i)) then return,i
  endfor
endelse
return,-1
end
;-----------------------------------------------------------------------
function ixfind_simple,a,v
  na = n_elements(a)
  for k=0,na-1 do begin
    if (v eq a[k]) then return,k
  endfor
  return,-1
end
;-----------------------------------------------------------------------
 pro set_output_device,stateptr
;
; Set output device:
;
  state = *stateptr
  if (state.device eq 'PS') then begin
    set_plot,'PS'
;   device,/color,/landscape,filename=state.ps_outfile    ; default width=9.5"
;   device,/color,/landscape,/inches,xsize=11.,ysize=8.,$ ; 11" wide
;     filename=state.ps_outfile
;   device,/color,/landscape,/inches,xsize=12.,ysize=8.,$ ; 12" wide
;     filename=state.ps_outfile
    device,/color,/landscape,/inches,xsize=13.,ysize=8.,$ ; 13" wide
      filename=state.ps_outfile
    id = widget_info(state.tlb,find_by_uname='SET_PS_DEVICE')
    widget_control,id,/set_button
    print,format="('Set output device to PS (output file = ',a,')')",$
      state.ps_outfile
  endif else if (state.device eq 'X') then begin
    set_plot,'X'
    device,decomposed=0,retain=2,set_font='Times',/tt_font ; TrueType font
;   device,decomposed=0,retain=2                           ; for TrueColor display
;   window,xsize=650,ysize=450,retain=2,title='utproc'
    window,xsize=750,ysize=550,retain=2,title='utproc'
    id = widget_info(state.tlb,find_by_uname='SET_X11_DEVICE')
    widget_control,id,/set_button
    print,'Set output device to X display.'
  endif
end
;-----------------------------------------------------------------------
pro find_mtimes,stateptr
;
; Given state.mtime_beg, find state.ifile_beg and state.itime_file_beg,
; and same for mtime_end.
;
  state = *stateptr
  mtime_beg = state.mtime_beg
  mtime_end = state.mtime_end
  files = *state.files
  nfiles = n_elements(files)
;
; Search for start time (state.mtime_beg):
;
  for ifile=0,nfiles-1 do begin
    ncid = ncdf_open(files[ifile],/nowrite)
    id = ncdf_varid(ncid,'mtime')
    ncdf_varget,ncid,id,mtime
    ntime = n_elements(mtime)/3
    found = 0
    for i=0,ntime-1 do begin
      if (array_equal(mtime[*,i],state.mtime_beg)) then begin
        found = 1
        itime_beg = i
        ifile_beg = ifile
        break
      endif
    endfor
    if (found eq 0) then continue
;   print,format="('Found start time ',3i4,' ifile=',i3,' itime_beg=',i3)",$
;     state.mtime_beg,ifile_beg,itime_beg
    ncdf_close,ncid
    break
  endfor
  if (found eq 0) then begin
    print,'>>> WARNING: find_mtimes: could not find mtime_beg ',mtime_beg
  endif
;
; Search for end time (state.mtime_end):
;
  for ifile=0,nfiles-1 do begin
    ncid = ncdf_open(files[ifile],/nowrite)
    id = ncdf_varid(ncid,'mtime')
    ncdf_varget,ncid,id,mtime
    ntime = n_elements(mtime)/3
    found = 0
    for i=0,ntime-1 do begin
      if (array_equal(mtime[*,i],state.mtime_end)) then begin
        found = 1
        itime_end = i
        ifile_end = ifile
        break
      endif
    endfor
    if (found eq 0) then continue
;   print,format="('Found end time ',3i4,' ifile=',i3,' itime_end=',i3)",$
;     state.mtime_end,ifile_end,itime_end
    ncdf_close,ncid
    break
  endfor
  if (found eq 0) then begin
    print,'>>> WARNING: find_mtimes: could not find mtime_beg ',mtime_beg
  endif
  state.ifile_beg = ifile_beg
  state.ifile_end = ifile_end
  *stateptr = state
end
;-----------------------------------------------------------------------
pro update_select_fields,stateptr
;
; Update text widget displaying currently selected fields:
;
; state = *stateptr
; fields = *state.fields

; select_fields_list = widget_info(state.tlb,find_by_uname='SELECT_TGCM_FIELDS')
; indices = widget_info(select_fields_list,/list_select)
; nfields = n_elements(indices)
; *state.sfields = fields[indices]
; print,format=$
;   "('Selected TGCM fields: ',(10a8))",fields[indices].name

; *stateptr = state ; updates state.sfields, set above 
end
;-----------------------------------------------------------------------
function mtime_to_mins,mtime ; model time to integer minutes
return,long(mtime(0))*24*60+long(mtime(1))*60+long(mtime(2))
end
;-----------------------------------------------------------------------
pro select_files_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
  tryagain:
  files = dialog_pickfile(path=state.openpath,/multiple_files,filter='*.nc',$
    get_path=path,title='Please select one or more files')
  if (files[0] eq '') then return ; no files were selected (pickfile was cancelled)
;
; Update state w/ dir path and selected files:
;
  state.openpath = path
  *state.files = files
  *stateptr = state
;
; Destroy plot windows if they exist:
;
  if (state.utvert_tlb gt 0) then begin
    if (widget_info(state.utvert_tlb,/valid_id) gt 0) then $
      widget_control,state.utvert_tlb,/destroy
    state.utvert_tlb = -1
    *stateptr = state
  endif
  if (state.utlat_tlb gt 0) then begin
    if (widget_info(state.utlat_tlb,/valid_id) gt 0) then $
      widget_control,state.utlat_tlb,/destroy
    state.utlat_tlb = -1
    *stateptr = state
  endif
;
; Open files and read mtimes:
;
  widget_control,/hourglass
  open_files,stateptr            ; open_files.pro
  state = *stateptr
  if (state.ntimes le 1) then begin
    print,format="('>>> ntimes=',i4,' -- cannot make plots.')",$
      state.ntimes
;
    goto,tryagain
  endif
;
; Update text widget showing currently selected files and 
; their mtimes:
;
  files = *state.files
  nfiles = n_elements(files)
  text = strarr(nfiles+1)
  text[0] = 'Currently selected files:'
  mtime_files = *state.mtime_files
  ntime_files = *state.ntime_files
  for i=0,nfiles-1 do begin
    text[i+1] = file_basename(files[i])
    text[i+1] = text[i+1] + string(format=$
      "(' (',i3,',',i3,',',i3,' to ',i3,',',i3,',',i3,')')",$
      mtime_files[*,0,i],mtime_files[*,ntime_files[i]-1,i])
  endfor
  id = widget_info(event.top,find_by_uname='SHOW_FILES_TEXT')
  widget_control,id,set_value=text
;
; Populate mtime lists:
;
  ntimes = state.ntimes
  mtimes = *state.mtimes
  mtimes_strings = strarr(ntimes)
  for i=0,state.ntimes-1 do mtimes_strings[i] = $
    string(format="(i3,',',i3,',',i3)",mtimes[*,i])
  select_mtimes_id = widget_info(state.tlb,find_by_uname='SELECT_MTIMES_BASE')
  widget_control,select_mtimes_id,sensitive=1
  list_beg_mtime_id = widget_info(select_mtimes_id,find_by_uname='SELECT_BEG_MTIME')
  list_end_mtime_id = widget_info(select_mtimes_id,find_by_uname='SELECT_END_MTIME')
  widget_control,list_beg_mtime_id,set_value=mtimes_strings,$
    set_list_select=0
  widget_control,list_end_mtime_id,set_value=mtimes_strings,$
    set_list_select=ntimes-1
;
; Update text widgets giving current start/end times:
;
  textid = widget_info(event.top,find_by_uname='SELECT_BEG_TEXT')
  value = string(format=$
    "('Start Time (',i3,',',i3,',',i3,')')",state.mtime_beg)
  widget_control,textid,set_value=value

  textid = widget_info(event.top,find_by_uname='SELECT_END_TEXT')
  value = string(format=$
    "('End Time (',i3,',',i3,',',i3,')')",state.mtime_end)
  widget_control,textid,set_value=value
;
; Populate TGCM fields list (see open_files.pro):
;
  select_fields_id = widget_info(state.tlb,find_by_uname='SELECT_FIELDS_BASE')
  widget_control,select_fields_id,sensitive=1

  select_fields_list = widget_info(state.tlb,find_by_uname='SELECT_TGCM_FIELDS')
  fields = *state.fields_tgcm
  widget_control,select_fields_list,set_value=fields[*].name,set_list_select=0
  *state.sfields_tgcm = fields[0]
  *state.sfields = fields[0]

  select_fields_list = widget_info(state.tlb,find_by_uname='SELECT_MSIS_FIELDS')
  fields = *state.fields_msis
  widget_control,select_fields_list,set_value=fields[*].name
  nullfield = replicate({field_struct},1)
  nullfield.name = ' '
  *state.sfields_msis = nullfield

  *stateptr = state
;
; Make base widgets sensitive:
;
  id = widget_info(event.top,find_by_uname='SHOW_FILES_BASE')
  widget_control,id,sensitive=1
  id = widget_info(event.top,find_by_uname='DEVICE_BASE')
  widget_control,id,sensitive=1

  utvert_id = widget_info(state.tlb,find_by_uname='UTVERT')
  if (utvert_id gt 0) then widget_control,utvert_id,sensitive=1
  utlat_id = widget_info(state.tlb,find_by_uname='UTLAT')
  if (utlat_id gt 0) then widget_control,utlat_id,sensitive=1
end ; pro select_files_event
;-----------------------------------------------------------------------
pro help_main_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
  display_help,state.tlb,'MAIN'
end
;-----------------------------------------------------------------------
pro select_mtime_event,event
  widget_control,event.top,get_uvalue=stateptr ; select_mtimes_base
  state = *stateptr
  mtimes = *state.mtimes
  widget_control,event.id,get_uvalue=widget
  case widget of
    'SELECT_BEG_MTIME': begin
      mtime_beg = mtimes[*,event.index]
      if (mtime_to_mins(mtime_beg) ge mtime_to_mins(state.mtime_end)) then begin
        print,'>>> Start time must be less than end time:'
        print,format=$
          "('Selected Start time = ',i3,',',i3,',',i3,' End time = ',i3,',',i3,',',i3)",$
          mtime_beg,state.mtime_end
        return
      endif
      state.mtime_beg = mtime_beg
      textid = widget_info(event.top,find_by_uname='SELECT_BEG_TEXT')
      value = string(format=$
        "('Start Time (',i3,',',i3,',',i3,')')",state.mtime_beg)
      widget_control,textid,set_value=value
      print,'Selected starting model time ',state.mtime_beg
    end
    'SELECT_END_MTIME': begin
      mtime_end = mtimes[*,event.index]
      if (mtime_to_mins(mtime_end) le mtime_to_mins(state.mtime_beg)) then begin
        print,'>>> End time must be greater than start time:'
        print,format=$
          "('Selected End time = ',i3,',',i3,',',i3,' Begin time = ',i3,',',i3,',',i3)",$
          mtime_end,state.mtime_beg
        return
      endif
      state.mtime_end = mtimes[*,event.index]
      textid = widget_info(event.top,find_by_uname='SELECT_END_TEXT')
      value = string(format=$
        "('End Time (',i3,',',i3,',',i3,')')",state.mtime_end)
      widget_control,textid,set_value=value
      print,'Selected ending model time ',state.mtime_end
    end
  endcase
  *stateptr = state
  find_mtimes,stateptr
end
;-----------------------------------------------------------------------
pro select_fields_event,event
  widget_control,event.top,get_uvalue=stateptr ; select_fields_base
  state = *stateptr
  widget_control,event.id,get_uvalue=uvalue ; one of the models
  indices = widget_info(event.id,/list_select)
; print,'select_fields_event: uvalue=',uvalue,' indices=',indices
  case uvalue of
    'SELECT_TGCM_FIELDS': begin
      fields_tgcm = *state.fields_tgcm
      sfields_tgcm = fields_tgcm[indices]
      *state.sfields_tgcm = sfields_tgcm
      print,'Selected TGCM fields ',sfields_tgcm.name
    end
    'SELECT_MSIS_FIELDS': begin
      fields_msis = *state.fields_msis
      sfields_msis = fields_msis[indices]
      *state.sfields_msis = sfields_msis
      print,'Selected MSIS fields ',sfields_msis.name

    end
    'SELECT_HWM_FIELDS': begin

    end
    'SELECT_IRI_FIELDS': begin

    end
  endcase
;
  ntgcm = 0
  sfields_tgcm = *state.sfields_tgcm
  if (sfields_tgcm[0].name ne ' ') then begin
    ntgcm = n_elements(sfields_tgcm)
    print,'ntgcm=',ntgcm
  endif
;
  nmsis = 0
  sfields_msis = *state.sfields_msis
  if (sfields_msis[0].name ne ' ') then nmsis = n_elements(sfields_msis)
;
  nhwm = 0
  sfields_hwm = *state.sfields_hwm
  if (sfields_hwm[0].name ne ' ') then nhwm = n_elements(sfields_hwm)
;
  niri = 0
  sfields_iri = *state.sfields_iri
  if (sfields_iri[0].name ne ' ') then niri = n_elements(sfields_iri)
;
  nsfields = ntgcm+nmsis+nhwm+niri
  sfields = replicate({field_struct},nsfields)
  i = 0
  if (ntgcm gt 0) then begin
    sfields[0:ntgcm-1] = sfields_tgcm
    i = i+ntgcm
  endif
  if (nmsis gt 0) then begin
    sfields[i:i+nmsis-1] = sfields_msis[0:nmsis-1]
    i = i+nmsis
  endif
  if (nhwm gt 0) then begin
    sfields[i:i+nhwm-1] = sfields_msis[0:nhwm-1]
    i = i+nhwm
  endif
  if (niri gt 0) then begin
    sfields[i:i+niri-1] = sfields_msis[0:niri-1]
    i = i+niri
  endif
  nsfields = n_elements(sfields)
  for i=0,nsfields-1 do print,'i=',i,' nsfields=',nsfields,' sfield=',sfields[i]
  *state.sfields = sfields
  *stateptr = state
end
;-----------------------------------------------------------------------
pro plot_utvert_event,event
  widget_control,event.top,get_uvalue=stateptr
  find_mtimes,stateptr
  state = *stateptr
;
; Check if ps file has changed (i.e., user typed new file name and
; did not hit return)
;
  open_ps_file,stateptr
  *stateptr = state
  mkutvert,stateptr
end
;-----------------------------------------------------------------------
pro plot_utlat_event,event
  widget_control,event.top,get_uvalue=stateptr
  find_mtimes,stateptr
  state = *stateptr
  open_ps_file,stateptr
  *stateptr = state
  mkutlat,stateptr
end
;-----------------------------------------------------------------------
pro select_device_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
  widget_control,event.id,get_uvalue=widget
  case widget of
    'X11_DEVICE': begin ; switch to X device
      if (event.select eq 0) then return
      state.device = 'X'
      id = widget_info(event.top,find_by_uname='SET_PS_BASE')
      widget_control,id,sensitive=0
      *stateptr = state
      set_output_device,stateptr
      print,'Directing graphics output to X display.'
    end 
    'POSTSCRIPT_DEVICE': begin ; switch to postscript device
      if (event.select eq 0) then return
      state.device = 'PS'
      *stateptr = state
      open_ps_file,stateptr
      id = widget_info(event.top,find_by_uname='SET_PS_BASE')
      widget_control,id,sensitive=1
      *stateptr = state
      set_plot,'PS'
      print,'Directing plots to ps file ',state.ps_outfile
    end 
    'SET_PS_OUTFILE': begin
      state.device = 'PS'
      *stateptr = state
      open_ps_file,stateptr
    end
  endcase
  *stateptr = state
end
;-----------------------------------------------------------------------
pro exit_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
; if (state.device eq 'PS') then begin
  if (!d.name eq 'PS') then begin
    device,/close_file
    print,format="(/,'Closed ps file ',a)",state.ps_outfile
  endif
  widget_control,event.top,/destroy
end
;-----------------------------------------------------------------------
pro utproc
;
; Main: GUI interface to utproc.pro
;
; State structure:
;
; initial_openpath = "/aim/d/tgcm/tiegcm1-82_yr02/py2s"
  initial_openpath = "/aim/d/tgcm/tiegcm1.9/imf_2002/pimf"
  cwd = ''
  cd,current=cwd 
  state = {                         $
    tlb:0,                          $ id of top level base
    utvert_tlb:0,                   $ id of top level base of utvert widgets
    utlat_tlb:0,                    $ id of top level base of utlat widgets
    openpath:initial_openpath,      $ initial directory path for pickfile
    files:ptr_new(/allocate_heap),  $ input files from pickfile
    ntimes:0,                       $ total times on all files
    mtimes:ptr_new(/allocate_heap), $ mtimes from all files (3,ntimes)
    mtime_files:ptr_new(/allocate_heap), $ mtimes on each file (3,mxtimes_file,nfiles)
    ntime_files:ptr_new(/allocate_heap), $ number of times on each file
    mtime_beg:[0,0,0],              $ starting mtime
    ifile_beg:0,                    $ index to file containing start mtime
    mtime_end:[0,0,0],              $ ending mtime
    ifile_end:0,                    $ index to file containing end mtime
;   fields:ptr_new(/allocate_heap), $ all fields
    sfields:ptr_new(/allocate_heap),$ selected fields
    lons:ptr_new(/allocate_heap),   $ grid longitudes
    lats:ptr_new(/allocate_heap),   $ grid latitudes
    levs:ptr_new(/allocate_heap),   $ midpoint levels
    ilevs:ptr_new(/allocate_heap),  $ interface levels
    device:'PS',                    $ output device ('PS' or 'X')
    ps_outfile:cwd+'/utproc.ps',    $ ps output file
    ps_confirm_open:1,              $ confirmation to open existing ps_outfile
    model:'TGCM',                   $ 'TGCM', 'MSIS', 'HWM', or 'IRI'
    fields_tgcm:ptr_new(/allocate_heap), $
    fields_msis:ptr_new(/allocate_heap), $
    fields_hwm:ptr_new(/allocate_heap),  $
    fields_iri:ptr_new(/allocate_heap),  $
    sfields_tgcm:ptr_new(/allocate_heap), $
    sfields_msis:ptr_new(/allocate_heap), $
    sfields_hwm:ptr_new(/allocate_heap),  $
    sfields_iri:ptr_new(/allocate_heap)   $
  }
  field = {field_struct,            $
    name:'',                        $ short name
    long_name:'',                   $ long name
    units:'',                       $ units
    model:''                        $ TGCM, MSIS, HWM, or IRI 
  }
;
; Top level base:
;
  title = "utproc"
  tlb = widget_base(row=6,title=title)
  state.tlb = tlb
;
; Button to select files:
;
  button = widget_button(tlb,value='Select TGCM File(s)',uvalue='Select_Files',$
    event_pro='select_files_event')
;
; Help button:
;
  button = widget_button(tlb,value='Help',uvalue=stateptr,$
    event_pro='help_main_event')
;
; Display currently selected files:
;
  files_base = widget_base(tlb,row=1,uvalue=stateptr,sensitive=0,frame=3,$
    uname='SHOW_FILES_BASE')
  text = widget_text(files_base,value='Currently selected files:',$
    uname='SHOW_FILES_TEXT',xsize=63,ysize=6,/scroll)
;
; Select start/end times:
;
  select_mtimes_base = widget_base(tlb,column=2,uvalue=stateptr,sensitive=0,$
    uname='SELECT_MTIMES_BASE',frame=0)
;
; Select start time:
;
  list_base = widget_base(select_mtimes_base,row=2,frame=3,$
    group_leader=select_mtimes_base)
  value = string(format=$
    "('Start Time (',i3,',',i3,',',i3,')')",state.mtime_beg)
  label = widget_label(list_base,value=value,group_leader=select_mtimes_base,$
    uname='SELECT_BEG_TEXT')
  list = widget_list(list_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_BEG_MTIME',event_pro='select_mtime_event',$
    uname ='SELECT_BEG_MTIME',group_leader=select_mtimes_base)
;
; Select end time:
;
  list_base = widget_base(select_mtimes_base,row=2,frame=3,$
    group_leader=select_mtimes_base)
  value = string(format=$
    "('End Time (',i3,',',i3,',',i3,')')",state.mtime_end)
  label = widget_label(list_base,value=value,group_leader=select_mtimes_base,$
    uname='SELECT_END_TEXT')
  list = widget_list(list_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_END_MTIME',event_pro='select_mtime_event',$
    uname ='SELECT_END_MTIME',group_leader=select_mtimes_base)
;
; Select fields:
;
  fields_base = widget_base(tlb,column=4,frame=3,sensitive=0,$
    uname='SELECT_FIELDS_BASE',uvalue=stateptr)
;
; Select TGCM fields:
;
  label = widget_label(fields_base,value='TGCM fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_TGCM_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_TGCM_FIELDS',group_leader=fields_base,/multiple)
;
; Select MSIS fields:
;
  label = widget_label(fields_base,value='MSIS fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_MSIS_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_MSIS_FIELDS',group_leader=fields_base,/multiple)
  msis_names = ['TN','O2','O1']
  nfields_msis = n_elements(msis_names)
  fields_msis = replicate({field_struct},nfields_msis)
  fields_msis[*].name = msis_names
  fields_msis[*].model = 'MSIS'
  widget_control,list,set_value=fields_msis[*].name
  *state.fields_msis = fields_msis 
  nullfield = replicate({field_struct},1)
  nullfield.name = ' '
  *state.sfields_msis = nullfield
;
; Select HWM fields:
;
  label = widget_label(fields_base,value='HWM fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_HWM_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_HWM_FIELDS',group_leader=fields_base,/multiple)
  *state.sfields_hwm = nullfield
;
; Select IRI fields:
;
  label = widget_label(fields_base,value='IRI fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_IRI_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_IRI_FIELDS',group_leader=fields_base,/multiple)
  *state.sfields_iri = nullfield
;
; Select output device (X11, ps, jpeg, etc):
;
  device_base = widget_base(tlb,row=2,frame=3,uname='DEVICE_BASE',sensitive=0)
  label = widget_label(device_base,value='Output Device:',$
    group_leader=device_base)
  radio_base = widget_base(device_base,row=1,/exclusive)
  x11_button = widget_button(radio_base,value='X11 Display',$
    uvalue='X11_DEVICE',uname='SET_X11_DEVICE',$
    event_pro='select_device_event')
  ps_button = widget_button(radio_base,value='Postscript',$
    uvalue='POSTSCRIPT_DEVICE',uname='SET_PS_DEVICE',$
    event_pro='select_device_event')
  widget_control,ps_button,/set_button

  file_base = widget_base(device_base,row=1,uname='SET_PS_BASE')
  label = widget_label(file_base,value='PS output file:')
  text = widget_text(file_base,value=state.ps_outfile+'            ',$
    /editable,event_pro='select_device_event',uvalue='SET_PS_OUTFILE',$
    uname='SET_PS_OUTFILE')
;
; Plot types. These buttons will spawn new widget hierarchies
; independent of the main window.
;
  pltbase = widget_base(tlb,row=1)
  button = widget_button(pltbase,value='Plot ut vs pressure',uvalue=stateptr,$
    event_pro='plot_utvert_event',uname='UTVERT',sensitive=0)
  button = widget_button(pltbase,value='Plot ut vs latitude',uvalue=stateptr,$
    event_pro='plot_utlat_event',uname='UTLAT',sensitive=0)
  button = widget_button(pltbase,value='Exit Utproc',uvalue='Exit',$
    event_pro='exit_event')
;
; Realize top level base:
;
  widget_control,tlb,/realize
  stateptr = ptr_new(state)
  set_output_device,stateptr ; this will truncate any pre-existing utproc.ps
  widget_control,tlb,set_uvalue=stateptr
  xmanager,'utproc',tlb,/no_block
end
