  pro get_ap, begdate, enddate, data, t, fix_missing=fix_flag
;
; Returns a time-series of the daily Ap geomagnetic activity indices.
; If the keyword parameter fix_missing is supplied and is non-zero,
; then missing data will be replaced by linear interpolates.
;
; B. Knapp, 1987-12-09
;
; Change log
;
;    1995-08-11  B. Knapp, Make portable to Unix platforms
;    1996-02-18  B. Knapp, Change name from ap to get_ap
;    1996-12-05  B. Knapp, Change sunos paths
;    1998-06-09  B. Knapp, IDL v. 5 compliance
;    1999-07-26  T. Rood,  Change data_dir path and add /f77 parameter to
;                          ap data file open.
;    1999-10-28  T. Rood,  Read new IDL-generated ap.dat data file.
;    2000-01-13  T. Rood,  Now use the  $solndx_data environment variable
;                          instead of a hard coded directory to define
;                          data_dir.
;    B. Knapp, 2007-05-02  Linux port (use getenv(), path_sep())
;
; Input:
;    begdate, enddate -- dates of the form yyyyddd (integer
;                        variables or contstants)
;    fix_flag -- flag to request replacement of missing data
;    
; Output:
;    data -- floating-point array of length dyd(begdate,enddate)+1
;    t    -- floating-point array giving the times (in years since
;            1900 Jan 0) corresponding to each element of data.
;

; Host-dependent parameters
  sep = path_sep()
; data_dir = getenv('solndx_data')+sep
; data_dir = '.'+sep
  data_dir = 'msis'+sep
;
; Print usage?
  if n_params() eq 0 then begin
     print,' '
     print,' GET_AP returns (in the output variable data) a time-series'
     print,' of the Fredericksburg daily Ap geomagnetic activity index'
     print,' for a specified time period.  The input parameters begdate'
     print,' and enddate must be "longdates" of the form yyyyddd in the'
     print,' range 1932001 to the present.  If the keyword parameter'
     print,' fix_missing is supplied and is non-zero, then missing data'
     print,' will be replaced by linear interpolates.'
     print,' '
     print,' Usage: '
     print,' '
     print,'     get_ap, begdate, enddate, data, t, /fix_missing'
     print,' '
     return
  endif

; Open the data file and retrieve the entire Ap data set
  common get_ap_save, day0, n, entire, last
  if n_elements( day0 ) eq 0 then begin
     get_lun, lun
     openr, lun, data_dir+'ap.dat'
     n = (fstat(lun)).size / 2
     entire = intarr( n )
     readu, lun, entire
     close, lun
     free_lun, lun
     last = max( where( entire gt 0 ) )
     day0 = long( ymd2yd( 1932, 1, 1 ) )
  endif

; Extract the requested time-series
  j0 = dyd( day0, begdate )
  j1 = dyd( day0, enddate ) < last
  if j0 lt 0 or j0 gt n-1 or j1 lt 0 or j1 gt n-1 then begin
     message,' begdate, enddate must be in range 1932001-present',/info
     return
  endif
  data = entire[j0:j1]
  x = findgen( n_elements( data ) )
  if keyword_set( fix_flag ) then data = replace( x, data, 0 )
  t = 1900.d0 + (x+dyd( 1900000L, begdate ))/365.25
  return

  end
