;-----------------------------------------------------------------------
function make_default_psfile
  cwd = ''
  cd,'.',current=cwd 
  default_psfile = cwd+'/idl.ps'    ; IDL default file
  return,default_psfile 
end
;-----------------------------------------------------------------------
function nullfield
  field = replicate({field_struct},1)
  field.name = ' '
  field.long_name = ' '
  field.units = ' '
  field.model = ' '
  return,field 
end
;-----------------------------------------------------------------------
function isnullfield,field
  is = 0
  if (n_elements(field) eq 1) then begin
    if (field.name  eq ' ' and field.long_name eq ' ' and $
        field.units eq ' ' and field.model eq ' ') then is = 1
    return,is
  endif
end
;-----------------------------------------------------------------------
pro bases_sensitivity,stateptr,sensitive
  state = *stateptr
  widget_control,state.tlb,sensitive=sensitive
  if (widget_info(state.utvert_tlb,/valid_id) gt 0) then $
    widget_control,state.utvert_tlb,sensitive=sensitive
  if (widget_info(state.utlat_tlb,/valid_id) gt 0) then $
    widget_control,state.utlat_tlb,sensitive=sensitive
end
;-----------------------------------------------------------------------
function ixfind, z,val,del
;
; Find nearest index to val in z, with increment del
; If not found, return -1
;
nz = n_elements(z)
if del gt 0 then begin
  for i=0,nz-1 do begin
    if (val ge z(i)-0.5*del and val le z(i)+0.5*del) then return,i
  endfor
endif else begin
  for i=0,nz-1 do begin
    if (val eq z(i)) then return,i
  endfor
endelse
return,-1
end
;-----------------------------------------------------------------------
function ixfind_simple,a,v
  na = n_elements(a)
  for k=0,na-1 do begin
    if (v eq a[k]) then return,k
  endfor
  return,-1
end
;-----------------------------------------------------------------------
pro find_mtimes,stateptr
;
; Given state.mtime_beg, find state.ifile_beg and state.itime_file_beg,
; and same for mtime_end.
;
  state = *stateptr
  mtime_beg = state.mtime_beg
  mtime_end = state.mtime_end
  files = *state.files
  nfiles = n_elements(files)
;
; Search for start time (state.mtime_beg):
;
  for ifile=0,nfiles-1 do begin
    ncid = ncdf_open(files[ifile],/nowrite)
    id = ncdf_varid(ncid,'mtime')
    ncdf_varget,ncid,id,mtime
    ntime = n_elements(mtime)/3
    found = 0
    for i=0,ntime-1 do begin
      if (array_equal(mtime[*,i],state.mtime_beg)) then begin
        found = 1
        itime_beg = i
        ifile_beg = ifile
        break
      endif
    endfor
    if (found eq 0) then continue
    print,format="('Found start time ',3i4,' ifile=',i3,' itime_beg=',i3)",$
      state.mtime_beg,ifile_beg,itime_beg
    ncdf_close,ncid
    break
  endfor
  if (found eq 0) then begin
    print,'>>> WARNING: find_mtimes: could not find mtime_beg ',mtime_beg
  endif
;
; Search for end time (state.mtime_end):
;
  for ifile=0,nfiles-1 do begin
    ncid = ncdf_open(files[ifile],/nowrite)
    id = ncdf_varid(ncid,'mtime')
    ncdf_varget,ncid,id,mtime
    ntime = n_elements(mtime)/3
    found = 0
    for i=0,ntime-1 do begin
      if (array_equal(mtime[*,i],state.mtime_end)) then begin
        found = 1
        itime_end = i
        ifile_end = ifile
        break
      endif
    endfor
    if (found eq 0) then continue
    print,format="('Found end time ',3i4,' ifile=',i3,' itime_end=',i3)",$
      state.mtime_end,ifile_end,itime_end
    ncdf_close,ncid
    break
  endfor
  if (found eq 0) then begin
    print,'>>> WARNING: find_mtimes: could not find mtime_beg ',mtime_beg
  endif
  state.ifile_beg = ifile_beg
  state.ifile_end = ifile_end
  *stateptr = state
end ; pro find_mtimes
;-----------------------------------------------------------------------
pro update_select_fields,stateptr
;
; Update text widget displaying currently selected fields:
;
; state = *stateptr
; fields = *state.fields

; select_fields_list = widget_info(state.tlb,find_by_uname='SELECT_TGCM_FIELDS')
; indices = widget_info(select_fields_list,/list_select)
; nfields = n_elements(indices)
; *state.sfields = fields[indices]
; print,format=$
;   "('Selected TGCM fields: ',(10a8))",fields[indices].name

; *stateptr = state ; updates state.sfields, set above 
end
;-----------------------------------------------------------------------
function mtime_to_mins,mtime ; model time to integer minutes
return,long(mtime(0))*24*60+long(mtime(1))*60+long(mtime(2))
end
;-----------------------------------------------------------------------
pro select_files_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
  tryagain:
  files = dialog_pickfile(path=state.openpath,/multiple_files,filter='*.nc',$
    get_path=path,title='Please select one or more files')
  if (files[0] eq '') then return ; no files were selected (pickfile was cancelled)
;
; Update state w/ dir path and selected files:
;
  state.openpath = path
  *state.files = files
  *stateptr = state
;
; Destroy plot windows if they exist:
;
  if (state.utvert_tlb gt 0) then begin
    if (widget_info(state.utvert_tlb,/valid_id) gt 0) then $
      widget_control,state.utvert_tlb,/destroy
    state.utvert_tlb = -1
    *stateptr = state
  endif
  if (state.utlat_tlb gt 0) then begin
    if (widget_info(state.utlat_tlb,/valid_id) gt 0) then $
      widget_control,state.utlat_tlb,/destroy
    state.utlat_tlb = -1
    *stateptr = state
  endif
;
; Open files and read mtimes:
;
  widget_control,/hourglass
  open_files,stateptr            ; open_files.pro
  state = *stateptr
  if (state.ntimes le 1) then begin
    print,format="('>>> ntimes=',i4,' -- cannot make plots.')",$
      state.ntimes
;
    goto,tryagain
  endif
;
; Update text widget showing currently selected files and 
; their mtimes:
;
  files = *state.files
  nfiles = n_elements(files)
  text = strarr(nfiles+1)
  text[0] = 'Currently selected files:'
  mtime_files = *state.mtime_files
  ntime_files = *state.ntime_files
  for i=0,nfiles-1 do begin
    text[i+1] = file_basename(files[i])
    text[i+1] = text[i+1] + string(format=$
      "(' (',i3,',',i3,',',i3,' to ',i3,',',i3,',',i3,')')",$
      mtime_files[*,0,i],mtime_files[*,ntime_files[i]-1,i])
  endfor
  id = widget_info(event.top,find_by_uname='SHOW_FILES_TEXT')
  widget_control,id,set_value=text
;
; Populate mtime lists:
;
  ntimes = state.ntimes
  mtimes = *state.mtimes
  mtimes_strings = strarr(ntimes)
  for i=0,state.ntimes-1 do mtimes_strings[i] = $
    string(format="(i3,',',i3,',',i3)",mtimes[*,i])
  select_mtimes_id = widget_info(state.tlb,find_by_uname='SELECT_MTIMES_BASE')
  widget_control,select_mtimes_id,sensitive=1
  list_beg_mtime_id = widget_info(select_mtimes_id,find_by_uname='SELECT_BEG_MTIME')
  list_end_mtime_id = widget_info(select_mtimes_id,find_by_uname='SELECT_END_MTIME')
  widget_control,list_beg_mtime_id,set_value=mtimes_strings,$
    set_list_select=0
  widget_control,list_end_mtime_id,set_value=mtimes_strings,$
    set_list_select=ntimes-1
;
; Update text widgets giving current start/end times:
;
  textid = widget_info(event.top,find_by_uname='SELECT_BEG_TEXT')
  value = string(format=$
    "('Start Time (',i3,',',i3,',',i3,')')",state.mtime_beg)
  widget_control,textid,set_value=value

  textid = widget_info(event.top,find_by_uname='SELECT_END_TEXT')
  value = string(format=$
    "('End Time (',i3,',',i3,',',i3,')')",state.mtime_end)
  widget_control,textid,set_value=value
;
; Populate TGCM fields list (see open_files.pro):
;
  select_fields_id = widget_info(state.tlb,find_by_uname='SELECT_FIELDS_BASE')
  widget_control,select_fields_id,sensitive=1

  select_fields_list = widget_info(state.tlb,find_by_uname='SELECT_TGCM_FIELDS')
  fields = *state.fields_tgcm
  widget_control,select_fields_list,set_value=fields[*].name,set_list_select=0
  *state.sfields_tgcm = fields[0]
  *state.sfields = fields[0]

  select_fields_list = widget_info(state.tlb,find_by_uname='SELECT_MSIS_FIELDS')
  fields = *state.fields_msis
  widget_control,select_fields_list,set_value=fields[*].name
  *state.sfields_msis = nullfield()
  *stateptr = state
;
; Make base widgets sensitive:
;
  id = widget_info(event.top,find_by_uname='SHOW_FILES_BASE')
  widget_control,id,sensitive=1
  id = widget_info(event.top,find_by_uname='OUTPUT_BASE')
  widget_control,id,sensitive=1

  id = widget_info(event.top,find_by_uname='OPTIONS_TABLE_BUTTON')
  widget_control,id,sensitive=1

  utvert_id = widget_info(state.tlb,find_by_uname='UTVERT')
  if (utvert_id gt 0) then widget_control,utvert_id,sensitive=1
  utlat_id = widget_info(state.tlb,find_by_uname='UTLAT')
  if (utlat_id gt 0) then widget_control,utlat_id,sensitive=1
;
; Open the ps file:
  open_psfile,stateptr
end ; pro select_files_event
;-----------------------------------------------------------------------
pro options_table_event,event
  widget_control,event.top,get_uvalue=stateptr
  options_table,stateptr
end
;-----------------------------------------------------------------------
pro help_main_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
  display_help,state.tlb,'MAIN'
end
;-----------------------------------------------------------------------
pro select_mtime_event,event
  widget_control,event.top,get_uvalue=stateptr ; select_mtimes_base
  state = *stateptr
  mtimes = *state.mtimes
  widget_control,event.id,get_uvalue=widget
  case widget of
    'SELECT_BEG_MTIME': begin
      mtime_beg = mtimes[*,event.index]
      if (mtime_to_mins(mtime_beg) ge mtime_to_mins(state.mtime_end)) then begin
        print,'>>> Start time must be less than end time:'
        print,format=$
          "('Selected Start time = ',i3,',',i3,',',i3,' End time = ',i3,',',i3,',',i3)",$
          mtime_beg,state.mtime_end
        return
      endif
      state.mtime_beg = mtime_beg
      textid = widget_info(event.top,find_by_uname='SELECT_BEG_TEXT')
      value = string(format=$
        "('Start Time (',i3,',',i3,',',i3,')')",state.mtime_beg)
      widget_control,textid,set_value=value
      print,'Selected starting model time ',state.mtime_beg
    end
    'SELECT_END_MTIME': begin
      mtime_end = mtimes[*,event.index]
      if (mtime_to_mins(mtime_end) le mtime_to_mins(state.mtime_beg)) then begin
        print,'>>> End time must be greater than start time:'
        print,format=$
          "('Selected End time = ',i3,',',i3,',',i3,' Begin time = ',i3,',',i3,',',i3)",$
          mtime_end,state.mtime_beg
        return
      endif
      state.mtime_end = mtimes[*,event.index]
      textid = widget_info(event.top,find_by_uname='SELECT_END_TEXT')
      value = string(format=$
        "('End Time (',i3,',',i3,',',i3,')')",state.mtime_end)
      widget_control,textid,set_value=value
      print,'Selected ending model time ',state.mtime_end
    end
  endcase
  *stateptr = state
  find_mtimes,stateptr
end
;-----------------------------------------------------------------------
;pro assemble_select_fields,stateptr
; state = *stateptr
; tgcm_list = widget_info(state.tlb,find_by_uname='SELECT_TGCM_FIELDS')
; msis_list = widget_info(state.tlb,find_by_uname='SELECT_MSIS_FIELDS')
; hwm_list = widget_info(state.tlb,find_by_uname='SELECT_HWM_FIELDS')
; iri_list = widget_info(state.tlb,find_by_uname='SELECT_IRI_FIELDS')
;
; tgcm_indices = widget_info(tgcm_list,/list_select)

; print,'assemble_select_fields: #indices=',n_elements(tgcm_indices),$
;   ' tgcm_indices=',tgcm_indices

; ntgcm = 0
; if tgcm_indices[0] ne -1 then begin
;   ntgcm = n_elements(tgcm_indices)
;   fields_tgcm = *state.fields_tgcm
;   sfields_tgcm = fields_tgcm[tgcm_indices]

;   for i=0,n_elements(tgcm_indices)-1 do begin  
;     print,'Enter assemble_select_fields: tgcm field ',sfields_tgcm[i].name,$
;       ' cmin,max,int=',sfields_tgcm[i].cmin,sfields_tgcm[i].cmax,$
;       sfields_tgcm[i].cint
;   endfor

; endif
;
; msis_indices = widget_info(msis_list,/list_select)
; nmsis = 0
; if msis_indices[0] ne -1 then begin
;   nmsis = n_elements(msis_indices)
;   fields_msis = *state.fields_msis
;   sfields_msis = fields_msis[msis_indices]
; endif
;  
; hwm_indices = widget_info(hwm_list,/list_select)
; nhwm = 0
; if hwm_indices[0] ne -1 then begin
;   nhwm = n_elements(hwm_indices)
;   fields_hwm = *state.fields_hwm
;   sfields_hwm = fields_hwm[hwm_indices]
; endif
;  
; iri_indices = widget_info(iri_list,/list_select)
; niri = 0
; if iri_indices[0] ne -1 then begin
;   niri = n_elements(iri_indices)
;   fields_iri = *state.fields_iri
;   sfields_iri = fields_iri[iri_indices]
; endif
;
; nsfields = ntgcm+nmsis+nhwm+niri
; if (nsfields eq 0) then begin
;   print,'>>> WARNING: no fields are selected: nsfields=',nsfields
;   print,'Please select at least one field to plot.'
;   *state.sfields = nullfield()
;   *stateptr = state
;   return
; endif

; sfields = replicate({field_struct},nsfields)
; i = 0
; if (ntgcm gt 0) then begin
;   sfields[0:ntgcm-1] = sfields_tgcm
;   i = i+ntgcm
; endif
; if (nmsis gt 0) then begin
;   sfields[i:i+nmsis-1] = sfields_msis[0:nmsis-1]
;   i = i+nmsis
; endif
; if (nhwm gt 0) then begin
;   sfields[i:i+nhwm-1] = sfields_msis[0:nhwm-1]
;   i = i+nhwm
; endif
; if (niri gt 0) then begin
;   sfields[i:i+niri-1] = sfields_msis[0:niri-1]
;   i = i+niri
; endif

; for i=0,nsfields-1 do print,'end assemble_select_fields: i=',i,' nsfields=',$
;   nsfields,' sfield cmin,cmax,cint=',sfields[i].cmin,sfields[i].cmax,$
;   sfields[i].cint

; *state.sfields = sfields
; *stateptr = state
;
; Update options table, if open:
;
; if (state.opts_tlb ne 0) then begin
;   widget_control,state.opts_tlb,/destroy
;   options_table,stateptr
; endif
;end
;-----------------------------------------------------------------------
pro select_fields_event,event
  widget_control,event.top,get_uvalue=stateptr ; select_fields_base
  state = *stateptr
  widget_control,event.id,get_uvalue=uvalue ; one of the models
  indices = widget_info(event.id,/list_select)

  fields_tgcm = *state.fields_tgcm

    print,'select_fields_event: TN cmin,max,int=',$
      fields_tgcm[0].cmin,fields_tgcm[0].cmax,$
      fields_tgcm[0].cint

  case uvalue of
    'SELECT_TGCM_FIELDS': begin
      if (indices[0] ne -1) then begin
        fields_tgcm = *state.fields_tgcm
        sfields_tgcm = fields_tgcm[indices]
        *state.sfields_tgcm = sfields_tgcm
        print,'Selected TGCM fields ',sfields_tgcm.name
      endif else begin
        sfields_tgcm = nullfield()
      endelse
    end
    'SELECT_MSIS_FIELDS': begin
      if (indices[0] ne -1) then begin
        fields_msis = *state.fields_msis
        sfields_msis = fields_msis[indices]
        *state.sfields_msis = sfields_msis
        print,'Selected MSIS fields ',sfields_msis.name
      endif else begin
        sfields_msis = nullfield()
      endelse
    end
    'SELECT_HWM_FIELDS': begin

    end
    'SELECT_IRI_FIELDS': begin

    end
  endcase
  assemble_select_fields,stateptr
end
;-----------------------------------------------------------------------
pro open_psfile,stateptr
  set_plot,'PS'
  state = *stateptr
  filewidget = widget_info(state.tlb,find_by_uname='SET_PSFILE')
  widget_control,filewidget,get_value=file
  ps_outfile = strcompress(file,/remove_all)
  if (strlen(ps_outfile) le 0) then begin
    print,'>>> Cannot open NULL file. Please enter PS file name'
    return
  endif
  fileinfo = file_info(ps_outfile)
  openfile = 0
  default_psfile = make_default_psfile()
  if (not fileinfo.exists or (fileinfo.exists and fileinfo.size eq 0)) then begin
    openfile = 1  
  endif else if (fileinfo.exists and fileinfo.size gt 0) then begin
    file_truncate_query,stateptr,ps_outfile
    state = *stateptr
    if (state.ps_confirm_open gt 0) then openfile = 1
  endif 
  if (openfile eq 0) then print,'Did NOT open file ',ps_outfile,$
    ' ps output will continue to go to this file'
;
; Close any pre-existing ps file before opening new file:
;
  if (openfile) then begin
    device,/close_file ; close any existing file
    state = *stateptr
;   device,/color,/landscape,/inches,xsize=13.,ysize=8.,$ ; 13" wide
;     filename=ps_outfile
    device,/color,/landscape,/inches,xsize=10.,ysize=7.,$
      filename=ps_outfile
    print,'Opened ps file ',ps_outfile
    state.ps_outfile = ps_outfile
    *stateptr = state
  endif
end ; pro open_psfile
;-----------------------------------------------------------------------
pro close_psfile,stateptr
  state = *stateptr
  device,/close_file
  print,'Closed ps output file ',state.ps_outfile
; default_psfile = make_default_psfile()
; filewidget = widget_info(state.tlb,find_by_uname='SET_PSFILE')
; widget_control,filewidget,set_value=default_psfile
; state.ps_outfile = ' '
  *stateptr = state
end
;-----------------------------------------------------------------------
pro openclose_psfile_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
  widget_control,event.id,get_uvalue=widget
  case widget of
    'SET_PSFILE': begin
      open_psfile,stateptr
    end
    'OPEN_PSFILE': begin
      open_psfile,stateptr
    end
    'CLOSE_PSFILE': begin
      close_psfile,stateptr
    end
  endcase
end
;-----------------------------------------------------------------------
pro plot_utvert_event,event
  widget_control,event.top,get_uvalue=stateptr
  find_mtimes,stateptr
  state = *stateptr
  sfields = *state.sfields
  if (isnullfield(sfields)) then begin
    print,'>>> No fields selected.'
    return
  endif
  open_psfile,stateptr
  *stateptr = state
  mkutvert,stateptr
end
;-----------------------------------------------------------------------
pro plot_utlat_event,event
  widget_control,event.top,get_uvalue=stateptr
  find_mtimes,stateptr
  state = *stateptr
  sfields = *state.sfields
  if (isnullfield(sfields)) then begin
    print,'>>> No fields selected.'
    return
  endif
  open_psfile,stateptr
  *stateptr = state
  mkutlat,stateptr
end
;-----------------------------------------------------------------------
pro exit_event,event
  widget_control,event.top,get_uvalue=stateptr
  state = *stateptr
  if (!d.name eq 'PS') then begin
    device,/close_file
    print,format="(/,'Closed ps file ',a)",state.ps_outfile
  endif
  widget_control,event.top,/destroy
end
;-----------------------------------------------------------------------
pro utproc
;
; Main: GUI interface to utproc.pro
;
; State structure:
;
; initial_openpath = "/aim/d/tgcm/tiegcm1-82_yr02/py2s"
; initial_openpath = "/aim/d/tgcm/tiegcm1.9/imf_2002/pimf"
; initial_openpath = "/oreo/d/foster/tiegcm1.9"
; initial_openpath = "/aim/d/tgcm/timegcm_ssw08"
; initial_openpath = "/aim/d/tgcm/timegcm_ssw08/dec07-feb08_daily"
; initial_openpath = "/aim/d/tgcm/tiegcm1.92_nov04/imf_dres"
  initial_openpath = "."
  cwd = ''
  cd,'.',current=cwd
  default_psfile = cwd+'/utproc.ps'
  state = {                         $
    tlb:0,                          $ id of top level base
    utvert_tlb:0,                   $ id of top level base of utvert widgets
    utlat_tlb:0,                    $ id of top level base of utlat widgets
    opts_tlb:0,                     $ id of top level base of options table
    openpath:initial_openpath,      $ initial directory path for pickfile
    files:ptr_new(/allocate_heap),  $ input files from pickfile
    ntimes:0,                       $ total times on all files
    mtimes:ptr_new(/allocate_heap), $ mtimes from all files (3,ntimes)
    mtime_files:ptr_new(/allocate_heap), $ mtimes on each file (3,mxtimes_file,nfiles)
    ntime_files:ptr_new(/allocate_heap), $ number of times on each file
    mtime_beg:[0,0,0],              $ starting mtime
    ifile_beg:0,                    $ index to file containing start mtime
    mtime_end:[0,0,0],              $ ending mtime
    ifile_end:0,                    $ index to file containing end mtime
    lons:ptr_new(/allocate_heap),   $ grid longitudes
    lats:ptr_new(/allocate_heap),   $ grid latitudes
    levs:ptr_new(/allocate_heap),   $ midpoint levels
    ilevs:ptr_new(/allocate_heap),  $ interface levels
    horiz_resolution:0.,            $ 5.0 or 2.5  (single,double)
    vert_resolution:0.,             $ 0.5 or 0.25 (single,double)
    ps_outfile:default_psfile,      $ ps output file
    ps_confirm_open:0,              $ confirmation to open and truncate psfile
    label_contours:0,               $ whether or not to label contours
    model:'TGCM',                   $ 'TGCM', 'MSIS', 'HWM', or 'IRI'
    sfields:ptr_new(/allocate_heap),$ selected fields (all models)
    fields_tgcm:ptr_new(/allocate_heap), $
    fields_msis:ptr_new(/allocate_heap), $
    fields_hwm:ptr_new(/allocate_heap),  $
    fields_iri:ptr_new(/allocate_heap),  $
    sfields_tgcm:ptr_new(/allocate_heap), $
    sfields_msis:ptr_new(/allocate_heap), $
    sfields_hwm:ptr_new(/allocate_heap),  $
    sfields_iri:ptr_new(/allocate_heap),  $
    outfile : 'utproc.out'          $ File to receive myprint calls
  }
  field = {field_struct,            $
    name:'',                        $ short name
    long_name:'',                   $ long name
    units:'',                       $ units
    model:'',                       $ TGCM, MSIS, HWM, or IRI 
    density:0,                      $ 1 if a density species
    log10:0,                        $ take log10 of field prior to plotting
    cmin:0.,cmax:0.,cint:0.         $ contour min,max,interval
  }
;
; Top level base:
;
  title = "utproc"
  tlb = widget_base(row=6,title=title)
  state.tlb = tlb
;
; Button to select files:
;
  top_row = widget_base(tlb,column=3)
  button = widget_button(top_row,value='Select TGCM File(s)',uvalue='Select_Files',$
    event_pro='select_files_event')
;
; Options table:
;
  button = widget_button(top_row,value='Options Table',uvalue='Options_Table',$
    event_pro='options_table_event',sensitive=0,uname='OPTIONS_TABLE_BUTTON')
;
; Help button:
;
  button = widget_button(top_row,value='Help',uvalue=stateptr,$
    event_pro='help_main_event')
;
; Display currently selected files:
;
  files_base = widget_base(tlb,row=1,uvalue=stateptr,sensitive=0,frame=3,$
    uname='SHOW_FILES_BASE')
  text = widget_text(files_base,value='Currently selected files:',$
    uname='SHOW_FILES_TEXT',xsize=63,ysize=6,/scroll)
;
; Select start/end times:
;
  select_mtimes_base = widget_base(tlb,column=2,uvalue=stateptr,sensitive=0,$
    uname='SELECT_MTIMES_BASE',frame=0)
;
; Select start time:
;
  list_base = widget_base(select_mtimes_base,row=2,frame=3,$
    group_leader=select_mtimes_base)
  value = string(format=$
    "('Start Time (',i3,',',i3,',',i3,')')",state.mtime_beg)
  label = widget_label(list_base,value=value,group_leader=select_mtimes_base,$
    uname='SELECT_BEG_TEXT')
  list = widget_list(list_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_BEG_MTIME',event_pro='select_mtime_event',$
    uname ='SELECT_BEG_MTIME',group_leader=select_mtimes_base)
;
; Select end time:
;
  list_base = widget_base(select_mtimes_base,row=2,frame=3,$
    group_leader=select_mtimes_base)
  value = string(format=$
    "('End Time (',i3,',',i3,',',i3,')')",state.mtime_end)
  label = widget_label(list_base,value=value,group_leader=select_mtimes_base,$
    uname='SELECT_END_TEXT')
  list = widget_list(list_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_END_MTIME',event_pro='select_mtime_event',$
    uname ='SELECT_END_MTIME',group_leader=select_mtimes_base)
;
; Select fields:
;
  fields_base = widget_base(tlb,column=4,frame=3,sensitive=0,$
    uname='SELECT_FIELDS_BASE',uvalue=stateptr)
;
; Select TGCM fields:
;
  label = widget_label(fields_base,value='TGCM fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_TGCM_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_TGCM_FIELDS',group_leader=fields_base,/multiple)
;
; Select MSIS fields:
;
  label = widget_label(fields_base,value='MSIS fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_MSIS_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_MSIS_FIELDS',group_leader=fields_base,/multiple)
  msis_names = ['TN','O2','O1','H','HE','N','N2','AR']
  nfields_msis = n_elements(msis_names)
  fields_msis = replicate({field_struct},nfields_msis)
  fields_msis[*].name = msis_names
  fields_msis[*].model = 'MSIS'
  fields_msis[1:7].density = 1
  widget_control,list,set_value=fields_msis[*].name
  *state.fields_msis = fields_msis 
  nullfield = replicate({field_struct},1)
  nullfield.name = ' '
  *state.sfields_msis = nullfield
;
; Select HWM fields:
;
  label = widget_label(fields_base,value='HWM fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_HWM_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_HWM_FIELDS',group_leader=fields_base,/multiple)
  *state.sfields_hwm = nullfield
;
; Select IRI fields:
;
  label = widget_label(fields_base,value='IRI fields',$
    group_leader=fields_base)
  list = widget_list(fields_base,value=' ',ysize=10,xsize=12,$
    uvalue='SELECT_IRI_FIELDS',event_pro='select_fields_event',$
    uname ='SELECT_IRI_FIELDS',group_leader=fields_base,/multiple)
  *state.sfields_iri = nullfield
;
; Set, open or close ps output file:
;
  output_base = widget_base(tlb,row=2,frame=3,uname='OUTPUT_BASE',sensitive=0)
  file_base = widget_base(output_base,row=1,uname='SET_PS_BASE')
  label = widget_label(file_base,value='PS Output File:')
  text = widget_text(file_base,value=state.ps_outfile+'            ',$
    /editable,event_pro='openclose_psfile_event',uvalue='SET_PSFILE',$
    uname='SET_PSFILE')

  openclose_base = widget_base(output_base,row=1)
  open_button = widget_button(openclose_base,value='Open PS File',$
    uvalue='OPEN_PSFILE',uname='OPEN_PSFILE',$
    event_pro='openclose_psfile_event')
  close_button = widget_button(openclose_base,value='Close PS File',$
    uvalue='CLOSE_PSFILE',uname='CLOSE_PSFILE',$
    event_pro='openclose_psfile_event')

;
; Plot types. These buttons will spawn new widget hierarchies
; independent of the main window.
;
  pltbase = widget_base(tlb,row=1)
  button = widget_button(pltbase,value='Plot ut vs pressure',uvalue=stateptr,$
    event_pro='plot_utvert_event',uname='UTVERT',sensitive=0)
  button = widget_button(pltbase,value='Plot ut vs latitude',uvalue=stateptr,$
    event_pro='plot_utlat_event',uname='UTLAT',sensitive=0)
  button = widget_button(pltbase,value='Exit Utproc',uvalue='Exit',$
    event_pro='exit_event')
;
; Open text file to receive myprint calls:
;
  openw,/get_lun,outfile_unit,state.outfile
  print,'Opened outfile ',state.outfile
;
; Realize top level base:
;
  widget_control,tlb,/realize
  stateptr = ptr_new(state)
; open_psfile,stateptr
  widget_control,tlb,set_uvalue=stateptr
  xmanager,'utproc',tlb,/no_block
end
