!
      subroutine getms(mspath,diskfile,tmpdir)
      implicit none
!
! Acquire mspath from ncar mass store to diskfile in cwd.
! If diskfile already exists in cwd, use it and do not go to mss, 
!   and save a link in tmpdir.
! If diskfile not in cwd, check in tmpdir. If it exists in tmpdir,
!   save a link in cwd.
! If not in cwd or tmpdir and mss is available (#ifdef MSS), get file
!   from mss to diskfile using dcs command msrcp. Save a link in tmpdir.
!
! On input:
!   mspath:   Mass store path including file name (required input).
!   diskfile: Disk file name (required input).
!   tmpdir:   Temporary directory. Must be on same file system as
!             cwd for links to work. If tmpdir is blank on input,
!             it will not be used (cwd and mss only).
! Method:
!   1. If diskfile already exists in cwd, link to tmpdir and return.
!   2. If diskfile in tmpdir, link to cwd and return.
!   3. Acquire mspath from mss as diskfile in cwd, link to tmpdir and 
!      return.
!
! Args:
      character(len=*),intent(in) :: mspath,diskfile,tmpdir
!
! Local:
      logical exists
      integer,parameter :: mxlen=1024
      character(len=mxlen) :: dcs_output,cmd,tmpfile
      integer :: istat,ier
!
! External (ilink, iunlink, and isystem are in util.F):
      integer,external :: ilink,iunlink,isystem
!
! Check args: mspath and diskfile must be non-blank: 
!
      write(6,"(' ',/72('-'),/'GETMS:')")
      if (len_trim(mspath) == 0) then
        write(6,"('>>> FATAL getms: mspath is blank.')")
        write(6,"(72('-'),/)")
        stop 'getms'
      endif
      if (len_trim(diskfile)==0) then
        write(6,"('>>> FATAL getms: diskfile is blank.')")
        write(6,"(72('-'),/)")
        stop 'getms'
      endif
!
! If tmpdir is non-blank and non-existant, make it:
!
      if (len_trim(tmpdir) > 0) then
        inquire(file=tmpdir,exist=exists)
        if (.not.exists) then
          cmd = ' '
          write(cmd,"('mkdir -p -m 777 ',a)") trim(tmpdir)
          write(6,"('getms: mkdir command: ',a)") trim(cmd)
          istat = isystem(trim(cmd))
          if (istat==0) then
            write(6,"('getms: Created tmpdir ',a)") trim(tmpdir)
          else
            write(6,"('>>> getms: error creating tmpdir ',a,
     |        ' istat=',i3)") trim(tmpdir),istat
            stop 'getms'
          endif
        endif ! tmpdir existence
!
! tmpfile = tmpdir/diskfile
        write(tmpfile,"(a,'/',a)") trim(tmpdir),trim(diskfile)
      endif
!
! Check for diskfile in cwd:
!
      inquire(file=diskfile,exist=exists)
      if (exists) then
        write(6,"('Found file ',a,' in cwd')") trim(diskfile)
!
! Save a link in tmpdir (remove any pre-existing file in tmpdir):
!
        if (len_trim(tmpdir) > 0) then
          istat = iunlink(tmpfile,0)
          istat = ilink(diskfile,tmpfile,1)
          if (istat /= 0) then
            write(6,"('>>> getms: error saving link of ',a,' in ',
     |        'tmpdir ',a)") trim(diskfile),trim(tmpdir)
          endif
        endif
        write(6,"(72('-'),/)")
        return
      endif ! diskfile exists in cwd
!
! File does not exist in cwd, so check in tmpdir. If it exists in tmpdir,
! save link to cwd and return. Note tmpfile was defined above. 
!
      if (len_trim(tmpdir) > 0) then
        inquire(file=tmpfile,exist=exists)
        if (exists) then
          istat = ilink(tmpfile,diskfile,1)
          if (istat /= 0) then
            write(6,"('>>> getms: error saving link of ',a,' in cwd ',
     |        'as ',a)") trim(tmpfile),trim(diskfile)
          endif
          write(6,"(72('-'),/)")
          return
        endif
      endif ! tmpdir non-blank
!
! File does not exist in cwd or tmpdir, so go to mss.
! Callable dcs commands are in libdcs.
!
#ifdef MSS
      write(6,"('getms calling msrcp: mspath=',a,' diskfile=',a)")
     |  trim(mspath),trim(diskfile)
#if defined(SUN) || defined(SGI) || defined(LINUX)
      call msrcp(ier,'mss:'//trim(mspath)//' '//trim(diskfile),
     |  dcs_output)
#else
      call msrcp_(ier,'mss:'//trim(mspath)//' '//trim(diskfile),
     |  dcs_output)
#endif
      if (ier /= 0) then
        write(6,"(/,'>>> getms: Error from msrcp reading mspath ',a,
     |    ' to diskfile ',a)") trim(mspath),trim(diskfile)
        write(6,"('    ier=',i4,' dcs_output=',a)") 
     |    ier,trim(dcs_output)
        stop 'msrcp'
!
! If msrcp was successful, save a link in tmpdir:
! (remove any pre-existing copy in tmpdir)
!
      else
        write(6,"('Acquired mss file ',a,' to disk file ',
     |    a)") trim(mspath),trim(diskfile)
        if (len_trim(tmpdir) > 0) then
          istat = iunlink(tmpfile,0)
          istat = ilink(diskfile,tmpfile,1)
          if (istat /= 0) then
            write(6,"('>>> getms: error saving link of ',a,' in cwd ',
     |        'as ',a)") trim(diskfile),trim(tmpfile)
          endif
        endif
      endif ! msrcp return
#else
!
! File not found, and mss not available:
      if (len_trim(tmpdir) > 0) then
        write(6,"(/,'>>> getms: diskfile ',a,' not found in cwd or in ',
     |    'tmpdir ',a,' and mss not available.')") 
     |    trim(diskfile),trim(tmpdir)
      else
        write(6,"(/,'>>> getms: diskfile ',a,' not found in cwd, and ',
     |    'mss not available.')") trim(diskfile)
      endif
      write(6,"(72('-'),/)")
      stop 'getms' 
#endif
      write(6,"(72('-'),/)")
      return
      end
