!
#include <defs.h>
      module cons_module
      use params_module,only: dlat,dlon,dz,nlon,nlonp1,nlonp4,nlat,
     |  nlatp1,dlev,nlev,nmlat,nmlon,nmlonp1,nmagphrlat,nmagphrlon,
     |  magphrlat1,magphrlat2,magphrlon1,zmbot,zmtop,zibot,zitop,
     |  glon,glat,glon1,glat1,zpmid,zpint,gmlon,gmlat,zpmag,zpimag,
     |  dmlev,nlevp1,nmlev
      implicit none
!
! Define model constants. 
! Parameter constants are cons_module module data and are accessed 
!   in subroutines via use-association. 
! Derived constants are cons_module module data, and are calculated
!   in sub init_cons (contained in cons_module). 
!   Sub init_cons is called by sub init (init_mod.F).
! Parameters referenced here are in params.h (s.a., dims.h)
!
! Real parameter constants:
!
      real,parameter ::
     |  dzp  = dz,           ! alias for dz (also dlev)
     |  re   = 6.37122e8,    ! earth radius (cm)                  C(51)
     |  re_inv = 1./re,      ! inverse of earth radius            C(52)
     |  avo  = 6.023e23,     ! avogadro number                    C(85)
     |  boltz = 1.38E-16,    ! boltzman's constant                C(84)
     |  p0   = 5.0e-4,       ! standard pressure                  C(81)
     |  gask = 8.314e7,      ! gas constant                       C(57)
!
! dipmin should be same as sin10 (see magfield.F):
!
#if (NLAT==36 && NLON==72)
     |  dipmin = 0.17,       ! minimum mag dip angle (5.0 deg horizontal res)
#elif (NLAT==72 && NLON==144)
     |  dipmin = 0.24,       ! minimum mag dip angle (2.5 deg horizontal res)
!    |  dipmin = 0.40,       ! minimum mag dip angle (2.5 deg horizontal res)
#else
      UNKNOWN NLAT,NLON      ! compilation will stop here if unknown res
#endif
     |  brn2d = 0.6,         !
!    |  tbound = 181.0,      ! background tn at lower boundary (tiegcm)
     |  tbound = 233.7,      ! background tn at lower boundary (timegcm)
     |  atm_amu = 28.9,      ! mean mass of surface atmosphere    C(24)
     |  shapiro = 3.0E-2,    ! shapiro smoother constant          C(26)
     |  dtsmooth = 0.95,     ! time smoothing constant            C(30)
     |  dtsmooth_div2 = 0.5*(1.-dtsmooth), !                      C(31)
     |  tgrad = 20.,         ! TN gradient from old /RUNMDS/ (timegcm)
     |  hor(nlat) = 1.0,     ! horizontal variation of eddy diffusion and
                             !  eddy thrmal conductivity. If unity, value
                             !  of KE at bottom is 5.e-6 (old /RUNMDS/)
     |  prndtl  = 10.,       ! timegcm1-2dev7 
     |  evergs = 1.602e-12,  ! 1 eV = 1.602e-12 ergs
     |  tsurplus=5.11*evergs,! surplus heat per event (ergs)    C(45)
     |  amu = 1.66e-24       ! atomic mass unit (g)
!
! Set calc_gw true to call waccm/ccm parameterization (see mgw.F)
! Set calc_gw false to zero out tendencies.
!
      logical,parameter :: calc_gw = .true. 
!
      integer :: nlonper=nlonp4 ! nlon + periodic points (alias for nlonp4)
!
! Many expressions require x/rmass, but its more efficient on most
! platforms to multiply rather than divide, so set rmassinv = 1./rmass 
! here, and use x*rmassinv in the code.
!
      real,parameter :: 
     |  rmass(3) = (/32.,16.,28./), ! o2,o,n2
     |  rmass_o2 = 32., rmass_o1  = 16., rmass_n2  = 28., 
     |  rmass_o3 = 48., rmass_n4s = 14., rmass_n2d = 14., 
     |  rmass_no = 30., rmass_op  = 16., rmass_co2 = 44.,
     |  rmass_h  =  1., rmass_h2  =  2., rmass_ho2 = 33., 
     |  rmass_h2o = 18.,rmass_h2o2= 34., rmass_he  =  4.,
     |  rmass_ch4 = 16.,rmass_co  = 28., rmass_oh  = 17.,
     |  rmass_hox = 1. ,rmass_ox  = 16., rmass_no2 = 46.,
     |  rmass_ar  = 40.
      real,parameter ::
     |  rmassinv_o2  = 1./rmass_o2,
     |  rmassinv_o1  = 1./rmass_o1,
     |  rmassinv_ox  = 1./rmass_o1,
     |  rmassinv_n2  = 1./rmass_n2,
     |  rmassinv_o3  = 1./rmass_o3,
     |  rmassinv_n4s = 1./rmass_n4s,
     |  rmassinv_n2d = 1./rmass_n2d,
     |  rmassinv_no  = 1./rmass_no,
     |  rmassinv_op  = 1./rmass_op,
     |  rmassinv_co2 = 1./rmass_co2,
     |  rmassinv_h   = 1./rmass_h,
     |  rmassinv_h2  = 1./rmass_h2,
     |  rmassinv_ho2 = 1./rmass_ho2,
     |  rmassinv_h2o = 1./rmass_h2o,
     |  rmassinv_h2o2= 1./rmass_h2o2,
     |  rmassinv_he  = 1./rmass_he,
     |  rmassinv_ch4 = 1./rmass_ch4,
     |  rmassinv_co  = 1./rmass_co,
     |  rmassinv_oh  = 1./rmass_oh,
     |  rmassinv_hox = 1./rmass_hox,
     |  rmassinv_no2 = 1./rmass_no2,
     |  rmassinv_ar  = 1./rmass_ar
!
! 2/00: these were in modsrc.snoe (tgcm13mt), but were unused.
! Low-energy protons:
!     real,parameter ::
!       alfalp = 10.,
!       efluxlp = 1.e-20
!
! Model derived constants (see sub init_cons in this module):
!
      real ::
     |  pi,             ! set with 4*atan(1)    C(110)
     |  rtd,            ! radians-to-degrees (180./pi)
     |  dtr,            ! degrees-to-radians (pi/180.)
     |  dphi,           ! delta lat (pi/nlat)   C(2)
     |  dphi_2div3,     ! 2./(3.*dphi)          C(12)
     |  dphi_1div12,    ! 1./(12.*dphi)         C(13)
     |  dlamda,         ! delta lon (2pi/nlon)  C(1)
     |  dlamda_2div3,   ! 2./(3.*dlamda)        C(10)
     |  dlamda_1div12,  ! 1./(12.*dlamda)       C(11)
     |  dt,             ! time step (secs)      C(4)
     |  dtx2,           ! 2*dt                  C(6)
     |  dtx2inv,        ! 1./(2*dt)             C(7)
     |  freq_3m3,       ! frequency of 2-day wave (rad/sec)       C(21)
     |  freq_semidi,    ! frequency of semidiurnal tide (rad/sec) C(23)
     |  freq_ann,       ! frequency of annual tide                C(25)
     |  expz(nlev+1),   ! exp(-z) at midpoints
     |  expzmid,        ! exp(-.5*dz)                             C(86)
     |  expzmid_inv,    ! 1./expzmid                              C(87)
     |  rmassinv(3),    ! inverse of rmass
     |  t0(nlev+1),     ! set by sub lowbound (bndry.F)
     |  racs(nlat),     ! 1./(re*cs(lat))
     |  cs(-1:nlat+2),  ! cos(phi)
     |  sn(nlat),       ! sin(phi)
     |  tn(nlat),       ! tan(phi)
     |  cor(nlat),
     |  grav,           ! accel due to gravity (dependent on lower boundary)
     |  dzgrav          ! grav/gask C(65)
!
! Constants for dynamo and electric field calculations:
      real,parameter :: re_dyn = 6.378165e8   ! earth radius for apex
!
! Save gdlat,gdlon(nmlonp1,nmlat) in degrees from apex for sub define_mag (mpi.F)
      real,dimension(nmlonp1,nmlat) :: gdlatdeg,gdlondeg
!
! btf 6/2/04: Increase h0 from 90 to 97 to avoid zigmxxx==0 at mag equator
!             (s.a., rtramrm in fieldline_integrals (dynamo.F))
!     real,parameter :: h0 =9.0e6, r0 =re+h0      ! does not work
!     real,parameter :: h0 =9.0e6, r0 =re_dyn+h0  ! does not work
      real,parameter :: h0 =9.7e6, r0 =re+h0      ! works

      real,parameter :: hs=1.3e7
      real :: 
     |  dlatg, dlong, dlatm, dlonm,dmagphrlon,
     |  ylatm(nmlat),    ! magnetic latitudes (radians)
     |  ylonm(nmlonp1),  ! magnetic longitudes (radians)
     |  ylatmagphr(nmagphrlat),  ! magnetosphere latitudes (radians)
     |  ylonmagphr(nmagphrlon),  ! magnetosphere longitudes (radians)
     |  rcos0s(nmlat),   ! cos(theta0)/cos(thetas)
     |  dt0dts(nmlat),   ! d(theta0)/d(thetas)
     |  dt1dts(nmlat),   ! dt0dts/abs(sinim) (non-zero at equator)
     |  table(91,2)
!
! Geographic grid in radians:
      real ::
     |  ylatg(0:nlatp1), ! geographic latitudes (radians)
     |  ylong(nlonp1)    ! geographic longitudes (radians)
!
! Critical colatitude limits for use of Heelis potential in dynamo:
      real,parameter :: 
     |  crit(2) = (/0.261799387, 0.523598775/)
!
! Test pdynamo with crit1,2 near zero to eliminate Heelis at high latitudes. 
!     real,parameter :: 
!    |  crit(2) = (/0.001, 0.002/)
!
! Kut is used in filtering longitudinal waves (see filter.F):
!
#if (NLAT==36 && NLON==72)                 /* 5.0 deg horizontal resolution */
      integer,parameter :: kut(nlat) =
     |  (/1,2,3,5,6,7,9,10,11,13,14,15,17,17,17,17,17,17,17,17,17,17,17,
     |    17,15,14,13,11,10,9,7,6,5,3,2,1/)
#elif (NLAT==72 && NLON==144)              /* 2.5 deg horizontal resolution */
       integer,parameter :: kut(nlat) =
     |   (/1  ,1  ,2  ,2  ,4  ,4  ,8  ,8  ,10 ,10 ,12 ,12,
     |     15 ,15 ,18 ,18 ,22 ,22 ,26 ,26 ,30 ,30 ,32 ,32,
     |     34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34,
     |     34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34 ,34,
     |     32 ,32 ,30 ,30 ,26 ,26 ,22 ,22 ,18 ,18 ,15 ,15,
     |     12 ,12 ,10 ,10 ,8  ,8  ,4  ,4  ,2  ,2  ,1  ,1/)
!      integer,parameter :: kut(nlat) =
!    |   (/0  ,1  ,1  ,2  ,3  ,4  ,5  ,6  , 7 , 8 , 9 ,10,
!    |     15 ,15 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17,
!    |     17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17,
!    |     17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17,
!    |     17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,17 ,15 ,15,
!    |     10 , 9 , 8 , 7 ,6  ,5  ,4  ,3  ,2  ,1  ,1  ,0/)
#else
      UNKNOWN NLAT,NLON      ! compilation will stop here if unknown res
#endif
!
! If check_exp is set true, certain routines will use expo() (util.F)
! instead of exp(). expo checks for out of range arguments to the
! exponential, substituting large or small results if the argument
! is out of range. This avoids NaNS fpe's, but degrades performance.
! It will also produce slightly different results.
!
#ifdef DEBUG
      logical,parameter :: check_exp = .true.
#else
      logical,parameter :: check_exp = .false.
#endif
!
! Special pi for mag field calculations. If pi=4.*atan(1.) and code is
! linked with -lmass lib, then the last 2 digits (16th and 17th) of pi
! are different (56 instead of 12), resulting in theta0(j=49)==0., which
! is wrong (should be .1110e-15).
!
      real,parameter ::
     |  pi_dyn=3.14159265358979312
!
      contains
!-----------------------------------------------------------------------
      subroutine init_cons
      use params_module,only: tgcm_version,tgcm_name
      use input_module,only: step
!
! Set derived constants (this is called from sub init in init_module)
!
! Local:
      real :: expdz,phi
      real :: omega = 7.292E-5
      integer :: k,i,j,nfsech,js
!
      pi = 4.*atan(1.)                ! C(110)
      rtd = 180./pi                   ! radians to degrees
      dtr = pi/180.                   ! degrees to radians
      dphi = pi/real(nlat)            ! C(2)
      dphi_2div3 = 2./(3.*dphi)       ! C(12)
      dphi_1div12 = 1./(12.*dphi)     ! C(13)
      dlamda = 2.*pi/real(nlon)       ! C(1)
      dlamda_2div3 = 2./(3.*dlamda)   ! C(10)
      dlamda_1div12 = 1./(12.*dlamda) ! C(11)
!
! expz(nlevp1) is exp(-zp) at midpoints (expz(nlev+1) not used):
!
      expz(:) = 0. ! init e(-z) at midpoints
      expz(1) = exp(-zmbot) ! zmbot is bottom midpoint level (see params.F)
      expdz = exp(-dlev)
      do k=2,nlev
        expz(k) = expz(k-1)*expdz
      enddo
      expz(nlev+1) = expz(nlev)*expdz ! btf added 5/17/04 (not in tgcm24)
      expzmid = exp(-.5*dlev)
      expzmid_inv = 1./expzmid
      do i=1,3
        rmassinv(i) = 1./rmass(i)
      enddo
      js=-(nlat/2)
      do j=1,nlat
        phi=(j+js-.5)*dphi
        cs(j)=cos(phi)
        sn(j)=sin(phi)
        tn(j)=tan(phi)
        cor(j)=2.*omega*sn(j)
        racs(j) = 1./(re*cs(j))
      enddo
!
! cs at 0, -1, nlat+1, and nlat+2 replace the old cssp and csnp:
      cs(-1) = -cs(2)
      cs(0)  = -cs(1)
      cs(nlat+1) = -cs(nlat)
      cs(nlat+2) = -cs(nlat-1)
      dt = real(step)                    ! was C(4)
      dtx2 = 2.*dt                       ! was C(6)
      dtx2inv = 1./dtx2                  ! was C(7)
      freq_3m3 = 2.*pi/(49.7789*60.*60.) ! was C(21)
      freq_semidi = 4.*pi/(24.*60.*60.)  ! was C(23)
      freq_ann = freq_semidi/(2.*365.25) ! was C(25)
!
! Set gravity according to lower boundary:
      grav = 945.          ! (is 870 for tiegcm)
      dzgrav = grav/gask   ! C(65)
!
! Set kut for wave filtering according to dlat (2.5 or 5.0):
!     call set_wave_filter(36,kut_5,nlat,kut)
!     write(6,"('init_cons: dlat=',f6.2,' nlat=',i3,' kut=',/,(12i4))")
!    |  dlat,nlat,kut
!
! Set dynamo constants:
!     call consdyn ! this is now called from tgcm.F
!
! Report to stdout:
      write(6,"(/,'Model name    = ',a)") tgcm_name
      write(6,"(  'Model version = ',a)") tgcm_version
      write(6,"(/,'Set constants:')")
      write(6,"('  nlat=',i3,' nlon=',i3,' nlev=',i3)") nlat,nlon,nlev
      write(6,"('  dz = ',f5.2)") dz
      write(6,"('  zmbot, zmtop = ',2f8.3,
     |  ' (bottom,top midpoint levels)')") zmbot,zmtop
      write(6,"('  zibot, zitop = ',2f8.3,
     |  ' (bottom,top interface levels)')") zibot,zitop
      write(6,"('  dt = ',f8.2,' secs')") dt
      write(6,"('  grav = ',f10.2)") grav
      write(6,"('  freq_3m3 = ',e12.4,'  freq_semidi=',e12.4,
     |  '  freq_ann=',e12.4)") freq_3m3,freq_semidi,freq_ann
      write(6,"('  dipmin = ',f8.3)") dipmin
      write(6,"('  check_exp = ',l1)") check_exp
      write(6,"('  calc_gw   = ',l1)") calc_gw
      write(6,"('  dlat=',f6.2,' dlon=',f6.2)") dlat,dlon
!     write(6,"('  kut(nlat)=',/,(12i4))") kut
!
      end subroutine init_cons
!-----------------------------------------------------------------------
      subroutine set_wave_filter(nlat5,kut5,nlat,kut)
!
! Args:
      integer,intent(in) :: nlat5,nlat
      integer,intent(in) :: kut5(nlat5)
      integer,intent(out) :: kut(nlat)
!
! Local:
      integer :: j
!
      if (nlat==nlat5) then ! nlat==nlat5==36 (5x5 degree res)
        do j=1,nlat
          kut(j) = kut5(j)
        enddo
      elseif (nlat==nlat5*2) then ! nlat==72 (2.5x2.5 degree res)
        do j=1,nlat5-1
          kut(j*2-1) = kut5(j) ! 1,3,5,...,65,67,69
          kut(j*2)   = kut5(j) ! 2,4,6,...,66,68,70
        enddo
        kut(nlat) = kut5(nlat5)
        kut(nlat-1) = kut5(nlat5)
      else
        write(6,"('set_wave_filter: nlat=',i3,' dlat=',f8.3,
     |    ' not supported.')") nlat,dlat
        call shutdown('dlat')
      endif
!     write(6,"('set_wave_filter: nlat=',i3,' kut=',/,(12i4))") nlat,kut
      end subroutine set_wave_filter
!-----------------------------------------------------------------------
      subroutine consdyn
      use input_module,only: dynamo
!
! Set derived constants used in dynamo.
!
! Local:
      integer :: j,i,n,k
      real,parameter :: e=1.e-6, r1=1.06e7, alfa=1.668
      real :: 
     |  tanth0(nmlat),
     |  tanths(nmlat),
     |  theta0(nmlat),
     |  hamh0(nmlat)
      real :: dtheta,table2(91,3:5),tanths2
      real :: fac,rmin,rmax,rmag
!
! Must define pi, rtd, dtr, because consdyn is called before init_cons:
      pi = 4.*atan(1.)                ! C(110)
      rtd = 180./pi                   ! radians to degrees
      dtr = pi/180.                   ! degrees to radians
!
! Set grid deltas:
      dlatg = pi/float(nlat)
      dlong = 2.*pi/float(nlon)
      dlatm = pi_dyn/float(nmlat-1) ! note use of pi_dyn
      dlonm = 2.*pi_dyn/float(nmlon) 
      dmagphrlon = 360./float(nmagphrlon)
!
! Set geographic latitude array ylatg:
      do j=1,nlat
        ylatg(j) = -0.5*(pi-dlatg)+float(j-1)*dlatg
      enddo ! j=1,nlat
      ylatg(0) = -pi/2.+e
      ylatg(nlatp1) = pi/2.-e
!
! Set geographic longitude array ylong:
      do i=1,nlonp1
        ylong(i) = -pi+float(i-1)*dlong
      enddo ! i=1,nmlonp1
!
! Set magnetic latitudes ylatm and magnetic longitudes ylonm:
!
! ylatm is equally spaced in theta0, but holds corresponding value
!   of thetas.
      do j=1,nmlat
        theta0(j) = -pi_dyn/2.+float(j-1)*dlatm ! note use of pi_dyn
      enddo ! j=1,nmlat
      do j=2,nmlat-1
        tanth0(j) = abs(tan(theta0(j)))
        hamh0(j) = r1*tanth0(j)+r0*tanth0(j)**(2.+2.*alfa)/
     |    (1.+tanth0(j)**2)**alfa
        tanths(j) = sqrt(hamh0(j)/r0)
        ylatm(j) = sign(atan(tanths(j)),theta0(j))
        rcos0s(j) = sqrt((1.+tanths(j)**2)/(1.+tanth0(j)**2))
!
! If dynamo==0 -> dynamo is not calculated
! If dynamo==1 -> dynamo is calculated
!
        if (dynamo <= 0) then ! no dynamo
          dt1dts(j) = 0.
          dt0dts(j) = (2.*r0*tanths(j)*(1.+tanths(j)**2))/
     |      (r1*(1.+tanth0(j)**2)+2.*r0*tanth0(j)**(2.*alfa+1.)*
     |      (1.+alfa+tanth0(j)**2)/(1.+tanth0(j)**2)**alfa)
        else                  ! with dynamo
          tanths2  = tanths(j)**2
          dt1dts(j) = 
     |      (r0*sqrt(1.+4.*tanths2)*(1.+tanths2))/
     |      (r1*(1.+tanth0(j)**2)+2.*r0*tanth0(j)**(2.*alfa+1.)*
     |      (1.+alfa+tanth0(j)**2)/(1.+tanth0(j)**2)**alfa)
          dt0dts(j) = dt1dts(j)*2.*tanths(j)/sqrt(1.+4.*tanths2)
        endif

!       write(6,"('consdyn: j=',i3,' tanth0=',e12.4,' hamh0=',e12.4,
!    |    ' tanths=',e12.4,' ylatm=',e12.4,' rcos0s=',e12.4,' dt0dts=',
!    |    e12.4,' dt1dts=',e12.4)") j,tanth0(j),hamh0(j),tanths(j),
!    |    ylatm(j),rcos0s(j),dt0dts(j),dt1dts(j) 

      enddo ! j=2,nmlat-1
!
! Magnetic poles:
      ylatm(1) = theta0(1)
      ylatm(nmlat) = theta0(nmlat)
      rcos0s(1) = 1.
      rcos0s(nmlat) = 1.
      dt0dts(1) = 1.
      dt0dts(nmlat) = 1.
!
! Magnetic longitudes:
      do i=1,nmlonp1
        ylonm(i) = -pi+float(i-1)*dlonm
      enddo ! i=1,nmlonp1
      dtheta = pi/(2.*90.)
!
      table(1,1) = 0. 
      table(1,2) = 0. 
      do i=2,91
        table(i,1) = table(i-1,1)+dtheta
      enddo
      do i=2,90
        table2(i,4) = tan(table(i,1))
        table(i,2) = table(i,1)
      enddo ! i=2,90
      table(91,2) = table(91,1)
!     table(91,2) = pi/2.
      do n=1,7
        do i=2,90
          table2(i,3) = table(i,2)
          table(i,2) = tan(table2(i,3))
          table2(i,5) = sqrt(r1/r0*table(i,2)+table(i,2)**(2.*(1.+alfa))
     |      /(1.+table(i,2)**2)**alfa)
          table(i,2) = table2(i,3)-(table2(i,5)-table2(i,4))*2.*
     |      table2(i,5)/(r1/r0*(1.+table(i,2)**2)+2.*table(i,2)**
     |      (2.*alfa+1.)*(1.+alfa+table(i,2)**2)/(1.+table(i,2)**2)**
     |      alfa)
        enddo ! i=2,90
      enddo ! n=1,7
!
! Define magnetospheric grid vars
      do i=1,nmagphrlon
        ylonmagphr(i) = magphrlon1+(i-1)*dmagphrlon
      enddo
      fac = pi/180.
      rmax = 1./(cos(magphrlat1*fac))**2
      rmin = 1./(cos(magphrlat2*fac))**2
      do i=1,nmagphrlat
        rmag = (rmax-(rmax-rmin)/(nmagphrlat-1)*real(i-1))
        ylatmagphr(i) = acos(sqrt(1./rmag))/fac
      enddo
!
! Define mag grid in degrees, and mag levels:
      gmlat(:) = ylatm(:)*rtd
      gmlon(:) = ylonm(:)*rtd
      do k=1,nmlev
        zpmag(k) = zmbot+(k-1)*dmlev
        zpimag(k) = zibot+(k-1)*dmlev
      enddo
!
!     write(6,"(/,'consdyn: ylatg =',/,(6e12.4))") ylatg
!     write(6,"(  'consdyn: ylong =',/,(6e12.4))") ylong
!     write(6,"(  'consdyn: ylatm =',/,(6e12.4))") ylatm
!     write(6,"(  'consdyn: ylonm =',/,(6e12.4))") ylonm
!     write(6,"(  'consdyn: rcos0s=',/,(6e12.4))") rcos0s
!     write(6,"(  'consdyn: dt0dts=',/,(6e12.4))") dt0dts
!     write(6,"(  'consdyn: table=',/,(6e12.4))") table
!     write(6,"(  'consdyn: table2=',/,(6e12.4))") table2
!
      end subroutine consdyn
!-----------------------------------------------------------------------
      subroutine set_geogrid
      integer :: i,j,k
!
! Define geographic grid:
      do i=1,nlon
        glon(i) = glon1+(i-1)*dlon
      enddo
      do j=1,nlat
        glat(j) = glat1+(j-1)*dlat
      enddo
      do k=1,nlevp1
        zpmid(k) = zmbot+(k-1)*dlev  ! midpoint levels
        zpint(k) = zibot+(k-1)*dlev  ! interface levels
      enddo
      end subroutine set_geogrid
!-----------------------------------------------------------------------
      end module cons_module
