!
      module hist
      implicit none
!
! History type:
!
      integer,parameter :: nsdtide=10, ndtide=2
      type history
        character*80 mssvol	! mss path of volume containing history
        character*240 histfile	! path to history file
        integer :: mtime(3)	! model time (day,hr,min)
        real :: ut		! ut (from mtime hr and min)
        integer :: iyd		! year-day yyddd (from hdrdate(2))
        integer :: nzp		! number of pressure levels (in lat slices)
        real,pointer :: lev(:)=>NULL()  ! midpoint levels coord array
        real,pointer :: ilev(:) =>NULL()! interface levels coord array
        real :: zpb,zpt		! bottom,top pressure levels (midpoints)
        real :: zpib,zpit	! bottom,top pressure levels (interfaces)
        real :: lbc             ! zp of lower boundary condition (LBC)
                                ! (should be same as zpb)
        integer :: nlat,nlon	! number of lats,lons  (in lat slices)
        integer :: nflds	! number of fields on history
        logical isdyn		! if true, is dynamo history
        logical istimes		! if true, is times-gcm history
        logical issech		! if true, is secondary history
        logical iscpl		! if true, is from coupled run
        logical ismtgcm		! if true, is mars tgcm
        logical isjtgcm		! if true, is jupiter tgcm
        logical isvtgcm		! if true, is venus tgcm
        logical isnew		! if true, is "new" history file format
        character*32 version	! model version name (e.g. tiegcm1.8)
        character*32 model_name	! model version name (e.g. tiegcm)
        character(len=8),pointer :: fnames(:)=>NULL()	! names of fields on history
        integer :: lu		! fort unit (== 0 if a netcdf file)
        integer :: ncid		! netcdf id (== 0 if a fortran file)
        integer :: itime        ! this is nth history on file
        real :: hpower,ctpoten,byimf,f107a,f107d
        real :: dtide(ndtide),sdtide(nsdtide)
        integer :: gpi,ncep,amie,year,day,step
        integer :: gswmdi,gswmsdi,gswmnmidi,gswmnmisdi
      end type history
!
! Old style tgcm header in common:
!
!     common /tgcmhdr/ iter,nday,nhr,nmin,label(40),
!    +  hdrdate(2),output(3,20),start(3),stp(6),hdrhist(6),sav(3),step,
!    +  mag(4),difhor,iuivi,sdtide(10),ipower,aurora,dispos,data(3),
!    +  source(3),sourct(3),dtide(2),dum(8),rdate,naur,hp,cp,byimf,
!    +  aurp(60),rmodnum,colfac,f107d,f107a
!     integer iter,nday,nhr,nmin,hdrdate,start,stp,hdrhist,step,sourct,
!    +  sav,difhor,aurora,dispos,ipower,naur,iuivi
!     real mag,sdtide,dtide,dum,hp,cp,byimf,aurp,rmodnum,colfac,
!    +  f107d,f107a
!     character*8 data,source,label,output,rdate
!
! f90 style header structure:
!
      type tgcmheader
        integer iter,nday,nhr,nmin,hdrdate(2),start(3),stp(6),
     +    hdrhist(6),step,sourct(3),sav(3),difhor,aurora,dispos,
     +    ipower,naur,iuivi
        real mag(4),sdtide(nsdtide),dtide(ndtide),hp,cp,byimf,aurp,
     |    rmodnum,colfac,f107d,f107a
        character*8 data(3),source(3),label(40),output(3,20),rdate
      end type tgcmheader
      type(tgcmheader) :: hdr
!
! tgcm summary (mxlen_summary is max len in words):
!
      integer,parameter :: mxlen_summary = 512
      real :: summary(mxlen_summary)
      contains
!-------------------------------------------------------------------
      subroutine inithist(h)
!
! Initialize and return a history structure:
!
      type(history) :: h
      real, parameter :: spval=1.e36		! real special value
!
      h%version = "unknown"
      h%mssvol = ' '
      h%histfile = ' '
      h%mtime = (/-1,-1,-1/)
      h%ut = -1.
      h%nzp = 0
      h%zpb = 0.
      h%zpt = 0.
      h%zpib = 0.
      h%zpit = 0.
      h%nlat = 0
      h%nlon = 0
      h%nflds = 0
      h%isdyn = .false.
      h%istimes = .false.
      h%issech = .false.
      h%iscpl = .false.
      h%ismtgcm = .false.
      h%isjtgcm = .false.
      h%version = ' '
      h%model_name = ' '
      h%lu = -1
      h%ncid = -1
      h%itime = -1
      h%year = -1
      h%day = -1
      h%gpi = -1
      h%ncep = -1
      h%gswmdi = -1
      h%gswmsdi = -1
      h%gswmnmidi = -1
      h%gswmnmisdi = -1
      h%hpower = spval
      h%ctpoten = spval
      h%byimf = spval
      h%f107a = spval
      h%f107d = spval
      h%dtide(:) = spval
      h%sdtide(:) = spval
      return
      end subroutine inithist
!
!-------------------------------------------------------------------
!
      subroutine print_hist(h)
!
! Print info about a history structure:
!
! Args:
      type(history),intent(in) :: h
!
! Locals:
      integer :: i
!
      if (h%lu > 0) then
        write(6,"('HISTORY (on cray-blocked history file):')")
        write(6,"('lu = ',i2)") h%lu
      else
        write(6,"('HISTORY (on netcdf history file):')")
        write(6,"('ncid = ',i8)") h%ncid
      endif
      write(6,"('Mss volume: ',a)") trim(h%mssvol)
      write(6,"('History file: ',a)") trim(h%histfile)
      write(6,"('Year-day (yyddd) = ',i5)") h%iyd
      write(6,"('Model time (day:hr:min): ',i3,':',i2,':',i2,
     +  ' (ut=',f7.3,')')") h%mtime,h%ut
      write(6,"('ut (hours) = ',f9.3)") h%ut
      write(6,"('year = ',i4)") h%year
      write(6,"('day = ',i4)") h%day
      write(6,"('Model name: ',a)") h%model_name
      write(6,"('Model version: ',a)") h%version
      write(6,"('nzp     = ',i3,' (number of vertical levels)')")h%nzp
      write(6,"('zpb,zpt = ',2f7.2,' (bottom and top midpoint pressure',
     +    ' levels)')") h%zpb,h%zpt
      write(6,"('zpib,zpit = ',2f7.2,' (bottom and top interface',
     |  ' pressure levels)')") h%zpib,h%zpit
      write(6,"('itime   = ',i3,' (nth history on the file)')") h%itime
      if (h%isdyn) then
        write(6,"('isdyn   = ',l1,' (history is from a dynamo model)')") 
     +    h%isdyn
      else
        write(6,"('isdyn   = ',l1,' (history is from a NON-dynamo ',
     +    'model)')") h%isdyn
      endif
      if (h%istimes) then
        write(6,"('istimes = ',l1,' (model is time-gcm)')") 
     +    h%istimes
      else
        write(6,"('istimes = ',l1,' (model is ti(e)gcm)')") 
     +    h%istimes
      endif
      if (h%ismtgcm) then
        write(6,"('ismtgcm = ',l1,' (model is mars mtgcm)')") h%ismtgcm
      endif
      if (h%isjtgcm) then
        write(6,"('isjtgcm = ',l1,' (model is jupiter mtgcm)')") 
     |    h%isjtgcm
      endif
      if (h%iscpl) then
        write(6,"('iscpl   = ',l1,' (history IS from a coupled ',
     +    'tgcm/ccm run)')") h%iscpl
      else
        write(6,"('iscpl   = ',l1,' (history is NOT from a coupled ',
     +    'tgcm/ccm run)')") h%iscpl
      endif
      if (h%issech) then
        write(6,"('issech  = ',l1,' (this is a secondary history)')") 
     +    h%issech
      else
        write(6,"('issech  = ',l1,' (this is a primary history)')")
     +    h%issech
      endif
      if (h%isnew) then
        write(6,"('isnew   = ',l1,' (is new history file format)')")
     |    h%isnew
        write(6,"('lbc     = ',f5.1,' (zp of bottom interface level)')")
     |    h%lbc
      else
        write(6,"('isnew   = ',l1,' (is old history file format)')")
     |    h%isnew
      endif

      write(6,"('hpower  = ',f10.3)") h%hpower
      write(6,"('ctpoten = ',f10.3)") h%ctpoten
      write(6,"('byimf   = ',f10.3)") h%byimf
      write(6,"('f107a   = ',f10.3)") h%f107a
      write(6,"('f107d   = ',f10.3)") h%f107d
      write(6,"('dtide   = ',2e12.4)") h%dtide
      write(6,"('sdtide  = ',/,(6e12.4))") h%sdtide
      write(6,"('gpi     = ',i3)") h%gpi
      write(6,"('ncep    = ',i3)") h%ncep
      write(6,"('gswmdi   , gswmsdi    = ',2i3)") h%gswmdi,h%gswmsdi
      write(6,"('gswmnmidi, gswmnmisdi = ',2i3)") h%gswmnmidi,
     |  h%gswmnmisdi

      if (associated(h%fnames)) then
        write(6,"('There are ',i3,' fields on this history,',
     +    ' as follows:')") h%nflds
        do i=1,h%nflds
          write(6,"(a)",advance="NO") h%fnames(i)
          if (mod(i,8)==0) write(6,"(' ')")
        enddo
        write(6,"(' ')")
      else
        write(6,"('Field names undefined (nflds=',i3,')')") 
     +    h%nflds
      endif
      write(6,"(' ')")
      return
      end subroutine print_hist
      end module hist
