!
      module protons_module
!
! Data from Charley Jackman for solar proton events.
!   (see ~foster/usr/jackman) 
!
! The 8 proton events (data is on the half hour):
!   Event  1: year 2000 days 196 to 200 (July 14-18)
!   Event  2: year 2000 days 313 to 318 (November 8-13)
!   Event  3: year 2001 days 267 to 276 (September 24-October 3)
!   Event  4: year 2001 days 308 to 314 (November 4-10)
!   Event  5: year 2001 days 326 to 331 (November 22-27)
!   Event  6: year 2002 days 107 to 120 (April 17-30)
!   Event  7: year 2003 days 299 to 311 (October 26-November 7)
! Event 8 added 4/21/05:
!   Event  8: year 2005 days 15  to 22  (January 15-22)
!
      implicit none
      integer,parameter :: nevents = 8, mxtimes = 336
      integer,parameter,dimension(nevents) :: 
     |  ntimes = (/120  ,144  ,240  ,168  ,144  ,336  ,312  ,192/),
     |  iyears = (/2000 ,2000 ,2001 ,2001 ,2001 ,2002 ,2003 ,2005/)
      real,dimension(mxtimes,nevents) :: 
     |  fe,   ! energy flux (MeV/cm2/s)
     |  eo,   ! characteristic energy (MeV)
     |  hour  ! ut (hrs) for each data
      integer,dimension(mxtimes,nevents) :: 
     |  iday  ! julian day for each data
      contains
!-----------------------------------------------------------------------
      subroutine get_protons(iyear,mday,secs,istep,nstep,alfad2,ed2,
     |  found)
!
! Args:
      integer,intent(in) :: iyear,mday,istep,nstep
      real,intent(in) :: secs
      real,intent(inout) :: 
     |  alfad2, ! characteristic energy (MeV), interpolated from Eo
     |  ed2     ! energy flux (ergs/cm2/s), interpolated from Fe
      logical,intent(out) :: found
!
! Locals:
      integer :: i,i0,i1,msecs,dsecs0,dsecs1,ievent
!
! External:
      logical,external :: time2print
!
      found = .false.
      msecs = mday*86400+int(secs)
!     write(6,"(/,'Enter get_protons: iyear=',i4,' mday=',i3,' secs=',
!    |  f10.3,' msecs=',i10,' istep=',i4,' nstep=',i4)") 
!    |  iyear,mday,secs,msecs,istep,nstep
!
! msecs = current model time in seconds (includes day)
! dsecs0,dsecs1 = beginning,ending data times (includes day)
! Note there may be multiple events within a single year.
!
      ievent = 0
      eventloop: do i=1,nevents
!
! Check if there is an event for current year:
	if (iyear /= iyears(i)) cycle eventloop
!
! We are in a year with an event, so check day and time:
        dsecs0 = iday(1,i)*86400+int(hour(1,i)*3600.)
        dsecs1 = iday(ntimes(i),i)*86400+int(hour(ntimes(i),i)*3600.)
!
! If we are within the date and time of an event, set ievent:
        if (msecs >= dsecs0 .and. msecs <= dsecs1) then
          ievent = i
          exit eventloop
        endif
      enddo eventloop ! i=1,nevents
!
! Proton data not available for current model time:
!
      if (ievent==0) then
!
!   If proton data is not available for this date, print to stdout according 
!   to time2print (util.F, usually every step or every 10th step), and return.
!
        if (time2print(nstep,istep))
     |    write(6,"('  Solar proton data not available for iyear=',i4,
     |      ' mday=',i3,' ut=',f8.2,'. Using default alfad2=',e12.4,
     |      ' ed2=',e12.4)") iyear,mday,secs/3600.,alfad2,ed2
        return
      endif
!
!     write(6,"('get_protons: using event ',i2,': year ',i4,' days ',
!    |  i3,' to ',i3)") ievent,iyears(ievent),iday(1,ievent),
!    |  iday(ntimes(ievent),ievent)
!
! We are in a proton event. Bracket model time:
      i0 = 0
      i1 = 0
      time_loop: do i=1,ntimes(ievent)-1
        dsecs0 = iday(i,ievent)  *86400+int(hour(i,ievent)  *3600.)
        dsecs1 = iday(i+1,ievent)*86400+int(hour(i+1,ievent)*3600.)
        if (msecs >= dsecs0 .and. msecs <= dsecs1) then
          i0 = i
          i1 = i+1
          exit time_loop
        endif
      enddo time_loop ! i=1,ntimes-1
!
! Stop if unable to bracket model time:
      if (i0==0.or.i1==0) then
        write(6,"(/,'>>> WARNING get_protons: could not',
     |    ' bracket model time: current model secs = ',f10.2)") secs
        call shutdown('get_protons')
      endif
      found = .true.
!
!     write(6,"('i0=',i3,' i1=',i3,' i0 day,hour=',i4,',',f5.2,
!    |  ' i1 day,hour=',i4,',',f5.2)") i0,i1,iday(i0,ievent),
!    |  hour(i0,ievent),iday(i1,ievent),hour(i1,ievent)
!     write(6,"('Fe i0,i1=',2e12.4,' Eo i0,i1=',2e12.4)")
!    |  fe(i0,ievent),fe(i1,ievent),eo(i0,ievent),eo(i1,ievent)
!
! Do simple linear interpolation:
      ed2    = (msecs-dsecs0)*(fe(i1,ievent)-fe(i0,ievent))/
     |  (dsecs1-dsecs0)+fe(i0,ievent)
      alfad2 = (msecs-dsecs0)*(eo(i1,ievent)-eo(i0,ievent))/
     |  (dsecs1-dsecs0)+eo(i0,ievent)
      ed2 = ed2*1.6022e-6 ! convert MeV to ergs
!
! Report to stdout: 
      if (time2print(nstep,istep))
     |  write(6,"('  Solar protons: iyear=',i4,' mday=',i3,' ut=',f5.2,
     |    ' ed2=',e12.4,' alfad2=',e12.4)") 
     |    iyear,mday,secs/3600.,ed2,alfad2
      end subroutine get_protons
!-----------------------------------------------------------------------
      subroutine protons_data
!
! These array-constructors were written (except for the last event, 
! April, 2002) by IDL procedure ~foster/usr/jackman/rdprotons.pro.
!
! Proton flux parameters for July 14-18, 2000 Solar Proton Event
      fe(1:120,1)=(/
     | 5.370e+02, 5.338e+02, 4.851e+02, 4.398e+02, 4.476e+02, 4.578e+02,
     | 4.291e+02, 4.227e+02, 4.001e+02, 3.796e+02, 1.552e+03, 3.803e+05,
     | 6.947e+05, 5.005e+05, 3.962e+05, 4.302e+05, 7.033e+05, 7.091e+05,
     | 7.551e+05, 7.224e+05, 6.921e+05, 7.458e+05, 7.575e+05, 7.253e+05,
     | 7.679e+05, 7.915e+05, 8.804e+05, 1.105e+06, 1.343e+06, 1.287e+06,
     | 1.306e+06, 1.322e+06, 1.359e+06, 1.695e+06, 1.709e+06, 1.827e+06,
     | 2.006e+06, 1.859e+06, 1.844e+06, 1.235e+06, 9.673e+05, 8.286e+05,
     | 6.540e+05, 3.678e+05, 1.133e+05, 1.214e+05, 1.007e+05, 7.554e+04,
     | 5.989e+04, 5.059e+04, 5.302e+04, 4.488e+04, 4.379e+04, 3.235e+04,
     | 3.047e+04, 2.951e+04, 2.511e+04, 2.263e+04, 2.857e+04, 3.500e+04,
     | 3.431e+04, 2.632e+04, 2.744e+04, 2.199e+04, 2.081e+04, 1.914e+04,
     | 1.620e+04, 1.443e+04, 1.438e+04, 1.425e+04, 1.355e+04, 1.372e+04,
     | 1.500e+04, 1.675e+04, 1.577e+04, 1.538e+04, 1.461e+04, 1.369e+04,
     | 1.257e+04, 1.132e+04, 1.027e+04, 9.116e+03, 8.037e+03, 7.090e+03,
     | 6.900e+03, 6.495e+03, 5.689e+03, 4.848e+03, 4.409e+03, 4.204e+03,
     | 4.313e+03, 4.365e+03, 4.285e+03, 4.292e+03, 4.170e+03, 4.097e+03,
     | 4.334e+03, 4.259e+03, 4.329e+03, 4.208e+03, 3.856e+03, 3.822e+03,
     | 3.562e+03, 3.403e+03, 3.145e+03, 2.983e+03, 2.990e+03, 2.849e+03,
     | 2.478e+03, 2.533e+03, 2.491e+03, 2.291e+03, 2.096e+03, 2.143e+03,
     | 2.028e+03, 2.052e+03, 2.040e+03, 2.067e+03, 1.999e+03, 1.872e+03
     | /)
      eo(1:120,1)=(/
     | 4.166e+00, 4.132e+00, 4.081e+00, 4.057e+00, 4.094e+00, 4.056e+00,
     | 4.128e+00, 4.141e+00, 4.207e+00, 4.248e+00, 1.594e+01, 1.564e+02,
     | 1.041e+02, 3.611e+01, 2.433e+01, 2.011e+01, 1.703e+01, 1.525e+01,
     | 1.400e+01, 1.346e+01, 1.355e+01, 1.276e+01, 1.233e+01, 1.204e+01,
     | 1.156e+01, 1.123e+01, 1.075e+01, 1.039e+01, 1.024e+01, 9.950e+00,
     | 9.680e+00, 9.252e+00, 9.029e+00, 9.186e+00, 8.754e+00, 8.842e+00,
     | 8.505e+00, 8.159e+00, 8.026e+00, 6.951e+00, 6.459e+00, 6.345e+00,
     | 6.199e+00, 5.997e+00, 6.022e+00, 6.092e+00, 6.127e+00, 5.929e+00,
     | 5.898e+00, 5.978e+00, 5.913e+00, 5.970e+00, 5.760e+00, 5.987e+00,
     | 6.027e+00, 6.079e+00, 6.062e+00, 6.035e+00, 6.372e+00, 6.160e+00,
     | 6.157e+00, 6.442e+00, 6.595e+00, 6.831e+00, 6.935e+00, 7.042e+00,
     | 7.221e+00, 7.179e+00, 7.074e+00, 6.896e+00, 6.606e+00, 6.355e+00,
     | 6.123e+00, 5.948e+00, 5.820e+00, 5.778e+00, 5.818e+00, 5.848e+00,
     | 5.846e+00, 5.955e+00, 5.906e+00, 5.958e+00, 5.968e+00, 6.022e+00,
     | 6.067e+00, 6.066e+00, 6.182e+00, 6.386e+00, 6.473e+00, 6.412e+00,
     | 6.286e+00, 6.179e+00, 6.147e+00, 5.991e+00, 5.855e+00, 5.716e+00,
     | 5.672e+00, 5.515e+00, 5.373e+00, 5.410e+00, 5.461e+00, 5.439e+00,
     | 5.318e+00, 5.441e+00, 5.490e+00, 5.386e+00, 5.554e+00, 5.600e+00,
     | 5.706e+00, 5.713e+00, 5.653e+00, 5.608e+00, 5.751e+00, 5.696e+00,
     | 5.499e+00, 5.584e+00, 5.410e+00, 5.475e+00, 5.439e+00, 5.416e+00
     | /)
      iday(1:120,1)=(/
     |  196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196,
     |  196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196, 196,
     |  197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197,
     |  197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197, 197,
     |  198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198,
     |  198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198, 198,
     |  199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199,
     |  199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199, 199,
     |  200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200,
     |  200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200, 200
     | /)
      hour(1:120,1)=(/
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
!
! Proton flux parameters for November 8-13, 2000 Solar Proton Event
      fe(1:144,2)=(/
     | 1.436e+01, 1.145e+01, 1.309e+01, 1.237e+01, 1.275e+01, 1.080e+01,
     | 1.425e+01, 1.341e+01, 1.119e+01, 1.231e+01, 1.292e+01, 1.361e+01,
     | 1.192e+01, 1.458e+01, 1.391e+01, 1.626e+01, 1.733e+01, 1.259e+01,
     | 1.379e+01, 1.566e+01, 1.321e+01, 1.341e+01, 1.248e+01, 1.228e+03,
     | 1.626e+05, 4.880e+05, 6.858e+05, 8.596e+05, 9.473e+05, 9.977e+05,
     | 1.054e+06, 1.026e+06, 9.610e+05, 1.084e+06, 1.113e+06, 1.080e+06,
     | 1.032e+06, 1.027e+06, 9.566e+05, 1.028e+06, 9.316e+05, 8.217e+05,
     | 7.966e+05, 7.797e+05, 7.148e+05, 6.469e+05, 5.629e+05, 5.041e+05,
     | 4.015e+05, 3.049e+05, 2.851e+05, 2.553e+05, 2.374e+05, 2.187e+05,
     | 1.896e+05, 1.101e+05, 7.151e+04, 9.275e+04, 9.392e+04, 8.342e+04,
     | 1.058e+05, 7.999e+04, 5.872e+04, 5.401e+04, 5.356e+04, 4.946e+04,
     | 5.108e+04, 4.712e+04, 4.328e+04, 4.208e+04, 3.955e+04, 3.488e+04,
     | 3.344e+04, 3.050e+04, 2.698e+04, 2.621e+04, 2.423e+04, 1.543e+04,
     | 1.175e+04, 6.632e+03, 3.996e+03, 2.214e+03, 1.305e+03, 1.071e+03,
     | 1.291e+03, 1.224e+03, 1.597e+03, 1.913e+03, 2.007e+03, 2.054e+03,
     | 1.946e+03, 1.923e+03, 1.912e+03, 1.987e+03, 1.853e+03, 1.988e+03,
     | 2.055e+03, 2.123e+03, 2.015e+03, 1.933e+03, 1.819e+03, 1.875e+03,
     | 1.869e+03, 1.817e+03, 1.751e+03, 1.722e+03, 1.786e+03, 1.744e+03,
     | 1.586e+03, 1.581e+03, 1.614e+03, 1.583e+03, 1.561e+03, 1.355e+03,
     | 1.310e+03, 1.325e+03, 1.280e+03, 1.001e+03, 8.489e+02, 8.619e+02,
     | 9.062e+02, 8.352e+02, 8.520e+02, 8.421e+02, 8.695e+02, 8.452e+02,
     | 7.974e+02, 7.981e+02, 7.706e+02, 7.451e+02, 7.021e+02, 7.074e+02,
     | 7.042e+02, 6.948e+02, 6.458e+02, 6.401e+02, 5.542e+02, 5.472e+02,
     | 5.510e+02, 5.538e+02, 5.294e+02, 5.152e+02, 5.162e+02, 4.861e+02
     | /)
      eo(1:144,2)=(/
     | 1.444e+01, 1.302e+01, 1.152e+01, 1.358e+01, 1.339e+01, 1.327e+01,
     | 1.487e+01, 1.530e+01, 1.495e+01, 1.566e+01, 1.693e+01, 1.299e+01,
     | 1.695e+01, 1.575e+01, 1.460e+01, 1.923e+01, 1.527e+01, 1.637e+01,
     | 1.414e+01, 1.727e+01, 1.510e+01, 1.357e+01, 1.278e+01, 2.428e+01,
     | 1.691e+01, 1.604e+01, 1.630e+01, 1.684e+01, 1.597e+01, 1.497e+01,
     | 1.371e+01, 1.322e+01, 1.278e+01, 1.227e+01, 1.202e+01, 1.174e+01,
     | 1.158e+01, 1.147e+01, 1.088e+01, 9.817e+00, 8.372e+00, 7.525e+00,
     | 7.154e+00, 6.990e+00, 6.859e+00, 6.667e+00, 6.522e+00, 6.463e+00,
     | 6.652e+00, 6.636e+00, 6.441e+00, 6.422e+00, 6.394e+00, 6.346e+00,
     | 6.274e+00, 6.341e+00, 6.475e+00, 6.264e+00, 6.234e+00, 6.234e+00,
     | 6.084e+00, 6.324e+00, 6.608e+00, 6.662e+00, 6.608e+00, 6.751e+00,
     | 6.604e+00, 6.840e+00, 6.994e+00, 6.892e+00, 6.856e+00, 6.998e+00,
     | 7.036e+00, 6.953e+00, 6.939e+00, 6.898e+00, 6.890e+00, 7.078e+00,
     | 7.197e+00, 7.700e+00, 8.049e+00, 8.523e+00, 8.938e+00, 9.126e+00,
     | 9.187e+00, 9.564e+00, 9.189e+00, 8.815e+00, 8.737e+00, 8.660e+00,
     | 8.655e+00, 8.684e+00, 8.676e+00, 8.448e+00, 8.244e+00, 8.219e+00,
     | 8.161e+00, 8.038e+00, 8.135e+00, 8.233e+00, 8.461e+00, 8.273e+00,
     | 8.167e+00, 8.255e+00, 8.165e+00, 8.110e+00, 8.368e+00, 8.329e+00,
     | 8.463e+00, 8.314e+00, 8.254e+00, 8.299e+00, 8.139e+00, 8.481e+00,
     | 8.495e+00, 8.248e+00, 8.116e+00, 8.237e+00, 8.936e+00, 8.564e+00,
     | 8.197e+00, 8.156e+00, 8.194e+00, 8.125e+00, 8.020e+00, 8.434e+00,
     | 8.298e+00, 8.275e+00, 8.338e+00, 8.316e+00, 8.534e+00, 8.451e+00,
     | 8.100e+00, 8.298e+00, 8.068e+00, 8.592e+00, 8.564e+00, 8.942e+00,
     | 8.725e+00, 8.614e+00, 8.560e+00, 8.299e+00, 8.099e+00, 8.242e+00
     | /)
      iday(1:144,2)=(/
     |  313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313,
     |  313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313,
     |  314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314,
     |  314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314,
     |  315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315,
     |  315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315, 315,
     |  316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316,
     |  316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316, 316,
     |  317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317,
     |  317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317, 317,
     |  318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318,
     |  318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318, 318
     | /)
      hour(1:144,2)=(/
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
!
! Proton flux parameters for September 24 - October 3, 2001 Solar Proton Event
      fe(1:240,3)=(/
     | 1.918e+01, 1.836e+01, 1.596e+01, 1.682e+01, 1.423e+01, 1.473e+01,
     | 1.931e+01, 1.748e+01, 1.439e+01, 1.759e+01, 1.664e+01, 1.492e+02,
     | 2.321e+03, 6.584e+03, 9.583e+03, 2.110e+04, 4.969e+04, 6.970e+04,
     | 7.833e+04, 7.814e+04, 8.655e+04, 1.019e+05, 1.310e+05, 1.456e+05,
     | 1.501e+05, 1.492e+05, 1.572e+05, 1.676e+05, 1.798e+05, 1.957e+05,
     | 1.989e+05, 2.239e+05, 2.346e+05, 2.280e+05, 2.183e+05, 1.987e+05,
     | 2.140e+05, 2.318e+05, 2.292e+05, 2.315e+05, 2.291e+05, 2.335e+05,
     | 2.513e+05, 2.922e+05, 3.984e+05, 7.409e+05, 1.037e+06, 7.887e+05,
     | 6.374e+05, 5.263e+05, 5.230e+05, 4.727e+05, 4.357e+05, 3.955e+05,
     | 3.556e+05, 3.291e+05, 2.924e+05, 2.710e+05, 2.493e+05, 2.241e+05,
     | 2.036e+05, 1.811e+05, 1.681e+05, 1.464e+05, 1.409e+05, 1.327e+05,
     | 1.192e+05, 1.140e+05, 1.079e+05, 1.025e+05, 9.665e+04, 9.264e+04,
     | 8.786e+04, 8.247e+04, 7.855e+04, 7.650e+04, 7.525e+04, 6.808e+04,
     | 6.435e+04, 5.994e+04, 5.817e+04, 5.362e+04, 5.006e+04, 4.346e+04,
     | 3.406e+04, 3.108e+04, 2.971e+04, 3.170e+04, 3.122e+04, 3.316e+04,
     | 3.253e+04, 2.906e+04, 2.922e+04, 2.791e+04, 2.699e+04, 2.711e+04,
     | 2.572e+04, 2.529e+04, 2.358e+04, 2.256e+04, 2.166e+04, 2.143e+04,
     | 1.992e+04, 1.982e+04, 1.766e+04, 1.692e+04, 1.765e+04, 1.598e+04,
     | 1.452e+04, 1.446e+04, 1.462e+04, 1.533e+04, 1.416e+04, 1.235e+04,
     | 1.089e+04, 1.192e+04, 1.188e+04, 1.078e+04, 1.031e+04, 1.077e+04,
     | 1.073e+04, 1.053e+04, 9.843e+03, 9.308e+03, 9.021e+03, 8.735e+03,
     | 8.462e+03, 8.414e+03, 7.947e+03, 7.775e+03, 6.352e+03, 5.380e+03,
     | 4.764e+03, 4.229e+03, 3.850e+03, 3.771e+03, 3.599e+03, 3.475e+03,
     | 3.195e+03, 3.087e+03, 2.907e+03, 2.844e+03, 2.600e+03, 2.478e+03,
     | 2.426e+03, 2.269e+03, 1.840e+03, 1.666e+03, 1.675e+03, 1.583e+03,
     | 1.442e+03, 1.388e+03, 1.281e+03, 1.134e+03, 1.077e+03, 1.008e+03,
     | 9.474e+02, 1.042e+03, 1.054e+03, 9.716e+02, 1.041e+03, 1.015e+03,
     | 9.065e+02, 9.758e+02, 9.098e+02, 9.063e+02, 6.932e+02, 7.855e+02,
     | 8.672e+02, 1.002e+03, 1.165e+03, 1.123e+03, 1.232e+03, 1.199e+03,
     | 1.106e+03, 1.062e+03, 1.042e+03, 1.129e+03, 1.155e+03, 1.219e+03,
     | 1.410e+03, 1.521e+03, 4.106e+03, 8.545e+03, 1.234e+04, 1.665e+04,
     | 2.180e+04, 2.886e+04, 3.638e+04, 4.268e+04, 4.548e+04, 3.954e+04,
     | 4.026e+04, 3.269e+04, 3.814e+04, 4.279e+04, 4.980e+04, 9.232e+04,
     | 8.882e+04, 1.305e+05, 1.662e+05, 1.209e+05, 7.966e+04, 6.636e+04,
     | 5.938e+04, 5.106e+04, 3.652e+04, 2.375e+04, 2.167e+04, 2.142e+04,
     | 1.975e+04, 1.474e+04, 1.453e+04, 1.194e+04, 1.384e+04, 1.191e+04,
     | 1.212e+04, 1.000e+04, 9.902e+03, 9.652e+03, 9.392e+03, 9.066e+03,
     | 7.339e+03, 1.013e+04, 1.487e+04, 1.247e+04, 1.205e+04, 1.194e+04,
     | 5.047e+03, 4.742e+03, 5.060e+03, 4.449e+03, 4.570e+03, 4.158e+03,
     | 3.421e+03, 3.296e+03, 3.219e+03, 2.975e+03, 2.808e+03, 2.767e+03
     | /)
      eo(1:240,3)=(/
     | 1.380e+01, 1.371e+01, 1.256e+01, 1.289e+01, 1.682e+01, 1.664e+01,
     | 1.439e+01, 1.336e+01, 1.374e+01, 1.267e+01, 1.242e+01, 3.500e+01,
     | 2.034e+01, 1.281e+01, 1.150e+01, 1.105e+01, 1.057e+01, 1.099e+01,
     | 1.069e+01, 1.129e+01, 1.071e+01, 9.773e+00, 9.535e+00, 9.541e+00,
     | 9.383e+00, 9.640e+00, 9.455e+00, 9.019e+00, 8.870e+00, 8.768e+00,
     | 9.140e+00, 9.430e+00, 9.440e+00, 9.443e+00, 9.546e+00, 9.679e+00,
     | 9.615e+00, 9.322e+00, 9.328e+00, 9.315e+00, 9.303e+00, 9.146e+00,
     | 8.628e+00, 8.078e+00, 7.262e+00, 6.539e+00, 6.215e+00, 6.478e+00,
     | 6.435e+00, 6.444e+00, 6.424e+00, 6.368e+00, 6.319e+00, 6.207e+00,
     | 6.133e+00, 6.117e+00, 6.085e+00, 6.077e+00, 6.064e+00, 5.990e+00,
     | 5.945e+00, 5.966e+00, 5.908e+00, 5.968e+00, 5.860e+00, 5.830e+00,
     | 5.851e+00, 5.810e+00, 5.709e+00, 5.681e+00, 5.649e+00, 5.576e+00,
     | 5.569e+00, 5.556e+00, 5.560e+00, 5.529e+00, 5.465e+00, 5.539e+00,
     | 5.555e+00, 5.596e+00, 5.585e+00, 5.596e+00, 5.578e+00, 5.580e+00,
     | 5.856e+00, 5.911e+00, 5.931e+00, 5.791e+00, 5.759e+00, 5.620e+00,
     | 5.600e+00, 5.704e+00, 5.592e+00, 5.631e+00, 5.597e+00, 5.518e+00,
     | 5.552e+00, 5.451e+00, 5.479e+00, 5.478e+00, 5.447e+00, 5.444e+00,
     | 5.480e+00, 5.450e+00, 5.556e+00, 5.589e+00, 5.447e+00, 5.545e+00,
     | 5.608e+00, 5.567e+00, 5.492e+00, 5.355e+00, 5.365e+00, 5.501e+00,
     | 5.619e+00, 5.423e+00, 5.361e+00, 5.483e+00, 5.388e+00, 5.370e+00,
     | 5.334e+00, 5.292e+00, 5.345e+00, 5.329e+00, 5.373e+00, 5.351e+00,
     | 5.385e+00, 5.338e+00, 5.388e+00, 5.443e+00, 5.474e+00, 5.547e+00,
     | 5.541e+00, 5.524e+00, 5.549e+00, 5.631e+00, 5.533e+00, 5.518e+00,
     | 5.559e+00, 5.604e+00, 5.648e+00, 5.625e+00, 5.690e+00, 5.807e+00,
     | 5.682e+00, 5.746e+00, 5.695e+00, 5.520e+00, 5.594e+00, 5.620e+00,
     | 5.631e+00, 5.556e+00, 5.589e+00, 5.821e+00, 5.641e+00, 5.889e+00,
     | 6.016e+00, 5.981e+00, 5.952e+00, 5.961e+00, 5.828e+00, 5.730e+00,
     | 5.858e+00, 5.802e+00, 5.693e+00, 5.623e+00, 5.746e+00, 5.856e+00,
     | 5.639e+00, 5.616e+00, 5.355e+00, 5.392e+00, 5.454e+00, 5.684e+00,
     | 5.734e+00, 5.764e+00, 5.683e+00, 5.848e+00, 5.839e+00, 5.911e+00,
     | 6.147e+00, 6.561e+00, 7.093e+00, 6.936e+00, 6.708e+00, 6.672e+00,
     | 6.656e+00, 6.904e+00, 6.701e+00, 6.752e+00, 6.600e+00, 6.546e+00,
     | 6.755e+00, 6.751e+00, 6.493e+00, 6.420e+00, 6.332e+00, 6.347e+00,
     | 6.326e+00, 6.132e+00, 6.091e+00, 6.190e+00, 6.103e+00, 5.959e+00,
     | 5.859e+00, 5.779e+00, 5.761e+00, 5.912e+00, 5.885e+00, 5.461e+00,
     | 5.315e+00, 5.337e+00, 5.190e+00, 5.244e+00, 5.179e+00, 5.223e+00,
     | 5.100e+00, 5.045e+00, 4.910e+00, 4.899e+00, 4.866e+00, 4.823e+00,
     | 4.843e+00, 4.640e+00, 4.738e+00, 4.955e+00, 4.613e+00, 4.516e+00,
     | 4.521e+00, 4.404e+00, 4.562e+00, 4.700e+00, 4.514e+00, 4.459e+00,
     | 4.328e+00, 4.411e+00, 4.299e+00, 4.396e+00, 4.220e+00, 4.376e+00
     | /)
      iday(1:240,3)=(/
     |  267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267,
     |  267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267, 267,
     |  268, 268, 268, 268, 268, 268, 268, 268, 268, 268, 268, 268,
     |  268, 268, 268, 268, 268, 268, 268, 268, 268, 268, 268, 268,
     |  269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269,
     |  269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269, 269,
     |  270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270,
     |  270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270, 270,
     |  271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271,
     |  271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271, 271,
     |  272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272,
     |  272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272, 272,
     |  273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273,
     |  273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273, 273,
     |  274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 274,
     |  274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 274, 274,
     |  275, 275, 275, 275, 275, 275, 275, 275, 275, 275, 275, 275,
     |  275, 275, 275, 275, 275, 275, 275, 275, 275, 275, 275, 275,
     |  276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276,
     |  276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276, 276
     | /)
      hour(1:240,3)=(/
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
!
! Proton flux parameters for November 4-10, 2001 Solar Proton Event
      fe(1:168,4)=(/
     | 2.003e+01, 1.678e+01, 1.733e+01, 2.371e+01, 1.991e+01, 1.649e+01,
     | 1.863e+01, 2.049e+01, 1.750e+01, 1.578e+01, 1.612e+01, 1.649e+01,
     | 2.018e+01, 1.710e+01, 2.772e+01, 2.172e+01, 2.417e+02, 3.039e+04,
     | 8.204e+04, 1.028e+05, 1.431e+05, 1.086e+05, 1.208e+05, 1.490e+05,
     | 1.847e+05, 1.903e+05, 1.917e+05, 2.567e+05, 2.790e+05, 2.308e+05,
     | 1.896e+05, 2.456e+05, 2.995e+05, 2.765e+05, 3.058e+05, 3.359e+05,
     | 2.182e+05, 2.355e+05, 3.234e+05, 3.099e+05, 4.059e+05, 6.510e+05,
     | 7.449e+05, 5.788e+05, 8.605e+05, 1.699e+06, 1.894e+06, 2.009e+06,
     | 2.115e+06, 2.069e+06, 2.509e+06, 1.786e+06, 2.115e+06, 2.231e+06,
     | 8.112e+05, 4.410e+05, 3.943e+05, 2.590e+05, 2.551e+05, 2.206e+05,
     | 2.175e+05, 1.442e+05, 1.141e+05, 1.050e+05, 7.540e+04, 7.110e+04,
     | 6.655e+04, 6.074e+04, 5.786e+04, 5.589e+04, 5.063e+04, 4.706e+04,
     | 4.567e+04, 4.738e+04, 4.421e+04, 4.114e+04, 3.789e+04, 3.372e+04,
     | 3.132e+04, 3.289e+04, 3.088e+04, 3.198e+04, 3.235e+04, 3.215e+04,
     | 2.866e+04, 2.735e+04, 2.506e+04, 2.257e+04, 1.919e+04, 1.991e+04,
     | 1.832e+04, 1.495e+04, 1.469e+04, 1.566e+04, 1.216e+04, 1.063e+04,
     | 1.073e+04, 1.275e+04, 9.742e+03, 9.941e+03, 1.281e+04, 1.203e+04,
     | 1.124e+04, 1.113e+04, 1.011e+04, 8.075e+03, 7.979e+03, 7.244e+03,
     | 6.855e+03, 5.782e+03, 4.819e+03, 4.120e+03, 3.694e+03, 3.554e+03,
     | 3.423e+03, 3.246e+03, 3.095e+03, 3.372e+03, 3.271e+03, 2.655e+03,
     | 2.711e+03, 3.185e+03, 2.656e+03, 2.785e+03, 2.116e+03, 1.589e+03,
     | 1.189e+03, 1.165e+03, 1.203e+03, 1.273e+03, 1.336e+03, 1.157e+03,
     | 9.881e+02, 9.799e+02, 8.816e+02, 8.940e+02, 8.908e+02, 9.610e+02,
     | 9.035e+02, 8.257e+02, 9.833e+02, 1.113e+03, 1.104e+03, 1.092e+03,
     | 1.116e+03, 1.121e+03, 1.095e+03, 1.103e+03, 1.048e+03, 9.960e+02,
     | 9.345e+02, 9.003e+02, 8.899e+02, 9.176e+02, 8.732e+02, 8.379e+02,
     | 7.442e+02, 6.475e+02, 6.119e+02, 6.609e+02, 6.049e+02, 6.188e+02,
     | 6.774e+02, 6.334e+02, 6.159e+02, 7.293e+02, 6.841e+02, 6.358e+02
     | /)
      eo(1:168,4)=(/
     | 1.334e+01, 1.471e+01, 1.539e+01, 1.242e+01, 1.233e+01, 1.355e+01,
     | 1.435e+01, 1.628e+01, 1.555e+01, 1.218e+01, 1.267e+01, 1.421e+01,
     | 1.673e+01, 1.490e+01, 1.317e+01, 1.629e+01, 4.753e+01, 9.127e+01,
     | 8.036e+01, 2.853e+01, 1.649e+01, 1.461e+01, 1.403e+01, 1.288e+01,
     | 1.112e+01, 1.086e+01, 1.048e+01, 1.107e+01, 1.095e+01, 1.047e+01,
     | 9.886e+00, 1.009e+01, 1.039e+01, 9.979e+00, 9.579e+00, 9.871e+00,
     | 9.977e+00, 9.309e+00, 9.043e+00, 8.661e+00, 9.003e+00, 9.925e+00,
     | 9.375e+00, 8.879e+00, 9.462e+00, 9.188e+00, 8.814e+00, 8.669e+00,
     | 8.505e+00, 8.467e+00, 8.343e+00, 8.332e+00, 7.919e+00, 7.905e+00,
     | 7.214e+00, 6.636e+00, 6.407e+00, 6.123e+00, 6.128e+00, 6.119e+00,
     | 6.221e+00, 6.171e+00, 6.682e+00, 6.681e+00, 7.095e+00, 6.818e+00,
     | 6.768e+00, 6.731e+00, 6.747e+00, 6.598e+00, 6.598e+00, 6.559e+00,
     | 6.477e+00, 6.510e+00, 6.419e+00, 6.409e+00, 6.405e+00, 6.503e+00,
     | 6.532e+00, 6.419e+00, 6.402e+00, 6.347e+00, 6.282e+00, 6.083e+00,
     | 5.959e+00, 5.931e+00, 5.962e+00, 6.015e+00, 6.143e+00, 6.002e+00,
     | 5.957e+00, 6.170e+00, 6.004e+00, 5.781e+00, 5.960e+00, 6.178e+00,
     | 5.969e+00, 5.536e+00, 5.902e+00, 5.763e+00, 5.314e+00, 5.274e+00,
     | 5.238e+00, 5.173e+00, 5.169e+00, 5.387e+00, 5.360e+00, 5.315e+00,
     | 5.294e+00, 5.333e+00, 5.591e+00, 5.553e+00, 5.678e+00, 5.621e+00,
     | 5.568e+00, 5.553e+00, 5.508e+00, 5.242e+00, 5.142e+00, 5.489e+00,
     | 5.360e+00, 5.129e+00, 5.262e+00, 5.066e+00, 5.310e+00, 5.582e+00,
     | 5.862e+00, 6.088e+00, 5.917e+00, 5.759e+00, 5.643e+00, 5.744e+00,
     | 6.000e+00, 6.028e+00, 5.660e+00, 5.942e+00, 5.920e+00, 5.898e+00,
     | 5.537e+00, 5.643e+00, 5.368e+00, 5.169e+00, 5.175e+00, 5.268e+00,
     | 5.423e+00, 5.327e+00, 5.262e+00, 5.397e+00, 5.452e+00, 5.626e+00,
     | 5.543e+00, 5.305e+00, 5.357e+00, 5.468e+00, 5.273e+00, 5.552e+00,
     | 5.362e+00, 5.308e+00, 5.717e+00, 5.267e+00, 5.522e+00, 5.563e+00,
     | 5.490e+00, 5.209e+00, 5.086e+00, 5.179e+00, 5.190e+00, 5.308e+00
     | /)
      iday(1:168,4)=(/
     |  308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308,
     |  308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308,
     |  309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309,
     |  309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309,
     |  310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310,
     |  310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310,
     |  311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311,
     |  311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311,
     |  312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312,
     |  312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312, 312,
     |  313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313,
     |  313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313, 313,
     |  314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314,
     |  314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314, 314
     | /)
      hour(1:168,4)=(/
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
!
! Proton flux parameters for November 22-27, 2001 Solar Proton Event
      fe(1:144,5)=(/
     | 4.481e+02, 3.458e+02, 4.009e+02, 3.790e+02, 3.685e+02, 2.899e+02,
     | 3.003e+02, 2.886e+02, 2.571e+02, 2.322e+02, 2.396e+02, 2.097e+02,
     | 1.895e+02, 1.916e+02, 1.808e+02, 1.512e+02, 1.437e+02, 1.291e+02,
     | 1.176e+02, 1.281e+02, 1.272e+02, 2.383e+02, 7.998e+02, 1.398e+03,
     | 2.098e+03, 5.164e+03, 1.539e+04, 2.959e+04, 5.659e+04, 9.295e+04,
     | 1.237e+05, 1.813e+05, 2.173e+05, 2.397e+05, 2.976e+05, 3.109e+05,
     | 3.298e+05, 3.257e+05, 3.166e+05, 3.268e+05, 3.288e+05, 3.247e+05,
     | 3.217e+05, 3.258e+05, 3.342e+05, 3.239e+05, 3.045e+05, 3.240e+05,
     | 3.611e+05, 4.521e+05, 5.760e+05, 7.038e+05, 9.063e+05, 1.525e+06,
     | 1.317e+06, 8.293e+05, 7.246e+05, 6.599e+05, 5.754e+05, 4.098e+05,
     | 2.747e+05, 2.625e+05, 1.793e+05, 1.721e+05, 1.188e+05, 8.612e+04,
     | 4.842e+04, 4.007e+04, 3.209e+04, 3.098e+04, 2.896e+04, 1.995e+04,
     | 2.160e+04, 2.137e+04, 2.091e+04, 2.189e+04, 2.076e+04, 1.878e+04,
     | 1.583e+04, 1.419e+04, 1.252e+04, 1.641e+04, 1.244e+04, 1.273e+04,
     | 1.050e+04, 9.253e+03, 7.753e+03, 7.431e+03, 7.646e+03, 6.618e+03,
     | 5.782e+03, 4.773e+03, 4.050e+03, 3.718e+03, 4.489e+03, 5.255e+03,
     | 5.315e+03, 5.030e+03, 4.015e+03, 3.741e+03, 3.294e+03, 2.870e+03,
     | 3.067e+03, 3.629e+03, 3.049e+03, 2.705e+03, 2.504e+03, 2.191e+03,
     | 2.051e+03, 1.883e+03, 1.620e+03, 1.360e+03, 1.281e+03, 1.211e+03,
     | 1.292e+03, 1.199e+03, 1.125e+03, 1.032e+03, 9.444e+02, 1.139e+03,
     | 1.683e+03, 1.712e+03, 1.577e+03, 1.431e+03, 1.327e+03, 1.214e+03,
     | 1.166e+03, 1.167e+03, 1.143e+03, 1.063e+03, 9.931e+02, 9.465e+02,
     | 8.502e+02, 7.575e+02, 7.253e+02, 6.928e+02, 7.234e+02, 6.842e+02,
     | 6.699e+02, 6.783e+02, 7.397e+02, 7.043e+02, 7.528e+02, 9.195e+02
     | /)
      eo(1:144,5)=(/
     | 4.170e+00, 4.051e+00, 4.022e+00, 4.286e+00, 4.570e+00, 4.387e+00,
     | 4.547e+00, 4.441e+00, 4.436e+00, 4.663e+00, 4.725e+00, 4.665e+00,
     | 5.021e+00, 4.926e+00, 4.748e+00, 5.222e+00, 5.345e+00, 5.244e+00,
     | 5.151e+00, 5.091e+00, 4.926e+00, 9.413e+00, 1.612e+01, 1.522e+01,
     | 1.327e+01, 1.214e+01, 9.950e+00, 8.652e+00, 8.083e+00, 8.222e+00,
     | 8.319e+00, 8.430e+00, 8.530e+00, 8.235e+00, 8.501e+00, 8.446e+00,
     | 8.393e+00, 7.948e+00, 7.677e+00, 7.608e+00, 7.153e+00, 6.901e+00,
     | 6.739e+00, 6.573e+00, 6.457e+00, 6.326e+00, 6.212e+00, 6.135e+00,
     | 6.427e+00, 6.421e+00, 6.222e+00, 6.076e+00, 5.911e+00, 5.410e+00,
     | 5.277e+00, 5.435e+00, 5.437e+00, 5.307e+00, 5.274e+00, 5.182e+00,
     | 5.010e+00, 4.890e+00, 4.924e+00, 4.881e+00, 4.922e+00, 4.997e+00,
     | 5.239e+00, 5.129e+00, 5.074e+00, 4.902e+00, 4.770e+00, 4.861e+00,
     | 4.735e+00, 4.578e+00, 4.584e+00, 4.443e+00, 4.354e+00, 4.339e+00,
     | 4.332e+00, 4.285e+00, 4.303e+00, 4.219e+00, 4.386e+00, 4.393e+00,
     | 4.600e+00, 4.397e+00, 4.437e+00, 4.460e+00, 4.326e+00, 4.354e+00,
     | 4.362e+00, 4.476e+00, 4.611e+00, 4.609e+00, 4.526e+00, 4.381e+00,
     | 4.567e+00, 4.511e+00, 4.711e+00, 4.779e+00, 4.757e+00, 4.888e+00,
     | 4.855e+00, 4.694e+00, 4.783e+00, 4.822e+00, 4.891e+00, 5.106e+00,
     | 5.004e+00, 4.989e+00, 5.055e+00, 5.278e+00, 5.473e+00, 5.400e+00,
     | 5.260e+00, 5.228e+00, 5.084e+00, 5.038e+00, 5.012e+00, 4.831e+00,
     | 4.516e+00, 4.368e+00, 4.504e+00, 4.561e+00, 4.550e+00, 4.587e+00,
     | 4.441e+00, 4.311e+00, 4.704e+00, 4.716e+00, 4.589e+00, 4.729e+00,
     | 4.914e+00, 5.041e+00, 4.943e+00, 5.175e+00, 4.976e+00, 4.901e+00,
     | 4.934e+00, 4.939e+00, 4.985e+00, 4.728e+00, 4.364e+00, 4.544e+00
     | /)
      iday(1:144,5)=(/
     |  326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326,
     |  326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326, 326,
     |  327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327,
     |  327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327, 327,
     |  328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328,
     |  328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328, 328,
     |  329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329,
     |  329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329, 329,
     |  330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330,
     |  330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330, 330,
     |  331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331,
     |  331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331, 331
     | /)
      hour(1:144,5)=(/
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
!
! For April, 2002:
      fe(1:336,6)=(/
     | 2.099e+01, 1.784e+01, 2.031e+01, 2.282e+01, 2.263e+01, 2.065e+01,
     | 2.068e+01, 2.598e+01, 2.164e+01, 3.633e+01, 3.724e+01, 8.233e+01,
     | 4.004e+02, 1.395e+03, 9.994e+02, 1.734e+03, 1.658e+03, 2.083e+03,
     | 2.128e+03, 1.799e+03, 2.279e+03, 2.453e+03, 2.581e+03, 2.270e+03,
     | 2.258e+03, 1.833e+03, 1.871e+03, 1.593e+03, 1.979e+03, 1.455e+03,
     | 1.085e+03, 1.045e+03, 9.711e+02, 9.570e+02, 8.752e+02, 7.338e+02,
     | 5.774e+02, 4.877e+02, 5.248e+02, 4.357e+02, 4.105e+02, 4.361e+02,
     | 3.984e+02, 4.143e+02, 4.671e+02, 6.433e+02, 5.334e+02, 4.522e+02,
     | 4.782e+02, 4.762e+02, 4.550e+02, 4.544e+02, 4.344e+02, 5.673e+02,
     | 6.707e+02, 9.610e+02, 1.580e+03, 5.863e+02, 3.556e+02, 3.327e+02,
     | 3.418e+02, 3.311e+02, 3.806e+02, 5.381e+02, 8.602e+02, 1.553e+03,
     | 1.356e+03, 1.519e+03, 1.561e+03, 1.286e+03, 1.100e+03, 1.004e+03,
     | 8.875e+02, 4.203e+02, 1.744e+02, 1.320e+02, 1.198e+02, 1.174e+02,
     | 1.177e+02, 1.271e+02, 1.062e+02, 1.095e+02, 9.124e+01, 1.132e+02,
     | 9.456e+01, 7.999e+01, 8.364e+01, 6.188e+01, 5.878e+01, 6.253e+01,
     | 6.510e+01, 6.090e+01, 4.644e+01, 6.037e+01, 4.620e+01, 5.405e+01,
     | 4.665e+01, 8.873e+01, 5.094e+03, 4.035e+04, 8.039e+04, 1.032e+05,
     | 1.264e+05, 1.249e+05, 1.547e+05, 1.602e+05, 1.775e+05, 1.628e+05,
     | 1.685e+05, 1.770e+05, 1.678e+05, 1.524e+05, 1.429e+05, 1.408e+05,
     | 1.408e+05, 1.341e+05, 1.596e+05, 1.727e+05, 2.004e+05, 2.172e+05,
     | 1.994e+05, 1.729e+05, 1.728e+05, 1.641e+05, 1.499e+05, 1.631e+05,
     | 1.579e+05, 1.471e+05, 1.235e+05, 1.083e+05, 9.808e+04, 8.865e+04,
     | 7.924e+04, 6.657e+04, 5.810e+04, 5.582e+04, 5.287e+04, 4.784e+04,
     | 4.533e+04, 4.268e+04, 3.984e+04, 3.489e+04, 3.264e+04, 2.993e+04,
     | 2.700e+04, 2.518e+04, 2.497e+04, 2.451e+04, 2.383e+04, 2.042e+04,
     | 1.809e+04, 1.583e+04, 1.425e+04, 1.193e+04, 1.124e+04, 1.090e+04,
     | 1.028e+04, 8.767e+03, 9.590e+03, 1.074e+04, 1.017e+04, 9.541e+03,
     | 9.106e+03, 9.420e+03, 9.197e+03, 9.395e+03, 9.734e+03, 1.008e+04,
     | 9.903e+03, 9.664e+03, 9.788e+03, 9.725e+03, 9.295e+03, 8.350e+03,
     | 7.792e+03, 7.593e+03, 7.208e+03, 6.576e+03, 5.973e+03, 4.561e+03,
     | 4.439e+03, 4.179e+03, 3.616e+03, 2.762e+03, 2.371e+03, 2.242e+03,
     | 2.122e+03, 1.998e+03, 2.110e+03, 2.071e+03, 1.997e+03, 1.936e+03,
     | 1.839e+03, 1.690e+03, 1.548e+03, 1.528e+03, 1.470e+03, 1.469e+03,
     | 1.366e+03, 1.254e+03, 1.218e+03, 1.201e+03, 1.173e+03, 1.174e+03,
     | 1.094e+03, 1.010e+03, 9.398e+02, 8.913e+02, 9.668e+02, 9.752e+02,
     | 8.941e+02, 7.212e+02, 8.199e+02, 8.087e+02, 7.746e+02, 7.932e+02,
     | 7.844e+02, 8.538e+02, 8.430e+02, 8.298e+02, 8.320e+02, 8.104e+02,
     | 7.955e+02, 7.559e+02, 7.507e+02, 7.368e+02, 6.003e+02, 5.422e+02,
     | 5.095e+02, 4.521e+02, 4.108e+02, 3.750e+02, 4.010e+02, 3.719e+02,
     | 3.912e+02, 4.092e+02, 4.669e+02, 4.743e+02, 4.328e+02, 4.878e+02,
     | 4.821e+02, 4.630e+02, 4.619e+02, 4.500e+02, 4.501e+02, 4.161e+02,
     | 4.054e+02, 4.054e+02, 3.964e+02, 3.856e+02, 3.729e+02, 3.570e+02,
     | 3.571e+02, 3.538e+02, 3.397e+02, 3.454e+02, 3.126e+02, 2.836e+02,
     | 2.891e+02, 2.822e+02, 2.700e+02, 2.511e+02, 2.041e+02, 2.141e+02,
     | 2.040e+02, 1.901e+02, 1.711e+02, 1.707e+02, 1.569e+02, 1.604e+02,
     | 1.551e+02, 1.574e+02, 1.723e+02, 1.690e+02, 1.699e+02, 1.766e+02,
     | 1.368e+02, 1.169e+02, 1.221e+02, 1.113e+02, 1.043e+02, 1.130e+02,
     | 1.069e+02, 1.042e+02, 9.219e+01, 9.797e+01, 9.369e+01, 9.203e+01,
     | 1.019e+02, 9.482e+01, 7.422e+01, 6.881e+01, 7.018e+01, 7.597e+01,
     | 6.634e+01, 7.728e+01, 7.047e+01, 7.055e+01, 6.165e+01, 7.185e+01,
     | 7.305e+01, 7.389e+01, 5.309e+01, 6.924e+01, 6.508e+01, 6.457e+01,
     | 6.187e+01, 6.951e+01, 6.490e+01, 7.266e+01, 7.643e+01, 5.670e+01,
     | 5.721e+01, 5.996e+01, 5.664e+01, 6.098e+01, 5.421e+01, 5.482e+01,
     | 6.242e+01, 6.649e+01, 6.228e+01, 5.664e+01, 5.051e+01, 5.851e+01,
     | 5.427e+01, 6.320e+01, 5.341e+01, 5.035e+01, 5.503e+01, 5.837e+01,
     | 5.452e+01, 5.430e+01, 5.363e+01, 5.239e+01, 6.233e+01, 6.123e+01
     | /)
!
! For April, 2002:
      eo(1:336,6)=(/
     | 1.156e+01, 1.250e+01, 1.275e+01, 1.045e+01, 9.804e+00, 1.203e+01,
     | 1.367e+01, 1.043e+01, 9.943e+00, 7.931e+00, 8.336e+00, 6.363e+00,
     | 5.214e+00, 4.721e+00, 4.892e+00, 4.792e+00, 4.902e+00, 4.546e+00,
     | 4.498e+00, 4.606e+00, 4.639e+00, 4.106e+00, 4.003e+00, 3.793e+00,
     | 3.838e+00, 3.929e+00, 3.820e+00, 3.757e+00, 3.603e+00, 3.829e+00,
     | 3.985e+00, 3.916e+00, 3.801e+00, 3.902e+00, 3.852e+00, 4.094e+00,
     | 3.946e+00, 4.068e+00, 3.927e+00, 4.397e+00, 4.143e+00, 4.343e+00,
     | 4.300e+00, 4.371e+00, 4.199e+00, 4.214e+00, 4.038e+00, 4.225e+00,
     | 4.411e+00, 4.292e+00, 4.433e+00, 4.275e+00, 4.131e+00, 4.081e+00,
     | 4.033e+00, 3.736e+00, 3.567e+00, 4.097e+00, 4.093e+00, 4.298e+00,
     | 4.237e+00, 4.132e+00, 4.159e+00, 4.050e+00, 3.759e+00, 3.545e+00,
     | 3.468e+00, 3.466e+00, 3.660e+00, 3.450e+00, 3.575e+00, 3.676e+00,
     | 3.822e+00, 3.988e+00, 4.512e+00, 4.874e+00, 5.095e+00, 4.845e+00,
     | 4.979e+00, 4.856e+00, 5.228e+00, 5.100e+00, 5.196e+00, 5.209e+00,
     | 5.517e+00, 5.315e+00, 5.419e+00, 6.042e+00, 6.077e+00, 6.402e+00,
     | 5.927e+00, 5.965e+00, 5.979e+00, 5.776e+00, 6.396e+00, 6.272e+00,
     | 6.344e+00, 1.213e+01, 3.399e+01, 1.698e+01, 1.285e+01, 1.190e+01,
     | 1.224e+01, 1.292e+01, 1.230e+01, 1.182e+01, 1.134e+01, 1.137e+01,
     | 1.087e+01, 1.036e+01, 1.039e+01, 1.049e+01, 1.054e+01, 1.062e+01,
     | 1.050e+01, 1.033e+01, 8.914e+00, 8.139e+00, 7.524e+00, 6.882e+00,
     | 6.839e+00, 6.973e+00, 6.840e+00, 6.842e+00, 6.940e+00, 6.648e+00,
     | 6.594e+00, 6.680e+00, 6.830e+00, 7.018e+00, 7.097e+00, 7.259e+00,
     | 7.515e+00, 7.995e+00, 8.296e+00, 8.185e+00, 7.954e+00, 7.917e+00,
     | 7.972e+00, 8.011e+00, 8.047e+00, 8.192e+00, 8.311e+00, 8.287e+00,
     | 8.273e+00, 8.418e+00, 8.424e+00, 8.369e+00, 8.288e+00, 8.369e+00,
     | 8.438e+00, 8.303e+00, 8.563e+00, 8.482e+00, 8.472e+00, 8.402e+00,
     | 8.483e+00, 8.421e+00, 8.429e+00, 8.316e+00, 8.221e+00, 8.306e+00,
     | 8.261e+00, 8.287e+00, 8.132e+00, 7.887e+00, 7.801e+00, 7.729e+00,
     | 7.726e+00, 7.622e+00, 7.499e+00, 7.559e+00, 7.502e+00, 7.331e+00,
     | 7.292e+00, 7.278e+00, 7.349e+00, 7.309e+00, 7.361e+00, 7.603e+00,
     | 7.458e+00, 7.321e+00, 7.478e+00, 7.835e+00, 8.049e+00, 7.889e+00,
     | 7.650e+00, 7.683e+00, 7.485e+00, 7.314e+00, 7.312e+00, 7.314e+00,
     | 7.266e+00, 7.253e+00, 7.353e+00, 7.377e+00, 7.431e+00, 7.385e+00,
     | 7.336e+00, 7.370e+00, 7.331e+00, 7.561e+00, 7.427e+00, 7.409e+00,
     | 7.468e+00, 7.752e+00, 8.234e+00, 7.948e+00, 7.776e+00, 7.557e+00,
     | 7.585e+00, 7.858e+00, 7.660e+00, 7.449e+00, 7.646e+00, 7.712e+00,
     | 7.371e+00, 7.144e+00, 7.037e+00, 6.963e+00, 6.989e+00, 7.052e+00,
     | 7.030e+00, 7.039e+00, 6.766e+00, 6.883e+00, 7.082e+00, 7.336e+00,
     | 7.678e+00, 8.015e+00, 8.283e+00, 8.328e+00, 8.139e+00, 8.281e+00,
     | 7.740e+00, 7.617e+00, 7.330e+00, 6.986e+00, 7.091e+00, 6.916e+00,
     | 6.703e+00, 6.871e+00, 6.672e+00, 6.803e+00, 6.874e+00, 6.759e+00,
     | 6.643e+00, 6.705e+00, 6.570e+00, 6.613e+00, 6.602e+00, 6.813e+00,
     | 6.742e+00, 7.049e+00, 6.649e+00, 7.140e+00, 6.578e+00, 7.056e+00,
     | 6.934e+00, 6.931e+00, 6.717e+00, 6.983e+00, 7.136e+00, 6.896e+00,
     | 7.210e+00, 6.982e+00, 6.576e+00, 7.254e+00, 6.896e+00, 8.133e+00,
     | 6.755e+00, 6.526e+00, 7.617e+00, 7.345e+00, 7.238e+00, 7.432e+00,
     | 7.793e+00, 7.789e+00, 7.764e+00, 9.048e+00, 9.332e+00, 8.166e+00,
     | 8.018e+00, 8.097e+00, 8.404e+00, 7.994e+00, 8.495e+00, 7.832e+00,
     | 7.661e+00, 7.862e+00, 9.013e+00, 8.314e+00, 8.717e+00, 8.830e+00,
     | 8.557e+00, 8.776e+00, 8.604e+00, 9.199e+00, 9.638e+00, 1.049e+01,
     | 9.385e+00, 8.287e+00, 9.086e+00, 8.765e+00, 8.613e+00, 9.013e+00,
     | 8.337e+00, 7.385e+00, 8.015e+00, 9.289e+00, 8.565e+00, 8.230e+00,
     | 8.977e+00, 7.686e+00, 9.437e+00, 9.481e+00, 8.564e+00, 8.453e+00,
     | 9.457e+00, 8.314e+00, 9.364e+00, 9.116e+00, 7.842e+00, 8.717e+00,
     | 8.608e+00, 8.630e+00, 9.877e+00, 8.562e+00, 8.858e+00, 9.914e+00,
     | 8.036e+00, 8.669e+00, 8.652e+00, 9.215e+00, 8.462e+00, 7.568e+00
     | /)
!
! For April, 2002:
      iday(1:336,6)=(/
     | 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107,
     | 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107, 107,
     | 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108,
     | 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108, 108,
     | 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109,
     | 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109, 109,
     | 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110,
     | 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110, 110,
     | 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111,
     | 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111, 111,
     | 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112,
     | 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112, 112,
     | 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113,
     | 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113, 113,
     | 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114,
     | 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114, 114,
     | 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115,
     | 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115, 115,
     | 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116,
     | 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116, 116,
     | 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117,
     | 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117, 117,
     | 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118,
     | 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118, 118,
     | 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119,
     | 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119, 119,
     | 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120,
     | 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120, 120
     | /)
!
! For April, 2002:
      hour(1:336,6)=(/
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     | 0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     |12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
!
! Proton flux parameters for October 26  November 7, 2003, Solar Eruptions
      fe(1:312,7)=(/
     | 1.873e+01, 4.627e+01, 5.817e+01, 4.375e+01, 2.683e+01, 1.938e+01,
     | 2.687e+01, 3.186e+01, 3.517e+01, 2.669e+01, 3.985e+01, 2.389e+01,
     | 2.591e+01, 2.595e+01, 2.436e+01, 2.541e+01, 1.970e+01, 1.891e+01,
     | 2.854e+03, 1.402e+04, 2.665e+04, 1.957e+04, 3.698e+04, 2.716e+04,
     | 2.690e+04, 3.653e+04, 3.837e+04, 3.292e+04, 3.091e+04, 2.957e+04,
     | 2.265e+04, 1.551e+04, 1.227e+04, 8.099e+03, 5.596e+03, 7.887e+03,
     | 6.789e+03, 4.985e+03, 2.637e+03, 2.261e+03, 1.644e+03, 1.734e+03,
     | 1.420e+03, 1.050e+03, 7.201e+02, 5.467e+02, 4.374e+02, 2.766e+02,
     | 2.374e+02, 3.391e+02, 5.829e+02, 7.159e+02, 6.682e+02, 6.047e+02,
     | 5.481e+02, 4.766e+02, 4.497e+02, 4.116e+02, 2.707e+02, 3.674e+02,
     | 9.665e+03, 8.931e+04, 1.995e+05, 3.683e+05, 4.747e+05, 5.365e+05,
     | 5.374e+05, 5.002e+05, 5.450e+05, 5.812e+05, 6.293e+05, 8.504e+05,
     | 1.067e+06, 1.204e+06, 1.298e+06, 1.487e+06, 1.870e+06, 2.229e+06,
     | 2.676e+06, 2.186e+06, 2.168e+06, 1.555e+06, 1.345e+06, 7.587e+05,
     | 3.711e+05, 1.371e+05, 6.887e+04, 9.668e+04, 7.936e+04, 7.504e+04,
     | 8.503e+04, 1.247e+05, 1.076e+05, 8.718e+04, 1.496e+05, 2.364e+05,
     | 2.297e+05, 1.751e+05, 2.101e+05, 2.089e+05, 1.693e+05, 1.496e+05,
     | 1.459e+05, 1.390e+05, 1.465e+05, 1.484e+05, 1.914e+05, 2.380e+05,
     | 2.218e+05, 2.264e+05, 2.273e+05, 2.327e+05, 2.492e+05, 2.047e+05,
     | 1.511e+05, 2.455e+05, 2.195e+05, 1.261e+05, 7.604e+04, 6.336e+04,
     | 6.465e+04, 4.289e+04, 2.580e+04, 1.517e+04, 1.327e+04, 1.164e+04,
     | 9.800e+03, 9.681e+03, 7.231e+03, 6.428e+03, 8.620e+03, 5.938e+03,
     | 4.228e+03, 3.441e+03, 3.229e+03, 2.900e+03, 2.670e+03, 2.144e+03,
     | 2.048e+03, 1.984e+03, 1.731e+03, 1.540e+03, 1.695e+03, 1.840e+03,
     | 1.726e+03, 1.319e+03, 1.448e+03, 1.490e+03, 1.271e+03, 1.316e+03,
     | 1.332e+03, 1.208e+03, 1.187e+03, 1.201e+03, 1.137e+03, 9.777e+02,
     | 9.680e+02, 1.036e+03, 9.558e+02, 7.674e+02, 6.234e+02, 6.889e+02,
     | 7.161e+02, 6.533e+02, 6.882e+02, 6.910e+02, 5.220e+02, 6.374e+02,
     | 6.106e+02, 6.892e+02, 7.503e+02, 7.659e+02, 7.880e+02, 6.797e+02,
     | 6.802e+02, 7.906e+02, 7.588e+02, 7.924e+02, 8.610e+02, 1.386e+03,
     | 1.628e+03, 2.401e+03, 3.131e+03, 2.858e+03, 2.229e+03, 2.904e+03,
     | 2.336e+04, 5.342e+04, 6.159e+04, 7.832e+04, 1.114e+05, 1.150e+05,
     | 9.806e+04, 1.014e+05, 9.676e+04, 7.826e+04, 8.655e+04, 1.084e+05,
     | 1.143e+05, 1.202e+05, 1.253e+05, 1.237e+05, 1.211e+05, 1.112e+05,
     | 1.101e+05, 1.362e+05, 1.307e+05, 1.130e+05, 8.655e+04, 9.107e+04,
     | 7.468e+04, 9.234e+04, 8.018e+04, 9.526e+04, 1.147e+05, 9.674e+04,
     | 7.553e+04, 6.092e+04, 5.260e+04, 5.812e+04, 4.499e+04, 5.814e+04,
     | 5.779e+04, 3.530e+04, 1.795e+04, 1.814e+04, 1.948e+04, 1.463e+04,
     | 1.360e+04, 1.214e+04, 9.505e+03, 6.888e+03, 5.833e+03, 4.164e+03,
     | 3.156e+03, 2.653e+03, 1.932e+03, 1.532e+03, 1.653e+03, 3.641e+03,
     | 7.449e+03, 9.055e+03, 9.758e+03, 1.358e+04, 1.737e+04, 2.566e+04,
     | 3.069e+04, 3.287e+04, 3.244e+04, 2.842e+04, 2.631e+04, 2.199e+04,
     | 1.966e+04, 1.754e+04, 1.358e+04, 1.176e+04, 1.054e+04, 9.645e+03,
     | 9.519e+03, 8.776e+03, 8.927e+03, 8.860e+03, 7.362e+03, 9.630e+03,
     | 1.020e+04, 8.881e+03, 8.203e+03, 6.840e+03, 6.597e+03, 6.950e+03,
     | 6.327e+03, 5.904e+03, 5.425e+03, 5.038e+03, 4.554e+03, 4.480e+03,
     | 4.610e+03, 4.060e+03, 3.359e+03, 2.834e+03, 3.374e+03, 3.024e+03,
     | 2.731e+03, 2.477e+03, 2.345e+03, 2.170e+03, 2.241e+03, 1.955e+03,
     | 1.920e+03, 1.760e+03, 1.503e+03, 1.149e+03, 1.183e+03, 1.231e+03,
     | 1.185e+03, 1.174e+03, 1.218e+03, 1.145e+03, 1.119e+03, 1.076e+03,
     | 1.009e+03, 9.681e+02, 8.907e+02, 8.866e+02, 8.663e+02, 7.556e+02,
     | 7.760e+02, 9.235e+02, 9.474e+02, 8.990e+02, 7.981e+02, 7.896e+02
     | /)
      eo(1:312,7)=(/
     | 9.556e+00, 8.038e+00, 6.279e+00, 6.678e+00, 9.293e+00, 9.540e+00,
     | 8.985e+00, 7.509e+00, 7.158e+00, 8.824e+00, 7.857e+00, 9.045e+00,
     | 9.611e+00, 8.799e+00, 1.057e+01, 1.205e+01, 1.039e+01, 1.241e+01,
     | 1.385e+01, 8.821e+00, 6.600e+00, 6.195e+00, 5.168e+00, 5.883e+00,
     | 6.142e+00, 5.806e+00, 5.779e+00, 5.870e+00, 5.677e+00, 5.601e+00,
     | 5.495e+00, 5.797e+00, 5.397e+00, 5.079e+00, 5.247e+00, 4.908e+00,
     | 4.890e+00, 4.896e+00, 5.372e+00, 5.461e+00, 5.621e+00, 5.381e+00,
     | 5.649e+00, 5.602e+00, 6.036e+00, 6.677e+00, 6.486e+00, 7.558e+00,
     | 8.367e+00, 7.225e+00, 5.968e+00, 5.110e+00, 5.159e+00, 5.488e+00,
     | 5.208e+00, 5.233e+00, 5.333e+00, 5.021e+00, 5.551e+00, 1.086e+01,
     | 1.535e+01, 1.218e+01, 1.188e+01, 1.273e+01, 1.300e+01, 1.275e+01,
     | 1.258e+01, 1.266e+01, 1.271e+01, 1.222e+01, 1.219e+01, 1.243e+01,
     | 1.249e+01, 1.163e+01, 1.111e+01, 1.046e+01, 9.230e+00, 8.088e+00,
     | 7.395e+00, 7.039e+00, 7.173e+00, 7.066e+00, 6.900e+00, 6.753e+00,
     | 6.581e+00, 6.521e+00, 6.296e+00, 6.199e+00, 6.277e+00, 6.300e+00,
     | 6.387e+00, 6.391e+00, 6.317e+00, 7.927e+00, 1.047e+01, 1.007e+01,
     | 9.421e+00, 1.055e+01, 1.050e+01, 1.099e+01, 1.127e+01, 1.124e+01,
     | 1.097e+01, 1.078e+01, 9.887e+00, 9.513e+00, 8.721e+00, 7.918e+00,
     | 7.700e+00, 7.445e+00, 7.310e+00, 7.005e+00, 6.596e+00, 6.721e+00,
     | 6.465e+00, 5.780e+00, 5.564e+00, 5.619e+00, 5.731e+00, 5.681e+00,
     | 5.515e+00, 5.544e+00, 5.772e+00, 5.899e+00, 5.948e+00, 5.971e+00,
     | 6.072e+00, 6.045e+00, 6.192e+00, 6.217e+00, 6.049e+00, 5.981e+00,
     | 6.077e+00, 6.146e+00, 6.114e+00, 6.142e+00, 6.076e+00, 6.209e+00,
     | 6.217e+00, 6.168e+00, 6.390e+00, 6.375e+00, 6.291e+00, 6.080e+00,
     | 6.267e+00, 6.343e+00, 6.196e+00, 6.205e+00, 6.318e+00, 6.257e+00,
     | 6.249e+00, 6.301e+00, 6.281e+00, 6.172e+00, 6.257e+00, 6.184e+00,
     | 6.167e+00, 6.136e+00, 6.272e+00, 6.429e+00, 6.555e+00, 6.401e+00,
     | 6.433e+00, 6.178e+00, 6.164e+00, 6.215e+00, 6.654e+00, 6.227e+00,
     | 5.843e+00, 5.852e+00, 6.045e+00, 6.142e+00, 6.007e+00, 6.325e+00,
     | 6.326e+00, 6.244e+00, 6.088e+00, 6.119e+00, 6.420e+00, 6.582e+00,
     | 6.449e+00, 5.932e+00, 5.421e+00, 5.328e+00, 5.442e+00, 7.940e+00,
     | 3.237e+01, 2.816e+01, 1.856e+01, 1.284e+01, 1.071e+01, 9.358e+00,
     | 8.563e+00, 7.713e+00, 7.330e+00, 7.502e+00, 7.387e+00, 7.060e+00,
     | 6.805e+00, 6.603e+00, 6.386e+00, 6.180e+00, 6.089e+00, 6.009e+00,
     | 5.802e+00, 5.455e+00, 5.328e+00, 5.254e+00, 5.332e+00, 5.064e+00,
     | 5.051e+00, 4.758e+00, 4.685e+00, 4.488e+00, 4.232e+00, 4.232e+00,
     | 4.391e+00, 4.403e+00, 4.369e+00, 4.161e+00, 4.179e+00, 3.974e+00,
     | 3.908e+00, 4.040e+00, 4.213e+00, 4.185e+00, 4.222e+00, 4.281e+00,
     | 4.359e+00, 4.280e+00, 4.385e+00, 4.461e+00, 4.546e+00, 4.708e+00,
     | 4.949e+00, 5.098e+00, 5.498e+00, 5.855e+00, 6.233e+00, 5.879e+00,
     | 5.545e+00, 5.567e+00, 5.478e+00, 6.048e+00, 6.269e+00, 6.662e+00,
     | 6.916e+00, 6.603e+00, 6.636e+00, 6.518e+00, 6.483e+00, 6.356e+00,
     | 6.395e+00, 6.430e+00, 6.593e+00, 6.690e+00, 7.049e+00, 7.061e+00,
     | 6.988e+00, 6.982e+00, 6.746e+00, 6.667e+00, 7.221e+00, 6.782e+00,
     | 6.431e+00, 6.419e+00, 6.421e+00, 6.563e+00, 6.477e+00, 6.316e+00,
     | 6.236e+00, 6.259e+00, 6.341e+00, 6.352e+00, 6.447e+00, 6.383e+00,
     | 6.159e+00, 6.186e+00, 6.351e+00, 6.509e+00, 6.109e+00, 6.074e+00,
     | 6.133e+00, 6.062e+00, 5.991e+00, 5.858e+00, 5.694e+00, 5.857e+00,
     | 5.800e+00, 5.724e+00, 5.728e+00, 5.967e+00, 6.028e+00, 5.730e+00,
     | 5.784e+00, 5.582e+00, 5.596e+00, 5.709e+00, 5.610e+00, 5.724e+00,
     | 5.637e+00, 5.672e+00, 5.585e+00, 5.677e+00, 5.663e+00, 5.917e+00,
     | 6.149e+00, 6.080e+00, 5.808e+00, 5.501e+00, 5.843e+00, 5.699e+00
     | /)
      iday(1:312,7)=(/
     |  299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299,
     |  299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299, 299,
     |  300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300,
     |  300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300, 300,
     |  301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301,
     |  301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301, 301,
     |  302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302,
     |  302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302, 302,
     |  303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303,
     |  303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303, 303,
     |  304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304,
     |  304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304, 304,
     |  305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305,
     |  305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305, 305,
     |  306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306,
     |  306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306, 306,
     |  307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307,
     |  307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307, 307,
     |  308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308,
     |  308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308, 308,
     |  309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309,
     |  309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309, 309,
     |  310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310,
     |  310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310, 310,
     |  311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311,
     |  311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311, 311
     | /)
      hour(1:312,7)=(/
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
!
!  4/21/05 btf: Added event 8 for Jan 15-22, 2005.
!  Proton flux parameters for January 15-22, 2005, Solar Eruptions
      fe(1:192,8)=(/
     | 2.097e+01, 2.655e+01, 2.634e+01, 2.256e+01, 2.514e+01, 2.765e+01,
     | 2.350e+01, 3.386e+01, 1.357e+02, 4.459e+02, 7.942e+02, 9.935e+02,
     | 1.070e+03, 1.022e+03, 9.725e+02, 8.582e+02, 7.748e+02, 7.445e+02,
     | 4.541e+02, 5.568e+02, 5.102e+02, 3.508e+02, 3.269e+02, 3.009e+02,
     | 3.499e+02, 7.030e+02, 1.772e+03, 3.068e+03, 4.388e+03, 4.609e+03,
     | 5.968e+03, 7.123e+03, 7.821e+03, 9.291e+03, 1.239e+04, 1.402e+04,
     | 1.714e+04, 1.818e+04, 2.360e+04, 3.035e+04, 3.113e+04, 3.732e+04,
     | 4.162e+04, 3.948e+04, 3.627e+04, 3.219e+04, 3.602e+04, 3.666e+04,
     | 3.808e+04, 3.645e+04, 4.259e+04, 4.074e+04, 3.164e+04, 2.711e+04,
     | 2.589e+04, 2.739e+04, 2.795e+04, 2.605e+04, 3.593e+04, 3.596e+04,
     | 2.341e+04, 5.159e+04, 1.405e+05, 1.585e+05, 3.038e+05, 3.580e+05,
     | 3.201e+05, 2.513e+05, 2.717e+05, 2.827e+05, 2.761e+05, 2.051e+05,
     | 1.762e+05, 1.914e+05, 2.106e+05, 1.726e+05, 1.788e+05, 1.655e+05,
     | 1.682e+05, 1.624e+05, 1.410e+05, 1.393e+05, 1.348e+05, 1.418e+05,
     | 1.107e+05, 8.431e+04, 8.006e+04, 7.573e+04, 7.053e+04, 6.352e+04,
     | 5.235e+04, 5.451e+04, 4.917e+04, 4.221e+04, 3.412e+04, 2.217e+04,
     | 1.779e+04, 1.835e+04, 1.094e+04, 1.149e+04, 1.176e+04, 1.130e+04,
     | 1.121e+04, 1.075e+04, 9.043e+03, 1.036e+04, 1.056e+04, 1.119e+04,
     | 9.786e+03, 9.211e+03, 8.681e+03, 8.133e+03, 8.490e+03, 7.496e+03,
     | 5.283e+03, 4.902e+03, 5.190e+03, 4.790e+03, 5.197e+03, 4.019e+03,
     | 3.642e+03, 3.301e+03, 3.415e+03, 3.147e+03, 2.509e+03, 2.597e+03,
     | 8.029e+03, 6.580e+05, 4.506e+05, 2.462e+05, 1.679e+05, 1.631e+05,
     | 1.318e+05, 1.155e+05, 1.152e+05, 1.052e+05, 9.456e+04, 7.608e+04,
     | 6.256e+04, 4.637e+04, 3.788e+04, 3.690e+04, 2.879e+04, 2.465e+04,
     | 2.938e+04, 2.082e+04, 1.979e+04, 1.914e+04, 1.959e+04, 2.461e+04,
     | 2.510e+04, 2.601e+04, 2.575e+04, 2.513e+04, 2.054e+04, 2.117e+04,
     | 2.497e+04, 2.636e+04, 2.908e+04, 3.271e+04, 3.078e+04, 5.363e+04,
     | 1.503e+04, 7.755e+03, 6.702e+03, 5.545e+03, 5.097e+03, 4.413e+03,
     | 4.235e+03, 3.919e+03, 3.737e+03, 3.360e+03, 2.783e+03, 2.919e+03,
     | 2.609e+03, 2.448e+03, 2.297e+03, 2.085e+03, 1.899e+03, 1.771e+03,
     | 1.792e+03, 1.744e+03, 1.535e+03, 1.544e+03, 1.353e+03, 1.083e+03,
     | 9.641e+02, 8.945e+02, 7.245e+02, 6.453e+02, 6.738e+02, 6.157e+02
     | /)
      eo(1:192,8)=(/
     | 1.245e+01, 1.349e+01, 1.324e+01, 1.541e+01, 1.561e+01, 1.345e+01,
     | 1.288e+01, 1.461e+01, 1.225e+01, 9.241e+00, 7.598e+00, 6.615e+00,
     | 6.222e+00, 5.969e+00, 5.877e+00, 5.688e+00, 5.427e+00, 5.366e+00,
     | 6.098e+00, 5.475e+00, 5.714e+00, 6.091e+00, 6.375e+00, 7.210e+00,
     | 6.966e+00, 8.032e+00, 7.727e+00, 7.397e+00, 7.031e+00, 7.069e+00,
     | 7.122e+00, 7.003e+00, 7.145e+00, 6.946e+00, 6.769e+00, 6.543e+00,
     | 6.559e+00, 6.873e+00, 6.921e+00, 6.765e+00, 6.660e+00, 6.185e+00,
     | 5.922e+00, 5.729e+00, 5.687e+00, 5.641e+00, 5.335e+00, 5.247e+00,
     | 5.310e+00, 5.183e+00, 4.971e+00, 4.869e+00, 4.885e+00, 4.916e+00,
     | 4.877e+00, 4.767e+00, 4.707e+00, 4.679e+00, 4.633e+00, 4.609e+00,
     | 5.515e+00, 7.115e+00, 9.217e+00, 9.785e+00, 9.850e+00, 9.718e+00,
     | 9.690e+00, 9.391e+00, 8.897e+00, 8.840e+00, 8.738e+00, 9.041e+00,
     | 9.203e+00, 8.874e+00, 8.580e+00, 8.590e+00, 7.831e+00, 8.002e+00,
     | 7.678e+00, 7.357e+00, 7.259e+00, 7.181e+00, 7.159e+00, 6.797e+00,
     | 6.960e+00, 7.358e+00, 7.610e+00, 8.228e+00, 8.090e+00, 8.016e+00,
     | 8.252e+00, 8.180e+00, 8.371e+00, 8.507e+00, 8.268e+00, 7.871e+00,
     | 7.389e+00, 7.255e+00, 7.024e+00, 6.707e+00, 6.513e+00, 6.478e+00,
     | 6.531e+00, 6.646e+00, 7.120e+00, 7.055e+00, 7.037e+00, 7.191e+00,
     | 7.052e+00, 7.028e+00, 7.084e+00, 7.098e+00, 7.545e+00, 7.836e+00,
     | 8.474e+00, 8.706e+00, 8.657e+00, 9.188e+00, 8.828e+00, 9.536e+00,
     | 9.620e+00, 9.734e+00, 9.564e+00, 9.438e+00, 9.897e+00, 9.405e+00,
     | 1.882e+01, 8.846e+01, 4.535e+01, 2.630e+01, 1.951e+01, 1.568e+01,
     | 1.371e+01, 1.269e+01, 1.067e+01, 9.978e+00, 1.001e+01, 1.006e+01,
     | 1.004e+01, 1.071e+01, 1.129e+01, 1.073e+01, 1.120e+01, 1.177e+01,
     | 9.737e+00, 1.117e+01, 1.122e+01, 1.068e+01, 9.776e+00, 8.290e+00,
     | 7.778e+00, 7.382e+00, 7.182e+00, 7.055e+00, 7.257e+00, 7.110e+00,
     | 6.597e+00, 6.227e+00, 5.934e+00, 5.617e+00, 5.581e+00, 4.665e+00,
     | 5.545e+00, 6.315e+00, 6.341e+00, 6.425e+00, 6.541e+00, 6.562e+00,
     | 6.651e+00, 6.782e+00, 6.760e+00, 6.805e+00, 7.002e+00, 6.795e+00,
     | 6.769e+00, 6.707e+00, 6.716e+00, 6.656e+00, 7.046e+00, 7.051e+00,
     | 7.134e+00, 7.153e+00, 7.579e+00, 7.297e+00, 7.567e+00, 7.747e+00,
     | 8.074e+00, 8.019e+00, 8.446e+00, 8.700e+00, 8.497e+00, 8.480e+00
     | /)
      iday(1:192,8)=(/
     |   15,  15,  15,  15,  15,  15,  15,  15,  15,  15,  15,  15,
     |   15,  15,  15,  15,  15,  15,  15,  15,  15,  15,  15,  15,
     |   16,  16,  16,  16,  16,  16,  16,  16,  16,  16,  16,  16,
     |   16,  16,  16,  16,  16,  16,  16,  16,  16,  16,  16,  16,
     |   17,  17,  17,  17,  17,  17,  17,  17,  17,  17,  17,  17,
     |   17,  17,  17,  17,  17,  17,  17,  17,  17,  17,  17,  17,
     |   18,  18,  18,  18,  18,  18,  18,  18,  18,  18,  18,  18,
     |   18,  18,  18,  18,  18,  18,  18,  18,  18,  18,  18,  18,
     |   19,  19,  19,  19,  19,  19,  19,  19,  19,  19,  19,  19,
     |   19,  19,  19,  19,  19,  19,  19,  19,  19,  19,  19,  19,
     |   20,  20,  20,  20,  20,  20,  20,  20,  20,  20,  20,  20,
     |   20,  20,  20,  20,  20,  20,  20,  20,  20,  20,  20,  20,
     |   21,  21,  21,  21,  21,  21,  21,  21,  21,  21,  21,  21,
     |   21,  21,  21,  21,  21,  21,  21,  21,  21,  21,  21,  21,
     |   22,  22,  22,  22,  22,  22,  22,  22,  22,  22,  22,  22,
     |   22,  22,  22,  22,  22,  22,  22,  22,  22,  22,  22,  22
     | /)
      hour(1:192,8)=(/
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5,
     |  0.5, 1.5, 2.5, 3.5, 4.5, 5.5, 6.5, 7.5, 8.5, 9.5,10.5,11.5,
     | 12.5,13.5,14.5,15.5,16.5,17.5,18.5,19.5,20.5,21.5,22.5,23.5
     | /)
      end subroutine protons_data
      end module protons_module
