!
      module qrj_module
!
! Calculate ionization and heating rates. Note there are 2 qrj routines:
! qrj and qrj_exp. If one is changed, the other must be changed as well.
!
      use params_module,only: nlevp1,nlonp4,zsb,dz
      implicit none
!
! Coefficients for wavelength scans in qrj.
! These are referenced in routines qrj, init_sflux, init_sigmas, and 
! set_qrj_coeff (all in qrj_module.F).
!
      integer,parameter :: lmax=59, l1=16
      real :: 
     |  sigeuv(8,lmax),  ! absorption coefficients
     |  sigmas(6,lmax),  ! ionization coefficients
     |  euveff(nlevp1),  !
     |  rlmeuv(lmax),    !
     |  feuv(lmax),      !
     |  fsrc(l1-1),      !
     |  sigsrc(l1-1),    !
     |  rlmsrc(l1-1),    !
     |  quench(4),       !
     |  sflux(lmax),     !
     |  brn2(lmax),      !
     |  bro2(lmax)       !
!
! For euvac, init_sigmas
      integer,parameter :: neuv=37
      real,dimension(neuv) ::
     |  wleuv1,wleuv2,
     |  sigao, sigao2, sigan2,
     |  sigio, sigio2, sigin2,
     |  sigop2p, sigop2d, sigop4s, sigin,
     |  brop4s, brop2d, brop2p, brn2np, bro2op 
!
! Heating and ionization terms set by qrj, and used by other routines.
! These are allocated for task subdomains by alloc_q (called from allocdata)
!   e.g.: allocate(qtotal(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
!
      real,dimension(:,:,:),allocatable ::
     |  qtotal, ! total heating             ! F(NQ)
     |  qop2p,  ! o+(2p) ionization         ! F(NQOP2P)
     |  qop2d,  ! o+(2d) ionization         ! F(NQOP2D)
     |  qo2p,   ! o2+ ionization            ! F(NQO2P)
     |  qo2j,   !                           ! F(NQO2J) zeroed out
     |  qop,    ! o+  ionization            ! F(NQOP)
     |  qn2p,   ! n2+ ionization            ! F(NQN2P)
     |  qnp,    ! n+  ionization            ! F(NQNP)
     |  qnop,   ! no+ ionization            ! F(NQNOP)
     |  qnoplya,! no+ nighttime ionization by lyman-alpha (XJNOPN)
     |  qnolya  ! no  ionization by lyman-alpha           (XJNOP)
!
! Photodissociation terms set by qrj, and used by other routines.
      real,dimension(:,:,:),allocatable ::
     |  pdo2,   ! total o2 photodissociation      (F(NRJ))
     |  pdo2d,  ! o2(d) photodissociation         (F(NJO2D))
     |  pdn2,   ! solar photodissociation of n2   (F(NQTEF))
     |  pdco2t, ! total photodissociation of co2  (F(NJCO2T))
     |  pdco2d, ! total photodissociation of co2  (XJCO2D)
     |  pdh2ol, ! lyman-alfa dissociation of h2o  (F(NJH2OL))
     |  pdh2ot, ! total photodissociation of h2o  (F(NJH2OT))
     |  pdo3d,  ! photodissociation of o3(d)      (F(NJO3D))
     |  pdo3p,  ! photodissociation of o3(p)      (F(NJO3P))
     |  pdch4a, ! partial photodissociation of ch4 (XJCH4A)
     |  pdch4b, ! partial photodissociation of ch4 (XJCH4B)
     |  pdch4t, ! total photodissociation of ch4  (F(NJCH4T))
     |  pdnoeuv,! photodissociation of no by euv  (DNOEUV)
     |  pdnosrb,! photodissociation of no by srb  (XJNO)
     |  pdh2o2, ! XJH2O2
     |  pdch2oa,! XJCH2OA
     |  pdch2ob,! XJCH2OB
     |  pdn2o,  ! XJN2O
     |  pdho2,  ! XJHO2
     |  pdno2,  ! XJNO2
     |  pdch3oo,! XJCH3OO
     |  pd762   ! XJ762
!
! Attenuation of lyman-alfa at night (s.a., qinite.F)
!     real,dimension(:,:,:),allocatable :: attlya ! SJNMLYA
!
      real,external :: expo
      contains
!-----------------------------------------------------------------------
      subroutine qrj(sco2,sco1,scn2,sco3,scno,vo2,tn,no,o2,o1,o3,n4s,
     |  o21d,xnmbari,cp,lev0,lev1,lon0,lon1,lat)
!
! Calculate heating, ionization, and dissociation rates.
! (Note that if check_exp is set true (cons.F), then qrj_exp is called
!  instead of this routine)
! Also note input arg o1 is actually ox.
!
      use input_module,only: f107
      use init_module,only: sfeps,istep
      use cons_module,only: t0,expz,avo,rmassinv_n4s,rmassinv_no,
     |  rmassinv_o2,rmassinv_o1,rmassinv_o3,rmassinv_n2,expzmid,gask,
     |  amu,expzmid_inv,p0
      use bndry_module,only: fb,b
      use chemrates_module,only: disn2p,beta9
      use o2srbc ! module in o2srbc.f for O2 heating/dissoc in Shumann-Runge
      use chapman_module,only: idn
!
! Args:
      integer,intent(in) :: lev0,lev1,lon0,lon1,lat
      real,dimension(lev0:lev1,lon0-2:lon1+2),intent(in) ::
     |  sco2,sco1,scn2,sco3,scno,  ! chapman slant column integrals
     |  vo2,                       ! chapman vertical integral
     |  tn,no,o2,o1,o3,n4s,        ! tn and species mass mixing ratios
     |  o21d,
     |  xnmbari,              ! p0*e(-z)*barm/kT at interfaces
     |  cp                    ! specific heat
!
! VT vampir tracing:
!
#ifdef VT
#include <VT.inc>
#endif
!
! Local:
      integer :: k,i,l,nlevs,ier,levmin
      real,parameter ::
     |  do2=8.203E-12   ,
     |  do22=1.1407E-11 ,   
     |  aband=0.143     , ! shumann-runge
     |  bband=9.64E8    , ! shumann-runge
     |  cband=9.03E-19  , ! shumann-runge
     |  e3=0.33         ,
     |  hc = 1.9845E-16 , ! C(60)
     |  zpmin = -7.0      ! was zkmin
      real :: rlmeuvinv(lmax),rlmsrcinv(l1-1),factor,fmin,fmax
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  r,      ! s14
     |  rp,     ! s15
     |  taur,   ! s1
     |  tauq,   ! s2
     |  tau1,   ! s3
     |  tau2,   ! s4
     |  tau3,   ! s5
     |  etaur,  ! s6
     |  etauq   ! s7
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  o2i,    ! o2  at interfaces (s1)
     |  o1i,    ! o   at interfaces (s2)
     |  o3i,    ! o3  at interfaces (s1)
     |  n2i,    ! n2  at interfaces (s3)
     |  n4si,   ! n4s at interfaces (s4)
     |  o21di,  ! o21d at interfaces (s3)
     |  tni     ! tn  at interfaces (s6)
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  zo2,       ! o2  ionization (s8)
     |  zo1,       ! o   ionization (s9)
     |  zn2,       ! n2  ionization (s10)
     |  zn4s,      ! n4s ionization (s11)
     |  zo2_bro2,  ! o2  ionization * bro2 (s12)
     |  zn2_brn2,  ! n2  ionization * brn2 (s13)
     |  qop2pd,    ! (s4)
     |  quenchfac, ! (s8)
     |  sigchap,   ! (s9)
     |  p3f,       ! (s7)
     |  pdnolya,   ! photodissociation of NO by lyman-alpha (SJMLYA)
     |  pdo2lya,   ! photodissociation of O2 by lyman-alpha (SJO2LYA)
     |  sco3t      ! temporary chapman integral
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  sum1,   ! sum(o2,o,n2)(sigma*chapman)   (s5)
     |  sum2,   ! sum(o2,o,n2)(sigma*psi/rmass) (s6)
     |  sum3    ! sum(o2,o,n2,n4s)(sigmas)      (s7)
!
! Declarations for O2 photolysis and heating in Shumann-Runge Continuum
! (SRC) and Shumann-Runge Bands (SRB) (see o2srbc.f).
! 12/11/00: declarations for o2srbc as per kibo12:
!  8/5/04: Sub jo2h2osrb returns xdo2srb and dh2osrb, but only dh2osrb
!          is being used (do2srb from sub mkdo2srb is used instead of xdo2srb)
      real ::
     |  xno2(lev0:lev1),         ! o2 column density (cm-2)
     |  rho(lev0:lev1),          ! total density
     |  do2src(lev0:lev1,lon0:lon1), ! o2 dissoc in SRC  (mkdo2src output)
     |  ho2src(lev0:lev1,lon0:lon1), ! o2 heating in SRC (mkho2src output)
     |  do2srb(lev0:lev1,lon0:lon1), ! o2 dissoc in SRB  (mkdo2srb output)
     |  ho2srb(lev0:lev1,lon0:lon1), ! o2 heating in SRB (mkdo2srb output)
     | xdo2srb(lev0,lev1,lon0:lon1), ! alternate o2 srb dissoc (sub jo2h2osrb)
     | dh2osrb(lev0:lev1,lon0:lon1)  ! h2o dissoc in SRB (jo2h2osrb output)
      real :: sfac,temp1,temp2
!
! Heating diagnostics:
      real,dimension(lev0:lev1,lon0:lon1) :: 
     |  qtotal1,qtotal2,qtotal3,qtotal4,qtotal5,qtotal6,qtotal7
      integer :: i0,i1,nk,nkm1
!
!   If mks >  0, ho2src or ho2srb are returned in deg K/sec (mks)
!   If mks <= 0, ho2src or ho2srb are returned in ergs/gm-1/s-1 (cgs)
      integer,parameter :: mks=0     ! units flag for ho2src
!
      logical,parameter :: 
     |  debug=.false.     ! insert print statements
!
      if (debug) write(6,"('Enter qrj: lat=',i3,' lon0,1=',2i3)") 
     |  lat,lon0,lon1
!
#ifdef VT
!     code = 118 ; state = 'qrj' ; activity='ModelCode'
      call vtbegin(118,ier)
#endif
!
! Exec:
      nlevs = lev1-lev0+1
      i0=lon0 ; i1=lon1 ; nk=lev1-lev0+1 ; nkm1=nk-1

!     write(6,"('qrj: lat=',i2,' lon0,1=',2i3,' idn(lon0:lon1,lat)=',/,
!    |  (15i3))") lat,lon0,lon1,idn(lon0:lon1,lat)
!     call addfsech('SCO2',' ',' ',sco2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCO1',' ',' ',sco1(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCN2',' ',' ',scn2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('XNMBARI',' ',' ',xnmbari(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!
! Will multiply by inverses:
      do i=1,lmax
        rlmeuvinv(i) = 1./rlmeuv(i)
      enddo
      do i=1,l1-1 ! 1,15
        rlmsrcinv(i) = 1./rlmsrc(i)
      enddo
!
! COMPUTE S14 = RSQ, S15 = RSP
! S1 = TAU(R),  S2 = TAU(Q)
!
!     write(6,"('qrj: lat=',i2,' lon0,1=',2i3,' idn(lon0:lon1,lat)=',/,
!    |  (12i3))") lat,lon0,lon1,idn(lon0:lon1,lat)
!
      do i=lon0,lon1
        do k=lev0,lev1
          taur(k,i) = sigeuv(1,49)*sco2(k,i)+sigeuv(2,49)*sco1(k,i)+
     |                sigeuv(3,49)*scn2(k,i)
          tauq(k,i) = sigeuv(1,20)*sco2(k,i)+sigeuv(2,20)*sco1(k,i)+
     |                sigeuv(3,20)*scn2(k,i)
          tau1(k,i) = 1.3*taur(k,i)
          tau2(k,i) = 2.0*taur(k,i)
          tau3(k,i) = 2.5*taur(k,i)
          if (taur(k,i) > 9.) taur(k,i) = 9.
          if (tauq(k,i) > 9.) tauq(k,i) = 9.
          if (tau1(k,i) > 9.) tau1(k,i) = 9.
          if (tau2(k,i) > 9.) tau2(k,i) = 9.
          if (tau3(k,i) > 9.) tau3(k,i) = 9.
!
! TAU(N) = EXP(-TAU(N)) FOR N = 1,3,1
!
          tau1(k,i) = exp(-tau1(k,i))
          tau2(k,i) = exp(-tau2(k,i))
          tau3(k,i) = exp(-tau3(k,i))
!
! taur = EXP(-TAU(R)), tauq = EXP(-TAU(Q))
!
          etaur(k,i) = exp(-taur(k,i))
          etauq(k,i) = exp(-tauq(k,i))
!
! rp = r(o2p), r = r(op,n2p,np)
!
          r(k,i) = etaur(k,i)+2.*(tau1(k,i)+tau2(k,i)+tau3(k,i))
          rp(k,i) = 1.5*etaur(k,i)/(r(k,i)+tauq(k,i)/taur(k,i)*
     |      etauq(k,i))
          r(k,i) = 2.4*etaur(k,i)/r(k,i)
        enddo
      enddo
      if (debug) write(6,"(/,'qrj after tau: lat=',i3)") lat
!
! Zero diffs, 7/30/03:
!     call addfsech('TAUQ',' ',' ',tauq,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('TAUR',' ',' ',taur,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('R'   ,' ',' ',r   ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('RP'  ,' ',' ',rp  ,lon0,lon1,nlevs,nlevs,lat)
!
! O2,O,N4S at interface levels:
      do i=lon0,lon1
        do k=lev0,lev1-1
          o2i (k+1,i) = 0.5*(o2(k,i)+o2(k+1,i))
          o1i (k+1,i) = 0.5*(o1(k,i)+o1(k+1,i))
          n4si(k+1,i) = 0.
        enddo
      enddo
!
! Bottom boundary:
      do i=lon0,lon1
        o2i(1,i) = .5*((b(i,1,1)+1.)*o2(1,i)+b(i,1,2)*o1(1,i)+
     |    fb(i,1,lat))
        o1i(1,i) = .5*(b(i,2,1)*o2(1,i)+(b(i,2,2)+1.)*o1(1,i)+
     |    fb(i,2,lat))
        n4si(1,i) = 0.
      enddo
!
! N2 at interfaces:
      do k=lev0,lev1
        n2i(k,:) = 1.-o2i(k,:)-o1i(k,:)
      enddo 
!
! Zero diffs, 7/30/03:
!     call addfsech('O2I'  ,' ',' ',o2i,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('O1I'  ,' ',' ',o1i,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('N2I'  ,' ',' ',n2i,lon0,lon1,nlevs,nlevs,lat)
!
! Initialize terms on current processor subdomain:
! (global module data for use by other routines)
! (in tgcm24, some of these inits occur in sub clearnq)
!
      qtotal(lev0:lev1,lon0:lon1,lat) = 0.  ! NQ
      qop2p (lev0:lev1,lon0:lon1,lat) = 0.  ! NQOP2P
      qop2d (lev0:lev1,lon0:lon1,lat) = 0.  ! NQOP2D
      qo2p  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQO2P
      qo2j  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQO2J
      qop   (lev0:lev1,lon0:lon1,lat) = 0.  ! NQOP
      qn2p  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQN2P
      qnp   (lev0:lev1,lon0:lon1,lat) = 0.  ! NQNP
      qnop  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQNOP
!
      pdo2   (lev0:lev1,lon0:lon1,lat) = 0. ! NRJ
      pdn2   (lev0:lev1,lon0:lon1,lat) = 0. ! NQTEF 
      pdco2t (lev0:lev1,lon0:lon1,lat) = 0. ! NJCO2T 
      pdh2ol (lev0:lev1,lon0:lon1,lat) = 0. ! NJH2OL 
      pdh2ot (lev0:lev1,lon0:lon1,lat) = 0. ! NJH2OT 
      pdo3d  (lev0:lev1,lon0:lon1,lat) = 0. ! NJO3D 
      pdo3p  (lev0:lev1,lon0:lon1,lat) = 0. ! NJO3P 
      pdo2d  (lev0:lev1,lon0:lon1,lat) = 0. ! NJO2D (added 5/25/04)
      pdch4t (lev0:lev1,lon0:lon1,lat) = 0. ! NJCH4T 
      pdnoeuv(lev0:lev1,lon0:lon1,lat) = 0. ! DNOEUV 
!
! Initialize ionization rates:
! (array ops on local arrays dimensioned (lon0:lon1,lev0:lev1)):
      zo2      = 0.
      zo1      = 0.
      zn2      = 0.
      zn4s     = 0.
      zo2_bro2 = 0.
      zn2_brn2 = 0.
!
! Summation over wavelength. Define parameters at task subdomains, and
!   vertical column only from zpmin to top.
!
      levmin = int((zpmin-zsb)/dz)+1 ! zpmin == -7.
!     write(6,"('qrj: lat=',i3,' zpmin=',f6.2,' levmin=',i3)")
!    |  lat,zpmin,levmin
!
      do l=l1,lmax     ! 16,59 (44 wavelengths)
        sum1(:,:) = 0. ! sum(o2,o,n2)(sigeuv*chapman)   ! s5
        sum2(:,:) = 0. ! sum(o2,o,n2)(sigeuv*psi/rmass) ! s6
        sum3(:,:) = 0. ! sum(o2,o,n2,n4s)(sigmas)       ! s7
        do i=lon0,lon1
          do k=levmin,lev1
            sum1(k,i) = sum1(k,i)+sigeuv(1,l)*sco2(k,i)+
     |                            sigeuv(2,l)*sco1(k,i)+
     |                            sigeuv(3,l)*scn2(k,i)
            sum2(k,i) = sum2(k,i)+sigeuv(1,l)*o2i(k,i)*rmassinv_o2+
     |                            sigeuv(2,l)*o1i(k,i)*rmassinv_o1+
     |                            sigeuv(3,l)*n2i(k,i)*rmassinv_n2
            sum3(k,i) = sum3(k,i)+sigmas(1,l)+sigmas(2,l)+
     |                            sigmas(3,l)
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
!
        do i=lon0,lon1
          do k=levmin,lev1
            sum3(k,i) = sum3(k,i)+sigmas(4,l)
            sum1(k,i) = feuv(l)*exp(-sum1(k,i))
            qtotal (k,i,lat) = qtotal (k,i,lat)+hc*rlmeuvinv(l)*        ! f(nq)
     |        sum1(k,i)*sum2(k,i)
            pdo2   (k,i,lat) = pdo2   (k,i,lat)+sum1(k,i)*sum3(k,i)     ! f(nrj)
            pdco2t (k,i,lat) = pdco2t (k,i,lat)+sum1(k,i)*sigeuv(5,l)   ! f(njco2t)
            pdnoeuv(k,i,lat) = pdnoeuv(k,i,lat)+sum1(k,i)*sigeuv(7,l)   ! dnoeuv
            pdo3d  (k,i,lat) = pdo3d  (k,i,lat)+sum1(k,i)*sigeuv(4,l)   ! f(njo3d)
            pdch4t (k,i,lat) = pdch4t (k,i,lat)+sum1(k,i)*sigeuv(8,l)   ! f(njch4t)
            qop2d(k,i,lat)   = qop2d  (k,i,lat)+sum1(k,i)*sigmas(5,l)*  ! f(nqop2d)
     |        o1i(k,i)*rmassinv_o1
            qop2p(k,i,lat)   = qop2p  (k,i,lat)+sum1(k,i)*sigmas(6,l)*  ! f(nqop2p)
     |        o1i(k,i)*rmassinv_o1
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
!
        do i=lon0,lon1
          do k=levmin,lev1
            zo2(k,i) = zo2(k,i)+sigmas(1,l)*sum1(k,i)*o2i(k,i)*  ! s8
     |        rmassinv_o2
            zo1(k,i) = zo1(k,i)+sigmas(2,l)*sum1(k,i)*o1i(k,i)*  ! s9
     |        rmassinv_o1
            zn2(k,i) = zn2(k,i)+sigmas(3,l)*sum1(k,i)*n2i(k,i)*  ! s10
     |        rmassinv_n2
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
!
        do i=lon0,lon1
          do k=levmin,lev1
            pdn2(k,i,lat) = pdn2(k,i,lat)+(sigeuv(3,l)-sigmas(3,l))*    ! f(nqtef)
     |        sum1(k,i)*n2i(k,i)*rmassinv_n2*2.
            zn4s(k,i) = zn4s(k,i)+sigmas(4,l)*sum1(k,i)*n4si(k,i)*       ! s11
     |        rmassinv_n4s
            zo2_bro2(k,i) = zo2_bro2(k,i)+sigmas(1,l)*sum1(k,i)*o2i(k,i) ! s12
     |        *rmassinv_o2*bro2(l)
            zn2_brn2(k,i) = zn2_brn2(k,i)+sigmas(3,l)*sum1(k,i)*n2i(k,i) ! s13
     |        *rmassinv_n2*brn2(l)
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
      enddo ! l=l1,lmax
!
!     qtotal1(:,:) = qtotal(:,:,lat)
!     call addfsech('QTOTAL1',' ',' ',qtotal1,lon0,lon1,nlevs,nlevs,lat)
!
! Zero diffs, 8/5/03:
!     call addfsech('QTOTAL1',' ',' ',qtotal(:,:,lat),lon0,lon1,    ! f(nq)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDO2'   ,' ',' ',pdo2(:,:,lat),lon0,lon1,      ! f(nrj)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDO3D'  ,' ',' ',pdo3d(:,:,lat),lon0,lon1,     ! f(njo3d)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDCO2T' ,' ',' ',pdco2t(:,:,lat),lon0,lon1,    ! f(njco2t)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDNOEUV',' ',' ',pdnoeuv(:,:,lat),lon0,lon1,   ! dnoeuv
!    |  nlevs,nlevs,lat)
!     call addfsech('PDCH4T' ,' ',' ',pdch4t(:,:,lat),lon0,lon1,    ! f(njch4t)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDN2'   ,' ',' ',pdn2(:,:,lat),lon0,lon1,      ! f(nqtef)
!    |  nlevs,nlevs,lat)
!
! ln vs bf diffs here due to exp() along terminator in sum1 above:
!
!     call addfsech('QOP2D'  ,' ',' ',qop2d(:,:,lat),               ! f(nqop2d)
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('QOP2P',' ',' '  ,qop2p(:,:,lat),               ! f(nqop2p)
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SUM1',' ',' '  ,sum1 ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SUM2',' ',' '  ,sum2 ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SUM3',' ',' '  ,sum3 ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('O2I' ,' ',' '  ,o2i  ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('O1I' ,' ',' '  ,o1i  ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('N2I' ,' ',' '  ,n2i  ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCO2',' ',' '  ,sco2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCO1',' ',' '  ,sco1(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCN2',' ',' '  ,scn2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('ZO2' ,' ',' ',zo2 ,lon0,lon1,nlevs,nlevs,lat)  ! s8
!     call addfsech('ZO1' ,' ',' ',zo1 ,lon0,lon1,nlevs,nlevs,lat)  ! s9
!     call addfsech('ZN2' ,' ',' ',zn2 ,lon0,lon1,nlevs,nlevs,lat)  ! s10
!     call addfsech('ZN4S',' ',' ',zn4s,lon0,lon1,nlevs,nlevs,lat)  ! s11
!     call addfsech('ZO2_BRO2',' ',' ',zo2_bro2,lon0,lon1,          ! s12
!    |  nlevs,nlevs,lat)
!     call addfsech('ZN2_BRN2',' ',' ',zn2_brn2,lon0,lon1,          ! s13
!    |  nlevs,nlevs,lat)
!
! Multiply Q by efficiency factor:
      do i=lon0,lon1
        do k=lev0,lev1
          qtotal(k,i,lat) = qtotal(k,i,lat)*euveff(k)*avo
        enddo
      enddo

!     qtotal2(:,:) = qtotal(:,:,lat)
!     call addfsech('QTOTAL2',' ',' ',qtotal2,lon0,lon1,nlevs,nlevs,lat)

!     call addfsech('qeffic'  ,' ',' ',qtotal(:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!
! Contributions to O2+, O+, N2+, N+, pdn2:
! (disn2p should probably be renamed and moved here from chemrates.F)
!
      do i=lon0,lon1
        do k=lev0,lev1
!
! disn2p: dissociation of N2+ (DISN2P):
          disn2p(k,i,lat) = zn2(k,i)*r(k,i)*xnmbari(k,i)
!
! qo2p: O2+ ionization (F(NQO2P)):
          qo2p(k,i,lat) = qo2p(k,i,lat)+(zo2(k,i)*(1.+rp(k,i))-
     |      zo2_bro2(k,i))*xnmbari(k,i)
!
! qn2p: N2+ ionization (F(NQN2P)):
          qn2p(k,i,lat) = qn2p(k,i,lat)+(zn2(k,i)*(1.+r(k,i))-
     |      zn2_brn2(k,i))*xnmbari(k,i) 
!
! qnp: N+ ionization (F(NQNP)):
          qnp(k,i,lat)  = qnp(k,i,lat)+(zn4s(k,i)+zn2_brn2(k,i))*
     |      xnmbari(k,i)
!
! pdn2: photodissociation of N2 (F(NQTEF)):
          pdn2(k,i,lat) = pdn2(k,i,lat)*xnmbari(k,i)
!
! qop2pd (local) (S4):
          qop2pd(k,i) = qop2p(k,i,lat)+qop2d(k,i,lat)+zo1(k,i)+
     |      zo2_bro2(k,i)
!
! qop2p: O+(2P) ionization (F(NQOP2P)):
          qop2p(k,i,lat) = (qop2p(k,i,lat)+qop2pd(k,i)*r(k,i)*0.22)*
     |      xnmbari(k,i)
!
! qop2d: O+(2D) ionization (F(NQOP2D)):
          qop2d(k,i,lat) = (qop2d(k,i,lat)+qop2pd(k,i)*r(k,i)*0.24)*
     |      xnmbari(k,i)
!
! qop: O+ ionization (F(NQOP)):
          qop(k,i,lat) = qop(k,i,lat)+(zo1(k,i)+zo2_bro2(k,i)+
     |      qop2pd(k,i)*r(k,i)*0.56)*xnmbari(k,i)
!
! pdnosrb: photodissociation of no by srb  (XJNO)
! 8/4/04 btf: see jno call below.
!
!         pdnosrb(k,i,lat) = 7.0e-6*(1.+0.11*(f107-65.)/165.)*
!    |      exp(-1.e-8*sco2(k,i)**0.38)*exp(-5.e-19*sco3(k,i))*sfeps
!
! pdnolya: photodissociation of NO by lyman-alpha (local) (SJMLYA)
          pdnolya(k,i) = (0.68431  *exp(-8.22114e-21*sco2(k,i))+
     |                    0.229841 *exp(-1.77556e-20*sco2(k,i))+
     |                    0.0865412*exp(-8.22112e-21*sco2(k,i)))*
     |                    sflux(12)
!
! pdo2lya: photodissociation of O2 by lyman-alpha (local) (SJO2LYA)
          pdo2lya(k,i) = (6.0073e-21*exp(-8.2166e-21 *sco2(k,i))+
     |                   4.28569e-21*exp(-1.63296e-20*sco2(k,i))+
     |                   1.28059e-20*exp(-4.85121e-17*sco2(k,i)))
     |                   *sflux(12)
!
! qnolya: no ionization by lyman-alpha (XJNOP)
          qnolya(k,i,lat) = 2.02e-18*pdnolya(k,i)
!
! Calc of qnoplya moved to qinite.
!
! attlya: attenuation of lyman-alpha at night (SJNMLYA)
!         attlya(k,i,lat) = (0.68431  *exp(-8.22114e-21*vo2(k,i))+
!    |                       0.229841 *exp(-1.77556e-20*vo2(k,i))+
!    |                       0.0865412*exp(-8.22112e-21*vo2(k,i)))
!    |                       *sfeps
!
! qnoplya: no+ nighttime ionization by lyman-alpha (XJNOPN)
!         qnoplya(k,i,lat) = 2.02e-18*sflux(12)/100.*attlya(k,i,lat)
!
        enddo ! k=lev0,lev1
      enddo ! i=lon0,lon1
!
! Total NO dissociation pdnosrb (XJNO) is returned by sub jno.
! allocate(pdnosrb(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
!
      call jno(
     |  pdnosrb(:,lon0:lon1,lat),  o2(:,lon0:lon1),  o1(:,lon0:lon1),
     |  xnmbari(:,lon0:lon1)    ,sco2(:,lon0:lon1),sco3(:,lon0:lon1),
     |  scno   (:,lon0:lon1),lev0,lev1,lon0,lon1,lat)
!     call addfsech('pdnosrb',' ',' ',pdnosrb(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
! 
! Add no ionization to qnop:
      do i=lon0,lon1
        qnop(lev0,i,lat) = qnop(lev0,i,lat)+(qnolya(lev0,i,lat)+
     |    pdnoeuv(lev0,i,lat))*no(lev0,i)*xnmbari(lev0,i)*rmassinv_no
      enddo
      do i=lon0,lon1
        do k=lev0+1,lev1
          qnop(k,i,lat) = qnop(k,i,lat)+(qnolya(k,i,lat)+
     |      pdnoeuv(k,i,lat))*.5*(no(k,i)+no(k-1,i))*xnmbari(k,i)*
     |      rmassinv_no
        enddo
      enddo
!
! Minimum values for ionization rates:
      do i=lon0,lon1
        do k=lev0,lev1
          if (qo2p (k,i,lat) < 1.e-20) qo2p (k,i,lat) = 1.e-20  ! (f(nqo2p))
          if (qop  (k,i,lat) < 1.e-20) qop  (k,i,lat) = 1.e-20  ! (f(nqop))
          if (qop2p(k,i,lat) < 1.e-20) qop2p(k,i,lat) = 1.e-20  ! (f(nqop2p))
          if (qop2d(k,i,lat) < 1.e-20) qop2d(k,i,lat) = 1.e-20  ! (f(nqop2d))
          if (qn2p (k,i,lat) < 1.e-20) qn2p (k,i,lat) = 1.e-20  ! (f(nqn2p))
          if (qnp  (k,i,lat) < 1.e-20) qnp  (k,i,lat) = 1.e-20  ! (f(nqnp))
          if (pdn2 (k,i,lat) < 1.e-20) pdn2 (k,i,lat) = 1.e-20  ! (f(nqtef))
          if (qnop (k,i,lat) < 1.e-20) qnop (k,i,lat) = 1.e-20  ! (f(nqnop))
        enddo
      enddo
!
! tn at interfaces (S6):
      do i=lon0,lon1
        tni(1,i) = tn(lev1,i) ! tn bottom boundary is stored in top slot
        do k=lev0+1,lev1-1
          tni(k,i) = .5*(tn(k-1,i)+tn(k,i))
        enddo
        tni(lev1,i) = tn(lev1-1,i) ! nlevp1 <- nlev
      enddo
!
! Quench:
      do i=lon0,lon1
        do k=lev0,lev1
          factor = avo*p0/gask*expz(1)*expzmid**(2*k-3)
          quenchfac(k,i) = factor/((o2i(k,i)*rmassinv_o2+   ! s8
     |                              o1i(k,i)*rmassinv_o1+
     |                              n2i(k,i)*rmassinv_n2)*tni(k,i))
          quenchfac(k,i) = quenchfac(k,i)*
     |      (quench(1)*n2i(k,i)*rmassinv_n2+
     |       quench(2)*o2i(k,i)*rmassinv_o2)
          quenchfac(k,i) = quench(3)*quenchfac(k,i)/
     |                    (quench(4)+quenchfac(k,i))
        enddo
      enddo
!
! Zero diffs, 8/03:
!     call addfsech('qo2p',' ',' ',qo2p(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qn2p',' ',' ',qn2p(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnp',' ',' ',qnp(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdn2',' ',' ',pdn2(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop2pd',' ',' ',qop2pd,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop2p',' ',' ',qop2p(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop2d',' ',' ',qop2d(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop',' ',' ',qop(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdnosrb',' ',' ',pdnosrb(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdnolya',' ',' ',pdnolya,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdo2lya',' ',' ',pdo2lya,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnolya',' ',' ',qnolya(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('attlya',' ',' ',attlya(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnoplya',' ',' ',qnoplya(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('quench',' ',' ',quenchfac,lon0,lon1,nlevs,nlevs,
!    |  lat)
!     call addfsech('qnop',' ',' ',qnop(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!
! Summation over wave length:
      do l=1,l1-1 ! 1->15
        do i=lon0,lon1
          do k=lev0,lev1
            sigchap(k,i) = fsrc(l)*exp(-sigsrc(l)*sco2(k,i))
            pdo3d (k,i,lat) = pdo3d (k,i,lat)+sigeuv(4,l)*sigchap(k,i)
            pdco2t(k,i,lat) = pdco2t(k,i,lat)+sigeuv(5,l)*sigchap(k,i)
            pdh2ot(k,i,lat) = pdh2ot(k,i,lat)+sigeuv(6,l)*sigchap(k,i)
            qnop  (k,i,lat) = qnop  (k,i,lat)+sigeuv(7,l)*sigchap(k,i)
          enddo ! k=lev0,lev1
        enddo ! i=lon0,lon1
      enddo ! l=1,l1-1
!
!     call addfsech('pdo3d',' ',' ',pdo3d(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdco2t',' ',' ',pdco2t(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdh2ot',' ',' ',pdh2ot(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnop',' ',' ',qnop(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!
! Most code from here to end of qrj is time-gcm only (not in tiegcm):
!
! Calculate O2 photolysis and heating in Shumann-Runge Continuum (src) 
! Shumann-Runge Bands (srb):
!
      do i=lon0,lon1
        ho2src(:,i) = 0.
        do2src(:,i) = 0.
        do k=lev0,lev1
          rho(k) = (o2i(k,i)*xnmbari(k,i)+
     |              o1i(k,i)*xnmbari(k,i)+
     |              n2i(k,i)*xnmbari(k,i))*amu        ! gm/cm3
          xno2(k) = o2i(k,i)*rmassinv_o2*xnmbari(k,i)
        enddo ! lev0,lev1
!
! Pass columns at each grid point to the o2src routines.
! mkdo2src returns do2src (dissociation), mkho2src returns ho2src (heating)
!   subroutine mkdo2src(sco2,f107d,do2src,nlev)
!   subroutine mkho2src(sco2,xno2,rho,cp,f107d,ho2src,nlev,mks,lat)
!
        call mkdo2src(sco2(:,i),f107,do2src(:,i),nlevs)
        call mkho2src(sco2(:,i),xno2,rho,cp(:,i),f107,ho2src(:,i),
     |    nlevs,mks)
!
! mkdo2srb returns do2srb and ho2srb columns:
        call mkdo2srb(sco2(:,i),xno2,rho,cp(:,i),f107,sfeps,
     |    do2srb(:,i),ho2srb(:,i),nlevs,mks)
      enddo ! i=lon0,lon1
!
! Add src and srb to total heating and o2 heating and dissociation:
      do i=lon0,lon1
        do k=lev0,lev1
          qtotal(k,i,lat) = qtotal(k,i,lat)+ho2src(k,i)+ho2srb(k,i) ! f(nq)
          pdo2(k,i,lat) = pdo2(k,i,lat)+do2src(k,i)+do2srb(k,i)     ! f(nrj)
          pdo2d(k,i,lat) = pdo2d(k,i,lat)+do2src(k,i)               ! f(njo2d)
        enddo ! k=lev0,lev1
      enddo ! i=lon0,lon1

!     qtotal3(:,:) = qtotal(:,:,lat)
!     call addfsech('QTOTAL3',' ',' ',qtotal3,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('do2src',' ',' ',do2src,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('ho2src',' ',' ',ho2src,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('do2srb',' ',' ',do2srb,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('ho2srb',' ',' ',ho2srb,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qtotal',' ',' ',qtotal(:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo2'  ,' ',' ',pdo2  (:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo2d' ,' ',' ',pdo2d (:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!
! Contributions from ozone dissociation from the Herbzberg, Hartley, 
!   Huggins and chapius bands.
! Contributions from solar lyman-alpha, SRB and Herzberg to O2 
!   dissociation and heating.
!
      sfac = 1.+0.11*(f107-65.)/165.
      do i=lon0,lon1
        do k=lev0,lev1-1
          o3i  (k+1,i) = 0.5*(o3  (k,i)+o3  (k+1,i))   ! s1
          o2i  (k+1,i) = 0.5*(o2  (k,i)+o2  (k+1,i))   ! s2
          o21di(k+1,i) = 0.5*(o21d(k,i)+o21d(k+1,i))   ! s3
        enddo ! k=lev0,lev1-1
        o3i  (lev0,i) = o3  (lev0,i)**1.5/sqrt(o3  (lev0+1,i))
        o2i  (lev0,i) = o2  (lev0,i)**1.5/sqrt(o2  (lev0+1,i))
        o21di(lev0,i) = o21d(lev0,i)**1.5/sqrt(o21d(lev0+1,i))
      enddo ! i=lon0,lon1
!
! Sub jo2h2osrb calculates o2 and h2o dissociation at SRB:
!   dh2osrb is dissociation of h2o at SRB.
!   xdo2srb is an alternate calculation of o2 dissociation at SRB,
!     and is not used in this version. (see sub mkdo2srb for o2
!     dissociation, called above)
!
      call jo2h2osrb(xdo2srb,dh2osrb,sco2,sco3,lev0,lev1,lon0,lon1,lat)
!     call addfsech('JO2SRB' ,' ',' ',xdo2srb,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('JH2OSRB',' ',' ',dh2osrb,lon0,lon1,nlevs,nlevs,lat)
!
! Scan subdomain:
      do i=lon0,lon1
        do k=lev0,lev1
          sco3t(k,i) = sco3(k,i)                         ! s9
          if (sco3t(k,i) < 1.e+5) sco3t(k,i) = 1.e+5
!
! Dissociation and heating of O3 from lyman-alpha:
          pdo3d (k,i,lat) = pdo3d (k,i,lat)+2.27e-17*pdnolya(k,i)
          qtotal(k,i,lat) = qtotal(k,i,lat)+2.27e-17*pdnolya(k,i)*
     |      o3i(k,i)*rmassinv_o3*avo*9.944e-12

!         qtotal4(k,i) = qtotal(k,i,lat) ! diagnostic

          temp1 = 1.0e-3*exp(-1.5577e-13*sco3t(k,i)**0.6932)
          if (sco3t(k,i) < 1.6e+20) 
     |      temp1 = 1.04e-2*exp(-1.0217e-6*sco3t(k,i)**0.3587)
!
! Hartley bands of O3:
          pdo3d(k,i,lat) = pdo3d(k,i,lat)+
     |      (temp1*0.68+exp(-1.4912e-14*sco2   (k,i)**0.5298)*
     |      (4.053e-4  *exp(-8.1381e-16*sco3t(k,i)**0.8856)+
     |       4.7e-6    *exp(-1.5871e-14*sco3t(k,i)**0.7665))*
     |       1.085     *exp(-1.4655e-25*sco2   (k,i)**1.0743)*0.68)*
     |      sfeps
!
! Chappius and Huggins bands:
          pdo3p(k,i,lat) = pdo3p(k,i,lat)+
     |      (4.5e-4*exp(-3.4786e-11*sco2(k,i)
     |      **0.3366)*exp(-1.0061e-20*sco3t(k,i)**0.9719)
     |      +(7.5e-4 *exp(-2.7663e-19*sco3t(k,i)**1.0801)
     |      +2.5e-4/(1.+1.5772e-18   *sco3t(k,i)**0.9516))
     |      *exp(-1.0719e-10*sco2(k,i)**0.3172))*sfeps
!
! Add ozone heating to total heat:
          temp1 = exp(-5.50e-24*sco2(k,i)-6.22E-18*sco3(k,i))
          temp2 = exp(-1.34e-26*sco2(k,i)-1.66E-18*sco3(k,i))
          qtotal(k,i,lat) = qtotal(k,i,lat)+
     |      ((5.512e-16*exp(-3.16E-21*sco3(k,i))
     |      +(41.21e-16*exp(-8.94E-19*sco3(k,i))
     |      + 10.90e-16*exp(-1.09E-19*sco3(k,i))
     |      + 52.80e-16*exp(-3.07E-18*sco3(k,i)))
     |      +(69.60e-16*exp(-9.65E-18*sco3(k,i))
     |      +  9.39e-16*exp(-4.79E-18*sco3(k,i)))
     |      +(14.94e-16*temp1+2.76E-16*temp2))
     |      *o3i(k,i)*rmassinv_o3*avo)*sfeps

!         qtotal5(k,i) = qtotal(k,i,lat) ! diagnostic

!
! Dissociation and heating of O2 from lyman-alpha:
          pdo2  (k,i,lat) = pdo2  (k,i,lat)+pdo2lya(k,i)
          pdo2d (k,i,lat) = pdo2d (k,i,lat)+pdo2lya(k,i)*0.53
          qtotal(k,i,lat) = qtotal(k,i,lat)+pdo2lya(k,i)*
     |      o2i(k,i)*rmassinv_o2*avo*8.13816E-12

!         qtotal6(k,i) = qtotal(k,i,lat) ! diagnostic

!
! Dissociation and heating of O2 from Herzberg:
          temp1 =
     |      7.4e-10*exp(-2.5352e-15*sco2 (k,i)**0.6288)*
     |              exp(-4.6661E-18*sco3t(k,i)**0.9538)*
     |          0.9*exp(-1.4110E-25*sco2 (k,i)**1.0667)
          pdo2(k,i,lat) = pdo2(k,i,lat)+temp1*sfeps
          qtotal(k,i,lat) = qtotal(k,i,lat)+temp1*
     |      o2i(k,i)*rmassinv_o2*avo*0.46458e-12*sfeps

!         qtotal7(k,i) = qtotal(k,i,lat) ! diagnostic

!
! O(1D) photodixxociation production from CO2.
! From lyman-alpha:
          pdco2t(k,i,lat) = pdco2t(k,i,lat)+8.14e-20*pdnolya(k,i)
          pdco2d(k,i,lat) = pdco2t(k,i,lat)
!
! Total co2 photodissociation:
          if (sco2(k,i) >= 6.3e+22) then
            pdco2t(k,i,lat) = pdco2t(k,i,lat)+(2.0e-11*
     |        exp(-1.9087e-20*sco2 (k,i)**0.8675)*
     |        exp(-2.9570e-14*sco3t(k,i)**0.7582)*
     |        exp(-5.9648e-15*sco2 (k,i)**0.6172))*sfeps
          else
            pdco2t(k,i,lat) = pdco2t(k,i,lat)+(8.5E-9*
     |        exp(-3.4368E-3 *sco2 (k,i)**0.1456)*
     |        exp(-2.9570E-14*sco3t(k,i)**0.7582)*
     |        exp(-5.9648E-15*sco2 (k,i)**0.6172))*sfeps
          endif
!
! Total h2o photodissociation:
          pdh2ol(k,i,lat) = 1.53e-17*pdnolya(k,i)
!
! 8/5/04 btf: Use dh2osrb from sub jo2h2osrb.
!         temp1=exp(-1.e-7*sco2(k,i)**0.35)*sfeps
!         pdh2ot(k,i,lat) = pdh2ot(k,i,lat)+pdh2ol(k,i,lat)+
!    |                      sfac*1.2e-6*temp1
          pdh2ot(k,i,lat) = (pdh2ot(k,i,lat)+pdh2ol(k,i,lat)+
     |                       dh2osrb(k,i))*sfeps
!
! Total o2(1d) ionization:
          qo2p(k,i,lat) = qo2p(k,i,lat)+
     |      (0.549e-9*exp(-2.406e-20*sco2(k,i))+2.6e-9*exp(-8.508E-20*
     |      sco2(k,i)))*o21di(k,i)*rmassinv_o2*xnmbari(k,i)*sfeps
!
! Total CH4 photodissociation:
          pdch4a(k,i,lat) = 1.3e-6*exp(-8.4899e-21*sco2(k,i)**1.0034)*
     |               exp(-3.1398e-17*sco3t(k,i)**1.0031)*
     |               exp(-0.2776/sco3t(k,i)**0.8240)
          pdch4b(k,i,lat) = 3.888e-6*exp(-4.7152e-21*sco2(k,i)**1.0153)*
     |               exp(-4.0876e-16*sco3t(k,i)**0.9347)*
     |               exp(-0.4976/sco3t(k,i)**0.0916)
          pdch4t(k,i,lat) = pdch4t(k,i,lat)+(pdch4a(k,i,lat)+
     |                      pdch4b(k,i,lat))*sfeps
!
! New rates for H2O2, CH2O, N2O AND HO2:
!
          pdh2o2(k,i,lat) =                                 ! XJH2O2
     |      ((1.0E-4/(1.+1.6951E-17*sco3t(k,i)**0.8573))+  
     |      1.0E-4*exp(-2.0818E-23*sco2(k,i)**0.9415)*
     |      exp(-8.5266E-14*sco3t(k,i)**0.7466)*0.8721*
     |      exp(-1.4871E-20*sco2 (k,i)**0.8573))*sfeps
          pdch2oa(k,i,lat) =                                ! XJCH2OA
     |      (1.2E-4*exp(-2.3481E-40*sco2(k,i)**1.4962)*
     |      exp(-2.1444E-10*sco3t(k,i)**0.5043)*
     |      0.95*exp(-7.1534E-54*sco2(k,i)**2.0170))*sfeps
          pdch2ob(k,i,lat) =                                ! XJCH2OB
     |      (1.1E-4*exp(-6.0858E-70*sco2(k,i)**2.6383)*
     |      exp(-1.8189E-10*sco3t(k,i)**0.4812)*
     |      exp(-2.4759E-7 *sco2 (k,i)**0.1970))*sfeps
          pdn2o(k,i,lat) =                                  ! XJN2O
     |      ((3.5E-6*exp(-1.1232E-5 *sco2(k,i)**0.2638)+
     |        3.5E-7*exp(-3.4971E-18*sco2(k,i)**0.7601))*
     |      exp(-7.5897E-14*sco3t(k,i)**0.7283)*
     |      exp(-1.8121E-33*sco3t(k,i)**1.7512))*sfeps
          pdho2(k,i,lat) =                                  ! XJHO2
     |      (8.2E-4*exp(-6.4971E-16*sco2(k,i)**0.6354)*
     |      exp(-3.2108E-12*sco3t(k,i)**0.6469)*
     |      exp(-7.2877E-21*sco3t(k,i)**1.1077))*sfeps
!
          pdno2(k,i,lat) = 1.3e-2*float(idn(i,lat))*sfeps    ! XJNO2
          pdch3oo(k,i,lat) = 2.71e-5*float(idn(i,lat))*sfeps ! XJCH3OO
          pd762(k,i,lat) = 0.                                ! XJ762
!
        enddo ! k=lev0,lev1-1
      enddo ! i=lon0,lon1

!     call addfsech('QTOTAL4',' ',' ',qtotal4,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('QTOTAL5',' ',' ',qtotal5,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('QTOTAL6',' ',' ',qtotal6,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('QTOTAL7',' ',' ',qtotal7,lon0,lon1,nlevs,nlevs,lat)
!
!     call addfsech('QTOTAL2',  ' ',' ',qtotal (:,:,lat),lon0,lon1, ! F(NQ)
!    |  nlevs,nlevs,lat)
!     call addfsech('qop2p' ,   ' ',' ',qop2p  (:,:,lat),lon0,lon1, ! F(NQOP2P)
!    |  nlevs,nlevs,lat)
!     call addfsech('qop2d' ,   ' ',' ',qop2d  (:,:,lat),lon0,lon1, ! F(NQOP2D)
!    |  nlevs,nlevs,lat)
!     call addfsech('qo2p'  ,   ' ',' ',qo2p   (:,:,lat),lon0,lon1, ! F(NQO2P)
!    |  nlevs,nlevs,lat)
!     call addfsech('qop'   ,   ' ',' ',qop    (:,:,lat),lon0,lon1, ! F(NQOP)
!    |  nlevs,nlevs,lat)
!     call addfsech('qn2p'  ,   ' ',' ',qn2p   (:,:,lat),lon0,lon1, ! F(NQN2P)
!    |  nlevs,nlevs,lat)
!     call addfsech('qnp'   ,   ' ',' ',qnp    (:,:,lat),lon0,lon1, ! F(NQNP)
!    |  nlevs,nlevs,lat)
!     call addfsech('qnop'  ,   ' ',' ',qnop   (:,:,lat),lon0,lon1, ! F(NQNOP)
!    |  nlevs,nlevs,lat)
!     call addfsech('qnoplya',  ' ',' ',qnoplya(:,:,lat),lon0,lon1, ! XJNOPN
!    |  nlevs,nlevs,lat)
!     call addfsech('qnolya',   ' ',' ',qnolya (:,:,lat),lon0,lon1, ! XJNOP
!    |  nlevs,nlevs,lat)
!
!     call addfsech('pdo2',   ' ',' ',pdo2   (:,:,lat),lon0,lon1, ! (F(NRJ))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo2d',  ' ',' ',pdo2d  (:,:,lat),lon0,lon1, ! (F(NJO2D))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdn2',   ' ',' ',pdn2   (:,:,lat),lon0,lon1, ! (F(NQTEF))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdco2t', ' ',' ',pdco2t (:,:,lat),lon0,lon1, ! (F(NJCO2T))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdh2ol', ' ',' ',pdh2ol (:,:,lat),lon0,lon1, ! (F(NJH2OL))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdh2ot', ' ',' ',pdh2ot (:,:,lat),lon0,lon1, ! (F(NJH2OT))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo3d',  ' ',' ',pdo3d  (:,:,lat),lon0,lon1, ! (F(NJO3D))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo3p',  ' ',' ',pdo3p  (:,:,lat),lon0,lon1, ! (F(NJO3P))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch4t', ' ',' ',pdch4t (:,:,lat),lon0,lon1, ! (F(NJCH4T))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdnoeuv',' ',' ',pdnoeuv(:,:,lat),lon0,lon1, ! DNOEUV
!    |  nlevs,nlevs,lat)
!     call addfsech('pdnosrb',' ',' ',pdnosrb(:,:,lat),lon0,lon1, ! XJNO
!    |  nlevs,nlevs,lat)
!     call addfsech('pdh2o2', ' ',' ',pdh2o2 (:,:,lat),lon0,lon1, ! XJH2O2
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch2oa',' ',' ',pdch2oa(:,:,lat),lon0,lon1, ! XJCH2OA
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch2ob',' ',' ',pdch2ob(:,:,lat),lon0,lon1, ! XJCH2OB
!    |  nlevs,nlevs,lat)
!     call addfsech('pdn2o',  ' ',' ',pdn2o  (:,:,lat),lon0,lon1, ! XJN2O
!    |  nlevs,nlevs,lat)
!     call addfsech('pdho2',  ' ',' ',pdho2  (:,:,lat),lon0,lon1, ! XJHO2
!    |  nlevs,nlevs,lat)
!     call addfsech('pdno2',  ' ',' ',pdno2  (:,:,lat),lon0,lon1, ! XJNO2
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch3oo',' ',' ',pdch3oo(:,:,lat),lon0,lon1, ! XJCH3OO
!    |  nlevs,nlevs,lat)
!     call addfsech('pd762',  ' ',' ',pd762  (:,:,lat),lon0,lon1, ! XJ762
!    |  nlevs,nlevs,lat)
!
#ifdef VT
!     code = 118 ; state = 'qrj' ; activity='ModelCode'
      call vtend(118,ier)
#endif
      end subroutine qrj
!-----------------------------------------------------------------------
      subroutine qrj_exp(sco2,sco1,scn2,sco3,scno,vo2,tn,no,o2,o1,o3,
     |  n4s,o21d,xnmbari,cp,lev0,lev1,lon0,lon1,lat)
!
! Calculate heating, ionization, and dissociation rates.
! Same as sub qrj, but uses expo for all exp's (debug only)
!
      use input_module,only: f107
      use init_module,only: sfeps,istep ! istep for diagnostic print only
      use cons_module,only: t0,expz,avo,rmassinv_n4s,rmassinv_no,
     |  rmassinv_o2,rmassinv_o1,rmassinv_o3,rmassinv_n2,expzmid,gask,
     |  amu,expzmid_inv,p0
      use bndry_module,only: fb,b
      use chemrates_module,only: disn2p,beta9
      use o2srbc ! module in o2srbc.f for O2 heating/dissoc in Shumann-Runge
      use chapman_module,only: idn
      implicit none
!
! Args:
      integer,intent(in) :: lev0,lev1,lon0,lon1,lat
      real,dimension(lev0:lev1,lon0-2:lon1+2),intent(in) ::
     |  sco2,sco1,scn2,sco3,scno, ! chapman slant column integrals
     |  vo2,                      ! chapman vertical integral
     |  tn,no,o2,o1,o3,n4s,       ! tn and species mass mixing ratios
     |  o21d,
     |  xnmbari,              ! p0*e(-z)*barm/kT at interfaces
     |  cp                    ! specific heat
!
! VT vampir tracing:
!
#ifdef VT
#include <VT.inc>
#endif
!
! Local:
      integer :: k,i,l,nlevs,ier,levmin
      real,parameter ::
     |  do2=8.203E-12   ,
     |  do22=1.1407E-11 ,   
     |  aband=0.143     , ! shumann-runge
     |  bband=9.64E8    , ! shumann-runge
     |  cband=9.03E-19  , ! shumann-runge
     |  e3=0.33         ,
     |  hc = 1.9845E-16 , ! C(60)
     |  zpmin = -7.0      ! was zkmin
      real :: rlmeuvinv(lmax),rlmsrcinv(l1-1),factor,fmin,fmax
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  r,      ! s14
     |  rp,     ! s15
     |  taur,   ! s1
     |  tauq,   ! s2
     |  tau1,   ! s3
     |  tau2,   ! s4
     |  tau3,   ! s5
     |  etaur,  ! s6
     |  etauq   ! s7
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  o2i,    ! o2  at interfaces (s1)
     |  o1i,    ! o   at interfaces (s2)
     |  o3i,    ! o3  at interfaces (s1)
     |  n2i,    ! n2  at interfaces (s3)
     |  n4si,   ! n4s at interfaces (s4)
     |  o21di,  ! o21d at interfaces (s3)
     |  tni     ! tn  at interfaces (s6)
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  zo2,       ! o2  ionization (s8)
     |  zo1,       ! o   ionization (s9)
     |  zn2,       ! n2  ionization (s10)
     |  zn4s,      ! n4s ionization (s11)
     |  zo2_bro2,  ! o2  ionization * bro2 (s12)
     |  zn2_brn2,  ! n2  ionization * brn2 (s13)
     |  qop2pd,    ! (s4)
     |  quenchfac, ! (s8)
     |  sigchap,   ! (s9)
     |  p3f,       ! (s7)
     |  pdnolya,   ! photodissociation of NO by lyman-alpha (SJMLYA)
     |  pdo2lya,   ! photodissociation of O2 by lyman-alpha (SJO2LYA)
     |  sco3t      ! temporary chapman integral
      real,dimension(lev0:lev1,lon0:lon1) ::
     |  sum1,   ! sum(o2,o,n2)(sigma*chapman)   (s5)
     |  sum2,   ! sum(o2,o,n2)(sigma*psi/rmass) (s6)
     |  sum3    ! sum(o2,o,n2,n4s)(sigmas)      (s7)
!
! Declarations for O2 photolysis and heating in Shumann-Runge Continuum
! (SRC) and Shumann-Runge Bands (SRB) (see o2srbc.f).
! 12/11/00: declarations for o2srbc as per kibo12:
      real ::
     |  xno2(lev0:lev1),         ! o2 column density (cm-2)
     |  rho(lev0:lev1),          ! total density
     |  do2src(lev0:lev1,lon0:lon1), ! o2 dissoc in SRC  (mkdo2src output)
     |  ho2src(lev0:lev1,lon0:lon1), ! o2 heating in SRC (mkho2src output)
     |  do2srb(lev0:lev1,lon0:lon1), ! o2 dissoc in SRB  (mkdo2srb output)
     |  ho2srb(lev0:lev1,lon0:lon1), ! o2 heating in SRB (mkdo2srb output)
     | xdo2srb(lev0,lev1,lon0:lon1), ! alternate o2 srb dissoc (sub jo2h2osrb)
     | dh2osrb(lev0:lev1,lon0:lon1)  ! h2o dissoc in SRB (jo2h2osrb output)
      real :: sfac,temp1,temp2
!
!   If mks >  0, ho2src or ho2srb are returned in deg K/sec (mks)
!   If mks <= 0, ho2src or ho2srb are returned in ergs/gm-1/s-1 (cgs)
      integer,parameter :: mks=0     ! units flag for ho2src
!
      logical,parameter :: 
     |  debug=.false.     ! insert print statements
!     real,external :: expo
!
! expo() (util.F) is used only if check_exp is true. This will avoid
! NaNS fpe, but will degrade performance. Check_exp is in cons.F.
!
      if (debug) write(6,"('Enter qrj_exp: lat=',i3,' lon0,1=',2i3)") 
     |  lat,lon0,lon1
!
#ifdef VT
!     code = 118 ; state = 'qrj' ; activity='ModelCode'
      call vtbegin(118,ier)
#endif
!
! Exec:
      nlevs = lev1-lev0+1

!     write(6,"('qrj: lat=',i2,' lon0,1=',2i3,' idn(lon0:lon1,lat)=',/,
!    |  (15i3))") lat,lon0,lon1,idn(lon0:lon1,lat)
!     call addfsech('SCO2',' ',' ',sco2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCO1',' ',' ',sco1(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCN2',' ',' ',scn2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('XNMBARI',' ',' ',xnmbari(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!
! Will multiply by inverses:
      do i=1,lmax
        rlmeuvinv(i) = 1./rlmeuv(i)
      enddo
      do i=1,l1-1 ! 1,15
        rlmsrcinv(i) = 1./rlmsrc(i)
      enddo
!
! COMPUTE S14 = RSQ, S15 = RSP
! S1 = TAU(R),  S2 = TAU(Q)
!
!     write(6,"('qrj: lat=',i2,' lon0,1=',2i3,' idn(lon0:lon1,lat)=',/,
!    |  (12i3))") lat,lon0,lon1,idn(lon0:lon1,lat)
!
      do i=lon0,lon1
        do k=lev0,lev1
          taur(k,i) = sigeuv(1,49)*sco2(k,i)+sigeuv(2,49)*sco1(k,i)+
     |                sigeuv(3,49)*scn2(k,i)
          tauq(k,i) = sigeuv(1,20)*sco2(k,i)+sigeuv(2,20)*sco1(k,i)+
     |                sigeuv(3,20)*scn2(k,i)
          tau1(k,i) = 1.3*taur(k,i)
          tau2(k,i) = 2.0*taur(k,i)
          tau3(k,i) = 2.5*taur(k,i)
          if (taur(k,i) > 9.) taur(k,i) = 9.
          if (tauq(k,i) > 9.) tauq(k,i) = 9.
          if (tau1(k,i) > 9.) tau1(k,i) = 9.
          if (tau2(k,i) > 9.) tau2(k,i) = 9.
          if (tau3(k,i) > 9.) tau3(k,i) = 9.
!
! TAU(N) = EXP(-TAU(N)) FOR N = 1,3,1
!
          tau1(k,i) = expo(-tau1(k,i))
          tau2(k,i) = expo(-tau2(k,i))
          tau3(k,i) = expo(-tau3(k,i))
!
! taur = EXP(-TAU(R)), tauq = EXP(-TAU(Q))
!
          etaur(k,i) = expo(-taur(k,i))
          etauq(k,i) = expo(-tauq(k,i))
!
! rp = r(o2p), r = r(op,n2p,np)
!
          r(k,i) = etaur(k,i)+2.*(tau1(k,i)+tau2(k,i)+tau3(k,i))
          rp(k,i) = 1.5*etaur(k,i)/(r(k,i)+tauq(k,i)/taur(k,i)*
     |      etauq(k,i))
          r(k,i) = 2.4*etaur(k,i)/r(k,i)
        enddo
      enddo
      if (debug) write(6,"(/,'qrj after tau: lat=',i3)") lat
!
! Zero diffs, 7/30/03:
!     call addfsech('TAUQ',' ',' ',tauq,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('TAUR',' ',' ',taur,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('R'   ,' ',' ',r   ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('RP'  ,' ',' ',rp  ,lon0,lon1,nlevs,nlevs,lat)
!
! O2,O,N4S at interface levels:
      do i=lon0,lon1
        do k=lev0,lev1-1
          o2i (k+1,i) = 0.5*(o2(k,i)+o2(k+1,i))
          o1i (k+1,i) = 0.5*(o1(k,i)+o1(k+1,i))
          n4si(k+1,i) = 0.
        enddo
      enddo
!
! Bottom boundary:
      do i=lon0,lon1
        o2i(1,i) = .5*((b(i,1,1)+1.)*o2(1,i)+b(i,1,2)*o1(1,i)+
     |    fb(i,1,lat))
        o1i(1,i) = .5*(b(i,2,1)*o2(1,i)+(b(i,2,2)+1.)*o1(1,i)+
     |    fb(i,2,lat))
        n4si(1,i) = 0.
      enddo
!
! N2 at interfaces:
      do k=lev0,lev1
        n2i(k,:) = 1.-o2i(k,:)-o1i(k,:)
      enddo 
!
! Zero diffs, 7/30/03:
!     call addfsech('O2I'  ,' ',' ',o2i,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('O1I'  ,' ',' ',o1i,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('N2I'  ,' ',' ',n2i,lon0,lon1,nlevs,nlevs,lat)
!
! Initialize terms on current processor subdomain:
! (global module data for use by other routines)
      qtotal(lev0:lev1,lon0:lon1,lat) = 0.  ! NQ
      qop2p (lev0:lev1,lon0:lon1,lat) = 0.  ! NQOP2P
      qop2d (lev0:lev1,lon0:lon1,lat) = 0.  ! NQOP2D
      qo2p  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQO2P (see sub clearnq)
      qo2j  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQO2J (see sub clearnq)
      qop   (lev0:lev1,lon0:lon1,lat) = 0.  ! NQOP  (see sub clearnq)
      qn2p  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQN2P (see sub clearnq)
      qnp   (lev0:lev1,lon0:lon1,lat) = 0.  ! NQNP  (see sub clearnq)
      qnop  (lev0:lev1,lon0:lon1,lat) = 0.  ! NQNOP (see sub clearnq)
!
      pdo2   (lev0:lev1,lon0:lon1,lat) = 0. ! NRJ
      pdn2   (lev0:lev1,lon0:lon1,lat) = 0. ! NQTEF 
      pdco2t (lev0:lev1,lon0:lon1,lat) = 0. ! NJCO2T 
      pdh2ol (lev0:lev1,lon0:lon1,lat) = 0. ! NJH2OL 
      pdh2ot (lev0:lev1,lon0:lon1,lat) = 0. ! NJH2OT 
      pdo3d  (lev0:lev1,lon0:lon1,lat) = 0. ! NJO3D 
      pdo3p  (lev0:lev1,lon0:lon1,lat) = 0. ! NJO3P 
      pdo2d  (lev0:lev1,lon0:lon1,lat) = 0. ! NJO2D (added 5/25/04)
      pdch4t (lev0:lev1,lon0:lon1,lat) = 0. ! NJCH4T 
      pdnoeuv(lev0:lev1,lon0:lon1,lat) = 0. ! DNOEUV 
!
! Initialize ionization rates:
! (array ops on local arrays dimensioned (lon0:lon1,lev0:lev1)):
      zo2      = 0.
      zo1      = 0.
      zn2      = 0.
      zn4s     = 0.
      zo2_bro2 = 0.
      zn2_brn2 = 0.
!
! Summation over wavelength. Define parameters at task subdomains, and
!   vertical column only from zpmin to top.
!
      levmin = int((zpmin-zsb)/dz)+1 ! zpmin == -7.
!     write(6,"('qrj: lat=',i3,' zpmin=',f6.2,' levmin=',i3)")
!    |  lat,zpmin,levmin
!
! This is in sub qrj_exp() (uses expo()):
!
      do l=l1,lmax     ! 16,59 (44 wavelengths)
        sum1(:,:) = 0. ! sum(o2,o,n2)(sigeuv*chapman)   ! s5
        sum2(:,:) = 0. ! sum(o2,o,n2)(sigeuv*psi/rmass) ! s6
        sum3(:,:) = 0. ! sum(o2,o,n2,n4s)(sigmas)       ! s7
        do i=lon0,lon1
          do k=levmin,lev1
            sum1(k,i) = sum1(k,i)+sigeuv(1,l)*sco2(k,i)+
     |                            sigeuv(2,l)*sco1(k,i)+
     |                            sigeuv(3,l)*scn2(k,i)
            sum2(k,i) = sum2(k,i)+sigeuv(1,l)*o2i(k,i)*rmassinv_o2+
     |                            sigeuv(2,l)*o1i(k,i)*rmassinv_o1+
     |                            sigeuv(3,l)*n2i(k,i)*rmassinv_n2
            sum3(k,i) = sum3(k,i)+sigmas(1,l)+sigmas(2,l)+
     |                            sigmas(3,l)
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
!
        do i=lon0,lon1
          do k=levmin,lev1
            sum3(k,i) = sum3(k,i)+sigmas(4,l)
            sum1(k,i) = feuv(l)*expo(-sum1(k,i))
            qtotal (k,i,lat) = qtotal (k,i,lat)+hc*rlmeuvinv(l)*        ! f(nq)
     |        sum1(k,i)*sum2(k,i)
            pdo2   (k,i,lat) = pdo2   (k,i,lat)+sum1(k,i)*sum3(k,i)     ! f(nrj)
            pdco2t (k,i,lat) = pdco2t (k,i,lat)+sum1(k,i)*sigeuv(5,l)   ! f(njco2t)
            pdnoeuv(k,i,lat) = pdnoeuv(k,i,lat)+sum1(k,i)*sigeuv(7,l)   ! dnoeuv
            pdo3d  (k,i,lat) = pdo3d  (k,i,lat)+sum1(k,i)*sigeuv(4,l)   ! f(njo3d)
            pdch4t (k,i,lat) = pdch4t (k,i,lat)+sum1(k,i)*sigeuv(8,l)   ! f(njch4t)
            qop2d(k,i,lat)   = qop2d  (k,i,lat)+sum1(k,i)*sigmas(5,l)*  ! f(nqop2d)
     |        o1i(k,i)*rmassinv_o1
            qop2p(k,i,lat)   = qop2p  (k,i,lat)+sum1(k,i)*sigmas(6,l)*  ! f(nqop2p)
     |        o1i(k,i)*rmassinv_o1
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
!
        do i=lon0,lon1
          do k=levmin,lev1
            zo2(k,i) = zo2(k,i)+sigmas(1,l)*sum1(k,i)*o2i(k,i)*  ! s8
     |        rmassinv_o2
            zo1(k,i) = zo1(k,i)+sigmas(2,l)*sum1(k,i)*o1i(k,i)*  ! s9
     |        rmassinv_o1
            zn2(k,i) = zn2(k,i)+sigmas(3,l)*sum1(k,i)*n2i(k,i)*  ! s10
     |        rmassinv_n2
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
!
        do i=lon0,lon1
          do k=levmin,lev1
            pdn2(k,i,lat) = pdn2(k,i,lat)+(sigeuv(3,l)-sigmas(3,l))*    ! f(nqtef)
     |        sum1(k,i)*n2i(k,i)*rmassinv_n2*2.
            zn4s(k,i) = zn4s(k,i)+sigmas(4,l)*sum1(k,i)*n4si(k,i)*       ! s11
     |        rmassinv_n4s
            zo2_bro2(k,i) = zo2_bro2(k,i)+sigmas(1,l)*sum1(k,i)*o2i(k,i) ! s12
     |        *rmassinv_o2*bro2(l)
            zn2_brn2(k,i) = zn2_brn2(k,i)+sigmas(3,l)*sum1(k,i)*n2i(k,i) ! s13
     |        *rmassinv_n2*brn2(l)
          enddo ! k=levmin,lev1
        enddo ! i=lon0,lon1
      enddo ! l=l1,lmax
!
! Zero diffs, 8/5/03:
!     call addfsech('qtotal' ,' ',' ',qtotal(:,:,lat),lon0,lon1,    ! f(nq)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDO2'   ,' ',' ',pdo2(:,:,lat),lon0,lon1,      ! f(nrj)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDO3D'  ,' ',' ',pdo3d(:,:,lat),lon0,lon1,     ! f(njo3d)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDCO2T' ,' ',' ',pdco2t(:,:,lat),lon0,lon1,    ! f(njco2t)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDNOEUV',' ',' ',pdnoeuv(:,:,lat),lon0,lon1,   ! dnoeuv
!    |  nlevs,nlevs,lat)
!     call addfsech('PDCH4T' ,' ',' ',pdch4t(:,:,lat),lon0,lon1,    ! f(njch4t)
!    |  nlevs,nlevs,lat)
!     call addfsech('PDN2'   ,' ',' ',pdn2(:,:,lat),lon0,lon1,      ! f(nqtef)
!    |  nlevs,nlevs,lat)
!     call addfsech('QOP2D'  ,' ',' ',qop2d(:,:,lat),               ! f(nqop2d)
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('QOP2P',' ',' '  ,qop2p(:,:,lat),               ! f(nqop2p)
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SUM1',' ',' '  ,sum1,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SUM2',' ',' '  ,sum2,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SUM3',' ',' '  ,sum3,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('O2I' ,' ',' '  ,o2i ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('O1I' ,' ',' '  ,o1i ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('N2I' ,' ',' '  ,n2i ,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCO2',' ',' '  ,sco2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCO1',' ',' '  ,sco1(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('SCN2',' ',' '  ,scn2(:,lon0:lon1),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('ZO2' ,' ',' ',zo2 ,lon0,lon1,nlevs,nlevs,lat)  ! s8
!     call addfsech('ZO1' ,' ',' ',zo1 ,lon0,lon1,nlevs,nlevs,lat)  ! s9
!     call addfsech('ZN2' ,' ',' ',zn2 ,lon0,lon1,nlevs,nlevs,lat)  ! s10
!     call addfsech('ZN4S',' ',' ',zn4s,lon0,lon1,nlevs,nlevs,lat)  ! s11
!     call addfsech('ZO2_BRO2',' ',' ',zo2_bro2,lon0,lon1,          ! s12
!    |  nlevs,nlevs,lat)
!     call addfsech('ZN2_BRN2',' ',' ',zn2_brn2,lon0,lon1,          ! s13
!    |  nlevs,nlevs,lat)
!
! Multiply Q by efficiency factor:
      do i=lon0,lon1
        do k=lev0,lev1
          qtotal(k,i,lat) = qtotal(k,i,lat)*euveff(k)*avo
        enddo
      enddo
!     call addfsech('qeffic'  ,' ',' ',qtotal(:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!
! Contributions to O2+, O+, N2+, N+, pdn2:
! (disn2p should probably be renamed and moved here from chemrates.F)
!
      do i=lon0,lon1
        do k=lev0,lev1
!
! disn2p: dissociation of N2+ (DISN2P):
          disn2p(k,i,lat) = zn2(k,i)*r(k,i)*xnmbari(k,i)
!
! qo2p: O2+ ionization (F(NQO2P)):
          qo2p(k,i,lat) = qo2p(k,i,lat)+(zo2(k,i)*(1.+rp(k,i))-
     |      zo2_bro2(k,i))*xnmbari(k,i)
!
! qn2p: N2+ ionization (F(NQN2P)):
          qn2p(k,i,lat) = qn2p(k,i,lat)+(zn2(k,i)*(1.+r(k,i))-
     |      zn2_brn2(k,i))*xnmbari(k,i) 
!
! qnp: N+ ionization (F(NQNP)):
          qnp(k,i,lat)  = qnp(k,i,lat)+(zn4s(k,i)+zn2_brn2(k,i))*
     |      xnmbari(k,i)
!
! pdn2: photodissociation of N2 (F(NQTEF)):
          pdn2(k,i,lat) = pdn2(k,i,lat)*xnmbari(k,i)
!
! qop2pd (local) (S4):
          qop2pd(k,i) = qop2p(k,i,lat)+qop2d(k,i,lat)+zo1(k,i)+
     |      zo2_bro2(k,i)
!
! qop2p: O+(2P) ionization (F(NQOP2P)):
          qop2p(k,i,lat) = (qop2p(k,i,lat)+qop2pd(k,i)*r(k,i)*0.22)*
     |      xnmbari(k,i)
!
! qop2d: O+(2D) ionization (F(NQOP2D)):
          qop2d(k,i,lat) = (qop2d(k,i,lat)+qop2pd(k,i)*r(k,i)*0.24)*
     |      xnmbari(k,i)
!
! qop: O+ ionization (F(NQOP)):
          qop(k,i,lat) = qop(k,i,lat)+(zo1(k,i)+zo2_bro2(k,i)+
     |      qop2pd(k,i)*r(k,i)*0.56)*xnmbari(k,i)
!
! pdnosrb: photodissociation of no by srb  (XJNO)
! 8/4/04 btf: see jno call below.
!         pdnosrb(k,i,lat) = 7.0e-6*(1.+0.11*(f107-65.)/165.)*
!    |      expo(-1.e-8*sco2(k,i)**0.38)*expo(-5.e-19*sco3(k,i))*sfeps
!
! pdnolya: photodissociation of NO by lyman-alpha (local) (SJMLYA)
          pdnolya(k,i) = (0.68431  *expo(-8.22114e-21*sco2(k,i))+
     |                    0.229841 *expo(-1.77556e-20*sco2(k,i))+
     |                    0.0865412*expo(-8.22112e-21*sco2(k,i)))*
     |                    sflux(12)
!
! pdo2lya: photodissociation of O2 by lyman-alpha (local) (SJO2LYA)
          pdo2lya(k,i) = (6.0073e-21*expo(-8.2166e-21 *sco2(k,i))+
     |                   4.28569e-21*expo(-1.63296e-20*sco2(k,i))+
     |                   1.28059e-20*expo(-4.85121e-17*sco2(k,i)))
     |                   *sflux(12)
!
! qnolya: no ionization by lyman-alpha (XJNOP)
          qnolya(k,i,lat) = 2.02e-18*pdnolya(k,i)
!
! Calc of qnoplya moved to qinite.
!
! attlya: attenuation of lyman-alpha at night (SJNMLYA)
!         attlya(k,i,lat) = (0.68431  *expo(-8.22114e-21*vo2(k,i))+
!    |                       0.229841 *expo(-1.77556e-20*vo2(k,i))+
!    |                       0.0865412*expo(-8.22112e-21*vo2(k,i)))
!    |                       *sfeps
!
! qnoplya: no+ nighttime ionization by lyman-alpha (XJNOPN)
!         qnoplya(k,i,lat) = 2.02e-18*sflux(12)/100.*attlya(k,i,lat)
!
        enddo ! k=lev0,lev1
      enddo ! i=lon0,lon1
!
! Total NO dissociation pdnosrb (XJNO) is returned by sub jno (this module).
! allocate(pdnosrb(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
!
!     call jno(
!    |  pdnosrb(:,lon0:lon1,lat),  o2(:,lon0:lon1),  o1(:,lon0:lon1),
!    |  xnmbari(:,lon0:lon1)    ,sco2(:,lon0:lon1),sco3(:,lon0:lon1),
!    |  scno   (:,lon0:lon1),lev0,lev1,lon0,lon1,lat)
      call jno_exp(
     |  pdnosrb(:,lon0:lon1,lat),  o2(:,lon0:lon1),  o1(:,lon0:lon1),
     |  xnmbari(:,lon0:lon1)    ,sco2(:,lon0:lon1),sco3(:,lon0:lon1),
     |  scno   (:,lon0:lon1),lev0,lev1,lon0,lon1,lat)
! 
! Add no ionization to qnop:
      do i=lon0,lon1
        qnop(lev0,i,lat) = qnop(lev0,i,lat)+(qnolya(lev0,i,lat)+
     |    pdnoeuv(lev0,i,lat))*no(lev0,i)*xnmbari(lev0,i)*rmassinv_no
      enddo
      do i=lon0,lon1
        do k=lev0+1,lev1
          qnop(k,i,lat) = qnop(k,i,lat)+(qnolya(k,i,lat)+
     |      pdnoeuv(k,i,lat))*.5*(no(k,i)+no(k-1,i))*xnmbari(k,i)*
     |      rmassinv_no
        enddo
      enddo
!
! Minimum values for ionization rates:
      do i=lon0,lon1
        do k=lev0,lev1
          if (qo2p (k,i,lat) < 1.e-20) qo2p (k,i,lat) = 1.e-20  ! (f(nqo2p))
          if (qop  (k,i,lat) < 1.e-20) qop  (k,i,lat) = 1.e-20  ! (f(nqop))
          if (qop2p(k,i,lat) < 1.e-20) qop2p(k,i,lat) = 1.e-20  ! (f(nqop2p))
          if (qop2d(k,i,lat) < 1.e-20) qop2d(k,i,lat) = 1.e-20  ! (f(nqop2d))
          if (qn2p (k,i,lat) < 1.e-20) qn2p (k,i,lat) = 1.e-20  ! (f(nqn2p))
          if (qnp  (k,i,lat) < 1.e-20) qnp  (k,i,lat) = 1.e-20  ! (f(nqnp))
          if (pdn2 (k,i,lat) < 1.e-20) pdn2 (k,i,lat) = 1.e-20  ! (f(nqtef))
          if (qnop (k,i,lat) < 1.e-20) qnop (k,i,lat) = 1.e-20  ! (f(nqnop))
        enddo
      enddo
!
! tn at interfaces (S6):
      do i=lon0,lon1
        tni(1,i) = tn(lev1,i) ! tn bottom boundary is stored in top slot
        do k=lev0+1,lev1-1
          tni(k,i) = .5*(tn(k-1,i)+tn(k,i))
        enddo
        tni(lev1,i) = tn(lev1-1,i) ! nlevp1 <- nlev
      enddo
!
! Quench:
      do i=lon0,lon1
        do k=lev0,lev1
          factor = avo*p0/gask*expz(1)*expzmid**(2*k-3)
          quenchfac(k,i) = factor/((o2i(k,i)*rmassinv_o2+   ! s8
     |                              o1i(k,i)*rmassinv_o1+
     |                              n2i(k,i)*rmassinv_n2)*tni(k,i))
          quenchfac(k,i) = quenchfac(k,i)*
     |      (quench(1)*n2i(k,i)*rmassinv_n2+
     |       quench(2)*o2i(k,i)*rmassinv_o2)
          quenchfac(k,i) = quench(3)*quenchfac(k,i)/
     |                    (quench(4)+quenchfac(k,i))
        enddo
      enddo
!
! Zero diffs, 8/03:
!     call addfsech('qo2p',' ',' ',qo2p(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qn2p',' ',' ',qn2p(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnp',' ',' ',qnp(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdn2',' ',' ',pdn2(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop2pd',' ',' ',qop2pd,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop2p',' ',' ',qop2p(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop2d',' ',' ',qop2d(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qop',' ',' ',qop(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdnosrb',' ',' ',pdnosrb(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdnolya',' ',' ',pdnolya,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdo2lya',' ',' ',pdo2lya,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnolya',' ',' ',qnolya(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('attlya',' ',' ',attlya(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnoplya',' ',' ',qnoplya(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('quench',' ',' ',quenchfac,lon0,lon1,nlevs,nlevs,
!    |  lat)
!     call addfsech('qnop',' ',' ',qnop(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!
! Summation over wave length:
      do l=1,l1-1 ! 1->15
        do i=lon0,lon1
          do k=lev0,lev1
            sigchap(k,i) = fsrc(l)*expo(-sigsrc(l)*sco2(k,i))
            pdo3d (k,i,lat) = pdo3d (k,i,lat)+sigeuv(4,l)*sigchap(k,i)
            pdco2t(k,i,lat) = pdco2t(k,i,lat)+sigeuv(5,l)*sigchap(k,i)
            pdh2ot(k,i,lat) = pdh2ot(k,i,lat)+sigeuv(6,l)*sigchap(k,i)
            qnop  (k,i,lat) = qnop  (k,i,lat)+sigeuv(7,l)*sigchap(k,i)
          enddo ! k=lev0,lev1
        enddo ! i=lon0,lon1
      enddo ! l=1,l1-1
!
!     call addfsech('pdo3d',' ',' ',pdo3d(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdco2t',' ',' ',pdco2t(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('pdh2ot',' ',' ',pdh2ot(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('qnop',' ',' ',qnop(:,lon0:lon1,lat),
!    |  lon0,lon1,nlevs,nlevs,lat)
!
! Most code from here to end of qrj is time-gcm only (not in tiegcm):
!
! Calculate O2 photolysis and heating in Shumann-Runge Continuum (src) 
! Shumann-Runge Bands (srb):
!
      do i=lon0,lon1
        ho2src(:,i) = 0.
        do2src(:,i) = 0.
        do k=lev0,lev1
          rho(k) = (o2i(k,i)*xnmbari(k,i)+
     |              o1i(k,i)*xnmbari(k,i)+
     |              n2i(k,i)*xnmbari(k,i))*amu        ! gm/cm3
          xno2(k) = o2i(k,i)*rmassinv_o2*xnmbari(k,i)
        enddo ! lev0,lev1
!
! Pass columns at each grid point to the o2src routines.
! mkdo2src returns do2src (dissociation), mkho2src returns ho2src (heating)
!   subroutine mkdo2src(sco2,f107d,do2src,nlev)
!   subroutine mkho2src(sco2,xno2,rho,cp,f107d,ho2src,nlev,mks,lat)
!
        call mkdo2src(sco2(:,i),f107,do2src(:,i),nlevs)
        call mkho2src(sco2(:,i),xno2,rho,cp(:,i),f107,ho2src(:,i),
     |    nlevs,mks)
!
! mkdo2srb returns do2srb and ho2srb columns:
        call mkdo2srb(sco2(:,i),xno2,rho,cp(:,i),f107,sfeps,
     |    do2srb(:,i),ho2srb(:,i),nlevs,mks)
      enddo ! i=lon0,lon1
!
! Add src and srb to total heating and o2 heating and dissociation:
      do i=lon0,lon1
        do k=lev0,lev1
          qtotal(k,i,lat) = qtotal(k,i,lat)+ho2src(k,i)+ho2srb(k,i) ! f(nq)
          pdo2(k,i,lat) = pdo2(k,i,lat)+do2src(k,i)+do2srb(k,i)     ! f(nrj)
          pdo2d(k,i,lat) = pdo2d(k,i,lat)+do2src(k,i)               ! f(njo2d)
        enddo ! k=lev0,lev1
      enddo ! i=lon0,lon1
!
!     call addfsech('do2src',' ',' ',do2src,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('ho2src',' ',' ',ho2src,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('do2srb',' ',' ',do2srb,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('ho2srb',' ',' ',ho2srb,lon0,lon1,nlevs,nlevs,lat)
!
!     call addfsech('qtotal',' ',' ',qtotal(:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo2'  ,' ',' ',pdo2  (:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo2d' ,' ',' ',pdo2d (:,:,lat),lon0,lon1,
!    |  nlevs,nlevs,lat)
!
! Contributions from ozone dissociation from the Herbzberg, Hartley, 
!   Huggins and chapius bands.
! Contributions from solar lyman-alpha, SRB and Herzberg to O2 
!   dissociation and heating.
!
      sfac = 1.+0.11*(f107-65.)/165.
      do i=lon0,lon1
        do k=lev0,lev1-1
          o3i  (k+1,i) = 0.5*(o3  (k,i)+o3  (k+1,i))   ! s1
          o2i  (k+1,i) = 0.5*(o2  (k,i)+o2  (k+1,i))   ! s2
          o21di(k+1,i) = 0.5*(o21d(k,i)+o21d(k+1,i))   ! s3
        enddo ! k=lev0,lev1-1
        o3i  (lev0,i) = o3  (lev0,i)**1.5/sqrt(o3  (lev0+1,i))
        o2i  (lev0,i) = o2  (lev0,i)**1.5/sqrt(o2  (lev0+1,i))
        o21di(lev0,i) = o21d(lev0,i)**1.5/sqrt(o21d(lev0+1,i))
      enddo ! i=lon0,lon1
!
! Sub jo2h2osrb calculates o2 and h2o dissociation at SRB:
!   dh2osrb is dissociation of h2o at SRB.
!   xdo2srb is an alternate calculation of o2 dissociation at SRB,
!     and is not used in this version. (see sub mkdo2srb for o2
!     dissociation, called above)
!
      call jo2h2osrb(xdo2srb,dh2osrb,sco2,sco3,lev0,lev1,lon0,lon1,lat)
!     call addfsech('JO2SRB' ,' ',' ',xdo2srb,lon0,lon1,nlevs,nlevs,lat)
!     call addfsech('JH2OSRB',' ',' ',dh2osrb,lon0,lon1,nlevs,nlevs,lat)
!
! Scan subdomain:
      do i=lon0,lon1
        do k=lev0,lev1
          sco3t(k,i) = sco3(k,i)                         ! s9
          if (sco3t(k,i) < 1.e+5) sco3t(k,i) = 1.e+5
!
! Dissociation and heating of O3 from lyman-alpha:
          pdo3d (k,i,lat) = pdo3d (k,i,lat)+2.27e-17*pdnolya(k,i)
          qtotal(k,i,lat) = qtotal(k,i,lat)+2.27e-17*pdnolya(k,i)*
     |      o3i(k,i)*rmassinv_o3*avo*9.944e-12
          temp1 = 1.0e-3*expo(-1.5577e-13*sco3t(k,i)**0.6932)
          if (sco3t(k,i) < 1.6e+20) 
     |      temp1 = 1.04e-2*expo(-1.0217e-6*sco3t(k,i)**0.3587)
!
! Hartley bands of O3:
          pdo3d(k,i,lat) = pdo3d(k,i,lat)+
     |      (temp1*0.68+expo(-1.4912e-14*sco2   (k,i)**0.5298)*
     |      (4.053e-4  *expo(-8.1381e-16*sco3t(k,i)**0.8856)+
     |       4.7e-6    *expo(-1.5871e-14*sco3t(k,i)**0.7665))*
     |       1.085     *expo(-1.4655e-25*sco2   (k,i)**1.0743)*0.68)*
     |      sfeps
!
! Chappius and Huggins bands:
          pdo3p(k,i,lat) = pdo3p(k,i,lat)+
     |      (4.5e-4*expo(-3.4786e-11*sco2(k,i)
     |      **0.3366)*expo(-1.0061e-20*sco3t(k,i)**0.9719)
     |      +(7.5e-4 *expo(-2.7663e-19*sco3t(k,i)**1.0801)
     |      +2.5e-4/(1.+1.5772e-18   *sco3t(k,i)**0.9516))
     |      *expo(-1.0719e-10*sco2(k,i)**0.3172))*sfeps
!
! Add ozone heating to total heat:
          temp1 = expo(-5.50e-24*sco2(k,i)-6.22E-18*sco3(k,i))
          temp2 = expo(-1.34e-26*sco2(k,i)-1.66E-18*sco3(k,i))
          qtotal(k,i,lat) = qtotal(k,i,lat)+
     |      ((5.512e-16*expo(-3.16E-21*sco3(k,i))
     |      +(41.21e-16*expo(-8.94E-19*sco3(k,i))
     |      + 10.90e-16*expo(-1.09E-19*sco3(k,i))
     |      + 52.80e-16*expo(-3.07E-18*sco3(k,i)))
     |      +(69.60e-16*expo(-9.65E-18*sco3(k,i))
     |      +  9.39e-16*expo(-4.79E-18*sco3(k,i)))
     |      +(14.94e-16*temp1+2.76E-16*temp2))
     |      *o3i(k,i)*rmassinv_o3*avo)*sfeps
!
! Dissociation and heating of O2 from lyman-alpha:
          pdo2  (k,i,lat) = pdo2  (k,i,lat)+pdo2lya(k,i)
          pdo2d (k,i,lat) = pdo2d (k,i,lat)+pdo2lya(k,i)*0.53
          qtotal(k,i,lat) = qtotal(k,i,lat)+pdo2lya(k,i)*
     |      o2i(k,i)*rmassinv_o2*avo*8.13816E-12
!
! Dissociation and heating of O2 from Herzberg:
          temp1 =
     |      7.4e-10*expo(-2.5352e-15*sco2 (k,i)**0.6288)*
     |              expo(-4.6661E-18*sco3t(k,i)**0.9538)*
     |          0.9*expo(-1.4110E-25*sco2 (k,i)**1.0667)
          pdo2(k,i,lat) = pdo2(k,i,lat)+temp1*sfeps
          qtotal(k,i,lat) = qtotal(k,i,lat)+temp1*
     |      o2i(k,i)*rmassinv_o2*avo*0.46458e-12*sfeps
!
! O(1D) photodixxociation production from CO2.
! From lyman-alpha:
          pdco2t(k,i,lat) = pdco2t(k,i,lat)+8.14e-20*pdnolya(k,i)
!
! Total co2 photodissociation:
          if (sco2(k,i) >= 6.3e+22) then
            pdco2t(k,i,lat) = pdco2t(k,i,lat)+(2.0e-11*
     |        expo(-1.9087e-20*sco2 (k,i)**0.8675)*
     |        expo(-2.9570e-14*sco3t(k,i)**0.7582)*
     |        expo(-5.9648e-15*sco2 (k,i)**0.6172))*sfeps
          else
            pdco2t(k,i,lat) = pdco2t(k,i,lat)+(8.5E-9*
     |        expo(-3.4368E-3 *sco2 (k,i)**0.1456)*
     |        expo(-2.9570E-14*sco3t(k,i)**0.7582)*
     |        expo(-5.9648E-15*sco2 (k,i)**0.6172))*sfeps
          endif
!
! Total h2o photodissociation:
          pdh2ol(k,i,lat) = 1.53e-17*pdnolya(k,i)
!
! 8/5/04 btf: Use dh2osrb from sub jo2h2osrb.
!         temp1=exp(-1.e-7*sco2(k,i)**0.35)*sfeps
!         pdh2ot(k,i,lat) = pdh2ot(k,i,lat)+pdh2ol(k,i,lat)+
!    |                      sfac*1.2e-6*temp1
          pdh2ot(k,i,lat) = (pdh2ot(k,i,lat)+pdh2ol(k,i,lat)+
     |                       dh2osrb(k,i))*sfeps
!
! Total o2(1d) ionization:
          qo2p(k,i,lat) = qo2p(k,i,lat)+
     |      (0.549e-9*expo(-2.406e-20*sco2(k,i))+2.6e-9*expo(-8.508E-20*
     |      sco2(k,i)))*o21di(k,i)*rmassinv_o2*xnmbari(k,i)*sfeps
!
! Total CH4 photodissociation:
          pdch4a(k,i,lat) = 1.3e-6*expo(-8.4899e-21*sco2(k,i)**1.0034)* ! XJCH4A
     |               expo(-3.1398e-17*sco3t(k,i)**1.0031)*
     |               expo(-0.2776/sco3t(k,i)**0.8240)
          pdch4b(k,i,lat) =3.888e-6*expo(-4.7152e-21*sco2(k,i)**1.0153)* ! XJCH4B
     |               expo(-4.0876e-16*sco3t(k,i)**0.9347)*
     |               expo(-0.4976/sco3t(k,i)**0.0916)
          pdch4t(k,i,lat) = pdch4t(k,i,lat)+(pdch4a(k,i,lat)+           ! XJCH4T
     |                      pdch4b(k,i,lat))*sfeps
!
! New rates for H2O2, CH2O, N2O AND HO2:
!
          pdh2o2(k,i,lat) =                                 ! XJH2O2
     |      ((1.0E-4/(1.+1.6951E-17*sco3t(k,i)**0.8573))+  
     |      1.0E-4*expo(-2.0818E-23*sco2(k,i)**0.9415)*
     |      expo(-8.5266E-14*sco3t(k,i)**0.7466)*0.8721*
     |      expo(-1.4871E-20*sco2 (k,i)**0.8573))*sfeps
          pdch2oa(k,i,lat) =                                ! XJCH2OA
     |      (1.2E-4*expo(-2.3481E-40*sco2(k,i)**1.4962)*
     |      expo(-2.1444E-10*sco3t(k,i)**0.5043)*
     |      0.95*expo(-7.1534E-54*sco2(k,i)**2.0170))*sfeps
          pdch2ob(k,i,lat) =                                ! XJCH2OB
     |      (1.1E-4*expo(-6.0858E-70*sco2(k,i)**2.6383)*
     |      expo(-1.8189E-10*sco3t(k,i)**0.4812)*
     |      expo(-2.4759E-7 *sco2 (k,i)**0.1970))*sfeps
          pdn2o(k,i,lat) =                                  ! XJN2O
     |      ((3.5E-6*expo(-1.1232E-5 *sco2(k,i)**0.2638)+
     |        3.5E-7*expo(-3.4971E-18*sco2(k,i)**0.7601))*
     |      expo(-7.5897E-14*sco3t(k,i)**0.7283)*
     |      expo(-1.8121E-33*sco3t(k,i)**1.7512))*sfeps
          pdho2(k,i,lat) =                                  ! XJHO2
     |      (8.2E-4*expo(-6.4971E-16*sco2(k,i)**0.6354)*
     |      expo(-3.2108E-12*sco3t(k,i)**0.6469)*
     |      expo(-7.2877E-21*sco3t(k,i)**1.1077))*sfeps
!
          pdno2(k,i,lat) = 1.3e-2*float(idn(i,lat))*sfeps    ! XJNO2
          pdch3oo(k,i,lat) = 2.71e-5*float(idn(i,lat))*sfeps ! XJCH3OO
          pd762(k,i,lat) = 0.                                ! XJ762
!
        enddo ! k=lev0,lev1-1
      enddo ! i=lon0,lon1
!
!     call addfsech('QTOT_QRJ',   ' ',' ',qtotal (:,:,lat),lon0,lon1, ! F(NQ)
!    |  nlevs,nlevs,lat)
!     call addfsech('qop2p' ,   ' ',' ',qop2p  (:,:,lat),lon0,lon1, ! F(NQOP2P)
!    |  nlevs,nlevs,lat)
!     call addfsech('qop2d' ,   ' ',' ',qop2d  (:,:,lat),lon0,lon1, ! F(NQOP2D)
!    |  nlevs,nlevs,lat)
!     call addfsech('qo2p'  ,   ' ',' ',qo2p   (:,:,lat),lon0,lon1, ! F(NQO2P)
!    |  nlevs,nlevs,lat)
!     call addfsech('qop'   ,   ' ',' ',qop    (:,:,lat),lon0,lon1, ! F(NQOP)
!    |  nlevs,nlevs,lat)
!     call addfsech('qn2p'  ,   ' ',' ',qn2p   (:,:,lat),lon0,lon1, ! F(NQN2P)
!    |  nlevs,nlevs,lat)
!     call addfsech('qnp'   ,   ' ',' ',qnp    (:,:,lat),lon0,lon1, ! F(NQNP)
!    |  nlevs,nlevs,lat)
!     call addfsech('qnop'  ,   ' ',' ',qnop   (:,:,lat),lon0,lon1, ! F(NQNOP)
!    |  nlevs,nlevs,lat)
!     call addfsech('qnoplya',  ' ',' ',qnoplya(:,:,lat),lon0,lon1, ! XJNOPN
!    |  nlevs,nlevs,lat)
!     call addfsech('qnolya',   ' ',' ',qnolya (:,:,lat),lon0,lon1, ! XJNOP
!    |  nlevs,nlevs,lat)
!
!     call addfsech('pdo2',   ' ',' ',pdo2   (:,:,lat),lon0,lon1, ! (F(NRJ))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo2d',  ' ',' ',pdo2d  (:,:,lat),lon0,lon1, ! (F(NJO2D))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdn2',   ' ',' ',pdn2   (:,:,lat),lon0,lon1, ! (F(NQTEF))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdco2t', ' ',' ',pdco2t (:,:,lat),lon0,lon1, ! (F(NJCO2T))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdh2ol', ' ',' ',pdh2ol (:,:,lat),lon0,lon1, ! (F(NJH2OL))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdh2ot', ' ',' ',pdh2ot (:,:,lat),lon0,lon1, ! (F(NJH2OT))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo3d',  ' ',' ',pdo3d  (:,:,lat),lon0,lon1, ! (F(NJO3D))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdo3p',  ' ',' ',pdo3p  (:,:,lat),lon0,lon1, ! (F(NJO3P))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch4t', ' ',' ',pdch4t (:,:,lat),lon0,lon1, ! (F(NJCH4T))
!    |  nlevs,nlevs,lat)
!     call addfsech('pdnoeuv',' ',' ',pdnoeuv(:,:,lat),lon0,lon1, ! DNOEUV
!    |  nlevs,nlevs,lat)
!     call addfsech('pdnosrb',' ',' ',pdnosrb(:,:,lat),lon0,lon1, ! XJNO
!    |  nlevs,nlevs,lat)
!     call addfsech('pdh2o2', ' ',' ',pdh2o2 (:,:,lat),lon0,lon1, ! XJH2O2
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch2oa',' ',' ',pdch2oa(:,:,lat),lon0,lon1, ! XJCH2OA
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch2ob',' ',' ',pdch2ob(:,:,lat),lon0,lon1, ! XJCH2OB
!    |  nlevs,nlevs,lat)
!     call addfsech('pdn2o',  ' ',' ',pdn2o  (:,:,lat),lon0,lon1, ! XJN2O
!    |  nlevs,nlevs,lat)
!     call addfsech('pdho2',  ' ',' ',pdho2  (:,:,lat),lon0,lon1, ! XJHO2
!    |  nlevs,nlevs,lat)
!     call addfsech('pdno2',  ' ',' ',pdno2  (:,:,lat),lon0,lon1, ! XJNO2
!    |  nlevs,nlevs,lat)
!     call addfsech('pdch3oo',' ',' ',pdch3oo(:,:,lat),lon0,lon1, ! XJCH3OO
!    |  nlevs,nlevs,lat)
!     call addfsech('pd762',  ' ',' ',pd762  (:,:,lat),lon0,lon1, ! XJ762
!    |  nlevs,nlevs,lat)
!
#ifdef VT
!     code = 118 ; state = 'qrj' ; activity='ModelCode'
      call vtend(118,ier)
#endif
      end subroutine qrj_exp
!-----------------------------------------------------------------------
      subroutine init_sflux(iprint)
      use input_module,only: f107,f107a
      use init_module,only: sfeps
!
! Flux initialization once per time step, called from advance.
!
! Args:
      integer,intent(in) :: iprint
! Local:
      real :: euvflx(neuv),wave1(lmax),wave2(lmax),sfluxout(lmax)
      real :: flya,hlybr,fexvir,hlya,heiew,xuvfac
      integer :: iscale,n,nn
!
! Use f107,f107a from input_mod (these are either provided by
! the user, or obtained from GPI database, see input_module.F and
! gpi_module.F).
!
      hlybr = 0.
      fexvir = 0.
!     hlya = 3.e+11+0.4E+10*(f107-70.)                            ! tiegcm
      hlya = 1.e+11*(0.5839+0.3554*sqrt(f107a)+0.1730*(sqrt(f107) ! timegcm
     |       -sqrt(f107a)))
      heiew = 0.
!     xuvfac = 4.0 - (f107-68.0) / (243.0-68.0) ! tiegcm
      xuvfac = 2.0 - (f107-68.0) / (243.0-68.0) ! timegcm
      if (xuvfac .lt. 1.0) xuvfac = 1.0
!
! ISCALE   =0 for Hinteregger contrast ratio method
!          =1 for Hinteregger linear interpolation
!          =2 for Tobiska EUV91 model
!          =3 for Woods & Rottman 10 Nov. 1988 measurement
!          =4 for Woods & Rottman 20 Jun. 1989 measurement
! F107     daily 10.7 cm flux (1.E-22 W m-2 Hz-1)
! F107A    81-day centered average 10.7 cm flux
! HLYBR    ratio of H Ly-b 1026A flux to solar minimum value (optional)
! FEXVIR   ratio of Fe XVI 335A flux to solar minimum value (optional)
! HLYA     H Lyman-alpha flux (photons cm-2 s-1) (optional)
! HEIEW    He I 10830A equivalent width, (milliAngstroms) (optional)
! XUVFAC   factor for scaling flux 16-250A (optional)
! WAVE1    longwave bound of spectral intervals (Angstroms)
! WAVE2    shortwave bound of intervals (= WAVE1 for indiv. lines)
! SFLUX    scaled solar flux returned by subroutine (photons cm-2 s-1)
!
      iscale = 0
      call ssflux(iscale,f107,f107a,hlybr,fexvir,hlya,
     |            heiew, xuvfac, wave1, wave2, sfluxout)
      sflux = sfluxout ! transfer flux from local to module array
      call euvac(f107,f107a,euvflx)
!
! Transfer values of sflux to appropriate slots
!
      do n = l1,lmax ! 16->59
        feuv(n) = sflux(n)*sfeps
      enddo
      do n = 1,l1-1 !   1->15
        fsrc(n)   = sflux(n)*sfeps
      enddo

!     write(6,"('init_sflux: sfeps=',e12.4,' sflux=',/,(6e12.4))")
!    |  sfeps,sflux
!     write(6,"('init_sflux: sfeps=',e12.4,' feuv =',/,(6e12.4))")
!    |  sfeps,feuv
!     write(6,"('init_sflux: sfeps=',e12.4,' fsrc =',/,(6e12.4))")
!    |  sfeps,fsrc

      do n=1,neuv
        nn = n+15
        feuv(nn) = euvflx(n)*sfeps
!       write(6,"('init_sflux: n=',i3,' nn=',i3,' euvflx(n)=',e12.4,  
!    |    ' sfeps=',e12.4,' feuv(nn)=',e12.4)") n,nn,euvflx(n),sfeps,  
!    |    feuv(nn)
      enddo
!
      if (iprint > 0) then
        write(6,"(/,30('-'),' init_sflux ',30('-'))")
        write(6,"('  n   wave1   wave2       sflux      ',
     |    'euvflx        fsrc        feuv')")
        do n=1,lmax                   ! 1->59
          if (n < l1) then            ! 1->15
            write(6,"(i3,2f8.2,e12.4,12x,e12.4,e12.4)") 
     |        n,wave1(n),wave2(n),sflux(n),fsrc(n),feuv(n)
          elseif (n < l1+neuv-1) then ! 16->52
            write(6,"(i3,2f8.2,e12.4,e12.4,12x,e12.4)") 
     |        n,wave1(n),wave2(n),sflux(n),euvflx(n-l1+1),feuv(n)
          else                        ! 53->59
            write(6,"(i3,2f8.2,e12.4,12x,12x,e12.4)") 
     |        n,wave1(n),wave2(n),sflux(n),feuv(n)
          endif 
        enddo
        write(6,"(28('-'),' end init_sflux ',28('-'),/)")
      endif
      end subroutine init_sflux
!-----------------------------------------------------------------------
      subroutine init_qrj
!
! Called once per run, from init_module.
!
! TOTAL ABSORPTION COEFFICIENTS
!   sigeuv(1,N) = SIGA(O2)
!   sigeuv(2,N) = SIGA(O)
!   sigeuv(3,N) = SIGA(N2)
!   sigeuv(4,N) = SIGA(O3)
!   sigeuv(5,N) = SIGA(CO2)
!   sigeuv(6,N) = SIGA(H2O)
!   sigeuv(7,N) = SIGA(NO)
!   sigeuv(8,N) = SIGA(CH4)
!
! TOTAL IONIZATION COEFFICIENTS
!   SIGMAS(1,N) = SIGI(O2)
!   SIGMAS(2,N) = SIGI(O+(4S))
!   SIGMAS(3,N) = SIGI(N2)
!   SIGMAS(4,N) = SIGI(N)
!   SIGMAS(5,N) = SIGI(O+(2D))
!   SIGMAS(6,N) = SIGI(O+(2P))
!
      RLMEUV = (/0.17250E-04, 0.16750E-04, 0.16250E-04, 0.15750E-04,
     |           0.15250E-04, 0.14750E-04, 0.14250E-04, 0.13750E-04,
     |           0.13250E-04, 0.12750E-04, 0.12250E-04, 0.12157E-04,
     |           0.11750E-04, 0.11250E-04, 0.10750E-04, 0.10250E-04,
     |           0.10319E-04, 0.10257E-04, 0.97500E-05, 0.97702E-05,
     |           0.92500E-05, 0.87500E-05, 0.82500E-05, 0.77500E-05,
     |           0.78936E-05, 0.77041E-05, 0.76515E-05, 0.72500E-05,
     |           0.70331E-05, 0.67500E-05, 0.62500E-05, 0.62973E-05,
     |           0.60976E-05, 0.57500E-05, 0.58433E-05, 0.55437E-05,
     |           0.52500E-05, 0.47500E-05, 0.46522E-05, 0.42500E-05,
     |           0.37500E-05, 0.36807E-05, 0.32500E-05, 0.30378E-05,
     |           0.30331E-05, 0.27500E-05, 0.28415E-05, 0.25630E-05,
     |           0.22500E-05, 0.17500E-05, 0.12500E-05, 0.75000E-06,
     |           0.41000E-06, 0.27500E-06, 0.19500E-06, 0.12000E-06,
     |           0.60000E-07, 0.30000E-07, 0.15000E-07/)
!
! Absorption coefficients:
!
! O2 absorption coefficient:
      sigeuv(1,:) = (/
     |             0.50, 1.50, 3.40, 6.00,10.00,13.00,
     |            15.00,12.00, 2.20, 0.30, 3.00, 0.01,
     |             0.30, 0.10, 1.00, 1.10, 1.00, 1.60,
     |            16.53, 4.00,15.54, 9.85,20.87,27.09,
     |            26.66,25.18,21.96,29.05,25.00,26.27,
     |            26.02,25.80,26.10,25.04,22.00,25.59,
     |            24.06,21.59,20.40,19.39,18.17,18.40,
     |            17.19,16.80,16.80,15.10,15.70,13.20,
     |            10.60, 7.10, 4.00, 1.18, 0.32, 0.10,
     |             1.02, 0.14, .024, .004, .0004/)
      sigeuv(2,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,
     |             0.00, 0.00, 2.12, 4.18, 4.38, 4.23,
     |             4.28, 4.18, 4.18, 8.00,11.35,10.04,
     |            12.21,12.22,12.23,11.90,12.17,12.13,
     |            11.91,11.64,11.25,11.21, 9.64, 9.95,
     |             8.67, 7.70, 7.68, 6.61, 7.13, 6.05,
     |             5.30, 2.90, 1.60, 0.59, 0.16, 0.05,
     |             0.51, 0.07, .012, .002, .0002/)
!
! N2 absorption coefficient:
      sigeuv(3,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,
     |            36.16, 0.70,16.99,46.63,15.05,30.71,
     |            19.26,26.88,35.46,30.94,26.30,29.75,
     |            23.22,23.20,23.10,22.38,23.20,24.69,
     |            24.53,21.85,21.80,21.07,17.51,18.00,
     |            13.00,11.60,11.60,10.30,10.60, 9.70,
     |             8.00, 4.40, 1.90, 0.60, 0.24, 1.16,
     |             0.48, 0.09, .015, .003, .0003/)
!
! O3 absorption coefficient:
      sigeuv(4,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             1.25, 0.92, 0.92, 0.92, 0.92, 0.80,
     |             0.95, 0.95, 1.30, 1.86, 2.96, 2.96,
     |             3.48, 3.48, 3.64, 0.44, 0.41, 3.62,
     |             3.46, 3.51, 3.51, 3.20, 3.06, 3.06,
     |             3.20, 0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0./)
!
! CO2 absorption coefficient:
      sigeuv(5,:) = (/
     |             0.05, 0.10, 0.15, 0.30, 0.40, 0.55,
     |             0.50, 0.50, 0.80, 0.50, 0.20, 0.00,
     |             0.00,18.52,14.80,18.50,14.20,15.10,
     |            29.60,42.90,74.10,18.50,14.80,22.20,
     |            31.90,37.00,93.90,22.20,35.20,18.50,
     |            29.60,34.30,35.30,31.50,34.20,30.40,
     |            33.30,25.20,27.60,20.40,27.80,27.80,
     |            18.50,23.40,23.40,22.90,25.90,28.20,
     |            22.20,11.10,0.,0.,0.,0.,0.,0.,0.,0.,0./)
!
! H2O absorption coefficient:
      sigeuv(6,:) = (/
     |             5.00, 5.00, 5.00, 3.00, 1.50, 0.80,
     |             0.80, 1.10, 5.00, 8.00, 8.00, 0.00,
     |             4.44, 4.44, 4.44,18.52,14.10,29.60,
     |            15.56, 9.63,18.52,18.52,35.19,37.04,
     |            48.90,35.19,33.33,25.93,14.81,25.93,
     |            24.07,24.44,31.10,22.20,17.04,26.67,
     |            24.07,29.60,26.67,26.67,24.07,23.70,
     |            22.20,22.20,22.20,22.20,32.60,18.52,
     |            14.81,11.11,0.,0.,0.,0.,0.,0.,0.,0.,0./)
!
! NO absorption coefficient:
      sigeuv(7,:) = (/
     |             0.00, 0.00, 0.00, 0.00, 0.00, 0.00,
     |             0.00, 0.00, 1.85, 2.04, 2.04, 0.00,
     |             2.41, 3.70, 6.48, 7.41, 3.70, 8.15,
     |            16.67,20.74,24.07,16.67,12.96,12.96,
     |            14.44,12.96, 7.41,14.44,14.44,18.52,
     |            20.00,20.00,20.00,20.00,20.00,20.00,
     |            18.52,16.67,24.81,22.22,22.22,21.85,
     |            18.52,22.96,22.96,25.93,19.26,25.93,
     |            22.22,22.22,0.,0.,0.,0.,0.,0.,0.,0.,0./)
!
! CH4 absorption coefficient:
      sigeuv(8,:) = (/
     |            0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |            2.97, 3.07, 3.46, 3.82, 4.23, 5.00, 4.67,
     |            4.33, 4.26, 4.14, 4.10, 4.00, 3.69, 3.67, 3.33,
     |            3.20, 3.07, 3.00, 2.90, 2.70, 2.67, 2.33, 2.10,
     |            2.00, 1.67, 1.45, 1.33, 1.03, 1.02, 
     |            1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1.,1./)
!
! Ionization coefficients:
!
! O2 ionization coefficient:
      SIGMAS(1,:) = (/
     |             0.00, 0.00, 0.00, 0.00, 0.00, 0.00,
     |             0.00, 0.00, 0.00, 0.00, 0.00, 0.00,
     |             0.00, 0.00, 0.00, 0.27, 0.00, 1.00,
     |            12.22, 2.50, 9.34, 4.69, 6.12, 9.39,
     |            11.05, 9.69, 8.59,23.81,23.00,22.05,
     |            25.94,25.80,26.10,25.04,22.00,25.59,
     |            24.06,21.59,20.40,19.39,18.17,18.40,
     |            17.19,16.80,16.80,15.10,15.70,13.20,
     |            10.60, 7.10, 4.00, 1.18, 0.32, 0.10,
     |             1.02, 0.14, .024, .004, .0004/)
!
! O+(4S) ionization coefficient:
      SIGMAS(2,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,
     |             2.12, 4.18, 4.38, 4.23,
     |             4.28, 4.18, 4.18, 4.20, 4.91, 4.01,
     |             3.78, 3.79, 3.67, 3.45, 3.53, 3.52,
     |             3.45, 3.26, 3.15, 3.03, 2.51, 2.59,
     |             2.25, 1.93, 1.92, 1.65, 1.78, 1.51,
     |             1.38, 0.78, 0.46, 0.18, 0.05, 0.015,
     |             0.015, 0.02, 0.004, 0.0006, 0.00006/)
!
! N2 ionization coefficient:
      SIGMAS(3,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,
     |             0.00, 0.00, 0.00, 0.00, 0.00,16.75,
     |            10.18,18.39,23.77,23.20,23.00,25.06,
     |            23.22,23.20,23.10,22.38,23.20,24.69,
     |            24.53,21.85,21.80,21.07,17.51,18.00,
     |            13.00,11.60,11.60,10.30,10.60, 9.70,
     |             8.00, 4.40, 1.90, 0.60, 0.24, 1.16,
     |             0.48, 0.09, .015, .003, .0003/)
!
! N ionization coefficient:
      SIGMAS(4,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             1.00, 1.00, 1.00, 1.00, 1.10, 1.10,
     |             1.10, 1.20, 1.20, 1.20, 1.20, 1.20,
     |             1.10, 1.20, 1.15, 1.10, 1.00, 1.00,
     |             1.00, 0.70, 0.80, 0.65, 0.60, 0.60,
     |             0.50, 0.50, 0.40, 0.35, 0.25, 0.20,
     |             0.10, 0.10, 0.10, 0.05, 0.01, 0.,0./)
!
! O+(2D) ionization coefficient:
      SIGMAS(5,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,
     |             3.80, 6.44, 5.52, 5.49,
     |             5.50, 5.50, 5.36, 5.48, 5.46, 5.36,
     |             5.24, 5.06, 4.71, 3.86, 3.98, 3.47,
     |             2.85, 2.84, 2.38, 2.64, 2.12, 1.86,
     |             0.99, 0.51, 0.19, 0.05, 0.015, 0.015,
     |             0.02, 0.004, 0.0006, 0.00006/)
!
! O+(2P) ionization coefficient:
      SIGMAS(6,:) = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,
     |             0.50, 2.93, 2.93, 3.06,
     |             3.09, 3.16, 3.15, 3.10, 3.14, 3.04,
     |             3.48, 3.28, 3.38, 2.95, 2.93, 2.92,
     |             2.58, 2.71, 2.42, 2.07, 1.13, 0.62,
     |             0.22, 0.06, 0.02, 0.02, 0.03, 0.004,
     |             0.0007, 0.00007/)
      BRN2 = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.01, 0.04, 0.04, 0.03, 0.05, 0.05,
     |             0.15, 0.20, 0.20, 0.25, 0.32, 0.34,
     |             0.36, 0.36, 0.36, 0.36, 0.36, 0.36,
     |             0.36, 0.36, 0.36, 0.36, 0.36/)
      BRO2 = (/
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,0.,
     |             0.,0.,0.,0.,0.,0.,
     |             .025, .036, .045, .120, .155, .189,
     |             .230, 0.20, 0.20, 0.20, 0.23, 0.25,
     |             0.29, 0.33, 0.33, 0.33, 0.33, 0.33,
     |             0.33, 0.33, 0.33, 0.33, 0.33, 0.33,
     |             0.33, 0.33, 0.33, 0.33, 0.33/)
      euveff(:) = 0.05
      quench = (/7.E-11,5.E-11,3.1401E-12,9.1E-3/)
      end subroutine init_qrj
!---------------------------------------------------------------
      subroutine init_sigmas
!
! Scale coefficients (called once per run from init_module).
!
      integer :: n,nn
      do n = 1,lmax
!
! Absorption coefficients:
        sigeuv(1,n) = sigeuv(1,n)*1.E-18 ! O2
        sigeuv(2,n) = sigeuv(2,n)*1.E-18 ! O
        sigeuv(3,n) = sigeuv(3,n)*1.E-18 ! N2
        sigeuv(4,n) = sigeuv(4,n)*1.E-17 ! O3
        sigeuv(5,n) = sigeuv(5,n)*1.E-18 ! CO2
        sigeuv(6,n) = sigeuv(6,n)*1.E-18 ! H2O
        sigeuv(7,n) = sigeuv(7,n)*1.E-18 ! NO
        sigeuv(8,n) = sigeuv(8,n)*1.E-17 ! CH4
!
! Ionization coefficients:
        sigmas(1,n) = sigmas(1,n)*1.E-18 ! O2
        sigmas(2,n) = sigmas(2,n)*1.E-18 ! O+(4S)
        sigmas(3,n) = sigmas(3,n)*1.E-18 ! N2
        sigmas(4,n) = sigmas(4,n)*1.E-17 ! N
        sigmas(5,n) = sigmas(5,n)*1.E-18 ! O+(2D)
        sigmas(6,n) = sigmas(6,n)*1.E-18 ! O+(2P)
      enddo
      do n = 1,l1-1
        rlmsrc(n) = rlmeuv(n)
        sigsrc(n) = sigeuv(1,n)
      enddo
      do n=1,neuv
        brop2p(n) = 0.
        if (n > 14) brop2p(n) = 1.-brop2d(n)-brop4s(n)
        sigop2p(n)=sigio(n)*brop2p(n)
        sigop2d(n)=sigio(n)*brop2d(n)
        sigop4s(n)=sigio(n)*brop4s(n)
      enddo
      do n = 1,neuv
        nn = n+15    ! 16:52
        sigeuv(1,nn) = sigao2(n)
        sigeuv(2,nn) = sigao(n)
        sigeuv(3,nn) = sigan2(n)
        sigmas(1,nn) = sigio2(n)
        sigmas(2,nn) = sigop4s(n)
        sigmas(3,nn) = sigin2(n)
        sigmas(4,nn) = sigin(n)
        sigmas(5,nn) = sigop2d(n)
        sigmas(6,nn) = sigop2p(n)
        brn2(nn) = brn2np(n)
        bro2(nn) = bro2op(n)
      enddo
      end subroutine init_sigmas
!-----------------------------------------------------------------------
      subroutine init_euvac
!
! Called once per run from init_module.
!
! lambdas:
      wleuv1 = 
     +    (/1000.00, 1031.91, 1025.72,  950.00,  977.02,  900.00,
     +       850.00,  800.00,  750.00,  789.36,  770.41,  765.15,
     +       700.00,  703.36,  650.00,  600.00,  629.73,  609.76,
     +       550.00,  584.33,  554.31,  500.00,  450.00,  465.22,
     +       400.00,  350.00,  368.07,  300.00,  303.78,  303.31,
     +       250.00,  284.15,  256.30,  200.00,  150.00,  100.00,
     +        50.00/)
      wleuv2 =
     +    (/1050.00, 1031.91, 1025.72, 1000.00,  977.02,  950.00,
     +       900.00,  850.00,  800.00,  789.36,  770.41,  765.15,
     +       750.00,  703.36,  700.00,  650.00,  629.73,  609.76,
     +       600.00,  584.33,  554.31,  550.00,  500.00,  465.22,
     +       450.00,  400.00,  368.07,  350.00,  303.78,  303.31,
     +       300.00,  284.15,  256.30,  250.00,  200.00,  150.00,
     +       100.00/)
c
c sigao
      sigao =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 1.32e-18,
     +      4.55e-18, 3.50e-18, 5.09e-18, 3.75e-18, 3.89e-18, 4.00e-18,
     +      1.07e-17, 1.15e-17, 1.72e-17, 1.34e-17, 1.34e-17, 1.34e-17,
     +      1.30e-17, 1.31e-17, 1.26e-17, 1.21e-17, 1.21e-17, 1.19e-17,
     +      1.15e-17, 9.69e-18, 9.84e-18, 8.69e-18, 7.70e-18, 7.68e-18,
     +      6.46e-18, 7.08e-18, 6.05e-18, 5.20e-18, 3.73e-18, 1.84e-18,
     +      7.30e-19/)
c
c sigao2
      sigao2 =
     +    (/1.35e-18, 1.00e-18, 1.63e-18, 2.11e-17, 1.87e-17, 1.28e-17,
     +      8.56e-18, 1.66e-17, 2.21e-17, 2.67e-17, 1.89e-17, 2.08e-17,
     +      2.85e-17, 2.74e-17, 2.19e-17, 2.60e-17, 3.21e-17, 2.81e-17,
     +      2.66e-17, 2.28e-17, 2.60e-17, 2.46e-17, 2.31e-17, 2.19e-17,
     +      2.03e-17, 1.81e-17, 1.83e-17, 1.74e-17, 1.68e-17, 1.68e-17,
     +      1.44e-17, 1.58e-17, 1.34e-17, 1.09e-17, 7.51e-18, 3.81e-18,
     +      1.32e-18/)
c
c sigan2
      sigan2 =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 5.10e-17, 2.24e-18, 9.68e-18,
     +      2.02e-17, 1.70e-17, 3.36e-17, 1.65e-17, 1.42e-17, 1.20e-16,
     +      2.47e-17, 2.65e-17, 3.18e-17, 2.33e-17, 2.34e-17, 2.28e-17,
     +      2.28e-17, 2.24e-17, 2.41e-17, 2.45e-17, 2.35e-17, 2.32e-17,
     +      2.17e-17, 1.64e-17, 1.69e-17, 1.39e-17, 1.17e-17, 1.17e-17,
     +      1.05e-17, 1.09e-17, 1.02e-17, 8.39e-18, 4.96e-18, 2.26e-18,
     +      7.20e-19/)
c
c sigio
      sigio =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 1.32e-18,
     +      4.55e-18, 3.50e-18, 5.09e-18, 3.75e-18, 3.89e-18, 4.00e-18,
     +      1.07e-17, 1.15e-17, 1.72e-17, 1.34e-17, 1.34e-17, 1.34e-17,
     +      1.30e-17, 1.31e-17, 1.26e-17, 1.21e-17, 1.21e-17, 1.19e-17,
     +      1.15e-17, 9.69e-18, 9.84e-18, 8.69e-18, 7.70e-18, 7.68e-18,
     +      6.46e-18, 7.08e-18, 6.05e-18, 5.20e-18, 3.73e-18, 1.84e-18,
     +      7.30e-19/)
c
c sigio2
      sigio2 =
     +    (/2.59e-19, 0.00e+00, 1.05e-18, 1.39e-17, 1.55e-17, 9.37e-18,
     +      5.49e-18, 6.41e-18, 1.06e-17, 1.02e-17, 8.47e-18, 1.17e-17,
     +      2.38e-17, 2.38e-17, 2.13e-17, 2.49e-17, 3.11e-17, 2.64e-17,
     +      2.66e-17, 2.28e-17, 2.60e-17, 2.46e-17, 2.31e-17, 2.19e-17,
     +      2.03e-17, 1.81e-17, 1.83e-17, 1.74e-17, 1.68e-17, 1.68e-17,
     +      1.44e-17, 1.58e-17, 1.34e-17, 1.09e-17, 7.51e-18, 3.81e-18,
     +      1.32e-18/)
c
c sigin2
      sigin2 =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.00e+00, 0.00e+00, 1.43e-17, 8.86e-18, 8.50e-18, 6.58e-17,
     +      1.51e-17, 2.55e-17, 2.92e-17, 2.33e-17, 2.34e-17, 2.28e-17,
     +      2.28e-17, 2.24e-17, 2.41e-17, 2.45e-17, 2.35e-17, 2.32e-17,
     +      2.17e-17, 1.64e-17, 1.69e-17, 1.39e-17, 1.17e-17, 1.17e-17,
     +      1.05e-17, 1.09e-17, 1.02e-17, 8.39e-18, 4.96e-18, 2.26e-18,
     +      7.20e-19/)
c
c sigin
      sigin =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.21e-18,10.29e-18,11.71e-18,10.96e-18,11.24e-18,11.32e-18,
     +     12.10e-18,13.26e-18,12.42e-18,11.95e-18,11.21e-18,11.80e-18,
     +     11.76e-18,11.78e-18,11.77e-18,11.50e-18,11.02e-18,10.58e-18,
     +      9.56e-18, 8.15e-18, 8.30e-18, 7.30e-18, 6.41e-18, 6.40e-18,
     +      5.24e-18, 5.73e-18, 4.87e-18, 3.95e-18, 2.49e-18, 0.99e-18,
     +      0.33e-18/)
c
c O 4S
      brop4s =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 1.00e+00,
     +      1.00e+00, 1.00e+00, 1.00e+00, 1.00e+00, 1.00e+00, 1.00e+00,
     +      6.30e-01, 4.30e-01, 3.00e-01, 3.20e-01, 2.90e-01, 2.70e-01,
     +      2.80e-01, 3.00e-01, 2.90e-01, 2.80e-01, 2.80e-01, 2.80e-01,
     +      2.70e-01, 2.60e-01, 2.60e-01, 2.60e-01, 2.50e-01, 2.50e-01,
     +      2.50e-01, 2.50e-01, 2.50e-01, 2.60e-01, 2.70e-01, 2.90e-01,
     +      3.00e-01/)
c O 2D
      brop2d =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      3.70e-01, 5.70e-01, 6.60e-01, 4.60e-01, 4.70e-01, 4.60e-01,
     +      4.50e-01, 4.60e-01, 4.50e-01, 4.50e-01, 4.50e-01, 4.50e-01,
     +      4.30e-01, 4.00e-01, 4.00e-01, 4.00e-01, 3.70e-01, 3.70e-01,
     +      3.60e-01, 3.70e-01, 3.50e-01, 3.50e-01, 3.30e-01, 3.20e-01,
     +      3.20e-01/)
c O 2P
c     brop2p =
c    +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
c    +      0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
c    +      0.00e+00, 0.00e+00, 4.00e-02, 2.20e-01, 2.40e-01, 2.70e-01,
c    +      2.70e-01, 2.40e-01, 2.60e-01, 2.70e-01, 2.70e-01, 2.70e-01,
c    +      2.60e-01, 2.50e-01, 2.60e-01, 2.50e-01, 2.50e-01, 2.50e-01,
c    +      2.30e-01, 2.30e-01, 2.30e-01, 2.20e-01, 2.20e-01, 2.10e-01,
c    +      2.10e-01/)
c N2 -> N+
      brn2np =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.00e+00, 0.00e+00, 0.00e+00, 1.30e-03, 3.00e-02, 4.60e-02,
     +      4.50e-02, 1.05e-01, 9.00e-02, 1.63e-01, 2.13e-01, 2.13e-01,
     +      3.00e-01, 2.57e-01, 3.35e-01, 3.77e-01, 3.64e-01, 3.46e-01,
     +      3.85e-01/)
c O2 -> O+
      bro2op =
     +    (/0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00, 0.00e+00,
     +      0.00e+00, 0.00e+00, 8.91e-03, 3.86e-02, 3.31e-02, 7.01e-02,
     +      1.44e-01, 1.71e-01, 1.71e-01, 2.08e-01, 2.31e-01, 2.25e-01,
     +      2.37e-01, 2.59e-01, 2.49e-01, 2.99e-01, 3.53e-01, 3.53e-01,
     +      3.71e-01, 3.73e-01, 3.66e-01, 3.93e-01, 4.48e-01, 3.84e-01,
     +      0.00e+00/)
      end subroutine init_euvac
!-----------------------------------------------------------------------
      subroutine euvac(F107,F107A,EUVFLX)
!
! This EUV flux model uses the F74113 solar reference spectrum and
! ratios determined from Hinteregger's SERF1 model. It uses the daily
! F10.7 flux (F107) and the 81 day mean (F107A) as a proxy for scaling
! The fluxes are returned in EUVFLX and correspond to the 37 wavelength
! bins of Torr et al. [1979] Geophys. Res. Lett. p771.
! See Richards et al. [1994] J. Geophys. Res. p8981 for details.
!
! Called from init_sflux, once per timestep.
!
! F107   = input daily 10.7 cm flux index. (e.g. 74)
! F107A  = input 81 day average of daily F10.7 centered on current day
! EUVFLX = output array for EUV flux in units of photons/cm2/sec.
!
      real :: wleuv1,wleuv2,sigao,sigao2,sigan2,sigio,sigio2,
     |  sigin2,brop4s,brop2d,brop2p,sigop2p,sigop2d,sigop4s,
     |  sigin,brn2np,bro2op
!
! Args:
      real,intent(in) :: f107,f107a
      real,intent(out) :: euvflx(neuv)
!
! Local:
      integer :: i
      real :: AFAC(neuv),F74113(neuv),flxfac
      real :: afac_tmp(neuv),f74113_tmp(neuv)
!
! F74113 reference spectrum (doubled below 150-250 A, tripled <150)
! Will be multiplied by 1.0E9 later
!
      F74113=
     | (/1.200,0.450,4.800,3.100,0.460,0.210,1.679,0.800,6.900,
     |   0.965,0.650,0.314,0.383,0.290,0.285,0.452,0.720,1.270,
     |   0.357,0.530,1.590,0.342,0.230,0.360,0.141,0.170,0.260,
     |   0.702,0.758,1.625,3.537,3.000,4.400,1.475,3.500,2.100,
     |   2.467/)
!
! Scaling factors(Ai) for the EUV flux
      AFAC=
     | (/1.0017E-02,7.1250E-03,1.3375E-02,1.9450E-02,2.7750E-03,
     |   1.3768E-01,2.6467E-02,2.5000E-02,3.3333E-03,2.2450E-02,
     |   6.5917E-03,3.6542E-02,7.4083E-03,7.4917E-03,2.0225E-02,
     |   8.7583E-03,3.2667E-03,5.1583E-03,3.6583E-03,1.6175E-02,
     |   3.3250E-03,1.1800E-02,4.2667E-03,3.0417E-03,4.7500E-03,
     |   3.8500E-03,1.2808E-02,3.2750E-03,4.7667E-03,4.8167E-03,
     |   5.6750E-03,4.9833E-03,3.9417E-03,4.4167E-03,5.1833E-03,
     |   5.2833E-03,4.3750E-03/)
!
! 6/20/03 btf: Reverse order of F74113 and AFAC to correspond to descending
! wavelength bins (as in wleuv1,2 and WAVEL,WAVES):
      do i=1,neuv
        afac_tmp(i) = afac(neuv-i+1)  
        f74113_tmp(i) = f74113(neuv-i+1)  
      enddo
      afac = afac_tmp      ! whole array
      f74113 = f74113_tmp
!
! Loop through the wavelengths calculating the scaling factors and
!   the resulting solar flux.
! The scaling factors are restricted to be greater than 0.8
!
      DO 50 I=1,neuv
        FLXFAC=(1.0 + AFAC(I) * (0.5*(F107+F107A) - 80.0))
        IF(FLXFAC.LT.0.8) FLXFAC=0.8
        EUVFLX(I)=F74113(I) * FLXFAC * 1.0E9
 50   CONTINUE
      end subroutine euvac
!-----------------------------------------------------------------------
      subroutine ssflux (iscale, f107, f107a, hlybr, fexvir, hlya,
     |                   heiew, xuvfac, wave1, wave2, sfluxout)
      integer,parameter :: lmax=59
!
! Args:
      integer,intent(in) :: iscale
      real,intent(in) :: f107,f107a,hlybr,fexvir,hlya,heiew,xuvfac
      real,intent(out) :: wave1(lmax),wave2(lmax),sfluxout(lmax) 
!
! Local:
      real ::
     |          wavel(lmax), waves(lmax), rflux(lmax), xflux(lmax),
     |          scale1(lmax), scale2(lmax),
     |          tchr0(lmax), tchr1(lmax), tchr2(lmax),
     |          tcor0(lmax), tcor1(lmax), tcor2(lmax),
     |          war1(lmax), war2(lmax),
     |          b1(3), b2(3)
      real :: frat,r1,r2,hlymod,heimod,xuvf
      integer :: l
!
! regression coefficients which reduce to solar min. spectrum:
      b1 = (/1.0, 0.0138, 0.005/) 
      b2 = (/1.0, 0.59425, 0.3811/)
!
! 'best fit' regression coefficients, commented out, for reference:
!     b1 = (/1.31, 0.01106, 0.00492/, B2/-6.618, 0.66159, 0.38319/)
!
      WAVEL =   (/1750.00, 1700.00, 1650.00, 1600.00, 1550.00, 1500.00,
     |            1450.00, 1400.00, 1350.00, 1300.00, 1250.00, 1215.67,
     |            1200.00, 1150.00, 1100.00, 1050.00, 1031.91, 1025.72,
     |            1000.00,  977.02,  950.00,  900.00,  850.00,  800.00,
     |             789.36,  770.41,  765.15,  750.00,  703.31,  700.00,
     |             650.00,  629.73,  609.76,  600.00,  584.33,  554.37,
     |             550.00,  500.00,  465.22,  450.00,  400.00,  368.07,
     |             350.00,  303.78,  303.31,  300.00,  284.15,  256.30,
     |             250.00,  200.00,  150.00,  100.00,   50.00,   32.00,
     |              23.00,   16.00,    8.00,    4.00,    2.00/)
      WAVES =   (/1700.00, 1650.00, 1600.00, 1550.00, 1500.00, 1450.00,
     |            1400.00, 1350.00, 1300.00, 1250.00, 1200.00, 1215.67,
     |            1150.00, 1100.00, 1050.00, 1000.00, 1031.91, 1025.72,
     |             950.00,  977.02,  900.00,  850.00,  800.00,  750.00,
     |             789.36,  770.41,  765.15,  700.00,  703.31,  650.00,
     |             600.00,  629.73,  609.76,  550.00,  584.33,  554.37,
     |             500.00,  450.00,  465.22,  400.00,  350.00,  368.07,
     |             300.00,  303.78,  303.31,  250.00,  284.15,  256.30,
     |             200.00,  150.00,  100.00,   50.00,   32.00,   23.00,
     |              16.00,    8.00,    4.00,    2.00,    1.00/)
      RFLUX =    (/322.00,  168.00,   95.00,   62.00,   44.00,   25.00,
     |              16.90,   11.80,   19.50,    4.10,   11.10,  249.00,
     |               2.78,    0.70,    3.07,    3.64,    3.18,    4.38,
     |               1.78,    5.96,    4.22,    4.43,    1.93,    0.87,
     |               0.79,    0.24,    0.20,    0.17,    0.39,    0.22,
     |               0.17,    1.50,    0.45,    0.48,    1.58,    0.80,
     |               0.51,    0.31,    0.18,    0.39,    0.21,    0.74,
     |               0.87,    6.00,    0.24,    0.84,    0.10,    0.27,
     |               0.92,    1.84,    0.13,    0.38,  0.0215,  0.0067,
     |              1.E-3,   2.E-3,   1.E-5,   5.E-8,   1.E-10/)
      XFLUX =    (/354.00,  191.00,  110.00,   76.00,   55.00,   28.00,
     |              19.60,   14.30,   25.30,    5.00,   17.20,  401.00,
     |               6.26,    1.51,    6.11,    8.66,    9.04,   13.12,
     |               4.42,   13.18,   12.03,   13.29,    5.01,    2.18,
     |               1.59,    0.67,    0.43,    0.43,    0.72,    0.46,
     |               0.48,    3.02,    1.46,    1.02,    4.86,    1.59,
     |               1.57,    1.67,    0.36,    0.99,    2.20,    1.39,
     |               5.63,   11.28,    2.50,    4.14,    3.16,    0.59,
     |               3.70,    4.85,    0.34,    1.15,    0.18,    0.08,
     |             2.5E-2,   5.E-2,   8.E-4,   3.E-5,   5.E-7/)
      SCALE1 = (/     0.0,     0.0,     0.0,     0.0,     0.0,     0.0,
     |                0.0,     0.0,     0.0,     0.0,     0.0,     0.0,
     |            1692.09,  405.95, 1516.20, 2731.70, 3314.57, 4375.00,
     |            1316.91, 3621.91, 3908.56, 4432.54, 1541.21,  531.73,
     |             364.83,    0.00,  116.00,  129.41,  162.48,   94.07,
     |              41.29,  709.50,    0.00,  268.47, 1561.05,  367.64,
     |             290.06,  184.36,    0.00,   86.15,    7.50,    0.00,
     |               0.00, 2220.00,    0.00,   61.00,    0.00,   86.95,
     |             206.00,  135.89,   60.35,  157.12,    7.06,    0.75,
     |               0.00,    0.00,    0.00,    0.00,    0.00/)
      SCALE2 =  (/   0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    5.34,    0.00,    0.00,    0.00,    0.54,
     |               3.30,    0.00,   12.60,    0.00,    0.00,    0.00,
     |               5.34,   11.63,    2.28,    5.56,   24.93,    8.16,
     |              60.69,    0.00,   28.20,   45.90,   40.80,    1.27,
     |              35.47,   42.80,    1.12,    6.19,    1.26,    0.69,
     |               0.23,    0.46,  7.6E-3,  2.9E-4,  4.8E-6/)
      TCHR0 = (/
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,-4.290E+00,-5.709E+00,-8.493E+00,
     |-1.161E+00,-3.429E+00,-5.464E+00,-6.502E+00,-1.912E+00,-4.034E-01,
     |-1.448E-01, 0.000E+00,-9.702E-02,-6.591E-02,-2.338E-02,-1.273E-01,
     |-2.406E-01,-3.351E-01, 0.000E+00,-1.465E+00,-2.405E+00,-7.975E-02,
     |-4.197E-01,-1.971E-01, 0.000E+00,-5.895E-02,-5.815E-03, 0.000E+00,
     | 0.000E+00, 2.138E-01, 0.000E+00,-7.713E-02, 0.000E+00,-3.035E-02,
     |-2.039E-01,-1.749E-01,-1.041E-01,-2.638E-01,-1.094E-02, 0.000E+00,
     |       0.0,       0.0,       0.0,       0.0,       0.0/)
      TCHR1 = (/
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,-3.023E-13,-3.745E-13,-5.385E-13,
     |-1.211E-13,-3.868E-13,-3.646E-13,-4.125E-13,-1.527E-13,-4.753E-14,
     |-3.411E-14, 0.000E+00,-1.190E-14,-1.034E-14,-1.343E-14,-1.539E-14,
     |-5.174E-14,-6.934E-14, 0.000E+00,-1.215E-13,-1.537E-13,-2.024E-14,
     |-4.596E-14,-1.562E-14, 0.000E+00,-1.221E-14,-1.123E-15, 0.000E+00,
     | 0.000E+00,-2.263E-13, 0.000E+00,-1.508E-14, 0.000E+00,-1.744E-14,
     |-2.100E-14,-1.805E-14,-8.224E-15,-1.919E-14,-7.944E-16, 0.000E+00,
     |       0.0,       0.0,       0.0,       0.0,       0.0/)
      TCHR2 = (/
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0, 3.275E-11, 4.057E-11, 6.160E-11,
     | 1.312E-11, 4.189E-11, 4.167E-11, 4.716E-11, 1.654E-11, 5.150E-12,
     | 3.901E-12, 0.000E+00, 1.289E-12, 1.120E-12, 1.455E-12, 1.667E-12,
     | 5.604E-12, 7.931E-12, 0.000E+00, 1.317E-11, 1.757E-11, 2.194E-12,
     | 4.978E-12, 1.693E-12, 0.000E+00, 1.324E-12, 1.285E-13, 0.000E+00,
     | 0.000E+00, 2.586E-11, 0.000E+00, 1.724E-12, 0.000E+00, 1.889E-12,
     | 2.400E-12, 2.063E-12, 8.911E-13, 2.193E-12, 9.090E-14, 0.000E+00,
     |       0.0,       0.0,       0.0,       0.0,       0.0/)
      TCOR0 = (/
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0, 0.000E+00, 0.000E+00, 0.000E+00,
     | 0.000E+00, 0.000E+00, 0.000E+00, 0.000E+00, 0.000E+00,-6.060E-02,
     | 0.000E+00,-3.399E-02, 0.000E+00, 0.000E+00, 0.000E+00, 4.866E-02,
     |-1.762E-01, 0.000E+00,-2.412E-01, 0.000E+00, 0.000E+00, 0.000E+00,
     |-4.743E-01,-9.713E-01, 5.891E-02,-1.263E-01,-1.246E+00, 2.870E-01,
     |-4.659E+00, 0.000E+00,-1.058E+00,-3.821E+00,-1.874E+00, 0.000E+00,
     |-1.896E+00,-8.505E-01,-2.101E-04,-2.012E-01,-6.097E-02,-2.925E-02,
     |-4.875E-03,       0.0,       0.0,       0.0,       0.0/)
      TCOR1 = (/
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0, 0.000E+00, 0.000E+00, 0.000E+00,
     | 0.000E+00, 0.000E+00, 0.000E+00, 0.000E+00, 0.000E+00, 2.877E-03,
     | 0.000E+00, 1.760E-03, 0.000E+00, 0.000E+00, 0.000E+00, 3.313E-04,
     | 3.643E-03, 0.000E+00, 5.225E-03, 0.000E+00, 0.000E+00, 0.000E+00,
     | 4.085E-03, 1.088E-02, 8.447E-04, 3.237E-03, 1.907E-02, 2.796E-03,
     | 4.460E-02, 0.000E+00, 1.007E-02, 3.481E-02, 1.604E-02, 0.000E+00,
     | 2.029E-02, 2.160E-02, 6.342E-04, 3.594E-03, 5.503E-04, 2.687E-04,
     | 4.479E-05,       0.0,       0.0,       0.0,       0.0/)
      TCOR2 = (/
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0,       0.0,       0.0,       0.0,
     |       0.0,       0.0,       0.0, 0.000E+00, 0.000E+00, 0.000E+00,
     | 0.000E+00, 0.000E+00, 0.000E+00, 0.000E+00, 0.000E+00, 1.846E-03,
     | 0.000E+00, 1.127E-03, 0.000E+00, 0.000E+00, 0.000E+00, 1.891E-04,
     | 2.326E-03, 0.000E+00, 2.801E-03, 0.000E+00, 0.000E+00, 0.000E+00,
     | 2.446E-03, 7.121E-03, 5.204E-04, 1.983E-03, 1.204E-02, 1.721E-03,
     | 2.911E-02, 0.000E+00, 7.177E-03, 2.272E-02, 9.436E-03, 0.000E+00,
     | 1.316E-02, 1.398E-02, 4.098E-04, 2.328E-03, 3.574E-04, 1.745E-04,
     | 2.909E-05,       0.0,       0.0,       0.0,       0.0/)
      WAR1 =  (/     0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    0.00,    3.80,    6.25,    4.93,    6.06,
     |               2.70,    7.07,    8.62,    9.60,    4.54,    2.37,
     |               0.82,    0.33,    0.24,    0.67,    0.28,    0.55,
     |               1.56,    1.11,    0.77,    1.32,    1.71,    0.44,
     |               1.11,    0.95,    0.39,    0.81,    2.00,    1.49,
     |               6.81,    5.07,    1.63,    5.62,    2.08,    0.59,
     |               3.89,    5.19,    0.35,    1.18,    0.099,   0.04,
     |               0.007,   0.00,    0.00,    0.00,    0.00/)
      WAR2 =  (/     0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    0.00,    0.00,    0.00,    0.00,    0.00,
     |               0.00,    0.00,   20.80,   17.90,    9.30,   14.30,
     |               6.90,   12.00,   15.60,   18.60,   10.10,    4.30,
     |              12.40,    8.00,    3.60,    1.80,    0.50,    1.40,
     |               3.90,    2.60,    1.60,    3.40,    4.10,    0.70,
     |               4.30,    4.30,    3.80,    2.60,    6.08,    1.35,
     |              12.60,    9.78,    2.96,   10.20,    4.11,    6.68,
     |               6.62,    8.07,    0.47,    1.73,    0.17,    0.075,
     |               0.012,   0.00,    0.00,    0.00,    0.00/)
C
C Linear Interpolation between SC#21REFW and F79050:
C
      FRAT = (F107-68.) / (243.-68.)
      DO 200 L=1,LMAX
        sfluxout(L) = RFLUX(L) + (XFLUX(L)-RFLUX(L)) * FRAT
  200 CONTINUE
C
C Hinteregger contrast ratio method:
C
      IF (ISCALE .EQ. 0) THEN
        IF (HLYBR .GT. 0.001) THEN
          R1 = HLYBR
        ELSE
          R1 =  B1(1) + B1(2)*(F107A-71.5) + B1(3)*(F107-F107A+3.9)
        ENDIF
        IF (FEXVIR .GT. 0.001) THEN
          R2 = FEXVIR
        ELSE
          R2 =  B2(1) + B2(2)*(F107A-71.5) + B2(3)*(F107-F107A+3.9)
        ENDIF
        DO 100 L=13,LMAX
          sfluxout(L) = (RFLUX(L) + ((R1-1.)*SCALE1(L)
     |                         +  (R2-1.)*SCALE2(L)) / 1000.)
  100   CONTINUE
      ENDIF
C
C Tobiska EUV91 Method:
C
      IF (ISCALE .EQ. 2) THEN
        IF (HLYA .GT. 0.001) THEN
          HLYMOD = HLYA
        ELSE
          IF (HEIEW .GT. 0.001) THEN
             HLYMOD = HEIEW * 3.77847E9 + 8.40317E10
          ELSE
             HLYMOD = 8.70E8 * F107 + 1.90E11
             HLYMOD = 8.70E8 * F107 + 1.90E11
          ENDIF
        ENDIF
        IF (HEIEW .GT. 0.001) THEN
          HEIMOD = HEIEW * 3.77847E9 + 8.40317E10
        ELSE
          HEIMOD = HLYMOD
        ENDIF
        DO 500 L=16,55
          sfluxout(L) = TCHR0(L) + TCHR1(L)*HLYMOD + TCHR2(L)*HEIMOD
     |             + TCOR0(L) + TCOR1(L)*F107 + TCOR2(L)*F107A
 500    CONTINUE
      ENDIF
C
C Woods and Rottman (10 Nov. 1988) spectrum:
C
      IF (ISCALE .EQ. 3) THEN
        DO 550 L=15,55
        sfluxout(L) = WAR1(L)
  550   CONTINUE
      ENDIF
C
C Woods and Rottman (20 June 1989) spectrum:
C
      IF (ISCALE .EQ. 4) THEN
        DO 560 L=15,55
        sfluxout(L) = WAR2(L)
  560   CONTINUE
      ENDIF
C
C Substitute in H Lyman-alpha and XUVFAC if provided:
C
      IF (HLYA .GT. 0.001) sfluxout(12) = HLYA / 1.E9
      IF (XUVFAC .GT. 0.001) THEN
        XUVF = XUVFAC
      ELSE
        XUVF = 1.0
      ENDIF
C
C Convert from gigaphotons to photons, etc.:
C
      DO 600 L=1,LMAX
        WAVE1(L) = WAVEL(L)
        WAVE2(L) = WAVES(L)
        IF (sfluxout(L) .LT. 0.0) sfluxout(L) = 0.0
        IF (WAVEL(L).LT.251.0 .AND. WAVES(L).GT.15.0)
     |     sfluxout(L)=sfluxout(L)*XUVF
        sfluxout(L) = sfluxout(L) * 1.E9
  600 CONTINUE
C
      end subroutine ssflux
!-----------------------------------------------------------------------
      subroutine alloc_q(lon0,lon1,lat0,lat1)
!
! Args:
      integer,intent(in) :: lon0,lon1,lat0,lat1
!
! Local:
      integer :: istat
!
! Allocate heating and ionization rates:
!
      allocate(qtotal(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qtotal: stat=',i3)") istat
      allocate(qop2p(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qop2p: stat=',i3)") istat
      allocate(qop2d(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qop2d: stat=',i3)") istat
      allocate(qo2p(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qo2p: stat=',i3)") istat
      allocate(qo2j(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qo2j: stat=',i3)") istat
      allocate(qop(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qop: stat=',i3)") istat
      allocate(qn2p(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qn2p: stat=',i3)") istat
      allocate(qnp(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qnp: stat=',i3)") istat
      allocate(qnop(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qnop: stat=',i3)") istat
      allocate(qnoplya(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qnoplya: stat=',i3)") istat
      allocate(qnolya(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' qnolya: stat=',i3)") istat
!
! Allocate photodissociation rates:
!
      allocate(pdo2(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdo2: stat=',i3)") istat
      allocate(pdo2d(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdo2d: stat=',i3)") istat
      allocate(pdn2(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdn2: stat=',i3)") istat
      allocate(pdco2t(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdco2t: stat=',i3)") istat
      allocate(pdco2d(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdco2d: stat=',i3)") istat
      allocate(pdh2ol(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdh2ol: stat=',i3)") istat
      allocate(pdh2ot(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdh2ot: stat=',i3)") istat
      allocate(pdo3d(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdo3d: stat=',i3)") istat
      allocate(pdo3p(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdo3p: stat=',i3)") istat
      allocate(pdch4a(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdch4a: stat=',i3)") istat
      allocate(pdch4b(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdch4b: stat=',i3)") istat
      allocate(pdch4t(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdch4t: stat=',i3)") istat
      allocate(pdnoeuv(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdnoeuv: stat=',i3)") istat
      allocate(pdnosrb(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdnosrb: stat=',i3)") istat
!     allocate(attlya(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
!     if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
!    |  ' attlya: stat=',i3)") istat
      allocate(pdh2o2(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdh2o2: stat=',i3)") istat
      allocate(pdch2oa(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdch2oa: stat=',i3)") istat
      allocate(pdch2ob(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdch2ob: stat=',i3)") istat
      allocate(pdn2o(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdn2o: stat=',i3)") istat
      allocate(pdho2(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdho2: stat=',i3)") istat
      allocate(pdno2(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdno2: stat=',i3)") istat
      allocate(pdch3oo(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pdch3oo: stat=',i3)") istat
      allocate(pd762(nlevp1,lon0:lon1,lat0:lat1),stat=istat)
      if (istat /= 0) write(6,"('>>> alloc_q: error allocating',
     |  ' pd762: stat=',i3)") istat
      end subroutine alloc_q
!-----------------------------------------------------------------------
      subroutine jno(xjno,o2mmr,o1mmr,xnmbar,sco2,sco3,scno,
     |  lev0,lev1,lon0,lon1,lat)
!
! Return NO dissociation in xjno. This is called from QRJ, and replaces 
!   former single-statement calculation of XJNO in QRJ.
! 8/4/04 btf: adapted for timegcm1.
!
      use cons_module,only: rmassinv_n2
      implicit none
!
! Args:
      integer,intent(in) :: lev0,lev1,lon0,lon1,lat
      real,intent(out) :: xjno(lev0:lev1,lon0:lon1)
      real,intent(in),dimension(lev0:lev1,lon0:lon1) :: 
     |  o2mmr,o1mmr,    ! o2,o densities (mmr)
     |  xnmbar,         ! mbar (for n2 number density)
     |  sco2,sco3,scno  ! slant column integrations from chapman
!
! Local:
      integer :: k,i,il
      real :: prob,xnn2,tauo3,sum,xjno1,xjno2,xjno3
!
! O2(5,0) BAND, NO D(0-0) BAND 190.2-192.5NM
!
      real,dimension(6) :: so2,w1no,s1no,w2no,s2no
      data so2  /1.12e-23,2.45e-23,7.19e-23,3.04e-22,1.75e-21,1.11e-20/
      data w1no /0.      ,5.12e-2 ,1.36e-1 ,1.65e-1 ,1.41e-1 ,4.5e-2  /
      data s1no /0.      ,1.32e-18,6.35e-19,7.09e-19,2.18e-19,4.67e-19/
      data w2no /0.      ,5.68e-3 ,1.52e-2 ,1.83e-2 ,1.57e-2 ,5.0e-3  /
      data s2no /0.      ,4.41e-17,4.45e-17,4.5e-17 ,2.94e-17,4.35e-17/
!
! O2(9,0) BAND, NO D(1-0) BAND 183.1-184.6NM
!
      real,dimension(6) :: s1o2,w11no,s11no,w12no,s12no
      data s1o2  /1.35e-22,2.99e-22,7.33e-22,3.07e-21,1.69e-20,1.66e-19/
      data w11no /0.      ,0.      ,1.93e-03,9.73e-2 ,9.75e-2 ,3.48e-2 /
      data s11no /0.      ,0.      ,3.05e-21,5.76e-19,2.29e-18,2.21e-18/
      data w12no /0.      ,0.      ,2.14e-4 ,1.08e-2 ,1.08e-2 ,3.86e-3 /
      data s12no /0.,0.,3.2e-21,5.71e-17,9.09e-17,6.0e-17/
!
! O2(10,0) BAND, NO D(1-0) BAND 181.6-183.1NM
!
      real,dimension(6) :: s2o2,w21no,s21no,w22no,s22no
      data s2o2  /2.97e-22,5.83e-22,2.05e-21,8.19e-21,4.8e-20 ,2.66e-19/
      data w21no /4.5e-2  ,1.8e-1  ,2.25e-1 ,2.25e-1 ,1.8e-1  ,4.5e-2  /
      data s21no /1.8e-18 ,1.5e-18 ,5.01e-19,7.20e-20,6.72e-20,1.49e-21/
      data w22no /5.e-3   ,2.e-2   ,2.5e-2  ,2.5e-2  ,2.0e-2  ,5.0e-3  /
      data s22no /1.4e-16 ,1.52e-16,7.e-17  ,2.83e-17,2.73e-17,6.57e-18/
!
! O3 ABSORPTION CROSS SECTION FOR EACH BAND
!
      real :: so3a1,so3a2,so3a3
      data so3a1,so3a2,so3a3 /42.8e-20, 62.2e-20, 68.8e-20/
      real :: dlam,dlam1,dlam2
      data dlam,dlam1,dlam2 /2.3, 1.5, 1.5/
!
! Loop over zp and longitude:
!
      do i=lon0,lon1
        do k=lev0,lev1
          xnn2 = (1.-o2mmr(k,i)-o1mmr(k,i))*xnmbar(k,i)*rmassinv_n2
          prob = 1.65e+9 / (5.1e+7 + 1.65e+9 + 1.5e-9 * xnn2)
          tauo3 = exp(-so3a1*sco3(k,i))
          sum = 0.
          do il = 1,6
            sum = sum + 
     |       (exp(-so2(il) *sco2(k,i))*(w1no(il)*s1no(il)*
     |        exp(-s1no(il)*scno(k,i))+w2no(il)*s2no(il)*
     |        exp(-s2no(il)*scno(k,i))))
          enddo
          xjno1 = dlam*3.98e+11*tauo3*sum*prob
!
          tauo3 = exp(-so3a2*sco3(k,i))
          sum = 0.
          do il = 1,6
            sum = sum +
     |       (exp(-s1o2(il) *sco2(k,i))*(w11no(il)*s11no(il)*
     |        exp(-s11no(il)*scno(k,i))+w12no(il)*s12no(il)*
     |        exp(-s12no(il)*scno(k,i))))
          enddo
          xjno2 = dlam1*2.21e+11*tauo3*sum*prob
!
          tauo3 = exp(-so3a3*sco3(k,i))
          sum = 0.
          do il = 1,6
            sum = sum +
     |       (exp(-s2o2(il) *sco2(k,i))*(w21no(il)*s21no(il)*
     |        exp(-s21no(il)*scno(k,i))+w22no(il)*s22no(il)*
     |        exp(-s22no(il)*scno(k,i))))
          enddo
          xjno3 = dlam2*2.30e+11*tauo3*sum*prob
!
! Total NO dissociation:
!
          xjno(k,i) = xjno1+xjno2+xjno3
        enddo
      enddo
      end subroutine jno
!-----------------------------------------------------------------------
      subroutine jno_exp(xjno,o2mmr,o1mmr,xnmbar,sco2,sco3,scno,
     |  lev0,lev1,lon0,lon1,lat)
!
! Return NO dissociation in xjno. This is called from QRJ, and replaces 
!   former single-statement calculation of XJNO in QRJ.
! 8/4/04 btf: adapted for timegcm1.
!
      use cons_module,only: rmassinv_n2
      implicit none
!
! Args:
      integer,intent(in) :: lev0,lev1,lon0,lon1,lat
      real,intent(out) :: xjno(lev0:lev1,lon0:lon1)
      real,intent(in),dimension(lev0:lev1,lon0:lon1) :: 
     |  o2mmr,o1mmr,    ! o2,o densities (mmr)
     |  xnmbar,         ! mbar (for n2 number density)
     |  sco2,sco3,scno  ! slant column integrations from chapman
!
! Local:
      integer :: k,i,il
      real :: prob,xnn2,tauo3,sum,xjno1,xjno2,xjno3
!
! O2(5,0) BAND, NO D(0-0) BAND 190.2-192.5NM
!
      real,dimension(6) :: so2,w1no,s1no,w2no,s2no
      data so2  /1.12e-23,2.45e-23,7.19e-23,3.04e-22,1.75e-21,1.11e-20/
      data w1no /0.      ,5.12e-2 ,1.36e-1 ,1.65e-1 ,1.41e-1 ,4.5e-2  /
      data s1no /0.      ,1.32e-18,6.35e-19,7.09e-19,2.18e-19,4.67e-19/
      data w2no /0.      ,5.68e-3 ,1.52e-2 ,1.83e-2 ,1.57e-2 ,5.0e-3  /
      data s2no /0.      ,4.41e-17,4.45e-17,4.5e-17 ,2.94e-17,4.35e-17/
!
! O2(9,0) BAND, NO D(1-0) BAND 183.1-184.6NM
!
      real,dimension(6) :: s1o2,w11no,s11no,w12no,s12no
      data s1o2  /1.35e-22,2.99e-22,7.33e-22,3.07e-21,1.69e-20,1.66e-19/
      data w11no /0.      ,0.      ,1.93e-03,9.73e-2 ,9.75e-2 ,3.48e-2 /
      data s11no /0.      ,0.      ,3.05e-21,5.76e-19,2.29e-18,2.21e-18/
      data w12no /0.      ,0.      ,2.14e-4 ,1.08e-2 ,1.08e-2 ,3.86e-3 /
      data s12no /0.,0.,3.2e-21,5.71e-17,9.09e-17,6.0e-17/
!
! O2(10,0) BAND, NO D(1-0) BAND 181.6-183.1NM
!
      real,dimension(6) :: s2o2,w21no,s21no,w22no,s22no
      data s2o2  /2.97e-22,5.83e-22,2.05e-21,8.19e-21,4.8e-20 ,2.66e-19/
      data w21no /4.5e-2  ,1.8e-1  ,2.25e-1 ,2.25e-1 ,1.8e-1  ,4.5e-2  /
      data s21no /1.8e-18 ,1.5e-18 ,5.01e-19,7.20e-20,6.72e-20,1.49e-21/
      data w22no /5.e-3   ,2.e-2   ,2.5e-2  ,2.5e-2  ,2.0e-2  ,5.0e-3  /
      data s22no /1.4e-16 ,1.52e-16,7.e-17  ,2.83e-17,2.73e-17,6.57e-18/
!
! O3 ABSORPTION CROSS SECTION FOR EACH BAND
!
      real :: so3a1,so3a2,so3a3
      data so3a1,so3a2,so3a3 /42.8e-20, 62.2e-20, 68.8e-20/
      real :: dlam,dlam1,dlam2
      data dlam,dlam1,dlam2 /2.3, 1.5, 1.5/
!
! Loop over zp and longitude:
!
      do i=lon0,lon1
        do k=lev0,lev1
          xnn2 = (1.-o2mmr(k,i)-o1mmr(k,i))*xnmbar(k,i)*rmassinv_n2
          prob = 1.65e+9 / (5.1e+7 + 1.65e+9 + 1.5e-9 * xnn2)
          tauo3 = expo(-so3a1*sco3(k,i))
          sum = 0.
          do il = 1,6
            sum = sum + 
     |       (expo(-so2(il) *sco2(k,i))*(w1no(il)*s1no(il)*
     |        expo(-s1no(il)*scno(k,i))+w2no(il)*s2no(il)*
     |        expo(-s2no(il)*scno(k,i))))
          enddo
          xjno1 = dlam*3.98e+11*tauo3*sum*prob
!
          tauo3 = expo(-so3a2*sco3(k,i))
          sum = 0.
          do il = 1,6
            sum = sum +
     |       (expo(-s1o2(il) *sco2(k,i))*(w11no(il)*s11no(il)*
     |        expo(-s11no(il)*scno(k,i))+w12no(il)*s12no(il)*
     |        expo(-s12no(il)*scno(k,i))))
          enddo
          xjno2 = dlam1*2.21e+11*tauo3*sum*prob
!
          tauo3 = expo(-so3a3*sco3(k,i))
          sum = 0.
          do il = 1,6
            sum = sum +
     |       (expo(-s2o2(il) *sco2(k,i))*(w21no(il)*s21no(il)*
     |        expo(-s21no(il)*scno(k,i))+w22no(il)*s22no(il)*
     |        expo(-s22no(il)*scno(k,i))))
          enddo
          xjno3 = dlam2*2.30e+11*tauo3*sum*prob
!
! Total NO dissociation:
!
          xjno(k,i) = xjno1+xjno2+xjno3
        enddo
      enddo
      end subroutine jno_exp
!-----------------------------------------------------------------------
      subroutine jo2h2osrb(xjo2srb,xjh2osrb,sco2,sco3,
     |  lev0,lev1,lon0,lon1,lat)
!
! Dissociation of O2 and H2O from Shumann-Runge bands.
!
      implicit none
!
! Args:
      integer,intent(in) :: lev0,lev1,lon0,lon1,lat
      real,intent(out),dimension(lev0:lev1,lon0:lon1) :: 
     |  xjo2srb,xjh2osrb
!
! 11/29/04 btf: fix dimension bug in sco2, sco3:
      real,intent(in),dimension(lev0:lev1,lon0-2:lon1+2) ::
     |  sco2,sco3
!
! Local:
      integer,parameter :: 
     |  nwt=6,  ! number of weights
     |  nwl=16  ! number of wavelength bins
      real :: sum,xjo2,xjh2o,trano3(nwl)
      integer :: i,k,il,jl
      real :: w(nwt) = (/0.05, 0.20, 0.25, 0.25, 0.20, 0.05/)
      real :: sg(nwt,nwl) = reshape(source=(/
     | 3.770E-21, 1.280E-20, 3.600E-20, 7.690E-20, 2.640E-19, 1.050E-18,
     | 1.033E-21, 1.750E-21, 4.588E-21, 1.714E-20, 1.012E-19, 1.097E-18,
     | 7.680E-22, 1.244E-21, 2.670E-21, 1.080E-20, 8.363E-20, 7.697E-19,
     | 1.127E-21, 2.015E-21, 4.660E-21, 1.650E-20, 9.295E-20, 5.017E-19,
     | 5.560E-22, 1.579E-21, 3.715E-21, 1.377E-20, 7.221E-20, 3.439E-19,
     | 2.968E-22, 5.831E-22, 2.053E-21, 8.192E-21, 4.802E-20, 2.655E-19,
     | 1.350E-22, 2.991E-22, 7.334E-22, 3.074E-21, 1.689E-20, 1.658E-19,
     | 1.079E-22, 2.092E-22, 5.875E-22, 2.590E-21, 1.578E-20, 1.028E-19,
     | 4.494E-23, 6.684E-23, 2.384E-22, 1.126E-21, 6.988E-21, 5.549E-20,
     | 1.910E-23, 3.438E-23, 1.168E-22, 4.736E-22, 2.653E-21, 2.529E-20,
     | 1.117E-23, 2.447E-23, 7.188E-23, 3.042E-22, 1.748E-21, 1.112E-20,
     | 9.600E-24, 1.264E-23, 2.553E-23, 1.049E-22, 5.190E-22, 2.824E-21,
     | 6.815E-24, 7.499E-24, 1.030E-23, 2.478E-23, 1.517E-22, 1.254E-21,
     | 6.977E-24, 7.493E-24, 9.308E-24, 1.653E-23, 7.869E-23, 4.630E-22,
     | 6.739E-24, 6.774E-24, 6.931E-24, 7.357E-24, 1.039E-23, 5.183E-23,
     | 6.842E-24, 6.850E-24, 6.878E-24, 7.113E-24, 8.407E-24, 2.869E-23
     | /),shape=(/nwt,nwl/))
      real :: si(nwl) = (/ 0.825E+10,
     |  1.334E+10, 1.535E+10, 1.531E+10, 1.835E+10, 2.301E+10, 
     |  2.208E+10, 2.200E+10, 2.879E+10, 3.370E+10, 3.980E+10, 
     |  4.176E+10, 5.823E+10, 6.270E+10, 6.554E+10, 7.928E+10/)
      real :: dl(nwl) = (/ 20.,
     |  8., 10., 11., 12., 15., 15., 17., 19., 20., 23., 22., 25., 13., 
     |  15.,25./)
      real :: sigo3(nwl) = (/ 81.1E-20,
     | 79.9E-20,78.6E-20,78.6E-20,76.3E-20,76.3E-20,68.8E-20,62.2E-20,
     | 57.6E-20,52.6E-20,47.6E-20,42.8E-20,38.3E-20,34.7E-20,32.3E-20,
     | 31.4E-20/)
      real :: sgh2o(nwt,nwl) = reshape(source=(/
     | 1.870E-18, 2.050E-18, 2.180E-18, 2.270E-18, 2.190E-18, 2.190E-18,
     | 1.600E-18, 1.610E-18, 1.670E-18, 1.690E-18, 1.720E-18, 1.720E-18,
     | 1.250E-18, 1.270E-18, 1.330E-18, 1.370E-18, 1.340E-18, 1.397E-18,
     | 8.450E-19, 8.520E-19, 9.150E-19, 9.780E-19, 9.690E-19, 1.050E-18,
     | 4.300E-19, 4.510E-19, 5.010E-19, 5.730E-19, 6.110E-19, 6.640E-19,
     | 1.890E-19, 2.050E-19, 2.550E-19, 2.900E-19, 3.020E-19, 3.400E-19,
     | 8.160E-20, 8.980E-20, 9.940E-20, 1.270E-19, 1.370E-19, 1.500E-19,
     | 4.720E-20, 4.980E-20, 5.360E-20, 6.070E-20, 6.430E-20, 6.760E-20,
     | 2.960E-20, 2.950E-20, 3.340E-20, 3.710E-20, 3.970E-20, 4.110E-20,
     | 1.190E-20, 1.270E-20, 1.430E-20, 1.890E-20, 2.180E-20, 2.300E-20,
     | 3.490E-21, 3.520E-21, 4.240E-21, 5.840E-21, 6.610E-21, 8.170E-21,
     | 1.600E-21, 1.810E-21, 2.040E-21, 2.380E-21, 2.780E-21, 2.940E-21,
     | 3.840E-22, 4.440E-22, 5.790E-22, 7.330E-22, 9.430E-22, 1.100E-21,
     | 1.840E-22, 1.920E-22, 1.990E-22, 2.280E-22, 2.430E-22, 2.620E-22,
     | 7.300E-23, 6.700E-23, 7.360E-23, 9.580E-23, 7.270E-23, 2.320E-23,
     | 0.,        0.,        0.,        0.,        0.,        0.
     | /),shape=(/nwt,nwl/))
!
! o2 photodissociation
!
      do i=lon0,lon1
        do k=lev0,lev1
          xjo2 = 0.
	  do il=1,16
            trano3(il) = exp(-sigo3(il)*sco3(k,i))
            sum = 0.
            do jl=1,6
              sum = sum+sg(jl,il)*w(jl)*exp(-sg(jl,il)*sco2(k,i))
            enddo
            xjo2 = xjo2+si(il)*dl(il)*sum*trano3(il)
          enddo
          xjo2srb(k,i) = xjo2
!
! h2o photodissociation
!
          xjh2o = 0.
	  do il=1,16
	    trano3(il) = exp(-sigo3(il)*sco3(k,i))
	    sum = 0.
	    do jl=1,6
	      sum = sum+sgh2o(jl,il)*w(jl)*exp(-sg(jl,il)*sco2(k,i))
            enddo
	    xjh2o = xjh2o+si(il)*dl(il)*sum*trano3(il)
          enddo
	  xjh2osrb(k,i) = xjh2o
        enddo ! k=lev0,lev1
      enddo   ! i=lon0,lon1
      end subroutine jo2h2osrb
!-----------------------------------------------------------------------
      end module qrj_module
