!
! Utility subprograms for tgcm:
!
!-------------------------------------------------------------------
      integer function nextlu()
      implicit none
!
! Return an unopened fortan logical unit number (not 5 or 6):
! (removed old conditional in which previously used lu's were not allowed)
!
      logical isopen
      integer lu
      do lu=7,99
        inquire(lu,opened=isopen)
        if (.not.isopen) then
          nextlu = lu
          return
        endif
      enddo
      write(6,"(/'>>> nextlu: all logical units apparently in use')")
      nextlu = 0
      call shutdown('nextlu')
      end function nextlu
!-------------------------------------------------------------------
      subroutine rmcomments(luin,luout,comcharin,echo)
      implicit none
!
! Read input lines from unit lu. If current line contains the comment
!   character comcharin, strip the line from position of comchar to end,
!   and write any remaining line to a new unit. If no comment in current 
!   line, write entire line to new unit. 
! Return new unit, rewound (e.g., ready to be read by namelist).
! If echo > 0, echo output lines to stdout.
! If comcharin is ' ', then default comment char is ';'
!
! Args:
      integer,intent(in) :: luin,luout,echo
      character(len=1),intent(in) :: comcharin
! Local:
      character(len=1) :: comchar
      logical isopen
      integer :: i,lens,ios,compos,nline,nlcfields
      character(len=6400) :: line
      character(len=64) :: newcfields(30)
!
      if (luin <= 0) then
        write(6,"('>>> rmcomments: bad input luin=',i5)") luin
        call shutdown('rmcomments')
      endif
      if (luout <= 0) then
        write(6,"('>>> rmcomments: bad input luout=',i5)") luout
        call shutdown('rmcomments')
      endif
      if (len_trim(comcharin) > 0) then
        comchar = comcharin
      else
        comchar = ';'
        write(6,"('rmcomments: using default semicolon as ',
     +    'comment character.')")
      endif
      inquire(unit=luin,opened=isopen)
      if (.not.isopen) then
        open(unit=luin,iostat=ios)
        if (ios /= 0) then
          write(6,"('>>> WARNING rmcomments: error opening input',
     +      ' file with unit luin=',i2,' ios=',i5)") luin,ios
          call shutdown('rmcomments')
        endif
      endif
      nline = 0
      read_loop: do
        line = ' '
        read(luin,"(a)",iostat=ios) line
        if (ios > 0) then
          write(6,"('>>> rmcomments: error reading from input',
     +      ' unit luin=',i3,' at line ',i5)") luin,nline
          return
        endif
        if (ios < 0) exit read_loop	! eof
        nline = nline+1
!
! Remove line if it has only "E" in column 1 (this was an
! old "Echo" directive from f77/cray namelist):
!
        if (line(1:1)=='E'.and.trim(line)=='E') cycle read_loop
!
! Use only non-commented part of line:
!
        compos = index(line,comchar)
        if (compos == 1) cycle read_loop
        if (compos > 0) line = line(1:compos-1)
        if (len_trim(adjustl(line))==0) cycle read_loop
!
! Write to new unit:
        write(luout,"(a)") trim(line)
        if (echo > 0) write(6,"(a)") line(1:len_trim(line))
      enddo read_loop  
!
! 8/3/04 btf: close luout (rather than just rewinding) to insure 
! that the file fort.7 is complete before barrier is passed and all 
! tasks do the namelist read.
!     rewind luout
      close(luout)
      return      
      end subroutine rmcomments
!-------------------------------------------------------------------
      subroutine mkdiskflnm(msspath,diskname)
      implicit none
!
! Given a mss path (msspath), construct and return a local 
! file name (diskname) which is the same as msspath except
! that slashes ('/') in the msspath are replaced by dots ('.')
! in the diskname. (The initial slash in msspath is ignored)
!
! For example, if on input msspath = '/FOSTER/dir1/file', then
! diskname would be returned as 'FOSTER.dir1.file'
!
! Args:
      character(len=*),intent(in) :: msspath
      character(len=*),intent(out) :: diskname
!
! Local:
      integer :: i,ii
!
      if (len_trim(msspath)==0) then
        write(6,"('WARNING mkdiskflnm: zero length msspath.',
     +    ' Returning diskname: ''disk.file''')") 
        diskname = 'disk.file'
        return
      endif
      diskname = ' '
      do i=1,len(diskname)
!
! Do not replace leading slash of msspath. 
! If there are no occurrences of '/' in the msspath (i.e., its
!   not really an mss path, but just a file name), return the 
!   msspath unchanged.
!
        ii = i+1 
        if (index(msspath,'/')==0) ii = i
        if (ii > len_trim(msspath)) exit
        if (msspath(ii:ii)=='/') then
          diskname(i:i)='.'
        else
          diskname(i:i) = msspath(ii:ii)
        endif
      enddo
      if (len_trim(diskname)==0) then
        write(6,"('WARNING mkdiskflnm: zero length diskname output.')")
        write(6,"('  msspath=',a)") trim(msspath)
        write(6,"('Returning diskname: ''disk.file''')")
        diskname = 'disk.file'
      endif
!     write(6,"('mkdiskflnm returning:',/,'  msspath=',a,
!    +  /,'  diskname=',a)") trim(msspath),trim(diskname)
      end subroutine mkdiskflnm
!-------------------------------------------------------------------
      subroutine fminmax(f,n,fmin,fmax)
!
! Return min and max of fields f(n) (between -1.e36,+1.e36)
!
      implicit none
!
! Args: 
      integer,intent(in) :: n
      real,intent(in) :: f(n)
      real,intent(out) :: fmin,fmax
!
! Local:
      integer :: i
!
      fmin = 1.e36
      fmax = -1.e36
      do i=1,n
        fmin = min(f(i),fmin)
        fmax = max(f(i),fmax)
      enddo
      end subroutine fminmax
!-------------------------------------------------------------------
      subroutine fminmaxspv(f,n,fmin,fmax,spv)
      implicit none
!
! Return min and max of fields f(n) (between -1.e36,+1.e36)
! Ignore any f(i)==spv.
!
! Args:
      integer,intent(in) :: n
      real,intent(in) :: f(n),spv
      real,intent(out) :: fmin,fmax
!
! Local:
      integer :: i
!
      fmin = 1.e36
      fmax = -1.e36
      do i=1,n
        if (f(i) /= spv) then
          fmin = min(f(i),fmin)
          fmax = max(f(i),fmax)
        endif
      enddo
      end subroutine fminmaxspv
!-------------------------------------------------------------------
      integer function ixfind(iarray,idim,itarget,icount)
!
! Search iarray(idim) for itarget, returning first index in iarray 
! where iarray(idim)==target. Also return number of elements of
! iarray that == itarget in icount.
!

!
! Args:
      integer,intent(in) :: idim,itarget
      integer,intent(in) :: iarray(idim)
      integer,intent(out) :: icount
!
! Local:
      integer :: i
!
      ixfind = 0
      icount = 0
      if (.not.any(iarray==itarget)) return
      icount = count(iarray==itarget)
      do i=1,idim
        if (iarray(i)==itarget) then
          ixfind = i
          exit
        endif
      enddo
      end function ixfind
!-------------------------------------------------------------------
      real function finterp(f0,f1,isec0,isec1,isec)
!
! Do linear interpolation between f0 (which is at isec0) and 
! f1 (which is at isec1) to isec.
!
! Args:
      real,intent(in) :: f0,f1
      integer,intent(in) :: isec0,isec1,isec
!
      finterp = f0+(f1-f0)*float(isec-isec0)/float(isec1-isec0)
      end function finterp
!-------------------------------------------------------------------
      real function finterp_bigints(f0,f1,isec0,isec1,isec)
!
! Do linear interpolation between f0 (which is at isec0) and 
! f1 (which is at isec1) to isec. Same as finterp except integer
! parameters are 8-byte.
!
! Args:
      real,intent(in) :: f0,f1
      integer(kind=8),intent(in) :: isec0,isec1,isec
!
      finterp_bigints = 
     |  f0+(f1-f0)*float(isec-isec0)/float(isec1-isec0)
      end function finterp_bigints
!-------------------------------------------------------------------
      real function sddot(n,x,y)
      implicit none
!
! Call sdot (single precision) if on Cray, or ddot (double precision) 
!   if on SGI. (ddot must be called even if -r8 on sgi compiler command 
!   line). Ddot is from -lblas on the sgi.
! On IBM AIX use dot_product()
!
! 2/10/00: removing incx,incy args (i.e., incx=incy=1 on unicos
!   and irix -- IBM dot_product does not use increment args --
!   this function must be called with stride-1 vectors 
!   (see bndry.f, bndry2.f, bndrya.f, threed.f, transf.f)
!
      integer,intent(in) :: n
      real,intent(in) :: x(n),y(n)
!
#ifdef UNICOS
      real,external :: sdot
      sddot = sdot(n,x,1,y,1)
#elif SUN
      real,external :: sdot
      sddot = dot_product(x,y)
#elif IRIX
      real,external :: ddot
      sddot = ddot(n,x,1,y,1)
#elif AIX
      sddot = dot_product(x,y)
#elif OSF1
      sddot = dot_product(x,y)
#elif LINUX
      sddot = dot_product(x,y)
#else
      write(6,"('>>> WARNING sddot: unresolved OS pre-processor',
     |  ' directive.')")
#endif
      end function sddot
!-------------------------------------------------------------------
      real function vsum(n,v,inc)
!
! Call single precision vector sum "ssum" on Cray/unicos, or
!   double precision "dsum" on SGI/irix (from -lblas on the sgi):
! On IBM AIX use sum
!
      integer,intent(in) :: n,inc
      real,intent(in) :: v(n)
!
#ifdef UNICOS
      vsum = ssum(n,v,inc)
#elif IRIX
      vsum = dsum(n,v,inc)
#elif AIX
      vsum = sum(v,n)
#elif SUN
      vsum = sum(v,n)
#elif OSF1
      vsum = sum(v,n) ! a wild guess
#elif LINUX
      vsum = sum(v,n) ! another guess
#else
      write(6,"('>>> WARNING sddot: OS system cpp directive',
     |  ' not found.')")
#endif
      end function vsum
!-------------------------------------------------------------------
      complex function sdcmplx(x1,x2)
!
! Call single precision cmplx on unicos, or
! double precision dcmplx on irix:
!
      real,intent(in) :: x1,x2
!
#if defined(IRIX) || defined(AIX)
      sdcmplx = dcmplx(x1,x2)
#elif UNICOS
      sdcmplx = cmplx(x1,x2)
#elif SUN
      sdcmplx = cmplx(x1,x2)
#elif OSF1
      sdcmplx = cmplx(x1,x2)
#elif LINUX
      sdcmplx = cmplx(x1,x2)
#else
      write(6,"('>>> WARNING sddot: OS system cpp directive',
     |  ' not found.')")
#endif
      end function sdcmplx
!-------------------------------------------------------------------
      real function expo(x)
!
! To avoid overflow/underflow, check argument range to exp() intrinsic:
!
!   Under AIX (xlf90):
!     exp arg range = -708.3964 < x  < 709.7827
!     huge = 0.179769313486231571E+309 as reported by huge() intrinsic
!     tiny = 0.222507385850720138E-307 as reported by tiny() intrinsic
!                            
!   Under Linux-mpi (pgf90):
!     exp arg range = -745.1332 < x < 709.7827
!     huge = 1.7976931348623167E+308 as reported by huge() intrinsic
!     tiny = 2.2250738585072010E-308 as reported by tiny() intrinsic
!
      real,intent(in) :: x 
      real,parameter :: xmin=-708., xmax=+709., 
     |  big=.1e305, small=.1e-305
      integer :: iprint=0
!     integer :: iprint=1
!
#if defined(IRIX) || defined(AIX) || defined(OSF1) || defined(SUN) || defined(LINUX)
      if (x >= xmin .and. x <= xmax) then
        expo = exp(x)
      elseif (x < xmin) then
        if (iprint > 0) write(6,"('expo iprint=',i2,' x=',e12.4,
     |    ' setting expo = 0.')") iprint,x
        expo = 0.
      else
        if (iprint > 0) write(6,"('expo iprint=',i2,' x=',e12.4,
     |    ' setting expo = big')") iprint,x
        expo = big
      endif
#elif UNICOS
      expo = exp(x)
#endif
      end function expo
!-------------------------------------------------------------------
      integer function isystem(command)
      implicit none
!
! Execute command to the shell. UNICOS and IRIX use ishell(),
! AIX uses system().
!
      character(len=*),intent(in) :: command
#if defined(UNICOS) || defined(IRIX)
      integer,external :: ishell
      isystem = ishell(trim(command))
#elif defined (AIX)
      integer,external :: system
      isystem = system(trim(command)//"\0")
#elif defined(OSF1)
      integer,external :: system
      isystem = system(trim(command))
#elif defined(SUN)
      integer,external :: system
      isystem = system(trim(command))
#elif defined(LINUX)
      integer,external :: system
      isystem = system(trim(command))
#endif
      end function isystem
!-------------------------------------------------------------------
      real function cputime()
!
! Return user cpu time:
!
      implicit none
!
! Cray unicos and SGI IRIX use the second() function:
!
#if defined(UNICOS) || defined(IRIX)
      real(kind=4),external :: second
      cputime = second()
#elif AIX || LINUX
!
! IBM AIX uses sub cpu_time
! To use other than default time type use setrteopts, e.g.:
!     call setrteopts('cpu_time_type=usertime')
! (see p.589 of AIX Language Reference)
!
      real :: time
      call cpu_time(time)
      cputime = time
#elif OSF1
      cputime = 0. ! don't know of one on OSF, except in ncaru
#elif SUN
      cputime = 0. ! don't know of one on OSF, except in ncaru
#endif
      end function cputime 
!-------------------------------------------------------------------
      integer function isymlink(from,to,iprint)
      implicit none
!
! Args:
      character(len=*),intent(in) :: from,to
      integer,intent(in) :: iprint
!
! Local:
      logical :: exists
      integer :: istat
      integer,parameter :: maxlen=1024
      character(len=maxlen) :: command
!
! External:
      integer,external :: isystem
!
! Check for existence of "from" file:
      inquire(file=trim(from),exist=exists)      
!     write(6,"('isymlink: Linking from=',a,' to=',a,' exists=',l1)") 
!    |  trim(from),trim(to),exists
!
! Make symbolic link:
      write(command,"('ln -s -f ',a,' ',a)") trim(from),trim(to)
      isymlink = isystem(command)
      if (isymlink == 0) then
        if (iprint > 0) 
     |    write(6,"('Made symbolic link of ',a,' to ',a)") 
     |      trim(from),trim(to)
      else
        write(6,"('>>> WARNING: symbolic link of ',a,' to ',a,
     |    ' failed: ierror=',i3,/)") trim(from),trim(to),isymlink
!       call shutdown('isymlink')
      endif
      end function isymlink
!-------------------------------------------------------------------
      integer function ilink(from,to,iprint)
!
! Make a link from file "from" to target "to"
!
! A regular link (link()) will fail if "from" and "to" are on different
!   file systems (e.g., /ptmp and /hao or /fis). If a regular link fails, 
!   try making symbolic link using function isymlink, also in this util.F.
!
      implicit none
!
! Args:
      character(len=*),intent(in) :: from,to
      integer,intent(in) :: iprint
!
! External:
      integer,external :: 
     |  link,     ! intrinsic?
     |  isymlink, ! in this file
     |  iunlink   ! in this file
!
! Local:
      logical :: exists
!
      inquire(file=trim(from),exist=exists)      
!     write(6,"('ilink: Linking from=',a,' to=',a,' exists=',l1)") 
!    |  trim(from),trim(to),exists
!
#if defined(UNICOS) || defined(IRIX) || defined(SUN) || defined(OSF1)
      ilink = link(trim(from),trim(to))
      if (ilink.eq.0) then
        if (iprint > 0) 
     |  write(6,"('Linked file ',a,' to ',a)") trim(from),trim(to)
      else
!       write(6,"(/,'>>> WARNING: link command of ',a,' to ',a,
!    |    ' failed.')") trim(from),trim(to)
      endif
#elif defined(AIX)  || defined(LINUX)
      ilink = link(trim(from)//"\0",trim(to)//"\0")
      if (ilink.eq.0) then
        if (iprint > 0) 
     |    write(6,"('Linked file ',a,' to ',a)") trim(from),trim(to)
      else
!       write(6,"('>>> WARNING: link command of ',a,' to ',a,
!    |    ' failed.')") trim(from),trim(to)
      endif
#endif
!
! Try symbolic link if regular link failed:
      if (ilink /= 0) then
        ilink = isymlink(from,to,0)
        if (ilink /= 0) then
          write(6,"(/,'>>> WARNING: Regular and symbolic links of ',
     |      a,' to ',a,' failed.')") trim(from),trim(to)
        endif
      endif
      end function ilink
!-------------------------------------------------------------------
      integer function iunlink(file,iprint)
      implicit none
      character(len=*),intent(in) :: file
      integer,intent(in) :: iprint
      integer,external :: unlink
!
#if defined(UNICOS) || defined(IRIX) || defined(SUN) || defined(OSF1)
      iunlink = unlink(trim(file))
      if (iunlink.eq.0) then
        if (iprint > 0) 
     |    write(6,"('Unlinked file ',a)") trim(file)
      else
        if (iprint > 0) 
     |    write(6,"('Note: unlink of ',a,' failed',
     |      ' (possibly non-existant file).')") trim(file)
      endif
#elif AIX || LINUX
      iunlink = unlink(trim(file)//"\0")
      if (iunlink.eq.0) then
        if (iprint > 0) 
     |    write(6,"('Unlinked file ',a)") trim(file)
      else
        if (iprint > 0) 
     |    write(6,"('Note: unlink of ',a,' failed: iunlink=',i3)")
     |      trim(file),iunlink
      endif
#endif
      end function iunlink
!-----------------------------------------------------------------------
      real function mtime_delta(mtime0,mtime1)
!
! Return difference in time (minutes) from mtime0 to mtime1:
!
      implicit none
      integer,intent(in) :: mtime0(4),mtime1(4)
      real :: rmins0,rmins1
!
      rmins0 = mtime0(1)*1440.+mtime0(2)*60.+mtime0(3)+mtime0(4)/60.
      rmins1 = mtime1(1)*1440.+mtime1(2)*60.+mtime1(3)+mtime1(4)/60.
      mtime_delta = rmins1-rmins0
!     if (mtime_delta < 0.) then
!       write(6,"('>>> WARNING mtime_delta: negative delta: mtime0=',
!    |    4i4,' mtime1=',4i4,' delta=',f10.2)") 
!    |    mtime0,mtime1,mtime_delta
!     endif
      end function mtime_delta
!-----------------------------------------------------------------------
      real function mtime_to_datestr(iyear,mtime,imo,ida,datestr)
      implicit none
!
! Given model time mtime (day,hr,min,sec) and iyear (4-digit year),
! return imo (2 digit month), ida (2 digit day of the month), and
! a date string "minutes since yyyy-m-d". Function value is real
! minutes into the day from modeltime(2:4) hour, min, and sec.
!
! Args:
      integer,intent(in) :: iyear,mtime(4)
      integer,intent(out) :: imo,ida
      character(len=*),intent(out) :: datestr
!
! Local:
      character(len=2) :: f_mon,f_day,f_hr,f_min
      character(len=120) :: format
      integer :: ndmon(12) =
!          J  F  M  A  M  J  J  A  S  O  N  D
     |  (/31,28,31,30,31,30,31,31,30,31,30,31/)
      integer :: m,id,iday
!
      mtime_to_datestr = -1.
      datestr = ' '
      ndmon(2) = 28
      if (mod(iyear,4).eq.0) ndmon(2) = 29
!
      iday = mtime(1)
      id = 0
      do m=1,12
        id = id+ndmon(m)
        if (id.eq.iday) then
          id = ndmon(m)
          goto 100
        endif
        if (id.gt.iday) then
          id = ndmon(m)-id+iday
          goto 100
        endif
      enddo
      write(6,"('>>> mtime_to_datestr: could not find date for mtime=',
     |  4i4,' iyear=',i4)") mtime,iyear
      imo = 0
      ida = 0
      return
 100  continue
      imo = m
      ida = id
!
! Return real minutes:
      mtime_to_datestr = float(mtime(2))*60.+float(mtime(3))+
     |                   float(mtime(4))/60.
!
! Return date string:
      if (imo <= 9) then
        if (ida <= 9) then
          f_day = 'i1'
          f_mon = 'i1'
        else ! ida > 9
          f_day = 'i2'
          f_mon = 'i1'
        endif
      else ! imo > 9
        if (ida <= 9) then
          f_day = 'i1'
          f_mon = 'i2'
        else ! ida > 9
          f_day = 'i2'
          f_mon = 'i2'
        endif
      endif
      if (mtime(2) <= 9) then
        if (mtime(3) <= 9) then
          f_hr = 'i1'
          f_min = 'i1'
        else
          f_hr = 'i1'
          f_min = 'i2'
        endif
      else
        if (mtime(3) <= 9) then
          f_hr = 'i2'
          f_min = 'i1'
        else
          f_hr = 'i2'
          f_min = 'i2'
        endif
      endif
!
! Example format:
!   ('minutes since ',i4,'-',i1,'-',i2,' ',i1,':',i2,':0')
! Example date string:
!   minutes since 1997-3-21 0:0:0
!
      format = '(''minutes since '',i4,''-'','//f_mon//',''-'','//f_day
      format = trim(format)//','' '','//f_hr//','':'','//f_min//','
      format = trim(format)//''':0'')'
      write(datestr,format) iyear,imo,ida,mtime(2),mtime(3)
      end function mtime_to_datestr
!-----------------------------------------------------------------------
      integer function mtime_to_mins(mtime)
      implicit none
!
! Given model time mtime (day,hr,min), return equivalent time 
! in minutes (includes day):
!
! Arg:
      integer,intent(in)  :: mtime(3)
!
      mtime_to_mins = mtime(1)*24*60+mtime(2)*60+mtime(3)
      end function mtime_to_mins
!-----------------------------------------------------------------------
      integer function mtime_to_nstep(mtime,istepsecs)
      implicit none
!
! Return number of steps from time 0,0,0 to given model time 
! mtime(day,hr,min) (istepsecs is time step in seconds)
! (This function used to be itera in earlier versions of tgcm)
!
! Args:
      integer,intent(in) :: mtime(3),istepsecs
!
      mtime_to_nstep=((mtime(1)*24+mtime(2))*60+mtime(3))*60/istepsecs
      end function mtime_to_nstep
!-----------------------------------------------------------------------
      integer(kind=8) function mtime_to_nsec(mtime)
      implicit none
      integer(kind=8),parameter :: nsecperday=24*60*60 ! 86400 seconds/day
      integer,intent(in) :: mtime(3)
      mtime_to_nsec = mtime(1)*nsecperday + mtime(2)*3600 + mtime(3)*60
      end function mtime_to_nsec
!-----------------------------------------------------------------------
      subroutine mins_to_mtime(mins,mtime)
      implicit none
!
! Given minutes mins, return equivalent model time (day,hr,min)
!
! Args:
      integer,intent(in) :: mins
      integer,intent(out) :: mtime(3)
!
! Local:
      integer,parameter :: minperday=24*60
!
      mtime(:) = 0
      if (mins==0) return
      mtime(1) = mins/minperday                        ! days
      mtime(2) = mod(mins,minperday)/60                ! hours
      mtime(3) = mins-(mtime(1)*minperday+mtime(2)*60) ! minutes
      end subroutine mins_to_mtime
!-------------------------------------------------------------------
      subroutine nsecs_to_modeltime(nsecs,modeltime)
      implicit none
!
! Given seconds nsecs, return equivalent model time (day,hr,min,sec)
!
! Args:
      integer(kind=8),intent(in) :: nsecs
      integer,intent(out) :: modeltime(4)
!
! Local:
      integer(kind=8),parameter :: nsecperday=24*60*60 ! 86400 seconds/day
!
      modeltime(:) = 0
      if (nsecs==0) return
      modeltime(1) = nsecs/nsecperday            ! days
      modeltime(2) = mod(nsecs,nsecperday)/3600  ! hours
      modeltime(3) = (nsecs-(modeltime(1)*nsecperday+modeltime(2)*3600))
     |               /60                         ! mins
      modeltime(4) = nsecs-(modeltime(1)*nsecperday+modeltime(2)*3600+
     |  modeltime(3)*60)                         ! seconds
      end subroutine nsecs_to_modeltime
!-------------------------------------------------------------------
      subroutine modeltime_to_nsecs(modeltime,nsecs)
!
! Given modeltime(4) (day,hr,min,sec), return total seconds
! in nsecs.
!
! Args:
      integer,intent(in)  :: modeltime(4)  
      integer(kind=8),intent(out) :: nsecs
! 
! Local:
      integer(kind=8),parameter :: nsecperday=24*60*60 ! 86400 seconds/day
!
      nsecs = modeltime(1)*nsecperday + modeltime(2)*3600 +
     |        modeltime(3)*60 + modeltime(4)
      end subroutine modeltime_to_nsecs
!-------------------------------------------------------------------
      subroutine setosys(system)
      implicit none
      character(len=*),intent(out) :: system
! 
      system = ' ' 
#ifdef UNICOS
      system = 'UNICOS'
#elif IRIX
      system = 'IRIX'
#elif AIX
      system = 'AIX'
#elif OSF1
      system = 'OSF1'
#elif SUN
      system = 'SUN'
#elif LINUX
      system = 'LINUX'
#else
      write(6,"('>>> WARNING setosys: unresolved OS cpp directive.')")
      system = 'unknown'
#endif
      end subroutine setosys
!-------------------------------------------------------------------
      subroutine settmpdir(logname,model_version,tmpdir)
!
! If tempdir is not provided by the user via input, set it here
!   according to platform. These defaults may need to be changed as
!   directory systems on individual machines change.
! If the user does not want to set it, and this routine is not
!   suitable, the user may set tempdir='.' in input. This will
!   work on all systems, but in this case history files will not be 
!   linked to tempdir, and could be lost if not saved to mss.
!   (e.g., if cwd is $TMPDIR, or some such scratch directory)
! Note however, that if the cwd and tmpdir are in different file
!   systems, the link (e.g., from savefile) will fail.
!
      implicit none
      character(len=*),intent(in) :: logname,model_version
      character(len=*),intent(out) :: tmpdir
!
      tmpdir = ' '
#ifdef UNICOS
      tmpdir = '/tmp/'//trim(logname)//'/'//trim(model_version)
#elif IRIX
      tmpdir = '/ptmp/'//trim(logname)//'/'//trim(model_version)
#elif AIX
      tmpdir = '/ptmp/'//trim(logname)//'/'//trim(model_version)
      tmpdir = trim(tmpdir)//'/tmpdir'
#elif OSF1
      tmpdir = '/ptmp/'//trim(logname)//'/'//trim(model_version)
#elif SUN
      tmpdir = '/e/foster/timegcm1/tmpdir'
#elif LINUX
      tmpdir = './tmpdir'
#else
      write(6,"('>>> WARNING settmpdir: unresolved OS cpp directive.')")
      write(6,"('            Will set to cwd (tempdir = ''.'')')")
      tmpdir = '.'
#endif
!     write(6,"('settmpdir: set tempdir = ',a)") trim(tmpdir)
      end subroutine settmpdir
!-------------------------------------------------------------------
      subroutine datetime(curdate,curtime)
!
! Return character*8 values for current date and time.
! (sub date_and_time is an f90 intrinsic)
!
      implicit none
!
! Args:
      character(len=*),intent(out) :: curdate,curtime
!
! Local:
      character(len=8) :: date
      character(len=10) :: time
      character(len=5) :: zone
      integer :: values(8)
!
      curdate = ' '
      curtime = ' '
      call date_and_time(date,time,zone,values)
!
!     write(6,"('datetime: date=',a,' time=',a,' zone=',a)")
!    |  date,time,zone
!     write(6,"('datetime: values=',8i8)") values
!
      curdate(1:2) = date(5:6)
      curdate(3:3) = '/'
      curdate(4:5) = date(7:8)
      curdate(6:6) = '/'
      curdate(7:8) = date(3:4)
!
      curtime(1:2) = time(1:2)
      curtime(3:3) = ':'
      curtime(4:5) = time(3:4)
      curtime(6:6) = ':'
      curtime(7:8) = time(5:6)
!
      end subroutine datetime
!-------------------------------------------------------------------
      subroutine gethostsname(host)
!
! Return a host name, by various methods depending on platform:
!
      implicit none
!
! Args:
      character(len=*),intent(out) :: host
!
! Local:
      integer :: istat
!
! External:
#if defined(UNICOS)
      integer,external :: gethost
#endif
#if defined(AIX)
      integer,external :: gethostname
#endif
!
      host = ' '
#ifdef UNICOS
      istat = gethost(host)
      if (istat <= 0) then
        write(6,"('>>> gethostname UNICOS: error ',i4,' from gethost')") 
     |    istat
        host = 'unknown'
      endif
#elif defined(AIX)
      istat = gethostname(host) ! this yields "bb0001en" on babyblue
      if (istat /= 0) then
        write(6,"('>>> gethostname AIX: error ',i4,
     |    ' from gethostname')") istat
        host = 'unknown'
      endif
#elif defined(OSF1)
!
! There is a gethostname function on prospect (compaq), but apparently 
! there is no fortran binding.
      write(6,"('>>> gethostsname: do not know how to get host name ',
     |  'under OSF1')")
      host = 'unknown'
#elif defined(IRIX)
!
! Under IRIX, the interactive environment (e.g. utefe) defines $HOST, 
! whereas the batch environment (e.g. ute) defines $QSUB_HOST.
! 
      call getenv('HOST',host)                             ! interactive
      if (len_trim(host)==0) call getenv('QSUB_HOST',host) ! batch
      if (len_trim(host)==0) then
        write(6,"(/,'>>> gethost under IRIX: Cannot get HOST or ',
     |    'QSUB_HOST environment variables.',/)")
        host = 'unknown'
      endif
#elif defined(SUN) || defined(LINUX)
      call getenv('HOST',host)
      if (len_trim(host)==0) then
        write(6,"(/,'>>> gethost: Cannot get HOST environment ',
     |    'variable.',/)")
        host = 'unknown'
      endif
#else
      write(6,"('>>> WARNING gethost: unresolved OS cpp directive.')")
      host = 'unknown'
#endif
      end subroutine gethostsname
!-----------------------------------------------------------------------
      logical function isadigit(s)
      implicit none
      character(len=1),intent(in) :: s
      character(len=1) :: digits(10)=
     +  (/'0','1','2','3','4','5','6','7','8','9'/)
      integer :: i
      isadigit = .false.
      do i=1,10
        if (s==digits(i)) then
          isadigit = .true.
          exit
        endif
      enddo
      end function isadigit
!-----------------------------------------------------------------------
      integer function decode_int(str,sout)
      implicit none
!
! Return integer decoded from digits in string str.
! Only use digits from end of str. Also return string
! sout, with integers replaced by dollar signs.
!
! Args:
      character(len=*),intent(in) :: str
      character(len=*),intent(out) :: sout
!
! Local:
      character(len=len(str)) :: snum,s
      character(len=8) :: format
      integer :: i,ipos,slen
!
! External:
      logical,external :: isadigit
!
      decode_int = -1
      s = adjustl(str)
      slen = len_trim(s)
      if (slen>9999) then
        write(6,"('>>> decode_int: string too long:',
     +    ' slen=',i10)") slen
        return
      endif
      ipos = 0
      snum = ' '
      sout = trim(s)
      do i=slen,1,-1
        if (.not.isadigit(s(i:i))) exit
        ipos = ipos+1
        snum(slen-ipos+1:slen-ipos+1) = s(i:i)
        sout(i:i) = '$'
      enddo
      if (ipos==0) return
      snum = adjustl(snum)
      write(format,"('(i',i4,')')") len_trim(snum)
!     read(trim(snum),format,err=100) decode_int ! IBM does not like this
      read(snum,format,err=100) decode_int
      return
 100  write(6,"('>>> decode_int: error making integer from ',
     +  'string: ',a)") trim(snum)
      decode_int = -1
      end function decode_int
!-----------------------------------------------------------------------
      subroutine encode_str(str,num)
      implicit none
!
! Given str containing n characters '$', return
! same string, but with the '$' replaced by num.
!
      character(len=*),intent(inout) :: str
      character(len=len(str)) :: newstr
      integer,intent(in) :: num
      integer :: i,ipos,ndollar
      character(len=16) :: format
!
      ndollar = 0
      do i=1,len(str)
        if (str(i:i)=='$') ndollar = ndollar+1
      enddo
      if (ndollar==0) return
      newstr = str
      write(format,"('(i',i4,'.',i4,')')") ndollar,ndollar
      format = trim(adjustl(format))
      ipos = index(str,'$')
      write(newstr(ipos:ipos+ndollar-1),format,err=100) num
      str = newstr
      return
 100  write(6,"('>>> encode_str: error writing integer ',
     +  'num=',i12,' to string.')") num
      return
      end subroutine encode_str
!-----------------------------------------------------------------------
      integer function strloc(strarray,nstr,str)
      implicit none
!
! Search for string str in string array strarray(nstr).
!
! Args:
      integer,intent(in) :: nstr
      character(len=*),intent(in) :: strarray(nstr)
      character(len=*) :: str
!
! Local:
      integer :: i
!
      strloc = 0
      aloop: do i=1,nstr
        if (len_trim(strarray(i)) > 0) then
          if (trim(str) == trim(strarray(i))) then
            strloc = i
            exit aloop
          endif
        endif
      enddo aloop
      end function strloc
!-----------------------------------------------------------------------
      subroutine packstr(strings,nstrings,nonblank)
      implicit none
!
! Collect non-blank elements of strings(nstrings) at beginning
!   of the array. Return number of non-blank elements in nonblank. 
! On output, elements strings(1->nonblank) are non-blank (unless
!   nonblank==0), and remaining elements are blank.
!
! Args
      integer,intent(in) :: nstrings
      character(len=*),intent(inout) :: strings(nstrings)
      integer,intent(out) :: nonblank
!
! local:
      character(len=len(strings(1))) :: strings_tmp(nstrings)
      integer :: i
!
      strings_tmp(:) = ' '
      nonblank = 0
      do i=1,nstrings
        if (len_trim(strings(i)) > 0) then 
          nonblank = nonblank+1
          strings_tmp(nonblank) = strings(i)    
        endif
      enddo
      strings(:) = strings_tmp(:)
      end subroutine packstr
!-----------------------------------------------------------------------
      subroutine fftrans(a,na,work,nw,trigs,ntrigs,ifax,inc,jump,n,lot,
     |  isign)
!
! Do fft. If unicos, the original library version, FFT991  is used.
! If non-unicos, call FFT999, which is in fft9.f (the cray lib source,
!   with FFT991 changed to FFT999).
! See also setfft below.
! This is called from filter.F, e.g.:
!   call fftrans(fx,wfft,trigs,ifax,1,nlonp4,nlon,nlevs,-1)
!
      implicit none
!
! Args:
      integer,intent(in) :: inc,jump,n,lot,isign,ntrigs,na,nw
      real,intent(inout) :: a(na)
      real,intent(in) :: work(nw),trigs(ntrigs)
      integer,intent(in) :: ifax(13) 
!
#ifdef UNICOS
      call fft991(a,work,trigs,ifax,inc,jump,n,lot,isign)
#else
      call fft999(a,na,work,nw,trigs,ntrigs,ifax,inc,jump,n,lot,isign)
#endif
      end subroutine fftrans
!-----------------------------------------------------------------------
      subroutine setfft(trigs,ifax,ntrigs,imax)
!
! Set up fft (called once per run from con.f). 
! If unicos, the original library version, SET99  is used.
! If non-unicos, call SET999, which is in fft9.f (the cray lib source,
!   with SET99 changed to SET999.
!
      implicit none
!
! Args:
      integer,intent(in) :: ifax(13),ntrigs,imax
      real,intent(in) :: trigs(ntrigs)
!
#ifdef UNICOS
      call set99(trigs,ifax,imax)
#else
      call set999(trigs,ifax,ntrigs,imax)
#endif
      end subroutine setfft
!-----------------------------------------------------------------------
      subroutine rcpfile(source,target)
!
! Do remote copy of source to target.
!
      implicit none
!
! Args:
      character(len=*),intent(in) :: source,target
!
! Local:
      integer,parameter :: maxlen=1024
      character(len=maxlen) :: command
      integer :: len,ier
!
! External:
      integer,external :: isystem
!
      len = len_trim(source)+len_trim(target)+5
      if (len > maxlen) then
        write(6,"(/,'>>> WARNING rcpfile: len(source)+len(target)+5=',
     |    i6,' is > maxlen=',i6)") len,maxlen
        write(6,"('Please increase maxlen in sub rcpfile to at least ',
     |    i6,' -- no rcp done.')") len
        return
      endif
      write(command,"('rcp ',a,' ',a)") trim(source),trim(target)
      ier = isystem(command)
      if (ier==0) then
        write(6,"('rcpfile: successful rcp of ',a,' to ',a)")
     |    trim(source),trim(target)
      else
        write(6,"('>>> rcpfile: Error from rcp of ',a,' to ',a)")
     |    trim(source),trim(target)
      endif
      end subroutine rcpfile
!-----------------------------------------------------------------------
      subroutine getcwd(cwd)
#ifdef MPI
      use mpi_module,only: mytid
#endif
!
! Return current working directory in cwd:
!
      implicit none
      character(len=*),intent(out) :: cwd
      integer,external :: isystem,nextlu
      integer :: istat,lu,ier
#ifdef MPI
#include <mpif.h>
#endif
!
#ifdef MPI
      if (mytid==0) then
        istat = isystem('pwd > cwd')
        if (istat /= 0) write(6,"('>>> WARNING getcwd: error return ',
     |    'isystem(''pwd > cwd'')')")
      endif
      call mpi_barrier(MPI_COMM_WORLD,ier)
      lu = nextlu()
      open(lu,file='cwd',status='old')
      read(lu,"(a)") cwd
      close(lu) 
#else
      istat = isystem('pwd > cwd')
      if (istat /= 0) write(6,"('>>> WARNING getcwd: error return ',
     |  'isystem(''pwd > cwd'')')")
      lu = nextlu()
      open(lu,file='cwd',status='old')
      read(lu,"(a)") cwd
      close(lu) 
#endif
!
      end subroutine getcwd
!-----------------------------------------------------------------------
      subroutine isec2hms(isec,ih,im,is)
      implicit none
      integer,intent(in) :: isec
      integer,intent(out) :: ih,im,is
      integer :: nsec
!
! Given integer seconds isec (not including days), return
! integer hour, minute, seconds in ih,im,is:
!
      ih = isec/3600
      nsec = mod(isec,3600)
      im = nsec/60
      is = mod(nsec,60)
      return
      end
!-----------------------------------------------------------------------
      subroutine check_nans(f,id1,id2,id3,name,n_total,ispval,spval,
     |  iprint,ifatal)
!
! Check for existence of +/-INF and NaN's in field f(id1,id2,id3).
!   If ispval > 0 -> replace any INF or NaNs with spval
!   If iprint==1  -> print warnings only if INF or NaNs are found
!   If iprint==2  -> always print number of INF and NaNs found
!   If ifatal > 0 -> stop program when first INF or NaNs are found
! Note: Can distinguish between +/-INF (not really NaNs), but cannot 
!       distinguish between types of actual NaNs (+/-NaNQ and NaNS).
! IBM only. See pp318-319 User's Guide Version 8 XL Fortran for AIX
!
      implicit none
!
! Args:
      integer,intent(in) :: id1,id2,id3,iprint,ifatal,ispval
      integer,intent(out) :: n_total ! total number of +/-INF+NaNs
      real,intent(inout) :: f(id1,id2,id3)
      real,intent(in) :: spval
      character(len=*),intent(in) :: name 
!
! Local:
      real :: plus_inf,minus_inf,plus_nanq,minus_nanq,sig_nan
!
! For double precision 8-byte reals (-qrealsize=8):
      data plus_inf   /z'7ff0000000000000'/  ! INF   (overflow)
      data minus_inf  /z'fff0000000000000'/  ! -INF  (underflow)
      data plus_nanq  /z'7ff8000000000000'/  ! NaNQ  (plus quiet NaN)
      data minus_nanq /z'fff8000000000000'/  ! -NaNQ (minus quiet NaN)
      data sig_nan    /z'7ff0000000000001'/  ! NaNS  (signalling NaN)
!
! For single precision (4-byte) reals:
!     data plus_inf   /z'7f800000'/  ! INF   (overflow)
!     data minus_inf  /z'ff800000'/  ! -INF  (underflow)
!     data plus_nanq  /z'7fc00000'/  ! NaNQ  (plus quiet NaN)
!     data minus_nanq /z'ffc00000'/  ! -NaNQ (minus quiet NaN)
!     data sig_nan    /z'7f800001'/  ! NaNS  (signalling NaN)
!
      integer :: i1,i2,i3
      integer :: 
     |  n_plus_inf,   ! number of INF
     |  n_minus_inf,  ! number of -INF
     |  n_nan         ! total number of NaNs (+/-NaNQ and NaNS)
!
! Init:
      n_plus_inf = 0
      n_minus_inf = 0
      n_nan = 0
      n_total = 0
!
! Scan array:
      do i3=1,id3
        do i2=1,id2
!
! +/-INF are detected by simple comparison:
          n_plus_inf   = n_plus_inf   + count(f(:,i2,i3)==plus_inf) 
          n_minus_inf  = n_minus_inf  + count(f(:,i2,i3)==minus_inf) 
!
! NaNs (NaNQ or NaNS) are detected by (a /= a):
          n_nan        = n_nan        + count(f(:,i2,i3)/=f(:,i2,i3))
          n_total = n_plus_inf+n_minus_inf+n_nan
!
!         write(6,"('i3=',i3,' i2=',i3,' n_plus_inf=',i8,' n_minus_inf='
!    |      ,i8,' n_nan=',i8,' n_total=',i8)") i3,i2,n_plus_inf,
!    |      n_minus_inf,n_nan,n_total
!
! Fatal when first INF or NaN is found:
          if (ifatal > 0 .and. n_total > 0) then
            write(6,"(/,'>>> FATAL: Found INF and/or NaNs in field ',
     |        a)") name
            write(6,"('  Dimensions id1,id2,id3=',3i4)") id1,id2,id3
            write(6,"('  First INF or NaN found at id2=',i4,', id3=',
     |        i4)") i2,i3
            write(6,"('  n_plus_inf   = ',i6)") n_plus_inf
            write(6,"('  n_minus_inf  = ',i6)") n_minus_inf
            write(6,"('  n_nan (NaNS or NaNQ) = ',i6)") n_nan
            write(6,"('  data(:,',i3,',',i3,') = ',/,(6e12.4))") 
     |        i2,i3,f(:,i2,i3)
            call shutdown('check_nans')
          endif ! ifatal > 0
!
! Replace any INF or NaNs with spval:
          if (ispval > 0 .and. n_total > 0) then
            do i1=1,id1
              if (f(i1,i2,i3)==plus_inf.or.f(i1,i2,i3)==minus_inf.or.
     |            f(i1,i2,i3)/=f(i1,i2,i3)) f(i1,i2,i3) = spval
            enddo
          endif 
        enddo ! i2=1,id2
      enddo ! i3=1,id3
!
! Print level 1 (print warnings only if INF or NaNs are found):
      if (iprint==1) then
        if (n_plus_inf > 0) write(6,"('>>> WARNING: found ',
     |    i6,' INF values in field ',a,' (id1,2,3=',3i4,')')") 
     |    n_plus_inf,name,id1,id2,id3
        if (n_minus_inf > 0) write(6,"('>>> WARNING: found ',
     |    i6,' -INF values in field ',a,' (id1,2,3=',3i4,')')") 
     |    n_minus_inf,name,id1,id2,id3
        if (n_nan > 0) write(6,"('>>> WARNING: found ',i6,
     |    ' NaNS or NaNQ values in field ',a,' (id1,2,3=',3i4,')')") 
     |    n_nan,name,id1,id2,id3
!       if (ispval > 0 .and. n_total > 0) 
!    |    write(6,"('>>> Replaced ',i8,' values with spval ',e12.4)")
!    |      n_total,spval
!
! Print level 2 (always print number of nans found):
      elseif (iprint==2) then 
        write(6,"('Checking for INF and NaNs in field ',a,' id1,2,3=',
     |    3i4)") name,id1,id2,id3
        print *,'  n_plus_inf   (',plus_inf,  ') = ',n_plus_inf
        print *,'  n_minus_inf  (',minus_inf, ') = ',n_minus_inf
        print *,'  n_nan        (',plus_nanq,'+',sig_nan,') = ',n_nan
        print *,'  n_total      (total INF+NaNs) = ',n_total
!       if (ispval > 0)
!    |  print *,'  Replaced ',n_total,' values with spval ',spval
      endif
      end subroutine check_nans
!-----------------------------------------------------------------------
      integer function my_msrcp(opts,src,dest)
      use mpi_module,only: mytid
      implicit none
#ifdef MPI
#include <mpif.h>
#endif
!
! Build and execute msrcp command to the shell. Assume "mss:" prefixes 
! either src (for a get) or dest (for a put).
!
! Args:
      character(len=*),intent(in) :: opts,src,dest
!
! Local:
      character(len=1024) :: cmd
      integer :: ier
!
! External:
      integer,external :: isystem
!
      if (src(1:4) /= "mss:" .and. dest(1:4) /= "mss:") then
        write(6,"(/,'>>> my_msrcp: need mss: prefix on either src or ',
     |    'dest')")
        write(6,"('src=',a)") trim(src)
        write(6,"('dest=',a)") trim(dest)
        call shutdown('my_msrcp')
      endif
!
! Only master task does the msrcp:
      if (mytid==0) then
        if (src(1:4) == "mss:") then
          write(6,"('my_msrcp: Obtaining file ',a,' from mss.')") 
     |      src(5:len_trim(src))
        else
          write(6,"('my_msrcp: Disposing file ',a,' to mss.')") 
     |      dest(5:len_trim(dest))
        endif
        cmd = ' '
        cmd = "msrcp "//trim(opts)//' '//trim(src)//' '//trim(dest)
        write(6,"(a)") trim(cmd)
        my_msrcp = isystem(cmd)
      else
        write(6,"('my_msrcp: Waiting for master task...')")
        my_msrcp = 0
      endif
#ifdef MPI
      call mpi_barrier(MPI_COMM_WORLD,ier)
#endif
      end function my_msrcp
!-----------------------------------------------------------------------
      logical function is_mspath(path)
!
! Return true if input path begins with "/LOGNAME/", otherwise false. 
!
      implicit none
!
! Args:
      character(len=*),intent(in) :: path
!
! Local:
      character(len=64) :: logname,lognamedir
!
! External:
      character(len=64),external :: trcase ! (util.F)
!
      is_mspath = .false.
      if (path(1:1) /= '/') return
!
! Get logname in upper case:
      logname = ' '
      call getenv('LOGNAME',logname) ! get logname from env var
      if (len_trim(logname)==0) then
        write(6,"('>>> WARNING is_mspath: Cannot get LOGNAME ',
     |    'environment variable.')")
        return
      elseif (len_trim(logname) > 62) then
        write(6,"('>>> WARNING is_mspath: please lengthen logname',
     |    ' (must be at least 62: len_trim(logname)=',i3)") 
     |    len_trim(logname)
        return
      endif
      logname = trcase(logname) ! translate from lower to upper case
!
      lognamedir = '/'//trim(logname)//'/'
      if (path(1:len_trim(lognamedir))==lognamedir) is_mspath = .true.
      end function is_mspath
!-----------------------------------------------------------------------
      subroutine check_mspath(mspathin,mspathout)
!
! Check validity of mspathin as an mss path (must begin with /LOGNAME)
!
      implicit none
!
! Args:
      character(len=*),intent(in) :: mspathin
      character(len=*),intent(out) :: mspathout
!
! Local:
      integer :: i,ii,istat,ipos,oldlen,newlen
      character(len=32) :: logname
      character(len=len(logname)),external :: trcase
!
! Init:
!     write(6,"('Enter check_mspath: mspathin=',a)") trim(mspathin)
      mspathout = mspathin
!
! Mss path must begin with a slash:
      if (mspathin(1:1) /= '/') then
        if (len(mspathout) < len_trim(mspathin)+1) then
          write(6,"('>>> WARNING check_mspath: length of mspathout',
     |      ' too small: ',i4, ' (must be at least ',i4,')')") 
     |      len(mspathout),len_trim(mspathin)+1
        endif
        mspathout = ' '       ! reinit
        mspathout(1:1) = '/'
        do i=2,len_trim(mspathin)+1
          mspathout(i:i) = mspathin(i-1:i-1)    
        enddo
!       write(6,"('check_mspath: added slash to front of mspath: ',a)")
!    |    trim(mspathout)
      endif
!
! Get logname in upper case:
      logname = ' '
      call getenv('LOGNAME',logname) ! get logname from env var
      if (len_trim(logname)==0) then
        write(6,"('>>> WARNING check_mspath: Cannot get LOGNAME ',
     |    'environment variable.')")
        return
      endif
      logname = trcase(logname) ! translate from lower to upper case
!
! Mss path must begin with /LOGNAME:
      if (mspathout(2:len_trim(logname)+1) /= trim(logname)) then
        if (len(mspathout) < len_trim(mspathout)+len_trim(logname)+1) 
     |    then
          write(6,"('>>> WARNING check_mspath: length of mspathout',
     |      ' not long enough to add /',a)") trim(logname)
          return
        endif 
        oldlen = len_trim(mspathout)
        newlen = oldlen+len_trim(logname)+1 ! +1 for leading slash
        do i=newlen,newlen-oldlen+1,-1
          ii = oldlen-newlen+i
          mspathout(i:i) = mspathout(ii:ii) ! copy existing part
        enddo
        mspathout(1:len_trim(logname)+1) = '/'//trim(logname) ! add /LOGNAME
!       write(6,"('check_mspath: added /',a,' to front of mspath: ',a)")
!    |    trim(logname),trim(mspathout)
      endif
!     write(6,"('check_mspath returning: mspathout=',a)") 
!    |  trim(mspathout)
      end subroutine check_mspath
!-----------------------------------------------------------------------
      character(len=*) function trcase(string)
!
! Translate case of input string string, i.e., return string like string,
! but with lower case character converted to upper case, and vice-versa.
!
      implicit none
!
! Args:
      character(len=*),intent(in) :: string
!
! Local:
      character(len=*),parameter :: ucase = "ABCDEFGHIJKLMNOPQRSTUVWXYZ"
      character(len=*),parameter :: lcase = "abcdefghijklmnopqrstuvwxyz"
      integer :: i,lpos,upos
!
      trcase = ' '
      do i=1,len_trim(string)
        trcase(i:i) = string(i:i)
        lpos = index(lcase,trcase(i:i))
        upos = index(ucase,trcase(i:i))
        if (lpos > 0) then
          trcase(i:i) = ucase(lpos:lpos) ! lower to upper
        elseif (upos > 0) then
          trcase(i:i) = lcase(upos:upos) ! upper to lower
        endif
      enddo
      end function trcase
!-----------------------------------------------------------------------
      subroutine shutdown(msg)
!
! An fatal error has occurred -- shut down the model, including MPI.
!
      implicit none
#ifdef MPI
#include <mpif.h>
#endif
!
! Args:
      character(len=*) :: msg
!
! Local:
      integer :: ier
      character(len=80) :: errorcode
!
      write(6,"(/,28('>'),' MODEL SHUTDOWN ',28('<'))")
      write(6,"('Shutdown: stop message: ',a)") trim(msg)
#ifdef MPI
      write(6,"('Shutdown calling mpi_abort..')")
      write(6,"(72('>'))")
      call mpi_abort(MPI_COMM_WORLD,errorcode,ier) 
#endif
      write(6,"(/,'For information, please see TGCM Users Guide: ',
     |  'http://download.hao.ucar.edu/pub/tgcm/doc')")
      stop
      end subroutine shutdown
!-----------------------------------------------------------------------
      recursive subroutine expand_path(path)
!
! Expand any environment variables imbedded in path, and return 
!   expanded path. 
! Procedure:
!   If '$' is found in input path, then an env var is defined as 
!   that part of path following the '$' up to (not including) the 
!   next delimiter. The value of the env var is substituted in place 
!   of the env var string. If no '$' is found, the routine returns 
!   without changing path. 
! Environment vars can be set (using setenv) in the user's .cshrc file, 
!   in the job script (e.g., setenv from a shell var), or set manually 
!   in the shell before executing the model.
!
! The 7 recognized delimiters (meaning end of env var name) are:
!   '/' (forward slash), 
!   '.' (dot), 
!   '_' (underscore), 
!   '-' (dash), 
!   ':' (colon), 
!   '#' (pound sign), and 
!   '%' (percent sign)
!
! This routine is recursive, so multiple env vars can be used in the
!   same path, and in combination with different delimiters, see 
!   examples below.
!
! Examples:
!   path = '$TGCMDATA/dir1/file.nc'   (the env var is $TGCMDATA)
!   path = '$MYDIR/$MYSUBDIR/file.nc' (env vars are $MYDIR, $MYSUBDIR)
!   path = '$USER.$MODEL_$NUM.nc'     (3 env vars and different delims)
!   path = '$FILEPATH'                (entire path in one env var)
! Last example:
!   In the job script:
!     set model = $tiegcm  ! set a shell var
!     setenv MODEL $model  ! set env var from shell var
!   In the namelist input:
!     histfile = '$TGCMDATA/TGCM.$MODEL.p001-2002-080.nc' or
!     histfile = '$TGCMDATA/TGCM.$MODEL.p001-$YEAR-$DAY.nc'
!
      implicit none
!
! Args:
      character(len=*),intent(inout) :: path
!
! Local:
      character(len=224) :: path_out,envvar_value
      character(len=80) :: envvar_name
      integer,parameter :: ndelim=7
      character(len=1) :: delimiters(ndelim) = 
     |  (/ '/', '.', '-', '_', ':', '#', '%'/)
      integer :: i,idollar,idelim
!
      if (len_trim(path)==0) then
        write(6,"('>>> WARNING expand_path: path is empty.')")
        return
      endif
!
!     write(6,"('Enter expand_path: path=',a)") trim(path)
!
      idollar = index(path,'$')
      if (idollar <= 0) return ! no env var in path
!
! Env var is between idollar and next slash 
! (or end of path if there is no slash after idollar):
!
      idelim = 0
      do i=idollar+1,len_trim(path) ! find next delimiter
        if (any(delimiters==path(i:i))) then
          idelim = i
          exit
        endif
      enddo
      if (idelim <= 0) idelim = len_trim(path)+1
      envvar_name = path(idollar+1:idelim-1)

!     write(6,"('expand_path: path=',a,' idollar=',i3,
!    |  ' idelim=',i3,' envvar_name=',a)") 
!    |  trim(path),idollar,idelim,trim(envvar_name)

!
! Get value of env var (getenv is f90 intrinsic):
      call getenv(trim(envvar_name),envvar_value)
      if (len_trim(envvar_value) <= 0) then
        write(6,"('>>> WARNING expand_path: error retrieving ',
     |    'value for env var ''',a,'''')") trim(envvar_name)
        return
      else
!       write(6,"('expand_path: envvar=',a,' value=',a)")
!    |    trim(envvar_name),trim(envvar_value)
      endif
!
! Put together the expanded output path:
      if (idollar > 1) then
        if (idelim < len_trim(path)) then
          path_out = path(1:idollar-1)//trim(envvar_value)//
     |      path(idelim:len_trim(path))
        else
          path_out = path(1:idollar-1)//trim(envvar_value)
        endif
      else     ! idollar == 1
        if (idelim < len_trim(path)) then
          path_out = trim(envvar_value)//path(idelim:len_trim(path))
        else
          path_out = trim(envvar_value)
        endif
      endif
!
! Return new path, and make recursive call for more env vars:
      path = trim(path_out)
      write(6,"('expand_path returning path = ''',a,'''')") trim(path)
!
! Recursive call to expand any additional env vars:
      call expand_path(path) ! expand next env var
!
      end subroutine expand_path
!-----------------------------------------------------------------------
      logical function time2print(nstep,istep)
      implicit none
        integer,intent(in) :: nstep,istep
        time2print = .false.
        if (nstep <= 100 .or. (nstep > 100 .and. mod(istep,10)==0))
     |    time2print = .true.
      end function time2print
