!
      module cco2gr_module
      implicit none
!
      real :: amat(43,9),bmat(43,9)
!
! Vertical grids to be used in IR scheme
! xr(67) - pressure scale heights, psh's, (=0-16.5) at which input parameter
!          should be given
! XRH(59) - psh's at which cooling rates compute (=2-16.5)
! XRF(17) - psh's at which the reccurence formula is utilized (=12.5-16.5)
!
      real :: xr(67), xrh(59), xrf(17)
!
! DATA for "LTE" parameterization (x=2-12.5)
! IG(9) - indexes of level which should be used to account for the internal
!         atmospheric heat exchange
! AO3(35,9) - coefficients for O3 scheme to calculate cooling rate in
!             "erg/g/s" at levels x=2-10.5 (with a step of 0.25)
! CO2O(4) - vmr for basic CO2
! "LTE-coefficients" for CO2 scheme using to calculate cooling rate in
! "erg/g/s" in region x=2-12.5 (with a step of 0.25). To account for internal
! heat exchange 9 level in atmosphere are needed. 
!  A150, B150(43,9) - for 150ppm of CO2
!  A360, B360(43,9) - for 360ppm of CO2
!  A540, B540(43,9) - for 540ppm of CO2
!  A720, B720(43,9) - for 720ppm of CO2
!
      integer :: nlevlte=9
      real,parameter :: dzco2=0.25, zp0co2=2., zp1co2=12.5
      integer,parameter :: nco2 = (zp1co2-zp0co2)/dzco2+1
      real :: a150(43,9),b150(43,9), a360(43,9),b360(43,9),
     |        a540(43,9),b540(43,9), a720(43,9),b720(43,9),co2o(4)
      real,parameter :: dzo3 =0.25, zp0o3 =2., zp1o3 =10.5
      integer,parameter :: no3 = (zp1o3-zp0o3)/dzo3+1
      real :: ao3(35,9)
      integer :: ig(9)
!
! DATA for NLTE parameterization CO2 (x=12.5-16.5)
! UCO2RO, ALO(51) - CO2 column amount and corresponding escape functions
!                                              (eventually, their log)
! COR150, COR360, COR540, COR720(6) - correction to escape functions to
!   calculate coefficients for the reccurence formula between x=12.5 and 13.75
! UCO2CO(6) - CO2 column amount at x=12.5-13.75 (step - 0.25) for 360 ppm
!             to be used to correct escape functions in this region
!
      real :: uco2ro(51),alo(51),cor150(6),cor360(6),cor540(6),
     |        cor720(6),uco2co(6)
!
! CO2VP(67) - volume mixing ratio (vmr) for basic CO2 profile at XR grid
! CO2CL(67) - CO2 column amount at XR grid calculated with use of CO2VP
!
      real :: co2vp(67), co2cl(67)

      contains
!-----------------------------------------------------------------------
      subroutine cco2gr(rco2)
      implicit none
!
! Args:
      real,intent(in) :: rco2 ! co2 volume mixing in region below x=12.5
!
! Local:
      real :: uref(4), co2int(4)
      integer :: i,j,isgn
      real :: a(1),rco2vec(1)
!
! Calculate coefficients for the matrix paramerization:
!
!     write(6,"('cco2gr: nco2=',i3)") nco2
!     write(6,"('cco2gr: co2o=',4e12.4)") co2o
      rco2vec(1) = rco2
!
      do j = 1,9
        do i = 1,43
          if ((i.le.5).and.(j.eq.2)) cycle
!
! amat:
          isgn = int(sign(1.,a150(i,j))+sign(1.,a360(i,j))+
     |           sign(1.,a540(i,j))+sign(1.,a720(i,j)))
          co2int(1)=a150(i,j)/co2o(1)
          co2int(2)=a360(i,j)/co2o(2)
          co2int(3)=a540(i,j)/co2o(3)
          co2int(4)=a720(i,j)/co2o(4)
          if(isgn.eq.-4) then
            co2int(1) = alog(-co2int(1))
            co2int(2) = alog(-co2int(2))
            co2int(3) = alog(-co2int(3))
            co2int(4) = alog(-co2int(4))
            a(1) = -exp(a18lin(rco2,co2o,co2int,1,4))
          else if (isgn.eq.4) then
            co2int(1) = alog(co2int(1))
            co2int(2) = alog(co2int(2))
            co2int(3) = alog(co2int(3))
            co2int(4) = alog(co2int(4))
            a(1) = exp(a18lin(rco2,co2o,co2int,1,4))
          else
            call a18int(co2o,co2int,rco2vec,a,4,1) ! a(1) is returned
          endif
          amat(i,j)=a(1)*rco2
!
! bmat:
          isgn = int(sign(1.,b150(i,j))+sign(1.,b360(i,j))+
     |               sign(1.,b540(i,j))+sign(1.,b720(i,j)))
          co2int(1)=b150(i,j)/co2o(1)
          co2int(2)=b360(i,j)/co2o(2)
          co2int(3)=b540(i,j)/co2o(3)
          co2int(4)=b720(i,j)/co2o(4)
          if(isgn.eq.-4) then
            co2int(1) = alog(-co2int(1))
            co2int(2) = alog(-co2int(2))
            co2int(3) = alog(-co2int(3))
            co2int(4) = alog(-co2int(4))
            a(1) = -exp(a18lin(rco2,co2o,co2int,1,4))
          else if (isgn.eq.4) then
            co2int(1) = alog(co2int(1))
            co2int(2) = alog(co2int(2))
            co2int(3) = alog(co2int(3))
            co2int(4) = alog(co2int(4))
            a(1) = exp(a18lin(rco2,co2o,co2int,1,4))
          else
            call a18int(co2o,co2int,rco2vec,a,4,1) ! a(1) is returned
          end if
          bmat(i,j)=a(1)*rco2
        enddo ! i=1,43
      enddo ! j=1,9

!     do j=1,9
!       write(6,"('cco2gr: j=',i3,' amat(:,j)=',/,(6e12.4))")j,amat(:,j)
!       write(6,"('cco2gr: j=',i3,' bmat(:,j)=',/,(6e12.4))")j,bmat(:,j)
!     enddo

      end subroutine cco2gr
!-----------------------------------------------------------------------
      real function a18lin(X,XN,YN,M,N)
      implicit none
!
! linear interpolation (called by cco2gr)
!
! input:
!  X - argument for which a value of function should be found
!  XN(N),YN(N) - values of function YN(N) at XN(N) grid. X(N) should be
!                ordered so that X(I-1) < X(I).
! output:
!  a18lin - value of function for X
!
! Args:
      integer,intent(in) :: m,n
      real,intent(in) :: XN(N),YN(N),x
! 
! Local:
      integer :: k,i
!
      k=m-1
      loop: do i=m,n
        k=k+1
        if (x <= xn(i)) exit loop
      enddo loop
      if (k==1) k=2
!
! k has been found so that xn(k).le.x.lt.xn(k+1)
!
      a18lin = (yn(k)-yn(k-1))/(xn(k)-xn(k-1))*(x-xn(k))+yn(k)

!     write(6,"('a18lin: k=',i3,' x=',e12.4,' yn(k)=',e12.4,' yn(k-1)=',
!    |  e12.4,' xn(k)=',e12.4,' xn(k-1)=',e12.4,' a18lin=',e12.4)")
!    |  k,x,yn(k),yn(k-1),xn(k),xn(k-1),a18lin

      end function a18lin
!-----------------------------------------------------------------------
      subroutine a18int(x1,y1,x2,y2,n1,n2)
      implicit none
!
! Third order spline interpolation
! input argument and function:  X1(1:N1),Y1(1:N1)
! output argument and function: X2(1:N2)X2(1:N2),Y2(1:N2)
! the necessary conditionts are: X1(I) < X1(I+1), and the same for X2 array.
!
! Args:
      integer,intent(in) :: n1,n2
      real,intent(in)  :: x1(n1),x2(n2),y1(n1)
      real,intent(out) :: y2(n2)
!
! Local:
      integer :: k,nvs,kr,l,i
      real :: h1,h2,g,f1,f2,f3
      real :: a(150),e(150),f(150),h(150)
!
      h2=x1(1)
      nvs=n1-1
      do k=1,nvs
        h1=h2
        h2=x1(k+1)
        h(k)=h2-h1
      enddo
      a(1)=0.
      a(n1)=0.
      e(n1)=0.
      f(n1)=0.
      h1=h(n1-1)
      f1=y1(n1-1)
      f2=y1(n1)
      do kr=2,nvs
        k=nvs+2-kr
        h2=h1
        h1=h(k-1)
        f3=f2
        f2=f1
        f1=y1(k-1)
        g=1/(h2*e(k+1)+2.*(h1+h2))
        e(k)=-h1*g
        f(k)=(3.*((f3-f2)/h2-(f2-f1)/h1)-h2*f(k+1))*g
      enddo
      g=0.
      do k=2,nvs
        g=e(k)*g+f(k)
        a(k)=g
      enddo
      l=1
      do i=1,n2
        g=x2(i)
        do 6 k=l,nvs
          if(g.gt.x1(k+1))goto6
          l=k
          goto 5
    6   continue
        l=nvs
    5   g=g-x1(l)
        h2=h(l)
        f2=y1(l)
        f1=h2**2
        f3=g**2
        y2(i)=f2+g/h2*(y1(l+1)-f2-(a(l+1)*(f1-f3)+
     |                 a(l)*(2.*f1-3.*g*h2+f3))/3.)
      enddo
      end subroutine a18int
!-----------------------------------------------------------------------
      subroutine set_cco2_data
!
! Set module data with array constructors:
! (In previous versions, these were in common and initialized in block data)
!
! 11/2002 btf: This new version of blockdata PCO2O3.F was received from 
!   Vladimir Ogibalov 11/7/02.
! 12/2003 btf: Convert from common and block data to module data and array
!   constructors. Used code in hao:/home/foster/timegcm/timegcm1/mkcco2gr 
!   to convert arrays ao3, a150, b150, etc.
!
      xr = (/
     | 0.0, 0.25, 0.5, 0.75, 1.0, 1.25, 1.5, 1.75, 2.0, 2.25, 2.5, 2.75,
     | 3.0, 3.25, 3.5, 3.75, 4.0, 4.25, 4.5, 4.75, 5.0, 5.25, 5.5, 5.75,
     | 6.0, 6.25, 6.5, 6.75, 7.0, 7.25, 7.5, 7.75, 8.0, 8.25, 8.5, 8.75,
     | 9.0, 9.25, 9.5, 9.75,10.0,10.25,10.5,10.75,11.0,11.25,11.5,11.75,
     |12.0,12.25,12.5,12.75,13.0,13.25,13.5,13.75,14.0,14.25,14.5,14.75,
     |15.0,15.25,15.5,15.75,16.0,16.25,16.5/)
!
      xrh = (/
     | 2.0, 2.25, 2.5, 2.75, 3.0, 3.25, 3.5, 3.75, 4.0, 4.25, 4.5, 4.75,
     | 5.0, 5.25, 5.5, 5.75, 6.0, 6.25, 6.5, 6.75, 7.0, 7.25, 7.5, 7.75,
     | 8.0, 8.25, 8.5, 8.75, 9.0, 9.25, 9.5, 9.75,10.0,10.25,10.5,10.75,
     |11.0,11.25,11.5,11.75,12.0,12.25,12.5,12.75,13.0,13.25,13.5,13.75,
     |14.0,14.25,14.5,14.75,15.0,15.25,15.5,15.75,16.0,16.25,16.5/)
!
      xrf = (/
     |12.5,12.75,13.0,13.25,13.5,13.75,14.0,14.25,14.5,14.75,15.0,15.25,
     |15.5,15.75,16.0,16.25,16.5/)

      co2vp(1:50) = 3.600e-04
      co2vp(51:67) = (/
     | 3.58E-04, 3.54E-04, 3.50E-04, 3.41E-04, 3.28E-04, 3.11E-04,
     | 2.93E-04, 2.75E-04, 2.56E-04, 2.37E-04, 2.18E-04, 1.99E-04,
     | 1.80E-04, 1.61E-04, 1.42E-04, 1.24E-04, 1.06E-04/)

      co2cl = (/
     | 7.760162E+21,6.043619E+21,4.706775E+21,3.665639E+21,2.854802E+21,
     | 2.223321E+21,1.731524E+21,1.348511E+21,1.050221E+21,8.179120E+20,
     | 6.369897E+20,4.960873E+20,3.863524E+20,3.008908E+20,2.343332E+20,
     | 1.824981E+20,1.421289E+20,1.106894E+20,8.620418E+19,6.713512E+19,
     | 5.228412E+19,4.071814E+19,3.171055E+19,2.469544E+19,1.923206E+19,
     | 1.497717E+19,1.166347E+19,9.082750E+18,7.072886E+18,5.507602E+18,
     | 4.288557E+18,3.339164E+18,2.599777E+18,2.023941E+18,1.575480E+18,
     | 1.226217E+18,9.542118E+17,7.423736E+17,5.773939E+17,4.489076E+17,
     | 3.488423E+17,2.709114E+17,2.102188E+17,1.629513E+17,1.261393E+17,
     | 9.747013E+16,7.514254E+16,5.775381E+16,4.421144E+16,3.366464E+16,
     | 2.546953E+16,1.913609E+16,1.425730E+16,1.052205E+16,7.700499E+15,
     | 5.596946E+15,4.044901E+15,2.905406E+15,2.073209E+15,1.469497E+15,
     | 1.033831E+15,7.212717E+14,4.986668E+14,3.415852E+14,2.320320E+14,
     | 1.565317E+14,1.052353E+14/)

      ig = (/-25,-12,-7,-3,-1,0,1,3,6/)
!
      co2o = (/150.e-6, 360.e-6, 540.e-6, 720.e-6/)
!
      uco2co = (/2.546953E+16,1.913609E+16,1.425730E+16,
     |           1.052205E+16,7.700499E+15,5.596946E+15/)
!
      cor150 = (/2.225581E-01,1.893568E-01,1.459362E-01,
     |           8.327750E-02,3.570287E-02,1.414133E-03/)
      cor360 = (/5.586429E-01,4.605370E-01,3.374704E-01,
     |           1.920729E-01,8.415350E-02,3.254796E-02/)
      cor540 = (/6.958750E-01,6.029840E-01,4.752609E-01,
     |           3.070259E-01,1.589305E-01,5.626345E-02/)
      cor720 = (/7.989408E-01,6.929009E-01,5.531082E-01,
     |           3.669253E-01,1.967602E-01,7.187870E-02/)
!
      uco2ro = (/2.699726E+11,5.810773E+11,1.106722E+12,
     | 1.952319E+12,3.306797E+12,5.480155E+12,8.858565E+12,1.390142E+13,
     | 2.129301E+13,3.209300E+13,4.784654E+13,7.091442E+13,1.052353E+14,
     | 1.565317E+14,2.320320E+14,3.415852E+14,4.986668E+14,7.212717E+14,
     | 1.033831E+15,1.469497E+15,2.073209E+15,2.905406E+15,4.044901E+15,
     | 5.596946E+15,7.700499E+15,1.052205E+16,1.425730E+16,1.913609E+16,
     | 2.546953E+16,3.366464E+16,4.421144E+16,5.775381E+16,7.514254E+16,
     | 9.747013E+16,1.261393E+17,1.629513E+17,2.102188E+17,2.709114E+17,
     | 3.488423E+17,4.489076E+17,5.773939E+17,7.423736E+17,9.542118E+17,
     | 1.226217E+18,1.575480E+18,2.023941E+18,2.599777E+18,3.339164E+18,
     | 4.288557E+18,5.507602E+18,7.072886E+18/)
!
      alo = (/-2.410106E-04,-5.471415E-04,-1.061586E-03,
     |        -1.879789E-03,-3.166020E-03,-5.185436E-03,-8.216667E-03,
     |        -1.250894E-02,-1.838597E-02,-2.631114E-02,-3.688185E-02,
     |        -5.096491E-02,-7.004056E-02,-9.603746E-02,-1.307683E-01,
     |        -1.762946E-01,-2.350226E-01,-3.095215E-01,-4.027339E-01,
     |        -5.178570E-01,-6.581256E-01,-8.265003E-01,-1.024684E+00,
     |        -1.252904E+00,-1.509470E+00,-1.788571E+00,-2.081700E+00,
     |        -2.379480E+00,-2.675720E+00,-2.967325E+00,-3.252122E+00,
     |        -3.530485E+00,-3.803720E+00,-4.072755E+00,-4.338308E+00,
     |        -4.601048E+00,-4.861585E+00,-5.120370E+00,-5.377789E+00,
     |        -5.634115E+00,-5.889388E+00,-6.143488E+00,-6.396436E+00,
     |        -6.648774E+00,-6.901465E+00,-7.155207E+00,-7.409651E+00,
     |        -7.663536E+00,-7.915682E+00,-8.165871E+00,-8.415016E+00/)
!
      AO3(:, 1) = (/
     |  5.6900E+09, 4.9970E+09, 4.1790E+09, 3.4690E+09, 2.9300E+09,
     |  2.0200E+09, 1.6520E+09, 1.3910E+09, 1.1970E+09, 9.9610E+08,
     |  8.4080E+08, 7.2500E+08, 6.4840E+08, 5.7850E+08, 5.2720E+08,
     |  4.8910E+08, 4.6430E+08, 4.4610E+08, 6.4710E+08, 5.3770E+08,
     |  7.0640E+08, 1.2100E+09, 2.1210E+09, 3.3930E+09, 4.8380E+09,
     |  5.8520E+09, 6.6240E+09, 7.0300E+09, 8.0990E+09, 8.3820E+09,
     |  9.5450E+09, 1.0360E+10, 1.1800E+10, 8.8780E+09, 4.6520E+09
     | /)
      AO3(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  3.8660E+08, 3.7110E+08, 3.4970E+08, 3.3290E+08, 3.3900E+08,
     |  4.9220E+08, 5.3530E+08, 5.2140E+08, 4.9870E+08, 4.9550E+08,
     |  5.1310E+08, 5.6680E+08, 6.3910E+08, 1.5360E+08, 1.1010E+08,
     |  1.3460E+08, 2.1330E+08, 3.4500E+08, 5.0660E+08, 6.5340E+08,
     |  1.2860E+09, 2.4390E+09, 3.8650E+09, 4.8430E+09, 6.4260E+09,
     |  7.6670E+09, 9.1460E+09, 1.0210E+10, 7.1840E+09, 3.4690E+09
     | /)
      AO3(:, 3) = (/
     |  1.9620E+09, 2.1680E+09, 2.1980E+09, 1.9410E+09, 1.6490E+09,
     |  1.3680E+09, 1.1750E+09, 1.0330E+09, 9.6890E+08, 9.8670E+08,
     |  9.2260E+08, 9.4930E+08, 1.0140E+09, 1.0370E+09, 1.1940E+09,
     |  1.2990E+09, 1.5000E+09, 1.8700E+09, 2.5670E+09, 3.4740E+09,
     |  4.5350E+09, 5.7720E+09, 7.1110E+09, 8.1760E+09, 9.1880E+09,
     |  9.8670E+09, 1.0440E+10, 1.0550E+10, 1.0650E+10, 1.0180E+10,
     |  9.4290E+09, 8.0140E+09, 6.7250E+09, 3.5690E+09, 1.3850E+09
     | /)
      AO3(:, 4) = (/
     |  5.0150E+09, 4.9810E+09, 4.5940E+09, 3.9210E+09, 3.3780E+09,
     |  3.0490E+09, 2.7470E+09, 2.5700E+09, 2.4270E+09, 2.3750E+09,
     |  2.3580E+09, 2.4970E+09, 2.6250E+09, 2.6940E+09, 2.9980E+09,
     |  3.4560E+09, 4.0960E+09, 4.8810E+09, 5.9090E+09, 6.9920E+09,
     |  7.8140E+09, 8.4020E+09, 8.6110E+09, 8.4190E+09, 7.8950E+09,
     |  7.1210E+09, 5.9500E+09, 4.7810E+09, 3.7320E+09, 2.7710E+09,
     |  1.9270E+09, 1.3550E+09, 9.5750E+08, 4.0230E+08, 1.1140E+08
     | /)
      AO3(:, 5) = (/
     |  7.1050E+09, 7.3270E+09, 7.7800E+09, 8.4230E+09, 8.9850E+09,
     |  9.4520E+09, 9.8180E+09, 1.0080E+10, 1.0240E+10, 1.0320E+10,
     |  1.0390E+10, 1.0400E+10, 1.0380E+10, 1.0390E+10, 1.0260E+10,
     |  1.0160E+10, 9.9720E+09, 9.6080E+09, 9.0880E+09, 8.3330E+09,
     |  7.3610E+09, 6.1840E+09, 4.8740E+09, 3.7770E+09, 2.7460E+09,
     |  2.0460E+09, 1.3800E+09, 1.0090E+09, 6.7980E+08, 4.5930E+08,
     |  3.1430E+08, 2.2800E+08, 1.0910E+08, 4.6560E+07, 1.3680E+07 
     | /)
      AO3(:, 6) = (/
     | -3.9520E+10,-3.8550E+10,-3.6890E+10,-3.5080E+10,-3.3620E+10,
     | -3.2490E+10,-3.1650E+10,-3.1020E+10,-3.0560E+10,-3.0330E+10,
     | -3.0300E+10,-3.0470E+10,-3.0580E+10,-3.0890E+10,-3.1600E+10,
     | -3.2750E+10,-3.4220E+10,-3.6210E+10,-3.8440E+10,-4.0650E+10,
     | -4.2480E+10,-4.4000E+10,-4.4990E+10,-4.5730E+10,-4.6280E+10,
     | -4.6630E+10,-4.6910E+10,-4.7080E+10,-4.7190E+10,-4.7260E+10,
     | -4.7310E+10,-4.7340E+10,-4.7360E+10,-3.0790E+10,-1.4210E+10 
     | /)
      AO3(:, 7) = (/
     |  8.9160E+09, 9.2960E+09, 9.6240E+09, 9.8590E+09, 9.9410E+09,
     |  1.0100E+10, 1.0140E+10, 1.0170E+10, 1.0110E+10, 1.0110E+10,
     |  1.0040E+10, 9.9710E+09, 9.9030E+09, 9.6980E+09, 9.4660E+09,
     |  9.0030E+09, 8.3450E+09, 7.3860E+09, 6.2480E+09, 4.9790E+09,
     |  3.7860E+09, 2.8830E+09, 2.0550E+09, 1.4180E+09, 9.5480E+08,
     |  6.6430E+08, 4.2090E+08, 3.2380E+08, 1.8520E+08, 1.0840E+08,
     |  7.0750E+07, 4.4590E+07, 3.0180E+07, 1.3740E+07, 3.5530E+06 
     | /)
      AO3(:, 8) = (/
     |  1.2040E+09, 1.3260E+09, 1.1070E+09, 8.7360E+08, 8.1100E+08,
     |  6.2120E+08, 5.4700E+08, 5.1210E+08, 4.5210E+08, 5.3220E+08,
     |  5.5510E+08, 8.3890E+08, 7.7200E+08, 1.0680E+09, 1.2390E+09,
     |  1.5930E+09, 1.7900E+09, 2.1050E+09, 2.1730E+09, 2.0190E+09,
     |  1.7330E+09, 1.5300E+09, 1.1260E+09, 8.9270E+08, 6.0330E+08,
     |  4.0340E+08, 2.6350E+08, 2.1070E+08, 1.2330E+08, 8.3680E+07,
     |  5.1790E+07, 3.4630E+07, 2.0000E+07, 0.0000E+00, 0.0000E+00 
     | /)
      AO3(:, 9) = (/
     |  5.4340E+09, 4.5750E+09, 3.7360E+09, 3.0580E+09, 2.3670E+09,
     |  2.1170E+09, 1.8430E+09, 1.5520E+09, 1.3120E+09, 1.1780E+09,
     |  1.1410E+09, 9.1220E+08, 9.4550E+08, 8.9030E+08, 9.6590E+08,
     |  1.0830E+09, 1.2560E+09, 1.4600E+09, 1.5360E+09, 1.5940E+09,
     |  1.4310E+09, 1.1630E+09, 8.8400E+08, 6.3560E+08, 4.8100E+08,
     |  2.2580E+08, 2.2110E+08, 1.0940E+08, 1.5130E+08, 0.0000E+00,
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00 
     | /)
!
      A150(:, 1) = (/
     |  2.5071E+01, 3.1500E+01, 3.6182E+01, 3.9423E+01, 4.3831E+01,
     |  1.4878E+01, 1.8783E+01, 2.3009E+01, 2.7052E+01, 3.0101E+01,
     |  3.3087E+01, 3.6750E+01, 4.0942E+01, 4.5502E+01, 5.0618E+01,
     |  5.6355E+01, 6.2614E+01, 6.9518E+01, 9.8718E+01, 1.4212E+02,
     |  2.0879E+02, 3.0514E+02, 4.3094E+02, 5.6236E+02, 6.5594E+02,
     |  7.1475E+02, 7.5797E+02, 8.0789E+02, 8.6499E+02, 9.3702E+02,
     |  1.0118E+03, 1.1289E+03, 1.2846E+03, 1.4396E+03, 1.5560E+03,
     |  1.5988E+03, 1.6379E+03, 1.9572E+03, 2.3277E+03, 2.4400E+03,
     |  2.1244E+03, 1.8447E+03, 1.6368E+03/)
      A150(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  5.2045E+01, 8.2394E+01, 1.2961E+02, 1.9750E+02, 2.8472E+02,
     |  3.8705E+02, 5.0001E+02, 6.2556E+02, 7.5169E+02, 8.8381E+02,
     |  1.0260E+03, 1.1804E+03, 1.3516E+03, 1.4643E+03, 1.5585E+03,
     |  1.6312E+03, 1.6987E+03, 1.7637E+03, 1.8547E+03, 1.9790E+03,
     |  2.1387E+03, 2.3270E+03, 2.5377E+03, 2.7558E+03, 2.9914E+03,
     |  3.2266E+03, 3.6523E+03, 4.3436E+03, 5.1762E+03, 5.9325E+03,
     |  6.5593E+03, 7.0124E+03, 8.9672E+03, 1.0922E+04, 1.1103E+04,
     |  9.4411E+03, 7.4007E+03, 7.3316E+03/)
      A150(:, 3) = (/
     |  9.4271E+01, 1.5663E+02, 2.5371E+02, 4.0505E+02, 6.3483E+02,
     |  8.8175E+02, 1.1183E+03, 1.3047E+03, 1.4428E+03, 1.5602E+03,
     |  1.6765E+03, 1.8177E+03, 1.9783E+03, 2.1208E+03, 2.2351E+03,
     |  2.3261E+03, 2.4203E+03, 2.5392E+03, 2.7198E+03, 2.9407E+03,
     |  3.1936E+03, 3.4636E+03, 3.7656E+03, 4.0928E+03, 4.4193E+03,
     |  4.7545E+03, 5.1240E+03, 5.5538E+03, 6.0549E+03, 6.5966E+03,
     |  7.1061E+03, 7.4369E+03, 7.9078E+03, 7.9222E+03, 7.2371E+03,
     |  5.7913E+03, 4.8550E+03, 4.6177E+03, 3.2674E+03, 1.6305E+03,
     |  1.9400E+03, 4.4430E+03, 8.1797E+03/)
      A150(:, 4) = (/
     |  2.4800E+03, 3.1888E+03, 3.8837E+03, 4.4637E+03, 4.9463E+03,
     |  5.4048E+03, 5.7955E+03, 6.1382E+03, 6.4393E+03, 6.6833E+03,
     |  6.8880E+03, 7.0808E+03, 7.3050E+03, 7.5845E+03, 7.9294E+03,
     |  8.3275E+03, 8.7576E+03, 9.2050E+03, 9.6728E+03, 1.0181E+04,
     |  1.0755E+04, 1.1429E+04, 1.2255E+04, 1.3288E+04, 1.4473E+04,
     |  1.5789E+04, 1.7208E+04, 1.8698E+04, 1.9828E+04, 2.0619E+04,
     |  1.9300E+04, 1.8642E+04, 1.5187E+04, 1.3106E+04, 1.2254E+04,
     |  1.5823E+04, 2.0499E+04, 2.8360E+04, 3.1782E+04, 3.0642E+04,
     |  3.2600E+04, 3.9264E+04, 4.2337E+04/)
      A150(:, 5) = (/
     |  3.2513E+03, 3.0558E+03, 2.8503E+03, 2.6177E+03, 2.3860E+03,
     |  2.1308E+03, 1.8912E+03, 1.6883E+03, 1.4998E+03, 1.3466E+03,
     |  1.2661E+03, 1.2371E+03, 1.1883E+03, 1.1224E+03, 1.0080E+03,
     |  8.5741E+02, 7.1144E+02, 5.9872E+02, 5.6542E+02, 6.2949E+02,
     |  8.0516E+02, 1.0746E+03, 1.4451E+03, 1.8658E+03, 2.2733E+03,
     |  2.6324E+03, 2.9732E+03, 3.3256E+03, 3.7624E+03, 4.4109E+03,
     |  4.9733E+03, 5.0113E+03, 3.5605E+03, 1.0243E+03, 1.4708E+03,
     |  5.6122E+03, 1.1029E+04, 1.7336E+04, 2.1909E+04, 1.7039E+04,
     |  1.6317E+04, 1.3974E+04, 7.1853E+03/)
      A150(:, 6) = (/
     | -1.1106E+04,-1.2561E+04,-1.3998E+04,-1.5190E+04,-1.6273E+04,
     | -1.7303E+04,-1.8251E+04,-1.9125E+04,-1.9898E+04,-2.0578E+04,
     | -2.1290E+04,-2.2132E+04,-2.3129E+04,-2.4219E+04,-2.5296E+04,
     | -2.6387E+04,-2.7544E+04,-2.8845E+04,-3.0324E+04,-3.2068E+04,
     | -3.4136E+04,-3.6586E+04,-3.9494E+04,-4.2976E+04,-4.6760E+04,
     | -5.0774E+04,-5.5092E+04,-5.9805E+04,-6.4113E+04,-6.8151E+04,
     | -6.8907E+04,-7.2643E+04,-7.5466E+04,-7.9021E+04,-8.2035E+04,
     | -9.1345E+04,-1.0239E+05,-1.3540E+05,-1.6359E+05,-1.6327E+05,
     | -1.4953E+05,-1.4300E+05,-1.3941E+05/)
      A150(:, 7) = (/
     |  4.0759E+02, 2.1714E+02,-3.8877E+00,-2.0215E+02,-4.0125E+02,
     | -5.7143E+02,-7.1052E+02,-8.4795E+02,-9.8534E+02,-1.1112E+03,
     | -1.2328E+03,-1.3990E+03,-1.5833E+03,-1.8075E+03,-2.0658E+03,
     | -2.3450E+03,-2.6296E+03,-2.9075E+03,-3.1708E+03,-3.4232E+03,
     | -3.6656E+03,-3.8845E+03,-4.0956E+03,-4.3148E+03,-4.5121E+03,
     | -4.6927E+03,-4.8750E+03,-5.0617E+03,-5.2269E+03,-5.4357E+03,
     | -5.4701E+03,-5.1896E+03,-3.7702E+03,-2.5666E+03,-2.5030E+03,
     | -4.3147E+03,-6.9774E+03,-9.7833E+03,-1.1857E+04,-1.3289E+04,
     | -1.6849E+04,-2.0222E+04,-2.1286E+04/)
      A150(:, 8) = (/
     |  2.1092E+03, 2.2159E+03, 2.2559E+03, 2.2435E+03, 2.2322E+03,
     |  2.2461E+03, 2.3035E+03, 2.4024E+03, 2.5407E+03, 2.7005E+03,
     |  2.8990E+03, 3.1369E+03, 3.4181E+03, 3.7347E+03, 4.0586E+03,
     |  4.3915E+03, 4.7430E+03, 5.1245E+03, 5.5386E+03, 5.9821E+03,
     |  6.4358E+03, 6.8935E+03, 7.3534E+03, 7.7738E+03, 8.1064E+03,
     |  8.3764E+03, 8.6999E+03, 9.1641E+03, 9.7665E+03, 1.0776E+04,
     |  1.2184E+04, 1.5425E+04, 1.9804E+04, 2.3508E+04, 2.4825E+04,
     |  2.7342E+04, 2.7750E+04, 3.9192E+04, 5.3784E+04, 5.9206E+04,
     |  5.1082E+04, 4.6098E+04, 4.2111E+04/)
      A150(:, 9) = (/
     |  6.2086E+02, 7.7191E+02, 9.0957E+02, 1.0115E+03, 1.0977E+03,
     |  1.1734E+03, 1.2338E+03, 1.2767E+03, 1.2947E+03, 1.2948E+03,
     |  1.2917E+03, 1.3031E+03, 1.3338E+03, 1.3777E+03, 1.4267E+03,
     |  1.4833E+03, 1.5496E+03, 1.6320E+03, 1.7411E+03, 1.8903E+03,
     |  2.0961E+03, 2.3808E+03, 2.7609E+03, 3.3142E+03, 4.0070E+03,
     |  4.7896E+03, 5.5051E+03, 6.0821E+03, 6.1482E+03, 5.5120E+03,
     |  3.5634E+03, 2.3983E+03, 2.7041E+03, 3.8499E+03, 3.9419E+03,
     |  2.8183E+03, 2.5084E+03, 4.2200E+03, 7.3094E+03, 1.0492E+04,
     |  7.9873E+03, 5.1589E+03, 1.7588E+03/)
!
      B150(:, 1) = (/
     |  3.5449E+03, 3.1198E+03, 3.0384E+03, 3.2962E+03, 3.5024E+03,
     |  1.8651E+03, 2.0892E+03, 2.2587E+03, 2.4497E+03, 2.7043E+03,
     |  3.0068E+03, 3.3240E+03, 3.6770E+03, 4.0167E+03, 4.3171E+03,
     |  4.6133E+03, 4.8908E+03, 5.1235E+03, 6.9450E+03, 9.4886E+03,
     |  1.3199E+04, 1.8354E+04, 2.4240E+04, 2.8903E+04, 3.2229E+04,
     |  3.4894E+04, 3.7634E+04, 4.0672E+04, 4.3148E+04, 4.5406E+04,
     |  4.7906E+04, 5.1633E+04, 5.7102E+04, 6.3653E+04, 7.0083E+04,
     |  7.4838E+04, 7.9294E+04, 9.1393E+04, 1.0735E+05, 1.1778E+05,
     |  1.0949E+05, 1.0023E+05, 9.5218E+04/)
      B150(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  2.7055E+03, 3.6224E+03, 4.7724E+03, 6.3000E+03, 8.2949E+03,
     |  1.0333E+04, 1.2019E+04, 1.3364E+04, 1.4412E+04, 1.5293E+04,
     |  1.6161E+04, 1.7128E+04, 1.8156E+04, 1.4807E+04, 1.0885E+04,
     |  6.7662E+03, 2.9583E+03,-7.2004E+00,-1.8052E+03,-2.3770E+03,
     | -1.7051E+03, 2.6926E+02, 4.1703E+03, 1.0496E+04, 1.8983E+04,
     |  2.9347E+04, 4.4741E+04, 6.8127E+04, 9.8287E+04, 1.2928E+05,
     |  1.5849E+05, 1.8189E+05, 2.5293E+05, 3.3378E+05, 3.6386E+05,
     |  3.0448E+05, 2.1333E+05, 1.9234E+05/)
      B150(:, 3) = (/
     |  6.0595E+03, 6.9779E+03, 9.0741E+03, 1.3150E+04, 1.8025E+04,
     |  1.9742E+04, 1.9185E+04, 1.7136E+04, 1.5322E+04, 1.4666E+04,
     |  1.4752E+04, 1.5114E+04, 1.5485E+04, 1.5478E+04, 1.5287E+04,
     |  1.4985E+04, 1.4688E+04, 1.4558E+04, 1.6724E+04, 1.9530E+04,
     |  2.2979E+04, 2.7019E+04, 3.1412E+04, 3.5604E+04, 4.1190E+04,
     |  4.9606E+04, 6.1632E+04, 7.7911E+04, 9.8423E+04, 1.2247E+05,
     |  1.4976E+05, 1.7661E+05, 2.0893E+05, 2.2878E+05, 2.2377E+05,
     |  1.9011E+05, 1.6701E+05, 1.5458E+05, 1.1192E+05, 5.7171E+04,
     |  7.2184E+04, 1.5491E+05, 2.7703E+05/)
      B150(:, 4) = (/
     |  5.9194E+04, 5.7206E+04, 5.5062E+04, 5.6108E+04, 5.4886E+04,
     |  5.4105E+04, 5.2441E+04, 4.9727E+04, 4.8638E+04, 4.9697E+04,
     |  5.1004E+04, 5.1233E+04, 5.0360E+04, 4.8940E+04, 4.8675E+04,
     |  4.9385E+04, 5.0625E+04, 5.2664E+04, 5.6069E+04, 6.0726E+04,
     |  6.6818E+04, 7.4760E+04, 8.4138E+04, 9.3639E+04, 1.0711E+05,
     |  1.2727E+05, 1.5445E+05, 1.8852E+05, 2.2262E+05, 2.5202E+05,
     |  2.5010E+05, 2.4940E+05, 1.9944E+05, 1.6430E+05, 1.4205E+05,
     |  1.8074E+05, 2.3450E+05, 3.2847E+05, 3.8338E+05, 4.2072E+05,
     |  4.9031E+05, 6.5267E+05, 8.3527E+05/)
      B150(:, 5) = (/
     |  2.4153E+04, 2.0241E+04, 1.3920E+04, 1.0427E+04, 7.8311E+03,
     |  5.9007E+03, 4.3259E+03, 2.5389E+03, 9.3945E+02,-6.8869E+02,
     | -2.2161E+03,-3.7530E+03,-5.5979E+03,-7.1718E+03,-7.7662E+03,
     | -7.9452E+03,-8.0331E+03,-8.0392E+03,-8.0590E+03,-8.1958E+03,
     | -8.7985E+03,-1.0239E+04,-1.2599E+04,-1.6266E+04,-2.1941E+04,
     | -2.9851E+04,-4.0290E+04,-5.3357E+04,-6.6689E+04,-7.8791E+04,
     | -8.6630E+04,-1.0188E+05,-1.2254E+05,-1.5321E+05,-1.4867E+05,
     | -1.1566E+05,-6.8623E+04,-3.5863E+04, 2.6834E+04, 3.3546E+04,
     |  9.4781E+04, 9.7905E+04, 1.7831E+05/)
      B150(:, 6) = (/
     | -1.8129E+05,-1.7101E+05,-1.6094E+05,-1.6701E+05,-1.7016E+05,
     | -1.7119E+05,-1.6734E+05,-1.5821E+05,-1.5318E+05,-1.5461E+05,
     | -1.5781E+05,-1.5930E+05,-1.5848E+05,-1.5613E+05,-1.5695E+05,
     | -1.6029E+05,-1.6495E+05,-1.7167E+05,-1.8189E+05,-1.9498E+05,
     | -2.1113E+05,-2.3088E+05,-2.5214E+05,-2.7080E+05,-2.9821E+05,
     | -3.4289E+05,-4.0704E+05,-4.9259E+05,-5.8856E+05,-6.8525E+05,
     | -7.4394E+05,-8.3102E+05,-9.0727E+05,-9.8704E+05,-1.0446E+06,
     | -1.1654E+06,-1.3079E+06,-1.7581E+06,-2.2692E+06,-2.5425E+06,
     | -2.5254E+06,-2.6110E+06,-3.1980E+06/)
      B150(:, 7) = (/
     | -7.3950E+02,-2.3974E+03,-4.9368E+03,-7.0614E+03,-8.4873E+03,
     | -9.3633E+03,-9.8269E+03,-1.0092E+04,-1.0680E+04,-1.1832E+04,
     | -1.3457E+04,-1.5161E+04,-1.6532E+04,-1.7638E+04,-1.8636E+04,
     | -1.9794E+04,-2.1113E+04,-2.2736E+04,-2.4936E+04,-2.7749E+04,
     | -3.1227E+04,-3.5388E+04,-3.9951E+04,-4.3922E+04,-4.9135E+04,
     | -5.6726E+04,-6.7143E+04,-8.0638E+04,-9.5576E+04,-1.1178E+05,
     | -1.2498E+05,-1.3861E+05,-1.3853E+05,-1.3830E+05,-1.4276E+05,
     | -1.6962E+05,-2.0312E+05,-2.6283E+05,-3.0076E+05,-3.1266E+05,
     | -3.4655E+05,-4.1475E+05,-4.4182E+05/)
      B150(:, 8) = (/
     |  1.8034E+04, 1.8323E+04, 1.6070E+04, 1.5189E+04, 1.4310E+04,
     |  1.3886E+04, 1.3665E+04, 1.3302E+04, 1.3769E+04, 1.4869E+04,
     |  1.6562E+04, 1.8095E+04, 1.9303E+04, 2.0516E+04, 2.2497E+04,
     |  2.5025E+04, 2.7578E+04, 3.0252E+04, 3.3115E+04, 3.5766E+04,
     |  3.7842E+04, 3.9280E+04, 3.9907E+04, 3.8275E+04, 3.6914E+04,
     |  3.7385E+04, 4.0052E+04, 4.5267E+04, 5.3393E+04, 6.7006E+04,
     |  8.6671E+04, 1.2319E+05, 1.6899E+05, 2.0933E+05, 2.2889E+05,
     |  2.6142E+05, 2.7163E+05, 3.9905E+05, 6.0663E+05, 7.7766E+05,
     |  7.4950E+05, 7.4781E+05, 9.3582E+05/)
      B150(:, 9) = (/
     |  1.7060E+04, 1.5738E+04, 1.4916E+04, 1.5516E+04, 1.5756E+04,
     |  1.5664E+04, 1.5016E+04, 1.3893E+04, 1.2975E+04, 1.2633E+04,
     |  1.2303E+04, 1.1925E+04, 1.1570E+04, 1.1213E+04, 1.1171E+04,
     |  1.1390E+04, 1.1918E+04, 1.2709E+04, 1.4017E+04, 1.5901E+04,
     |  1.8556E+04, 2.2036E+04, 2.5644E+04, 2.9581E+04, 3.4891E+04,
     |  4.2339E+04, 5.1508E+04, 6.2006E+04, 6.8931E+04, 6.7610E+04,
     |  4.9025E+04, 3.6928E+04, 4.2544E+04, 5.7412E+04, 5.6638E+04,
     |  4.0323E+04, 3.4030E+04, 5.2762E+04, 8.5341E+04, 1.3038E+05,
     |  1.0466E+05, 7.5526E+04, 2.6983E+04/)
!
      A360(:, 1) = (/
     |  1.7111E+01, 2.2908E+01, 2.7319E+01, 3.0783E+01, 3.5472E+01,
     |  9.1326E+00, 1.2883E+01, 1.7280E+01, 2.1699E+01, 2.5391E+01,
     |  2.8737E+01, 3.2285E+01, 3.6154E+01, 4.0169E+01, 4.4132E+01,
     |  4.8657E+01, 5.4171E+01, 6.0574E+01, 8.8449E+01, 1.3063E+02,
     |  1.9624E+02, 2.9254E+02, 4.2077E+02, 5.6574E+02, 6.8167E+02,
     |  7.5814E+02, 8.1945E+02, 8.8385E+02, 9.6448E+02, 1.0715E+03,
     |  1.2038E+03, 1.3612E+03, 1.5487E+03, 1.7301E+03, 1.8900E+03,
     |  2.0692E+03, 2.1760E+03, 2.5894E+03, 2.9949E+03, 3.0603E+03,
     |  2.7634E+03, 2.5643E+03, 2.3111E+03/)
      A360(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  4.8883E+01, 8.0588E+01, 1.3155E+02, 2.0536E+02, 2.9812E+02,
     |  4.0318E+02, 5.2377E+02, 6.6239E+02, 8.1877E+02, 9.9459E+02,
     |  1.1939E+03, 1.4199E+03, 1.6710E+03, 1.8763E+03, 2.0511E+03,
     |  2.1885E+03, 2.3000E+03, 2.4050E+03, 2.5475E+03, 2.7281E+03,
     |  2.9531E+03, 3.2233E+03, 3.5298E+03, 3.8826E+03, 4.2901E+03,
     |  4.7887E+03, 5.3828E+03, 6.1490E+03, 7.1153E+03, 8.2448E+03,
     |  9.6136E+03, 1.1041E+04, 1.3824E+04, 1.6743E+04, 1.7847E+04,
     |  1.6868E+04, 1.4901E+04, 1.6379E+04/)
      A360(:, 3) = (/
     |  8.3056E+01, 1.4182E+02, 2.3141E+02, 3.7215E+02, 5.9760E+02,
     |  8.6250E+02, 1.1538E+03, 1.4159E+03, 1.6358E+03, 1.8381E+03,
     |  2.0522E+03, 2.3292E+03, 2.6533E+03, 2.9509E+03, 3.1969E+03,
     |  3.3771E+03, 3.5341E+03, 3.6989E+03, 3.9280E+03, 4.2093E+03,
     |  4.5366E+03, 4.9059E+03, 5.3231E+03, 5.7811E+03, 6.2515E+03,
     |  6.7397E+03, 7.2807E+03, 7.9050E+03, 8.6494E+03, 9.4989E+03,
     |  1.0370E+04, 1.1248E+04, 1.2211E+04, 1.2856E+04, 1.2581E+04,
     |  1.0840E+04, 9.0377E+03, 8.7650E+03, 6.5477E+03, 3.9306E+03,
     |  3.4609E+03, 6.2837E+03, 1.1679E+04/)
      A360(:, 4) = (/
     |  2.5855E+03, 3.4310E+03, 4.3432E+03, 5.2496E+03, 6.1343E+03,
     |  7.0511E+03, 7.8740E+03, 8.5702E+03, 9.1933E+03, 9.7158E+03,
     |  1.0147E+04, 1.0541E+04, 1.0916E+04, 1.1314E+04, 1.1815E+04,
     |  1.2370E+04, 1.2996E+04, 1.3668E+04, 1.4427E+04, 1.5251E+04,
     |  1.6132E+04, 1.7119E+04, 1.8290E+04, 1.9753E+04, 2.1475E+04,
     |  2.3465E+04, 2.5739E+04, 2.8301E+04, 3.0407E+04, 3.1468E+04,
     |  2.8570E+04, 2.6092E+04, 2.2644E+04, 2.0070E+04, 1.8483E+04,
     |  2.3003E+04, 2.9335E+04, 3.9365E+04, 4.0752E+04, 4.3998E+04,
     |  5.1441E+04, 6.7507E+04, 7.9888E+04/)
      A360(:, 5) = (/
     |  4.5027E+03, 4.6059E+03, 4.7061E+03, 4.6999E+03, 4.5724E+03,
     |  4.2696E+03, 3.8769E+03, 3.5184E+03, 3.1760E+03, 2.8735E+03,
     |  2.6693E+03, 2.5438E+03, 2.3783E+03, 2.2541E+03, 2.1576E+03,
     |  2.0391E+03, 1.9211E+03, 1.8189E+03, 1.7944E+03, 1.8907E+03,
     |  2.1482E+03, 2.5686E+03, 3.1729E+03, 3.9230E+03, 4.7162E+03,
     |  5.5030E+03, 6.2595E+03, 6.9181E+03, 7.1935E+03, 6.9181E+03,
     |  5.1174E+03, 2.4969E+03,-8.7818E+02,-5.0162E+03,-5.1310E+03,
     |  1.6322E+03, 1.5432E+04, 2.8457E+04, 3.1402E+04, 3.2664E+04,
     |  3.8380E+04, 4.6750E+04, 4.0175E+04/)
      A360(:, 6) = (/
     | -1.3407E+04,-1.5740E+04,-1.8226E+04,-2.0552E+04,-2.2686E+04,
     | -2.4658E+04,-2.6402E+04,-2.7931E+04,-2.9310E+04,-3.0517E+04,
     | -3.1669E+04,-3.2977E+04,-3.4416E+04,-3.6034E+04,-3.7847E+04,
     | -3.9666E+04,-4.1595E+04,-4.3655E+04,-4.6000E+04,-4.8668E+04,
     | -5.1710E+04,-5.5269E+04,-5.9490E+04,-6.4681E+04,-7.0536E+04,
     | -7.7006E+04,-8.4192E+04,-9.2124E+04,-9.9832E+04,-1.0652E+05,
     | -1.0819E+05,-1.1022E+05,-1.1138E+05,-1.1378E+05,-1.1693E+05,
     | -1.3502E+05,-1.5673E+05,-2.0555E+05,-2.3253E+05,-2.4673E+05,
     | -2.4938E+05,-2.7271E+05,-2.9008E+05/)
      A360(:, 7) = (/
     |  9.4422E+02, 7.9373E+02, 5.5617E+02, 3.1025E+02,-9.3943E-02,
     | -3.0594E+02,-5.6194E+02,-8.2225E+02,-1.0873E+03,-1.3365E+03,
     | -1.5681E+03,-1.8641E+03,-2.1413E+03,-2.4453E+03,-2.7954E+03,
     | -3.1925E+03,-3.6226E+03,-4.0517E+03,-4.4724E+03,-4.8669E+03,
     | -5.2296E+03,-5.5589E+03,-5.8661E+03,-6.1730E+03,-6.4553E+03,
     | -6.7320E+03,-7.0626E+03,-7.4735E+03,-7.7309E+03,-7.6437E+03,
     | -6.4167E+03,-5.2361E+03,-3.8634E+03,-2.6394E+03,-2.2711E+03,
     | -3.7570E+03,-7.4804E+03,-9.6090E+03,-9.8386E+03,-1.0971E+04,
     | -1.5923E+04,-2.2419E+04,-2.9267E+04/)
      A360(:, 8) = (/
     |  2.8589E+03, 3.2297E+03, 3.4655E+03, 3.5427E+03, 3.5415E+03,
     |  3.5635E+03, 3.6568E+03, 3.8271E+03, 4.0626E+03, 4.3239E+03,
     |  4.6188E+03, 4.9639E+03, 5.3781E+03, 5.8815E+03, 6.4504E+03,
     |  7.0210E+03, 7.5925E+03, 8.1543E+03, 8.7616E+03, 9.4103E+03,
     |  1.0078E+04, 1.0806E+04, 1.1592E+04, 1.2439E+04, 1.3250E+04,
     |  1.4003E+04, 1.4725E+04, 1.5399E+04, 1.6192E+04, 1.7212E+04,
     |  1.9248E+04, 2.1694E+04, 2.4767E+04, 2.9051E+04, 3.1239E+04,
     |  3.8950E+04, 4.1112E+04, 5.8046E+04, 7.2890E+04, 8.1503E+04,
     |  7.6517E+04, 8.1330E+04, 8.6023E+04/)
      A360(:, 9) = (/
     |  5.4724E+02, 7.5433E+02, 9.8395E+02, 1.1995E+03, 1.3987E+03,
     |  1.5685E+03, 1.7015E+03, 1.7947E+03, 1.8442E+03, 1.8521E+03,
     |  1.8345E+03, 1.8229E+03, 1.8272E+03, 1.8573E+03, 1.9134E+03,
     |  1.9871E+03, 2.0833E+03, 2.1994E+03, 2.3413E+03, 2.5196E+03,
     |  2.7540E+03, 3.0453E+03, 3.4428E+03, 4.0784E+03, 5.0019E+03,
     |  6.2390E+03, 7.6783E+03, 9.1977E+03, 1.0655E+04, 1.1888E+04,
     |  1.2630E+04, 1.3204E+04, 1.3320E+04, 1.2734E+04, 1.1002E+04,
     |  8.0859E+03, 5.7454E+03, 6.1217E+03, 7.6533E+03, 8.9888E+03,
     |  6.6528E+03, 4.9529E+03, 1.7649E+03/)
!
      B360(:, 1) = (/
     |  3.3253E+03, 3.2351E+03, 3.4143E+03, 3.8189E+03, 4.1614E+03,
     |  2.0223E+03, 2.3376E+03, 2.5827E+03, 2.8409E+03, 3.1770E+03,
     |  3.5809E+03, 4.0187E+03, 4.4980E+03, 4.9533E+03, 5.3978E+03,
     |  5.8217E+03, 6.2216E+03, 6.5519E+03, 8.8577E+03, 1.2116E+04,
     |  1.6918E+04, 2.3499E+04, 3.1018E+04, 3.6721E+04, 4.0347E+04,
     |  4.3147E+04, 4.5804E+04, 4.8953E+04, 5.2255E+04, 5.5590E+04,
     |  5.9005E+04, 6.3282E+04, 6.8584E+04, 7.4935E+04, 8.2070E+04,
     |  9.0657E+04, 9.7564E+04, 1.1300E+05, 1.3001E+05, 1.3940E+05,
     |  1.3700E+05, 1.2925E+05, 1.1594E+05/)
      B360(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  3.6547E+03, 4.9471E+03, 6.5468E+03, 8.7473E+03, 1.1660E+04,
     |  1.4753E+04, 1.7468E+04, 1.9883E+04, 2.1854E+04, 2.3722E+04,
     |  2.5604E+04, 2.7643E+04, 2.9766E+04, 2.6042E+04, 2.1469E+04,
     |  1.6596E+04, 1.2023E+04, 8.2278E+03, 5.6666E+03, 4.5079E+03,
     |  4.5944E+03, 6.1822E+03, 9.7297E+03, 1.5767E+04, 2.5136E+04,
     |  3.8115E+04, 5.7153E+04, 8.2712E+04, 1.1797E+05, 1.6340E+05,
     |  2.1983E+05, 2.8663E+05, 3.9899E+05, 5.2486E+05, 6.0382E+05,
     |  6.0803E+05, 5.0771E+05, 5.0653E+05/)
      B360(:, 3) = (/
     |  7.3641E+03, 8.9308E+03, 1.2220E+04, 1.8039E+04, 2.5046E+04,
     |  2.7659E+04, 2.6748E+04, 2.3744E+04, 2.1170E+04, 2.0298E+04,
     |  2.0861E+04, 2.2092E+04, 2.3916E+04, 2.5055E+04, 2.5321E+04,
     |  2.5212E+04, 2.4964E+04, 2.5021E+04, 2.7715E+04, 3.1218E+04,
     |  3.5622E+04, 4.0691E+04, 4.6128E+04, 5.0962E+04, 5.6891E+04,
     |  6.5637E+04, 7.8483E+04, 9.6834E+04, 1.2138E+05, 1.5344E+05,
     |  1.9255E+05, 2.4365E+05, 3.0300E+05, 3.6309E+05, 3.9755E+05,
     |  3.7686E+05, 3.4017E+05, 3.3795E+05, 2.6815E+05, 1.6870E+05,
     |  1.3898E+05, 2.1755E+05, 3.5367E+05/)
      B360(:, 4) = (/
     |  7.6881E+04, 7.6985E+04, 7.7525E+04, 7.7801E+04, 7.5543E+04,
     |  7.4391E+04, 7.3315E+04, 7.1147E+04, 6.9972E+04, 7.1729E+04,
     |  7.5266E+04, 7.7973E+04, 8.0231E+04, 8.0518E+04, 7.9686E+04,
     |  8.0755E+04, 8.2698E+04, 8.6342E+04, 9.1263E+04, 9.7914E+04,
     |  1.0750E+05, 1.2033E+05, 1.3581E+05, 1.5161E+05, 1.7254E+05,
     |  2.0208E+05, 2.4209E+05, 2.9452E+05, 3.4909E+05, 3.9843E+05,
     |  3.9550E+05, 3.9739E+05, 3.6570E+05, 3.3721E+05, 3.0167E+05,
     |  3.5112E+05, 4.1334E+05, 5.3471E+05, 5.4849E+05, 6.3536E+05,
     |  8.3429E+05, 1.1018E+06, 1.2254E+06/)
      B360(:, 5) = (/
     |  3.6490E+04, 3.5829E+04, 2.9970E+04, 2.1659E+04, 1.6587E+04,
     |  1.3449E+04, 1.1575E+04, 9.2354E+03, 6.3346E+03, 3.4963E+03,
     |  1.1059E+03,-1.3367E+03,-4.0742E+03,-6.6799E+03,-8.9239E+03,
     | -9.5016E+03,-9.0282E+03,-7.6910E+03,-5.6447E+03,-3.2713E+03,
     | -5.1923E+02, 2.3747E+03, 5.0914E+03, 6.0806E+03, 4.4006E+03,
     | -1.4351E+03,-1.2292E+04,-2.8869E+04,-5.1910E+04,-8.2160E+04,
     | -1.2700E+05,-1.8409E+05,-2.4814E+05,-3.2813E+05,-3.5233E+05,
     | -3.2063E+05,-1.9923E+05,-1.3794E+05,-1.1518E+05,-5.2386E+04,
     |  1.5581E+05, 2.2252E+05, 1.0956E+05/)
      B360(:, 6) = (/
     | -2.4105E+05,-2.4228E+05,-2.4221E+05,-2.4252E+05,-2.4380E+05,
     | -2.4497E+05,-2.4288E+05,-2.3382E+05,-2.2646E+05,-2.2840E+05,
     | -2.3729E+05,-2.4566E+05,-2.5521E+05,-2.5943E+05,-2.5954E+05,
     | -2.6611E+05,-2.7609E+05,-2.9154E+05,-3.1026E+05,-3.3353E+05,
     | -3.6513E+05,-4.0494E+05,-4.4907E+05,-4.8821E+05,-5.3623E+05,
     | -6.0196E+05,-6.9086E+05,-8.0987E+05,-9.4838E+05,-1.1036E+06,
     | -1.2140E+06,-1.3616E+06,-1.4783E+06,-1.6029E+06,-1.7072E+06,
     | -1.9842E+06,-2.2941E+06,-3.0072E+06,-3.4794E+06,-3.9077E+06,
     | -4.3693E+06,-4.6486E+06,-4.6402E+06/)
      B360(:, 7) = (/
     |  4.5539E+03, 2.8594E+03,-8.9474E+02,-4.7691E+03,-7.3681E+03,
     | -9.0663E+03,-1.0362E+04,-1.1456E+04,-1.2748E+04,-1.4823E+04,
     | -1.7837E+04,-2.1336E+04,-2.4740E+04,-2.7550E+04,-2.9813E+04,
     | -3.1850E+04,-3.3650E+04,-3.5707E+04,-3.8081E+04,-4.1177E+04,
     | -4.5426E+04,-5.0820E+04,-5.7149E+04,-6.2948E+04,-7.0083E+04,
     | -8.0121E+04,-9.4008E+04,-1.1274E+05,-1.3342E+05,-1.5455E+05,
     | -1.6383E+05,-1.7815E+05,-1.8813E+05,-2.0384E+05,-2.2432E+05,
     | -2.8298E+05,-3.5716E+05,-4.5280E+05,-4.8361E+05,-4.8895E+05,
     | -5.2083E+05,-6.3321E+05,-6.8491E+05/)
      B360(:, 8) = (/
     |  1.9832E+04, 2.5710E+04, 2.5160E+04, 2.1026E+04, 1.8443E+04,
     |  1.7597E+04, 1.8251E+04, 1.8910E+04, 1.9799E+04, 2.1595E+04,
     |  2.4710E+04, 2.7984E+04, 3.1861E+04, 3.5273E+04, 3.8233E+04,
     |  4.2622E+04, 4.7500E+04, 5.3095E+04, 5.8972E+04, 6.4963E+04,
     |  7.1570E+04, 7.8743E+04, 8.5650E+04, 8.8321E+04, 8.9429E+04,
     |  9.0302E+04, 9.2023E+04, 9.5354E+04, 1.0201E+05, 1.1466E+05,
     |  1.3444E+05, 1.6984E+05, 2.1204E+05, 2.6621E+05, 2.9638E+05,
     |  3.9368E+05, 4.3031E+05, 6.2812E+05, 8.2933E+05, 1.0119E+06,
     |  1.0957E+06, 1.1404E+06, 1.1434E+06/)
      B360(:, 9) = (/
     |  2.4329E+04, 2.2442E+04, 2.2327E+04, 2.3431E+04, 2.4019E+04,
     |  2.4035E+04, 2.3133E+04, 2.1482E+04, 2.0047E+04, 1.9383E+04,
     |  1.8964E+04, 1.8407E+04, 1.7806E+04, 1.6943E+04, 1.5983E+04,
     |  1.5465E+04, 1.5526E+04, 1.6182E+04, 1.7413E+04, 1.9366E+04,
     |  2.2532E+04, 2.7037E+04, 3.2393E+04, 3.9596E+04, 5.0070E+04,
     |  6.4637E+04, 8.2657E+04, 1.0414E+05, 1.2777E+05, 1.5292E+05,
     |  1.7726E+05, 2.0210E+05, 2.1699E+05, 2.2092E+05, 1.9769E+05,
     |  1.4922E+05, 1.0556E+05, 1.0813E+05, 1.2682E+05, 1.4073E+05,
     |  1.0328E+05, 7.8845E+04, 2.4000E+04/)
!
      A540(:, 1) = (/
     |  1.4060E+01, 1.8377E+01, 2.2830E+01, 2.6759E+01, 3.1289E+01,
     |  6.4033E+00, 9.6287E+00, 1.3553E+01, 1.7715E+01, 2.1440E+01,
     |  2.4849E+01, 2.8293E+01, 3.2218E+01, 3.6136E+01, 4.0265E+01,
     |  4.4909E+01, 5.0037E+01, 5.6386E+01, 8.3103E+01, 1.2390E+02,
     |  1.8905E+02, 2.8398E+02, 4.1272E+02, 5.5917E+02, 6.7738E+02,
     |  7.6094E+02, 8.2150E+02, 8.9854E+02, 9.9130E+02, 1.1111E+03,
     |  1.2665E+03, 1.4432E+03, 1.6549E+03, 1.8638E+03, 2.0574E+03,
     |  2.2699E+03, 2.4627E+03, 2.9118E+03, 3.3417E+03, 3.4941E+03,
     |  3.2281E+03, 3.1118E+03, 2.8235E+03/)
      A540(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  4.6827E+01, 7.8898E+01, 1.3079E+02, 2.0671E+02, 3.0064E+02,
     |  4.0573E+02, 5.2148E+02, 6.6543E+02, 8.3140E+02, 1.0300E+03,
     |  1.2605E+03, 1.5231E+03, 1.8219E+03, 2.0790E+03, 2.3040E+03,
     |  2.4866E+03, 2.6278E+03, 2.7609E+03, 2.9309E+03, 3.1386E+03,
     |  3.3935E+03, 3.6896E+03, 4.0424E+03, 4.4524E+03, 4.9345E+03,
     |  5.5124E+03, 6.2272E+03, 7.0886E+03, 8.1885E+03, 9.5203E+03,
     |  1.1202E+04, 1.3225E+04, 1.6403E+04, 1.9805E+04, 2.1727E+04,
     |  2.1445E+04, 1.9728E+04, 2.2047E+04/)
      A540(:, 3) = (/
     |  8.4204E+01, 1.3804E+02, 2.2646E+02, 3.6600E+02, 5.8693E+02,
     |  8.5350E+02, 1.1551E+03, 1.4395E+03, 1.6862E+03, 1.9206E+03,
     |  2.1803E+03, 2.5291E+03, 2.9571E+03, 3.3611E+03, 3.7030E+03,
     |  3.9632E+03, 4.1776E+03, 4.3862E+03, 4.6427E+03, 4.9506E+03,
     |  5.3127E+03, 5.7306E+03, 6.2068E+03, 6.7334E+03, 7.2792E+03,
     |  7.8571E+03, 8.4908E+03, 9.2302E+03, 1.0105E+04, 1.1120E+04,
     |  1.2160E+04, 1.3333E+04, 1.4486E+04, 1.5469E+04, 1.5457E+04,
     |  1.3909E+04, 1.1952E+04, 1.2072E+04, 9.7922E+03, 6.5839E+03,
     |  5.2999E+03, 8.0009E+03, 1.3555E+04/)
      A540(:, 4) = (/
     |  2.7027E+03, 3.5499E+03, 4.5279E+03, 5.5560E+03, 6.6148E+03,
     |  7.7808E+03, 8.8812E+03, 9.8123E+03, 1.0642E+04, 1.1364E+04,
     |  1.1983E+04, 1.2543E+04, 1.3075E+04, 1.3577E+04, 1.4175E+04,
     |  1.4836E+04, 1.5564E+04, 1.6359E+04, 1.7254E+04, 1.8258E+04,
     |  1.9354E+04, 2.0565E+04, 2.1963E+04, 2.3664E+04, 2.5648E+04,
     |  2.7951E+04, 3.0564E+04, 3.3586E+04, 3.6277E+04, 3.8117E+04,
     |  3.5834E+04, 3.4238E+04, 3.0854E+04, 2.8030E+04, 2.5885E+04,
     |  3.0631E+04, 3.7125E+04, 4.6215E+04, 4.8781E+04, 5.3596E+04,
     |  6.4105E+04, 8.4928E+04, 1.0191E+05/)
      A540(:, 5) = (/
     |  4.9905E+03, 5.2485E+03, 5.5611E+03, 5.7942E+03, 5.8580E+03,
     |  5.6466E+03, 5.2343E+03, 4.8044E+03, 4.3795E+03, 3.9986E+03,
     |  3.7287E+03, 3.5341E+03, 3.2601E+03, 3.0506E+03, 2.9136E+03,
     |  2.8043E+03, 2.7216E+03, 2.6568E+03, 2.6678E+03, 2.8034E+03,
     |  3.1104E+03, 3.6013E+03, 4.3172E+03, 5.2284E+03, 6.2259E+03,
     |  7.2728E+03, 8.3709E+03, 9.4369E+03, 1.0168E+04, 1.0252E+04,
     |  8.7129E+03, 5.6958E+03, 1.5757E+03,-5.2061E+03,-5.7114E+03,
     |  1.1017E+03, 1.5068E+04, 2.5398E+04, 2.9058E+04, 3.1920E+04,
     |  4.1364E+04, 5.5218E+04, 5.5284E+04/)
      A540(:, 6) = (/
     | -1.4442E+04,-1.7032E+04,-2.0050E+04,-2.3102E+04,-2.5973E+04,
     | -2.8669E+04,-3.1033E+04,-3.3003E+04,-3.4753E+04,-3.6327E+04,
     | -3.7841E+04,-3.9468E+04,-4.1248E+04,-4.3163E+04,-4.5347E+04,
     | -4.7646E+04,-5.0067E+04,-5.2658E+04,-5.5501E+04,-5.8727E+04,
     | -6.2409E+04,-6.6650E+04,-7.1636E+04,-7.7709E+04,-8.4582E+04,
     | -9.2310E+04,-1.0096E+05,-1.1083E+05,-1.2092E+05,-1.3040E+05,
     | -1.3464E+05,-1.3936E+05,-1.4073E+05,-1.4197E+05,-1.4496E+05,
     | -1.6644E+05,-1.9061E+05,-2.4125E+05,-2.7072E+05,-2.9018E+05,
     | -3.0120E+05,-3.3864E+05,-3.7593E+05/)
      A540(:, 7) = (/
     |  1.1944E+03, 1.1093E+03, 9.0601E+02, 6.7608E+02, 3.3693E+02,
     | -3.2394E+01,-3.5412E+02,-6.8678E+02,-1.0293E+03,-1.3568E+03,
     | -1.6693E+03,-2.0713E+03,-2.4396E+03,-2.8083E+03,-3.2111E+03,
     | -3.6657E+03,-4.1709E+03,-4.6930E+03,-5.2101E+03,-5.7009E+03,
     | -6.1553E+03,-6.5565E+03,-6.9176E+03,-7.2589E+03,-7.5576E+03,
     | -7.8459E+03,-8.1785E+03,-8.6409E+03,-9.0101E+03,-9.1184E+03,
     | -8.0997E+03,-7.2301E+03,-6.0081E+03,-4.5439E+03,-4.1776E+03,
     | -5.5456E+03,-9.1839E+03,-1.0031E+04,-1.0233E+04,-1.1182E+04,
     | -1.5779E+04,-2.2527E+04,-2.9600E+04/)
      A540(:, 8) = (/
     |  3.1142E+03, 3.6452E+03, 4.0618E+03, 4.2710E+03, 4.3178E+03,
     |  4.3535E+03, 4.4749E+03, 4.6972E+03, 5.0048E+03, 5.3436E+03,
     |  5.7159E+03, 6.1206E+03, 6.6096E+03, 7.2024E+03, 7.8965E+03,
     |  8.6263E+03, 9.3612E+03, 1.0075E+04, 1.0809E+04, 1.1577E+04,
     |  1.2369E+04, 1.3213E+04, 1.4131E+04, 1.5151E+04, 1.6207E+04,
     |  1.7305E+04, 1.8425E+04, 1.9566E+04, 2.0774E+04, 2.2059E+04,
     |  2.4037E+04, 2.6967E+04, 2.9443E+04, 3.3830E+04, 3.5710E+04,
     |  4.5225E+04, 4.8917E+04, 6.8778E+04, 8.3245E+04, 9.3457E+04,
     |  8.9674E+04, 9.7044E+04, 1.0812E+05/)
      A540(:, 9) = (/
     |  5.2344E+02, 7.1552E+02, 9.6670E+02, 1.2352E+03, 1.5074E+03,
     |  1.7551E+03, 1.9484E+03, 2.0782E+03, 2.1498E+03, 2.1696E+03,
     |  2.1524E+03, 2.1321E+03, 2.1184E+03, 2.1300E+03, 2.1781E+03,
     |  2.2568E+03, 2.3629E+03, 2.4986E+03, 2.6617E+03, 2.8675E+03,
     |  3.1337E+03, 3.4532E+03, 3.8718E+03, 4.5151E+03, 5.4501E+03,
     |  6.7467E+03, 8.3333E+03, 1.0207E+04, 1.2236E+04, 1.4228E+04,
     |  1.5884E+04, 1.6977E+04, 1.7767E+04, 1.7635E+04, 1.5995E+04,
     |  1.3261E+04, 9.7717E+03, 1.0121E+04, 1.0732E+04, 1.0119E+04,
     |  6.1920E+03, 4.1260E+03, 1.0669E+03/)
!
      B540(:, 1) = (/
     |  2.9738E+03, 3.1230E+03, 3.4557E+03, 3.9327E+03, 4.3466E+03,
     |  2.0002E+03, 2.3635E+03, 2.6734E+03, 2.9926E+03, 3.3620E+03,
     |  3.7894E+03, 4.2515E+03, 4.7534E+03, 5.2430E+03, 5.7030E+03,
     |  6.1469E+03, 6.5915E+03, 6.9727E+03, 9.5116E+03, 1.3033E+04,
     |  1.8170E+04, 2.5310E+04, 3.3635E+04, 3.9987E+04, 4.4115E+04,
     |  4.7267E+04, 5.0260E+04, 5.3453E+04, 5.6809E+04, 6.0214E+04,
     |  6.3841E+04, 6.7921E+04, 7.3205E+04, 7.9458E+04, 8.6700E+04,
     |  9.5908E+04, 1.0490E+05, 1.2181E+05, 1.3944E+05, 1.5088E+05,
     |  1.5082E+05, 1.4554E+05, 1.3311E+05/)
      B540(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  4.0525E+03, 5.6007E+03, 7.5433E+03, 1.0196E+04, 1.3633E+04,
     |  1.7346E+04, 2.0517E+04, 2.3440E+04, 2.5810E+04, 2.8059E+04,
     |  3.0373E+04, 3.3069E+04, 3.5940E+04, 3.2665E+04, 2.8267E+04,
     |  2.3391E+04, 1.8720E+04, 1.4752E+04, 1.1962E+04, 1.0698E+04,
     |  1.0742E+04, 1.2320E+04, 1.6019E+04, 2.2181E+04, 3.1619E+04,
     |  4.4808E+04, 6.2243E+04, 8.9138E+04, 1.2657E+05, 1.7794E+05,
     |  2.4649E+05, 3.3171E+05, 4.5944E+05, 6.0163E+05, 7.0765E+05,
     |  7.5237E+05, 6.6097E+05, 7.0996E+05/)
      B540(:, 3) = (/
     |  7.2703E+03, 9.6534E+03, 1.3767E+04, 2.0580E+04, 2.8795E+04,
     |  3.1953E+04, 3.1035E+04, 2.7741E+04, 2.4968E+04, 2.4052E+04,
     |  2.4729E+04, 2.6510E+04, 2.8837E+04, 3.0774E+04, 3.1620E+04,
     |  3.1880E+04, 3.1930E+04, 3.2202E+04, 3.5520E+04, 3.9658E+04,
     |  4.4578E+04, 5.0232E+04, 5.6241E+04, 6.1765E+04, 6.8459E+04,
     |  7.7892E+04, 9.1512E+04, 1.1039E+05, 1.3574E+05, 1.6925E+05,
     |  2.1070E+05, 2.6297E+05, 3.3090E+05, 4.0671E+05, 4.6359E+05,
     |  4.6878E+05, 4.4317E+05, 4.6611E+05, 3.9923E+05, 2.8063E+05,
     |  2.1748E+05, 2.7723E+05, 3.9590E+05/)
      B540(:, 4) = (/
     |  8.2482E+04, 8.7240E+04, 9.1673E+04, 9.1647E+04, 8.7298E+04,
     |  8.4060E+04, 8.2437E+04, 8.1726E+04, 8.2259E+04, 8.5262E+04,
     |  8.9836E+04, 9.4143E+04, 9.7896E+04, 1.0003E+05, 1.0027E+05,
     |  1.0155E+05, 1.0401E+05, 1.0820E+05, 1.1452E+05, 1.2269E+05,
     |  1.3378E+05, 1.4886E+05, 1.6739E+05, 1.8747E+05, 2.1401E+05,
     |  2.5024E+05, 2.9871E+05, 3.6078E+05, 4.2638E+05, 4.8823E+05,
     |  4.9800E+05, 5.1245E+05, 4.9986E+05, 4.8930E+05, 4.5892E+05,
     |  5.2317E+05, 5.8655E+05, 6.9944E+05, 7.1903E+05, 8.2941E+05,
     |  1.0919E+06, 1.3627E+06, 1.5504E+06/)
      B540(:, 5) = (/
     |  4.6823E+04, 4.5569E+04, 4.2010E+04, 3.0547E+04, 2.1468E+04,
     |  1.6198E+04, 1.3925E+04, 1.1965E+04, 9.0799E+03, 5.7358E+03,
     |  2.8374E+03,-1.9532E+02,-3.3667E+03,-6.4178E+03,-9.2567E+03,
     | -1.0767E+04,-1.0609E+04,-8.9146E+03,-5.9077E+03,-1.9534E+03,
     |  2.9295E+03, 8.6203E+03, 1.4929E+04, 2.0140E+04, 2.3265E+04,
     |  2.2431E+04, 1.5720E+04, 1.6511E+03,-2.1599E+04,-5.4160E+04,
     | -1.0031E+05,-1.7644E+05,-2.5556E+05,-3.8124E+05,-4.2900E+05,
     | -4.1689E+05,-3.0154E+05,-2.8942E+05,-2.8233E+05,-2.2621E+05,
     |  7.5296E+03, 6.8124E+04, 7.6836E+04/)
      B540(:, 6) = (/
     | -2.6922E+05,-2.8105E+05,-2.9550E+05,-2.9208E+05,-2.8448E+05,
     | -2.7904E+05,-2.7605E+05,-2.7155E+05,-2.6880E+05,-2.7334E+05,
     | -2.8458E+05,-2.9698E+05,-3.1099E+05,-3.2107E+05,-3.2458E+05,
     | -3.3169E+05,-3.4444E+05,-3.6384E+05,-3.9000E+05,-4.2141E+05,
     | -4.6135E+05,-5.1238E+05,-5.7055E+05,-6.2684E+05,-6.9598E+05,
     | -7.8520E+05,-9.0000E+05,-1.0442E+06,-1.2077E+06,-1.3907E+06,
     | -1.5426E+06,-1.6899E+06,-1.8613E+06,-2.0222E+06,-2.1824E+06,
     | -2.5829E+06,-2.9617E+06,-3.7503E+06,-4.2341E+06,-4.7115E+06,
     | -5.3122E+06,-5.5044E+06,-5.8396E+06/)
      B540(:, 7) = (/
     |  8.8823E+03, 6.5320E+03, 2.4694E+03,-2.5667E+03,-6.3482E+03,
     | -8.6211E+03,-1.0284E+04,-1.1904E+04,-1.3759E+04,-1.6334E+04,
     | -2.0168E+04,-2.4827E+04,-2.9361E+04,-3.3384E+04,-3.6684E+04,
     | -3.9542E+04,-4.1936E+04,-4.4292E+04,-4.7090E+04,-5.0475E+04,
     | -5.4855E+04,-6.0514E+04,-6.7288E+04,-7.3846E+04,-8.2139E+04,
     | -9.3712E+04,-1.0983E+05,-1.3122E+05,-1.5519E+05,-1.8023E+05,
     | -1.9432E+05,-2.1571E+05,-2.3445E+05,-2.5613E+05,-2.8878E+05,
     | -3.6778E+05,-4.6214E+05,-5.6349E+05,-6.0593E+05,-6.2428E+05,
     | -6.5451E+05,-7.6108E+05,-8.0827E+05/)
      B540(:, 8) = (/
     |  2.2658E+04, 3.0562E+04, 3.2836E+04, 2.6463E+04, 2.1014E+04,
     |  1.8717E+04, 1.9288E+04, 2.0903E+04, 2.2864E+04, 2.5172E+04,
     |  2.9233E+04, 3.3519E+04, 3.8825E+04, 4.3872E+04, 4.8025E+04,
     |  5.3025E+04, 5.8992E+04, 6.5957E+04, 7.3913E+04, 8.2188E+04,
     |  9.1167E+04, 1.0140E+05, 1.1219E+05, 1.1936E+05, 1.2567E+05,
     |  1.3164E+05, 1.3731E+05, 1.4282E+05, 1.5010E+05, 1.6296E+05,
     |  1.8619E+05, 2.1442E+05, 2.5969E+05, 3.2120E+05, 3.5213E+05,
     |  4.8173E+05, 5.3546E+05, 7.7528E+05, 9.6918E+05, 1.1723E+06,
     |  1.2725E+06, 1.2650E+06, 1.3701E+06/)
      B540(:, 9) = (/
     |  2.7005E+04, 2.5605E+04, 2.6433E+04, 2.7976E+04, 2.8604E+04,
     |  2.8404E+04, 2.7431E+04, 2.5929E+04, 2.4611E+04, 2.4055E+04,
     |  2.3542E+04, 2.3026E+04, 2.2227E+04, 2.1209E+04, 1.9903E+04,
     |  1.8825E+04, 1.8367E+04, 1.8635E+04, 1.9693E+04, 2.1613E+04,
     |  2.4805E+04, 2.9438E+04, 3.5067E+04, 4.3085E+04, 5.5473E+04,
     |  7.3652E+04, 9.7257E+04, 1.2589E+05, 1.5761E+05, 1.9149E+05,
     |  2.2637E+05, 2.5862E+05, 2.8879E+05, 3.0929E+05, 2.9778E+05,
     |  2.5922E+05, 1.9398E+05, 1.9633E+05, 1.9708E+05, 1.8196E+05,
     |  1.1256E+05, 7.2751E+04, 1.8345E+04/)
!
      A720(:, 1) = (/
     |  1.1010E+01, 1.5345E+01, 1.9840E+01, 2.4316E+01, 2.8812E+01,
     |  4.8874E+00, 7.5931E+00, 1.1095E+01, 1.5052E+01, 1.8840E+01,
     |  2.2382E+01, 2.5957E+01, 2.9751E+01, 3.3663E+01, 3.7756E+01,
     |  4.2314E+01, 4.7502E+01, 5.3642E+01, 7.9167E+01, 1.1946E+02,
     |  1.8388E+02, 2.7905E+02, 4.0918E+02, 5.5656E+02, 6.7749E+02,
     |  7.6418E+02, 8.3204E+02, 9.1295E+02, 1.0179E+03, 1.1486E+03,
     |  1.3138E+03, 1.5057E+03, 1.7323E+03, 1.9578E+03, 2.1819E+03,
     |  2.4405E+03, 2.7131E+03, 3.1846E+03, 3.6235E+03, 3.8274E+03,
     |  3.5841E+03, 3.5834E+03, 3.3267E+03/)
      A720(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  4.4830E+01, 7.6651E+01, 1.2919E+02, 2.0679E+02, 3.0255E+02,
     |  4.0784E+02, 5.2110E+02, 6.6433E+02, 8.3823E+02, 1.0514E+03,
     |  1.3042E+03, 1.5972E+03, 1.9308E+03, 2.2283E+03, 2.4954E+03,
     |  2.7142E+03, 2.8864E+03, 3.0448E+03, 3.2374E+03, 3.4641E+03,
     |  3.7351E+03, 4.0549E+03, 4.4294E+03, 4.8794E+03, 5.4150E+03,
     |  6.0570E+03, 6.8522E+03, 7.7974E+03, 8.9997E+03, 1.0491E+04,
     |  1.2401E+04, 1.4804E+04, 1.8307E+04, 2.2124E+04, 2.4661E+04,
     |  2.5089E+04, 2.3844E+04, 2.6192E+04/)
      A720(:, 3) = (/
     |  8.1644E+01, 1.3676E+02, 2.2535E+02, 3.6522E+02, 5.8162E+02,
     |  8.4529E+02, 1.1497E+03, 1.4469E+03, 1.7089E+03, 1.9637E+03,
     |  2.2533E+03, 2.6546E+03, 3.1652E+03, 3.6613E+03, 4.0885E+03,
     |  4.4216E+03, 4.6934E+03, 4.9417E+03, 5.2269E+03, 5.5573E+03,
     |  5.9432E+03, 6.3940E+03, 6.9158E+03, 7.4980E+03, 8.1047E+03,
     |  8.7527E+03, 9.4666E+03, 1.0289E+04, 1.1273E+04, 1.2420E+04,
     |  1.3592E+04, 1.4948E+04, 1.6299E+04, 1.7481E+04, 1.7775E+04,
     |  1.6430E+04, 1.4373E+04, 1.4833E+04, 1.2599E+04, 9.0573E+03,
     |  7.5061E+03, 9.9093E+03, 1.5671E+04/)
      A720(:, 4) = (/
     |  2.8070E+03, 3.6943E+03, 4.6995E+03, 5.7657E+03, 6.9114E+03,
     |  8.2511E+03, 9.5619E+03, 1.0703E+04, 1.1711E+04, 1.2603E+04,
     |  1.3377E+04, 1.4107E+04, 1.4803E+04, 1.5429E+04, 1.6124E+04,
     |  1.6889E+04, 1.7700E+04, 1.8569E+04, 1.9564E+04, 2.0699E+04,
     |  2.1964E+04, 2.3373E+04, 2.4986E+04, 2.6906E+04, 2.9117E+04,
     |  3.1669E+04, 3.4554E+04, 3.7861E+04, 4.0831E+04, 4.3080E+04,
     |  4.0982E+04, 4.0555E+04, 3.7579E+04, 3.4978E+04, 3.2908E+04,
     |  3.8516E+04, 4.5489E+04, 5.4162E+04, 5.7294E+04, 6.3155E+04,
     |  7.5432E+04, 9.9843E+04, 1.2020E+05/)
      A720(:, 5) = (/
     |  5.3949E+03, 5.7541E+03, 6.2012E+03, 6.5960E+03, 6.8277E+03,
     |  6.7534E+03, 6.3715E+03, 5.9176E+03, 5.4368E+03, 4.9982E+03,
     |  4.6718E+03, 4.4263E+03, 4.0629E+03, 3.7723E+03, 3.5826E+03,
     |  3.4687E+03, 3.4074E+03, 3.3790E+03, 3.4343E+03, 3.6179E+03,
     |  3.9794E+03, 4.5329E+03, 5.3347E+03, 6.3593E+03, 7.4940E+03,
     |  8.7109E+03, 1.0045E+04, 1.1444E+04, 1.2417E+04, 1.2576E+04,
     |  1.0400E+04, 7.2300E+03, 2.8743E+03,-4.4267E+03,-6.1044E+03,
     |  7.9113E+02, 1.4910E+04, 2.4978E+04, 2.9127E+04, 3.4744E+04,
     |  4.4067E+04, 6.1302E+04, 6.3801E+04/)
      A720(:, 6) = (/
     | -1.5300E+04,-1.8141E+04,-2.1462E+04,-2.4921E+04,-2.8302E+04,
     | -3.1635E+04,-3.4546E+04,-3.6978E+04,-3.9058E+04,-4.0951E+04,
     | -4.2744E+04,-4.4703E+04,-4.6824E+04,-4.9050E+04,-5.1542E+04,
     | -5.4253E+04,-5.7080E+04,-6.0059E+04,-6.3324E+04,-6.7014E+04,
     | -7.1223E+04,-7.6070E+04,-8.1725E+04,-8.8536E+04,-9.6204E+04,
     | -1.0484E+05,-1.1460E+05,-1.2583E+05,-1.3746E+05,-1.4874E+05,
     | -1.5412E+05,-1.6177E+05,-1.6517E+05,-1.6749E+05,-1.7092E+05,
     | -1.9500E+05,-2.2069E+05,-2.7296E+05,-3.0460E+05,-3.2815E+05,
     | -3.4394E+05,-3.9156E+05,-4.3894E+05/)
      A720(:, 7) = (/
     |  1.3904E+03, 1.3591E+03, 1.1947E+03, 9.9091E+02, 6.4209E+02,
     |  2.3646E+02,-1.2947E+02,-5.1511E+02,-9.2071E+02,-1.3149E+03,
     | -1.6985E+03,-2.1984E+03,-2.6526E+03,-3.0906E+03,-3.5451E+03,
     | -4.0467E+03,-4.6057E+03,-5.1925E+03,-5.7858E+03,-6.3573E+03,
     | -6.8919E+03,-7.3645E+03,-7.7840E+03,-8.1646E+03,-8.4800E+03,
     | -8.7686E+03,-9.0917E+03,-9.5384E+03,-9.9044E+03,-1.0060E+04,
     | -8.9692E+03,-8.3858E+03,-7.2803E+03,-5.9857E+03,-5.6654E+03,
     | -7.5243E+03,-1.1462E+04,-1.2429E+04,-1.2758E+04,-1.4144E+04,
     | -1.7750E+04,-2.4147E+04,-3.0635E+04/)
      A720(:, 8) = (/
     |  3.3083E+03, 3.9544E+03, 4.5085E+03, 4.8171E+03, 4.9109E+03,
     |  4.9712E+03, 5.1155E+03, 5.3859E+03, 5.7636E+03, 6.1785E+03,
     |  6.6183E+03, 7.0852E+03, 7.6406E+03, 8.3076E+03, 9.0955E+03,
     |  9.9591E+03, 1.0828E+04, 1.1668E+04, 1.2522E+04, 1.3399E+04,
     |  1.4298E+04, 1.5251E+04, 1.6275E+04, 1.7403E+04, 1.8602E+04,
     |  1.9909E+04, 2.1347E+04, 2.2895E+04, 2.4603E+04, 2.6367E+04,
     |  2.8776E+04, 3.2026E+04, 3.4860E+04, 3.9133E+04, 4.1022E+04,
     |  5.0741E+04, 5.5167E+04, 7.5802E+04, 9.0959E+04, 1.0104E+05,
     |  9.8925E+04, 1.0823E+05, 1.2134E+05/)
      A720(:, 9) = (/
     |  5.1024E+02, 6.9177E+02, 9.4484E+02, 1.2403E+03, 1.5610E+03,
     |  1.8761E+03, 2.1269E+03, 2.3001E+03, 2.3915E+03, 2.4226E+03,
     |  2.4061E+03, 2.3843E+03, 2.3603E+03, 2.3572E+03, 2.3926E+03,
     |  2.4690E+03, 2.5806E+03, 2.7252E+03, 2.9022E+03, 3.1294E+03,
     |  3.4237E+03, 3.7771E+03, 4.2295E+03, 4.8997E+03, 5.8476E+03,
     |  7.1537E+03, 8.7921E+03, 1.0785E+04, 1.3099E+04, 1.5558E+04,
     |  1.7850E+04, 1.9654E+04, 2.1106E+04, 2.1728E+04, 2.0471E+04,
     |  1.8375E+04, 1.3856E+04, 1.4273E+04, 1.3669E+04, 1.1855E+04,
     |  6.3800E+03, 3.7496E+03, 3.9188E+02/)
!
      B720(:, 1) = (/
     |  2.8405E+03, 3.0395E+03, 3.4408E+03, 3.9717E+03, 4.4535E+03,
     |  1.9917E+03, 2.3613E+03, 2.7008E+03, 3.0362E+03, 3.4208E+03,
     |  3.8614E+03, 4.3499E+03, 4.8714E+03, 5.3807E+03, 5.8595E+03,
     |  6.3290E+03, 6.8067E+03, 7.2173E+03, 9.8640E+03, 1.3557E+04,
     |  1.8960E+04, 2.6435E+04, 3.5153E+04, 4.1834E+04, 4.6296E+04,
     |  4.9877E+04, 5.3221E+04, 5.6581E+04, 5.9405E+04, 6.2639E+04,
     |  6.6401E+04, 7.0680E+04, 7.6279E+04, 8.2793E+04, 9.0707E+04,
     |  1.0054E+05, 1.1166E+05, 1.2928E+05, 1.4749E+05, 1.6040E+05,
     |  1.6061E+05, 1.5859E+05, 1.4883E+05/)
      B720(:, 2) = (/
     |  0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00, 0.0000E+00,
     |  4.3034E+03, 6.0062E+03, 8.1598E+03, 1.1082E+04, 1.4882E+04,
     |  1.8997E+04, 2.2593E+04, 2.5845E+04, 2.8553E+04, 3.1115E+04,
     |  3.3734E+04, 3.6921E+04, 4.0340E+04, 3.7426E+04, 3.3316E+04,
     |  2.8651E+04, 2.4077E+04, 2.0134E+04, 1.7323E+04, 1.6071E+04,
     |  1.6065E+04, 1.7738E+04, 2.1719E+04, 2.8331E+04, 3.8250E+04,
     |  5.2068E+04, 6.9586E+04, 9.7056E+04, 1.3593E+05, 1.8878E+05,
     |  2.6168E+05, 3.5601E+05, 4.9256E+05, 6.4638E+05, 7.7329E+05,
     |  8.4121E+05, 7.6916E+05, 8.3991E+05/)
      B720(:, 3) = (/
     |  7.4936E+03, 1.0131E+04, 1.4751E+04, 2.2216E+04, 3.1258E+04,
     |  3.4953E+04, 3.4166E+04, 3.0621E+04, 2.7699E+04, 2.6873E+04,
     |  2.7876E+04, 3.0243E+04, 3.3144E+04, 3.5605E+04, 3.6913E+04,
     |  3.7290E+04, 3.7617E+04, 3.8385E+04, 4.2372E+04, 4.7167E+04,
     |  5.2619E+04, 5.8637E+04, 6.5032E+04, 7.0915E+04, 7.8134E+04,
     |  8.8227E+04, 1.0254E+05, 1.2219E+05, 1.4764E+05, 1.8196E+05,
     |  2.2482E+05, 2.7870E+05, 3.5181E+05, 4.3639E+05, 5.0913E+05,
     |  5.3097E+05, 5.1509E+05, 5.6018E+05, 5.0425E+05, 3.8214E+05,
     |  3.0817E+05, 3.5079E+05, 4.6067E+05/)
      B720(:, 4) = (/
     |  8.7341E+04, 9.3355E+04, 9.9562E+04, 1.0150E+05, 9.7066E+04,
     |  9.2307E+04, 9.0119E+04, 8.9641E+04, 9.1664E+04, 9.6021E+04,
     |  1.0221E+05, 1.0820E+05, 1.1336E+05, 1.1671E+05, 1.1787E+05,
     |  1.1897E+05, 1.2180E+05, 1.2715E+05, 1.3471E+05, 1.4417E+05,
     |  1.5646E+05, 1.7286E+05, 1.9332E+05, 2.1622E+05, 2.4683E+05,
     |  2.8824E+05, 3.4326E+05, 4.1394E+05, 4.8695E+05, 5.5900E+05,
     |  5.7373E+05, 6.1047E+05, 6.1597E+05, 6.2339E+05, 6.0879E+05,
     |  6.9832E+05, 7.7427E+05, 8.8127E+05, 9.0404E+05, 1.0231E+06,
     |  1.3100E+06, 1.6460E+06, 1.8838E+06/)
      B720(:, 5) = (/
     |  4.9192E+04, 4.9040E+04, 4.7145E+04, 3.8150E+04, 2.7664E+04,
     |  1.9547E+04, 1.6574E+04, 1.4405E+04, 1.1561E+04, 7.9976E+03,
     |  4.8500E+03, 1.4568E+03,-2.1801E+03,-5.6919E+03,-9.1133E+03,
     | -1.1695E+04,-1.2080E+04,-1.0377E+04,-6.8135E+03,-1.7421E+03,
     |  4.6977E+03, 1.2577E+04, 2.1793E+04, 3.0628E+04, 3.8080E+04,
     |  4.2518E+04, 4.1670E+04, 3.3028E+04, 1.1952E+04,-2.3563E+04,
     | -8.0794E+04,-1.6617E+05,-2.5748E+05,-4.0507E+05,-4.8413E+05,
     | -4.8597E+05,-3.7861E+05,-3.8153E+05,-3.8801E+05,-3.2254E+05,
     | -1.5147E+05,-5.9600E+04,-3.0351E+04/)
      B720(:, 6) = (/
     | -2.8333E+05,-2.9971E+05,-3.2191E+05,-3.2913E+05,-3.2090E+05,
     | -3.0857E+05,-3.0392E+05,-2.9990E+05,-3.0091E+05,-3.0879E+05,
     | -3.2423E+05,-3.4127E+05,-3.5932E+05,-3.7306E+05,-3.7920E+05,
     | -3.8505E+05,-3.9926E+05,-4.2332E+05,-4.5532E+05,-4.9363E+05,
     | -5.4086E+05,-5.9984E+05,-6.6797E+05,-7.3687E+05,-8.2277E+05,
     | -9.3297E+05,-1.0730E+06,-1.2464E+06,-1.4319E+06,-1.6381E+06,
     | -1.8038E+06,-1.9916E+06,-2.2175E+06,-2.4290E+06,-2.6263E+06,
     | -3.0992E+06,-3.5297E+06,-4.3802E+06,-4.9000E+06,-5.4117E+06,
     | -5.9565E+06,-6.3166E+06,-6.7962E+06/)
      B720(:, 7) = (/
     |  1.0239E+04, 8.3936E+03, 4.6792E+03,-3.5717E+02,-4.8002E+03,
     | -7.8721E+03,-9.9420E+03,-1.1980E+04,-1.4404E+04,-1.7492E+04,
     | -2.2088E+04,-2.7798E+04,-3.3242E+04,-3.8152E+04,-4.2342E+04,
     | -4.5911E+04,-4.8849E+04,-5.1755E+04,-5.4983E+04,-5.8650E+04,
     | -6.3169E+04,-6.8828E+04,-7.5644E+04,-8.2391E+04,-9.1160E+04,
     | -1.0344E+05,-1.2081E+05,-1.4426E+05,-1.6963E+05,-1.9721E+05,
     | -2.1264E+05,-2.3986E+05,-2.6471E+05,-2.9629E+05,-3.3955E+05,
     | -4.3916E+05,-5.5374E+05,-6.7059E+05,-7.2800E+05,-7.6483E+05,
     | -8.0552E+05,-9.0012E+05,-9.5292E+05/)
      B720(:, 8) = (/
     |  2.1849E+04, 3.1428E+04, 3.5767E+04, 3.1331E+04, 2.4649E+04,
     |  2.0410E+04, 2.0706E+04, 2.2601E+04, 2.5526E+04, 2.8626E+04,
     |  3.3725E+04, 3.9168E+04, 4.5512E+04, 5.1589E+04, 5.6627E+04,
     |  6.1683E+04, 6.8127E+04, 7.6327E+04, 8.5856E+04, 9.5871E+04,
     |  1.0667E+05, 1.1902E+05, 1.3247E+05, 1.4292E+05, 1.5362E+05,
     |  1.6515E+05, 1.7693E+05, 1.8800E+05, 1.9868E+05, 2.1417E+05,
     |  2.4342E+05, 2.7442E+05, 3.2799E+05, 3.9401E+05, 4.2481E+05,
     |  5.6066E+05, 6.2336E+05, 8.8036E+05, 1.0841E+06, 1.2775E+06,
     |  1.3618E+06, 1.3832E+06, 1.5174E+06/)
      B720(:, 9) = (/
     |  2.9371E+04, 2.7742E+04, 2.8804E+04, 3.1031E+04, 3.2076E+04,
     |  3.1861E+04, 3.0832E+04, 2.9281E+04, 2.8022E+04, 2.7559E+04,
     |  2.7139E+04, 2.6704E+04, 2.5934E+04, 2.4863E+04, 2.3388E+04,
     |  2.1879E+04, 2.1062E+04, 2.0957E+04, 2.1689E+04, 2.3536E+04,
     |  2.6810E+04, 3.1515E+04, 3.7241E+04, 4.5592E+04, 5.8728E+04,
     |  7.8532E+04, 1.0552E+05, 1.3974E+05, 1.7822E+05, 2.2020E+05,
     |  2.6171E+05, 3.0483E+05, 3.4849E+05, 3.8673E+05, 3.8833E+05,
     |  3.6508E+05, 2.8233E+05, 2.8721E+05, 2.6558E+05, 2.2449E+05,
     |  1.2915E+05, 7.3788E+04, 1.6388E+04/)
      end subroutine set_cco2_data
!-----------------------------------------------------------------------
      end module cco2gr_module
