      module tgcm_module
      use netcdf
      implicit none
!
! B. Foster Sep-Oct, 2011.
! Neutrals are to be read from TIMEGCM or TIEGCM model netcdf output 
! history files. Path to the history file (tgcm_ncfile) was read from 
! the sami namelist, and is passed to sub read_tgcm.
!
! 10/7/11: Does not work w/ tiegcm files if target_zkm < 97 km.
!   Extrapolation to 78 km is not set up yet.
!
! Dimensions and variables to be returned to sami3 (formerly in common /UNVN/):
      integer :: jlon, jlat, jalt, jtime
      real,allocatable,save :: glatt(:),glont(:),tgcm_day(:),tgcm_ut(:)
      real,allocatable,dimension(:,:,:,:),save :: ! (jlon,jlat,jalt,jtime)
     |  zt, unt, vntt, tn2, to2, to1, tno, tn4s, thyd, ttn
!
      contains
!-----------------------------------------------------------------------
      subroutine read_tgcm(taskid,tgcm_ncfile)
!
! Locals are read from the file, and transferred (with necessary unit
! conversions) to the module data arrays for use by sami3.
!
! Dimensions:
!   Local:  nlon, nlat, nlev, ntime
!   Module: JLON, JLAT, JALT, JTIME
!
! Coordinate Variables:
!   Local:  glat,  glon, glev
!   Module: glatt, glont
!
! 4d variables (lon,lat,lev,time):
!   Local:  z,  un,  vn,   tn,   n2  o2,  o1,  n4s,  no,  h
!   Module: zt, unt, vntt, ttn, tn2, to2, to1, tn4s, tno, thyd
!
! Args:
      integer,intent(in) :: taskid
      character(len=*),intent(in) :: tgcm_ncfile
!
! Local dimensions and variables to be read from the file:
      integer :: nlon,nlat,nlev,ntime
      real,allocatable,save :: glat(:),glon(:),glev(:)
      real,allocatable,dimension(:,:,:),save ::    ! (nlon,nlat,nlev)
     |  barm,pkt 
      real,allocatable,dimension(:,:,:,:),save ::  ! (nlon,nlat,nlev,ntime)
     |  z,  un,  vn,   tn,  n2,  o2,  o1,  n4s,  no,  h
      real :: p0
      real :: target_zkm = 78. ! low bound (km) for vars returned to sami
      character(len=1024) :: msg
!
! Other local:
      integer :: istat,id,ncid,k,k0,k1,it
      character(len=NF90_MAX_NAME) :: varname
      integer :: ndims,nvars,natts,unlimid,formatnum
      real :: zkm0,zkm1
      real,parameter :: boltz=1.3805e-16
      real,allocatable,save :: zmalt(:) ! zonal mean alts (jalt)
!
! Later, add tgcm_ncfile to namelist
!     tgcm_ncfile = 'timegcm1.42.scntr_eqnx_smin_001.nc'
!
! Open netcdf file read-only:
      istat = nf90_open(tgcm_ncfile,NF90_NOWRITE,ncid)
      if (istat /= NF90_NOERR) then
        write(msg,"('Error opening tgcm_ncfile ',a)") trim(tgcm_ncfile)
        call handle_ncerr(istat,trim(msg),1)
      endif
      if (taskid == 0)
     |  write(6,"(/,'Opened TGCM netcdf file ',a,' ncid=',i4)")
     |    trim(tgcm_ncfile),ncid
      istat = nf90_inquire(ncid,ndims,nvars,natts,unlimid,formatnum)
!
! Get dimensions on the file:
      istat = nf90_inq_dimid(ncid,'lon',id)
      istat = nf90_inquire_dimension(ncid,id,varname,nlon)
      istat = nf90_inq_dimid(ncid,'lat',id)
      istat = nf90_inquire_dimension(ncid,id,varname,nlat)
      istat = nf90_inq_dimid(ncid,'lev',id)
      istat = nf90_inquire_dimension(ncid,id,varname,nlev)
      istat = nf90_inquire_dimension(ncid,unlimid,varname,ntime)

!     write(6,"('nlon=',i4,' nlat=',i4,' nlev=',i4,' ntime=',i4)") 
!    |  nlon,nlat,nlev,ntime
!
! Allocate variables to be read from the file:
      allocate(glon(nlon),glat(nlat),glev(nlev))
      allocate(z  (nlon,nlat,nlev,ntime))
      allocate(un (nlon,nlat,nlev,ntime))
      allocate(vn (nlon,nlat,nlev,ntime))
      allocate(tn (nlon,nlat,nlev,ntime))
      allocate(n2 (nlon,nlat,nlev,ntime))
      allocate(o2 (nlon,nlat,nlev,ntime))
      allocate(o1 (nlon,nlat,nlev,ntime))
      allocate(n4s(nlon,nlat,nlev,ntime))
      allocate(no (nlon,nlat,nlev,ntime))
      allocate(h  (nlon,nlat,nlev,ntime))
!
! day of year and ut are public module data (will not be deallocated):
! (prefix name w/ tgcm to avoid naming conflicts w/ sami3)
! (ntime == jtime)
      allocate(tgcm_day(ntime))
      allocate(tgcm_ut(ntime))
      tgcm_day = 0.
      tgcm_ut  = 0.
!
! Read variables:
      do id=1,nvars
        istat = nf90_inquire_variable(ncid,id,name=varname)
        select case (trim(varname))
          case ('lon') 
            istat = nf90_get_var(ncid, id, glon)
          case ('lat')
            istat = nf90_get_var(ncid, id, glat)
          case ('lev')
            istat = nf90_get_var(ncid, id, glev)
          case ('Z')
            istat = nf90_get_var(ncid, id, z)
          case ('UN')
            istat = nf90_get_var(ncid, id, un)
          case ('VN')
            istat = nf90_get_var(ncid, id, vn)
          case ('TN')
            istat = nf90_get_var(ncid, id, tn)
          case ('O2')
            istat = nf90_get_var(ncid, id, o2)
          case ('O1')
            istat = nf90_get_var(ncid, id, o1)
          case ('N4S')
            istat = nf90_get_var(ncid, id, n4s)
          case ('NO')
            istat = nf90_get_var(ncid, id, no)
          case ('H')
            istat = nf90_get_var(ncid, id, h)
          case ('p0')
            istat = nf90_get_var(ncid, id, p0)
          case ('day')
            istat = nf90_get_var(ncid, id, tgcm_day)
          case ('ut')
            istat = nf90_get_var(ncid, id, tgcm_ut)
          case default
!           write(6,"('Did not read variable ',a)") trim(varname)
        end select
        if (istat /= NF90_NOERR) then
          write(msg,"('Error getting var ',a)") trim(varname)
          call handle_ncerr(istat,msg,1)
        endif
      enddo ! id=1,nvars on the file
!
! n2 from o1,o2 mass mixing ratio: 
      n2 = 1.-o2-o1 ! mmr
!
! Report min,max of reads to stdout:
!     write(6,"('Read nlon=',i4,' glon min,max=',2f9.3)")
!    |  nlon,minval(glon),maxval(glon)
!     write(6,"('Read nlat=',i4,' glat min,max=',2f9.3)")
!    |  nlat,minval(glat),maxval(glat)
!     write(6,"('Read nlev=',i4,' glev min,max=',2f9.3)")
!    |  nlev,minval(glev),maxval(glev)
!     write(6,"('Read TN  min,max=',2e12.4)") minval(z),maxval(z)
!     write(6,"('Read UN  min,max=',2e12.4)") minval(un),maxval(un)
!     write(6,"('Read VN  min,max=',2e12.4)") minval(vn),maxval(vn)
!     write(6,"('Read Z   min,max=',2e12.4)") minval(z),maxval(z)
!     write(6,"('Read O2  min,max=',2e12.4)") minval(o2),maxval(o2)
!     write(6,"('Read O1  min,max=',2e12.4)") minval(o1),maxval(o1)
!     write(6,"('Read N2  min,max=',2e12.4)") minval(n2),maxval(n2)
!     write(6,"('Read N4S min,max=',2e12.4)") minval(n4s),maxval(n4s)
!     write(6,"('Read NO  min,max=',2e12.4)") minval(no),maxval(no)
!     write(6,"('Read H   min,max=',2e12.4)") minval(h),maxval(h)
!
! Find pressure level w/ geopotential near target_zkm (78 km): 
! (just look at first time):
      k0 = 0
      kloop: do k=2,nlev
        zkm0 = sum(z(:,:,k-1,1)/1.e5) / (float(nlon*nlat)) 
        zkm1 = sum(z(:,:,k  ,1)/1.e5) / (float(nlon*nlat)) 
        if (zkm0 <= target_zkm .and. zkm1 >= target_zkm) then
          k0 = k-1
          if (zkm1-target_zkm < target_zkm-zkm0) k0 = k
          exit kloop
        endif 
      enddo kloop
      if (k0==0) then
        write(6,"('>>> Could not bracket target_zkm ',f8.2)")
     |    target_zkm
        stop 'target_zkm'
      endif
      k1 = nlev-1
      jalt = k1-k0+1 ! number of altitudes for sami arrays
!
! Allocate sami arrays:
!   Module: zt, unt, vntt, ttn, tn2, to2, to1, tn4s, tno, thyd
!
      jlat = nlat ; jlon = nlon ; jtime = ntime
      allocate(glatt(jlat))
      allocate(glont(jlon))
      allocate(zmalt(jalt)) ! zonal mean heights (local for info only)
      allocate(zt  (jlon,jlat,jalt,jtime))
      allocate(unt (jlon,jlat,jalt,jtime))
      allocate(vntt(jlon,jlat,jalt,jtime))
      allocate(ttn (jlon,jlat,jalt,jtime))
      allocate(tn2 (jlon,jlat,jalt,jtime))
      allocate(to2 (jlon,jlat,jalt,jtime))
      allocate(to1 (jlon,jlat,jalt,jtime))
      allocate(tn4s(jlon,jlat,jalt,jtime))
      allocate(tno (jlon,jlat,jalt,jtime))
      allocate(thyd(jlon,jlat,jalt,jtime))
!
! Transfer to sami arrays, w/ unit conversion as necessary:
      glatt = glat 
      glont = glon
      zt  = z(:,:,k0:k1,:)/1.e5   ! cm to km
      ttn = tn(:,:,k0:k1,:)
      unt  = un(:,:,k0:k1,:)/100. ! cm/s to m/s
      vntt = vn(:,:,k0:k1,:)/100. ! cm/s to m/s
!
! Get zonal mean altitudes from k0 to k1 for information:
      do k=k0,k1
        zmalt(k-k0+1) = sum(z(:,:,k,1)/1.e5) / (float(nlon*nlat)) 
      enddo
!
! Density unit conversion:
      allocate(pkt(nlon,nlat,nlev),barm(nlon,nlat,nlev)) 
      do it=1,ntime
        do k=1,nlev
          pkt(:,:,k) = p0*exp(-glev(k))/(boltz*tn(:,:,k,it))
          barm(:,:,k) = 1./(o2(:,:,k,it)/32.+o1(:,:,k,it)/16.+
     |                      n2(:,:,k,it)/28.)
        enddo
        to2(:,:,:,it) = o2(:,:,k0:k1,it) * pkt(:,:,k0:k1) * 
     |    barm(:,:,k0:k1) / 32.
        to1(:,:,:,it) = o1(:,:,k0:k1,it) * pkt(:,:,k0:k1) * 
     |    barm(:,:,k0:k1) / 16.
        tn2(:,:,:,it) = n2(:,:,k0:k1,it) * pkt(:,:,k0:k1) * 
     |    barm(:,:,k0:k1) / 28.
        tno(:,:,:,it) = no(:,:,k0:k1,it) * pkt(:,:,k0:k1) * 
     |    barm(:,:,k0:k1) / 30.
        tn4s(:,:,:,it) = n4s(:,:,k0:k1,it) * pkt(:,:,k0:k1) * 
     |    barm(:,:,k0:k1) / 14.
        thyd(:,:,:,it) = h(:,:,k0:k1,it) * pkt(:,:,k0:k1) * 
     |    barm(:,:,k0:k1) / 1.
      enddo ! it=1,ntime
!
! Only master task reports to stdout:
      if (taskid == 0) then
        write(6,"('jlon=',i4,' jlat=',i4,' jalt=',i4,' jtime=',i4)")
     |    jlon,jlat,jalt,jtime
        write(6,"('glatt min,max=',2f9.2)")  minval(glatt),maxval(glatt)
        write(6,"('glont min,max=',2f9.2)")  minval(glont),maxval(glont)
        write(6,"('glev  min,max=',2f9.2,' (tgcm ln(p0/p)')")  
     |    minval(glev(k0:k1)),maxval(glev(k0:k1))
        write(6,"('zmalt min,max=',2f9.2,' (approx zm alt (km))')")  
     |    minval(zmalt),maxval(zmalt)
        write(6,"('ttn   min,max=',2e12.4)") minval(ttn),maxval(ttn)
        write(6,"('unt   min,max=',2e12.4)") minval(unt),maxval(unt)
        write(6,"('vnt   min,max=',2e12.4)") minval(vntt),maxval(vntt)
        write(6,"('zt    min,max=',2e12.4)") minval(zt),maxval(zt)
        write(6,"('to2   min,max=',2e12.4)") minval(to2),maxval(to2)
        write(6,"('to1   min,max=',2e12.4)") minval(to1),maxval(to1)
        write(6,"('tn2   min,max=',2e12.4)") minval(tn2),maxval(tn2)
        write(6,"('tn4s  min,max=',2e12.4)") minval(tn4s),maxval(tn4s)
        write(6,"('tno   min,max=',2e12.4)") minval(tno),maxval(tno)
        write(6,"('thyd  min,max=',2e12.4)") minval(thyd),maxval(thyd)
        write(6,"('day   min,max= ',2f8.2)") 
     |    minval(tgcm_day),maxval(tgcm_day)
        write(6,"('tgcm_ut  = ',/,(10f8.2))") tgcm_ut
        write(6,"(/)")
      endif
!
! Release local memory:
      deallocate(glon,glat,glev,pkt,barm,zmalt)
      deallocate(z)  
      deallocate(un)
      deallocate(vn)
      deallocate(tn)
      deallocate(n2)
      deallocate(o2)
      deallocate(o1)
      deallocate(n4s)
      deallocate(no)
      deallocate(h)

      end subroutine read_tgcm
!-----------------------------------------------------------------------
      subroutine handle_ncerr(istat,msg,ifatal)
!
! Handle a netcdf lib error:
!
      integer,intent(in) :: istat,ifatal
      character(len=*),intent(in) :: msg
!
      write(6,"(/72('-'))")
      write(6,"('>>> Error from netcdf library:')")
      write(6,"(a)") trim(msg)
      write(6,"('istat=',i5)") istat
      write(6,"(a)") nf90_strerror(istat)
      write(6,"(72('-')/)")
      if (ifatal > 0) stop 'Fatal netcdf error'
      end subroutine handle_ncerr
!-----------------------------------------------------------------------
      end module tgcm_module
