;
pro getzmean,info,slice 
;
;  This routine gets the latitude/longitude mean of altitude for map projections.  Not implemented for ilev fields.
; 
;  Get interface levels to test if this is an interface field
;
  file = *info.pfile
  levs = file.levs

;  Get variables from map info structures
;
  field = slice.field
  ixtime = slice.imtime
  iLev = slice.izvert

  fields = *info.fields
;
;  Check if this field is on interface levels and, if so, set the mean to -1 and quit
;
  if field.nlev ne n_elements(levs) then begin
    print, 'Field is on  levels, mean altitude not calculated'
    slice.zmean = -1
    return
  endif
; 
;  Get z coordinate data index
;
  ixz = -1
  ztype = ' '

  for i=0,n_elements(fields)-1 do begin

    IF info.ftype eq 'WACCM' THEN BEGIN

      IF field.grid_type eq 'magnetic' THEN BEGIN

        if strcompress(fields[i].name,/remove_all) eq 'ZMAG' then ixz = i

      ENDIF ELSE BEGIN

        if strcompress(fields[i].name,/remove_all) eq 'Z3' then ixz = i
    
      ENDELSE

    ENDIF ELSE BEGIN

      if field.grid_type eq 'magnetic' then begin

        if strcompress(fields[i].name,/remove_all) eq 'ZMAG' then ixz = i    

      endif else begin

        if strcompress(fields[i].name,/remove_all) eq 'Z' then ixz = i    
        if strcompress(fields[i].name,/remove_all) eq 'ZG' then ixz = i   
        if strcompress(fields[i].name,/remove_all) eq 'ZG' then ztype = 'ZG'
  
      endelse

    ENDELSE

  endfor
  
  IF ixz EQ -1 THEN BEGIN
    PRINT, 'No Z axis found from input file. Cannot get average altitude'
    zmeanlatlon = -1
    RETURN
  ENDIF      
;
; Read Z field if necessary(not read before):
;
  if not ptr_valid(fields[ixz].data) then begin ; field has not been read
;  print,'Reading Z field for altitude mapping..'
    widget_control,/hourglass
;
; Read Z data from file:
;
    varget,info,fields[ixz],zdata
;
;  Adjust altitude to kilometers and, for WACCM files, find vertical level to access original file 
;  altitude data since slice altitude data is already interpolated to TGCM-like zp levels
;
    IF info.ftype EQ 'WACCM' THEN BEGIN ;Geopotential altitude is in meters in WACCM history files

      levs = *fields[ixz].levs
      iLev = ixfind_nearest(levs,slice.levs[slice.izvert])

      zdata = zdata * 1.e-3  ; m to km

    ENDIF ELSE BEGIN; Centimeters in TGCM history files 

      zdata = zdata * 1.e-5  ; cm to km

    ENDELSE

  ENDIF ELSE BEGIN ;Field already read so grab it and set map level to get mean altitude
  
    zdata = *fields[ixz].data

    levs = *fields[ixz].levs
    iLev = ixfind_nearest(levs,slice.levs[slice.izvert])

  ENDELSE  
;
;  Convert geopotential to geometric height if requested   
;   
  IF ztype NE 'ZG' THEN BEGIN

    zdatatemp = zdata
    zdata = zdatatemp * (1. + zdatatemp/6370.0)
 
  ENDIF
;
;  Get latitude/longitude dimensions from zdata and set up output array
; 
  dims = size(zdata,/dimensions)
  ndims = n_elements(dims)
;
;  Check for up to 4 dimensional zdata.  Only processes data for 4 dimensions.  Less than 4 cancels altitude
;  plotting and returns
;  
  case ndims of
    4: begin
  	
    	if dims[2] ne field.nlev and field.nlev ne 0 then $
  	  print,'>>> WARNING getzmean: 3rd dimension of 4d data ',$
  	    'should be nlev=',field.nlev,' but is ',dims[2] 
  	if dims[3] ne field.ntime and field.ntime ne 1 then $
  	  print,'>>> WARNING getzmean: 4th dimension of data ',$
  	    'should be ntime=',field.ntime,' but is ',dims[3] 
;
;  Pick out altitude for current time.  Assume dimensions lon,lat,lev,time
;    
        zmeanlatlon = MEAN(zdata[*,*,iLev,ixtime])
;
;  For dimensions less than 4, assign pressure level data and quit altitude interpolation
;
    end
    3: begin

  	if dims[2] ne field.ntime and field.ntime ne 1 then $
  	  print,'>>> WARNING getzmean: 3rd dimension of data ',$
  	    'assumed be ntime=',field.ntime,' but is ',dims[2] 
        zmeanlatlon = MEAN(zdata[*,*,ixtime])
      
    end
    2: begin

        zmeanlatlon = MEAN(zdata[*,*])
      
    end
    else: begin
        print,'>>> getaltzdata: var data has ',ndims,' dimensions --',$
  	' cannot make slice data.'
        zmeanlatlon = -1
    end
  endcase

slice.zmean = zmeanlatlon

return

end
