pro checkalt,info,slice,alt
;
;  This routine takes the requested altitude for mapping and checks to see it is in the range of current 
;  altitude values
; 
;  Get map and info structure variables
;

  IF slice.plotz LT 4 THEN BEGIN

  field = slice.field
  ixtime = slice.imtime

  fields = *info.fields

  ixz = -1
  for i=0,n_elements(fields)-1 do begin
;
;  So far 3 types of altitude fields so check according to slice.plotz flag
;
    IF slice.plotz EQ 1 OR slice.plotz EQ 2 THEN BEGIN
     
      if strcompress(fields[i].name,/remove_all) eq 'Z' then ixz = i    
      if strcompress(fields[i].name,/remove_all) eq 'Z3' then ixz = i

    ENDIF ELSE IF slice.plotz EQ 3 THEN BEGIN
 
      if strcompress(fields[i].name,/remove_all) eq 'ZG' then ixz = i   

    ENDIF
  
  endfor
;  
;  If can't find altitude field in input field list then quit altitude level plotting  
;  
  if ixz eq -1 AND (slice.plotz EQ 1 OR slice.plotz EQ 2) then begin
    print,'>>> WARNING setalt: Could not find Z or Z3 field - no altitude plotting done'
    alt = -2
    return
  endif
;
  if ixz eq -1 AND slice.plotz EQ 3 then begin
    print,'>>> WARNING setalt: Could not find ZG field - no altitude plotting done'
    alt = -2
    return
  endif
;
; Read Z field if necessary(not read before):
;
  if not ptr_valid(fields[ixz].data) then begin ; field has not been read
;    print,'Reading altitude field for altitude plotting..'
    widget_control,/hourglass
;
; Read Z data from file:
    varget,info,fields[ixz],zdata
;
;  Adjust altitude to kilometers
;
    IF info.ftype EQ 'WACCM' THEN BEGIN ;Geopotential altitude is in meters in WACCM history files

      zdata = zdata * 1.e-3  ; m to km
      fields[ixz].data = ptr_new(zdata)
;
;  Change to approximate geometric altitude if requested
;        
      IF slice.plotz EQ 2 THEN zdata = zdatatemp * (1. + zdatatemp/6370.0)

    ENDIF ELSE BEGIN; Centimeters in TGCM history files 

      zdata = zdata * 1.e-5  ; cm to km
      fields[ixz].data = ptr_new(zdata)	       

    ENDELSE
  
  ENDIF ELSE BEGIN ;Field already read so grab it
  
    zdata = *fields[ixz].data

  ENDELSE   
;
;  Get zdata dimensions
;  
  dims = size(zdata,/dimensions)
  ndims = n_elements(dims)
 
  IF ndims NE 4 AND dims[2] NE field.nlev  THEN BEGIN
  
    print, 'Cannot interpolate to altitude.  Data is not 4 dimensional with level as 3rd dimension'
    alt = -1
    RETURN
    
  ENDIF  
;
;  Pick out altitude for current time.  Assume dimensions lon,lat,lev,time
;
  zdata1time = zdata[*,*,*,ixtime]
  
  nzd = SIZE(zdata1time,/dimensions)

  nLev = nzd[2]
;    
;  Need to average over longitudes and latitudes to get one altitude profile and then choose
;  level to map
; 
  zdata1timemean = FLTARR(nLev)
     
  FOR iLev = 0, nLev-1 DO BEGIN
  
    zdata1timemean[iLev] = MEAN(zdata[*,*,iLev])
  
  ENDFOR

  ENDIF ELSE BEGIN ; if slice.plotz lt 4
    
    levs = slice.levs
    nLev = N_ELEMENTS(levs)
    
    zdata1timemean = FLTARR(nLev)

    zdata1timemean = levs

  ENDELSE ; if slice.plotz eq 4
;
;  Check that altitude is between min/max
;  
  IF  alt LT zdata1timemean[0] OR alt GT zdata1timemean[nLev-1] THEN BEGIN
  
    print, 'Altitude outside data vertical range.
    print, 'Enter an altitude between ', zdata1timemean[0], ' -',zdata1timemean[nLev-1],' km' 
    alt = -1
    return

  ENDIF

END ; End of checkalt
;-----------------------------------------------------------------------
pro setalt_event,event
;
;  Set up window to get altitude from user
;
  widget_control,event.id,get_uvalue=widget
  widget_control,event.top,get_uvalue=paltstruct
  altstruct = *paltstruct
  case widget of
    'TEXTALT': begin
      widget_control,event.id,get_value=altstring
      altstruct.alt = mkfloat(altstring,altstruct.alt,event.id)
      print,'Set altitude to km = ',altstruct.alt
      *paltstruct = altstruct
    end
    'APPLY': begin
;
; Read alt value from text widget:
      widget_control,altstruct.textalt,get_value=altstring
      altstruct.alt = mkfloat(altstring,altstruct.alt,altstruct.textalt)
;
; Update state and quit:
      altstruct.alt = mkfloat(altstring,altstruct.alt,altstruct.textalt)
      altstruct.mode = 'APPLY'
      *paltstruct = altstruct
      widget_control,event.top,/destroy
    end ; APPLY
    'OK': begin
;
; Read alt value from text widget:
      widget_control,altstruct.textalt,get_value=altstring
      altstruct.alt = mkfloat(altstring,altstruct.alt,altstruct.textalt)
;
; Update state and quit:
      altstruct.alt = mkfloat(altstring,altstruct.alt,altstruct.textalt)
      *paltstruct = altstruct
      widget_control,event.top,/destroy
    end ; OK
    'HELP': begin
      xdisplayfile,"setalt.help",done_button='Done',title= $
        'Help for Selection of Altitude Level for Interpolation and Image Display',$
        group=event.top,modal=widget_info(event.top,/modal), $
        width=90,height=50,text=[ $
      " ",$
      "Set Altitude Level for Interpolation and Image Display.",$
      "('Altitude Level' option from the PlotOptions menu)"," ",$
      "('Set Geopotential','Set Geometric','Set Geometric ZG' options from the ",$
      "above 'Altitude Level' menu)"," ",$
      "There is one text field option:",$
      " ",$
      "  1. Enter altitude level in kilometers to which the plot will be interpolated.",$
      "     If value is outside range an error is given and able to re-enter altitude.",$
      " ",$
      "Buttons:"," ",$
      "  'APPLY':",$
      "     This button will validate and apply the altitude text entry and leave",$
      "     the window open for another entry.",$
      " ",$   
      "  'OK':",$
      "     This button will validate and apply the altitude text entry and close",$
      "     the window.",$
      " ",$   
      "  'HELP'",$
      "     This button displays this help information ",$
      " ",$   
      "Notes:"," ",$
      "  Geometric altitude is altitude converted from input model geopotential altitude",$
      "  using equation gmalt = gpalt*(1.+gpalt/Re) where radius of the earth, Re=6370.0 .",$ 
      " ",$
      "  Geometric altitude ZG is altitude calculated inside the model (just TGCMs for now)",$
      " ",$
      "  If altitude entered < minimum or > maximum altitude in data:",$
      "    Valid altitude range is given at command line and window remains for a valid entry",$
      " "$
      ]
    end
    else: print,'>>> setminmax: unrecognized event: ',widget
  endcase
end
;-----------------------------------------------------------------------
pro setalt,info,slice,alt,mode
;
; Widget to allow user to choose the altitude for the map plot.
;   alt is returned.
; Info is used only to provide tlb with a group_leader (info.tlb)
;   (tlb cannot be modal without a group leader)
;
; print,'Enter setalt: input alt=',alt

  altin = alt
  mode = 'OK'

  WIDGETALT: title = 'User Set Altitude'
  tlb = widget_base(column=1,title=title,group_leader=info.tlb,/modal)
;
  label = widget_label(tlb,value='Enter altitude in km ')
  textalt = widget_text(tlb,/editable,value=string(alt),uvalue='TEXTALT')
;
  base = widget_base(tlb,column=3,/frame)
  button = widget_button(base,value='APPLY',uvalue='APPLY')
  button = widget_button(base,value='OK',uvalue='OK')
  button = widget_button(base,value='HELP',uvalue='HELP')
;
; Set up structure to contain information for altitude selection window and start up window
;
  altstruct = {alt_struct,     $
    alt:alt,      $ ; input default altitude
    textalt:textalt,           $ ; widget id of min text
    mode:mode           $ ; mode of widget ('APPLY' to restore this widget after plotting or 'OK' not to restore)
  }
;
  paltstruct = ptr_new(altstruct)
  widget_control,tlb,set_uvalue=paltstruct
  widget_control,tlb,/realize
  xmanager,'setalt',tlb,/no_block
;
; Return altitude to calling procedure:
;
  alt = (*paltstruct).alt
;
;  Check to see if chosen altitude is within input range
;
  checkalt,info,slice,alt
  
;
;  If value not valid then ask again until valid value entered
;  
  IF alt EQ -1 THEN BEGIN
  
    alt = altin
    GOTO, WIDGETALT
  
  ENDIF
;
;  Set mode to identify if 'APPLY' button hit in window so selection window remains for another input
;
  mode = (*paltstruct).mode

  IF alt GT -1 THEN print,'Setting altitude to ',alt,' km

end
